/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdint.h>
#include <stdio.h>
#include <errno.h>
#include <sys/time.h>
#include <pthread.h>

#include <xine.h>
#include <xine/xineutils.h>

#include "common.h"
#include "event_sender.h"
#include "panel.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "xine-toolkit/image.h"
#include "xine-toolkit/labelbutton.h"

#define WINDOW_WIDTH    250
#define WINDOW_HEIGHT   221

static const struct {
  uint8_t x, y, w, h;
  int type;
  char label[8];
  char skin[16];
} _es_event_types[] = {
  /* 0 */
  {  5,  5, 80, 23, XINE_EVENT_INPUT_MENU1, "", "NavMenu1" },
  {  5, 28, 80, 23, XINE_EVENT_INPUT_MENU2, "", "NavMenu2" },
  {  5, 52, 80, 23, XINE_EVENT_INPUT_MENU3, "", "NavMenu3" },
  { 85,  5, 80, 23, XINE_EVENT_INPUT_MENU4, "", "NavMenu4" },
  {165,  5, 80, 23, XINE_EVENT_INPUT_MENU5, "", "NavMenu5" },
  {165, 28, 80, 23, XINE_EVENT_INPUT_MENU6, "", "NavMenu6" },
  {165, 52, 80, 23, XINE_EVENT_INPUT_MENU7, "", "NavMenu7" },
  /* 7 */
  { 57,124, 23, 23, XINE_EVENT_INPUT_NUMBER_9, "9", "NavNum9" },
  { 34,124, 23, 23, XINE_EVENT_INPUT_NUMBER_8, "8", "NavNum8" },
  { 11,124, 23, 23, XINE_EVENT_INPUT_NUMBER_7, "7", "NavNum7" },
  { 57,147, 23, 23, XINE_EVENT_INPUT_NUMBER_6, "6", "NavNum6" },
  { 34,147, 23, 23, XINE_EVENT_INPUT_NUMBER_5, "5", "NavNum5" },
  { 11,147, 23, 23, XINE_EVENT_INPUT_NUMBER_4, "4", "NavNum4" },
  { 57,170, 23, 23, XINE_EVENT_INPUT_NUMBER_3, "3", "NavNum3" },
  { 34,170, 23, 23, XINE_EVENT_INPUT_NUMBER_2, "2", "NavNum2" },
  { 11,170, 23, 23, XINE_EVENT_INPUT_NUMBER_1, "1", "NavNum1" },
  { 11,193, 23, 23, XINE_EVENT_INPUT_NUMBER_0, "0", "NavNum0" },
  { 34,193, 46, 23, XINE_EVENT_INPUT_NUMBER_10_ADD, "+10", "NavNumPlus10" },
  /* 18 */
  { 90, 39, 70, 40, XINE_EVENT_INPUT_UP,     N_("Up"), "NavUp" },
  { 20, 79, 70, 40, XINE_EVENT_INPUT_LEFT,   N_("Left"), "NavLeft" },
  { 95, 84, 60, 30, XINE_EVENT_INPUT_SELECT, N_("Select"), "NavSelect" },
  {160, 79, 70, 40, XINE_EVENT_INPUT_RIGHT,  N_("Right"), "NavRight" },
  { 90,119, 70, 40, XINE_EVENT_INPUT_DOWN,   N_("Down"), "NavDown" },
  {165,124, 80, 23, XINE_EVENT_INPUT_ANGLE_NEXT,     N_("Angle +"), "NavAnglePlus" },
  {165,147, 80, 23, XINE_EVENT_INPUT_ANGLE_PREVIOUS, N_("Angle -"), "NavAngleMinus" }
};
#define _ES_NUM_EVENTS (sizeof (_es_event_types) / sizeof (_es_event_types[0]))

static const char _es_menu_names[][12] = {
  /*  0 */ N_("Menu"),
  /*  1 */ N_("Menu 1"), N_("Menu 2"), N_("Menu 3"), N_("Menu 4"),
           N_("Menu 5"), N_("Menu 6"), N_("Menu 7"),
  /*  8 */ N_("Menu toggle"), N_("Title"), N_("Root"), N_("Subpicture"),
           N_("Audio"), N_("Angle"), N_("Part"),
  /* 15 */ N_("Top Menu"), N_("Popup Menu"),
  /* 17 */ N_("Red"), N_("Green"), N_("Yellow"), N_("Blue")
};

static const uint8_t _es_menu_index[][7] = {
  /* "dvd:/", "dvdnav:/" */
  {  8, 9,10,11,12,13,14 },
  /* "bd:/" */
  { 15,16, 3, 4, 5, 6, 7 },
  /* "vdr:/", "xvdr" */
  {  0,17,18,19,20, 6, 7 },
  /* default */
  {  1, 2, 3, 4, 5, 6, 7 }
};

struct xui_event_sender_st {
  gui_new_window_t      nw;

  uint8_t               helipad[_ES_NUM_EVENTS];

  const char           *label[_ES_NUM_EVENTS];
  xitk_widget_t        *wdgts[_ES_NUM_EVENTS + 1];

  int                   visible;
  int                   skin;
  xitk_register_key_t   widget_key;

  uint32_t              shown_menu; /** << 0 .. sizeof (_es_menu_index) / sizeof (_es_menu_index[0]) */
};

static int event_sender_test_move (xui_event_sender_t *es, int move) {
  xitk_rect_t _panel = { .x = 0 };
  int x, y1, y2, diff;
  int display_width = 0, display_height;

  /* simulste an actual magnet behaviour. */
  xitk_get_display_size (es->nw.gui->xitk, &display_width, &display_height);
  panel_get_window_position (es->nw.gui->panel, &_panel);
  x = _panel.x + _panel.width;
  if (x + es->nw.wr.width > display_width)
    x = _panel.x - es->nw.wr.width;
  y1 = y2 = _panel.y;
  diff = _panel.height - es->nw.wr.height;
  if (diff < 0)
    y1 += diff;
  else
    y2 += diff;
  diff = es->nw.wr.y;
  if (diff < y1)
    diff = y1;
  else if (diff > y2)
    diff = y2;

  if ((x != es->nw.wr.x) || (diff != es->nw.wr.y)) {
    es->nw.wr.x = x;
    es->nw.wr.y = diff;
    if (!move)
      return 1;
    _panel.x = es->nw.wr.x;
    _panel.y = es->nw.wr.y;
    _panel.width = XITK_INT_KEEP;
    _panel.height = XITK_INT_KEEP;
    xitk_window_move_resize (es->nw.xwin, &_panel);
    return 1;
  }
  return 0;
}

void event_sender_sticky_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  int old_sticky_value = gui->eventer_sticky;

  gui->eventer_sticky = cfg->num_value;
  if (gui->eventer) {
    xitk_labelbutton_set_state (gui->eventer->wdgts[_ES_NUM_EVENTS], gui->eventer_sticky);
    if ((!old_sticky_value) && gui->eventer_sticky)
      event_sender_test_move (gui->eventer, 1);
  }
}

/* Send given event to xine engine */
void event_sender_send (gGui_t *gui, int event) {
  xine_event_t   xine_event;

  if (!gui || !event)
    return;

  xine_event.type        = event;
  xine_event.data_length = 0;
  xine_event.data        = NULL;
  xine_event.stream      = gui->stream;
  xitk_gettime_tv (&xine_event.tv);
  xine_event_send (gui->stream, &xine_event);
}

static void event_sender_send2 (xitk_widget_t *w, void *data, int state) {
  xui_event_sender_t *es;
  uint8_t *land = (uint8_t *)data;
  int type = *land;
  xine_event_t event;

  (void)w;
  (void)state;
  xitk_container (es, land - type, helipad[0]);
  event.type        = _es_event_types[type].type;
  event.data_length = 0;
  event.data        = NULL;
  event.stream      = es->nw.gui->stream;
  xitk_gettime_tv (&event.tv);
  xine_event_send (es->nw.gui->stream, &event);
}

static void event_sender_magnet (xitk_widget_t *w, void *data, int state) {
  xui_event_sender_t *es = data;

  (void)w;
  if (state == es->nw.gui->eventer_sticky)
    return;

  /* will call event_sender_sticky_cb () above. */
  config_update_num (es->nw.gui->xine, "gui.eventer_sticky", state);
}

static void event_sender_exit (xitk_widget_t *w, void *data, int state) {
  xui_event_sender_t *es = data;

  (void)w;
  (void)state;
  if (es) {
    es->visible = 0;

    gui_save_window_pos (es->nw.gui, "eventer", es->widget_key);

    xitk_unregister_event_handler (es->nw.gui->xitk, &es->widget_key);
    xitk_window_destroy_window (es->nw.xwin);
    es->nw.xwin = NULL;
    es->nw.gui->eventer = NULL;
    video_window_set_input_focus (es->nw.gui->vwin);
    free (es);
  }
}

static int event_sender_event (void *data, const xitk_be_event_t *e) {
  xui_event_sender_t *es = data;
  xine_event_t xine_event;

  xine_event.type = -1;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      event_sender_exit (NULL, es, 0);
      return 1;
    case XITK_EV_POS_SIZE:
      es->nw.wr.x = e->x;
      es->nw.wr.y = e->y;
      es->nw.wr.width = e->w;
      es->nw.wr.height = e->h;
      return 1;
    case XITK_EV_BUTTON_UP:
      /* If we tried to move sticky window, move it back to stored position. */
      if (es->nw.gui->eventer_sticky && (panel_is_visible (es->nw.gui->panel) > 1))
        event_sender_test_move (es, 1);
      return 1;
    case XITK_EV_KEY_DOWN:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        switch (e->utf8[1]) {
          case XITK_KEY_UP:    xine_event.type = XINE_EVENT_INPUT_UP; break;
          case XITK_KEY_DOWN:  xine_event.type = XINE_EVENT_INPUT_DOWN; break;
          case XITK_KEY_LEFT:  xine_event.type = XINE_EVENT_INPUT_LEFT; break;
          case XITK_KEY_RIGHT: xine_event.type = XINE_EVENT_INPUT_RIGHT; break;
          case XITK_KEY_RETURN: xine_event.type = XINE_EVENT_INPUT_SELECT; break;
          case XITK_KEY_ESCAPE:
            event_sender_exit (NULL, es, 0);
            return 1;
          default: ;
        }
      } else if ((uint8_t)(e->utf8[0] ^ '0') < 10u) {
        xine_event.type = XINE_EVENT_INPUT_NUMBER_0 + (e->utf8[0] - '0');
      } else if (e->utf8[0] == '+') {
        xine_event.type = XINE_EVENT_INPUT_NUMBER_10_ADD;
      }
      break;
    default: ;
  }
  if (xine_event.type >= 0) {
    xine_event.data_length = 0;
    xine_event.data        = NULL;
    xine_event.stream      = es->nw.gui->stream;
    xitk_gettime_tv (&xine_event.tv);
    xine_event_send (es->nw.gui->stream, &xine_event);
    return 1;
  }
  return gui_handle_be_event (es->nw.gui, e);
}

static uint32_t _es_menu_mode (gGui_t *gui) {
  uint32_t mode = 3;

  gui_playlist_lock (gui);
  if (gui->mmk.mrl) {
    if ((!strncmp (gui->mmk.mrl, "dvd:/", 5)) || (!strncmp (gui->mmk.mrl, "dvdnav:/", 8))) {
      mode = 0;
    } else if (!strncmp (gui->mmk.mrl, "bd:/", 4)) {
      mode = 1;
    } else if (!strncmp (gui->mmk.mrl, "xvdr", 4) || !strncmp (gui->mmk.mrl, "vdr:/", 5)) {
      mode = 2;
    }
  }
  gui_playlist_unlock (gui);
  return mode;
}

void event_sender_menu_names (gGui_t *gui, const char *names[7]) {
  uint32_t mode, i;

  if (!gui || !names)
    return;

  mode = _es_menu_mode (gui);
  for (i = 0; i < 7; i++)
    names[i] = gettext (_es_menu_names[_es_menu_index[mode][i]]);
}

static int _es_get_menu_mode (xui_event_sender_t *es) {
  static const char menu_modes[][8] = { "DVD", "Bluray", "VDR", "FILE" };
  uint32_t mode, i;

  mode = _es_menu_mode (es->nw.gui);
  if (es->shown_menu == mode)
    return 0;
  es->shown_menu = mode;
  for (i = 0; i < 7; i++)
    es->label[i] = gettext (_es_menu_names[_es_menu_index[mode][i]]);
  if (es->nw.gui->verbosity >= 2)
    printf ("gui.eventer: menu mode %s.\n", menu_modes[mode]);
  return 1;
}

void event_sender_update_menu_buttons (gGui_t *gui) {
  if (gui && gui->eventer && _es_get_menu_mode (gui->eventer)) {
    xui_event_sender_t *es = gui->eventer;
    uint32_t i;

    for (i = 0; i < 7; i++)
      xitk_labelbutton_change_label (es->wdgts[i], gettext (es->label[i]));
  }
}

void event_sender_move (gGui_t *gui, int x, int y) {
  (void)x;
  (void)y;
  if (gui && gui->eventer && gui->eventer_sticky)
    event_sender_test_move (gui->eventer, 1);
}

void event_sender_change_skins (xui_event_sender_t *es, int synthetic) {
  (void)synthetic;
  if (es)
    event_sender_exit (NULL, es, 0);
}

static void _es_adjust (gui_new_window_t *nw) {
  xui_event_sender_t *es;

  xitk_container (es, nw, nw);
  if (es->nw.gui->eventer_sticky && (panel_is_visible (es->nw.gui->panel) > 1))
    event_sender_test_move (es, 0);
}

void event_sender_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_event_sender_t *es;
  uint32_t i;

  if (!gui)
    return;

  es = gui->eventer;
  if (mode == XUI_W_OFF) {
    if (!es)
      return;
    event_sender_exit (NULL, es, 0);
    return;
  } else if (mode == XUI_W_ON) {
    if (es) {
      gui_raise_window (gui, es->nw.xwin, 1, 0);
      xitk_window_set_input_focus (es->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (es) {
      event_sender_exit (NULL, es, 0);
      return;
    }
  }

  es = (xui_event_sender_t *)calloc (1, sizeof (*es));
  if (!es)
    return;

  for (i = 0; i < _ES_NUM_EVENTS; i++)
    es->helipad[i] = i;

  /* Create window */
  es->nw.gui = gui;
  es->nw.title = _("xine Event Sender");
  es->nw.id = "eventer";
  es->nw.skin = "NavBG";
  es->nw.wfskin = "NavWF";
  es->nw.adjust = _es_adjust;
  es->nw.wr.x = 80;
  es->nw.wr.y = 80;
  es->nw.wr.width = WINDOW_WIDTH;
  es->nw.wr.height = WINDOW_HEIGHT;
  es->skin = gui_window_new (&es->nw);
  if (es->skin < 0) {
    free (es);
    return;
  }

  gui->eventer = es;
  xitk_window_get_window_position (es->nw.xwin, &es->nw.wr);

  if (!es->skin) {
    xitk_image_t *wf = xitk_image_new (es->nw.gui->xitk, XINE_SKINDIR "/nav.png", 0, 0, 0);

    if (wf) {
      int w = xitk_image_width (wf), h = xitk_image_height (wf);
      xitk_image_widget_t iw = {
        .nw = {
          .wl = es->nw.wl,
          .tips = "",
          .userdata = es,
          .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
        }
      };
      xitk_part_image_t image = {
        .image = wf,
        .x = 0,
        .y = 0,
        .width = w,
        .height = h,
        .num_states = 2
      };
      xitk_widget_t *sym = xitk_noskin_part_image_create (&iw,
        &image, (2 * WINDOW_WIDTH - w) >> 2, 119 + 40 + ((WINDOW_HEIGHT - 2 - 119 - 40 - h) >> 1));

      xitk_widget_set_window_focusable (sym);
      xitk_image_free_image (&wf);
    }
  }

  es->shown_menu = ~0u;
  _es_get_menu_mode (es);
  for (i = 7; i < 18; i++)
    es->label[i] = _es_event_types[i].label;
  for (; i < _ES_NUM_EVENTS; i++)
    es->label[i] = gettext (_es_event_types[i].label);

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = es->nw.wl,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
      },
      .button_type       = CLICK_BUTTON,
      .align             = ALIGN_CENTER,
      .callback          = event_sender_send2
    };

    if (es->skin) {
      for (i = 0; i < _ES_NUM_EVENTS; i++) {
        lb.nw.skin_element_name = _es_event_types[i].skin;
        lb.nw.userdata = es->helipad + i;
        lb.nw.tips = lb.label = es->label[i];
        es->wdgts[i] = xitk_labelbutton_create (&lb, es->nw.gui->skin_config);
      }
      lb.nw.userdata = es;
      lb.nw.skin_element_name = "NavMagnet";
      lb.nw.add_state = es->nw.gui->eventer_sticky
                      ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                      : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      lb.nw.tips = _("Event sender window stick to main panel"); /** << same as in event.c */
      lb.button_type = RADIO_BUTTON;
      lb.callback = event_sender_magnet;
      lb.label = "<<";
      es->wdgts[_ES_NUM_EVENTS] = xitk_labelbutton_create (&lb, es->nw.gui->skin_config);
      lb.nw.skin_element_name = "NavDismiss";
      lb.nw.tips = lb.label = _("Close");
      lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
      lb.button_type = CLICK_BUTTON;
      lb.callback = event_sender_exit;
      xitk_labelbutton_create (&lb, es->nw.gui->skin_config);
    } else {
      for (i = 0; i < _ES_NUM_EVENTS; i++) {
        lb.nw.userdata = es->helipad + i;
        lb.label = es->label[i];
        es->wdgts[i] = xitk_noskin_labelbutton_create (&lb,
          _es_event_types[i].x, _es_event_types[i].y,
          _es_event_types[i].w, _es_event_types[i].h,
          XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
      }
      lb.nw.userdata = es;
      lb.nw.tips = _("Event sender window stick to main panel"); /** << same as in event.c */
      lb.nw.add_state = es->nw.gui->eventer_sticky
                      ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                      : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      lb.button_type = RADIO_BUTTON;
      lb.callback = event_sender_magnet;
      lb.label = "<<";
      es->wdgts[_ES_NUM_EVENTS] = xitk_noskin_labelbutton_create (&lb,
        WINDOW_WIDTH - 23 - 70 - 5, WINDOW_HEIGHT - 23 - 5, 23, 23,
        XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
      lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
      lb.label = _("Close");
      lb.callback = event_sender_exit;
      xitk_noskin_labelbutton_create (&lb,
        WINDOW_WIDTH - 70 - 5, WINDOW_HEIGHT - 23 - 5, 70, 23,
        XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
    }
  }

  es->widget_key = xitk_be_register_event_handler ("eventer", es->nw.xwin, event_sender_event, es, NULL, NULL);
  es->visible = 1;
  gui_raise_window (es->nw.gui, es->nw.xwin, 1, 0);
  xitk_window_set_input_focus (es->nw.xwin);
}
