/*  xdemorse.h
 *
 *  Header file for xdemorse
 */

/*
 *  xdemorse: An application to decode Morse signals to text
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#ifndef XDEMORSE_H
#define XDEMORSE_H 1

#include <gtk/gtk.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <stdio.h>
#include <math.h>

/************ CUSTOMIZE THESE DEFINITIONS FOR YOUR NEEDS *************/

/* Definitions for Morse code conversion to ascii */
#define NUMBER_OF_CHAR           55 /* Number of chars in ASCII table */

/* ASCII equivalents to Morse hex code. Last */
/* one (*) used for unrecognized characters. */
#define MORSE_ASCII_CHAR \
{ 'A','B','C','D','E','F','G','H','I','J','K','L','M', \
  'N','O','P','Q','R','S','T','U','V','W','X','Y','Z', \
  '1','2','3','4','5','6','7','8','9','0','.',',',':', \
  '?','\'','-','/','(','"',';','$','#','<','!','@',']',\
  '=','~',' ','*' }

/* Hex equivalents to Morse code chars above except (*).  */
/* Formed by starting with a 1 and following with a 0 for */
/* dash and 1 for dit e.g: A = ditdah = 101 = Hex 0x06.   */
#define MORSE_HEX_CODE \
{ 0x06,0x17,0x15,0x0b,0x03,0x1d,0x09,0x1f,0x07,0x18,0x0a,0x1b,0x04,\
  0x05,0x08,0x19,0x12,0x0d,0x0f,0x02,0x0e,0x1e,0x0c,0x16,0x14,0x13,\
  0x30,0x38,0x3c,0x3e,0x3f,0x2f,0x27,0x23,0x21,0x20,0x6a,0x4c,0x47,\
  0x73,0x61,0x5e,0x2d,0x52,0x6d,0x55,0xf6,0x35,0x7a,0x2a,0x37,0x29,\
  0x2e,0xff,0x01 }

/***************** END OF CUSTOMIZABLE SECTION ***********************/

/* Signal Detector parameter definitions */
#define BUFFER_SIZE            1024 /* Size of buffer for signal samples */
#define WFALL_WIDTH				256 /* Width of Waterfall drawingarea */
#define CYCLES_PER_FRAG           2 /* Cycles of signal/signal fragment  */
#define BINMAX_AVE_WIN            5 /* Length of bin max averaging window  */
#define FFT_SIZE	        	512 /* Size of fft input/output arrays */
#define FFT_SIZE2				256	/* Size/2 of fft input/output arrays */
#define GOERTZEL_SCALE		   1024 /* Scale factor for integer Goertzel algorithm*/
#define STEP_THRESHOLD			105 /* % diff between successive sig levels */
#define MAX_SPEED				 60 /* Maximum allowed Morse unit (dot) length */
#define MIN_SPEED				  6 /* Minimum allowed Morse unit (dot) length */

/* Transceiver type */
#define FT847	1
#define FT857	2
#define NONE	0

/* Definitions of contexts (stages) in Morse decoding process */
#define MARK_SIGNAL		 0x000001 /* Count fragments of a mark element   */
#define ELEM_SPACE		 0x000002 /* Count frag. of inter-element space  */
#define CHAR_SPACE		 0x000004 /* Count fragments of inter-char space */
#define WAIT_WORD_SPACE	 0x000008 /* Wait for an inter-word space        */
#define WORD_SPACE		 0x000010 /* Count fragments of inter-word space */
#define WAIT_FOR_MARK	 0x000020 /* Count fragments of no-signal space  */
#define WAIT_FOR_SPACE	 0x000040 /* Wait for a space after a long dash  */
#define MARK_TONE		 0x000080 /* Input signal is mark tone (key down)*/

/* Definitions of flags for various actions */
#define ADAPT_SPEED		 0x000100 /* Enable speed tracking */
#define MAN_THRESHOLD	 0x000200 /* Enable manual detector threshold */
#define WORD_WRAP		 0x000400 /* Enable word wrapping  */
#define DISPLAY_SIGNAL	 0x000800 /* Display output of signal detector */
#define DISPLAY_DETECTOR 0x001000 /* Display output of signal detector */
#define DSP_IO_ERROR	 0x002000 /* Error reading from dsp  */
#define REVERSE_VIDEO	 0x004000 /* Reverse waterfall video */
#define ENABLE_CAT		 0x008000 /* Enable CAT for transceiver */
#define CAT_SETUP		 0x010000 /* CAT is set up */
#define ENABLE_CAT_847	 0x020000 /* Enable CAT for FT847 */
#define ENABLE_CAT_857	 0x040000 /* Enable CAT for FT857 */
#define SERIAL_IO_ERROR	 0x080000 /* Error occured in serial i/o */
#define LONG_SPACE		 0x100000 /* Long period of space (no tone) */
#define ENABLE_RECEIVE	 0x200000 /* Enable reception of Morse code */

/* Channel to be used in audio capture */
#define LEFT        0 /* Left channel  */
#define RIGHT       1 /* Right channel */

/* Special characters */
#define LF      0x0A /* Line Feed */
#define CR      0x0D /* Carriage Return */
#define HT      0x09 /* Horizontal Tab  */

/* Return values */
#define ERROR			1	 /* Error condition */
#define SUCCESS			0	 /* No error condition */

/*-------------------------------------------------------------------*/

/* Runtine configuration data */
typedef struct
{
  char
	mix_dev[21], /* Mixer device name */
	dsp_dev[21], /* DSP device name   */
	rec_src[21], /* Recording/capture source */
	inp_lev[21]; /* Input level contol name  */

  /* Transceiver serial devices */
  char
	ft847_serial[32],
	ft857_serial[32];
  /* Transceiver type 1=FT847 2=FT867 0=NONE */
  int tcvr_type;

  int
	num_chn,	/* Number of audio channels (2=stereo, 1=mono) */
	use_chn,	/* Channel in use: 0=left, 1=right */
	rec_lev,	/* Recording/Capture level */
	dsp_speed;	/* DSP speed samples/sec */

  int
	tone_freq,	/* Receiver BFO Tone freq */
	red_line,	/* Waterfall BFO Tone freq marker line */
	fft_stride; /* Stride of fft over input samples */

  int
	unit_elem,	/* Morse unit element (dot) length */
	max_unit,	/* Maximum length of unit element  */
	min_unit,	/* Minimum length of unit element  */
	spd_wpm,	/* Current Morse speed words/min   */
	wrd_wrp,	/* Maximum line length if word wrap enabled   */
	det_thr;	/* Signal threshold for Mark/Space separation */


  /* Signal/dsp samples buffer */
  unsigned char *buffer;
  int
	buffer_idx,	 /* Index to signal samples buffer */
	buffer_size; /* Buffer size according to stereo/mono mode */

} rc_data_t;

/*-------------------------------------------------------------------*/

/* Function prototypes produced by cproto */

/* callbacks.c */
void Error_Dialog(char *message);
/* cat.c */
void Open_Tcvr_Serial(void);
void Read_Tcvr_Serial(unsigned char *cmnd, int block_size);
void Write_Tcvr_Command(unsigned char *cmnd);
void Close_Tcvr_Serial(void);
void Tune_Tcvr(double x);
int Read_Rx_Freq(void);
void Write_Rx_Freq(int freq);
/* decode.c */
char Get_Character(void);
void Adapt_Decoder(void);
char Hex_to_Ascii(int *hex_code);
/* detect.c */
int Get_Fragment(void);
gboolean Initialize_Detector( void );
/* display.c */
void Display_Detector(int plot);
void Display_Signal(int plot);
void Display_Waterfall(void);
void Clear_Pixbuf(void);
/* fft.c */
void Ifft_init(int fft_size);
void Ifft(int fft_size);
gboolean mem_alloc(void **ptr, int req);
/* main.c */
int main(int argc, char *argv[]);
gboolean Print_Character(gpointer data);
int Load_Line(char *buff, FILE *pfile, char *messg);
gboolean Load_Config(gpointer data);
void Usage(void);
int isFlagSet(int flag);
int isFlagClear(int flag);
void Set_Flag(int flag);
void Clear_Flag(int flag);
void Toggle_Flag(int flag);
void Cleanup(void);
/* sound.c */
gboolean Setup_Sound_Card(void);
int Signal_Sample(void);

#endif /* End of xdemorse.h */

