/* Chromium Unity integration extension
 * 
 *   Copyright 2012 Canonical Ltd.
 *
 *   This program is free software: you can redistribute it and/or modify it 
 *   under the terms of the GNU General Public License version 3, as published 
 *   by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but 
 *   WITHOUT ANY WARRANTY; without even the implied warranties of 
 *   MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 *   PURPOSE.  See the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along 
 *   with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

var background_page = (function () {

   // TODO extract based on logging settings
   var log = {
     info: function (msg) {
       if (msg) {
	 console.log ('Info: ' + msg);
       }
     }
     ,
     error: function (msg) {
       if (msg) {
	 console.log ('Error: ' + msg);
       }
     }
   };

   // connection & init w/ plugin NPAPI layer
   var plugin = document.getElementById ('unityChromiumExtensionId');
   if ( ! plugin) {
     log.error ("Unable to retrieve the Unity Chromium plugin");
     return null;
   }
   var service = plugin.service_new ();
   if (! service) {
     log.error ("Unable to retrieve a connection to Unity Webapps service");
     return null;
   }


   /////////////////////////////////////////////////////////
   // 
   // Handle login & connection to online-accounts project's
   // extension
   // 
   ////////////////////////////////////////////////////////
   var loginListeners = [];
   var onLoginListenerRequest = function (request, sender, sendResponse) {
     if (request.action == "registerLoginListener") {
       log.info ("Login listener request from " + sender.id);
       loginListeners.push (sender.id);
     }
   };
   chrome.runtime.onMessageExternal.addListener (onLoginListenerRequest);

   var onLoginEvent = function (request, sender, sendResponse) {
     if (request.action == "loginEvent") {
       log.info ("Login on " + request.domain + ": " + request.login);
       var len = loginListeners.length;
       for (var i = 0; i < len; i++) {
         var extensionId = loginListeners[i];
         log.info("Notifying login listener: " + extensionId);
         chrome.runtime.sendMessage (extensionId, {
                                         type: "login",
                                         login: request.login,
                                         domain: request.domain
                                       });
       }
     }
   };
   chrome.runtime.onMessage.addListener (onLoginEvent);


   /////////////////////////////////////////////////////////
   // 
   // Scafolding to keep track of data associated w/ infobar requests 
   // (Chromium's structure imposes some kind of state being maintained
   // in order to communicate data) 
   // 
   ////////////////////////////////////////////////////////
   // 
   // list of callback that are to be called asynchronously
   //  per tab. Used in the user integration/installation resquests context.
   // 
   // One thing to keep in mind is that one constraint, that bring some amount of
   //  'soundness' is that there is a hard limit (provided by the browser) of one infobar per tab.

   var user_infobar_request_callbacks = {};
   var addInfobarRequestCallbackFor = function (infobarRequestId, callback, message, details) {
     user_infobar_request_callbacks[infobarRequestId] = {callback: callback
						         , message: message
                                                         , details: details
						        };
   };
   var getDataIfAnyFor = function (infobarRequestId) {
     if (user_infobar_request_callbacks[infobarRequestId] === undefined) {
       return "";
     }
     return { message: user_infobar_request_callbacks[infobarRequestId].message,
              details: user_infobar_request_callbacks[infobarRequestId].details };
   };
   var invokeAndRemoveCallbackIfAnyFor = function (infobarRequestId, arguments) {
     if (user_infobar_request_callbacks[infobarRequestId] === undefined) {
       return;
     }
     var callback = user_infobar_request_callbacks[infobarRequestId].callback;
     user_infobar_request_callbacks[infobarRequestId] = undefined;
     if (typeof(callback) === 'function') {
       callback(arguments);
     }
   };


   /////////////////////////////////////////////////////////
   // 
   // Extract installed userscripts & webapps info
   // 
   ////////////////////////////////////////////////////////
   var initializeIntegrationScriptRepositoryAccess = function (plugin) {

     var repo = plugin.application_repository_new_default ();
     plugin.application_repository_prepare (repo);
     return repo;
   };
   var repo_ = initializeIntegrationScriptRepositoryAccess(plugin);
   var repo_install_request_stamps = {};

   /**
    * Performs a match on the list of loaded integration scripts
    * given a url.
    *
    */
   var matchesIntegrationScripts = function (plugin, url, repo, windowInfos, callback) {
     function askForApplicationInstallation(plugin, appPackageName, appDomain, url, installationCallback) {

       if (plugin.permissions_get_domain_dontask(appDomain)) {
         log.info ("WebApp domain was found in the 'dont ask' list, won't install: " + appPackageName);
         installationCallback ();
         return;
       }
       if (repo_install_request_stamps[appPackageName] !== undefined) {
         log.info ('WebApp not being considered for install (request is only issues once per session): ' + appPackageName);
         installationCallback ();
         return;
       }
       repo_install_request_stamps[appPackageName] = true;

       function installApp() {
         log.info ('Installing application: ' + appPackageName);
         plugin.permissions_allow_domain(appDomain);
         plugin.application_repository_install_application(repo, appPackageName, installationCallback, null);
       }
       function addToIgnoreList() {
         plugin.permissions_dontask_domain(appDomain);
       }
       var isInstallationConfirmed = function (response) {
	  return response && response.integrate;
       };

       if (!plugin.permissions_get_domain_preauthorized(appDomain)) {
         var appName = plugin.application_repository_get_resolved_application_name(repo, appPackageName);
         addInfobarRequestCallbackFor (windowInfos.tabId,
                                       function (result) {
                                         log.info ('Asking for installation : ' + windowInfos.tabId);
                                         if (result === undefined) {
                                           installationCallback();
                                         }
                                         if (isInstallationConfirmed(result))
                                           installApp();
                                         else
                                           addToIgnoreList();
                                       },
	                               chrome.i18n.getMessage("integration_copy", [appName, appDomain]),
                                       null);
         chrome.infobars.show ({tabId: windowInfos.tabId, path: "infobar.html"});
       }
       else {
         plugin.permissions_allow_domain(appDomain);
         installApp();
       }

     } // function askForApplicationInstallation

     var APPLICATION_STATUS_AVAILABLE = 0;
     var APPLICATION_STATUS_INSTALLED = 1;
     var APPLICATION_STATUS_UNRESOLVED = 2;

     var formatMatchedWebAppInfo = function (packageName, appName, appDomain, src) {
       return {
         packageName: packageName,
         appName: appName,
         appDomain: appDomain,
         content: src,
         requires: null,
         includes: null
       };
     };

     /**
      * Gathers the lists of installed and (if any) available apps for a given list of app names
      * (available for a given url).
      * 
      * @returns object with "installed" and "available" list properties
      */
     var gatherApplicationsStatuses = function (packageNames) {
       var installed = [];
       var available = [];
       for (var i = 0; i < packageNames.length; ++i) {
         var packageName = packageNames[i];

         var status = plugin.application_repository_get_resolved_application_status (repo, packageName);
         var appName = plugin.application_repository_get_resolved_application_name (repo, packageName);
         var appDomain = plugin.application_repository_get_resolved_application_domain (repo, packageName);

         log.info ('A WebApp application has been found: ' + appName + ', status: ' + status);

         switch (status) {
           case APPLICATION_STATUS_INSTALLED:
             // we have found an installed application, use it directly
             var src = plugin.application_repository_get_userscript_contents (repo, packageName);
             installed.push (formatMatchedWebAppInfo(packageName, appName, appDomain, src));
             break;
           case APPLICATION_STATUS_AVAILABLE:
             // we have found an application that can apply and be installed
             available.push (formatMatchedWebAppInfo(packageName, appName, appDomain));
             break;
         }
       }
       return {
         installed: installed,
         available: available
       };
     };

     var packageNames = JSON.parse(plugin.application_repository_resolve_url_as_json(repo_, url));
     if (null != packageNames) {
       var appStatuses = gatherApplicationsStatuses(packageNames);

       if (appStatuses && appStatuses.installed && appStatuses.installed.length > 0) {
         var app = appStatuses.installed[0];
         if (repo_install_request_stamps[app.packageName] == undefined
             && ! plugin.permissions_get_domain_dontask(app.appDomain)
             && ! plugin.permissions_get_domain_allowed(app.appDomain)) {
           // requesting for install
           addInfobarRequestCallbackFor (windowInfos.tabId,
                                         function (result) {
                                           if (result && result.integrate) {
                                             plugin.application_repository_add_desktop_to_launcher(app.appName, app.appDomain);
                                             plugin.permissions_allow_domain(app.appDomain);
                                           }
                                           else {
                                             plugin.permissions_dontask_domain(app.appDomain);
                                           }
                                         },
	                                 chrome.i18n.getMessage("integration_copy", [app.appName, app.appDomain]),
                                         null);
           chrome.infobars.show ({tabId: windowInfos.tabId, path: "infobar.html"});
         }
         return;
       }

       // we should have at most one script for a given app
       if (appStatuses && appStatuses.available && appStatuses.available.length > 0) {
         var app = appStatuses.available[0];
         askForApplicationInstallation (plugin,
                                        app.packageName,
                                        app.appDomain,
                                        url,
                                        function (_repository, name, status, data) {
                                          log.info ('Application installed: ' + name + ', (status ' + status + ')');

                                          if (name !== undefined && status == APPLICATION_STATUS_INSTALLED) {
                                            callback (true, app.packageName, app.appName, app.appDomain);
                                          }
                                        }
                                       );
       }
       else
         callback (false);
     } // if (null != names) {

   }; // var matchesIntegrationScripts = function (plugin, url, repo, windowInfos, callback) {
   
   
   ////////////////////////////////////////
   // 
   // main request handler
   // 
   ///////////////////////////////////////
   
   /**
    * Handles & responds to content script requests.
    * 
    */
   var init_requested_stamps = {};
   var contentScriptsRequestHandler = function (request, sender, callback) {
     
     var handlers = {
       get_extension_settings: function (request, sender, callback) {
	 
	 var settings = {
	   logging: false,
           incognito: sender.tab.incognito
	 };
	 
	 try {
	   if (window.localStorage) {
	     settings.logging = localStorage['logging'];
	   }
	 }
	 catch (e) {
	   log.error ("Error while trying to retrieve logging information: " + str(e));
	 }
	 
	 callback (settings);
       }
       ,
       on_user_infobar_request_result: function (request, sender, callback) {
	 
	 invokeAndRemoveCallbackIfAnyFor (request.tabId, request);
       }
       ,
       init_requested: function (request, sender, callback) {
         var name = request.options.name;
         var domain = request.options.domain;
         var iconUrl = request.options.iconUrl + '';
         var homepage = request.options.homepage || request.options.url;

         if (! domain || ! name || typeof(domain) !== 'string' || typeof(name) !== 'string')
           return;

         var islocal = request.options.protocol && request.options.protocol === 'file:';
         var host = request.options.hostname || request.options.host;
         if ( ! islocal && (! host || typeof(host) !== 'string'))
           return;

         if (-1 === homepage.indexOf(domain) || -1 === host.indexOf(domain)) {
           log.error('Invalid call to init(), homepage, domain and host are not consistent');
           return;
         }
         log.info('init_requested() called for ' + name + ' domain: ' + domain);

         if (plugin.permissions_get_domain_dontask(domain) || plugin.permissions_get_domain_allowed(domain)) {
           log.info('Domain already listed in allow or dont ask lists');
           return;
         }

         var stampname = name + domain;
         if (init_requested_stamps[stampname] == undefined) {
           addInfobarRequestCallbackFor (sender.tab.id,
                                         function (result) {
                                           if (result && result.integrate) {
                                             plugin.application_repository_generate_desktop_for_url_launch(name, domain, homepage, iconUrl);
                                             plugin.application_repository_add_desktop_to_launcher(name, domain);
                                             plugin.permissions_allow_domain(domain);
                                           }
                                           else {
                                             plugin.permissions_dontask_domain(domain);
                                           }
                                         },
	                                 chrome.i18n.getMessage("integration_copy", [name, domain]),
                                         null);
           init_requested_stamps[stampname] = true;
           chrome.infobars.show ({tabId: sender.tab.id, path: "infobar.html"});
         }
       }
     }; // handlers
     
     
     // validate request
     if ( ! request  || ! request.method) {
       callback ({error: "Invalid request structure"});
       return true;
     }
     if ( ! sender) {
       callback ({error: "Invalid sender"});
       return true;
     }
     if ( typeof (request.method) != 'string' || request.method.length == 0) {
       callback ({error: "Invalid request method"});
       return true;
     }

     log.info ('Got request: ' + request.method);

     var handler = handlers [request.method];
     
     if (handler !== undefined && typeof(handler) == 'function') {
       
       log.info ('Calling handler for: ' + request.method);
       
       handler (request, sender, callback);
       
       return true;
     }
     return false;
   } // var contentScriptsRequestHandler =

   // Main event handler and communication link
   //   w/ content scripts
   chrome.runtime.onMessage.addListener (contentScriptsRequestHandler);


   ///////////////////////////////////////////////////////////////////////
   // 
   // Window management related functions. In chromeless mode, we have specific
   //  rules for tab management to make webapps feel more "native" than plain
   //  web applications.
   // 
   ///////////////////////////////////////////////////////////////////////

  var onTabChanged = function (tabId, windowId, url) {
    var onInstalled = function (installed, packageName, appName, appDomain) { };
    matchesIntegrationScripts (plugin,
                               url,
                               repo_,
                               {
                                 windowId: windowId,
                                 tabId: tabId
                               },
                               onInstalled
                              );
   };
   chrome.tabs.onCreated.addListener(
     function(tab) {
       if (tab && tab.url) {
         onTabChanged(tab.id, tab.windowId, tab.url);
       } // if (tab && tab.url) {
     }
   );
   chrome.tabs.onUpdated.addListener (
     function(tabId, changeInfo, tab) {
       if (changeInfo && changeInfo.url) {
         onTabChanged(tabId, tab.windowId, changeInfo.url);
       }
     }
   );

   return {
     /*
      *  Returns a potential message associated with a tab id (infobar)
      */
     getMessageForTabId: function (tabId)  {
       return getDataIfAnyFor (tabId).message;
     }
   };
 }) ();

