/*
 * Copyright 2023 KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QFileInfo>
#include <QString>
#include <QTextStream>
#include <QDebug>

#include "devicepowerinfo.h"

DevicePowerInfo::DevicePowerInfo(const QString &devType,
                                 const QString &devPath,
                                 const QString &devName,
                                 const QString &controlFile,
                                 QObject *parent) :
    QObject(parent), m_devType(devType), m_devPath(devPath), m_devName(devName)
{
    m_powerControlFilePath = devPath + controlFile;

    QFile file(m_powerControlFilePath);
    if (file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QTextStream textStream(&file);
        m_defaultPowerValue = textStream.readLine();
        m_currentPowerValue = m_defaultPowerValue;
        file.close();
    }
}

QString DevicePowerInfo::getDevicePath()
{
    return m_devPath;
}

QString DevicePowerInfo::getDeviceName()
{
    return m_devName;
}

QString DevicePowerInfo::getCurrentPowerStat()
{
    return m_currentPowerValue;
}

QString DevicePowerInfo::getDefaultPowerStat()
{
    return m_defaultPowerValue;
}

bool DevicePowerInfo::ignoreDeviceList()
{
    if (m_devType == "pci") {
        if (m_devName.contains("Ethernet", Qt::CaseInsensitive)) {
            return true;
        }
        if (m_devName.contains("usb", Qt::CaseInsensitive)) {
            return true;
        }
    }
    return false;
}

bool DevicePowerInfo::setDevicePowerStat(const QString &stat)
{
    bool ret = false;

    if (true == ignoreDeviceList()) {
        return false;
    }

    QFile controlFile(m_powerControlFilePath);

    if (!controlFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return false;
    }
    QTextStream in(&controlFile);

    if (stat == "default") {
        in << m_defaultPowerValue << endl;
        m_currentPowerValue = m_defaultPowerValue;
        ret = true;
    } else {
        if (m_devType == "sata") {
            if (stat == "min_power" || stat == "max_performance") {
                in << stat << endl;
                m_currentPowerValue = stat;
                ret = true;
            } else {
                ret = false;
            }
        } else {
            if (stat == "auto" || stat == "on") {
                in << stat << endl;
                m_currentPowerValue = stat;
                ret = true;
            } else {
                ret = false;
            }
        }
    }
    controlFile.close();
    return ret;
}

