use test;
use markdown;
use lang:bs:macro;

class CountVisitor extends Visitor {
	Nat count;
	core:lang:Type type;

	init(core:lang:Type t) {
		init { type = t; }
	}

	ElementSeq visit(ElementSeq seq) {
		if (core:lang:typeOf(seq) is type)
			count++;
		seq;
	}

	Element visit(Element element) {
		if (core:lang:typeOf(element) is type)
			count++;
		element;
	}

	FormattedText visit(FormattedText text) {
		if (core:lang:typeOf(text) is type)
			count++;
		text;
	}

	TextSpan visit(TextSpan span) {
		if (core:lang:typeOf(span) is type)
			count++;
		span;
	}

}

Nat countType(Document d, core:lang:Type t) {
	CountVisitor x(t);
	d.visit(x);
	x.count;
}

suite BasicMarkdown {
	var simple = str {
		this is a
		paragraph

		this is the
		next *paragraph*
		with four
		lines

		single line

		this is a heading
		=================

		### deep heading

		this is another paragraph
	};

	Document result = parse(simple);

	test result.elements.elements.count == 6;
	test result.countType(named{Paragraph}) == 4;
	test result.countType(named{Heading}) == 2;
	// print("Result:");
	// print(result.toS);
	// print("\nHTML:");
	// print(result.toHtml.toS);
}

suite MarkdownLists {
	var nested = str {
		- First
		  - Second
		  - Third
		- Fourth
	};
	{
		Document result = parse(nested);
		test result.countType(named{List}) == 2;

		// print("Result:");
		// print(result.toHtml.toS);
	}

	var lists = str {
		Heading
		=======

		This is a short list:
		- First element
		end of list

		This is a list:

		- First element
		- Second element
		  with additional text
		- Third element
		  - Nested list

		  - Second element
		    with additional text

		- Fourth element

		1. Number first
		2. number second
	};

	{
		Document result = parse(lists);
		test result.countType(named{List}) == 4;

		// print("Result:");
		// print(result.toHtml.toS);
	}
}

suite MarkdownCode {
	var code = str {
		Paragraph

		```c++
		inline code in C++
		has two lines
		```

		another with `inline code`!

		- In a list:

		  ```java
		  test code
		  second line
		  ```
		- Next element
	};

	Document result = parse(code);
	test result.countType(named{CodeBlock}) == 2;
	test result.countType(named{Paragraph}) == 4;
	test result.countType(named{InlineCode}) == 1;
	test result.countType(named{List}) == 1;

	// print(result.toHtml.toS);
}

suite MarkdownCodeNoSpace {
	var code = str {
		Paragraph
		```c++
		inline code
		```
		END
	};

	Document result = parse(code);
	test result.countType(named{CodeBlock}) == 1;
	test result.countType(named{Paragraph}) == 2;
}

suite MarkdownMultilineMarkup {
	var code = str {
		test *multiline
		bold* text and `multiline
		code` in paragraphs

		- list item

		  test [multiline
		  link text](linktarget)
		  in the paragraph and `multiline
		  code` in a list paragraph
	};

	Document result = parse(code);
	test result.countType(named{Paragraph}) == 3;
	test result.countType(named{ItalicText}) == 1;
	test result.countType(named{InlineCode}) == 2;
	test result.countType(named{Link}) == 1;
}
