use lang;
use lang:bs;
use lang:bs:macro;
use lang:bnf;
use core:lang;
use core:io;
use core:debug;
use core:asm;

// Package reader for BF files.

PkgReader reader(Url[] files, Package pkg) on Compiler {
	FilePkgReader(files, pkg, &createFile(FileInfo));
}

FileReader createFile(FileInfo info) on Compiler {
	BfFile(info);
}

class BfFile extends FileReader {
	init(FileInfo info) {
		init(info) {}
	}

	void readFunctions() {
		// TODO: We do not want to read the file twice.
		BfFunction fn(info.url.title, info.url);
		info.pkg.add(fn);
	}

	Rule rootRule() {
		named{SFile};
	}
}

class BfFunction extends Function {
	Url file;

	init(Str name, Url file) {
		Value strVal = named{Str};
		init(strVal, name, [strVal]) {
			file = file;
		}

		setCode(LazyCode(&this.load()));
	}

	init(Url file) {
		Value strVal = rootScope.value(name{core:Str});
		init(strVal, file.title, [strVal]) {
			file = file;
		}
	}

	CodeGen load() {
		Parser<SFile> parser;
		parser.parse(file.read.readText.readAll, file);
		if (parser.hasError) {
			parser.throwError;
		}

		BfExpr o = parser.tree.transform();
		Value strType = named{Str};

		CodeGen r(RunOn(), false, strType);
		Listing l = r.l;
	    var input = l.createParam(strType.desc);

		l << prolog();

		CodeResult result(strType, r.block);

		Expr oInput = o.input;
		if (oInput as BareVarAccess) {
			oInput.var = input;
		}

		o.code(r, result);

		l << fnRet(result.location(r));

		r;
	}
}

// Brainfuck expression
class BfExpr extends lang:bs:Expr {
	Nat space;
	BfToken[] tokens;
	Expr input;

	init(SrcPos pos) {
		init(pos) {
			input(pos);
		}
	}

	init(SrcPos pos, Str space, BfToken[] data) {
		init(pos) {
			space = space.toNat;
			tokens = data;
			input = BareVarAccess(pos, Value(named{Str}), core:asm:Var());
		}
	}

	init(SrcPos pos, Str space, BfToken[] data, lang:bs:Expr input) {
		init(pos) {
			space = space.toNat;
			tokens = data;
			input = input;
		}
	}

	ExprResult result() {
		Value(named{Str});
	}

	void code(CodeGen g, CodeResult to) {
		// Generate some code!
		BfState state(g, space, input);

		// Initialize some variables...
		state.createOutput(pos);

		// Generate code!
		for (Nat i = 0; i < tokens.count; i++) {
			tokens[i].code(state, g);
		}

		// Generate the result.
		state.createResult(pos, to);
	}

	Str toS() {
		StrBuf o;
		o.add("bf(" # input # ", " # space # ") {");
		for (Nat i = 0; i < tokens.count; i++) {
			o.add(tokens[i].toS);
		}
		o.add("}");
		o.toS;
	}
}

// State passed around.
class BfState on Compiler {
	CodeGen g;

	// Pointer to a StrBuf.
    core:asm:Var output;

	// The data pointer is stored in ptrA, and has to be pushed/popped for each
	// function call.

	// Temporary variable (int sized).
	core:asm:Var tmpData;

	// Temporary storage of the data pointer while performing IO.
	core:asm:Var tmpPtr;

	// Data area. A large variable.
	core:asm:Var data;

	// Input expression and variable the TextInput is stored in.
	Expr inputExpr;
	core:asm:Var input;

	// StrBuf type. Needed a couple of times...
	Value strBuf;

	// TextInput type.
	Value textInput;

	// Nat type.
	Value natType;

	init(CodeGen g, Nat size, Expr inputExpr) {
		init() {
			g = g;
			strBuf = named{StrBuf};
			textInput = named{core:io:TextInput};
			natType = named{Nat};
			inputExpr = inputExpr;
		}

		output = g.createVar(strBuf).v;
		input = g.createVar(textInput).v;
		tmpData = g.l.createVar(g.block, sInt());
		tmpPtr = g.l.createVar(g.block, sPtr());
		data = g.l.createVar(g.block, Size(size));
	}

	// Create output. (TODO: Make use of createObject in CodeGen.h!)
	void createOutput(SrcPos pos) {
		Block block(pos, rootScope);
		Expr ctor = namedExpr(block, name{core:StrBuf}, Actuals());
		ctor.code(g, CodeResult(strBuf, VarInfo(output)));

		Expr create = namedExpr(block, name{core:io:readStr}, Actuals(inputExpr));
		create.code(g, CodeResult(textInput, VarInfo(input)));

		// Initialize our pointer as well.
		g.l << lea(ptrA, ptrRel(data, Offset()));
	}

	// Create the result.
	void createResult(SrcPos pos, CodeResult to) {
		Block block(pos, Scope(named{StrBuf}));
		BareVarAccess outAccess(pos, strBuf, output);
		Expr e = namedExpr(block, SStr("toS", pos), outAccess, Actuals());
		e.code(g, to);
	}
}

class BfToken on Compiler {
	SrcPos pos;

	init(SrcPos pos) {
		init { pos = pos; }
	}

	void code(BfState state, CodeGen to) {}
}

class BfBack extends BfToken {
	init(SrcPos pos) { init(pos); }
	void code(BfState state, CodeGen to) {
		to.l << sub(ptrA, ptrConst(Offset(1)));
	}
	Str toS() { "<"; }
}

class BfFwd extends BfToken {
	init(SrcPos pos) { init(pos); }
	void code(BfState state, CodeGen to) {
		to.l << add(ptrA, ptrConst(Offset(1)));
	}
	Str toS() { ">"; }
}

class BfInput extends BfToken {
	init(SrcPos pos) { init(pos); }
	void code(BfState state, CodeGen to) {
		to.l << mov(state.tmpPtr, ptrA);

		Block block(pos, Scope(named{core:io:TextInput}));
		BareVarAccess inAccess(pos, state.textInput, state.input);
		Expr e = namedExpr(block, SStr("read", pos), inAccess, Actuals());
		e.code(to, CodeResult(state.natType, VarInfo(state.tmpData)));

		to.l << lea(ptrC, state.tmpData);
		to.l << mov(ptrA, state.tmpPtr);
		to.l << mov(byteRel(ptrA, Offset()), byteRel(ptrC, Offset()));
	}
	Str toS() { ","; }
}

class BfOutput extends BfToken {
	init(SrcPos pos) { init(pos); }
	void code(BfState state, CodeGen to) {
		to.l << mov(state.tmpPtr, ptrA);
		to.l << lea(ptrC, state.tmpData);
		to.l << mov(byteRel(ptrC, Offset()), byteRel(ptrA, Offset()));

		Block block(pos, Scope(named{StrBuf}));
		BareVarAccess outAccess(pos, state.strBuf, state.output);
		BareVarAccess tmpData(pos, state.natType, state.tmpData);
		Expr ch = CtorCall(SrcPos(), rootScope, named{Char:__init<Char, Nat>}, Actuals(tmpData));
		Expr e = namedExpr(block, SStr("add", pos), outAccess, Actuals(ch));
		e.code(to, CodeResult());

		to.l << mov(ptrA, state.tmpPtr);
	}
	Str toS() { "."; }
}

class BfInc extends BfToken {
	init(SrcPos pos) { init(pos); }
	void code(BfState state, CodeGen to) {
		to.l << add(byteRel(ptrA, Offset()), byteConst(1.byte));
	}
	Str toS() { "+"; }
}

class BfDec extends BfToken {
	init(SrcPos pos) { init(pos); }
	void code(BfState state, CodeGen to) {
		to.l << sub(byteRel(ptrA, Offset()), byteConst(1.byte));
	}
	Str toS() { "-"; }
}

class BfLoop extends BfToken {
	BfToken[] tokens;

	init(SrcPos pos, BfToken[] tokens) {
		init(pos) {
			tokens = tokens;
		}
	}

	void code(BfState state, CodeGen to) {
		Label begin = to.l.label();
		Label end = to.l.label();

		to.l << begin;
		to.l << cmp(byteRel(ptrA, Offset()), byteConst(0.byte));
		to.l << jmp(end, CondFlag:ifEqual);

		// Inner code...
		for (Nat i = 0; i < tokens.count; i++) {
			tokens[i].code(state, to);
		}

		to.l << jmp(begin);
		to.l << end;
	}

	Str toS() {
		StrBuf t;
		t.add("[");
		for (Nat i = 0; i < tokens.count; i++) {
			t.add(tokens[i].toS);
		}
		t.add("]");
		t.toS;
	}
}
