/* -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of signon
 *
 * Copyright (C) 2011 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "keyring-test.h"
#include "mocked-gnome-keyring.h"
#include "secrets-storage.h"

#include <QDebug>
#include <QTest>

#define TEST_CREDENTIALS_ID 0x7fffffff

void KeyringTest::initTestCase()
{
    qDebug() << Q_FUNC_INFO;

    mocked_gnome_keyring_init();

    plugin = new KeyringPlugin;
    QVERIFY(plugin != 0);

    storage = plugin->secretsStorage(this);
    QVERIFY(storage != 0);
}

void KeyringTest::init()
{
    /* Before running each test, ensure that the storage is open */
    if (!storage->isOpen()) {
        QVariantMap conf;
        bool ok;

        /* execute tests in a different keyring, not to destroy the developer's
         * default keyring :-) */
        conf.insert(QLatin1String("KeyringName"), QLatin1String(TEST_KEYRING));
        ok = storage->initialize(conf);
        QVERIFY(ok);
    }
}

void KeyringTest::clear()
{
    bool ok;

    /* insert a few credentials */
    QStringList usernames, passwords;
    usernames << "first user" << "second user" << "third user";
    passwords << "pass 1" << "pass two" << "3rd pass";

    for (int i = 0; i < usernames.count(); i++) {
        ok = storage->updateCredentials(i, usernames[i], passwords[i]);
        QVERIFY(ok);
    }

    /* check that credentials are there */
    for (int i = 0; i < usernames.count(); i++) {
        QString username, password;
        ok = storage->loadCredentials(i, username, password);
        QVERIFY(ok);
        QCOMPARE(username, usernames[i]);
        QCOMPARE(password, passwords[i]);
    }

    /* clear the DB */
    ok = storage->clear();
    QVERIFY(ok);

    /* check the credentials are not there anymore */
    for (int i = 0; i < usernames.count(); i++) {
        QString username, password;
        ok = storage->loadCredentials(i, username, password);
        QVERIFY(!ok);
    }
}


void KeyringTest::dataStorage()
{
    const int id = TEST_CREDENTIALS_ID;
    const int method1 = 53, method2 = 7;

    QVariantMap data1;
    data1.insert(QLatin1String("key"), QLatin1String("Value"));

    QVariantMap data2;
    data2.insert(QLatin1String("name"),
                 QString::fromUtf8("\u00C4\u026B\u0253\u1EB9\u0159\u01AB\u2126"));
    QVariantMap innerDictionary;
    innerDictionary.insert(QLatin1String("value"), 300);
    QStringList names = QStringList() << "Tom" << "Dick" << "Harry";
    innerDictionary.insert(QLatin1String("names"), names);
    data2.insert(QLatin1String("dictionary"), innerDictionary);

    bool ok;

    ok = storage->storeData(id, method1, data1);
    QVERIFY(ok);

    ok = storage->storeData(id, method2, data2);
    QVERIFY(ok);

    QVariantMap loaded = storage->loadData(id, method1);
    QCOMPARE(loaded, data1);

    loaded = storage->loadData(id, method2);
    QCOMPARE(loaded, data2);

    /* delete the data for method1 */
    ok = storage->removeData(id, method1);
    QVERIFY(ok);

    /* check that it has been deleted, and that the data for method2 is still
     * there */
    loaded = storage->loadData(id, method1);
    QCOMPARE(loaded, QVariantMap());

    loaded = storage->loadData(id, method2);
    QCOMPARE(loaded, data2);
}

void KeyringTest::defaultKeyring()
{
    QVariantMap conf;
    bool ok;

    /* close the test keyring, and open the default one */
    ok = storage->close();
    QVERIFY(ok);
    ok = storage->initialize(conf);
    QVERIFY(ok);

    /* test just the basic operations on one identity only */
    QString testUsername = QLatin1String("Test username");
    QString testPassword = QLatin1String("Test p4ssw0rd");
    ok = storage->updateCredentials(TEST_CREDENTIALS_ID,
                                    testUsername,
                                    testPassword);
    QVERIFY(ok);

    QString username, password;
    ok = storage->loadCredentials(TEST_CREDENTIALS_ID, username, password);
    QVERIFY(ok);
    QCOMPARE(username, testUsername);
    QCOMPARE(password, testPassword);

    ok = storage->removeCredentials(TEST_CREDENTIALS_ID);
    QVERIFY(ok);

    ok = storage->close();
    QVERIFY(ok);
}

void KeyringTest::cleanupTestCase()
{
    delete storage;
    delete plugin;
}

QTEST_MAIN(KeyringTest)

