# -*- coding: utf-8 -*- #
# frozen_string_literal: true

require 'open-uri'
require 'json'

GHERKIN_SYNTAX_URI = "https://raw.githubusercontent.com/cucumber/cucumber/master/gherkin/gherkin-languages.json"
GHERKIN_KEYWORDS_FILE = "./lib/rouge/lexers/gherkin/keywords.rb"

namespace :builtins do
  task :gherkin do
    generator = Rouge::Tasks::Builtins::Gherkin.new

    input = JSON.load(URI.open(GHERKIN_SYNTAX_URI))
    keywords = generator.extract_keywords(input)

    output = generator.render_output(keywords)

    File.write(GHERKIN_KEYWORDS_FILE, output)
  end
end

module Rouge
  module Tasks
    module Builtins
      class Gherkin
        def extract_keywords(input)
          keywords = Hash.new { |h,k| h[k] = Set.new }

          input.values.each do |lang|
            lang.each do |k,v|
              case k
              when "feature"
                keywords[:feature].merge v
              when "background", "scenario", "scenarioOutline"
                keywords[:element].merge v
              when "examples"
                keywords[:examples].merge v
              when "given", "when", "then", "and", "but"
                keywords[:step].merge v
              end
            end
          end

          keywords
        end

        def render_output(keywords, &b)
          return enum_for(:render_output, keywords).to_a.join("\n") unless b

          yield   "# -*- coding: utf-8 -*- #"
          yield   "# frozen_string_literal: true"
          yield   ""
          yield   "# DO NOT EDIT"
          yield   "# This file is automatically generated by `rake builtins:gherkin`."
          yield   "# See tasks/builtins/gherkin.rake for more info."
          yield   ""
          yield   "module Rouge"
          yield   "  module Lexers"
          yield   "    def Gherkin.keywords"
          yield   "      @keywords ||= {}.tap do |k|"
          keywords.each do |t, kws|
            yield "        k[#{t.inspect}] = Set.new #{kws.to_a.sort.inspect}"
          end
          yield   "      end"
          yield   "    end"
          yield   "  end"
          yield   "end"
        end
      end
    end
  end
end
