/*
*  RAL -- Rubrica Addressbook Library
*  file: csv.c
*
*  the vcard import engine v0.5 
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "csv.h"
#include "../../error.h"
#include "../../plugin.h"
#include "../../abook.h"
#include "../../card.h"
#include "../../personal.h"
#include "../../company.h"
#include "../../contact.h"
#include "../../work.h"
#include "../../notes.h"
#include "../../address.h"
#include "../../net.h"
#include "../../telephone.h"
#include "../../lookup.h"
// #include "../../ref.h"


#define COMMA_DELIM ','
#define TAB_DELIM   '\t'
#define QUOTES      '"'

/* Thunderbird (Mozilla/SeaMonkey/Netscape)'s csv fields:
   First name, Last name, Display name, Nickname
   Email1, Email2, 
   Work Tel, Home Tel, Fax, Pager, Mobile,
   Home Address, Home Address2, City, Province, Zip, State,
   Work Address, Work Address2, City, Province, Zip, State,
   Title, Department, Organization, 
   Work web, Personal Web,
   Undef1, Undef2, Undef3, Undef4,
   UserDefined1,  UserDefined2, UserDefined3, UserDefined4,
   Notes, Other
 */

typedef enum {
  CSV_FIRST_NAME,               // 1
  CSV_LAST_NAME,                // 2
  CSV_DISPLAY_NAME,             // 3
  CSV_NICKNAME,                 // 4
  CSV_EMAIL1,                   // 5
  CSV_EMAIL2,                   // 6
  CSV_TEL_WORK,                 // 7
  CSV_TEL_HOME,                 // 8
  CSV_TEL_FAX,                  // 9
  CSV_TEL_PAGER,                // 10
  CSV_TEL_MOBILE,               // 11
  CSV_HOME_STREET,              // 12
  CSV_HOME_STREET_MORE,         // 13
  CSV_HOME_CITY,                // 14
  CSV_HOME_STATE_PROVINCE,      // 15
  CSV_HOME_ZIP,                 // 16
  CSV_HOME_COUNTRY,             // 17
  CSV_WORK_STREET,              // 18
  CSV_WORK_STREET_MORE,         // 19
  CSV_WORK_CITY,                // 20
  CSV_WORK_STATE_PROVINCE,      // 21
  CSV_WORK_ZIP,                 // 22
  CSV_WORK_COUNTRY,             // 23
  CSV_TITLE,                    // 24
  CSV_DEPARTMENT,               // 25
  CSV_ORGANIZATION,             // 26
  CSV_WORK_WEB,                 // 27
  CSV_WEB,                      // 28
  CSV_UNDEF1,                   // 29
  CSV_UNDEF2,                   // 30
  CSV_UNDEF3,                   // 31
  CSV_CUSTOM1,                  // 32
  CSV_CUSTOM2,                  // 33
  CSV_CUSTOM3,                  // 34
  CSV_CUSTOM4,                  // 35
  CSV_NOTE,                     // 36
  CSV_MORE,                     // 37
} RCsvField;


RLookupTable look [] ={
  {"first name",    N_("first name"), CSV_FIRST_NAME },
  {"last  name",    N_("last  name"), CSV_LAST_NAME},                
  {"display name",  N_("display name"), CSV_DISPLAY_NAME},             
  {"nickname",      N_("nickname"), CSV_NICKNAME },                 
  {"email 1",       N_("email 1"), CSV_EMAIL1 },                   
  {"email 2",       N_("email 2"), CSV_EMAIL2 },                   
  {"tel work",      N_("tel work"), CSV_TEL_WORK },                 
  {"tel home",      N_("tel home"), CSV_TEL_HOME },                
  {"tel fax",       N_("tel fax"), CSV_TEL_FAX },                
  {"tel pager",     N_("tel pager"), CSV_TEL_PAGER },              
  {"tel mobile",    N_("tel mobile"), CSV_TEL_MOBILE },             
  {"home street",   N_("home street"), CSV_HOME_STREET },            
  {"home street 2", N_("home street 2"), CSV_HOME_STREET_MORE },       
  {"home city",     N_("home city"), CSV_HOME_CITY },              
  {"home state",    N_("home state"), CSV_HOME_STATE_PROVINCE },    
  {"home zip",      N_("home zip"), CSV_HOME_ZIP },               
  {"home country",  N_("home country"), CSV_HOME_COUNTRY },           
  {"work street",   N_("work street"), CSV_WORK_STREET },            
  {"work street 2", N_("work street 2"), CSV_WORK_STREET_MORE },       
  {"work city",     N_("work city"), CSV_WORK_CITY },              
  {"work state",    N_("work state"), CSV_WORK_STATE_PROVINCE },    
  {"work zip",      N_("work zip"), CSV_WORK_ZIP },               
  {"work country",  N_("work country"), CSV_WORK_COUNTRY },           
  {"title",         N_("title"), CSV_TITLE },                  
  {"department",    N_("department"), CSV_DEPARTMENT },             
  {"org",           N_("org"), CSV_ORGANIZATION },           
  {"web work",      N_("web work"), CSV_WORK_WEB },               
  {"web",           N_("web"), CSV_WEB },                    
  {"undef 1",       N_("undef 1"), CSV_UNDEF1 },                 
  {"undef 2",       N_("undef 2"), CSV_UNDEF2 },                 
  {"undef 3",       N_("undef 3"), CSV_UNDEF3 },                 
  {"custom 1",      N_("custom 1"), CSV_CUSTOM1 },                
  {"custom 2",      N_("custom 2"), CSV_CUSTOM2 },                
  {"custom 3",      N_("custom 3"), CSV_CUSTOM3 },                
  {"custom 4",      N_("custom 4"), CSV_CUSTOM4 },                
  {"note",          N_("note"), CSV_NOTE },                   
  {"more",          N_("more"), CSV_MORE }
};

typedef enum {
  R_CSV_START,
  R_CSV_QUOTES,
  R_CSV_DQUOTES,
  R_CSV_CLOSE_QUOTES,
  R_CSV_CLOSE_DQUOTES,  
  R_CSV_BUILD,
  R_CSV_BUILD_QUOTES,
  R_CSV_BUILD_DQUOTES,
  R_CSV_STORE                    // not used state
} RCsvState;


static GObject* parent_class = NULL;


struct _RCsvPrivate{
  RCsvState state;          /* automoton's state */

  RCsvField field;

  glong file_pos;
  GString* buffer;          /* data currently read  */

  gboolean dispose_has_run;
};


gboolean       new_card     = TRUE;
RPersonalCard* card         = NULL;
RContact*      contact      = NULL;
RAddress*      home_address = NULL;
RAddress*      work_address = NULL;
RWork*         work         = NULL;
RNotes*        notes        = NULL;
RNetAddress*   net          = NULL;
RTelephone*    phone        = NULL;


static void   r_csv_init       (RCsv* obj);
static void   r_csv_class_init (RCsvClass* klass);

static void   r_csv_dispose    (GObject* obj);
static void   r_csv_finalize   (GObject* obj);



/* private functions
*/
static gboolean  r_csv_is_separator        (gunichar ch);
static gboolean  r_csv_is_quote            (gunichar ch);
static RCsvState r_csv_get_state           (RCsv* self);

static void      r_csv_store_decoded_data  (RCsv* self, RCard* card);
static void      r_csv_append_unichar      (RCsv* self, gunichar ch);
static void      r_csv_clean_buffer        (RCsv* self);
static void      r_csv_set_next_state      (RCsv* self, RCsvState state);
static void      r_csv_set_next_field      (RCsv* self, RCsvField field);
static void      r_csv_build_data          (RCsv* self, gunichar ch);
static void      r_csv_store_file_position (RCsv* self);

static gint      r_csv_open_file           (RAbook* abook, gchar* filename);
static gboolean  r_csv_save_file           (RAbook* abook, gchar* filename);


#undef DEBUG

#ifdef DEBUG				  
#define r_debug(x, y) {			              \
    gchar* str;                                       \
                                                      \
    if (x >= 0)                                       \
      { str = r_lookup_table_enum2str(look, x);	      \
	g_print("\nDEBUG *** %-14s\t%s", str, y); }   \
    else                                              \
      { g_print("\nDEBUG *** %s", y); }               \
  }
#endif

#ifndef DEBUG
#define r_debug(x, y)
#endif

GType
r_csv_get_type()
{
  static GType r_csv_type = 0;
  
  if (!r_csv_type)
    {
      static const GTypeInfo r_csv_info =
	{
	  sizeof(RCsvClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_csv_class_init,
	  NULL,
	  NULL,
	  sizeof(RCsv),
	  0,
	  (GInstanceInitFunc) r_csv_init
	};

      r_csv_type = g_type_register_static (R_ABOOK_TYPE, "RCsv",
					   &r_csv_info, 0);
    }
  
  return r_csv_type;
}


static void
r_csv_class_init(RCsvClass* klass)
{
  GObjectClass *class;
  
  class = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_csv_dispose;
  class->finalize = (GObjectFinalizeFunc) r_csv_finalize;

  parent_class = g_type_class_peek_parent (klass);
}


static void
r_csv_init(RCsv* self)
{
  g_return_if_fail(IS_R_CSV(self));

  self->fp  = NULL;
  
  self->private = g_new(RCsvPrivate, 1);
  if (!self->private)
    g_error("\nOut of memory");

  self->private->field    = CSV_FIRST_NAME;
  self->private->state    = R_CSV_START;
  self->private->file_pos = 0L;  
  self->private->buffer   = g_string_new(NULL);
  
  self->private->dispose_has_run = FALSE;
}


static void 
r_csv_dispose (GObject* obj)
{
  RCsv* self;
 
  self = R_CSV(obj);
  g_return_if_fail(IS_R_CSV(self));

  if (self->private->dispose_has_run)
    return;

  g_string_free(self->private->buffer, TRUE);
  self->private->dispose_has_run = TRUE;
}


static void 
r_csv_finalize (GObject* obj)
{
  RCsv* self;

  self = R_CSV(obj);
  g_return_if_fail(IS_R_CSV(self));
  
  g_free(self->private);
  if (self->fp)
    fclose(self->fp);
}


static gboolean 
r_csv_is_separator(gunichar ch)
{
  if ((ch == COMMA_DELIM) ||   /* comma separator */
      (ch == TAB_DELIM))       /* tab   separator */
    return TRUE;
  
  return FALSE;
}


static gboolean 
r_csv_is_quote(gunichar ch)
{
  if (ch == QUOTES) 
    return TRUE;

  return FALSE;
}


static RCsvState
r_csv_get_state(RCsv* self)
{
  g_return_val_if_fail(IS_R_CSV(self), R_CSV_START);

  return (RCsvState) self->private->state;
}


static void 
r_csv_store_decoded_data(RCsv* self, RCard* card)
{
  gchar* more = NULL;
  gchar* tmp = NULL, *str = NULL;

  g_return_if_fail(IS_R_CSV(self));

  if (self->private->buffer->len > 0)
    {  
      tmp = self->private->buffer->str;

      switch (self->private->field)
	{
	case CSV_FIRST_NAME:
	  r_debug(-1, "");
	  r_debug(self->private->field, tmp);

	  if (!contact)
	    contact = r_contact_new();

	  g_object_set(contact, "first-name", tmp, NULL);
	  break;
	  
	case CSV_LAST_NAME:
	  r_debug(self->private->field, tmp);

	  if (!contact)
	    contact = r_contact_new();

	  g_object_set(contact, "last-name", tmp, NULL);
	  break;
	  
	case CSV_DISPLAY_NAME:
	  r_debug(self->private->field, tmp);

	  g_object_set(card, "card-name", tmp, NULL);
	  break;
	  
	case CSV_NICKNAME:
	  r_debug(self->private->field, tmp);

	  if (!contact)
	    contact = r_contact_new();

	  g_object_set(contact, "nick-name", tmp, NULL);
	  break;
	  
	case CSV_EMAIL1:
	case CSV_EMAIL2:
	  r_debug(self->private->field, tmp);

	  net = r_net_address_new();
	  g_object_set(net, "url", tmp, "url-type", R_NET_ADDRESS_EMAIL, NULL);

	  r_card_add_net_address(card, net);
	  break;
	  
	case CSV_TEL_WORK:
	  r_debug(self->private->field, tmp);

	  phone = r_telephone_new();
	  g_object_set(phone, 
		       "telephone-number", tmp, 
		       "telephone-type", R_TELEPHONE_WORK, NULL);

	  r_card_add_telephone(card, phone);
	  break;
	  
	case CSV_TEL_HOME:
	  r_debug(self->private->field, tmp);

	  phone = r_telephone_new();
	  g_object_set(phone,
		       "telephone-number", tmp, 
		       "telephone-type", R_TELEPHONE_HOME, NULL);

	  r_card_add_telephone(card, phone);	  
	  break;
	  
	case CSV_TEL_FAX:
	  r_debug(self->private->field, tmp);

	  phone = r_telephone_new();
	  g_object_set(phone, 
		       "telephone-number", tmp, 
		       "telephone-type", R_TELEPHONE_FAX, NULL);

	  r_card_add_telephone(card, phone);
	  break;
	  
	case CSV_TEL_PAGER:
	  r_debug(self->private->field, tmp);

	  phone = r_telephone_new();
	  g_object_set(phone, 
		       "telephone-number", tmp, 
		       "telephone-type", R_TELEPHONE_OTHER, NULL);

	  r_card_add_telephone(card, phone);
	  break;
	  
	case CSV_TEL_MOBILE:
	  r_debug(self->private->field, tmp);

	  phone = r_telephone_new();
	  g_object_set(phone, 
		       "telephone-number", tmp, 
		       "telephone-type", R_TELEPHONE_CELLPHONE, NULL);

	  r_card_add_telephone(card, phone);
	  break;
	  
	case CSV_HOME_STREET:
	  r_debug(self->private->field, tmp);

	  if (!home_address)
	    home_address = r_address_new();
	  
	  g_object_set(home_address,
		       "street", tmp,
		       "address-type", R_ADDRESS_HOME, NULL);
	  break;
	  
	case CSV_HOME_STREET_MORE:
	  r_debug(self->private->field, tmp);

	  if (!home_address)
	    {
	      home_address = r_address_new();
	      g_object_set(home_address, "address-type", R_ADDRESS_HOME, 
			   "street", tmp, NULL);
	    }
	  else
	    {
	      g_object_get(home_address, "street", &str, NULL);
	      more = g_strdup_printf("%s %s", str, tmp);
	      
	      g_object_set(home_address, "street", more, NULL);
	  
	      g_free(str);
	      g_free(more);	  
	    }
	  break;
	  
	case CSV_HOME_CITY:
	  r_debug(self->private->field, tmp);

	  if (!home_address)
	    {
	      home_address = r_address_new();
	      g_object_set(home_address, "address-type", R_ADDRESS_HOME, NULL);
	    }

	  g_object_set(home_address, "city", tmp, NULL);
	  break;
	  
	case CSV_HOME_STATE_PROVINCE:
	  r_debug(self->private->field, tmp);

	  if (!home_address)
	    {
	      home_address = r_address_new();
	      g_object_set(home_address, "address-type", R_ADDRESS_HOME, NULL);
	    }

	  g_object_set(home_address, "province", tmp, NULL);
	  break;
	  
	case CSV_HOME_ZIP:
	  r_debug(self->private->field, tmp);

	  if (!home_address)
	    {
	      home_address = r_address_new();
	      g_object_set(home_address, "address-type", R_ADDRESS_HOME, NULL);
	    }

	  g_object_set(home_address, "zip", tmp, NULL);
	  break;
	  
	case CSV_HOME_COUNTRY:
	  r_debug(self->private->field, tmp);

	  if (!home_address)
	    {
	      home_address = r_address_new();
	      g_object_set(home_address, "address-type", R_ADDRESS_HOME, NULL);
	    }

	  g_object_set(home_address, "country", tmp, NULL);
	  break;
	  
	case CSV_WORK_STREET:
	  r_debug(self->private->field, tmp);

	  if (!work_address)
	    work_address = r_address_new();

	  g_object_set(work_address, 
		       "address-type", R_ADDRESS_WORK, "street", tmp, NULL);
	  break;
	  
	case CSV_WORK_STREET_MORE:
	  r_debug(self->private->field, tmp);

	  if (!work_address)
	    {
	      work_address = r_address_new();
	      g_object_set(work_address, "type", R_ADDRESS_WORK,
			   "street", tmp, NULL);
	    }
	  else
	    {
	      g_object_get(work_address, "street", &str, NULL);
	      more = g_strdup_printf("%s %s", str, tmp);
	      
	      g_object_set(work_address, "street", more, NULL);
	      
	      g_free(str);
	      g_free(more);	  	  
	    }
	  break;

	case CSV_WORK_CITY:
	  r_debug(self->private->field, tmp);

	  if (!work_address)
	    {
	      work_address = r_address_new();
	      
	      g_object_set(work_address, "address-type", R_ADDRESS_WORK, NULL);
	    }

	  g_object_set(work_address, "city", tmp, NULL);	  
	  break;
	  
	case CSV_WORK_STATE_PROVINCE:
	  r_debug(self->private->field, tmp);

	  if (!work_address)
	    {
	      work_address = r_address_new();
	      g_object_set(work_address, "address-type", R_ADDRESS_WORK, NULL);
	    }

	  g_object_set(work_address, "province", tmp, NULL);
	  break;
	  
	case CSV_WORK_ZIP:
	  r_debug(self->private->field, tmp);

	  if (!work_address)
	    {
	      work_address = r_address_new();
	      g_object_set(work_address, "address-type",R_ADDRESS_WORK, NULL);
	    }
	  
	  g_object_set(work_address, "zip", tmp, NULL);
	  break;
	  
	case CSV_WORK_COUNTRY:
	  r_debug(self->private->field, tmp);

	  if (!work_address)
	    {
	      work_address = r_address_new();
	      g_object_set(work_address, "address-type", R_ADDRESS_WORK, NULL);
	    }

	  g_object_set(work_address, "country", tmp, NULL);
	  break;
	  
	case CSV_TITLE:
	  r_debug(self->private->field, tmp);

	  if (!work)
	    work = r_work_new();

	  g_object_set(work, "assignment", tmp, NULL);
	  break;
	  
	case CSV_DEPARTMENT:
	  r_debug(self->private->field, tmp);

	  if (!work)
	    work = r_work_new();

	  g_object_set(work, "department", tmp, NULL);
	  break;
	  
	case CSV_ORGANIZATION:
	  r_debug(self->private->field, tmp);

	  if (!work)
	    work = r_work_new();

	  g_object_set(work, "organization", tmp, NULL);
	  break;
	  
	case CSV_WORK_WEB:
	  r_debug(self->private->field, tmp);

	  net = r_net_address_new();

	  g_object_set(net, "url", tmp, "url-type", R_NET_ADDRESS_WORK_WEB, 
		       NULL);
	  r_card_add_net_address(card, net);
	  break;

	case CSV_WEB:
	  r_debug(self->private->field, tmp);

	  net = r_net_address_new();

	  g_object_set(net, "url", tmp, "url-type", R_NET_ADDRESS_WEB, NULL);
	  r_card_add_net_address(card, net);
	  break;
	  
	case CSV_UNDEF1:
	case CSV_UNDEF2:
	case CSV_UNDEF3:
	  r_debug(self->private->field, tmp);

	  break;
	  
	case CSV_CUSTOM1:
	  r_debug(self->private->field, tmp);

	  if (!notes)
	    notes = r_notes_new();

	  r_notes_append_other_notes(notes, tmp);
	  break;
	  
	case CSV_CUSTOM2:
	  r_debug(self->private->field, tmp);

	  if (!notes)
	    notes = r_notes_new();

	  r_notes_append_other_notes(notes, tmp);
	  break;
	  
	case CSV_CUSTOM3:
	  r_debug(self->private->field, tmp);

	  if (!notes)
	    notes = r_notes_new();

	  r_notes_append_other_notes(notes, tmp);
	  break;
	  
	case CSV_CUSTOM4:
	  r_debug(self->private->field, tmp);

	  if (!notes)
	    notes = r_notes_new();

	  r_notes_append_other_notes(notes, tmp);
	  break;
	  
	case CSV_NOTE:
	  r_debug(self->private->field, tmp);

	  if (!notes)
	    notes = r_notes_new();

	  r_notes_append_other_notes(notes, tmp);
	  break;
	  
	case CSV_MORE:
	  r_debug(self->private->field, tmp);

	  if (!notes)
	    notes = r_notes_new();

	  r_notes_append_other_notes(notes, tmp);
	  break;
	  
	default:
	  //	  g_free(tmp);
	  break;
	}
    }
  else
    r_debug(self->private->field, "-");

  if (self->private->field != CSV_MORE)
    self->private->field++;
  else
    self->private->field = CSV_FIRST_NAME;
}


static void
r_csv_append_unichar(RCsv* self, gunichar ch)
{
  g_return_if_fail(IS_R_CSV(self));

  g_string_append_unichar(self->private->buffer, ch);
}


static void 
r_csv_set_next_state(RCsv* self, RCsvState state)
{
  g_return_if_fail(IS_R_CSV(self));

  self->private->state = state;
}


static void 
r_csv_set_next_field(RCsv* self, RCsvField field)
{
  g_return_if_fail(IS_R_CSV(self));

  self->private->field = field;
}


static void 
r_csv_clean_buffer (RCsv* self)
{
  g_return_if_fail(IS_R_CSV(self));
  
  if (self->private->buffer)
    g_string_truncate (self->private->buffer, 0);
}


static void 
r_csv_build_data(RCsv* self, gunichar ch)
{
  RCsvState current_state;

  g_return_if_fail(IS_R_CSV(self));

  current_state = r_csv_get_state(self);
  switch (current_state)
    {
    case R_CSV_START:
      if (new_card)
	{
	  card     = r_personal_card_new(); 
	  new_card = FALSE;
	}
      
      r_csv_clean_buffer(self);      
      if (r_csv_is_quote(ch))
	r_csv_set_next_state(self, R_CSV_QUOTES);	
      else
	{
	  if (r_csv_is_separator(ch))
	    {
	      r_csv_store_decoded_data(self, R_CARD(card));
	      r_csv_set_next_state(self, R_CSV_START);
	    }
	  else
	    {
	      r_csv_append_unichar(self, ch);
	      r_csv_set_next_state(self, R_CSV_BUILD);
	    }
	}
      break;
      
    case R_CSV_QUOTES:
      if (r_csv_is_quote(ch))
	r_csv_set_next_state(self, R_CSV_DQUOTES);	
      else
	{
	  r_csv_append_unichar(self, ch);
	  r_csv_set_next_state(self, R_CSV_BUILD_QUOTES);
	}
      break;
      
    case R_CSV_DQUOTES:
      if (r_csv_is_quote(ch))
	r_csv_set_next_state(self, R_CSV_CLOSE_DQUOTES);
      else
	{
	  if (r_csv_is_separator(ch))
	    {
	      r_csv_store_decoded_data(self, R_CARD(card));
	      r_csv_set_next_state(self, R_CSV_START);
	    }
	  else
	    {
	      r_csv_append_unichar(self, ch);
	      r_csv_set_next_state(self, R_CSV_BUILD_DQUOTES);
	    }
	}     
      break;
      
    case R_CSV_BUILD:
      if (r_csv_is_separator(ch))
	{
	  r_csv_store_decoded_data(self, R_CARD(card));
	  r_csv_set_next_state(self, R_CSV_START);
	}
      else
	{
	  r_csv_append_unichar(self, ch);
	  r_csv_set_next_state(self, R_CSV_BUILD);
	}               
      break;	  
      
    case R_CSV_BUILD_QUOTES:
      if (r_csv_is_quote(ch))
	r_csv_set_next_state(self, R_CSV_CLOSE_QUOTES);
      else
	{
	  r_csv_append_unichar(self, ch);
	  r_csv_set_next_state(self, R_CSV_BUILD_QUOTES);
	}       
      break;	  
      
    case R_CSV_BUILD_DQUOTES:
      if (r_csv_is_quote(ch))
	r_csv_set_next_state(self, R_CSV_CLOSE_DQUOTES);
      else
	{
	  r_csv_append_unichar(self, ch);
	  r_csv_set_next_state(self, R_CSV_BUILD_DQUOTES);
	} 
      break;	 

    case R_CSV_CLOSE_QUOTES:
      if (!r_csv_is_separator(ch))
	r_csv_set_next_state(self, R_CSV_CLOSE_QUOTES);
      else
	{
	  r_csv_store_decoded_data(self, R_CARD(card));
	  r_csv_set_next_state(self, R_CSV_START);
	  /* */
	}
      break;
      
    case R_CSV_CLOSE_DQUOTES:
      if (r_csv_is_quote(ch))
	r_csv_set_next_state(self, R_CSV_CLOSE_QUOTES);	      
      else
	r_csv_set_next_state(self, R_CSV_CLOSE_DQUOTES);
      break;	  
      
    default:
      break;
    }
}


void 
r_csv_store_file_position(RCsv *self)
{
  g_return_if_fail(IS_R_CSV(self));

  self->private->file_pos = ftell(self->fp);
}



/* *****************************  Public functions  ***********************
 */

/**
 * r_csv_new
 *
 * create a new #RCsv
 *
 * returns: a #RCsv*
 */
RCsv* 
r_csv_new(void)
{
  return (RCsv*) g_object_new(r_csv_get_type(), NULL);
}


/**
 * r_csv_free
 * @csv: a #RCsv
 * 
 * free the #RCsv object
 */
void  
r_csv_free(RCsv *csv) 
{ 
  g_return_if_fail(IS_R_CSV(csv)); 

  g_object_unref(csv); 
} 


/**
 * r_csv_open_file
 * @csv: #RCsv
 * @file: the file's name
 *
 * Read a csv file. 
 * When a csv file is open a "file_opened" signal is emitted, 
 * with an error code on file status.
 * Each time a csv's record is decoded, a "card_decoded" signal is 
 * emitted and a #RPersonalCard will be sent to the user callback
 * function. The "addressbook_decoded" signal is emitted when csv file
 * if successfully read.
 *
 * returns: a gint, %FILE_OPEN_FAIL or %FILE_IMPORT_SUCCESS
 */
static gboolean          
r_csv_open_file (RAbook* abook, gchar* filename)
{
  RCsv* csv;
  gunichar ch;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);

  if (!filename)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    NO_FILENAME, G_TYPE_INT);
      
      return FALSE;
    }
  
  if (!g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_NOT_EXIST, G_TYPE_INT);
      
      return FALSE;
    }
  
  csv = (RCsv*) r_abook_get_plugin(abook);
  csv->fp = fopen(filename, "r");
  if (!csv->fp)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_OPEN_FAIL, G_TYPE_INT);

      return FALSE; 
    }
  
  g_object_set(R_ABOOK(abook),
	       "addressbook-name", g_path_get_basename(filename),
	       "addressbook-path", g_path_get_dirname(filename),
	       NULL);
  do 
    {
      ch = fgetc(csv->fp);
      while ((ch == '\n') && (csv->private->state == R_CSV_START))
	ch = fgetc(csv->fp);

      /* read characters from file until \n or \n\r or EOF is reached 
       */
      if ((ch != feof(csv->fp)))
	{
	  if ((ch != '\n') && (ch != '\r'))
	    r_csv_build_data(csv, ch);
	  else 	                 /* card successfully read */
	    {
	      //	      g_print("\nCARD READ");
	      r_csv_store_file_position(csv);
	      
	      if (contact)
		{
		  r_personal_card_set_contact(card, contact);
		  contact = NULL;
		}

	      if (work)
		{
		  r_personal_card_set_work(card, work);
		  work = NULL;
		}

	      if (notes)
		{
		  r_personal_card_set_notes(card, notes);		  
		  notes = NULL; 
		}

	      if (home_address)
		{
		  r_card_add_address(R_CARD(card), home_address);
		  home_address = NULL;
		}

	      if (work_address)
		{
		  r_card_add_address(R_CARD(card), work_address);
		  work_address = NULL;
		}
	  
	      r_abook_add_loaded_card(R_ABOOK(abook), R_CARD(card));

	      g_signal_emit_by_name(R_ABOOK(abook), "card_read", 
				    card, G_TYPE_POINTER);

	      r_csv_set_next_field(csv, CSV_FIRST_NAME);
	      r_csv_set_next_state(csv, R_CSV_START);
	      new_card = TRUE;	      
	    }
	}
    }
  while(!feof(csv->fp));

  g_signal_emit_by_name(R_ABOOK(abook), "addressbook_read", NULL, G_TYPE_NONE);
  
  return TRUE;
}


static void
write_card (RCard* card, FILE* fp)
{
  RNotes*   notes;
  RAddress* address;
  RNetAddress* net;
  RTelephone* tel;

  gchar *type, *dn;
  gchar *fn = NULL, *ln = NULL, *nick = NULL;
  gchar *str = NULL, *city = NULL, *state = NULL;
  gchar *zip = NULL, *country = NULL;
  gchar *wstr = NULL, *wcity = NULL, *wstate = NULL;
  gchar *wzip = NULL, *wcountry = NULL;
  gchar *email1 = NULL, *email2 = NULL;
  gchar *web = NULL, *wweb = NULL;
  gchar *wtel = NULL, *htel = NULL, *fax = NULL;
  gchar *pager = NULL, *mobile = NULL;
  gchar *role = NULL, *dep = NULL, *org = NULL;
  gchar *note = NULL;

  g_object_get(card, "card-type", &type, "card-name", &dn, NULL);
  if (g_ascii_strcasecmp(type, "personal") == 0)
    {
      RContact* contact;
      RWork* work;
  
      contact = r_personal_card_get_contact(R_PERSONAL_CARD(card));
      work    = r_personal_card_get_work(R_PERSONAL_CARD(card));
      notes   = r_personal_card_get_notes(R_PERSONAL_CARD(card));
      
      g_object_get(contact, 
		   "first-name", &fn,
		   "last-name",  &ln, 
		   "nick-name",  &nick, NULL);
      g_object_get(work, 
		   "assignment", &role, 
		   "department", &dep, 
		   "organization", &org, NULL);
      g_object_get(notes, "other-notes", &note, NULL);      
    }
  else // company
    {
      g_object_get(R_COMPANY_CARD(card), "notes", &note, NULL);           
    }
        
  /* reset iter */
  r_card_reset_address(card);
  
  /* find address of type: R_ADDRESS_HOME */
  address = r_card_find_address(card, R_ADDRESS_HOME);
  if (address)
    g_object_get(address, 
		 "street",  &str,
		 "city",    &city,
		 "state",   &state,
		 "zip",     &zip,
		 "country", &country,
		 NULL);
  
  /* reset iter */
  r_card_reset_address(card);
  
  /* find address of type: R_ADDRESS_WORK */
  address = r_card_find_address(card, R_ADDRESS_WORK);
  if (address)
    g_object_get(address, 
		 "street",  &wstr,
		 "city",    &wcity,
		 "state",   &wstate,
		 "zip",     &wzip,
		 "country", &wcountry,
		 NULL);
  
  /* reset net iter */
  r_card_reset_net_address(card);
  
  /* find first net address */
  net = r_card_find_net_address(card, R_NET_ADDRESS_EMAIL);
  if (net) 
    g_object_get(net, "url", &email1, NULL);
  
  /* find next next address */
  net = r_card_find_net_address(card, R_NET_ADDRESS_EMAIL);
  if (net) 
    g_object_get(net, "url", &email2, NULL);     
  
  r_card_reset_net_address(card);   
  net = r_card_find_net_address(card, R_NET_ADDRESS_WEB);
  if (net) 
    g_object_get(net, "url", &web, NULL);
  
  r_card_reset_net_address(card);        
  net = r_card_find_net_address(card, R_NET_ADDRESS_WORK_WEB);
  if (net) 
    g_object_get(net, "url", &wweb, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_HOME);
  if (tel)
    g_object_get(tel, "telephone-number", &htel, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_WORK);
  if (tel)
    g_object_get(tel, "telephone-number", &wtel, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_FAX);
  if (tel)
    g_object_get(tel, "telephone-number", &fax, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_CELLPHONE);
  if (tel)
    g_object_get(tel, "telephone-number", &mobile, NULL);
  
  r_card_reset_telephone(card);
  tel = r_card_find_telephone(card, R_TELEPHONE_PAGER);
  if (tel)
    g_object_get(tel, "telephone-number", &pager, NULL); 
  
  fprintf(fp,  
	  "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\","
	  "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\","
	  "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\","
	  "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\","
	  "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\"\n", 
	  fn ? fn : "", ln ? ln : "", dn ? dn : "", nick ? nick : "",
	  email1 ? email1 : "", email2 ? email2 : "",
	  wtel ? wtel : "", htel ? htel : "", fax ? fax : "", 
	  pager ? pager : "", mobile ? pager : "",
	  str ? str : "", "", city ? city : "", state ? state : "",
	  zip ? zip : "", country ? country : "",
	  wstr ? wstr : "", "", wcity ? wcity :"", wstate ? wstate : "",
	  wzip ? wzip : "", wcountry ? wcountry : "",
	  role ? role : "", dep ? dep : "", org ? org : "",
	  wweb ? wweb : "", web ? web : "",
	  "", "", "", "", "", "", "",
	  note ? note : "", "");  
}

static gboolean 
r_csv_save_file (RAbook* abook, gchar* filename)
{  
  FILE* fp;
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);

  if (!(fp = fopen(filename, "w")))
    {
      g_warning("\nCan't write file: %s", filename);
      
      return FALSE;
    }
  
  r_abook_foreach_card (abook, (RFunc) write_card, fp);
  fflush(fp);

  return TRUE;
}


static gboolean
r_csv_overwrite_file(RAbook* abook)
{
  gchar* file;
  gchar* path;
  gchar* filename;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  g_object_get(abook, "addressbook-path", &path, 
	       "addressbook-name", &file, NULL);
  filename = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, file);
  
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    g_remove(filename);

  if (!r_csv_save_file(abook, filename))
    {
      g_signal_emit_by_name(abook, "save_fail", OVERWRITING, G_TYPE_INT);
      
      g_free(filename);
      return FALSE;
    }
    
  g_free(filename);
  g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);  
  
  return TRUE;  
}


/**
 * r_csv_read_previous_record
 * @csv: #RCsv
 *
 * read the previous record from csv file
 */
void        
r_csv_read_previous_record (RCsv *csv) 
{
  g_return_if_fail(IS_R_CSV(csv)); 

  fseek(csv->fp, csv->private->file_pos, SEEK_SET); 
}





G_MODULE_EXPORT void 
plugin_init (RPlugin* plugin, gchar* file)
{
  RPluginAction* action;
  RFilter* filter;

  g_return_if_fail(plugin != NULL);
 
  r_plugin_set_obj(plugin, r_csv_new());
  g_object_set(plugin, 
	       "plugin-name", "csv", 
	       "plugin-filename", file,
	       "plugin-info", "This plugin manages the csv file format",
	       "plugin-configurable", TRUE, NULL);
  
  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", "CSV", 
	       "filter-mime", "text/x-comma-separated-values", NULL);
  r_filter_add_pattern(filter, "csv");
  r_filter_add_pattern(filter, "*.csv");
  r_plugin_add_filter (plugin, filter);


  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("read");
  action->handle = (gpointer) r_csv_open_file;
  r_plugin_add_action(plugin, action);  
    
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("write");
  action->handle = (gpointer) r_csv_save_file;
  r_plugin_add_action(plugin, action);  

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("overwrite");
  action->handle = (gpointer) r_csv_overwrite_file;
  r_plugin_add_action(plugin, action);  
}


G_MODULE_EXPORT void     
plugin_fini (void)
{
  // TODO: liberare la memoria e rilasciare l'handler al plugin
}
