//plugins.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2014
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "roard.h"

#define MAX_PLUGINS    8

static struct _roard_plugin {
 struct roar_dl_lhandle     * lhandle;
} g_plugins[MAX_PLUGINS];
static struct _roard_plugin * _pp = NULL;

static int _plugins_inited = 0;

static int plugin_callback(enum roar_dl_fnreg_action action, int fn, int subtype, const void * object, size_t objectlen, int version, int options, void * userdata, struct roar_dl_lhandle * lhandle);
static const struct roar_dl_fnreg _plugin_callbacks = {
 .fn = -1,
 .subtype = -1,
 .version = -1,
 .callback = plugin_callback,
 .userdata = NULL
};

static struct _roard_plugin * _find_free(void) {
 int i;

 ROAR_DBG("_find_free(void) = ?");

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle == NULL ) {
   memset(&(g_plugins[i]), 0, sizeof(struct _roard_plugin));
   ROAR_DBG("_find_free(void) = %p // i=%i", &(g_plugins[i]), i);
   return &(g_plugins[i]);
  }
 }

 ROAR_DBG("_find_free(void) = NULL");
 return NULL;
}

int plugins_preinit  (void) {
 ROAR_DBG("plugins_preinit(void) = ?");

 memset(g_plugins, 0, sizeof(g_plugins));

#ifdef DEBUG
 print_pluginlist(FORMAT_NATIVE);
#endif

 ROAR_DBG("plugins_preinit(void) = 0");
 return 0;
}

static inline void plugins_delete(struct _roard_plugin * plugin) {
 ROAR_DBG("plugins_delete(plugin=%p) = ?", plugin);

 roar_dl_appsched_trigger(plugin->lhandle, ROAR_DL_APPSCHED_FREE);

 roar_dl_close(plugin->lhandle);
 memset(plugin, 0, sizeof(struct _roard_plugin));
 plugin->lhandle = NULL;
}

static int plugins_init_one(struct _roard_plugin * plugin) {
 if ( plugin == NULL )
  return -1;

 ROAR_DBG("plugins_init_one(plugin=%p) = ?", plugin);

 _pp = plugin;

 if ( roar_dl_ra_init(plugin->lhandle, NULL, NULL) == -1 ) {
  ROAR_WARN("plugins_init(void): Can not RA init lib at %p: %s", plugin->lhandle, roar_error2str(roar_error));
  plugins_delete(plugin);
  return -1;
 }

 roar_dl_appsched_trigger(plugin->lhandle, ROAR_DL_APPSCHED_INIT);

 _pp = NULL;

 return 0;
}

int plugins_init  (void) {
 size_t i;

 ROAR_DBG("plugins_init(void) = ?");

#ifdef DEBUG
 print_pluginlist(FORMAT_NATIVE);
#endif

 if ( _plugins_inited ) {
  ROAR_DBG("plugins_init(void) = -1 // error=BUSY");
  roar_err_set(ROAR_ERROR_BUSY);
  return -1;
 }

 ROAR_DL_RFNREG(ROAR_DL_HANDLE_APPLICATION, _plugin_callbacks);

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle != NULL ) {
   plugins_init_one(&(g_plugins[i]));
  }
 }

 _plugins_inited = 1;

#ifdef DEBUG
 print_pluginlist(FORMAT_NATIVE);
#endif

 ROAR_DBG("plugins_init(void) = 0");
 return 0;
}

int plugins_free  (void) {
 int i;

 ROAR_DBG("plugins_free(void) = ?");

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle != NULL ) {
   plugins_delete(&(g_plugins[i]));
  }
 }

 return plugins_preinit();
}

int plugins_update   (void) {
 int ret = 0;
 int i;

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle != NULL ) {
   if ( roar_dl_appsched_trigger(g_plugins[i].lhandle, ROAR_DL_APPSCHED_UPDATE) == -1 )
    if ( roar_error != ROAR_ERROR_NOENT )
     ret = -1;
  }
 }

 return ret;
}

int plugins_load  (const char * filename, const char * args) {
 struct _roard_plugin * next = _find_free();
 struct roar_dl_librarypara * para;

 ROAR_DBG("plugins_load(filename=\"%s\", args=\"%s\") = ?", filename, args);

 if ( next == NULL )
  return -1;

 if ( (para = roar_dl_para_new(args, NULL, ROARD_DL_APPNAME, ROARD_DL_ABIVERSION)) == NULL ) {
  ROAR_WARN("Can not load plugin (allocate para set): %s: %s", filename, roar_error2str(roar_error));
  return -1;
 }

 next->lhandle = roar_dl_open(filename, ROAR_DL_FLAG_PLUGIN, 0 /* we delay this until plugins_init() */, para);
 roar_dl_para_unref(para);

 if ( next->lhandle == NULL ) {
  ROAR_ERR("plugins_load(filename='%s'): can not load plugin: %s", filename, roar_dl_errstr(NULL));
  return -1;
 }

 if ( _plugins_inited )
  return plugins_init_one(next);

 return 0;
}

void print_pluginlist(enum output_format format) {
 const struct roar_dl_libraryname * libname;
 struct _roard_plugin * p;
 size_t i;

 switch (format) {
  case FORMAT_NATIVE:
    printf("  Name\n");
    printf("   Attributes\n");
    printf("-----------------------------------------------------\n");
   break;
  case FORMAT_WIKI:
  case FORMAT_CSV:
  default:
    roar_err_set(ROAR_ERROR_NOTSUP);
    return;
 }

 for (i = 0; i < MAX_PLUGINS; i++) {
  p = &(g_plugins[i]);
  if ( p->lhandle == NULL )
   continue;

  libname = roar_dl_getlibname(p->lhandle);
  if ( libname == NULL ) {
  } else {
   printf("  %s\n", libname->libname);
   printf("   Flags:       %s\n", "");
   if ( libname->description != NULL )
    printf("   Description: %s\n", libname->description);
   if ( libname->contact != NULL )
    printf("   Contact:     %s\n", libname->contact);
  }
 }
}

static int plugin_callback(enum roar_dl_fnreg_action action, int fn, int subtype, const void * object, size_t objectlen, int version, int options, void * userdata, struct roar_dl_lhandle * lhandle) {

 (void)options, (void)userdata;

 ROAR_DBG("plugin_callback(action=%i, fn=%i, subtype=%i, object=%p, objectlen=%llu, version=%i, options=0x%x, userdata=%p, lhandle=%p) = ?", (int)action, fn, subtype, object, (long long unsigned int)objectlen, version, options, userdata, lhandle);

 switch (fn) {
  case ROAR_DL_FN_PROTO:
    if ( subtype != ROAR_DL_PROTO_SUBTYPE ) {
     ROAR_DBG("plugin_callback(action=%i, fn=%i, subtype=%i, object=%p, objectlen=%llu, version=%i, options=0x%x, userdata=%p, lhandle=%p) = -1 // error=TYPEMM", (int)action, fn, subtype, object, (long long unsigned int)objectlen, version, options, userdata, lhandle);

     roar_err_set(ROAR_ERROR_TYPEMM);
     return -1;
    }
    if ( objectlen != ROAR_DL_PROTO_SIZE ) {
     ROAR_DBG("plugin_callback(action=%i, fn=%i, subtype=%i, object=%p, objectlen=%llu, version=%i, options=0x%x, userdata=%p, lhandle=%p) = -1 // error=BADLIB", (int)action, fn, subtype, object, (long long unsigned int)objectlen, version, options, userdata, lhandle);
     roar_err_set(ROAR_ERROR_BADLIB);
     return -1;
    }
    if ( version != ROAR_DL_PROTO_VERSION ) {
     ROAR_DBG("plugin_callback(action=%i, fn=%i, subtype=%i, object=%p, objectlen=%llu, version=%i, options=0x%x, userdata=%p, lhandle=%p) = -1 // error=BADVERSION", (int)action, fn, subtype, object, (long long unsigned int)objectlen, version, options, userdata, lhandle);
     roar_err_set(ROAR_ERROR_BADVERSION);
     return -1;
    }
    switch (action) {
     case ROAR_DL_FNREG:
       return clients_register_proto_common(object, lhandle);
      break;
     case ROAR_DL_FNUNREG:
       return clients_unregister_proto(((const struct roard_proto *)object)->proto);
      break;
    }
   break;
 }

 ROAR_DBG("plugin_callback(action=%i, fn=%i, subtype=%i, object=%p, objectlen=%llu, version=%i, options=0x%x, userdata=%p, lhandle=%p) = -1 // error=NOTSUP", (int)action, fn, subtype, object, (long long unsigned int)objectlen, version, options, userdata, lhandle);
 roar_err_set(ROAR_ERROR_NOTSUP);
 return -1;
}

//ll
