///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo qmg inputs / outputs
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 15 february 2002
//

// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
#include "rheolef/geo-qmg.h"
using namespace rheolef;
using namespace std;


// ============================================================================
// qmg data utilities
// ============================================================================

void
qmg::reverse ()
{
    _T.reverse();
    for (slist<slist<qmg::tetra> >::iterator p =_T.begin(); p != _T.end(); p++) {
        (*p).reverse();
    }
    _t.reverse();
    for (slist<slist<qmg::triangle> >::iterator p =_t.begin(); p != _t.end(); p++) {
        (*p).reverse();
    }
    _e.reverse();
    for (slist<slist<qmg::edge> >::iterator p =_e.begin(); p != _e.end(); p++) {
        (*p).reverse();
    }
    _p.reverse();
    for (slist<slist<qmg::vertex> >::iterator p =_p.begin(); p != _p.end(); p++) {
        (*p).reverse();
    }
    for (unsigned int d = 0; d < 4; d++) {
	_name_list [d].reverse();
    }
}
// ============================================================================
// qmg input utilities
// ============================================================================

template <class Element>
inline
typename slist<Element>::size_type
cumul_size (const slist<slist<Element> >& ll)
{
    typedef typename slist<Element>::size_type size_type;
    size_type n = 0;
    for (typename slist<slist<Element> >::const_iterator p = ll.begin();
        p != ll.end(); p++) {
        n += (*p).size();
    }
    return n;
}
template <class Element, class OutputIterator, 
	class Table1, class Table2, class Size>
inline
void
cumul_store (
        const slist<slist<Element> >& ll, 
	OutputIterator iter_elt,
	Table1 count_geo,
	Table2 count_element,
	Size dim,
	Size n_vert)
{
    typedef typename iterator_traits<OutputIterator>::value_type ElementDest;
    Size K_idx = 0;
    for (typename slist<slist<Element> >::const_iterator p = ll.begin(); p != ll.end(); p++) {
        for (typename slist<Element>::const_iterator q = (*p).begin(); q != (*p).end(); q++) {

      	    const Element& K0 = (*q);
      	    ElementDest&   K  = (*iter_elt++);
      	    K.set_variant (K0.size(), dim);
      	    K.set_index(K_idx++) ;
      	    for (size_type i = 0 ; i < (*q).size(); i++) {
		K[i] = K0[i];
		if (K[i] >= n_vert) {
	    	    error_macro("qmg input: element " << K.index()+1 << ": "
		        << i+1 << "-th vertice index "
              	        << K[i] << " out of range 0:" << n_vert-1);
		}
      	    }
            count_element [K.variant()]++;
      	    count_geo     [dim]++;
        }
    }
}
template <class Element>
inline
void
cumul_store_domains (
        const slist<slist<Element> >& ll,
	unsigned int		      dim,
        const slist<string>& 	      name_list,
	georep::domlist_type::iterator   q)
{
    if (name_list.size() != 0) {
        check_macro (ll.size() == name_list.size(), 
	    ll.size() << " domain names expected, "
	    << name_list.size() << " founded");
    }
    georep::size_type i_dom = 0;
    typename slist<string>::const_iterator p_name = name_list.begin();
    for (typename slist<slist<Element> >::const_iterator p = ll.begin(); p != ll.end(); p++, q++) {
	domain& d = *q;
	d.set_dimension(dim);
        if (name_list.size() != 0) {
            d.set_name (*p_name++);
	} else {
            d.set_name ("boundary" + itos(i_dom++));
	}
	d.resize((*p).size());
	domain::iterator s = (*q).begin();
        for (typename slist<Element>::const_iterator r = (*p).begin(); r != (*p).end(); r++, s++) {

      	    const Element& K0 = *r;
      	    geo_element&   K  = *s;
	    K.set_variant (K0.variant());
	    for (georep::size_type i = 0; i < K0.size(); i++) {
		K[i] = K0[i];
	    }
   	}
    }
}
// ============================================================================
// qmg input
// ============================================================================

istream&
georep::get_qmg(istream& s) 
{
  typedef georep::size_type size_type;
  if (!s) error_macro("bad input stream for qmg.");
  bool verbose = iorheo::getverbose(s);

  struct qmg data;
  s >> data;

  // qmg has only boundary edges that are numbered
  // and have domain numbers 
  // => this is only a version 1 mesh file format type
  //    not consistent for P2 elements
  _version = 1;
  _serial_number = numeric_limits<size_type>::max();
  _dim = data._dim;
  check_macro (_dim == 2 || _dim == 3, "unexpected qmg mesh dimension: " << _dim);
  reset_counters();
  //
  // get coordinates
  //
  size_type n_vert = data._x.size();
  _x.resize(n_vert);

  for (size_type j = 0; j < _dim; j++) {
    _xmin[j] =  numeric_limits<Float>::max();
    _xmax[j] = -numeric_limits<Float>::max();
  }
  for (size_type j = _dim+1; j < 3; j++) {
    _xmin [j] = _xmax [j] = 0;
  }
  size_type i = 0;
  nodelist_type::iterator last_p = _x.end();
  for (nodelist_type::iterator iter_p = _x.begin(); iter_p != last_p; iter_p++, i++) {
    *iter_p = data._x[i];
    const point& p = *iter_p;
    for (size_type j = 0 ; j < _dim; j++) {
      _xmin[j] = ::min(p[j], _xmin[j]);
      _xmax[j] = ::max(p[j], _xmax[j]);
    }
  }
  _count_geo     [0] = n_vert;
  _count_element [reference_element::p] = n_vert;
  //
  // get elements
  //
  size_type n_elt = (_dim == 2 ) ? cumul_size (data._t) : cumul_size (data._T);
  resize(n_elt);
  if (_dim == 3) {
      cumul_store (data._T, begin(), &_count_geo[0], &_count_element[0], _dim, n_vert);
  } else {
      cumul_store (data._t, begin(), &_count_geo[0], &_count_element[0], _dim, n_vert);
  }
  //
  // get domains: TODO
  //
  if (_dim == 3) {
      _domlist.resize(data._t.size());
      cumul_store_domains (data._t, 2, data._name_list[2], _domlist.begin()); 
  } else { 
      _domlist.resize(data._e.size());
      cumul_store_domains (data._e, 1, data._name_list[1], _domlist.begin()); 
  }
#ifdef TODO
#endif // TODO
  return s;
}
// =======================================================================
// qmg output
// =======================================================================

ostream& 
georep::put_qmg (ostream& os) const
{
  fatal_macro ("qmg output not yet supported");
  return os;
}
