#ifndef _RHEO_FEM_H
#define _RHEO_FEM_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

//! Helps the management of the finite element method.
//! Helps the geometry coordinate system.
//! Helps fields that have tensor value : 
//! doubles indices (i,j) are mapped to single index depending upon
//! the case : symmetric tensor, axisymmetric problems with extra 
//! theta-theta tensor component or tridimensionnal case.
//! This class is used internally by field and space classes.
//! It contains no data, only typedefs and member functions.

#include "rheolef/compiler.h"
namespace rheolef { 

class fem_helper {
public:
// -------------------------------------------------------------
// coordinate system helper
// -------------------------------------------------------------

// typedefs and static data:

  typedef size_t size_type;

  typedef enum {
          cartesian       = 0,
          axisymmetric_rz = 1,
          axisymmetric_zr = 2,
          max_coord_sys   = 3
  } coordinate_type;

  static const char* coord_sys_table [max_coord_sys];

// static members:

  static coordinate_type coordinate_system      (std::string sys_coord);
  static std::string     coordinate_system_name (coordinate_type);
  static void check_coord_sys_and_dimension     (coordinate_type, size_type d);

// -------------------------------------------------------------
// valued field helper
// -------------------------------------------------------------

// typedefs and static data:
  
  typedef enum {
	  scalar		 = 0,
	  vectorial		 = 1,
	  tensorial		 = 2, // symmetric, D_ij
	  unsymmetric_tensorial	 = 3,
	  tensorial_4 		 = 4, // symmetric, A_ijkl
	  hybrid	         = 5,
	  max_valued_field	 = 6
  } valued_field_type;

  // Note: 3 symmetries for 4-order tensors
  //     A_ijkl=A_ijlk, A_ijkl=A_jikl & A_ijkl=Aklij
  // => 6 components in 2d, 21 in 3d and 10 in 2d-axisymmetric(rz)

  struct pair_size_type {
          size_type first;
          size_type second;
  };
  struct double_pair_size_type {
          pair_size_type first;
          pair_size_type second;
  };
  static pair_size_type make_pair_size(size_type a, size_type b);
  static double_pair_size_type make_double_pair_size(
	  size_type a, size_type b, size_type c, size_type d);

  static const char*    valued_field_table [max_valued_field];

  static size_type      unsymmetric_tensor_index [3][3];
  static pair_size_type unsymmetric_tensor_subscript      [9];
  static const char*    unsymmetric_tensor_subscript_name [9];

  static size_type        symmetric_tensor_index [3][3];
  static pair_size_type   symmetric_tensor_subscript      [6];
  static const char*      symmetric_tensor_subscript_name [6];

  static size_type             symmetric_tensor4_index [6][6];
  static double_pair_size_type symmetric_tensor4_subscript      [21];
  static const char*           symmetric_tensor4_subscript_name [21];
  static double_pair_size_type symmetric_tensor4_subscript_rz      [10];
  static const char*           symmetric_tensor4_subscript_rz_name [10];

// static members:
  
  static valued_field_type valued_field      (std::string valued);
  static std::string       valued_field_name (valued_field_type);

  static size_type tensor_index (std::string valued, std::string sys_coord, size_type i, size_type j);
  static size_type tensor_index (valued_field_type, coordinate_type, size_type i, size_type j);

  static pair_size_type tensor_subscript      (valued_field_type, coordinate_type, size_type i_comp);
  static std::string    tensor_subscript_name (valued_field_type, coordinate_type, size_type i_comp);
  static pair_size_type tensor_subscript      (std::string valued, std::string sys_coord, size_type i_comp);
  static std::string    tensor_subscript_name (std::string valued, std::string sys_coord, size_type i_comp);
  
  static size_type tensor4_index (std::string valued, std::string sys_coord,
		  size_type i, size_type j, size_type k, size_type l);
  static size_type tensor4_index (valued_field_type, coordinate_type,
		  size_type i, size_type j, size_type k, size_type l);
  
  static double_pair_size_type tensor4_subscript      (valued_field_type, coordinate_type, size_type i_comp);
  static std::string           tensor4_subscript_name (valued_field_type, coordinate_type, size_type i_comp);
  static double_pair_size_type tensor4_subscript      (std::string valued, std::string sys_coord, size_type i_comp);
  static std::string           tensor4_subscript_name (std::string valued, std::string sys_coord, size_type i_comp);

// -------------------------------------------------------------
// approximation name -> basis and numbering names
// -------------------------------------------------------------
  static std::string basis_name (std::string approx_name);
  static std::string numbering_name (std::string approx_name);
};
}// namespace rheolef
#endif // _RHEO_FEM_H
