% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expand.R
\name{expand}
\alias{expand}
\alias{crossing}
\alias{nesting}
\title{Expand data frame to include all combinations of values}
\usage{
expand(data, ...)

crossing(...)

nesting(...)
}
\arguments{
\item{data}{A data frame.}

\item{...}{Specification of columns to expand. Columns can be atomic vectors
or lists.

To find all unique combinations of x, y and z, including those not
found in the data, supply each variable as a separate argument.
To find only the combinations that occur in the data, use nest:
\code{expand(df, nesting(x, y, z))}.

You can combine the two forms. For example,
\code{expand(df, nesting(school_id, student_id), date)} would produce
a row for every student for each date.

For factors, the full set of levels (not just those that appear in the
data) are used. For continuous variables, you may need to fill in values
that don't appear in the data: to do so use expressions like
\code{year = 2010:2020} or \verb{year = \link{full_seq}(year,1)}.

Length-zero (empty) elements are automatically dropped.}
}
\description{
\code{expand()} is often useful in conjunction with \code{left_join()} if
you want to convert implicit missing values to explicit missing values.
Or you can use it in conjunction with \code{anti_join()} to figure
out which combinations are missing.
}
\details{
\code{crossing()} is a wrapper around \code{\link[=expand_grid]{expand_grid()}} that deduplicates and sorts
each input. \code{nesting()} is the complement to \code{crossing()}: it only keeps
combinations of values that appear in the data.
}
\examples{
library(dplyr)
# All possible combinations of vs & cyl, even those that aren't
# present in the data
expand(mtcars, vs, cyl)

# Only combinations of vs and cyl that appear in the data
expand(mtcars, nesting(vs, cyl))

# Implicit missings ---------------------------------------------------------
df <- tibble(
  year   = c(2010, 2010, 2010, 2010, 2012, 2012, 2012),
  qtr    = c(   1,    2,    3,    4,    1,    2,    3),
  return = rnorm(7)
)
df \%>\% expand(year, qtr)
df \%>\% expand(year = 2010:2012, qtr)
df \%>\% expand(year = full_seq(year, 1), qtr)
df \%>\% complete(year = full_seq(year, 1), qtr)

# Nesting -------------------------------------------------------------------
# Each person was given one of two treatments, repeated three times
# But some of the replications haven't happened yet, so we have
# incomplete data:
experiment <- tibble(
  name = rep(c("Alex", "Robert", "Sam"), c(3, 2, 1)),
  trt  = rep(c("a", "b", "a"), c(3, 2, 1)),
  rep = c(1, 2, 3, 1, 2, 1),
  measurement_1 = runif(6),
  measurement_2 = runif(6)
)

# We can figure out the complete set of data with expand()
# Each person only gets one treatment, so we nest name and trt together:
all <- experiment \%>\% expand(nesting(name, trt), rep)
all

# We can use anti_join to figure out which observations are missing
all \%>\% anti_join(experiment)

# And use right_join to add in the appropriate missing values to the
# original data
experiment \%>\% right_join(all)
# Or use the complete() short-hand
experiment \%>\% complete(nesting(name, trt), rep)

# Generate all combinations with expand():
formulas <- list(
  formula1 = Sepal.Length ~ Sepal.Width,
  formula2 = Sepal.Length ~ Sepal.Width + Petal.Width,
  formula3 = Sepal.Length ~ Sepal.Width + Petal.Width + Petal.Length
)
data <- split(iris, iris$Species)
crossing(formula = formulas, data)
}
\seealso{
\code{\link[=complete]{complete()}} for a common application of \code{expand}:
completing a data frame with missing combinations. \code{\link[=expand_grid]{expand_grid()}}
is low-level that doesn't deduplicate or sort values.
}
