% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/testit.R
\name{assert}
\alias{assert}
\alias{\%==\%}
\title{Assertions with an optional message}
\usage{
assert(fact, ...)

x \%==\% y
}
\arguments{
\item{fact}{a message for the assertions when any of them fails; treated the
same way as expressions in \code{...} if it is not a character string,
which means you do not have to provide a message to this function}

\item{...}{any number of R expressions; see Details}

\item{x, y}{two R objects to be compared}
}
\value{
Invisible \code{NULL} if all expressions returned \code{TRUE},
  otherwise an error is signalled and the user-provided message is emitted.
}
\description{
The function \code{assert()} was inspired by \code{\link{stopifnot}()}. It
emits a message in case of errors, which can be a helpful hint for diagnosing
the errors (\code{stopifnot()} only prints the possibly truncated source code
of the expressions).

The infix operator \code{\%==\%} is simply an alias of the
  \code{\link{identical}()} function to make it slightly easier and intuitive
  to write test conditions. \code{x \%==\% y} is the same as
  \code{identical(x, y)}.
}
\details{
There are two ways to write R expressions in the \code{...} argument.

The first way is a series of R expressions (each expression is passed as an
individual argument) that return vectors of \code{TRUE}'s (if \code{FALSE} is
returned anywhere, an error will show up).

The second way is a single R expression wrapped in \code{{}} and passed as a
single argument. This expression may contain multiple sub-expressions. A
sub-expression is treated as a test condition if it is wrapped in \code{()}
(meaning its value will be checked to see if it is a logical vector
containing any \code{FALSE} values) , otherwise it is evaluated in the normal
way and its value will not be checked. If the value of the last
sub-expression is logical, it will also be treated as a test condition.
}
\note{
The internal implementation of \code{assert()} is different with the
  \code{stopifnot()} function in R \pkg{base}: (1) the custom message
  \code{fact} is emitted if an error occurs; (2) \code{assert()} requires the
  logical values to be non-empty (\code{logical(0)} will trigger an error);
  (3) if \code{...} contains a compound expression in \code{{}} that returns
  \code{FALSE} (e.g., \code{if (TRUE) {1+1; FALSE}}), the first and the last
  but one line of the source code from \code{\link{deparse}()} are printed in
  the error message, otherwise the first line is printed; (4) the arguments
  in \code{...} are evaluated sequentially, and \code{assert()} will signal
  an error upon the first failed assertion, and will ignore the rest of
  assertions.
}
\examples{
## The first way to write assertions --------------------

assert("one equals one", 1 == 1)
assert("seq and : produce equal sequences", seq(1L, 10L) == 1L:10L)
assert("seq and : produce identical sequences", identical(seq(1L, 10L), 1L:10L))

# multiple tests
T = FALSE
F = TRUE
assert("T is bad for TRUE, and so is F for FALSE", T != TRUE, F != FALSE)

# a mixture of tests
assert("Let's pray all of them will pass", 1 == 1, 1 != 2, letters[4] == "d", 
    rev(rev(letters)) == letters)

# logical(0) cannot pass assert(), although stopifnot() does not care
try(assert("logical(0) cannot pass", 1 == integer(0)))
stopifnot(1 == integer(0))  # it's OK!

# a compound expression
try(assert("this if statement returns TRUE", if (TRUE) {
    x = 1
    x == 2
}))

# no message
assert(!FALSE, TRUE, is.na(NA))


## The second way to write assertions -------------------

assert("T is bad for TRUE, and so is F for FALSE", {
    T = FALSE
    F = TRUE
    (T != TRUE)  # note the parentheses
    (F != FALSE)
})

assert("A Poisson random number is non-negative", {
    x = rpois(1, 10)
    (x >= 0)
    x > -1  # do not need () here because it's the last expression
})
}
