% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discriminantValidity.R
\name{discriminantValidity}
\alias{discriminantValidity}
\title{Calculate discriminant validity statistics}
\usage{
discriminantValidity(object, cutoff = 0.9, merge = FALSE, level = 0.95,
  boot.ci.type = "perc")
}
\arguments{
\item{object}{The \link[lavaan:lavaan-class]{lavaan::lavaan} model object returned by
the \code{\link[lavaan:cfa]{lavaan::cfa()}} function.}

\item{cutoff}{A cutoff to be used in the constrained models in likelihood
ratio tests.}

\item{merge}{Whether the constrained models should be constructed by merging
two factors as one. Implies \code{cutoff} = 1.}

\item{level}{The confidence level required.}

\item{boot.ci.type}{If bootstrapping was used, the type of interval required.
  The value should be one of \code{"norm"}, \code{"basic"}, \code{"perc"},
  or \code{"bca.simple"}. For the first three options, see the help page of
  the \code{boot.ci} function in the boot package. The
  \code{"bca.simple"} option produces intervals using the adjusted bootstrap
  percentile (BCa) method, but with no correction for acceleration (only for
  bias). Note that the p-value is still computed assuming that the z-statistic
  follows a standard normal distribution.}
}
\value{
A \code{data.frame} of latent variable correlation estimates, their
confidence intervals, and a likelihood ratio tests against constrained models.
with the following attributes:
\describe{
\item{baseline}{The baseline model after possible rescaling.}
\item{constrained}{A \code{list} of the fitted constrained models
used in the likelihood ratio test.}
}
}
\description{
Calculate discriminant validity statistics based on a fitted lavaan object
}
\details{
Evaluated on the measurement scale level, discriminant validity is commonly
evaluated by checking if each pair of latent correlations is sufficiently
below one (in absolute value) that the latent variables can be thought of
representing two distinct constructs.

\code{discriminantValidity} function calculates two sets of statistics that
are commonly used in discriminant validity evaluation. The first set are
factor correlation estimates and their confidence intervals. The second set
is a series of nested model tests, where the baseline model is compared
against a set of constrained models that are constructed by constraining
each factor correlation to the specified cutoff one at a time.

The function assume that the \code{object} is set of confirmatory
factor analysis results where the latent variables are scaled by fixing their
variances to 1s. If the model is not a CFA model, the function will calculate
the statistics for the correlations among exogenous latent variables, but
for the \emph{residual} variances with endogenous variables. If the
latent variables are scaled in some other way (e.g. fixing the first loadings),
the function issues a warning and re-estimates the model by fixing latent
variances to 1 (and estimating all loadings) so that factor covariances are
already estimated as correlations.

The likelihood ratio tests are done by comparing the original baseline model
against more constrained alternatives. By default, these alternatives are
constructed by fixing each correlation at a time to a cutoff value. The
typical purpose of this test is to demonstrate that the estimated factor
correlation is well below the cutoff and a significant \eqn{chi^2} statistic
thus indicates support for discriminant validity. In some cases, the original
correlation estimate may already be greater than the cutoff, making it
redundant to fit a "restricted" model. When this happens, the likelihood
ratio test will be replaced by comparing the baseline model against itself.
For correlations that are estimated to be negative, a negation of the cutoff
is used in the constrained model.

Another alternative is to do a nested model comparison against a model where
two factors are merged as one by setting the \code{merge} argument to
\code{TRUE}. In this comparison, the constrained model is constructed by
removing one of the correlated factors from the model and assigning its
indicators to the factor that remains in the model.
}
\examples{

library(lavaan)

HS.model <- ' visual  =~ x1 + x2 + x3
              textual =~ x4 + x5 + x6
              speed   =~ x7 + x8 + x9 '

fit <- cfa(HS.model, data = HolzingerSwineford1939)
discriminantValidity(fit)
discriminantValidity(fit, merge = TRUE)

}
\references{
Rönkkö, M., & Cho, E. (2022). An updated guideline for assessing
discriminant validity. \emph{Organizational Research Methods}, 25(1), 6–14.
\doi{10.1177/1094428120968614}
}
\author{
Mikko Rönkkö (University of Jyväskylä; \email{mikko.ronkko@jyu.fi}):
}
