% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/autoencoder.R
\docType{class}
\name{AutoEncoder-class}
\alias{AutoEncoder-class}
\alias{AutoEncoder}
\title{AutoEncoder}
\description{
An S4 Class implementing an Autoencoder
}
\details{
Autoencoders are neural networks that try to reproduce their input. Consider
this method unstable, as the internals may still be changed.
}
\section{Slots}{

\describe{
\item{\code{fun}}{A function that does the embedding and returns a
dimRedResult object.}

\item{\code{stdpars}}{The standard parameters for the function.}
}}

\section{General usage}{

Dimensionality reduction methods are S4 Classes that either be used
directly, in which case they have to be initialized and a full
list with parameters has to be handed to the \code{@fun()}
slot, or the method name be passed to the embed function and
parameters can be given to the \code{...}, in which case
missing parameters will be replaced by the ones in the
\code{@stdpars}.
}

\section{Parameters}{

Autoencoder can take the following parameters:
\describe{
  \item{ndim}{The number of dimensions for reduction.}
  \item{n_hidden}{The number of neurons in the hidden
      layers, the length specifies the number of layers,
      the length must be impair, the middle number must
      be the same as ndim.}
  \item{activation}{The activation functions for the layers,
      one of "tanh", "sigmoid", "relu", "elu", everything
      else will silently be ignored and there will be no
      activation function for the layer.}
  \item{weight_decay}{the coefficient for weight decay,
      set to 0 if no weight decay desired.}
  \item{learning_rate}{The learning rate for gradient descend}
  \item{graph}{Optional: A list of bits and pieces that define the
     autoencoder in tensorflow, see details.}
  \item{keras_graph}{Optional: A list of keras layers that define
     the encoder and decoder, specifying this, will ignore all
     other topology related variables, see details.}
  \item{batchsize}{If NA, all data will be used for training,
      else only a random subset of size batchsize will be used}
  \item{n_steps}{the number of training steps.}
}
}

\section{Details}{

There are several ways to specify an autoencoder, the simplest is to pass the
number of neurons per layer in \code{n_hidden}, this must be a vector of
integers of impair length and it must be symmetric and the middle number must
be equal to \code{ndim}, For every layer an activation function can be
specified with \code{activation}.

For regularization weight decay can be specified by setting
\code{weight_decay} > 0.

Currently only a gradient descent optimizer is used, the learning rate can be
specified by setting \code{learning_rate}.
The learner can operate on batches if \code{batchsize} is not \code{NA}.
The number of steps the learner uses is specified using \code{n_steps}.
}

\section{Further training a model}{

If the model did not converge in the first training phase or training with
different data is desired, the \code{\link{dimRedResult}} object may be
passed as \code{autoencoder} parameter; In this case all topology related
parameters will be ignored.
}

\section{Using Keras layers}{

The encoder and decoder part can be specified using a list of \pkg{keras}
layers. This requires a list with two entries, \code{encoder} should contain
a LIST of keras layers WITHOUT the \code{\link[keras]{layer_input}}
that will be concatenated in order to form the encoder part.
\code{decoder} should be
defined accordingly, the output of \code{decoder} must have the same number
of dimensions as the input data.
}

\section{Using Tensorflow}{

The model can be entirely defined in \pkg{tensorflow}, it must contain a
list with the following entries:
\describe{
  \item{encoder}{A tensor that defines the encoder.}
  \item{decoder}{A tensor that defines the decoder.}
  \item{network}{A tensor that defines the reconstruction (encoder + decoder).}
  \item{loss}{A tensor that calculates the loss (network + loss function).}
  \item{in_data}{A \code{placeholder} that points to the data input of
    the network AND the encoder.}
  \item{in_decoder}{A \code{placeholder} that points to the input of
    the decoder.}
  \item{session}{A \pkg{tensorflow} \code{Session} object that holds
    the values of the tensors.}
}
}

\section{Implementation}{

Uses \pkg{tensorflow} as a backend, for details an
  problems relating tensorflow, see \url{https://tensorflow.rstudio.com}.
}

\examples{
\dontrun{
dat <- loadDataSet("3D S Curve")

emb <- embed(dat, "AutoEncoder")

# predicting is possible:
samp <- sample(floor(nrow(dat) / 10))
emb2 <- embed(dat[samp])
emb3 <- predict(emb2, dat[-samp])

plot(emb, type = "2vars")
plot(emb2, type = "2vars")
points(getData(emb3))
}

}
\seealso{
Other dimensionality reduction methods: 
\code{\link{DRR-class}},
\code{\link{DiffusionMaps-class}},
\code{\link{DrL-class}},
\code{\link{FastICA-class}},
\code{\link{FruchtermanReingold-class}},
\code{\link{HLLE-class}},
\code{\link{Isomap-class}},
\code{\link{KamadaKawai-class}},
\code{\link{MDS-class}},
\code{\link{NNMF-class}},
\code{\link{PCA-class}},
\code{\link{PCA_L1-class}},
\code{\link{UMAP-class}},
\code{\link{dimRedMethod-class}},
\code{\link{dimRedMethodList}()},
\code{\link{kPCA-class}},
\code{\link{nMDS-class}},
\code{\link{tSNE-class}}
}
\concept{dimensionality reduction methods}
