#' Plot the main indel contexts
#'
#' @details
#' Plots the number of indels per main COSMIC context per sample.
#' The contexts are not subdivided into the number of repeats/microhomology length.
#' It takes a tibble with counts as its input. This tibble can be generated by count_indel_contexts
#' Each sample is plotted in a separate facet.
#' The same y axis can be used for all samples or a separate y axis can be used.
#'
#' @param counts A tibble containing the number of indels per COSMIC context.
#' @param same_y A boolean describing whether the same y axis should be used for all samples.
#'
#' @return A ggplot figure.
#'
#' @examples
#' ## Get The indel counts
#' ## See 'count_indel_contexts()' for more info on how to do this.
#' indel_counts <- readRDS(system.file("states/blood_indel_counts.rds",
#'   package = "MutationalPatterns"
#' ))
#'
#' ## Plot contexts
#' plot_main_indel_contexts(indel_counts)
#'
#' ## Use the same y axis for all samples.
#' plot_main_indel_contexts(indel_counts, same_y = TRUE)
#' @import ggplot2
#' @importFrom magrittr %>%
#' @family Indels
#'
#' @seealso \code{\link{count_indel_contexts}}, \code{\link{plot_indel_contexts}}
#'
#' @export


plot_main_indel_contexts <- function(counts, same_y = FALSE) {

  # These variables use non standard evaluation.
  # To avoid R CMD check complaints we initialize them to NULL.
  count <- muttype <- muttype_sub <- muttype_total <- NULL

  # Turn matrix into tibble
  counts <- counts %>%
    as.data.frame() %>%
    tibble::rownames_to_column("muttype_total") %>%
    tidyr::separate(muttype_total,
      c("muttype", "muttype_sub"),
      sep = "_(?=[0-9])"
    ) %>%
    dplyr::mutate(muttype = factor(muttype, levels = unique(muttype)))

  # Summarise per muttype and make data long
  counts_main <- counts %>%
    dplyr::select(-muttype_sub) %>%
    dplyr::group_by(muttype) %>%
    dplyr::summarise_all(list(~ sum(.))) %>%
    tidyr::gather(key = "sample", value = "count", -.data$muttype) %>% 
    dplyr::mutate(sample = factor(sample, levels = unique(sample)))

  # Count nr mutations. (This is used for the facets)
  nr_muts <- counts_main %>%
    dplyr::group_by(sample) %>%
    dplyr::summarise(nr_muts = sum(count))

  # Create facet text
  facet_labs_y <- stringr::str_c(nr_muts$sample, " (n = ", nr_muts$nr_muts, ")")
  names(facet_labs_y) <- nr_muts$sample

  # Set plotting parameters
  if (same_y) {
    facet_scale <- "free_x"
  } else {
    facet_scale <- "free"
  }

  # Create figure
  fig <- ggplot(counts_main, aes(x = muttype, y = count, fill = muttype)) +
    geom_bar(stat = "identity") +
    facet_grid(sample ~ ., labeller = labeller(sample = facet_labs_y), scales = facet_scale) +
    labs(x = "", y = "Nr of indels") +
    scale_fill_manual(guide = "none", values = INDEL_COLORS) +
    theme_bw() +
    theme(
      panel.grid.major.x = element_blank(),
      panel.grid.minor.y = element_blank(),
      axis.text.x = element_text(angle = 90, hjust = 1, vjust = 0.5)
    )
  return(fig)
}
