// Copyright (c) 2007, Google Inc.
// All rights reserved.
//
// Author: Li Liu
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
#ifndef CLIENT_LINUX_HANDLER_CPU_CONTEXT_H__
#define CLIENT_LINUX_HANDLER_CPU_CONTEXT_H__

#include <stdint.h>

#include "google_breakpad/common/minidump_format.h"

#ifdef __i386__
typedef MDRawContextX86 RawContext;
#elif defined(__x86_64__)
typedef MDRawContextAMD64 RawContext;
#else
#error "Unsupported platform!"
#endif

// Fowards
// System defined registers.
struct user_regs_struct;
struct user_fpregs_struct;

namespace google_breakpad {

// Fowards
// Debug register set.
struct DebugRegs;

// CPUContext
//
// A wrapper class around the raw cpu context dependent on each CPU type(i386
// and x86_64).
// It provides CPU independant way of accessing common cpu contexts by wrapping
// those details into this class.
class CPUContext {
  public:
    CPUContext(const struct user_regs_struct *user_regs,
               const struct user_fpregs_struct *fp_regs,
               const DebugRegs *debug_regs);

    // Get the frame pointer.
    uintptr_t GetFramePointer() const;

    // Get the stack pointer.
    uintptr_t GetStackPointer() const;

    // Fill the minidump context section with the current context.
    bool CopyTo(RawContext *context) const;

  private:
    void CopyGeneralRegisters(RawContext *context) const;
    void CopyFloatingPointRegisters(RawContext *context) const;
    void CopyDebugRegisters(RawContext *context) const;

    // General purpose registers.
    const struct user_regs_struct *user_regs_;
    // Floating pointers.
    const struct user_fpregs_struct *fp_regs_;
    // Debug registers.
    const DebugRegs *debug_regs_;
};

}  // namespace google_breakpad

#endif  // CLIENT_LINUX_HANDLER_CPU_CONTEXT_H__
