/*
 * ib-vexdm - discover QLogic VEx targets over IB
 *
 * Copyright (c) 2005 Topspin Communications.  All rights reserved.
 * Copyright (c) 2006 Cisco Systems, Inc.  All rights reserved.
 * Copyright (c) 2007 QLogic, Inc.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <endian.h>
#include <byteswap.h>
#include <errno.h>
#include <getopt.h>
#include <infiniband/umad.h>

#include "ib-vexdm.h"

#define IBVEXDM_VERSION "1.3.0.1"
enum {
	QUERY_ALL_PORTS,
	GUID_AVAILABLE,
	LOCAL_HCA_INFO
};

static char *umad_dev   = "/dev/infiniband/umad0";
static char *port_sysfs_path;
static int   timeout_ms = 25000;
static int   max_mad_retries = 3;
static int   node_table_response_size = 1 << 18;
static uint16_t sm_lid;
static uint32_t tid = 1;

static int cmd     = 0;
static int verbose = 0;

static int cmd_vex_str = 0, cmd_vex = 0;

static char *port_state_str[] = {
        "???",
        "Down",
        "Initializing",
        "Armed",
        "Active"
};

#define PFX "\t"

#define pr_human(arg...)					\
	do {							\
		if (!cmd && !cmd_vex && !cmd_vex_str)		\
			printf(PFX arg);			\
	} while (0)

#define pr_cmd_vex(arg...)			\
	do {					\
		if (cmd_vex)			\
			printf(PFX arg);	\
	} while (0)

#define pr_cmd_vex_str(arg...)			\
	do {					\
		if (cmd_vex_str)		\
			printf(arg);		\
	} while (0)

static char *sysfs_path = "/sys";

#define MAD_BLOCK_SIZE 256
#define MAD_RMPP_HDR_SIZE 36
#define get_data_ptr(mad) ((void *) ((mad).hdr.data))

struct vnic_ib_user_mad {
	struct ib_user_mad hdr;
	__u8 data[256];
};

static void usage(const char *argv0)
{
	fprintf(stderr, "Usage: %s [-ves] [-L] [-d <umad device>] [-C <HCA Name>] [-P <Port No>] [-G <Port GUID>][-V]\n", argv0);
	fprintf(stderr, "       -e reports EVIC/VEx IOCGUID information -s reports IOC profile string\n");
	fprintf(stderr, "       -V reports Version information for EVIC/VEx Discovery Tool.\n");
	fprintf(stderr, "       -L reports local HCA information(HCA No, HCA Name, Port No, Port GUID.\n");
	fprintf(stderr, "       -d <umad device> reports EVIC/VEx IOCs reachable through given IB port.\n");
	fprintf(stderr, "       -C <HCA Name> reports EVIC/VEx IOCs reachable from given HCA.\n");
	fprintf(stderr, "       -P <Port No> reports EVIC/VEx IOCs reachable through given IB port.\n");
	fprintf(stderr, "       -G <Port GUID> reports EVIC/VEx IOCs reachable through given port GUID.\n");
	fprintf(stderr, "eg. \n");
	fprintf(stderr, "	ib_qlgc_vnic_query -C mthca0 -P 1	 - EVIC IOCs reachable through mthca0 port 1.\n");
	fprintf(stderr, "	ib_qlgc_vnic_query -G 0x0002c90300000785 - EVIC IOCs reachable through IB port corresponding to given Port GUID.\n");
	fprintf(stderr, "	ib_qlgc_vnic_query -es			 - EVIC IOCs reachable through ALL available IB ports.\n");
}

int send_and_get(int port_id, int agent, struct vnic_ib_user_mad *out_mad,
		 struct vnic_ib_user_mad *in_mad, int in_mad_size)
{
	struct vex_dm_mad *out_dm_mad = (void *) out_mad->data;
	struct vex_dm_mad *in_dm_mad = (void *) in_mad->data;
	int i, len, ret, in_agent;

	in_mad_size = in_mad_size ? in_mad_size : sizeof (struct vnic_ib_user_mad);
	for (i = 0; i < max_mad_retries; ++i) {
		((uint32_t *) &out_dm_mad->tid)[1] = ++tid;

		ret = umad_send(port_id, agent, (void *) out_mad, MAD_BLOCK_SIZE,
				timeout_ms, 0);
		if (ret < 0) {
			fprintf(stderr, PFX "umad_send to %u failed\n",
					(uint16_t) ntohs(out_mad->hdr.addr.lid));
			return ret;
		}

		do {
			len = in_mad_size ? in_mad_size : MAD_BLOCK_SIZE;
			in_agent = umad_recv(port_id, (void *)in_mad, &len, timeout_ms);
			if (in_agent < 0) {
				fprintf(stderr, PFX "umad_recv from %u failed - %d\n", 
					(uint16_t) ntohs(out_mad->hdr.addr.lid), 
					in_agent);
				return in_agent;
			}
			if (in_agent != agent) {
				fprintf(stderr, PFX "umad_recv returned different agent\n");
				continue;
			}

			ret = umad_status((struct ib_user_mad *) in_mad);
			if (ret) {
				fprintf(stderr, PFX
					"bad MAD status (%u) from lid %d\n", 
					ret, (uint16_t) ntohs(out_mad->hdr.addr.lid));
				return -ret;
			} 

			if (len > 0 && in_mad->hdr.status == ETIMEDOUT && verbose > 0)
				fprintf(stderr, PFX "MAD timed out\n"); 

			if (tid != ((uint32_t *) &in_dm_mad->tid)[1] && verbose > 0)
				fprintf(stderr, PFX "umad_recv returned different transaction id sent %d got %d\n", 
					 tid, ((uint32_t *) &in_dm_mad->tid)[1]);

		} while (tid > ((uint32_t *) &in_dm_mad->tid)[1]);

		if (len > 0)
			return len;
	}

	return -1;
}

static int read_file(const char *dir, const char *file, char *buf, size_t size)
{
	char *path;
	int fd;
	int len;

	asprintf(&path, "%s/%s", dir, file);

	fd = open(path, O_RDONLY);
	if (fd < 0)
		return -1;

	len = read(fd, buf, size);

	close(fd);
	free(path);

	if (len > 0 && buf[len - 1] == '\n') {
		--len;
		buf[len] = '\0';
	}

	return len;
}

static int setup_port_sysfs_path(char *ibdev, int* port_num) {
	char *env;
	char class_dev_path[256];
	char ibport[16];
	char *umad_dev_name;

	env = getenv("SYSFS_PATH");
	if (env) {
		int len;

		sysfs_path = strndup(env, 256);
		len = strlen(sysfs_path);
		while (len > 0 && sysfs_path[len - 1] == '/') {
			--len;
			sysfs_path[len] = '\0';
		}
	}

	umad_dev_name = rindex(umad_dev, '/');
	if (!umad_dev_name) {
		fprintf(stderr, PFX "Couldn't find device name in '%s'\n", umad_dev_name);
		return -1;
	}

	snprintf(class_dev_path, sizeof class_dev_path,
		 "%s/class/infiniband_mad/%s", sysfs_path, umad_dev_name);

	if (read_file(class_dev_path, "ibdev", ibdev, sizeof ibdev) < 0) {
		fprintf(stderr, PFX "Couldn't read ibdev attribute\n");
		return -1;
	}

	if (read_file(class_dev_path, "port", ibport, sizeof ibport) < 0) {
		fprintf(stderr, PFX "Couldn't read port attribute\n");
		return -1;
	}

	asprintf(&port_sysfs_path, "%s/class/infiniband/%s/ports/%s",
		 sysfs_path, ibdev, ibport);

	*port_num=atoi(ibport);
	return 0;
}

int create_agent( char * ibdev, int port_num, int* agent, int* port_id)
{
	*port_id = umad_open_port(ibdev, port_num);

	if ( *port_id < 0){
		fprintf(stderr, PFX "umad_open_port failed for device %s port %d\n",
			ibdev, port_num);
		return -ENXIO;
	}

	*agent = umad_register(*port_id, VEX_MGMT_CLASS_SA,
					 VEX_MGMT_CLASS_SA_VERSION,
					 VEX_SA_RMPP_VERSION, 0);

	if (*agent < 0) {
		fprintf(stderr, PFX "umad_register failed\n");
		return *agent;
	}

	return 0;
} 

static void init_vex_dm_mad(struct vnic_ib_user_mad *out_mad, uint32_t agent,
			    uint16_t dlid, uint16_t attr_id, uint32_t attr_mod)
{
	struct vex_dm_mad *out_dm_mad;

	memset(out_mad, 0, sizeof *out_mad);

	out_mad->hdr.agent_id        = agent;
	out_mad->hdr.addr.qpn        = htonl(1);
	out_mad->hdr.addr.qkey       = htonl(0x80010000);
	out_mad->hdr.addr.lid        = htons(dlid);

	out_dm_mad = (void *) out_mad->data;

	out_dm_mad->base_version  = 1;
	out_dm_mad->mgmt_class    = VEX_MGMT_CLASS_DM;
	out_dm_mad->class_version = 1;
	out_dm_mad->method 	  = VEX_DM_METHOD_GET;
	out_dm_mad->attr_id       = htons(attr_id);
	out_dm_mad->attr_mod      = htonl(attr_mod);
}

static int get_iou_info(int port_id, int agent, uint16_t dlid,
			struct vex_dm_iou_info *iou_info)
{
	struct vnic_ib_user_mad		in_mad, out_mad;
	struct vex_dm_mad	       *in_dm_mad;

	init_vex_dm_mad(&out_mad, agent, dlid, VEX_DM_ATTR_IO_UNIT_INFO, 0);

	if (send_and_get(port_id, agent, &out_mad, &in_mad, 0) < 0)
		return -1;

	in_dm_mad = (void *) in_mad.data;
	if (in_dm_mad->status) {
		fprintf(stderr, PFX "IO Unit Info query returned status 0x%04x\n",
			ntohs(in_dm_mad->status));
		return -1;
	}

	memcpy(iou_info, in_dm_mad->data, sizeof *iou_info);

	return 0;
}

static int get_ioc_prof(int port_id, int agent, uint16_t dlid, int ioc,
			struct vex_dm_ioc_prof *ioc_prof)
{
	struct vnic_ib_user_mad		in_mad, out_mad;
	struct vex_dm_mad	       *in_dm_mad;

	init_vex_dm_mad(&out_mad, agent, dlid, VEX_DM_ATTR_IO_CONTROLLER_PROFILE, ioc);

	if (send_and_get(port_id, agent, &out_mad, &in_mad, 0) < 0)
		return -1;

	if (in_mad.hdr.status != 0) {
		fprintf(stderr, PFX "IO Controller Profile query timed out\n");
		return -1;
	}

	in_dm_mad = (void *) in_mad.data;
	if (in_dm_mad->status) {
		fprintf(stderr, PFX "IO Controller Profile query returned status 0x%04x\n",
			ntohs(in_dm_mad->status));
		return -1;
	}

	memcpy(ioc_prof, in_dm_mad->data, sizeof *ioc_prof);

	return 0;
}

static int get_svc_entries(int port_id, int agent, uint16_t dlid, int ioc,
			   int start, int end, struct vex_dm_svc_entries *svc_entries)
{
	struct vnic_ib_user_mad		in_mad, out_mad;
	struct vex_dm_mad	       *in_dm_mad;

	init_vex_dm_mad(&out_mad, agent, dlid, VEX_DM_ATTR_SERVICE_ENTRIES,
			(ioc << 16) | (end << 8) | start);

	if (send_and_get(port_id, agent, &out_mad, &in_mad, 0) < 0)
		return -1;

	if (in_mad.hdr.status != 0) {
		fprintf(stderr, PFX "Service Entries query timed out\n");
		return -1;
	}

	in_dm_mad = (void *) in_mad.data;
	if (in_dm_mad->status) {
		fprintf(stderr, PFX "Service Entries query returned status 0x%04x\n",
			ntohs(in_dm_mad->status));
		return -1;
	}

	memcpy(svc_entries, in_dm_mad->data, sizeof *svc_entries);

	return 0;
}

static void display_iou_info(struct vex_dm_iou_info iou_info, uint16_t dlid,
			     uint64_t subnet_prefix, uint64_t guid)
{
	int i;

	pr_human("IO Unit Info:\n");
	pr_human("    port LID:        %04x\n", dlid);
	pr_human("    port GID:        %016llx%016llx\n",
		 (unsigned long long) subnet_prefix,
		 (unsigned long long) guid);
	pr_human("    change ID:       %04x\n",
		 ntohs(iou_info.change_id));
	pr_human("    max controllers: 0x%02x\n",
		 iou_info.max_controllers);

	pr_human("\n");

	if (verbose > 0)
		for (i = 0; i < iou_info.max_controllers; ++i) {
			pr_human("    controller[%3d]: ", i + 1);
			switch ((iou_info.controller_list[i / 2] >>
				 (4 * (1 - i % 2))) & 0xf) {
			case VEX_DM_NO_IOC:      pr_human("not installed\n"); break;
			case VEX_DM_IOC_PRESENT: pr_human("present\n");       break;
			case VEX_DM_NO_SLOT:     pr_human("no slot\n");       break;
			default:                 pr_human("<unknown>\n");     break;
			}
		}

	pr_human("\n");
}

static int do_port(int port_id, int agent, uint16_t dlid, uint64_t subnet_prefix,
		   uint64_t guid)
{
	struct vex_dm_iou_info		iou_info;
	struct vex_dm_ioc_prof		ioc_prof;
	struct vex_dm_svc_entries	svc_entries;
	int				once = 0;
	int				i, j, k;

	if (get_iou_info(port_id, agent, dlid, &iou_info))
		return 1;

	for (i = 0; i < iou_info.max_controllers; ++i) {
		if (((iou_info.controller_list[i / 2] >> (4 * (1 - i % 2))) & 0xf) ==
		    VEX_DM_IOC_PRESENT) {

			if (get_ioc_prof(port_id, agent, dlid, i + 1, &ioc_prof))
				continue;
			/*Display all information only for VEx devices*/

			if ((ioc_prof.io_class == htons(VEX_IO_CLASS)) && 
				(ioc_prof.io_subclass == htons(VEX_SUB_CLASS)) &&
				(ioc_prof.protocol == htons(VEX_PROTOCOL)) && 
				(ioc_prof.protocol_version == htons(VEX_PROTOCOL_VERSION)) && 
				((ioc_prof.vendor_id >> 8) == htons(VEX_VENDOR_ID))) {
				
				if (!once) {
					display_iou_info(iou_info, dlid, subnet_prefix, guid);
					once = 1;
				}

				pr_human("    controller[%3d]\n", i + 1);

				pr_human("        GUID:      %016llx\n",
					 (unsigned long long) ntohll(ioc_prof.guid));
				pr_human("        vendor ID: %06x\n", ntohl(ioc_prof.vendor_id) >> 8);
				pr_human("        device ID: %06x\n", ntohl(ioc_prof.device_id));
				pr_human("        IO class : %04hx\n", ntohs(ioc_prof.io_class));
				pr_human("        ID:        %s\n", ioc_prof.id);
				pr_human("        service entries: %d\n", ioc_prof.service_entries);

				for (j = 0; j < ioc_prof.service_entries; j += 4) {
					int n;
				
					n = j + 3;
					if (n >= ioc_prof.service_entries)
						n = ioc_prof.service_entries - 1;
					
					if (get_svc_entries(port_id, agent, dlid, i + 1,
							    j, n, &svc_entries))
						continue;

					for (k = 0; k <= n - j; ++k) {
						pr_human("            service[%3d]: %016llx / %s\n",
							 j + k,
							 (unsigned long long) 
								ntohll(svc_entries.service[k].id),
							 svc_entries.service[k].name);
					}
				}

				pr_cmd_vex("ioc_guid=%016llx,"
					    "dgid=%016llx%016llx,"
					    "pkey=ffff",
					    (unsigned long long) ntohll(ioc_prof.guid),
					    (unsigned long long) subnet_prefix,
					    (unsigned long long) guid);

				if (cmd_vex)
					pr_cmd_vex_str(",\"%s\"", ioc_prof.id);
				else
					pr_cmd_vex_str("\"%s\"", ioc_prof.id);

				pr_cmd_vex("\n");

				if (!cmd_vex)
					pr_cmd_vex_str("\n");

				pr_human("\n");
				fflush(stdout);
			}
		}
	}
	
	return 0;
}

static int get_port_info(int port_id, int agent, uint16_t dlid,
			 uint64_t *subnet_prefix, int *isdm)
{
	struct vnic_ib_user_mad		out_mad, in_mad;
	struct vex_dm_rmpp_sa_mad      *out_sa_mad, *in_sa_mad;
	struct vex_dm_mad	       *in_dm_mad;
	struct vex_sa_port_info_rec    *port_info;

	in_sa_mad  = (void *) in_mad.data;
	in_dm_mad  = (void *) in_mad.data;
	out_sa_mad = (void *) get_data_ptr(out_mad);

	init_vex_dm_mad(&out_mad, agent, sm_lid, VEX_SA_ATTR_PORT_INFO, 0);

	out_sa_mad->mgmt_class 	  = VEX_MGMT_CLASS_SA;
	out_sa_mad->class_version = 2;
	out_sa_mad->comp_mask     = htonll(1); /* LID */
	port_info                 = (void *) out_sa_mad->data;
	port_info->endport_lid	  = htons(dlid);

	if (send_and_get(port_id, agent, &out_mad, &in_mad, 0) < 0)
		return -1;

	port_info = (void *) in_sa_mad->data;
	*subnet_prefix = ntohll(port_info->subnet_prefix);
	*isdm          = !!(ntohl(port_info->capability_mask) & (1 << 19));

	return 0;
}

static int do_full_port_list(int port_id, int agent)
{
	uint8_t                         in_mad_buf[node_table_response_size];
	struct vnic_ib_user_mad		out_mad, *in_mad;
	struct vex_dm_rmpp_sa_mad      *out_sa_mad, *in_sa_mad;
	struct vex_sa_node_rec	       *node;
	ssize_t len;
	char val[64];
	int size;
	int i;
	uint64_t subnet_prefix;
	int isdm;
	int sm_lid_try = 0;
	char down_state[] ="1: DOWN";
	int read_state_try = 0;
	int state = -1;

	/* Check the state of the port. If it is down wait for 2 seconds
	 * and read the state again. If it is still down the cable is not
	 * connected and so abort
	 */
 	
	while (read_state_try < 2){

		if (read_file(port_sysfs_path, "state", val, sizeof val) < 0) {
			fprintf(stderr, PFX "Couldn't read PORT STATE\n");
			return -1;
		}
		state = strcmp(val, down_state);

		if (!state && !read_state_try) {
			sleep(2);
			read_state_try++;
		}
		else
			break;
	}

	if (!state){
		fprintf(stderr, PFX "Port state is down. Aborting\n");
		return -1;
	}
				

	if (read_file(port_sysfs_path, "sm_lid", val, sizeof val) < 0) {
		fprintf(stderr, PFX "Couldn't read SM LID\n");
		return -1;
	}

	sm_lid = strtol(val, NULL, 0);

	/* The IB spec says that a LID value of 0x0 is reserved
	 * If we see that the sm_lid is 0x0, err on the side of caution
	 * and read it again after some time to check if we get a non-zero value.
	 */
		
	while ((sm_lid == 0 || sm_lid == 0xffff) && sm_lid_try < 5) {
		fprintf(stderr, PFX "smlid value is 0x%x. Waiting to see if we get a saner value...\n",
				sm_lid);
		sleep(5);
		if (read_file(port_sysfs_path, "sm_lid", val, sizeof val) < 0) {
			fprintf(stderr, PFX "Couldn't read SM LID\n");
			return -1;
		}
		sm_lid = strtol(val, NULL, 0);
		sm_lid_try++;
	}
	if (sm_lid == 0 || sm_lid == 0xffff ) {
		fprintf(stderr, PFX "Got sm_lid value to be 0x%x, Aborting !\n", sm_lid);
		return -1;
	}

	in_mad     = (void *) in_mad_buf;
	in_sa_mad  = (void *) in_mad->data;
	out_sa_mad = (void *) get_data_ptr(out_mad);

	init_vex_dm_mad(&out_mad, agent, sm_lid, VEX_SA_ATTR_NODE, 0);

	out_sa_mad->mgmt_class 	  = VEX_MGMT_CLASS_SA;
	out_sa_mad->method     	  = VEX_SA_METHOD_GET_TABLE;
	out_sa_mad->class_version = 2;
	out_sa_mad->comp_mask     = htonll(1ul << 4); /* node type */
	out_sa_mad->rmpp_version  = 1;
	out_sa_mad->rmpp_type     = 1;
	node                      = (void *) out_sa_mad->data;

	node->type		  = 1; /* CA */

	len = send_and_get(port_id, agent, &out_mad, (struct vnic_ib_user_mad *)in_mad, node_table_response_size);
	if (len < 0)
		return -1;

	if (verbose > 1) {
		fprintf(stdout, PFX "Length %ld from Node Record Table query\n", len);
	}

	size = ntohs(in_sa_mad->attr_offset) * 8;

	for (i = 0; (i + 1) * size <= len - MAD_RMPP_HDR_SIZE; ++i) {
		node = (void *) in_sa_mad->data + i * size;

		if (get_port_info(port_id, agent, ntohs(node->lid),
				  &subnet_prefix, &isdm))
			continue;

		if (!isdm)
			continue;

		do_port(port_id, agent, ntohs(node->lid),
			subnet_prefix, ntohll(node->port_guid));
	}

	if (verbose > 1) {
		fprintf(stdout, PFX "Performed port info query for %d ports\n", (i-1));
	}

	return 0;
}

static int get_node(int port_id, int agent, uint16_t dlid, uint64_t *guid)
{
	struct vnic_ib_user_mad		out_mad, in_mad;
	struct vex_dm_rmpp_sa_mad      *out_sa_mad, *in_sa_mad;
	struct vex_sa_node_rec	       *node;

	in_sa_mad  = (void *) in_mad.data;
	out_sa_mad = (void *) get_data_ptr(out_mad);

	init_vex_dm_mad(&out_mad, agent, sm_lid, VEX_SA_ATTR_NODE, 0);

	out_sa_mad->mgmt_class 	  = VEX_MGMT_CLASS_SA;
	out_sa_mad->class_version = 2;
	out_sa_mad->comp_mask     = htonll(1); /* LID */
	node			  = (void *) out_sa_mad->data;
	node->lid		  = htons(dlid);

	if (send_and_get(port_id, agent, &out_mad, &in_mad, 0) < 0)
		return -1;

	node  = (void *) in_sa_mad->data;
	*guid = ntohll(node->port_guid);

	return 0;
}

static int do_dm_port_list(int port_id, uint32_t agent)
{
	uint8_t                         in_mad_buf[node_table_response_size];
	struct vnic_ib_user_mad		out_mad, *in_mad;
	struct vex_dm_rmpp_sa_mad      *out_sa_mad, *in_sa_mad;
	struct vex_sa_port_info_rec    *port_info;
	ssize_t len;
	int size;
	int i;
	uint64_t guid;

	in_mad     = (void *) in_mad_buf;
	in_sa_mad  = (void *) in_mad->data;
	out_sa_mad = (void *) get_data_ptr(out_mad);

	init_vex_dm_mad(&out_mad, agent, sm_lid, VEX_SA_ATTR_PORT_INFO,
			VEX_SM_CAP_MASK_MATCH_ATTR_MOD);

	out_sa_mad->mgmt_class 	   = VEX_MGMT_CLASS_SA;
	out_sa_mad->method     	   = VEX_SA_METHOD_GET_TABLE;
	out_sa_mad->class_version  = 2;
	out_sa_mad->comp_mask      = htonll(1 << 7); /* Capability mask */
	out_sa_mad->rmpp_version   = 1;
	out_sa_mad->rmpp_type      = 1;
	port_info		   = (void *) out_sa_mad->data;
	port_info->capability_mask = htonl(VEX_IS_DM); /* IsDM */
	
	len = send_and_get(port_id, agent, &out_mad, in_mad, node_table_response_size);
	if (len < 0)
		return -1;

	size = ntohs(in_sa_mad->attr_offset) * 8;

	for (i = 0; (i + 1) * size <= len - MAD_RMPP_HDR_SIZE; ++i) {
		port_info = (void *) in_sa_mad->data + i * size;

		if (get_node(port_id, agent, ntohs(port_info->endport_lid), &guid))
			continue;

		do_port(port_id, agent, ntohs(port_info->endport_lid),
			ntohll(port_info->subnet_prefix), guid);
	}
	if (verbose > 1) {
		fprintf(stdout, PFX "Performed port info query for %d ports\n", (i-1));
	}

	return 0;
}

static int check_sm_cap(int port_id, int agent, int *mask_match)
{
        struct vnic_ib_user_mad              out_mad, in_mad;
        struct vex_dm_rmpp_sa_mad      *out_sa_mad, *in_sa_mad;
        struct vex_class_port_info     *cpi;

        in_sa_mad  = (void *) in_mad.data;
		out_sa_mad = (void *) get_data_ptr(out_mad);
	
		init_vex_dm_mad(&out_mad, agent, sm_lid, VEX_DM_ATTR_CLASS_PORT_INFO, 0);

        out_sa_mad->mgmt_class    = VEX_MGMT_CLASS_SA;
        out_sa_mad->class_version = 2;
        if (send_and_get(port_id, agent, &out_mad, &in_mad, 0) < 0)
                return -1;

        cpi = (void *) in_sa_mad->data;

        *mask_match = !!(ntohs(cpi->cap_mask) & VEX_SM_SUPPORTS_MASK_MATCH);

        return 0;
}

int run_query(char *ibdev, int port_num)
{
	int agent, mask_match, state = -1, port_id, sm_lid_try = 0, read_state_try = 0;
	char down_state[] ="1: DOWN", val[16];

	asprintf(&port_sysfs_path, "%s/class/infiniband/%s/ports/%d",
		 sysfs_path, ibdev, port_num);

	if (create_agent(ibdev, port_num, &agent, &port_id))
		return 1;
	

	/* Check the state of the port. If it is down wait for 2 seconds
	 * and read the state again. If it is still down the cable is not
	 * connected and so abort
	 */
 	
	while (read_state_try < 2){

		if (read_file(port_sysfs_path, "state", val, sizeof val) < 0) {
			fprintf(stderr, PFX "Couldn't read PORT STATE\n");
			return -1;
		}
		state = strcmp(val, down_state);

		if (!state && !read_state_try) {
			sleep(2);
			read_state_try++;
		}
		else
			break;
	}

	if (!state){
		fprintf(stderr, PFX "Port state is down. Aborting\n");
		return -1;
	}
				

	if (read_file(port_sysfs_path, "sm_lid", val, sizeof val) < 0) {
		fprintf(stderr, PFX "Couldn't read SM LID\n");
		return -1;
	}

	sm_lid = strtol(val, NULL, 0);

	/* The IB spec says that a LID value of 0x0 is reserved
	 * If we see that the sm_lid is 0x0, err on the side of caution
	 * and read it again after some time to check if we get a non-zero value.
	 */
		
	while ((sm_lid == 0 || sm_lid == 0xffff) && sm_lid_try < 5) {
		fprintf(stderr, PFX "smlid value is 0x%x. Waiting to see if we get a saner value...\n",
				sm_lid);
		sleep(5);
		if (read_file(port_sysfs_path, "sm_lid", val, sizeof val) < 0) {
			fprintf(stderr, PFX "Couldn't read SM LID\n");
			return -1;
		}
		sm_lid = strtol(val, NULL, 0);
		sm_lid_try++;
	}
	if (sm_lid == 0 || sm_lid == 0xffff ) {
		fprintf(stderr, PFX "Got sm_lid value to be 0x%x, Aborting !\n", sm_lid);
		return -1;
	}

	if (check_sm_cap(port_id, agent, &mask_match))
		return 1;

	if (mask_match)
		do_dm_port_list( port_id, agent);
	else
		do_full_port_list(port_id, agent);

	return 0;
}

int get_hca_no(char *hca_name)
{
	int i, len;
	char *ptr;

	len = strlen(hca_name);

	for (i = len - 1; i >= 0; i--)
		if (!isdigit(hca_name[i]))
			break;

	ptr = hca_name + i + 1;
	return atoi(ptr);
}

int get_hca_name(int hca_no, char *hca_name)
{
	int n, i;
	char names[20][UMAD_CA_NAME_LEN];
	umad_ca_t hca;

	n = umad_get_cas_names((void *)names, UMAD_CA_NAME_LEN);
	if (n < 0) {
		fprintf(stderr, PFX "Couldn't list IB device names.\n");
		return 1;
	}

	for (i = 0; i < n; i++) {
		if (umad_get_ca(names[i], &hca) < 0) {
			fprintf(stderr, PFX "Error: retrieving %s attributes.\n", names[i]);
			return 1;
		}

		if (hca_no == get_hca_no(names[i])) {
			strncpy(hca_name, names[i], UMAD_CA_NAME_LEN);
			return 0;
		}
	}

	fprintf(stderr, PFX "Invalid HCA No = %d input.\n", hca_no);
	return 1;
}

int get_ca_information(char *hca_name)
{
	umad_ca_t hca;
	int i;

	if (umad_get_ca(hca_name, &hca) < 0) {
		fprintf(stderr, PFX "Error: retrieving %s attributes.\n\n", hca_name);
		return 1;
	}

	for (i = 1; i <= hca.numports; i++) {
		if (hca.ports[i]->state == 4) { /* Port State Active */
			printf("HCA No = %d, HCA = %s, Port = %d, Port GUID = 0x%016llx\n\n",
				get_hca_no(hca_name), hca_name, i,
				(long long unsigned)ntohll(hca.ports[i]->port_guid));
			return run_query(hca_name, i);
		}
	}
	fprintf(stderr, PFX "None of the port is in Active State. Skipping search of"
		" DM nodes on the HCA = %s.\n\n", hca_name);
	return 1;
}

int get_ca_port_information(long long unsigned port_guid, int flag)
{
	int n, i, j;
	char names[20][UMAD_CA_NAME_LEN];
	umad_ca_t hca;

	n = umad_get_cas_names((void *)names, UMAD_CA_NAME_LEN);
	if (n < 0) {
		fprintf(stderr, PFX "Couldn't list IB device names.\n");
		return 1;
	}

	for (i = 0; i < n; i++) {
		if (umad_get_ca(names[i], &hca) < 0) {
			fprintf(stderr, PFX "Error: retrieving %s attributes.\n", names[i]);
			return 1;
		}

		for (j = 1; j <= hca.numports; j++) {
			if (flag & GUID_AVAILABLE) { 
				if (port_guid == (long long unsigned)ntohll(hca.ports[j]->port_guid)) {
					printf("HCA No = %d, HCA = %s, Port = %d, Port GUID = 0x%016llx, "
					       "State = %s\n\n", get_hca_no(names[i]), names[i], j,
						port_guid, port_state_str[hca.ports[j]->state]);
					if (hca.ports[j]->state == 4)	/* Search only if port Active. */
						return run_query(names[i], j);
					else
						fprintf(stderr, PFX "Port State is %s. Skipping search"
							" of DM nodes on this port.\n\n",
							port_state_str[hca.ports[j]->state]);
				}
			} else if (flag & LOCAL_HCA_INFO) {
				printf("%d,%s,%d,0x%016llx\n", get_hca_no(names[i]), names[i], j,
					(long long unsigned)ntohll(hca.ports[j]->port_guid));
			} else {
				printf("HCA No = %d, HCA = %s, Port = %d, Port GUID = 0x%016llx, "
				       "State = %s\n\n", get_hca_no(names[i]), names[i], j,
					(long long unsigned)ntohll(hca.ports[j]->port_guid),
					port_state_str[hca.ports[j]->state]);
				if (hca.ports[j]->state == 4) {
					if (run_query(names[i],j))
						fprintf(stderr, PFX "Error while finding DM nodes on "
					       	       "HCA = %s, Port = %d\n\n",names[i], j);
				} else
					fprintf(stderr, PFX "Port State is %s. Skipping search of "
						"DM nodes on this port.\n\n",
						port_state_str[hca.ports[j]->state]);
			}
		}
	}

	return 0;
}

int main(int argc, char *argv[])
{
	int port_num = 1, umad_dev_input = 0, hca_input = 0;
	int local_hca_info = 0, port_input = 0, guid_input = 0;
	char *cmd_name = strdup(argv[0]);
	char ibdev[16];
	long long unsigned port_guid;

	while (1) {
		int c;

		c = getopt(argc, argv, "LVevsd:C:P:G:");
		if (c == -1)
			break;

		switch (c) {
		case 'd':
			++umad_dev_input;
			umad_dev = optarg;
			break;

		case 'V':
			printf("QLogic VNIC EVIC/VEx Discovery Tool. Version : %s\n", IBVEXDM_VERSION);
			exit(0);

		case 'e':
			++cmd_vex;
			break;

		case 's':
			++cmd_vex_str;
			break;

		case 'v':
			++verbose;
			break;
		case 'C':
			++hca_input;
			strncpy(ibdev, optarg, 16);
			break;
		case 'P':
			++port_input;
			port_num = atoi(optarg);
			break;
		case 'G':
			++guid_input;
			sscanf(optarg, "0x%016llx\n", &port_guid);
			break;
		case 'L':
			++local_hca_info;
			break;
		default:
			usage(cmd_name);
			return 1;
		}
	}

	int ret = -1;
	if (umad_init() < 0) {
		fprintf(stderr, "Couldn't initialize UMAD library.\n");
		return 1;
	}

	printf("\n");
	if (umad_dev_input) {
		if ((ret = setup_port_sysfs_path(ibdev, &port_num)))
			goto end;
	
		printf("HCA No = %d, HCA = %s, Port = %d\n\n",
			get_hca_no(ibdev), ibdev, port_num);	
		ret = run_query(ibdev, port_num);
		goto end;
	}

	if (guid_input) {
		ret = get_ca_port_information(port_guid, GUID_AVAILABLE);
		goto end;
	}

	if (local_hca_info) {
		ret = get_ca_port_information(0, LOCAL_HCA_INFO);
		goto end;
	}

	if (hca_input) {
		if (port_input)
			ret = run_query(ibdev, port_num);
		else
			ret = get_ca_information(ibdev);
		goto end;
	}

	if (port_input && !hca_input) {
		if (!get_hca_name(0, ibdev)) { /* Default HCA - HCA 0 */
			printf("HCA No = %d, HCA = %s, Port = %d\n\n",
				get_hca_no(ibdev), ibdev, port_num);	
			ret = run_query(ibdev, port_num);
		}
		goto end;
	}

	ret = get_ca_port_information(0, QUERY_ALL_PORTS);
end:
	printf("\n");
	umad_done();
	return ret;
}
