# nmblookup.py
# Copyright (C) 2008-2009 Stefan J. Betz <stefan_betz@gmx.net>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from re import compile
from copy import copy
from subprocess import Popen, PIPE
from pyneighborhood import config, db, dblock
from pyneighborhood.misc import print_debug
from socket import gethostbyaddr

masterRe = compile("\n((?:\d{1,3}\.){3}\d{1,3})")
lookupRe = compile("\n\t([^\s]*)\s*(<(?:20|1e)>)")
workgroupcommand = [ config.get("Main", "nmblookup") ]
lookupcommand = [ config.get("Main", "nmblookup"), "-A" ]
mastercommand = [ config.get("Main", "nmblookup"), "-M", "--", "-" ]

def lookup(ip):
    """
    Lookups SMB name and workgroup
    
    returns: (name, workgroup)
    """
    command = copy(lookupcommand)
    command.append(ip)
    process = Popen( command, executable = command[0],
                     stdout = PIPE,
                     stderr = PIPE )
    process.wait()
    output = process.stdout.read()
    match = lookupRe.search(output)
    
    # preinitialize name & workgroup variables
    # to avoid reference before assignment!
    name = None
    workgroup = None
    
    while match:
        print_debug("%s" % match.group(0))
        if match.group(2) == "<20>":
            name = match.group(1)
        elif match.group(2) == "<1e>":
            workgroup = match.group(1)
        match = lookupRe.search(output, match.end())
    del match
    return (name, workgroup)

def query_masters():
    """
    A Generator which queries all Master Browsers.
    """
    process = Popen(mastercommand, stdout=PIPE)
    process.wait()
    stdout = process.stdout.read()
    process.stdout.close()
    match = masterRe.search(stdout)
    while match:
        yield match.group(1)
        match = masterRe.search(stdout, match.end())

def query_workgroup(ip, manual=False):
    """
    Resolve workgroup of the host 'ip' and add workgroup to database.

    returns: (workgroup, id)
    """
    # Step 1: nmblookup -A ip (hostlookupcommand)
    # Step 2: Check if workgroup already in database
    manual = 1 if manual else 0
    name, workgroup = lookup(ip)
    cursor = db.cursor()
    result = cursor.execute("""SELECT id FROM workgroups WHERE name=?;""", (workgroup,)).fetchall()
    if not result:
        dblock.acquire()
        cursor.execute("""INSERT INTO workgroups (name,manual) VALUES (?,?);""", (workgroup,manual,))
        dblock.release()
    elif len(result) == 1:
        dblock.acquire()
        cursor.execute("""UPDATE workgroups SET validated = 1 WHERE id = ?""", (result[0][0],))
        dblock.release()
    cursor.execute("""SELECT * FROM workgroups WHERE name=?;""", (workgroup,))
    return (workgroup, cursor.fetchone()[0])

def query_workgroup_hosts(workgroup):
    """
    Query all hosts of 'workgroup'.

    returns: IP List of Hosts.
    """
    # Step 1: nmblookup workgroup (workgrouplookupcommand)
    command = copy(workgroupcommand)
    command.append(workgroup)
    process = Popen(command, stdout=PIPE)
    process.wait()
    stdout = process.stdout.read()
    process.stdout.close()
    match = masterRe.search(stdout)
    while match:
        yield match.group(1)
        match = masterRe.search(stdout, match.end())

def query_host(ip, manual=False):
    """
    Query the host 'ip' and added it to the database.

    returns: None
    """
    print_debug("Querying host %s..." % ip)
    # Step 1: Check if Host is already in Database
    manual = 1 if manual else 0
    cursor = db.cursor()
    result = cursor.execute("""SELECT id,ip,name FROM hosts WHERE ip = ?""", ( ip, )).fetchall()
    if len(result) == 1:
        dblock.acquire()
        cursor.execute("""UPDATE hosts SET validated = 1 WHERE id = ?""", (result[0][0],))
        dblock.release()
        print_debug("...already in database!")
        return
    # Step 2: Create a new Database Entry...
    print_debug("...looking up hostname & IP...")
    name, workgroup = lookup(ip)
    if not name or not workgroup:
        print_debug("...could not retrieve samba name or workgroup!")
        return
    workgroupid = cursor.execute("""SELECT id FROM workgroups WHERE name = ?""", ( workgroup, )).fetchone()[0]
    try:
        hostname = gethostbyaddr(ip)[0]
    except:
        hostname = ""
    print_debug("... workgroup = %s, hostname = %s, name = %s..." % ( workgroup, hostname, name ))
    dblock.acquire()
    cursor.execute("""INSERT INTO hosts ( ip,name,workgroup,hostname,manual ) VALUES ( ?, ?, ?, ?, ? )""", ( ip, name, workgroupid, hostname, manual, ))
    dblock.release()
    print_debug("...done")
