#!/usr/bin/python

"""Read power meters available via ACPI."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import discovery
import pwrkap_data
import datetime
import os
import dircache
import util

class acpi_power_meter(pwrkap_data.power_meter):
	"""Driver for battery power meters available via ACPI."""

	def __init__(self, sensor_filename):
		self.sensor_filename = sensor_filename
		self.latency = None

	def read(self):
		def is_discharging(acpi_data):
			"""Return true if the battery is discharging."""
			discharge_data = ["charging", "state:", "discharging"]

			for line in acpi_data:
				if line == discharge_data:
					return True
			return False
		def extract_rate(acpi_data):
			"""Return discharge rate."""
			for line in acpi_data:
				if line[0] == "present" and line[1] == "rate:":
					return int(line[2]) / 1000.0
			return None
		try:
			before = datetime.datetime.utcnow()
			x = util.read_file_as_array(self.sensor_filename)
			if x == None:
				return None
			if not is_discharging(x):
				return None
			return extract_rate(x)
		finally:
			after = datetime.datetime.utcnow()
			if self.latency == None:
				self.latency = (after - before)
			else:
				self.latency = (8 * self.latency + 2 * (after - before)) / 10

	def get_latency(self):
		return self.latency

	def inventory(self):
		return ("acpimeter", {"name": self.sensor_filename})

SYSFS_ACPI_DIR = "/proc/acpi/battery/"
def acpi_meter_discover():
	"""Discover ACPI meters."""
	global SYSFS_ACPI_DIR

	os.system("modprobe -q battery")

	# Now look for /proc/acpi/battery/BAT*/state
	for hwmon_dev in dircache.listdir(SYSFS_ACPI_DIR):
		if not hwmon_dev.startswith("BAT"):
			continue
		hwmon_dev_dir = SYSFS_ACPI_DIR + hwmon_dev + "/state"
		meter = acpi_power_meter(hwmon_dev_dir)
		discovery.PWRKAP_POWER_METERS.append(meter)

def acpi_meter_init():
	"""Set up acpi meter discovery functions."""
	discovery.PWRKAP_METER_DISCOVERY.append(acpi_meter_discover)

	return True

acpi_meter_init()
