import pytest
from utils import *

import numpy as np
import psi4
from psi4.driver import qcdb


def vibanal_str(mass, coord, fcm, dipder=None, hess=None, project_trans=True, project_rot=True):
    """Vibrational analysis driver similar to psi4.driver.driver.vibanal_wfn only
    takes in arrays more directly than from a core.Wavefunction.

    """
    if hess is None:
        nmwhess = qcdb.hessparse.load_hessian(fcm, dtype='fcmfinal')
    else:
        nmwhess = hess

    if dipder is not None:
        dipder = np.asarray(dipder)

    mol = psi4.geometry(coord)
    m = np.asarray(mass)  # not good permanent
    geom = np.asarray(mol.geometry())
    symbols = [mol.symbol(at) for at in range(mol.natom())]
    irrep_labels = mol.irrep_labels()

    wfn = psi4.core.Wavefunction.build(mol, "STO-3G")  # dummy, obviously. only used for SALCs
    basisset = wfn.basisset()

    vibinfo, vibtext = qcdb.vib.harmonic_analysis(
        nmwhess, geom, m, basisset, irrep_labels, dipder, project_trans=project_trans, project_rot=project_rot)
    print(vibtext)
    print(qcdb.vib.print_vibs(vibinfo, shortlong=True, normco='q', atom_lbl=symbols))

    return vibinfo


# yapf:disable

# <<<  Cfour Ethene  >>>

c4_eth_xyz = """
     C                  1.24804346     0.00000000     0.00000000
     C                 -1.24804346     0.00000000     0.00000000
     H                  2.32303862     1.74354469     0.00000000
     H                  2.32303862    -1.74354469     0.00000000
     H                 -2.32303862     1.74354469     0.00000000
     H                 -2.32303862    -1.74354469     0.00000000
     units au
"""

c4_eth_mass = [12.000000000, 12.000000000, 1.007825035, 1.007825035, 1.007825035, 1.007825035]

c4_eth_dipder = [
    [  -0.0109113955,       0.0000000000,       0.0000000000,
       -0.0109113955,       0.0000000000,       0.0000000000,
        0.0054556978,      -0.0873021078,       0.0000000000,
        0.0054556978,       0.0873021078,       0.0000000000,
        0.0054556978,       0.0873021078,       0.0000000000,
        0.0054556978,      -0.0873021078,       0.0000000000],
    [   0.0000000000,       0.1296195826,       0.0000000000,
        0.0000000000,       0.1296195826,       0.0000000000,
       -0.0490863766,      -0.0648097913,       0.0000000000,
        0.0490863766,      -0.0648097913,       0.0000000000,
        0.0490863766,      -0.0648097913,       0.0000000000,
       -0.0490863766,      -0.0648097913,       0.0000000000],
    [   0.0000000000,       0.0000000000,      -0.2891402044,
        0.0000000000,       0.0000000000,      -0.2891402044,
        0.0000000000,       0.0000000000,       0.1445701022,
        0.0000000000,       0.0000000000,       0.1445701022,
        0.0000000000,       0.0000000000,       0.1445701022,
        0.0000000000,       0.0000000000,       0.1445701022]]

c4_eth_fcm = """
    6   18
        0.9979858052        0.0000000000        0.0000000000
       -0.6809467692        0.0000000000        0.0000000000
       -0.1462470353       -0.1371517395        0.0000000000
       -0.1462470353        0.1371517395        0.0000000000
       -0.0122724826       -0.0036710828        0.0000000000
       -0.0122724826        0.0036710828        0.0000000000
        0.0000000000        0.7089049217        0.0000000000
        0.0000000000       -0.1306667067        0.0000000000
       -0.1304888217       -0.2934169318        0.0000000000
        0.1304888217       -0.2934169318        0.0000000000
        0.0343100734        0.0042978243        0.0000000000
       -0.0343100734        0.0042978243        0.0000000000
        0.0000000000        0.0000000000        0.1351284712
        0.0000000000        0.0000000000       -0.0598756949
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000        0.0073781116
       -0.6809467692        0.0000000000        0.0000000000
        0.9979858052        0.0000000000        0.0000000000
       -0.0122724826        0.0036710828        0.0000000000
       -0.0122724826       -0.0036710828        0.0000000000
       -0.1462470353        0.1371517395        0.0000000000
       -0.1462470353       -0.1371517395        0.0000000000
        0.0000000000       -0.1306667067        0.0000000000
        0.0000000000        0.7089049217        0.0000000000
       -0.0343100734        0.0042978243        0.0000000000
        0.0343100734        0.0042978243        0.0000000000
        0.1304888217       -0.2934169318        0.0000000000
       -0.1304888217       -0.2934169318        0.0000000000
        0.0000000000        0.0000000000       -0.0598756949
        0.0000000000        0.0000000000        0.1351284712
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000       -0.0450044998
       -0.1462470353       -0.1304888217        0.0000000000
       -0.0122724826       -0.0343100734        0.0000000000
        0.1518654225        0.1484742948        0.0000000000
        0.0112766553        0.0153816024        0.0000000000
        0.0015013358        0.0002775169        0.0000000000
       -0.0061238955        0.0006654810        0.0000000000
       -0.1371517395       -0.2934169318        0.0000000000
        0.0036710828        0.0042978243        0.0000000000
        0.1484742948        0.3028992171        0.0000000000
       -0.0153816024       -0.0172298679        0.0000000000
       -0.0002775169        0.0015472586        0.0000000000
        0.0006654810        0.0019024996        0.0000000000
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000        0.0287456462
        0.0000000000        0.0000000000        0.0041387542
        0.0000000000        0.0000000000       -0.0099324434
        0.0000000000        0.0000000000        0.0146744312
       -0.1462470353        0.1304888217        0.0000000000
       -0.0122724826        0.0343100734        0.0000000000
        0.0112766553       -0.0153816024        0.0000000000
        0.1518654225       -0.1484742948        0.0000000000
       -0.0061238955       -0.0006654810        0.0000000000
        0.0015013358       -0.0002775169        0.0000000000
        0.1371517395       -0.2934169318        0.0000000000
       -0.0036710828        0.0042978243        0.0000000000
        0.0153816024       -0.0172298679        0.0000000000
       -0.1484742948        0.3028992171        0.0000000000
       -0.0006654810        0.0019024996        0.0000000000
        0.0002775169        0.0015472586        0.0000000000
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000        0.0041387542
        0.0000000000        0.0000000000        0.0287456462
        0.0000000000        0.0000000000        0.0146744312
        0.0000000000        0.0000000000       -0.0099324434
       -0.0122724826        0.0343100734        0.0000000000
       -0.1462470353        0.1304888217        0.0000000000
        0.0015013358       -0.0002775169        0.0000000000
       -0.0061238955       -0.0006654810        0.0000000000
        0.1518654225       -0.1484742948        0.0000000000
        0.0112766553       -0.0153816024        0.0000000000
       -0.0036710828        0.0042978243        0.0000000000
        0.1371517395       -0.2934169318        0.0000000000
        0.0002775169        0.0015472586        0.0000000000
       -0.0006654810        0.0019024996        0.0000000000
       -0.1484742948        0.3028992171        0.0000000000
        0.0153816024       -0.0172298679        0.0000000000
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000       -0.0099324434
        0.0000000000        0.0000000000        0.0146744312
        0.0000000000        0.0000000000        0.0287456462
        0.0000000000        0.0000000000        0.0041387542
       -0.0122724826       -0.0343100734        0.0000000000
       -0.1462470353       -0.1304888217        0.0000000000
       -0.0061238955        0.0006654810        0.0000000000
        0.0015013358        0.0002775169        0.0000000000
        0.0112766553        0.0153816024        0.0000000000
        0.1518654225        0.1484742948        0.0000000000
        0.0036710828        0.0042978243        0.0000000000
       -0.1371517395       -0.2934169318        0.0000000000
        0.0006654810        0.0019024996        0.0000000000
       -0.0002775169        0.0015472586        0.0000000000
       -0.0153816024       -0.0172298679        0.0000000000
        0.1484742948        0.3028992171        0.0000000000
        0.0000000000        0.0000000000        0.0073781116
        0.0000000000        0.0000000000       -0.0450044998
        0.0000000000        0.0000000000        0.0146744312
        0.0000000000        0.0000000000       -0.0099324434
        0.0000000000        0.0000000000        0.0041387542
        0.0000000000        0.0000000000        0.0287456462
"""

ref_eth_vibonly = {

    # freq from cfour after proj
    'omega': qcdb.vib.QCAspect('', '', np.asarray([885.4386, 1073.4306, 1080.4104, 1135.9390, 1328.7166,  1467.8428, 1565.5328, 1831.6445, 3287.3173, 3312.1889, 3371.7974, 3399.4232]), ''),
    'gamma': qcdb.vib.QCAspect('', '', ['B2u', 'B1u', 'B2g', 'Au', 'B1g', 'Ag', 'B3u', 'Ag', 'B3u', 'Ag', 'B1g', 'B2u'], ''),
    'IR_intensity': qcdb.vib.QCAspect('', '', np.array([0.2122, 94.4517, 0.0000, 0.0000, 0.0000, 0.0000, 10.3819, 0.0000, 19.2423, 0.0000, 0.0000, 28.0922]), ''),  # [km/mol]

    # from vibsuite after cfour
    'mu': qcdb.vib.QCAspect('', '', np.asarray([1.0423,  1.1607,  1.5200,  1.0078,  1.5276,  1.2029,  1.1122,  3.2209,  1.0475,  1.0799,  1.1150,  1.1182]), ''),
    'k': qcdb.vib.QCAspect('', '', np.asarray([0.4814, 0.7880, 1.0454, 0.7662, 1.5890, 1.5270, 1.6060, 6.3666, 6.6695, 6.9800, 7.4688, 7.6133]), ''),
    'Qtp0': qcdb.vib.QCAspect('', '', np.asarray([0.3687, 0.3349, 0.3338, 0.3256, 0.3010, 0.2864, 0.2773, 0.2564, 0.1914, 0.1907, 0.1890, 0.1882]), ''),
    'DQ0': qcdb.vib.QCAspect('', '', np.asarray([0.2607, 0.2368, 0.2360, 0.2302, 0.2128, 0.2025, 0.1961, 0.1813, 0.1353, 0.1348, 0.1336, 0.1331]), ''),

    # q from vibsuite after cfour
    'q': qcdb.vib.QCAspect('', '', np.asarray(
           [[  0.00000000E+00, -1.34306554E-01,  0.00000000E+00,  0.00000000E+00, -1.34306554E-01,  0.00000000E+00, -4.32765855E-01,  2.31720930E-01,  0.00000000E+00,  4.32765855E-01,  2.31720930E-01,  0.00000000E+00,  4.32765855E-01,  2.31720930E-01,  0.00000000E+00, -4.32765855E-01,  2.31720930E-01,  0.00000000E+00],
            [  0.00000000E+00,  0.00000000E+00, -2.68155030E-01,  0.00000000E+00,  0.00000000E+00, -2.68155030E-01,  0.00000000E+00,  0.00000000E+00,  4.62651532E-01,  0.00000000E+00,  0.00000000E+00,  4.62651532E-01,  0.00000000E+00,  0.00000000E+00,  4.62651532E-01,  0.00000000E+00,  0.00000000E+00,  4.62651532E-01],
            [  0.00000000E+00,  0.00000000E+00, -4.28876240E-01,  0.00000000E+00,  0.00000000E+00,  4.28876240E-01,  0.00000000E+00,  0.00000000E+00,  3.97533125E-01,  0.00000000E+00,  0.00000000E+00,  3.97533125E-01,  0.00000000E+00,  0.00000000E+00, -3.97533125E-01,  0.00000000E+00,  0.00000000E+00, -3.97533125E-01],
            [  0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00, -5.00000000E-01,  0.00000000E+00,  0.00000000E+00,  5.00000000E-01,  0.00000000E+00,  0.00000000E+00,  5.00000000E-01,  0.00000000E+00,  0.00000000E+00, -5.00000000E-01],
            [  0.00000000E+00,  4.30954858E-01,  0.00000000E+00,  0.00000000E+00, -4.30954858E-01,  0.00000000E+00,  3.79464410E-01, -1.14654771E-01,  0.00000000E+00, -3.79464410E-01, -1.14654771E-01,  0.00000000E+00,  3.79464410E-01,  1.14654771E-01,  0.00000000E+00, -3.79464410E-01,  1.14654771E-01,  0.00000000E+00],
            [ -2.97546697E-01,  0.00000000E+00,  0.00000000E+00,  2.97546697E-01,  0.00000000E+00,  0.00000000E+00, -4.15835441E-01,  1.81145984E-01,  0.00000000E+00, -4.15835441E-01, -1.81145984E-01,  0.00000000E+00,  4.15835441E-01,  1.81145984E-01,  0.00000000E+00,  4.15835441E-01, -1.81145984E-01,  0.00000000E+00],
            [  2.26323959E-01,  0.00000000E+00,  0.00000000E+00,  2.26323959E-01,  0.00000000E+00,  0.00000000E+00, -3.90479813E-01,  2.68168321E-01,  0.00000000E+00, -3.90479813E-01, -2.68168321E-01,  0.00000000E+00, -3.90479813E-01, -2.68168321E-01,  0.00000000E+00, -3.90479813E-01,  2.68168321E-01,  0.00000000E+00],
            [  6.12410369E-01,  0.00000000E+00,  0.00000000E+00, -6.12410369E-01,  0.00000000E+00,  0.00000000E+00, -1.24747482E-01,  2.16598328E-01,  0.00000000E+00, -1.24747482E-01, -2.16598328E-01,  0.00000000E+00,  1.24747482E-01,  2.16598328E-01,  0.00000000E+00,  1.24747482E-01, -2.16598328E-01,  0.00000000E+00],
            [  1.43821368E-01,  0.00000000E+00,  0.00000000E+00,  1.43821368E-01,  0.00000000E+00,  0.00000000E+00, -2.48136969E-01, -4.22002075E-01,  0.00000000E+00, -2.48136969E-01,  4.22002075E-01,  0.00000000E+00, -2.48136969E-01,  4.22002075E-01,  0.00000000E+00, -2.48136969E-01, -4.22002075E-01,  0.00000000E+00],
            [  1.90838944E-01,  0.00000000E+00,  0.00000000E+00, -1.90838944E-01,  0.00000000E+00,  0.00000000E+00, -2.48030143E-01, -4.12639427E-01,  0.00000000E+00, -2.48030143E-01,  4.12639427E-01,  0.00000000E+00,  2.48030143E-01, -4.12639427E-01,  0.00000000E+00,  2.48030143E-01,  4.12639427E-01,  0.00000000E+00],
            [  0.00000000E+00,  2.29057027E-01,  0.00000000E+00,  0.00000000E+00, -2.29057027E-01,  0.00000000E+00, -2.51182885E-01, -4.00841113E-01,  0.00000000E+00,  2.51182885E-01, -4.00841113E-01,  0.00000000E+00, -2.51182885E-01,  4.00841113E-01,  0.00000000E+00,  2.51182885E-01,  4.00841113E-01,  0.00000000E+00],
            [  0.00000000E+00, -2.32096682E-01,  0.00000000E+00,  0.00000000E+00, -2.32096682E-01,  0.00000000E+00,  2.50427065E-01,  4.00439571E-01,  0.00000000E+00, -2.50427065E-01,  4.00439571E-01,  0.00000000E+00, -2.50427065E-01,  4.00439571E-01,  0.00000000E+00,  2.50427065E-01,  4.00439571E-01,  0.00000000E+00]]
    ).T, ''),
}


# <<<  Cfour Ammonia  >>>

c4_nh3_xyz = """
     N                  0.09284449     0.00000000     0.00000000
     H                 -0.43000462     0.91560345    -1.58587170
     H                 -0.43000462    -1.83120690    -0.00000000
     H                 -0.43000462     0.91560345     1.58587170
     units au
"""

c4_nh3_mass = [14.003074002,      1.007825035,      1.007825035,      1.007825035]

c4_nh3_dipder = [
   [   -0.3708419404,       0.0000000010,       0.0000000000,
        0.1356506900,      -0.0583104805,      -0.0208481931,
        0.1356506900,      -0.0583104805,       0.0208481931,
        0.0995405605,       0.1166209602,       0.0000000000],
   [   -0.0000000003,      -0.7895340805,       0.0000000000,
       -0.0311511286,       0.2631780268,      -0.0539553375,
       -0.0311511286,       0.2631780268,       0.0539553375,
        0.0623022575,       0.2631780268,       0.0000000000],
   [    0.0000000000,       0.0000000000,      -0.3708419390,
       -0.0208481922,      -0.1009967143,       0.1115772701,
        0.0208481922,       0.1009967143,       0.1115772701,
        0.0000000000,       0.0000000000,       0.1476873986]]

c4_nh3_fcm = """
    4   12
        0.1285212330        0.0000000000        0.0000000000
       -0.0428404110        0.0528973676       -0.0916209283
       -0.0428404110       -0.1057947352        0.0000000000
       -0.0428404110        0.0528973676        0.0916209283
        0.0000000000        0.7858359564        0.0000000000
        0.0775885595       -0.1696087451        0.1599316370
       -0.1551771188       -0.4466184661        0.0000000000
        0.0775885595       -0.1696087451       -0.1599316370
        0.0000000000        0.0000000000        0.7858359564
       -0.1343873270        0.1599316370       -0.3542818924
        0.0000000000        0.0000000000       -0.0772721715
        0.1343873270       -0.1599316370       -0.3542818924
       -0.0428404110        0.0775885595       -0.1343873270
        0.0470388159       -0.0610713659        0.1057787087
       -0.0020992025       -0.0081739983        0.0143531521
       -0.0020992025       -0.0083431952        0.0142554663
        0.0528973676       -0.1696087451        0.1599316370
       -0.0610713659        0.1697901072       -0.1744891202
        0.0165171935       -0.0126978313        0.0347019057
       -0.0083431952        0.0125164692       -0.0201444224
       -0.0916209283        0.1599316370       -0.3542818924
        0.1057787087       -0.1744891202        0.3712727882
        0.0000976859       -0.0055869392        0.0041117024
       -0.0142554663        0.0201444224       -0.0211025982
       -0.0428404110       -0.1551771188        0.0000000000
       -0.0020992025        0.0165171935        0.0000976859
        0.0470388159        0.1221427318        0.0000000000
       -0.0020992025        0.0165171935       -0.0000976859
       -0.1057947352       -0.4466184661        0.0000000000
       -0.0081739983       -0.0126978313       -0.0055869392
        0.1221427318        0.4720141288        0.0000000000
       -0.0081739983       -0.0126978313        0.0055869392
        0.0000000000        0.0000000000       -0.0772721715
        0.0143531521        0.0347019057        0.0041117024
        0.0000000000        0.0000000000        0.0690487667
       -0.0143531521       -0.0347019057        0.0041117024
       -0.0428404110        0.0775885595        0.1343873270
       -0.0020992025       -0.0083431952       -0.0142554663
       -0.0020992025       -0.0081739983       -0.0143531521
        0.0470388159       -0.0610713659       -0.1057787087
        0.0528973676       -0.1696087451       -0.1599316370
       -0.0083431952        0.0125164692        0.0201444224
        0.0165171935       -0.0126978313       -0.0347019057
       -0.0610713659        0.1697901072        0.1744891202
        0.0916209283       -0.1599316370       -0.3542818924
        0.0142554663       -0.0201444224       -0.0211025982
       -0.0000976859        0.0055869392        0.0041117024
       -0.1057787087        0.1744891202        0.3712727882
"""

ref_nh3_vibonly = {
    # I think due to ZMAT confusion, the NH3 geom is slightly non-eq

    # freq from cfour after proj
    'omega': qcdb.vib.QCAspect('', '', np.asarray([760.3060, 1763.7937, 1763.7937, 3639.0750, 3817.6146, 3817.6146]), ''),
    'degeneracy': qcdb.vib.QCAspect('', '', np.asarray([1, 2, 2, 1, 2, 2]), ''),
    'gamma': qcdb.vib.QCAspect('', '', ['A1', 'E', 'E', 'A1', 'E', 'E'], ''),
    #'IR_intensity': qcdb.vib.QCAspect('', '', np.array([255.2339, 18.5931, 18.5931, 0.4207, 9.7416, 9.7416]), ''),  # unproj cfour
    'IR_intensity': qcdb.vib.QCAspect('', '', np.array([255.214, 20.523, 20.523, 0.421, 9.971, 9.971]), ''),  # proj nwchem

    # from vibsuite after cfour
    'mu': qcdb.vib.QCAspect('', '', np.asarray([1.1920,  1.0762,  1.0762,  1.0183,  1.0976,  1.0976]), ''),
    'k': qcdb.vib.QCAspect('', '', np.asarray([0.4060,  1.9725,  1.9725,  7.9450,  9.4251,  9.4251] ), ''),
    'Qtp0': qcdb.vib.QCAspect('', '', np.asarray([0.3979,  0.2613,  0.2613,  0.1819,  0.1776,  0.1776]), ''),
    'DQ0': qcdb.vib.QCAspect('', '', np.asarray([0.2814, 0.1847, 0.1847, 0.1286, 0.1256, 0.1256]), ''),

    # q from vibsuite after cfour
    'q': qcdb.vib.QCAspect('', '', np.asarray(
        [[ -4.08080346E-01, -7.62936600E-12,  0.00000000E+00,  5.07041545E-01,  7.19936610E-02, -1.24696679E-01,  5.07041545E-01, -1.43987322E-01,  0.00000000E+00,  5.07041545E-01,  7.19936610E-02,  1.24696679E-01],
        [  -4.97572587E-12, -2.61585766E-01,  5.32441764E-08, -9.28006642E-02,  5.41620888E-01,  3.75161405E-01,  1.85601263E-01, -1.08177689E-01, -1.54331130E-07, -9.28005987E-02,  5.41621041E-01, -3.75161449E-01],
        [   0.00000000E+00, -5.32441763E-08, -2.61585766E-01,  1.60735389E-01,  3.75161537E-01,  1.08421938E-01,  3.77779972E-08, -2.20189043E-08,  7.58220516E-01, -1.60735427E-01, -3.75161317E-01,  1.08421786E-01],
        [  -1.05093235E-01,  7.07141136E-11,  0.00000000E+00,  1.30578787E-01, -2.79553655E-01,  4.84201134E-01,  1.30578787E-01,  5.59107310E-01,  0.00000000E+00,  1.30578787E-01, -2.79553655E-01, -4.84201134E-01],
        [   0.00000000E+00, -2.24739897E-06, -2.96902468E-01,  1.82435503E-01, -3.30534434E-01,  5.59734985E-01,  1.59458251E-06,  5.68144710E-06, -1.27672829E-02, -1.82437097E-01,  3.30537130E-01,  5.59739989E-01],
        [   1.01604943E-11,  2.96902467E-01, -2.24739897E-06,  1.05331028E-01, -1.78070142E-01,  3.30540019E-01, -2.10659295E-01, -7.50572409E-01, -9.66417646E-08,  1.05328266E-01, -1.78065138E-01, -3.30531545E-01]]
    ).T, ''),
}


# <<<  Cfour Hydrogen Peroxide TS  >>>

c4_hooh_xyz = """
     H                  1.83199008    -1.57622903    -0.00000556
     O                  1.31720978     0.13813086     0.00000035
     O                 -1.31720978    -0.13813086     0.00000035
     H                 -1.83199008     1.57622903    -0.00000556
     units au
"""

c4_hooh_mass = [1.007825035, 15.994914630, 15.994914630, 1.007825035]

##c4_hooh_dipder = [
##    [   0.2780463304,      -0.0627424853,      -0.0000001662,
##        0.2780463304,      -0.0627424853,       0.0000001662,
##       -0.2780463304,       0.0627424853,       0.0000007872,
##       -0.2780463304,       0.0627424853,      -0.0000007872],
##    [  -0.0452363418,       0.2701572439,      -0.0000004247,
##       -0.0452363418,       0.2701572439,       0.0000004247,
##        0.0452363418,      -0.2701572439,       0.0000007936,
##        0.0452363418,      -0.2701572439,      -0.0000007936],
##    [  -0.0000001575,      -0.0000004725,       0.4019549247,
##        0.0000001575,       0.0000004725,       0.4019549247,
##       -0.0000000523,       0.0000008401,      -0.4019549247,
##        0.0000000523,      -0.0000008401,      -0.4019549247]]

# DIPDER was HHOO, not HOOH like FCMFINAL, so manually reordered
c4_hooh_dipder = [
    [
        0.2780463304,      -0.0627424853,      -0.0000001662,
       -0.2780463304,       0.0627424853,       0.0000007872,
       -0.2780463304,       0.0627424853,      -0.0000007872,
        0.2780463304,      -0.0627424853,       0.0000001662],
    [
       -0.0452363418,       0.2701572439,      -0.0000004247,
        0.0452363418,      -0.2701572439,       0.0000007936,
        0.0452363418,      -0.2701572439,      -0.0000007936,
       -0.0452363418,       0.2701572439,       0.0000004247],
    [
       -0.0000001575,      -0.0000004725,       0.4019549247,
       -0.0000000523,       0.0000008401,      -0.4019549247,
        0.0000000523,      -0.0000008401,      -0.4019549247,
        0.0000001575,       0.0000004725,       0.4019549247]]

c4_hooh_fcm = """
    4   12
        0.1263897452       -0.1480441159       -0.0000005106
       -0.0825705803        0.0894682153        0.0000003592
       -0.0297329883        0.0620787276        0.0000001151
       -0.0140861766       -0.0035028271        0.0000000362
       -0.1480441159        0.5705825960        0.0000019741
        0.1705432009       -0.5855055918       -0.0000020194
       -0.0189962579        0.0141465336        0.0000000431
       -0.0035028271        0.0007764624        0.0000000023
       -0.0000005106        0.0000019741       -0.0013924954
        0.0000005485       -0.0000020476        0.0013925128
       -0.0000000017        0.0000000757        0.0013926103
       -0.0000000362       -0.0000000023       -0.0013926278
       -0.0825705803        0.1705432009        0.0000005485
        0.4710873284       -0.1324537721       -0.0000003971
       -0.3587837598       -0.0190931709       -0.0000001530
       -0.0297329883       -0.0189962579        0.0000000017
        0.0894682153       -0.5855055918       -0.0000020476
       -0.1324537721        0.6803496340        0.0000020930
       -0.0190931709       -0.1089905757        0.0000000304
        0.0620787276        0.0141465336       -0.0000000757
        0.0000003592       -0.0000020194        0.0013925128
       -0.0000003971        0.0000020930       -0.0013922639
        0.0000001530       -0.0000000304       -0.0013928591
       -0.0000001151       -0.0000000431        0.0013926103
       -0.0297329883       -0.0189962579       -0.0000000017
       -0.3587837598       -0.0190931709        0.0000001530
        0.4710873284       -0.1324537721        0.0000003971
       -0.0825705803        0.1705432009       -0.0000005485
        0.0620787276        0.0141465336        0.0000000757
       -0.0190931709       -0.1089905757       -0.0000000304
       -0.1324537721        0.6803496340       -0.0000020930
        0.0894682153       -0.5855055918        0.0000020476
        0.0000001151        0.0000000431        0.0013926103
       -0.0000001530        0.0000000304       -0.0013928591
        0.0000003971       -0.0000020930       -0.0013922639
       -0.0000003592        0.0000020194        0.0013925128
       -0.0140861766       -0.0035028271       -0.0000000362
       -0.0297329883        0.0620787276       -0.0000001151
       -0.0825705803        0.0894682153       -0.0000003592
        0.1263897452       -0.1480441159        0.0000005106
       -0.0035028271        0.0007764624       -0.0000000023
       -0.0189962579        0.0141465336       -0.0000000431
        0.1705432009       -0.5855055918        0.0000020194
       -0.1480441159        0.5705825960       -0.0000019741
        0.0000000362        0.0000000023       -0.0013926278
        0.0000000017       -0.0000000757        0.0013926103
       -0.0000005485        0.0000020476        0.0013925128
        0.0000005106       -0.0000019741       -0.0013924954
"""

ref_hooh_vibonly = {

    # freq from cfour after proj
    'omega': qcdb.vib.QCAspect('', '', np.asarray([278.6136j, 1128.8155, 1364.3495, 1698.2924, 4140.0257, 4146.4749]), ''),
    'gamma': qcdb.vib.QCAspect('', '', ['A', 'A', 'B', 'A', 'A', 'B'], ''),
    'IR_intensity': qcdb.vib.QCAspect('', '', np.array([332.2633, 0.0000, 129.1661, 0.0000, 0.0000, 192.2182]), ''),

    # q from vibsuite after cfour
    'q': qcdb.vib.QCAspect('', '', np.asarray(
            [[  7.39506085E-08, -2.30708287E-06,  6.85830054E-01,  2.87191142E-07,  8.13343225E-07, -1.72154397E-01, -2.87191142E-07, -8.13343225E-07, -1.72154397E-01, -7.39506085E-08,  2.30708287E-06,  6.85830054E-01],
             [ -4.79362074E-02,  3.05642567E-02,  4.55759137E-07, -7.01350959E-01, -6.98196096E-02, -1.14402889E-07,  7.01350959E-01,  6.98196096E-02, -1.14402889E-07,  4.79362074E-02, -3.05642567E-02,  4.55759137E-07],
             [  6.56843220E-01,  1.97281139E-01,  4.29128663E-07, -1.64878235E-01, -4.95207454E-02, -8.89584240E-07, -1.64878235E-01, -4.95207454E-02,  8.89584240E-07,  6.56843220E-01,  1.97281139E-01, -4.29128663E-07],
             [ -6.51975530E-01, -1.27322670E-01, -6.12488055E-07,  1.49361767E-02,  2.41854827E-01,  1.53744374E-07, -1.49361767E-02, -2.41854827E-01,  1.53744374E-07,  6.51975530E-01,  1.27322670E-01, -6.12488055E-07],
             [  1.89066152E-01, -6.58050100E-01, -2.26480206E-06, -5.82086285E-02,  1.66840678E-01,  5.68501819E-07,  5.82086285E-02, -1.66840678E-01,  5.68501819E-07, -1.89066152E-01,  6.58050100E-01, -2.26480206E-06],
             [ -1.97281139E-01,  6.56843220E-01,  2.26438557E-06,  4.95207454E-02, -1.64878235E-01, -5.68345734E-07,  4.95207454E-02, -1.64878235E-01,  5.68345734E-07, -1.97281139E-01,  6.56843220E-01, -2.26438557E-06]]
    ).T, ''),
}


# <<<  Cfour Methane  >>>

c4_ch4_xyz = """
     C                  0.00000000     0.00000000     0.00000000
     H                  0.00000000    -1.68303316     1.19008416
     H                 -0.00000000     1.68303316     1.19008416
     H                  1.68303316    -0.00000000    -1.19008416
     H                 -1.68303316     0.00000000    -1.19008416
     units au
"""

c4_ch4_mass = [12.000000000, 1.007825035, 1.007825035, 1.007825035, 1.007825035]

c4_ch4_dipder = [
      [ 0.0605757483,       0.0000000000,       0.0000000000,
       -0.0151439370,      -0.0746404771,       0.0746404770,
       -0.0151439370,      -0.0746404771,      -0.0746404770,
       -0.0151439370,       0.0746404771,       0.0746404770,
       -0.0151439370,       0.0746404771,      -0.0746404770],
      [ 0.0000000000,       0.0605757483,       0.0000000000,
       -0.0746404771,      -0.0151439370,       0.0746404770,
       -0.0746404771,      -0.0151439370,      -0.0746404770,
        0.0746404771,      -0.0151439370,      -0.0746404770,
        0.0746404771,      -0.0151439370,       0.0746404770],
      [ 0.0000000000,       0.0000000000,       0.0605757483,
        0.0746404771,       0.0746404771,      -0.0151439371,
       -0.0746404771,      -0.0746404771,      -0.0151439371,
        0.0746404771,      -0.0746404771,      -0.0151439371,
       -0.0746404771,       0.0746404771,      -0.0151439371]]

c4_ch4_fcm = """
    5   15
        0.6065645250        0.0000000000        0.0000000000
       -0.0530280942        0.0000000000        0.0000000000
       -0.0530280942        0.0000000000        0.0000000000
       -0.2502541684        0.0000000000        0.1394598945
       -0.2502541684        0.0000000000       -0.1394598945
        0.0000000000        0.6065645250        0.0000000000
        0.0000000000       -0.2502541684        0.1394598945
        0.0000000000       -0.2502541684       -0.1394598945
        0.0000000000       -0.0530280942        0.0000000000
        0.0000000000       -0.0530280942        0.0000000000
        0.0000000000        0.0000000000        0.6065645249
        0.0000000000        0.1394598945       -0.1516411312
        0.0000000000       -0.1394598945       -0.1516411312
        0.1394598945        0.0000000000       -0.1516411312
       -0.1394598945        0.0000000000       -0.1516411312
       -0.0530280942        0.0000000000        0.0000000000
        0.0504323812        0.0000000000        0.0000000000
       -0.0013513088        0.0000000000        0.0000000000
        0.0019735108       -0.0258917766        0.0159572484
        0.0019735108        0.0258917766       -0.0159572484
        0.0000000000       -0.2502541684        0.1394598945
        0.0000000000        0.2673258620       -0.1533668511
        0.0000000000       -0.0210187153        0.0180075402
       -0.0004252691        0.0019735108       -0.0020502918
        0.0004252691        0.0019735108       -0.0020502918
        0.0000000000        0.1394598945       -0.1516411312
        0.0000000000       -0.1533668511        0.1588791215
        0.0000000000       -0.0180075402        0.0151320337
       -0.0020502918        0.0159572484       -0.0111850120
        0.0020502918        0.0159572484       -0.0111850120
       -0.0530280942        0.0000000000        0.0000000000
       -0.0013513088        0.0000000000        0.0000000000
        0.0504323812        0.0000000000        0.0000000000
        0.0019735108        0.0258917766        0.0159572484
        0.0019735108       -0.0258917766       -0.0159572484
        0.0000000000       -0.2502541684       -0.1394598945
        0.0000000000       -0.0210187153       -0.0180075402
        0.0000000000        0.2673258620        0.1533668511
        0.0004252691        0.0019735108        0.0020502918
       -0.0004252691        0.0019735108        0.0020502918
        0.0000000000       -0.1394598945       -0.1516411312
        0.0000000000        0.0180075402        0.0151320337
        0.0000000000        0.1533668511        0.1588791215
       -0.0020502918       -0.0159572484       -0.0111850120
        0.0020502918       -0.0159572484       -0.0111850120
       -0.2502541684        0.0000000000        0.1394598945
        0.0019735108       -0.0004252691       -0.0020502918
        0.0019735108        0.0004252691       -0.0020502918
        0.2673258620        0.0000000000       -0.1533668511
       -0.0210187153        0.0000000000        0.0180075402
        0.0000000000       -0.0530280942        0.0000000000
       -0.0258917766        0.0019735108        0.0159572484
        0.0258917766        0.0019735108       -0.0159572484
        0.0000000000        0.0504323812        0.0000000000
        0.0000000000       -0.0013513088        0.0000000000
        0.1394598945        0.0000000000       -0.1516411312
        0.0159572484       -0.0020502918       -0.0111850120
        0.0159572484        0.0020502918       -0.0111850120
       -0.1533668511        0.0000000000        0.1588791215
       -0.0180075402        0.0000000000        0.0151320337
       -0.2502541684        0.0000000000       -0.1394598945
        0.0019735108        0.0004252691        0.0020502918
        0.0019735108       -0.0004252691        0.0020502918
       -0.0210187153        0.0000000000       -0.0180075402
        0.2673258620        0.0000000000        0.1533668511
        0.0000000000       -0.0530280942        0.0000000000
        0.0258917766        0.0019735108        0.0159572484
       -0.0258917766        0.0019735108       -0.0159572484
        0.0000000000       -0.0013513088        0.0000000000
        0.0000000000        0.0504323812        0.0000000000
       -0.1394598945        0.0000000000       -0.1516411312
       -0.0159572484       -0.0020502918       -0.0111850120
       -0.0159572484        0.0020502918       -0.0111850120
        0.0180075402        0.0000000000        0.0151320337
        0.1533668511        0.0000000000        0.1588791215
"""

ref_ch4_vibonly = {

    # freq from cfour after proj
    'omega': qcdb.vib.QCAspect('', '', np.asarray([1433.2785, 1433.2785, 1433.2785, 1647.8698, 1647.8698, 3164.8164, 3285.6166, 3285.6166, 3285.6166]), ''),
    'gamma': qcdb.vib.QCAspect('', '', ['T2', 'T2', 'T2', 'E', 'E', 'A1', 'T2', 'T2', 'T2'], ''),
    'IR_intensity': qcdb.vib.QCAspect('', '', np.array([9.8963, 9.8963, 9.8963, 0.0000, 0.0000, 0.0000, 34.4012, 34.4012, 34.4012]), ''),

    # vibsuite seems to have contamination for evec from 1,2 degen, so
    #    from NORMCO (q; pre-proj) from cfour
    'q': qcdb.vib.QCAspect('', '', np.asarray(
            [[     -0.0000000000,        0.0000000000,        0.3971847614,
                    0.0000000000,       -0.3052259977,       -0.3426341440,
                    0.0000000000,        0.3052259977,       -0.3426341440,
                   -0.3052259977,        0.0000000000,       -0.3426341440,
                    0.3052259977,        0.0000000000,       -0.3426341440],
            [       0.0000000000,        0.3971847621,        0.0000000000,
                    0.0000000000,       -0.1268067711,       -0.3052259975,
                    0.0000000000,       -0.1268067711,        0.3052259975,
                   -0.0000000000,       -0.5584615167,        0.0000000000,
                    0.0000000000,       -0.5584615167,        0.0000000000],
            [       0.3971847621,        0.0000000000,        0.0000000000,
                   -0.5584615167,        0.0000000000,        0.0000000000,
                   -0.5584615167,        0.0000000000,        0.0000000000,
                   -0.1268067711,        0.0000000000,       -0.3052259975,
                   -0.1268067711,        0.0000000000,        0.3052259975],
            [       0.0000000000,        0.0000000000,       -0.0000000000,
                    0.0000000000,        0.2886751345,        0.4082482906,
                    0.0000000000,       -0.2886751345,        0.4082482906,
                   -0.2886751345,        0.0000000000,       -0.4082482906,
                    0.2886751345,        0.0000000000,       -0.4082482906],
            [       0.0000000000,       -0.0000000000,        0.0000000000,
                    0.5000000000,        0.0000000000,        0.0000000000,
                   -0.5000000000,        0.0000000000,        0.0000000000,
                    0.0000000000,       -0.5000000000,        0.0000000000,
                    0.0000000000,        0.5000000000,        0.0000000000],
            [       0.0000000000,       -0.0000000000,       -0.0000000000,
                   -0.0000000000,        0.4082482906,       -0.2886751345,
                   -0.0000000000,       -0.4082482906,       -0.2886751345,
                   -0.4082482906,        0.0000000000,        0.2886751345,
                    0.4082482906,        0.0000000000,        0.2886751345],
            [       0.0000000000,        0.0000000000,        0.3061185954,
                   -0.0000000000,        0.3960266283,       -0.2640752943,
                   -0.0000000000,       -0.3960266283,       -0.2640752943,
                    0.3960266283,        0.0000000000,       -0.2640752943,
                   -0.3960266283,        0.0000000000,       -0.2640752943],
            [       0.0000000000,        0.3061185953,        0.0000000000,
                    0.0000000000,       -0.5441084087,        0.3960266283,
                   -0.0000000000,       -0.5441084087,       -0.3960266283,
                    0.0000000000,        0.0159578203,        0.0000000000,
                   -0.0000000000,        0.0159578203,       -0.0000000000],
            [       0.3061185953,        0.0000000000,        0.0000000000,
                    0.0159578203,        0.0000000000,        0.0000000000,
                    0.0159578203,        0.0000000000,        0.0000000000,
                   -0.5441084087,        0.0000000000,        0.3960266283,
                   -0.5441084087,        0.0000000000,       -0.3960266283]]
            ).T, ''),

}


# <<<  Cfour CO2  >>>

c4_co2_xyz = """
     C                  0.00000000     0.00000000     0.00000000
     O                  0.00000000     0.00000000    -2.15536093
     O                 -0.00000000     0.00000000     2.15536093
     units au
"""

c4_co2_mass = [12.0, 15.994914630, 15.994914630]

c4_co2_dipder = [
     [  0.7687899340,        0.0000000000,        0.0000000000,
       -0.3843949670,        0.0000000000,        0.0000000000,
       -0.3843949670,        0.0000000000,        0.0000000000],
     [  0.0000000000,        0.7687899340,        0.0000000000,
        0.0000000000,       -0.3843949670,        0.0000000000,
        0.0000000000,       -0.3843949670,        0.0000000000],
     [  0.0000000000,        0.0000000000,        3.0485336770,
        0.0000000000,        0.0000000000,       -1.5242668385,
        0.0000000000,        0.0000000000,       -1.5242668385]]

c4_co2_fcm = """
    3    9
        0.1913268270        0.0000000000        0.0000000000
       -0.0956634136        0.0000000000        0.0000000000
       -0.0956634136        0.0000000000        0.0000000000
        0.0000000000        0.1913268270        0.0000000000
        0.0000000000       -0.0956634136        0.0000000000
        0.0000000000       -0.0956634136        0.0000000000
        0.0000000000        0.0000000000        2.1984779701
        0.0000000000        0.0000000000       -1.0992389851
        0.0000000000        0.0000000000       -1.0992389851
       -0.0956634136        0.0000000000        0.0000000000
        0.0478318229        0.0000000000        0.0000000000
        0.0478315907        0.0000000000        0.0000000000
        0.0000000000       -0.0956634136        0.0000000000
        0.0000000000        0.0478318229        0.0000000000
        0.0000000000        0.0478315907        0.0000000000
        0.0000000000        0.0000000000       -1.0992389851
        0.0000000000        0.0000000000        1.2427267603
        0.0000000000        0.0000000000       -0.1434877751
       -0.0956634136        0.0000000000        0.0000000000
        0.0478315907        0.0000000000        0.0000000000
        0.0478318229        0.0000000000        0.0000000000
        0.0000000000       -0.0956634136        0.0000000000
        0.0000000000        0.0478315907        0.0000000000
        0.0000000000        0.0478318229        0.0000000000
        0.0000000000        0.0000000000       -1.0992389851
        0.0000000000        0.0000000000       -0.1434877751
        0.0000000000        0.0000000000        1.2427267603
"""

ref_co2_vibonly = {

    # freq from cfour after proj
    'omega': qcdb.vib.QCAspect('', '', np.asarray([761.1526, 761.1526, 1513.3122, 2580.1495]), ''),
    'gamma': qcdb.vib.QCAspect('', '', ['PIu', 'PIu', 'SGg+', 'SGu-'], ''),
    'IR_intensity': qcdb.vib.QCAspect('', '', np.array([66.0268, 66.0268, 0.0000, 1038.2141]), ''),

    # from vibsuite after cfour
    'mu': qcdb.vib.QCAspect('', '', np.asarray([ 12.8774,   12.8774,   15.9949,   12.8774]), ''),
    'k': qcdb.vib.QCAspect('', '', np.asarray([ 4.3956, 4.3956, 21.5819, 50.5088]), ''),
    'Qtp0': qcdb.vib.QCAspect('', '', np.asarray([ 0.3977,    0.3977,    0.2821,    0.2160]), ''),
    'DQ0': qcdb.vib.QCAspect('', '', np.asarray([ 0.2812, 0.2812, 0.1994, 0.1527]), ''),

    # q from vibsuite after cfour
    'q': qcdb.vib.QCAspect('', '', np.asarray(
            [[ 8.52765895E-01,  0.00000000E+00,  0.00000000E+00, -3.69317160E-01,  0.00000000E+00,  0.00000000E+00, -3.69317160E-01,  0.00000000E+00,  0.00000000E+00],
             [ 0.00000000E+00,  8.52765895E-01,  0.00000000E+00,  0.00000000E+00, -3.69317160E-01,  0.00000000E+00,  0.00000000E+00, -3.69317160E-01,  0.00000000E+00],
             [ 0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00,  0.00000000E+00, -7.07106781E-01,  0.00000000E+00,  0.00000000E+00,  7.07106781E-01],
             [ 0.00000000E+00,  0.00000000E+00,  8.52765895E-01,  0.00000000E+00,  0.00000000E+00, -3.69317160E-01,  0.00000000E+00,  0.00000000E+00, -3.69317160E-01]]
    ).T, ''),
}


# <<<  Cfour Formaldehyde  >>>

c4_form_xyz = """
     C                  0.00000000     0.00000000     1.11607307
     O                  0.00000000     0.00000000    -1.11771672
     H                  0.00000000     1.76217215     2.22504227
     H                  0.00000000    -1.76217215     2.22504227
     units au
"""

c4_form_mass = [12.000000000, 15.994914630, 1.007825035, 1.007825035]

c4_form_dipder = [
     [  0.2435938429,       0.0000000000,       0.0000000000,
       -0.3830997164,       0.0000000000,       0.0000000000,
        0.0697529367,       0.0000000000,       0.0000000000,
        0.0697529367,       0.0000000000,       0.0000000000],

     [  0.0000000000,       0.9433671590,       0.0000000000,
        0.0000000000,      -0.5373880589,       0.0000000000,
        0.0000000000,      -0.2029895500,      -0.0738515561,
        0.0000000000,      -0.2029895500,       0.0738515561],

     [  0.0000000000,       0.0000000000,       1.2038372293,
        0.0000000000,       0.0000000000,      -1.1307467270,
        0.0000000000,      -0.1078376670,      -0.0365452511,
        0.0000000000,       0.1078376670,      -0.0365452511]]


c4_form_fcm = """
    4   12
        0.2132177929        0.0000000000        0.0000000000
       -0.0707359336        0.0000000000        0.0000000000
       -0.0712409296        0.0000000000        0.0000000000
       -0.0712409296        0.0000000000        0.0000000000
        0.0000000000        0.6710231116        0.0000000000
        0.0000000000       -0.1230826712        0.0000000000
        # stuff
        0.0000000000       -0.2739702202       -0.0944034742
        0.0000000000       -0.2739702202        0.0944034742
        0.0000000000        0.0000000000        1.2160957100
        0.0000000000        0.0000000000       -0.9878123593
        0.0000000000       -0.0930644339       -0.1141416753
        0.0000000000        0.0930644339       -0.1141416753
       -0.0707359336        0.0000000000        0.0000000000
        0.0234665344        0.0000000000        0.0000000000
        0.0236346996        0.0000000000        0.0000000000
        0.0236346996        0.0000000000        0.0000000000
        0.0000000000       -0.1230826712        0.0000000000
        0.0000000000        0.0856392896        0.0000000000
        0.0000000000        0.0187216908       -0.0424974191
        0.0000000000        0.0187216908        0.0424974191
        0.0000000000        0.0000000000       -0.9878123593
        0.0000000000        0.0000000000        1.0793209672
        0.0000000000       -0.0167443025       -0.0457543039
        0.0000000000        0.0167443025       -0.0457543039
       -0.0712409296        0.0000000000        0.0000000000
        0.0236346996        0.0000000000        0.0000000000
        0.0238028786        0.0000000000        0.0000000000
        0.0238033515        0.0000000000        0.0000000000
        0.0000000000       -0.2739702202       -0.0930644339
        0.0000000000        0.0187216908       -0.0167443025
        0.0000000000        0.2753690118        0.1233548149
        0.0000000000       -0.0201204824       -0.0135460785
        0.0000000000       -0.0944034742       -0.1141416753
        0.0000000000       -0.0424974191       -0.0457543039
        0.0000000000        0.1233548149        0.1499605310
        0.0000000000        0.0135460785        0.0099354482
       -0.0712409296        0.0000000000        0.0000000000
        0.0236346996        0.0000000000        0.0000000000
        0.0238033515        0.0000000000        0.0000000000
        0.0238028786        0.0000000000        0.0000000000
        0.0000000000       -0.2739702202        0.0930644339
        0.0000000000        0.0187216908        0.0167443025
        0.0000000000       -0.0201204824        0.0135460785
        0.0000000000        0.2753690118       -0.1233548149
        0.0000000000        0.0944034742       -0.1141416753
        0.0000000000        0.0424974191       -0.0457543039
        0.0000000000       -0.0135460785        0.0099354482
        0.0000000000       -0.1233548149        0.1499605310
"""

ref_form_vibonly = {

    # freq from cfour after proj
    'omega': qcdb.vib.QCAspect('', '', np.asarray([1325.3286, 1359.7579, 1637.4774, 2013.4255, 3108.9786, 3183.3975]), ''),
    'gamma': qcdb.vib.QCAspect('', '', ['B1', 'B2', 'A1', 'A1', 'A1', 'B2'], ''),
    'IR_intensity': qcdb.vib.QCAspect('', '', np.array([1.0137, 26.1737, 11.0592, 158.8806, 50.8031, 134.5380]), ''),

    # from vibsuite after cfour
    'mu': qcdb.vib.QCAspect('', '', np.asarray([1.3722, 1.3499, 1.0901, 7.8589, 1.0499, 1.1203]), ''),
    'k': qcdb.vib.QCAspect('', '', np.asarray([1.4201, 1.4706, 1.7222, 18.7708, 5.9790, 6.6892]), ''),
    'Qtp0': qcdb.vib.QCAspect('', '', np.asarray([0.3014, 0.2976, 0.2711, 0.2445, 0.1968, 0.1945]), ''),
    'DQ0': qcdb.vib.QCAspect('', '', np.asarray([0.2131, 0.2104, 0.1917, 0.1729, 0.1391, 0.1375]), ''),

    # q from vibsuite after cfour
    'q': qcdb.vib.QCAspect('', '', np.asarray(
        [[ -5.17011996E-01,  0.00000000E+00,  0.00000000E+00,  1.48564403E-01,  0.00000000E+00,  0.00000000E+00,  5.96081880E-01,  0.00000000E+00,  0.00000000E+00,  5.96081880E-01,  0.00000000E+00,  0.00000000E+00],
         [  0.00000000E+00,  4.45428632E-01,  0.00000000E+00,  0.00000000E+00, -2.76578605E-01,  0.00000000E+00,  0.00000000E+00, -2.17586159E-01,  5.61431267E-01,  0.00000000E+00, -2.17586159E-01, -5.61431267E-01],
         [  0.00000000E+00,  0.00000000E+00, -1.22436444E-02,  0.00000000E+00,  0.00000000E+00, -2.83617137E-01,  0.00000000E+00, -3.40935268E-01,  5.86062155E-01,  0.00000000E+00,  3.40935268E-01,  5.86062155E-01],
         [  0.00000000E+00,  0.00000000E+00,  7.45852498E-01,  0.00000000E+00,  0.00000000E+00, -6.21723599E-01,  0.00000000E+00,  1.61980873E-01, -4.84159650E-02,  0.00000000E+00, -1.61980873E-01, -4.84159650E-02],
         [  0.00000000E+00,  0.00000000E+00,  2.09033240E-01,  0.00000000E+00,  0.00000000E+00, -6.71018103E-03,  0.00000000E+00, -5.97934227E-01, -3.47281873E-01,  0.00000000E+00,  5.97934227E-01, -3.47281873E-01],
         [  0.00000000E+00, -3.31072792E-01,  0.00000000E+00,  0.00000000E+00,  2.07904060E-03,  0.00000000E+00,  0.00000000E+00,  5.67063166E-01,  3.51614288E-01,  0.00000000E+00,  5.67063166E-01, -3.51614288E-01]]
    ).T, ''),
}

# yapf:enable

_cfour_ref = {
    'nh3': {
        'xyz': c4_nh3_xyz,
        'mass': c4_nh3_mass,
        'dipder': c4_nh3_dipder,
        'fcm': c4_nh3_fcm,
        'vibonly': ref_nh3_vibonly
    },
    'hooh': {
        'xyz': c4_hooh_xyz,
        'mass': c4_hooh_mass,
        'dipder': c4_hooh_dipder,
        'fcm': c4_hooh_fcm,
        'vibonly': ref_hooh_vibonly
    },
    'c2h4': {
        'xyz': c4_eth_xyz,
        'mass': c4_eth_mass,
        'dipder': c4_eth_dipder,
        'fcm': c4_eth_fcm,
        'vibonly': ref_eth_vibonly
    },
    'co2': {
        'xyz': c4_co2_xyz,
        'mass': c4_co2_mass,
        'dipder': c4_co2_dipder,
        'fcm': c4_co2_fcm,
        'vibonly': ref_co2_vibonly
    },
    'ch4': {
        'xyz': c4_ch4_xyz,
        'mass': c4_ch4_mass,
        'dipder': c4_ch4_dipder,
        'fcm': c4_ch4_fcm,
        'vibonly': ref_ch4_vibonly
    },
    'h2co': {
        'xyz': c4_form_xyz,
        'mass': c4_form_mass,
        'dipder': c4_form_dipder,
        'fcm': c4_form_fcm,
        'vibonly': ref_form_vibonly
    },
}

# <<<  Cfour geometries in Psi4 notation  >>>

_psi4_systems = {
    'hooh':
    """
units au
H  1.8327917647 -1.5752960165 -0.0000055594
O  1.3171390326  0.1388012713  0.0000003503
O -1.3171390326 -0.1388012713  0.0000003503
H -1.8327917647  1.5752960165 -0.0000055594
""",
    'co2':
    """
units au
C  0.0000000000  0.0000000000  0.0000000000
O  2.1553604887  0.0000000000  0.0000000000
O -2.1553604887 -0.0000000000  0.0000000000
""",
    'c2h4':
    """
units au
C  1.2480432017  0.0000000000  0.0000000000
C -1.2480432017 -0.0000000000  0.0000000000
H  2.3230381483  1.7435443352  0.0000000000
H  2.3230381483 -1.7435443352  0.0000000000
H -2.3230381483  1.7435443352  0.0000000000
H -2.3230381483 -1.7435443352  0.0000000000
""",
    'ch4':
    """
units au
C  0.0000000000  0.0000000000  0.0000000000
H  0.0000000000  1.6830328110 -1.1900839136
H  1.6830328110  0.0000000000  1.1900839136
H -1.6830328110  0.0000000000  1.1900839136
H  0.0000000000 -1.6830328110 -1.1900839136
""",
    'nh3':
    """
     N                    0.00000000     0.00000000    0.09284449
     H                    0.91560345    -1.58587170   -0.43000462
     H                   -1.83120690    -0.00000000   -0.43000462
     H                    0.91560345     1.58587170   -0.43000462
     units au
""",
    'h2co':
    """
units au
     C                  0.00000000     0.00000000     1.11607307
     O                  0.00000000     0.00000000    -1.11771672
     H                  0.00000000     1.76217215     2.22504227
     H                  0.00000000    -1.76217215     2.22504227
""",
}

# <<<  Section I: testing vibrational analysis vs Cfour  >>>


@pytest.mark.quick
@pytest.mark.parametrize(
    "subject", [
        'co2',
        'c2h4',
        pytest.param('ch4', marks=pytest.mark.xfail(reason="unaligned degen mode pair")),
        'nh3',
        'h2co',
        'hooh',
    ],
    ids=['CO2', 'ethene', 'methane', 'ammonia', 'formaldehyde', 'HOOH_TS'])
def test_harmonic_analysis_vs_cfour(subject, request):
    # one of the ch4 degen modes has slight noise (when geom is properly bohr, so 4->2)
    digits = 2 if subject in ['ch4'] else 4
    # IR matches to 4 when scale factor tuned, but have to loosen to 1 since Cfour using 974.868 and we're using ~974.880
    toldict = {'IR_intensity': 1}
    verbose = 2
    forgive = ['gamma'] if subject in ['co2', 'ch4', 'nh3'] else []  # since Psi can't classify degen symmetries
    if subject in ['nh3', 'hooh']:
        forgive.append('IR_intensity')
        # there's problems with orientation for DIPDER

    pvibinfo = vibanal_str(
        mass=_cfour_ref[subject]['mass'],
        coord=_cfour_ref[subject]['xyz'],
        fcm=_cfour_ref[subject]['fcm'],
        dipder=_cfour_ref[subject]['dipder'])
    pvibonly = qcdb.vib.filter_nonvib(pvibinfo)
    rvibonly = _cfour_ref[subject]['vibonly']

    assert qcdb.compare_vibinfos(
        rvibonly, pvibonly, digits, request.node.name, verbose=verbose, forgive=forgive, toldict=toldict)


# <<<  Section II: testing Psi4 findif-by-grad Hessians vs Cfour  >>>
# <<<  Section III: testing Psi4 findif-by-energy Hessians vs Cfour  >>>
# <<<  Section IV: testing Psi4 analytic Hessians vs Cfour  >>>


@pytest.mark.parametrize(
    "dertype", [
        2,
        1,
        pytest.param(0, marks=pytest.mark.long),
    ], ids=['H_analytic', 'H_by_grad', 'H_by_ene'])
@pytest.mark.parametrize(
    "subject", [
        pytest.param('co2'),
        pytest.param('c2h4', marks=pytest.mark.long),
        pytest.param('ch4', marks=pytest.mark.xfail(reason="unaligned degen mode pair")),
        pytest.param('nh3'),
        pytest.param('h2co', marks=pytest.mark.quick),
        pytest.param('hooh'),
    ],
    ids=['CO2', 'ethene', 'methane', 'ammonia', 'formaldehyde', 'HOOH_TS'])
def test_hessian_vs_cfour(subject, dertype, request):
    """compare analytic, findif by G, findif by E vibrational analyses for several mols"""

    digits = 2
    toldict = {'IR_intensity': 1} if subject in ['nh3'] else {}
    verbose = 2
    forgive = ['gamma'] if subject in ['co2', 'ch4', 'nh3'] else []  # since Psi can't classify degen symmetries
    if dertype != 2:
        forgive.append('IR_intensity')

    qmol = qcdb.Molecule(_psi4_systems[subject])
    rqmol = qcdb.Molecule(_cfour_ref[subject]['xyz'])
    rmsd, mill, aqmol = qmol.B787(rqmol, atoms_map=True, mols_align=True, verbose=verbose - 1)
    apmol = psi4.geometry(aqmol.create_psi4_string_from_molecule())

    psi4.set_options({
        'e_convergence': 10,
        'd_convergence': 10,
        'points': 5,
        'scf_type': 'pk',
    })

    E, pwfn = psi4.frequency('hf/cc-pvdz', return_wfn=True, molecule=apmol, dertype=dertype)

    pvibonly = qcdb.vib.filter_nonvib(pwfn.frequency_analysis)
    rvibonly = _cfour_ref[subject]['vibonly']

    assert qcdb.compare_vibinfos(
        rvibonly, pvibonly, digits, request.node.name, verbose=verbose, forgive=forgive, toldict=toldict)


# <<<  Section V: Thermo  >>>


@pytest.mark.quick
def test_thermochemistry():
    modelchem = 'hf/3-21g'

    ch4_thermo_mol = psi4.geometry("""
        C
        H 1 r
        H 1 r 2 TDA
        H 1 r 2 TDA 3 120
        H 1 r 2 TDA 4 120
        r = 1.09
        """)

    psi4.set_options({
        'g_convergence': 'gau_verytight',
        'e_convergence': 10,
        'd_convergence': 10,
        'points': 5,
        'scf_type': 'pk',
    })

    psi4.optimize(modelchem, molecule=ch4_thermo_mol)
    E, thermo_wfn = psi4.freq(modelchem, return_wfn=True, molecule=ch4_thermo_mol, dertype=1)

    thermo_vibinfo = psi4.driver.vibanal_wfn(thermo_wfn)

    rot_const = np.asarray(ch4_thermo_mol.rotational_constants())
    therminfo, text = qcdb.vib.thermo(
        thermo_vibinfo,
        T=298.15,
        P=101325.0,
        multiplicity=1,
        molecular_mass=16.0313,
        E0=-39.97687755,
        sigma=12,
        rot_const=rot_const)
    print(text)

    ch4_hf_321g_thermoinfo = {
        'S_elec' :   qcdb.vib.QCAspect('', '', 0.00000000, ''),
        'S_trans' :  qcdb.vib.QCAspect('', '', 0.05459886, ''),
        'S_rot' :    qcdb.vib.QCAspect('', '', 0.01606470, ''),
        'S_vib' :    qcdb.vib.QCAspect('', '', 0.00006509, ''),
        'S_tot' :    qcdb.vib.QCAspect('', '', 0.07072865, ''),

        'Cv_elec' :  qcdb.vib.QCAspect('', '', 0.00000000, ''),
        'Cv_trans' : qcdb.vib.QCAspect('', '', 0.00475022, ''),
        'Cv_rot' :   qcdb.vib.QCAspect('', '', 0.00475022, ''),
        'Cv_vib' :   qcdb.vib.QCAspect('', '', 0.00043482, ''),
        'Cv_tot' :   qcdb.vib.QCAspect('', '', 0.00993526, ''),

        'Cp_elec' :  qcdb.vib.QCAspect('', '', 0.00000000, ''),
        'Cp_trans' : qcdb.vib.QCAspect('', '', 0.00791704, ''),
        'Cp_rot' :   qcdb.vib.QCAspect('', '', 0.00475022, ''),
        'Cp_vib' :   qcdb.vib.QCAspect('', '', 0.00043482, ''),
        'Cp_tot' :   qcdb.vib.QCAspect('', '', 0.01310208, ''),

        'E0'     :   qcdb.vib.QCAspect('', '', -39.97687755, ''),

        'ZPE_elec' :  qcdb.vib.QCAspect('', '',   0.00000000, ''),
        'ZPE_trans' : qcdb.vib.QCAspect('', '',   0.00000000, ''),
        'ZPE_rot' :   qcdb.vib.QCAspect('', '',   0.00000000, ''),
        'ZPE_vib' :   qcdb.vib.QCAspect('', '',   0.04799572, ''),
        'ZPE_corr' :  qcdb.vib.QCAspect('', '',   0.04799572, ''),
        'ZPE_tot' :   qcdb.vib.QCAspect('', '', -39.92888184, ''),

        'E_elec'  :   qcdb.vib.QCAspect('', '',   0.00000000, ''),
        'E_trans' :   qcdb.vib.QCAspect('', '',   0.00141628, ''),
        'E_rot'   :   qcdb.vib.QCAspect('', '',   0.00141628, ''),
        'E_vib'   :   qcdb.vib.QCAspect('', '',   0.04801283, ''),
        'E_corr'  :   qcdb.vib.QCAspect('', '',   0.05084539, ''),
        'E_tot'   :   qcdb.vib.QCAspect('', '', -39.92603216, ''),

        'H_elec'  :   qcdb.vib.QCAspect('', '',   0.00000000, ''),
        'H_trans' :   qcdb.vib.QCAspect('', '',   0.00236046, ''),
        'H_rot'   :   qcdb.vib.QCAspect('', '',   0.00141628, ''),
        'H_vib'   :   qcdb.vib.QCAspect('', '',   0.04801283, ''),
        'H_corr'  :   qcdb.vib.QCAspect('', '',   0.05178958, ''),
        'H_tot'   :   qcdb.vib.QCAspect('', '', -39.92508798, ''),

        'G_elec'  :   qcdb.vib.QCAspect('', '',  -0.00000000, ''),
        'G_trans' :   qcdb.vib.QCAspect('', '',  -0.01391819, ''),
        'G_rot'   :   qcdb.vib.QCAspect('', '',  -0.00337341, ''),
        'G_vib'   :   qcdb.vib.QCAspect('', '',   0.04799343, ''),
        'G_corr'  :   qcdb.vib.QCAspect('', '',   0.03070183, ''),
        'G_tot'   :   qcdb.vib.QCAspect('', '', -39.94617572, ''),
    }  # yapf: disable

    assert qcdb.compare_vibinfos(ch4_hf_321g_thermoinfo, therminfo, 4, 'asdf', forgive=['omega', 'IR_intensity'])


c4_neqh2o_xyz = """
O          0.0 0.0 0.0
H          0.0 1.0 0.0
H          0.0 0.0 1.0
"""

c4_neqh2o_mass = [15.994914630, 1.007825035, 1.007825035]

c4_neqh2o_fcm_analytic = """
    3    9
        0.0711843445        0.0000000000        0.0000000000
       -0.0355921722        0.0000000000        0.0000000000
       -0.0355921722        0.0000000000        0.0000000000
        0.0000000000        0.4725104616        0.0000000000
        0.0000000000       -0.2362552308       -0.2006630586
        0.0000000000       -0.2362552308        0.2006630586
        0.0000000000        0.0000000000        0.4698256319
        0.0000000000       -0.1475753402       -0.2349128160
        0.0000000000        0.1475753402       -0.2349128160
       -0.0355921722        0.0000000000        0.0000000000
        0.0234474351        0.0000000000        0.0000000000
        0.0121447371        0.0000000000        0.0000000000
        0.0000000000       -0.2362552308       -0.1475753402
        0.0000000000        0.2885428832        0.1741191993
        0.0000000000       -0.0522876525       -0.0265438592
        0.0000000000       -0.2006630586       -0.2349128160
        0.0000000000        0.1741191993        0.2234392863
        0.0000000000        0.0265438592        0.0114735296
       -0.0355921722        0.0000000000        0.0000000000
        0.0121447371        0.0000000000        0.0000000000
        0.0234474351        0.0000000000        0.0000000000
        0.0000000000       -0.2362552308        0.1475753402
        0.0000000000       -0.0522876525        0.0265438592
        0.0000000000        0.2885428832       -0.1741191993
        0.0000000000        0.2006630586       -0.2349128160
        0.0000000000       -0.0265438592        0.0114735296
        0.0000000000       -0.1741191993        0.2234392863
"""

c4_neqh2o_fcm_findifproj = """
    3    9
        0.0000000000        0.0000000000        0.0000000000
       -0.0000000000        0.0000000000        0.0000000000
       -0.0000000000        0.0000000000        0.0000000000
        0.0000000000        0.4262958076        0.0000000000
        0.0000000000       -0.2131479038       -0.2131479038
        0.0000000000       -0.2131479038        0.2131479038
        0.0000000000        0.0000000000        0.4698256163
        0.0000000000       -0.1475753470       -0.2349128081
        0.0000000000        0.1475753470       -0.2349128081
       -0.0000000000        0.0000000000        0.0000000000
        0.0000000000        0.0000000000        0.0000000000
       -0.0000000000        0.0000000000        0.0000000000
        0.0000000000       -0.2131479038       -0.1475753470
        0.0000000000        0.2769892550        0.1803616254
        0.0000000000       -0.0638413512       -0.0327862784
        0.0000000000       -0.2131479038       -0.2349128081
        0.0000000000        0.1803616254        0.2240303560
        0.0000000000        0.0327862784        0.0108824522
       -0.0000000000        0.0000000000        0.0000000000
       -0.0000000000        0.0000000000        0.0000000000
        0.0000000000        0.0000000000        0.0000000000
        0.0000000000       -0.2131479038        0.1475753470
        0.0000000000       -0.0638413512        0.0327862784
        0.0000000000        0.2769892550       -0.1803616254
        0.0000000000        0.2131479038       -0.2349128081
        0.0000000000       -0.0327862784        0.0108824522
        0.0000000000       -0.1803616254        0.2240303560
"""

c4_neqh2o_fcm_findifrot = """
    3    9
        0.0711843601        0.0000000000        0.0000000000
       -0.0355921801        0.0000000000        0.0000000000
       -0.0355921801        0.0000000000        0.0000000000
        0.0000000000        0.4725103356        0.0000000000
        0.0000000000       -0.2362551678       -0.2006630655
        0.0000000000       -0.2362551678        0.2006630655
        0.0000000000        0.0000000000        0.4698256163
        0.0000000000       -0.1475753470       -0.2349128081
        0.0000000000        0.1475753470       -0.2349128081
       -0.0355921801        0.0000000000        0.0000000000
        0.0234474448        0.0000000000        0.0000000000
        0.0121447353        0.0000000000        0.0000000000
        0.0000000000       -0.2362551678       -0.1475753470
        0.0000000000        0.2885428870        0.1741192062
        0.0000000000       -0.0522877192       -0.0265438592
        0.0000000000       -0.2006630655       -0.2349128081
        0.0000000000        0.1741192062        0.2234392406
        0.0000000000        0.0265438592        0.0114735676
       -0.0355921801        0.0000000000        0.0000000000
        0.0121447353        0.0000000000        0.0000000000
        0.0234474448        0.0000000000        0.0000000000
        0.0000000000       -0.2362551678        0.1475753470
        0.0000000000       -0.0522877192        0.0265438592
        0.0000000000        0.2885428870       -0.1741192062
        0.0000000000        0.2006630655       -0.2349128081
        0.0000000000       -0.0265438592        0.0114735676
        0.0000000000       -0.1741192062        0.2234392406
"""

_cfour_ref['neqh2o'] = {
    'xyz': c4_neqh2o_xyz,
    'mass': c4_neqh2o_mass,
    'fcm': {
        'analytic': c4_neqh2o_fcm_analytic,
        'findifproj': c4_neqh2o_fcm_findifproj,
        'findifrot': c4_neqh2o_fcm_findifrot,
    },
    'vibonly': {},
}

# <<<  Section VI: Non-stationary geometries  >>>

# Cfour on quantum chemists' water
#   cfour_CALC_level hf
#   basis cc-pvdz
#   cfour_SCF_CONV 12
#   cfour_vibration findif
#   cfour_vibration exact       # TOGGLE analytic Hess (analytic)
#   cfour_FD_PROJECT off        # TOGGLE Hess by findif G, project_rot=False --> RV-space (findifrot)
#   cfour_FD_PROJECT on         # TOGGLE Hess by findif G, project_rot=True --> V-space (findifproj)


def test_nonequilibrium_harmonic_analysis():
    w_right = 3448.7842  # value when rotations allowed to mix in at non-eq geom
    w_wrong = 3446.9559  # value when rotations not computed (findif) or rashly projected out (analysis)
    block = _cfour_ref['neqh2o']

    projtrans = True
    projrot = False

    analytic_vibinfo = vibanal_str(
        mass=block['mass'],
        coord=block['xyz'],
        fcm=block['fcm']['analytic'],
        project_trans=projtrans,
        project_rot=projrot)
    _cfour_ref['neqh2o']['vibonly']['analytic'] = qcdb.vib.filter_nonvib(analytic_vibinfo)

    findifproj_vibinfo = vibanal_str(
        mass=block['mass'],
        coord=block['xyz'],
        fcm=block['fcm']['findifproj'],
        project_trans=projtrans,
        project_rot=projrot)
    _cfour_ref['neqh2o']['vibonly']['findifproj'] = qcdb.vib.filter_nonvib(findifproj_vibinfo)

    findifrot_vibinfo = vibanal_str(
        mass=block['mass'],
        coord=block['xyz'],
        fcm=block['fcm']['findifrot'],
        project_trans=projtrans,
        project_rot=projrot)
    _cfour_ref['neqh2o']['vibonly']['findifrot'] = qcdb.vib.filter_nonvib(findifrot_vibinfo)

    assert compare_values(w_right, block['vibonly']['analytic']['omega'].data[-2].real, 0.2,
                          'Cfour analytic, analysis-T-projected mode')
    assert compare_values(w_wrong, block['vibonly']['findifproj']['omega'].data[-2].real, 0.2,
                          'Cfour G-findif-TR-projected, analysis-T-projected mode')
    assert compare_values(w_right, block['vibonly']['findifrot']['omega'].data[-2].real, 0.2,
                          'Cfour G-findif-T-projected, analysis-T-projected mode')

    assert qcdb.compare_vibinfos(block['vibonly']['analytic'], block['vibonly']['findifrot'], 3,
                                 'Cfour analytic vs. Cfour G-findif-T-projected')

    h2o = psi4.geometry(block['xyz'])

    psi4.set_options({
        'g_convergence': 'gau_verytight',
        'e_convergence': 10,
        'd_convergence': 10,
        'points': 5,
        'scf_type': 'pk'
    })

    # 1. findif-by-G, auto inclusion of rot dof b/c non-eq
    e, wfn = psi4.frequency('hf/cc-pvdz', return_wfn=True, molecule=h2o, dertype=1)
    pvibonly = qcdb.vib.filter_nonvib(wfn.frequency_analysis)

    assert qcdb.compare_vibinfos(block['vibonly']['analytic'], pvibonly, 3,
                                 'Cfour analytic vs. Psi4 G-findif-T-projected')

    # 2. analytic, always include rot dof
    e, wfn = psi4.frequency('hf/cc-pvdz', return_wfn=True, molecule=h2o, dertype=2)
    pvibonly = qcdb.vib.filter_nonvib(wfn.frequency_analysis)

    assert qcdb.compare_vibinfos(block['vibonly']['analytic'], pvibonly, 3, 'Cfour analytic vs. Psi4 analytic')

    # 3. even though same non-eq geometry, forcibly turn off rot dof
    psi4.set_options({'fd_project': True})
    e, wfn = psi4.frequency('hf/cc-pvdz', return_wfn=True, molecule=h2o, dertype=1)
    pvibonly = qcdb.vib.filter_nonvib(wfn.frequency_analysis)

    assert qcdb.compare_vibinfos(block['vibonly']['findifproj'], pvibonly, 3,
                                 'Cfour G-findif-TR-projected vs. Psi4 G-findif-TR-projected')


#@pytest.mark.parametrize("dertype,opts,ref,tagl", [
#    pytest.param(1, {}, 'analytic', 'Cfour analytic vs. Psi4 G-findif-T-projected'),
#    pytest.param(2, {}, 'analytic', 'Cfour analytic vs. Psi4 analytic'),
#    pytest.param(1, {'fd_project': True}, 'findifproj', 'Cfour G-findif-TR-projected vs. Psi4 G-findif-TR-projected'),
#])
#def test_rotational_projection(dertype, opts, ref, tagl):
#    block = _cfour_ref['neqh2o']
#    h2o = psi4.geometry(block['xyz'])
#
#    psi4.set_options({'g_convergence': 'gau_verytight',
#                      'e_convergence': 10,
#                      'd_convergence': 10,
#                      'points': 5,
#                      'scf_type': 'pk'})
#    psi4.set_options(opts)
#
#
#    # 1. findif-by-G, auto inclusion of rot dof b/c non-eq
#    # 2. analytic, always include rot dof
#    # 3. even though same non-eq geometry, forcibly turn off rot dof
#    e, wfn = psi4.frequency('hf/cc-pvdz', return_wfn=True, molecule=h2o, dertype=dertype)
#    pvibonly = qcdb.vib.filter_nonvib(wfn.frequency_analysis)
#
#    assert qcdb.compare_vibinfos(block['vibonly'][ref], pvibonly, 3, 'Cfour analytic vs. Psi4 G-findif-T-projected')
