/* Printer.cpp
 *
 * Copyright (C) 1998-2011,2012,2013,2014,2015 Paul Boersma
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * pb 2003/02/05 distinct treatment of NT and non-NT
 * pb 2004/05/25 font choice strategy
 * pb 2004/09/25 use /tmp as temporary directory
 * pb 2005/05/19 preference to switch screen preview off
 * pb 2005/07/21 removed bug in previous change for Linux
 * pb 2006/10/28 erased MacOS 9 stuff
 * pb 2006/12/28 theCurrentPraat
 * pb 2007/04/28 Mac: error messages for failing PostScript passthrough
 * pb 2007/08/12 wchar
 * pb 2007/10/05 less char
 * pb 2007/12/09 enums
 * pb 2009/01/18 arguments to UiForm callbacks
 * pb 2011/05/15 C++
 * pb 2011/07/11 C++
 */

#include <unistd.h> // close

#include "melder.h"

#if defined (_WIN32)
	#include <windows.h>
#endif

#include "Printer.h"
#include "praat.h"   // topShell
#include "Ui.h"
#include "site.h"
#include "GraphicsP.h"

#if cocoa
	#include "Picture.h"
#endif

/*
 * Everything must look the same on every printer, including on PDF,
 * so the margins must be constant with respect to the paper, not to the writable page.
 */

/* exported */ struct Printer thePrinter = {
	kGraphicsPostscript_spots_DEFAULT, kGraphicsPostscript_paperSize_DEFAULT, kGraphicsPostscript_orientation_DEFAULT, false,
	true, kGraphicsPostscript_fontChoiceStrategy_DEFAULT,
	600, 5100, 6600,
	1.0,
	NULL
};

void Printer_prefs () {
	Preferences_addEnum (U"Printer.spots", & thePrinter. spots, kGraphicsPostscript_spots, kGraphicsPostscript_spots_DEFAULT);
	Preferences_addEnum (U"Printer.paperSize", & thePrinter. paperSize, kGraphicsPostscript_paperSize, kGraphicsPostscript_paperSize_DEFAULT);
	Preferences_addBool (U"Printer.allowDirectPostScript", & thePrinter. allowDirectPostScript, true);
	Preferences_addEnum (U"Printer.fontChoiceStrategy", & thePrinter. fontChoiceStrategy, kGraphicsPostscript_fontChoiceStrategy, kGraphicsPostscript_fontChoiceStrategy_DEFAULT);
}

#if cocoa
	static NSView *theMacView;
#elif defined (macintosh)
	static PMPrintSession theMacPrintSession;
	static PMPageFormat theMacPageFormat;
	static PMPrintSettings theMacPrintSettings;
	static GrafPtr theMacPort;
	static PMRect paperSize;
#endif
#ifdef _WIN32
	static PRINTDLG theWinPrint;
	static HDC theWinDC;
#endif

#if defined (_WIN32) || defined (macintosh)
	int Printer_postScript_printf (void *stream, const char *format, ... ) {
		#if defined (_WIN32)
			static union { char chars [3002]; short shorts [1501]; } theLine;
		#elif cocoa
		#elif defined (macintosh)
			static Handle theLine;
		#endif
		int length;
		va_list args;
		va_start (args, format);
		(void) stream;
		#if cocoa
		#elif defined (_WIN32)
			vsprintf (theLine.chars + 2, format, args);
			length = strlen (theLine.chars + 2);
			theLine.shorts [0] = length;
			if (length > 0 && theLine.chars [length + 1] == '\n') {
				theLine.chars [length + 1] = '\r';
				theLine.chars [length + 2] = '\n';
				theLine.chars [length + 3] = '\0';
				length ++;
			}
			Escape (theWinDC, POSTSCRIPT_PASSTHROUGH, length + 2, theLine.chars, NULL);
		#elif defined (macintosh)
			if (! theLine) {
				theLine = NewHandle (3000);
				HLock (theLine);
			}
			vsprintf (*theLine, format, args);
			length = strlen (*theLine);
			if (length > 0 && (*theLine) [length - 1] == '\n')
				(*theLine) [length - 1] = '\r';
			SetPort (theMacPort);
			PMSessionPostScriptData (theMacPrintSession, *theLine, strlen (*theLine));
		#endif
		va_end (args);
		return 1;
	}
#endif

#if defined (_WIN32)
	static void initPostScriptPage () {
		/*
		 * Save the driver's state.
		 */
		Printer_postScript_printf (NULL, "/PraatPictureSaveObject save def\n");
		/*
		 * The LaserWriter driver puts the coordinates upside down.
		 * According to the PostScript Reference Manual,
		 * "There are few situations in which a PostScript language program
		 *  should execute initgraphics explicitly."
		 * This is one of them.
		 BUG: it probably is *not* one of them. Just do something like
		 	currentmatrix [1 0 -1 0 0 0] mul setmatrix
		 or whatever it is.
		 */
		#if 1
		Printer_postScript_printf (NULL, "initmatrix initclip\n");
		#else
Printer_postScript_printf (NULL, "8 8 scale initclip\n");
		#endif
	}
	static void exitPostScriptPage () {
		Printer_postScript_printf (NULL, "PraatPictureSaveObject restore\n");
	}
#endif

#if cocoa
#elif defined (_WIN32)
	static void initPrinter () {
	}
#elif defined (macintosh)
	static void initPrinter () {
		Boolean result;
		PMResolution res300 = { 300, 300 }, res600 = { 600, 600 };
		if (theMacPrintSettings == NULL) {   /* Once. */
			PMCreateSession (& theMacPrintSession);   /* Initialize the Printing Manager. */
			PMCreatePageFormat (& theMacPageFormat);
			PMCreatePrintSettings (& theMacPrintSettings);
			PMSessionDefaultPageFormat (theMacPrintSession, theMacPageFormat);
			PMSessionDefaultPrintSettings (theMacPrintSession, theMacPrintSettings);
		}
		PMSessionValidatePageFormat (theMacPrintSession, theMacPageFormat, & result);
		PMSessionValidatePrintSettings (theMacPrintSession, theMacPrintSettings, & result);
		/*
		 * BUG.
		 * If we now ask for the available printer resolutions,
		 * we may get the answer that there's only 300 dpi (perhaps PostScript drivers say so?).
		 * So we don't rely on that and have a buggy assumption instead.
		 */
		PMSetResolution (theMacPageFormat, & res300);   /* Perhaps all printers have this... */
		PMSetResolution (theMacPageFormat, & res600);   /* ... but this is preferred. */
	}
#endif

void Printer_nextPage () {
	#if cocoa
		[theMacView endPage];
		[theMacView beginPageInRect: [theMacView bounds] atPlacement: NSMakePoint (0, 0)];
	#elif defined (_WIN32)
		if (thePrinter. postScript) {
			exitPostScriptPage ();
			EndPage (theWinDC);
			StartPage (theWinDC);
			initPostScriptPage ();
		} else {
			if (EndPage (theWinDC) < 0) ;   /* BUG: should give the opportunity of cancellation. */
			StartPage (theWinDC);
			/*
			 * Since StartPage has reset the DC, restore some of our non-default settings.
			 */
			SetBkMode (theWinDC, TRANSPARENT);
			SetTextAlign (theWinDC, TA_LEFT | TA_BASELINE | TA_NOUPDATECP);
		}
	#elif defined (macintosh)
		PMSessionEndPage (theMacPrintSession);
		PMSessionBeginPage (theMacPrintSession, theMacPageFormat, NULL);
		PMSessionGetGraphicsContext (theMacPrintSession, kPMGraphicsContextQuickdraw, (void **) & theMacPort);
		SetPort (theMacPort);
		SetOrigin (- paperSize. left, - paperSize. top);
	#endif
}

int Printer_pageSetup () {
	#if cocoa
		NSPageLayout *cocoaPageSetupDialog = [NSPageLayout pageLayout];
		[cocoaPageSetupDialog runModal];
	#elif defined (_WIN32)
	#elif defined (macintosh)
		Boolean accepted;
		initPrinter ();
		PMSessionPageSetupDialog (theMacPrintSession, theMacPageFormat, & accepted);
	#endif
	return 1;
}

static void DO_Printer_postScriptSettings (UiForm dia, int /* narg */, Stackel /* args */,
	const char32 * /* sendingString_dummy */, Interpreter /* interpreter_dummy */,
	const char32 * /* invokingButtonTitle */, bool /* modified */, void *)
{
	#if defined (_WIN32)
		thePrinter. allowDirectPostScript = GET_INTEGER (U"Allow direct PostScript");
	#endif
	thePrinter. spots = GET_ENUM (kGraphicsPostscript_spots, U"Grey resolution");
	#if defined (UNIX)
		thePrinter. paperSize = GET_ENUM (kGraphicsPostscript_paperSize, U"Paper size");
	 	if (thePrinter. paperSize == kGraphicsPostscript_paperSize_A3) {
	 		thePrinter. paperWidth = 842 * thePrinter. resolution / 72;
	 		thePrinter. paperHeight = 1191 * thePrinter. resolution / 72;
		} else if (thePrinter. paperSize == kGraphicsPostscript_paperSize_US_LETTER) {
			thePrinter. paperWidth = 612 * thePrinter. resolution / 72;
			thePrinter. paperHeight = 792 * thePrinter. resolution / 72;
		} else {
			thePrinter. paperWidth = 595 * thePrinter. resolution / 72;
			thePrinter. paperHeight = 842 * thePrinter. resolution / 72;
		}
		thePrinter. orientation = GET_ENUM (kGraphicsPostscript_orientation, U"Orientation");
		thePrinter. magnification = GET_REAL (U"Magnification");
		Site_setPrintCommand (GET_STRING (U"printCommand"));
	#endif
	thePrinter. fontChoiceStrategy = GET_ENUM (kGraphicsPostscript_fontChoiceStrategy, U"Font choice strategy");
}

int Printer_postScriptSettings () {
	static UiForm dia;
	if (dia == NULL) {
		Any radio;
		dia = UiForm_create (theCurrentPraatApplication -> topShell, U"PostScript settings", DO_Printer_postScriptSettings, NULL, U"PostScript settings...", U"PostScript settings...");
		#if defined (_WIN32)
			BOOLEAN (U"Allow direct PostScript", true);
		#endif
		RADIO_ENUM (U"Grey resolution", kGraphicsPostscript_spots, DEFAULT)
		#if defined (UNIX)
			RADIO_ENUM (U"Paper size", kGraphicsPostscript_paperSize, DEFAULT);
			RADIO_ENUM (U"Orientation", kGraphicsPostscript_orientation, DEFAULT);
			POSITIVE (U"Magnification", U"1.0");
			LABEL (U"label", U"Print command:");
			#if defined (linux)
				TEXTFIELD (U"printCommand", U"lpr %s");
			#else
				TEXTFIELD (U"printCommand", U"lp -c %s");
			#endif
		#endif
		RADIO_ENUM (U"Font choice strategy", kGraphicsPostscript_fontChoiceStrategy, DEFAULT);
		UiForm_finish (dia);
	}
	#if defined (_WIN32)
		SET_INTEGER (U"Allow direct PostScript", thePrinter. allowDirectPostScript);
	#endif
	SET_ENUM (U"Grey resolution", kGraphicsPostscript_spots, thePrinter. spots);
	#if defined (UNIX)
		SET_ENUM (U"Paper size", kGraphicsPostscript_paperSize, thePrinter. paperSize);
		SET_ENUM (U"Orientation", kGraphicsPostscript_orientation, thePrinter. orientation);
		SET_REAL (U"Magnification", thePrinter. magnification);
		SET_STRING (U"printCommand", Site_getPrintCommand ());
	#endif
	SET_ENUM (U"Font choice strategy", kGraphicsPostscript_fontChoiceStrategy, thePrinter. fontChoiceStrategy);
	UiForm_do (dia, false);
	return 1;
}

#ifdef _WIN32
	static BOOL CALLBACK AbortFunc (HDC hdc, int nCode) {
		MSG msg;
		(void) hdc;
		(void) nCode;
		while (PeekMessage (& msg, 0, 0, 0, PM_REMOVE)) {
			TranslateMessage (& msg);
			DispatchMessage (& msg);
		}
		return true;
	}
	HDC Printer_getDC () {
		if (! theWinPrint. hDevMode) {
			memset (& theWinPrint, 0, sizeof (PRINTDLG));
			theWinPrint. lStructSize = sizeof (PRINTDLG);
			theWinPrint. Flags = PD_RETURNDEFAULT | PD_RETURNDC;
			PrintDlg (& theWinPrint);
		}
		return theWinPrint. hDC;
	}
#endif

#if cocoa
	static void (*theDraw) (void *boss, Graphics g);
	static void *theBoss;
	@interface GuiCocoaPrintingArea : NSView @end
	@implementation GuiCocoaPrintingArea {
		//GuiButton d_userData;
	}
	- (void) drawRect: (NSRect) dirtyRect {
		trace (U"printing ", dirtyRect. origin. x, U" ", dirtyRect. origin. y, U" ", dirtyRect. size. width, U" ", dirtyRect. size. height);
		int currentPage = [[NSPrintOperation currentOperation] currentPage];
		thePrinter. graphics = Graphics_create_screenPrinter (NULL, self);
		theDraw (theBoss, thePrinter. graphics);
		forget (thePrinter. graphics);
	}
	- (BOOL) isFlipped {
		return YES;
	}
	- (NSPoint) locationOfPrintRect: (NSRect) aRect {
		(void) aRect;
		return NSMakePoint (0.0, 0.0);   // the origin of the rect's coordinate system is always the top left corner of the physical page
	}
	- (BOOL) knowsPageRange: (NSRangePointer) range {
		range -> length = 1;
		return YES;
	}
	- (NSRect) rectForPage: (NSInteger) pageNumber {
		(void) pageNumber;   // every page has the same rectangle
		return [self bounds];
	}
	- (void) printOperationDidRun: (NSPrintOperation *) printOperation  success: (BOOL) success  contextInfo: (void *) contextInfo {
	}
	@end
#endif

int Printer_print (void (*draw) (void *boss, Graphics g), void *boss) {
	try {
		#if defined (UNIX)
			structMelderFile tempFile = { 0 };
			char tempPath_utf8 [] = "/tmp/picXXXXXX";
			close (mkstemp (tempPath_utf8));
			Melder_pathToFile (Melder_peek8to32 (tempPath_utf8), & tempFile);
			thePrinter. graphics = Graphics_create_postscriptjob (& tempFile, thePrinter. resolution,
				thePrinter. spots, thePrinter. paperSize, thePrinter. orientation, thePrinter. magnification);
			draw (boss, thePrinter. graphics);
			forget (thePrinter. graphics);
			char command [500];
			sprintf (command, Melder_peek32to8 (Site_getPrintCommand ()), tempPath_utf8);
			system (command);
			MelderFile_delete (& tempFile);
		#elif cocoa
			theDraw = draw;
			theBoss = boss;
			NSPrintInfo *info = [NSPrintInfo sharedPrintInfo];
			NSSize paperSize = [info paperSize];
			//NSLog (@"%f %f", paperSize. width, paperSize. height);
			thePrinter. paperWidth = paperSize. width / 0.12;
			thePrinter. paperHeight = paperSize. height / 0.12;
			[info setLeftMargin: 0.0];
			[info setRightMargin: 0.0];
			[info setTopMargin: 0.0];
			[info setBottomMargin: 0.0];
			/*
			 * Although the paper size reported may be 595 x 842 points (A4),
			 * 783 points (just under 11 inches) is the largest height that keeps the view on one page.
			 */
			int viewWidth = paperSize. width;
			int viewHeight = paperSize. height;
			NSLog (@"%d %d", viewWidth, viewHeight);
			NSRect rect = NSMakeRect (0, 0, viewWidth, viewHeight);
			NSView *cocoaPrintingArea = [[GuiCocoaPrintingArea alloc] initWithFrame: rect];
			theMacView = cocoaPrintingArea;
			[cocoaPrintingArea setBoundsSize: NSMakeSize (viewWidth / 0.12, viewHeight / 0.12)];   // 72 points per inch / 600 dpi = 0.12 points per dot
			[cocoaPrintingArea setBoundsOrigin: NSMakePoint (0, 0)];
			NSPrintOperation *op = [NSPrintOperation
				printOperationWithView: cocoaPrintingArea];
			#if 1
				if (op) [op runOperation];
			#else
				/*
				 * This may crash with multiple pages.
				 */
				if (op) {
					[op setCanSpawnSeparateThread: NO];
					NSView *pictureView = ((GraphicsScreen) Picture_getGraphics ((Picture) boss)) -> d_macView;
					[op
						runOperationModalForWindow: [pictureView window]
						delegate: cocoaPrintingArea
						didRunSelector: @selector(printOperationDidRun:success:contextInfo:)
						contextInfo: NULL
					];
				}
			#endif
		#elif defined (_WIN32)
			int postScriptCode = POSTSCRIPT_PASSTHROUGH;
			DOCINFO docInfo;
			DEVMODE *devMode;
			initPrinter ();
			if (! theWinPrint. hDevMode) {
				memset (& theWinPrint, 0, sizeof (PRINTDLG));
				theWinPrint. lStructSize = sizeof (PRINTDLG);
				theWinPrint. Flags = PD_RETURNDEFAULT;
				if (! PrintDlg (& theWinPrint)) Melder_throw (U"Cannot initialize printer.");
			}
			if (Melder_backgrounding) {
				theWinPrint. Flags = PD_RETURNDEFAULT | PD_RETURNDC;
				if (! PrintDlg (& theWinPrint) || theWinPrint. hDC == NULL) {
					Melder_throw (U"Cannot print from a script on this computer.");
				}
			} else {
				theWinPrint. Flags &= ~ PD_RETURNDEFAULT;
				theWinPrint. Flags |= PD_RETURNDC;
				if (! PrintDlg (& theWinPrint)) return 1;
			}
			theWinDC = theWinPrint. hDC;
			thePrinter. postScript = thePrinter. allowDirectPostScript &&
				Escape (theWinDC, QUERYESCSUPPORT, sizeof (int), (LPSTR) & postScriptCode, NULL);
			/*
			 * The HP colour inkjet printer returns in dmFields:
			 * 0, 1, 8, 9, 10, 11, 12, 13, 14, 15, 23, 24, 25, 26 = DM_ORIENTATION |
			 *    DM_PAPERSIZE | DM_COPIES | DM_DEFAULTSOURCE | DM_PRINTQUALITY |
			 *    DM_COLOR | DM_DUPLEX | DM_YRESOLUTION | DM_TTOPTION | DM_COLLATE |
			 *    DM_ICMMETHOD | DM_ICMINTENT | DM_MEDIATYPE | DM_DITHERTYPE
			 */
			devMode = * (DEVMODE **) theWinPrint. hDevMode;
			thePrinter. resolution = devMode -> dmFields & DM_YRESOLUTION ? devMode -> dmYResolution :
				devMode -> dmFields & DM_PRINTQUALITY ?
				( devMode -> dmPrintQuality > 0 ? devMode -> dmPrintQuality : 300 ) : 300;
			if (devMode -> dmFields & DM_PAPERWIDTH) {
				thePrinter. paperWidth = devMode -> dmPaperWidth * thePrinter. resolution / 254;
				thePrinter. paperHeight = devMode -> dmPaperLength * thePrinter. resolution / 254;
			} else if (devMode -> dmFields & DM_PAPERSIZE) {
				static struct { float width, height; } sizes [] = { { 0, 0 }, { 8.5, 11 }, { 8.5, 11 }, { 11, 17 },
					{ 17, 11 }, { 8.5, 14 }, { 5.5, 8.5 }, { 7.25, 10.5 }, { 297/25.4, 420/25.4 },
					{ 210/25.4, 297/25.4 }, { 210/25.4, 297/25.4 }, { 148.5/25.4, 210/25.4 },
					{ 250/25.4, 354/25.4 }, { 182/25.4, 257/25.4 }, { 8.5, 13 },
					{ 215/25.4, 275/25.4 }, { 10, 14 }, { 11, 17 }, { 8.5, 11 }, { 3.875, 8.875 },
					{ 4.125, 9.5 }, { 4.5, 10.375 } };
				int paperSize = devMode -> dmPaperSize;
				if (paperSize <= 0 || paperSize > 21) paperSize = 1;
				thePrinter. paperWidth = sizes [paperSize]. width * thePrinter. resolution;
				thePrinter. paperHeight = sizes [paperSize]. height * thePrinter. resolution;
				if (devMode -> dmOrientation == DMORIENT_LANDSCAPE) {
					long dummy = thePrinter. paperWidth;
					thePrinter. paperWidth = thePrinter. paperHeight;
					thePrinter. paperHeight = dummy;
				}
			} else {
				thePrinter. paperWidth = 1000;
				thePrinter. paperHeight = 1000;
			}
			EnableWindow ((HWND) XtWindow (theCurrentPraatApplication -> topShell -> d_xmShell), false);
			SetAbortProc (theWinDC, AbortFunc);
			memset (& docInfo, 0, sizeof (DOCINFO));
			docInfo. cbSize = sizeof (DOCINFO);
			docInfo. lpszDocName = L"Praatjes";
			docInfo. lpszOutput = NULL;
			if (thePrinter. postScript) {
				StartDoc (theWinDC, & docInfo);
				StartPage (theWinDC);
				initPostScriptPage ();
				thePrinter. graphics = Graphics_create_postscriptprinter ();
				draw (boss, thePrinter. graphics);
				forget (thePrinter. graphics);
				exitPostScriptPage ();
				EndPage (theWinDC);
				EndDoc (theWinDC);
			} else {
				StartDoc (theWinDC, & docInfo);
				StartPage (theWinDC);
				thePrinter. graphics = Graphics_create_screenPrinter (NULL, theWinDC);
				draw (boss, thePrinter. graphics);
				forget (thePrinter. graphics);
				if (EndPage (theWinDC) < 0) {
					Melder_throw (U"Cannot print page.");
				} else {
					EndDoc (theWinDC);
				}
			}
			EnableWindow ((HWND) XtWindow (theCurrentPraatApplication -> topShell -> d_xmShell), true);
			DeleteDC (theWinDC), theWinDC = NULL;
		#elif defined (macintosh)
			Boolean result;
			initPrinter ();
			if (Melder_backgrounding) {
				PMSessionValidatePageFormat (theMacPrintSession, theMacPageFormat, & result);
				PMSessionValidatePrintSettings (theMacPrintSession, theMacPrintSettings, & result);
			} else {
				Boolean accepted;
				PMSessionPrintDialog (theMacPrintSession, theMacPrintSettings, theMacPageFormat, & accepted);
				if (! accepted) return 1;   // normal cancelled return
			}
			PMSessionValidatePageFormat (theMacPrintSession, theMacPageFormat, & result);
			PMSessionValidatePrintSettings (theMacPrintSession, theMacPrintSettings, & result);
			PMResolution res;
			PMGetResolution (theMacPageFormat, & res);
			thePrinter. resolution = res. hRes;
			PMGetAdjustedPaperRect (theMacPageFormat, & paperSize);
			thePrinter. paperWidth = paperSize. right - paperSize. left;
			thePrinter. paperHeight = paperSize. bottom - paperSize. top;
			thePrinter. postScript = false;
			PMOrientation orientation;
			PMGetOrientation (theMacPageFormat, & orientation);
			thePrinter. orientation = orientation == kPMLandscape ||
				orientation == kPMReverseLandscape ? kGraphicsPostscript_orientation_LANDSCAPE : kGraphicsPostscript_orientation_PORTRAIT;
			PMSessionBeginDocument (theMacPrintSession, theMacPrintSettings, theMacPageFormat);
			PMSessionBeginPage (theMacPrintSession, theMacPageFormat, NULL);
			PMSessionGetGraphicsContext (theMacPrintSession, kPMGraphicsContextQuickdraw, (void **) & theMacPort);
			/*
			 * On PostScript, the point (0, 0) is the bottom left corner of the paper, which is fine.
			 * On the screen, however, the point (0, 0) is the top left corner of the writable page.
			 * Since we want paper-related margins, not writable-page-related margins,
			 * we require that this point gets the coordinates (250, 258) or so,
			 * so that the top left corner of the paper gets coordinates (0, 0).
			 * The "left" and "top" attributes of rPaper are negative values (e.g. -250 and -258),
			 * so multiply them by -1.
			 *
			 * Under Carbon, the port has to be set inside the page.
			 */
			SetPort (theMacPort);
			SetOrigin (- paperSize. left, - paperSize. top);
			thePrinter. graphics = Graphics_create_screenPrinter (NULL, theMacPort);
			draw (boss, thePrinter. graphics);
			forget (thePrinter. graphics);
			if (theMacPort) {
				PMSessionEndPage (theMacPrintSession);
				PMSessionEndDocument (theMacPrintSession);
				theMacPort = NULL;
			}
		#endif
		return 1;
	} catch (MelderError) {
		#if cocoa
		#elif defined (macintosh)
			if (theMacPort) {
				PMSessionEndPage (theMacPrintSession);
				PMSessionEndDocument (theMacPrintSession);
				theMacPort = NULL;
			}
		#endif
		Melder_throw (U"Not printed.");
	}
}

/* End of file Printer.cpp */
