/*	Icons

PIRL CVS ID: Icons.java,v 1.15 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2005-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import javax.swing.ImageIcon;
import java.net.URL;
import java.net.MalformedURLException;
import java.io.File;
import java.lang.SecurityException;

//	For the test stub.
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JLabel;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;


/**	The <i>Icons</i> class provides static methods to load ImageIcons.
<p>
	@author	Bradford Castalia
	@version	1.15
*/
public class Icons
{
//!	Class ID.
public static final String
	ID = "PIRL.Utilities.Icons (1.15 2012/04/16 06:22:58)";

//!	Default directory name for icon files.
public static final String
	DEFAULT_ICON_DIRECTORY_NAME = "Icons";


private static File
	Directory				= new File (DEFAULT_ICON_DIRECTORY_NAME);

private static Class
	Class_Relative			= null;

private static final char
	FILE_SEPARATOR 			= System.getProperty ("file.separator").charAt (0);


//	DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_UTILITIES			= 1 << 0,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;	

/*==============================================================================
	Constructors
*/
//	This class only contains static functions.
private Icons ()
{}

/*==============================================================================
	Functions
*/
/**	Loads an ImageIcon from a source.
<p>
	If the source is null or the empty String, nothing is done and null
	is returned.
<p>
	First, an attempt is made to construct a URL from the source String.
	If this succeeds the URL is used to try and load an ImageIcon. No
	other attempts are made to find the source.
<p>
	Next the source is treated as a file pathname. If this pathname is
	absolute or the source is not {@link #Class_Relative()
	class-relative} an attempt will be made to load an ImageIcon from the
	pathname. If the pathname is relative it is prepended with the icon
	{@link #Directory(String) directory} or {@link #Directory(String,
	Class) class-relative directory} pathname, whichever is current. If
	the resultant pathname is for a normal file (rather than a directory)
	and read access is permitted by any existing security manager the
	pathname is used to try and load an ImageIcon.
<p>
	If an ImageIcon is not yet loaded the pathname from the previous
	attempt is used to try and locate a system resource.  This resource 
	may be found relative to the CLASSPATH or inside a jar file. If this
	succeeds in producing a URL it is used to try and load an ImageIcon.
<p>
	@param	source	The URL or pathname for the icon image source.
	@return	An ImageIcon, or null if one can not be found or loaded.
*/
public static ImageIcon Load_Icon
	(
	String source
	)
{
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println (">>> Icons.Load_Icon: " + source);
if (source == null ||
	source.length () == 0)
	{
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println ("<<< Icons.Load_Icon");
	return null;
	}

ImageIcon
	image_icon = null;
URL
	url = null;

//	Treat the source as a URL.
try {url = new URL (source);}
catch (MalformedURLException exception) {}
if (url != null)
	{
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println ("    Using URL " + url);
	image_icon = new ImageIcon (url);
	}
else
	{
	//	Treat the source as a file pathname.
	boolean
		try_file = (Class_Relative == null);
	File
		file = new File (source);
	if (file.isAbsolute ())
		try_file = true;
	else
		file = new File (Directory.getPath () + FILE_SEPARATOR + source);
	if (try_file)
		{
		if ((DEBUG & DEBUG_UTILITIES) != 0)
			System.out.println
				("    Trying file " + file);
		if (file.canRead () &&
			file.isFile ())
			{
			try {image_icon = new ImageIcon (file.getPath ());}
			catch (SecurityException exception) {}
			}
		else if ((DEBUG & DEBUG_UTILITIES) != 0)
			System.out.println ("    Not readable");
		}

	if (image_icon == null)
		{
		//	Treat the source as a system resource.
		if ((DEBUG & DEBUG_UTILITIES) != 0)
			System.out.println
				("    Trying system resource at " + file.getPath ());
		url = ClassLoader.getSystemResource (file.getPath ());
		if (url != null)
			{
			if ((DEBUG & DEBUG_UTILITIES) != 0)
				System.out.println
					("    Using system resource URL " + url);
			image_icon = new ImageIcon (url);
			}
		}
	}
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println
		("<<< Icons.Load_Icon: "
			+ ((image_icon == null) ? "not " : "") + "found");
return image_icon;
}

/**	Sets the name of the directory where icon image files are to be found.
<p>
	<b>N.B.</b>: The directory pathname is marked as being not {@link
	#Class_Relative() class-relative}.
<p>
	@param	directory_path	The pathname for the directory. If null or empty
		the {@link #DEFAULT_ICON_DIRECTORY_NAME} will be used.
	@return	The previous directory pathname.
*/
public static String Directory
	(
	String	directory_path
	)
{
String
	previous_directory_path = Directory.getPath ();
if (directory_path == null ||
	directory_path.length () == 0)
	Directory = new File (DEFAULT_ICON_DIRECTORY_NAME);
else
	Directory = new File (directory_path);
Class_Relative = null;
return previous_directory_path;
}

/**	Sets the name of the directory where icon image files may be found
	relative to the location of a class.
<p>
	If the directory name is null or empty the {@link
	#DEFAULT_ICON_DIRECTORY_NAME} will be used. If the directory name is
	an absolute pathname, the class location is not used. Otherwise the
	directory pathname to the class location is prepended to the
	directory name and the resultant {@link #Directory() directory
	pathname} is marked as being {@link #Class_Relative()
	class-relative}.
<p>
	@param	directory_path	The pathname for the icons directory. If
		this is a relative pathname it will be relative to the location
		of the class.
	@param	the_class	The Class location to use as the base for a
		relative directory pathname. Ignored if null.
	@return	The previous directory pathname.
*/
public static String Directory
	(
	String	directory_path,
	Class	the_class
	)
{
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println
		(">>> Icons.Directory: " + directory_path);
String
	previous_directory_path = Directory (directory_path);
if (the_class != null &&
	! Directory.isAbsolute ())
	{
	String
		class_name = the_class.getName ();
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println
			("    Class name: " + class_name);
	int
		index = class_name.lastIndexOf ('.');
	if (index >= 0)
		/*
			Get the CLASSPATH relative driectory path to the icons
			by removing the final class name segment to get the package name,
			replacing the Java '.' segment delimiters with file separators,
			and appending the directory path afer a file separator.
		*/
		Directory = new File
			(class_name.substring (0, index)
				.replace ('.', FILE_SEPARATOR)
			+ FILE_SEPARATOR
			+ Directory.getPath ());
	Class_Relative = the_class;
	}
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println
		("<<< Icons.Directory: " + Directory);
return previous_directory_path;
}

/**	Gets the Class to which the current {@link #Directory() directory}
	is relative.
<p>
	@return	The last Class used to set a {@link #Directory(String, Class)
		class-relative} directory pathname. This will be null if the
		current {@link #Directory() directory pathname} is not
		class-relative.
*/
public static Class Class_Relative ()
{return Class_Relative;}

/**	Gets the pathname of the directory where icon image files are to be found.
<p>
	@return	The directory name.
*/
public static String Directory ()
{return Directory.getPath ();}


/*==============================================================================
	Test stub
*/
public static void main (String arguments[])
{
System.out.println (ID);
if (arguments.length > 0)
	{
	System.out.println ("Default Directory: " + Directory ());
	ImageIcon
		icon = Load_Icon (arguments[0]);
	if (icon == null)
		{
		System.out.println ("No icon at " + arguments[0]);
		Class
			this_class = null;
		try {this_class = Class.forName ("PIRL.Viewers.Icons");}
		catch (ClassNotFoundException exception)
			{
			System.out.println ("Icons class not found!");
			System.exit (1);
			}
		Directory (Directory (), this_class);
		System.out.println ("Class-relative Directory: " + Directory ());
		icon = Load_Icon (arguments[0]);
		}
	if (icon != null)
		{
		final JFrame
			frame = new JFrame ("Icon " + arguments[0]);
		frame.addWindowListener (new WindowAdapter ()
			{public void windowClosing (WindowEvent e) {System.exit (0);}});
		JPanel
			content = new JPanel ();
		frame.setContentPane (content);
		JLabel
			label = new JLabel (icon);
		content.add (label);
		frame.pack ();
		frame.setVisible (true);
		}
	else
		System.out.println ("Icon file not found.");
	}
else
	System.out.println ("Usage: Icons <filename>");
}

}
