#include "osl/search/simpleHashTable.h"
#include "osl/search/simpleHashRecord.h"
#include "osl/moveLogProb.h"
#include "osl/hash/hashKey.h"
//#include "osl/state/simpleHashState.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

// 一部を simpleHashRecordTest に分離する必要がある

class SimpleHashTableTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(SimpleHashTableTest);
  CPPUNIT_TEST(testCreation);
  CPPUNIT_TEST(testSize);
  CPPUNIT_TEST(testAllocate);
  CPPUNIT_TEST(testIsRecordedUpper);
  CPPUNIT_TEST(testIsRecordedLower);
  CPPUNIT_TEST(testOverWrite);
  CPPUNIT_TEST(testCapacity);
  CPPUNIT_TEST(testClear);
  CPPUNIT_TEST(testSetMinimumRecordLimit);
  CPPUNIT_TEST_SUITE_END();
public:
  void testCreation();
  void testSize();
  void testAllocate();
  void testIsRecordedUpper();
  void testIsRecordedLower();
  void testOverWrite();
  void testCapacity();
  void testClear();
  void testSetMinimumRecordLimit();
};

CPPUNIT_TEST_SUITE_REGISTRATION(SimpleHashTableTest);

using namespace osl;
using namespace osl::search;

void SimpleHashTableTest::testCreation() {
  SimpleHashTable table(1,1,false);
  CPPUNIT_ASSERT_EQUAL(1, table.minimumRecordLimit());

  SimpleHashTable table2(3,-4,false);
  CPPUNIT_ASSERT_EQUAL(-4, table2.minimumRecordLimit());
}

void SimpleHashTableTest::testSize() {
  SimpleHashTable table(1,1,false);
  CPPUNIT_ASSERT_EQUAL((size_t)0, table.size());
}

void SimpleHashTableTest::testAllocate() {
  SimpleHashTable table(100,1,false);
  HashKey k;
  CPPUNIT_ASSERT_EQUAL((size_t)0, table.size());

  SimpleHashRecord *record = table.allocate(k, 100);
  CPPUNIT_ASSERT_EQUAL((size_t)1, table.size());
  CPPUNIT_ASSERT(! record->hasLowerBound(0));
  CPPUNIT_ASSERT(! record->hasUpperBound(0));
}

void SimpleHashTableTest::testIsRecordedUpper() {
  SimpleHashTable table(100,1,false);
  MoveLogProb m1;
  const HashKey k;
  const int limit = 100;
  const int value = 8;
  {
    SimpleHashRecord *record = table.allocate(k, limit);
    record->setUpperBound(BLACK, limit, m1, value);
  }
  const SimpleHashRecord *record = table.find(k);
  CPPUNIT_ASSERT(record);
  CPPUNIT_ASSERT_EQUAL(value, record->upperBound());
}

void SimpleHashTableTest::testIsRecordedLower() {
  SimpleHashTable table(100,1,false);
  MoveLogProb m1;
  const HashKey k;
  const int limit = 100;
  const int value = 8;
  {
    SimpleHashRecord *record = table.allocate(k, limit);
    record->setLowerBound(BLACK, limit, m1, value);
  }
  const SimpleHashRecord *record = table.find(k);
  CPPUNIT_ASSERT(record);
  CPPUNIT_ASSERT_EQUAL(value, record->lowerBound());
}

/** 
 * 2回登録したら深いほうの結果が残る 
 */
void SimpleHashTableTest::testOverWrite() {
  const HashKey k;
  MoveLogProb m1;
  const int shallow_limit = 100;
  const int shallow_value = 8;
  const int deep_limit = 200;
  const int deep_value = 888;
  {
    // shallow -> deep
    SimpleHashTable table(100,1,false);
    SimpleHashRecord *record = table.allocate(k, shallow_limit);
    record->setLowerBound(BLACK, shallow_limit, m1, shallow_value);
    record->setLowerBound(BLACK, deep_limit, m1, deep_value);
    
    CPPUNIT_ASSERT(record->hasLowerBound(deep_limit));
    CPPUNIT_ASSERT_EQUAL(deep_value, record->lowerBound());
  }
  {
    // deep -> shallow
    SimpleHashTable table(100,1,false);
    SimpleHashRecord *record = table.allocate(k, shallow_limit);
    record->setLowerBound(BLACK, deep_limit, m1, deep_value);
    record->setLowerBound(BLACK, shallow_limit, m1, shallow_value);

    CPPUNIT_ASSERT(record->hasLowerBound(deep_limit));
    CPPUNIT_ASSERT_EQUAL(deep_value, record->lowerBound());
  }
}

/** capacity を越えて insert しない */
void SimpleHashTableTest::testCapacity() {
  const size_t capacity = 128;
  SimpleHashTable table(capacity,1,false);
  CPPUNIT_ASSERT_EQUAL(capacity, table.capacity());
  int table_full = 0;
  
  for (unsigned int i=1; i<=capacity*5; ++i)
  {
    HashKey k;
    k.setRandom();
    const int limit = i;
    // const int value = i*2;
    if (table.size() < capacity)
    {
      try 
      {
	SimpleHashRecord *r = table.allocate(k, limit);
	CPPUNIT_ASSERT_EQUAL((size_t)i, table.size()+table_full);
	const SimpleHashRecord *record = table.find(k);
	// std::cerr << r << " " << record << "\n";
	CPPUNIT_ASSERT_EQUAL(const_cast<const SimpleHashRecord*>(r), record);
      }
      catch (TableFull&)
      {
	CPPUNIT_ASSERT(table.size() >= (capacity / table.divSize()));
	++table_full;
      }
    }
    else 
    {
      if (table.find(k) || (limit < table.minimumRecordLimit()))
      {
	// already recorded
	CPPUNIT_ASSERT_EQUAL(capacity, table.size());
	table.allocate(k, limit);
	CPPUNIT_ASSERT_EQUAL(capacity, table.size());
      }
      else
      {
	try
	{
	  CPPUNIT_ASSERT_EQUAL(capacity, table.size());
	  table.allocate(k, limit);
	  CPPUNIT_ASSERT(! table.find(k));
	  CPPUNIT_ASSERT_EQUAL(capacity, table.size());
	}
	catch (TableFull&)
	{
	  CPPUNIT_ASSERT(! table.find(k));
	  CPPUNIT_ASSERT_EQUAL(capacity, table.size());
	}
      }
    }
  }
}

void SimpleHashTableTest::testClear() {
  SimpleHashTable table(100,1,false);
  const HashKey k;
  CPPUNIT_ASSERT_EQUAL((size_t)0, table.size());
  table.allocate(k, 100);
  CPPUNIT_ASSERT_EQUAL((size_t)1, table.size());
  table.clear();
  CPPUNIT_ASSERT_EQUAL((size_t)0, table.size());
}

void SimpleHashTableTest::testSetMinimumRecordLimit() {
  SimpleHashTable table(100,10000,false);
  const HashKey k;
  table.setMinimumRecordLimit(1000);
  table.allocate(k, 100);
  CPPUNIT_ASSERT_EQUAL((size_t)0, table.size());
  table.allocate(k, 10000);
  CPPUNIT_ASSERT_EQUAL((size_t)1, table.size());
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
