/* Data types for DPD format output.

   Copyright (C) 2006, 2007 IBM Corporation.
   Copyright (C) 2007-2015 Free Software Foundation, Inc.

   This file is part of the Decimal Floating Point C Library.

   Author(s): Steve Munroe <sjmunroe@us.ibm.com>

   The Decimal Floating Point C Library is free software; you can
   redistribute it and/or modify it under the terms of the GNU Lesser
   General Public License version 2.1.

   The Decimal Floating Point C Library is distributed in the hope that
   it will be useful, but WITHOUT ANY WARRANTY; without even the implied
   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
   the GNU Lesser General Public License version 2.1 for more details.

   You should have received a copy of the GNU Lesser General Public
   License version 2.1 along with the Decimal Floating Point C Library;
   if not, write to the Free Software Foundation, Inc., 59 Temple Place,
   Suite 330, Boston, MA 02111-1307 USA.

   Please see dfp/COPYING.txt for more information.  */

#include <float.h>
#include <math.h>
#include <fenv.h>
#include <string.h>

#include "dpd-private.h"
#include "get_digits.h"

const struct ieee754r_c_field c_decoder[32] = {
	{0, 0, 0, 0}, {0, 0, 0, 1}, {0, 0, 0, 2}, {0, 0, 0, 3},	/* 00000-00011 */
	{0, 0, 0, 4}, {0, 0, 0, 5}, {0, 0, 0, 6}, {0, 0, 0, 7},	/* 00100-00111 */
	{0, 0, 1, 0}, {0, 0, 1, 1}, {0, 0, 1, 2}, {0, 0, 1, 3},	/* 01000-01011 */
	{0, 0, 1, 4}, {0, 0, 1, 5}, {0, 0, 1, 6}, {0, 0, 1, 7},	/* 01100-01111 */
	{0, 0, 2, 0}, {0, 0, 2, 1}, {0, 0, 2, 2}, {0, 0, 2, 3},	/* 10000-10011 */
	{0, 0, 2, 4}, {0, 0, 2, 5}, {0, 0, 2, 5}, {0, 0, 2, 7},	/* 10100-10111 */
	{0, 0, 0, 8}, {0, 0, 0, 9}, {0, 0, 1, 8}, {0, 0, 1, 9},	/* 11000-11011 */
	{0, 0, 2, 8}, {0, 0, 2, 9}, {0, 1, 0, 0}, {1, 0, 0, 0}	/* 11100-11111 */
	};

/* Table to encode the "left most" 2 bits of the biased exponent and the
   "Left Most Digit" into the "C"ombination field.  Indexed by [lmd,lm2bits].  */
const unsigned char lm2lmd_to_c[10][4] = {
    {0x00, 0x08, 0x10, 0x1e},
    {0x01, 0x09, 0x11, 0x1e},
    {0x02, 0x0a, 0x12, 0x1e},
    {0x03, 0x0b, 0x13, 0x1e},
    {0x04, 0x0c, 0x14, 0x1e},
    {0x05, 0x0d, 0x15, 0x1e},
    {0x06, 0x0e, 0x16, 0x1e},
    {0x07, 0x0f, 0x17, 0x1e},
    {0x18, 0x1a, 0x1c, 0x1e},
    {0x19, 0x1b, 0x1d, 0x1e}};

/* Table to convert 3 Densely Packed Decimal digits (10-bits) into
   3 ASCII digits (24-bits, 3 bytes).  */
const char	dpd_to_char[1024][4] = {
	"000", "001", "002", "003", "004", "005", "006", "007", /* 000-007 */
	"008", "009", "080", "081", "800", "801", "880", "881", /* 008-00f */
	"010", "011", "012", "013", "014", "015", "016", "017", /* 010-017 */
	"018", "019", "090", "091", "810", "811", "890", "891", /* 018-01f */
	"020", "021", "022", "023", "024", "025", "026", "027", /* 020-027 */
	"028", "029", "082", "083", "820", "821", "808", "809", /* 028-02f */
	"030", "031", "032", "033", "034", "035", "036", "037", /* 030-037 */
	"038", "039", "092", "093", "830", "831", "818", "819", /* 038-03f */

	"040", "041", "042", "043", "044", "045", "046", "047", /* 040-047 */
	"048", "049", "084", "085", "840", "841", "088", "089", /* 048-04f */
	"050", "051", "052", "053", "054", "055", "056", "057", /* 050-057 */
	"058", "059", "094", "095", "850", "851", "098", "099", /* 058-05f */
	"060", "061", "062", "063", "064", "065", "066", "067", /* 060-067 */
	"068", "069", "086", "087", "860", "861", "888", "889", /* 068-06f */
	"070", "071", "072", "073", "074", "075", "076", "077", /* 070-077 */
	"078", "079", "096", "097", "870", "871", "898", "899", /* 078-07f */

	"100", "101", "102", "103", "104", "105", "106", "107", /* 080-087 */
	"108", "109", "180", "181", "900", "901", "980", "981", /* 088-08f */
	"110", "111", "112", "113", "114", "115", "116", "117", /* 090-097 */
	"118", "119", "190", "191", "910", "911", "990", "991", /* 098-09f */
	"120", "121", "122", "123", "124", "125", "126", "127", /* 0a0-0a7 */
	"128", "129", "182", "183", "920", "921", "908", "909", /* 0a8-0af */
	"130", "131", "132", "133", "134", "135", "136", "137", /* 0b0-0b7 */
	"138", "139", "192", "193", "930", "931", "918", "919", /* 0b8-0bf */

	"140", "141", "142", "143", "144", "145", "146", "147", /* 0c0-0c7 */
	"148", "149", "184", "185", "940", "941", "188", "189", /* 0c8-0cf */
	"150", "151", "152", "153", "154", "155", "156", "157", /* 0d0-0d7 */
	"158", "159", "194", "195", "950", "951", "198", "199", /* 0d8-0df */
	"160", "161", "162", "163", "164", "165", "166", "167", /* 0e0-0e7 */
	"168", "169", "186", "187", "960", "961", "988", "989", /* 0e8-0ef */
	"170", "171", "172", "173", "174", "175", "176", "177", /* 0f0-0f7 */
	"178", "179", "196", "197", "970", "971", "998", "999", /* 0f8-0ff */

	"200", "201", "202", "203", "204", "205", "206", "207", /* 100-107 */
	"208", "209", "280", "281", "802", "803", "882", "883", /* 108-10f */
	"210", "211", "212", "213", "214", "215", "216", "217", /* 110-117 */
	"218", "219", "290", "291", "812", "813", "892", "893", /* 118-11f */
	"220", "221", "222", "223", "224", "225", "226", "227", /* 120-127 */
	"228", "229", "282", "283", "822", "823", "828", "829", /* 128-12f */
	"230", "231", "232", "233", "234", "235", "236", "237", /* 130-137 */
	"238", "239", "292", "293", "832", "833", "838", "839", /* 138-13f */

	"240", "241", "242", "243", "244", "245", "246", "247", /* 140-147 */
	"248", "249", "284", "285", "842", "843", "288", "289", /* 148-14f */
	"250", "251", "252", "253", "254", "255", "256", "257", /* 150-157 */
	"258", "259", "294", "295", "852", "853", "298", "299", /* 158-15f */
	"260", "261", "262", "263", "264", "265", "266", "267", /* 160-167 */
	"268", "269", "286", "287", "862", "863", "888", "889", /* 168-16f */
	"270", "271", "272", "273", "274", "275", "276", "277", /* 170-177 */
	"278", "279", "296", "297", "872", "873", "898", "899", /* 178-17f */

	"300", "301", "302", "303", "304", "305", "306", "307", /* 180-187 */
	"308", "309", "380", "381", "902", "903", "982", "983", /* 188-18f */
	"310", "311", "312", "313", "314", "315", "316", "317", /* 190-197 */
	"318", "319", "390", "391", "912", "913", "992", "993", /* 198-19f */
	"320", "321", "322", "323", "324", "325", "326", "327", /* 1a0-1a7 */
	"328", "329", "382", "383", "922", "923", "928", "929", /* 1a8-1af */
	"330", "331", "332", "333", "334", "335", "336", "337", /* 1b0-1b7 */
	"338", "339", "392", "393", "932", "933", "938", "939", /* 1b8-1bf */

	"340", "341", "342", "343", "344", "345", "346", "347", /* 1c0-1c7 */
	"348", "349", "384", "385", "942", "943", "388", "389", /* 1c8-1cf */
	"350", "351", "352", "353", "354", "355", "356", "357", /* 1d0-1d7 */
	"358", "359", "394", "395", "952", "953", "398", "399", /* 1d8-1df */
	"360", "361", "362", "363", "364", "365", "366", "367", /* 1e0-1e7 */
	"368", "369", "386", "387", "962", "963", "988", "989", /* 1e8-1ef */
	"370", "371", "372", "373", "374", "375", "376", "377", /* 1f0-1f7 */
	"378", "379", "396", "397", "972", "973", "998", "999", /* 1f8-1ff */

	"400", "401", "402", "403", "404", "405", "406", "407", /* 200-207 */
	"408", "409", "480", "481", "804", "805", "884", "885", /* 208-20f */
	"410", "411", "412", "413", "414", "415", "416", "417", /* 210-217 */
	"418", "419", "490", "491", "814", "815", "894", "895", /* 218-21f */
	"420", "421", "422", "423", "424", "425", "426", "427", /* 220-227 */
	"428", "429", "482", "483", "824", "825", "848", "849", /* 228-22f */
	"430", "431", "432", "433", "434", "435", "436", "437", /* 230-237 */
	"438", "439", "492", "493", "834", "835", "858", "859", /* 238-23f */

	"440", "441", "442", "443", "444", "445", "446", "447", /* 240-247 */
	"448", "449", "484", "485", "844", "845", "488", "489", /* 248-24f */
	"450", "451", "452", "453", "454", "455", "456", "457", /* 250-257 */
	"458", "459", "494", "495", "854", "855", "498", "499", /* 258-25f */
	"460", "461", "462", "463", "464", "465", "466", "467", /* 260-267 */
	"468", "469", "486", "487", "864", "865", "888", "889", /* 268-26f */
	"470", "471", "472", "473", "474", "475", "476", "477", /* 270-277 */
	"478", "479", "496", "497", "874", "875", "898", "899", /* 278-27f */

	"500", "501", "502", "503", "504", "505", "506", "507", /* 280-287 */
	"508", "509", "580", "581", "904", "905", "984", "985", /* 288-28f */
	"510", "511", "512", "513", "514", "515", "516", "517", /* 290-297 */
	"518", "519", "590", "591", "914", "915", "994", "995", /* 298-29f */
	"520", "521", "522", "523", "524", "525", "526", "527", /* 2a0-2a7 */
	"528", "529", "582", "583", "924", "925", "948", "949", /* 2a8-2af */
	"530", "531", "532", "533", "534", "535", "536", "537", /* 2b0-2b7 */
	"538", "539", "592", "593", "934", "935", "958", "959", /* 2b8-2bf */

	"540", "541", "542", "543", "544", "545", "546", "547", /* 2c0-2c7 */
	"548", "549", "584", "585", "944", "945", "588", "589", /* 2c8-2cf */
	"550", "551", "552", "553", "554", "555", "556", "557", /* 2d0-2d7 */
	"558", "559", "594", "595", "954", "955", "598", "599", /* 2d8-2df */
	"560", "561", "562", "563", "564", "565", "566", "567", /* 2e0-2e7 */
	"568", "569", "586", "587", "964", "965", "988", "989", /* 2e8-2ef */
	"570", "571", "572", "573", "574", "575", "576", "577", /* 2f0-2f7 */
	"578", "579", "596", "597", "974", "975", "998", "999", /* 2f8-2ff */

	"600", "601", "602", "603", "604", "605", "606", "607", /* 300-307 */
	"608", "609", "680", "681", "806", "807", "886", "887", /* 308-30f */
	"610", "611", "612", "613", "614", "615", "616", "617", /* 310-317 */
	"618", "619", "690", "691", "816", "817", "896", "897", /* 318-31f */
	"620", "621", "622", "623", "624", "625", "626", "627", /* 320-327 */
	"628", "629", "682", "683", "826", "827", "868", "869", /* 328-32f */
	"630", "631", "632", "633", "634", "635", "636", "637", /* 330-337 */
	"638", "639", "692", "693", "836", "837", "878", "879", /* 338-33f */

	"640", "641", "642", "643", "644", "645", "646", "647", /* 340-347 */
	"648", "649", "684", "685", "846", "847", "688", "689", /* 348-34f */
	"650", "651", "652", "653", "654", "655", "656", "657", /* 350-357 */
	"658", "659", "694", "695", "856", "857", "698", "699", /* 358-35f */
	"660", "661", "662", "663", "664", "665", "666", "667", /* 360-367 */
	"668", "669", "686", "687", "866", "867", "888", "889", /* 368-36f */
	"670", "671", "672", "673", "674", "675", "676", "677", /* 370-377 */
	"678", "679", "696", "697", "876", "877", "898", "899", /* 378-37f */

	"700", "701", "702", "703", "704", "705", "706", "707", /* 380-387 */
	"708", "709", "780", "781", "906", "907", "986", "987", /* 388-38f */
	"710", "711", "712", "713", "714", "715", "716", "717", /* 390-397 */
	"718", "719", "790", "791", "916", "917", "996", "997", /* 398-39f */
	"720", "721", "722", "723", "724", "725", "726", "727", /* 3a0-3a7 */
	"728", "729", "782", "783", "926", "927", "968", "969", /* 3a8-3af */
	"730", "731", "732", "733", "734", "735", "736", "737", /* 3b0-3b7 */
	"738", "739", "792", "793", "936", "937", "978", "979", /* 3b8-3bf */

	"740", "741", "742", "743", "744", "745", "746", "747", /* 3c0-3c7 */
	"748", "749", "784", "785", "946", "947", "788", "789", /* 3c8-3cf */
	"750", "751", "752", "753", "754", "755", "756", "757", /* 3d0-3d7 */
	"758", "759", "794", "795", "956", "957", "798", "799", /* 3d8-3df */
	"760", "761", "762", "763", "764", "765", "766", "767", /* 3e0-3e7 */
	"768", "769", "786", "787", "966", "967", "988", "989", /* 3e8-3ef */
	"770", "771", "772", "773", "774", "775", "776", "777", /* 3f0-3f7 */
	"778", "779", "796", "797", "976", "977", "998", "999"  /* 3f8-3ff */
	};

//	"770", "771", "772", "773", "974", "775", "776", "777", /* 3f0-3f7 */

/* Table to convert 3 Densely Packed Decimal digits (10-bits) into
   3 Binary coded Decimal digits (12-bits).  */
const short int	dpd_to_bcd[1024] = {
	0x000, 0x001, 0x002, 0x003, 0x004, 0x005, 0x006, 0x007, /* 000-007 */
	0x008, 0x009, 0x080, 0x081, 0x800, 0x801, 0x880, 0x881, /* 008-00f */
	0x010, 0x011, 0x012, 0x013, 0x014, 0x015, 0x016, 0x017, /* 010-017 */
	0x018, 0x019, 0x090, 0x091, 0x810, 0x811, 0x890, 0x891, /* 018-01f */
	0x020, 0x021, 0x022, 0x023, 0x024, 0x025, 0x026, 0x027, /* 020-027 */
	0x028, 0x029, 0x082, 0x083, 0x820, 0x821, 0x808, 0x809, /* 028-02f */
	0x030, 0x031, 0x032, 0x033, 0x034, 0x035, 0x036, 0x037, /* 030-037 */
	0x038, 0x039, 0x092, 0x093, 0x830, 0x831, 0x818, 0x819, /* 038-03f */

	0x040, 0x041, 0x042, 0x043, 0x044, 0x045, 0x046, 0x047, /* 040-047 */
	0x048, 0x049, 0x084, 0x085, 0x840, 0x841, 0x088, 0x089, /* 048-04f */
	0x050, 0x051, 0x052, 0x053, 0x054, 0x055, 0x056, 0x057, /* 050-057 */
	0x058, 0x059, 0x094, 0x095, 0x850, 0x851, 0x098, 0x099, /* 058-05f */
	0x060, 0x061, 0x062, 0x063, 0x064, 0x065, 0x066, 0x067, /* 060-067 */
	0x068, 0x069, 0x086, 0x087, 0x860, 0x861, 0x888, 0x889, /* 068-06f */
	0x070, 0x071, 0x072, 0x073, 0x074, 0x075, 0x076, 0x077, /* 070-077 */
	0x078, 0x079, 0x096, 0x097, 0x870, 0x871, 0x898, 0x899, /* 078-07f */

	0x100, 0x101, 0x102, 0x103, 0x104, 0x105, 0x106, 0x107, /* 080-087 */
	0x108, 0x109, 0x180, 0x181, 0x900, 0x901, 0x980, 0x981, /* 088-08f */
	0x110, 0x111, 0x112, 0x113, 0x114, 0x115, 0x116, 0x117, /* 090-097 */
	0x118, 0x119, 0x190, 0x191, 0x910, 0x911, 0x990, 0x991, /* 098-09f */
	0x120, 0x121, 0x122, 0x123, 0x124, 0x125, 0x126, 0x127, /* 0a0-0a7 */
	0x128, 0x129, 0x182, 0x183, 0x920, 0x921, 0x908, 0x909, /* 0a8-0af */
	0x130, 0x131, 0x132, 0x133, 0x134, 0x135, 0x136, 0x137, /* 0b0-0b7 */
	0x138, 0x139, 0x192, 0x193, 0x930, 0x931, 0x918, 0x919, /* 0b8-0bf */

	0x140, 0x141, 0x142, 0x143, 0x144, 0x145, 0x146, 0x147, /* 0c0-0c7 */
	0x148, 0x149, 0x184, 0x185, 0x940, 0x941, 0x188, 0x189, /* 0c8-0cf */
	0x150, 0x151, 0x152, 0x153, 0x154, 0x155, 0x156, 0x157, /* 0d0-0d7 */
	0x158, 0x159, 0x194, 0x195, 0x950, 0x951, 0x198, 0x199, /* 0d8-0df */
	0x160, 0x161, 0x162, 0x163, 0x164, 0x165, 0x166, 0x167, /* 0e0-0e7 */
	0x168, 0x169, 0x186, 0x187, 0x960, 0x961, 0x988, 0x989, /* 0e8-0ef */
	0x170, 0x171, 0x172, 0x173, 0x174, 0x175, 0x176, 0x177, /* 0f0-0f7 */
	0x178, 0x179, 0x196, 0x197, 0x970, 0x971, 0x998, 0x999, /* 0f8-0ff */

	0x200, 0x201, 0x202, 0x203, 0x204, 0x205, 0x206, 0x207, /* 100-107 */
	0x208, 0x209, 0x280, 0x281, 0x802, 0x803, 0x882, 0x883, /* 108-10f */
	0x210, 0x211, 0x212, 0x213, 0x214, 0x215, 0x216, 0x217, /* 110-117 */
	0x218, 0x219, 0x290, 0x291, 0x812, 0x813, 0x892, 0x893, /* 118-11f */
	0x220, 0x221, 0x222, 0x223, 0x224, 0x225, 0x226, 0x227, /* 120-127 */
	0x228, 0x229, 0x282, 0x283, 0x822, 0x823, 0x828, 0x829, /* 128-12f */
	0x230, 0x231, 0x232, 0x233, 0x234, 0x235, 0x236, 0x237, /* 130-137 */
	0x238, 0x239, 0x292, 0x293, 0x832, 0x833, 0x838, 0x839, /* 138-13f */

	0x240, 0x241, 0x242, 0x243, 0x244, 0x245, 0x246, 0x247, /* 140-147 */
	0x248, 0x249, 0x284, 0x285, 0x842, 0x843, 0x288, 0x289, /* 148-14f */
	0x250, 0x251, 0x252, 0x253, 0x254, 0x255, 0x256, 0x257, /* 150-157 */
	0x258, 0x259, 0x294, 0x295, 0x852, 0x853, 0x298, 0x299, /* 158-15f */
	0x260, 0x261, 0x262, 0x263, 0x264, 0x265, 0x266, 0x267, /* 160-167 */
	0x268, 0x269, 0x286, 0x287, 0x862, 0x863, 0x888, 0x889, /* 168-16f */
	0x270, 0x271, 0x272, 0x273, 0x274, 0x275, 0x276, 0x277, /* 170-177 */
	0x278, 0x279, 0x296, 0x297, 0x872, 0x873, 0x898, 0x899, /* 178-17f */

	0x300, 0x301, 0x302, 0x303, 0x304, 0x305, 0x306, 0x307, /* 180-187 */
	0x308, 0x309, 0x380, 0x381, 0x902, 0x903, 0x982, 0x983, /* 188-18f */
	0x310, 0x311, 0x312, 0x313, 0x314, 0x315, 0x316, 0x317, /* 190-197 */
	0x318, 0x319, 0x390, 0x391, 0x912, 0x913, 0x992, 0x993, /* 198-19f */
	0x320, 0x321, 0x322, 0x323, 0x324, 0x325, 0x326, 0x327, /* 1a0-1a7 */
	0x328, 0x329, 0x382, 0x383, 0x922, 0x923, 0x928, 0x929, /* 1a8-1af */
	0x330, 0x331, 0x332, 0x333, 0x334, 0x335, 0x336, 0x337, /* 1b0-1b7 */
	0x338, 0x339, 0x392, 0x393, 0x932, 0x933, 0x938, 0x939, /* 1b8-1bf */

	0x340, 0x341, 0x342, 0x343, 0x344, 0x345, 0x346, 0x347, /* 1c0-1c7 */
	0x348, 0x349, 0x384, 0x385, 0x942, 0x943, 0x388, 0x389, /* 1c8-1cf */
	0x350, 0x351, 0x352, 0x353, 0x354, 0x355, 0x356, 0x357, /* 1d0-1d7 */
	0x358, 0x359, 0x394, 0x395, 0x952, 0x953, 0x398, 0x399, /* 1d8-1df */
	0x360, 0x361, 0x362, 0x363, 0x364, 0x365, 0x366, 0x367, /* 1e0-1e7 */
	0x368, 0x369, 0x386, 0x387, 0x962, 0x963, 0x988, 0x989, /* 1e8-1ef */
	0x370, 0x371, 0x372, 0x373, 0x374, 0x375, 0x376, 0x377, /* 1f0-1f7 */
	0x378, 0x379, 0x396, 0x397, 0x972, 0x973, 0x998, 0x999, /* 1f8-1ff */

	0x400, 0x401, 0x402, 0x403, 0x404, 0x405, 0x406, 0x407, /* 200-207 */
	0x408, 0x409, 0x480, 0x481, 0x804, 0x805, 0x884, 0x885, /* 208-20f */
	0x410, 0x411, 0x412, 0x413, 0x414, 0x415, 0x416, 0x417, /* 210-217 */
	0x418, 0x419, 0x490, 0x491, 0x814, 0x815, 0x894, 0x895, /* 218-21f */
	0x420, 0x421, 0x422, 0x423, 0x424, 0x425, 0x426, 0x427, /* 220-227 */
	0x428, 0x429, 0x482, 0x483, 0x824, 0x825, 0x848, 0x849, /* 228-22f */
	0x430, 0x431, 0x432, 0x433, 0x434, 0x435, 0x436, 0x437, /* 230-237 */
	0x438, 0x439, 0x492, 0x493, 0x834, 0x835, 0x858, 0x859, /* 238-23f */

	0x440, 0x441, 0x442, 0x443, 0x444, 0x445, 0x446, 0x447, /* 240-247 */
	0x448, 0x449, 0x484, 0x485, 0x844, 0x845, 0x488, 0x489, /* 248-24f */
	0x450, 0x451, 0x452, 0x453, 0x454, 0x455, 0x456, 0x457, /* 250-257 */
	0x458, 0x459, 0x494, 0x495, 0x854, 0x855, 0x498, 0x499, /* 258-25f */
	0x460, 0x461, 0x462, 0x463, 0x464, 0x465, 0x466, 0x467, /* 260-267 */
	0x468, 0x469, 0x486, 0x487, 0x864, 0x865, 0x888, 0x889, /* 268-26f */
	0x470, 0x471, 0x472, 0x473, 0x474, 0x475, 0x476, 0x477, /* 270-277 */
	0x478, 0x479, 0x496, 0x497, 0x874, 0x875, 0x898, 0x899, /* 278-27f */

	0x500, 0x501, 0x502, 0x503, 0x504, 0x505, 0x506, 0x507, /* 280-287 */
	0x508, 0x509, 0x580, 0x581, 0x904, 0x905, 0x984, 0x985, /* 288-28f */
	0x510, 0x511, 0x512, 0x513, 0x514, 0x515, 0x516, 0x517, /* 290-297 */
	0x518, 0x519, 0x590, 0x591, 0x914, 0x915, 0x994, 0x995, /* 298-29f */
	0x520, 0x521, 0x522, 0x523, 0x524, 0x525, 0x526, 0x527, /* 2a0-2a7 */
	0x528, 0x529, 0x582, 0x583, 0x924, 0x925, 0x948, 0x949, /* 2a8-2af */
	0x530, 0x531, 0x532, 0x533, 0x534, 0x535, 0x536, 0x537, /* 2b0-2b7 */
	0x538, 0x539, 0x592, 0x593, 0x934, 0x935, 0x958, 0x959, /* 2b8-2bf */

	0x540, 0x541, 0x542, 0x543, 0x544, 0x545, 0x546, 0x547, /* 2c0-2c7 */
	0x548, 0x549, 0x584, 0x585, 0x944, 0x945, 0x588, 0x589, /* 2c8-2cf */
	0x550, 0x551, 0x552, 0x553, 0x554, 0x555, 0x556, 0x557, /* 2d0-2d7 */
	0x558, 0x559, 0x594, 0x595, 0x954, 0x955, 0x598, 0x599, /* 2d8-2df */
	0x560, 0x561, 0x562, 0x563, 0x564, 0x565, 0x566, 0x567, /* 2e0-2e7 */
	0x568, 0x569, 0x586, 0x587, 0x964, 0x965, 0x988, 0x989, /* 2e8-2ef */
	0x570, 0x571, 0x572, 0x573, 0x574, 0x575, 0x576, 0x577, /* 2f0-2f7 */
	0x578, 0x579, 0x596, 0x597, 0x974, 0x975, 0x998, 0x999, /* 2f8-2ff */

	0x600, 0x601, 0x602, 0x603, 0x604, 0x605, 0x606, 0x607, /* 300-307 */
	0x608, 0x609, 0x680, 0x681, 0x806, 0x807, 0x886, 0x887, /* 308-30f */
	0x610, 0x611, 0x612, 0x613, 0x614, 0x615, 0x616, 0x617, /* 310-317 */
	0x618, 0x619, 0x690, 0x691, 0x816, 0x817, 0x896, 0x897, /* 318-31f */
	0x620, 0x621, 0x622, 0x623, 0x624, 0x625, 0x626, 0x627, /* 320-327 */
	0x628, 0x629, 0x682, 0x683, 0x826, 0x827, 0x868, 0x869, /* 328-32f */
	0x630, 0x631, 0x632, 0x633, 0x634, 0x635, 0x636, 0x637, /* 330-337 */
	0x638, 0x639, 0x692, 0x693, 0x836, 0x837, 0x878, 0x879, /* 338-33f */

	0x640, 0x641, 0x642, 0x643, 0x644, 0x645, 0x646, 0x647, /* 340-347 */
	0x648, 0x649, 0x684, 0x685, 0x846, 0x847, 0x688, 0x689, /* 348-34f */
	0x650, 0x651, 0x652, 0x653, 0x654, 0x655, 0x656, 0x657, /* 350-357 */
	0x658, 0x659, 0x694, 0x695, 0x856, 0x857, 0x698, 0x699, /* 358-35f */
	0x660, 0x661, 0x662, 0x663, 0x664, 0x665, 0x666, 0x667, /* 360-367 */
	0x668, 0x669, 0x686, 0x687, 0x866, 0x867, 0x888, 0x889, /* 368-36f */
	0x670, 0x671, 0x672, 0x673, 0x674, 0x675, 0x676, 0x677, /* 370-377 */
	0x678, 0x679, 0x696, 0x697, 0x876, 0x877, 0x898, 0x899, /* 378-37f */

	0x700, 0x701, 0x702, 0x703, 0x704, 0x705, 0x706, 0x707, /* 380-387 */
	0x708, 0x709, 0x780, 0x781, 0x906, 0x907, 0x986, 0x987, /* 388-38f */
	0x710, 0x711, 0x712, 0x713, 0x714, 0x715, 0x716, 0x717, /* 390-397 */
	0x718, 0x719, 0x790, 0x791, 0x916, 0x917, 0x996, 0x997, /* 398-39f */
	0x720, 0x721, 0x722, 0x723, 0x724, 0x725, 0x726, 0x727, /* 3a0-3a7 */
	0x728, 0x729, 0x782, 0x783, 0x926, 0x927, 0x968, 0x969, /* 3a8-3af */
	0x730, 0x731, 0x732, 0x733, 0x734, 0x735, 0x736, 0x737, /* 3b0-3b7 */
	0x738, 0x739, 0x792, 0x793, 0x936, 0x937, 0x978, 0x979, /* 3b8-3bf */

	0x740, 0x741, 0x742, 0x743, 0x744, 0x745, 0x746, 0x747, /* 3c0-3c7 */
	0x748, 0x749, 0x784, 0x785, 0x946, 0x947, 0x788, 0x789, /* 3c8-3cf */
	0x750, 0x751, 0x752, 0x753, 0x754, 0x755, 0x756, 0x757, /* 3d0-3d7 */
	0x758, 0x759, 0x794, 0x795, 0x956, 0x957, 0x798, 0x799, /* 3d8-3df */
	0x760, 0x761, 0x762, 0x763, 0x764, 0x765, 0x766, 0x767, /* 3e0-3e7 */
	0x768, 0x769, 0x786, 0x787, 0x966, 0x967, 0x988, 0x989, /* 3e8-3ef */
	0x770, 0x771, 0x772, 0x773, 0x974, 0x775, 0x776, 0x777, /* 3f0-3f7 */
	0x778, 0x779, 0x796, 0x797, 0x976, 0x977, 0x998, 0x999  /* 3f8-3ff */
	};

/* Table to convert 3 Binary Coded Decimal digits (12-bits) into
   3 Densely Packed Decimal digits (10-bits).  */
const short int	bcd_to_dpd[2464] = {
	0x000, 0x001, 0x002, 0x003, 0x004, 0x005, 0x006, 0x007,	/* 000-007 */
	0x008, 0x009, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 008-00f */
	0x010, 0x011, 0x012, 0x013, 0x014, 0x015, 0x016, 0x017,	/* 010-017 */
	0x018, 0x019, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 018-01f */
	0x020, 0x021, 0x022, 0x023, 0x024, 0x025, 0x026, 0x027,	/* 020-027 */
	0x028, 0x029, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 028-02f */
	0x030, 0x031, 0x032, 0x033, 0x034, 0x035, 0x036, 0x037,	/* 030-037 */
	0x038, 0x039, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 038-03f */
	0x040, 0x041, 0x042, 0x043, 0x044, 0x045, 0x046, 0x047,	/* 040-047 */
	0x048, 0x049, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 048-04f */
	0x050, 0x051, 0x052, 0x053, 0x054, 0x055, 0x056, 0x057,	/* 050-057 */
	0x058, 0x059, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 058-05f */
	0x060, 0x061, 0x062, 0x063, 0x064, 0x065, 0x066, 0x067,	/* 060-067 */
	0x068, 0x069, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 068-06f */
	0x070, 0x071, 0x072, 0x073, 0x074, 0x075, 0x076, 0x077,	/* 070-077 */
	0x078, 0x079, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 078-07f */
	0x00a, 0x00b, 0x02a, 0x02b, 0x04a, 0x04b, 0x06a, 0x06b,	/* 080-087 */
	0x04e, 0x04f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 088-08f */
	0x01a, 0x01b, 0x03a, 0x03b, 0x05a, 0x05b, 0x07a, 0x07b,	/* 090-097 */
	0x05e, 0x05f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 098-09f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0a0-0a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0a8-0af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0b0-0b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0b8-0bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0c0-0c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0c8-0cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0d0-0d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0d8-0df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0e0-0e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0e8-0ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0f0-0f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 0f8-0ff */
	0x080, 0x081, 0x082, 0x083, 0x084, 0x085, 0x086, 0x087,	/* 100-107 */
	0x088, 0x089, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 108-10f */
	0x090, 0x091, 0x092, 0x093, 0x094, 0x095, 0x096, 0x097,	/* 110-117 */
	0x098, 0x099, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 118-11f */
	0x0a0, 0x0a1, 0x0a2, 0x0a3, 0x0a4, 0x0a5, 0x0a6, 0x0a7,	/* 120-127 */
	0x0a8, 0x0a9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 128-12f */
	0x0b0, 0x0b1, 0x0b2, 0x0b3, 0x0b4, 0x0b5, 0x0b6, 0x0b7,	/* 130-137 */
	0x0b8, 0x0b9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 138-13f */
	0x0c0, 0x0c1, 0x0c2, 0x0c3, 0x0c4, 0x0c5, 0x0c6, 0x0c7,	/* 140-147 */
	0x0c8, 0x0c9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 148-14f */
	0x0d0, 0x0d1, 0x0d2, 0x0d3, 0x0d4, 0x0d5, 0x0d6, 0x0d7,	/* 150-157 */
	0x0d8, 0x0d9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 158-15f */
	0x0e0, 0x0e1, 0x0e2, 0x0e3, 0x0e4, 0x0e5, 0x0e6, 0x0e7,	/* 160-167 */
	0x0e8, 0x0e9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 168-16f */
	0x0f0, 0x0f1, 0x0f2, 0x0f3, 0x0f4, 0x0f5, 0x0f6, 0x0f7,	/* 170-177 */
	0x0f8, 0x0f9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 178-17f */
	0x08a, 0x08b, 0x0aa, 0x0ab, 0x0ca, 0x0cb, 0x0ea, 0x0eb,	/* 180-187 */
	0x0ce, 0x0cf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 188-18f */
	0x09a, 0x09b, 0x0ba, 0x0bb, 0x0da, 0x0db, 0x0fa, 0x0fb,	/* 190-197 */
	0x0de, 0x0df, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 198-19f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1a0-1a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1a8-1af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1b0-1b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1b8-1bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1c0-1c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1c8-1cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1d0-1d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1d8-1df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1e0-1e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1e8-1ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1f0-1f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 1f8-1ff */
	0x100, 0x101, 0x102, 0x103, 0x104, 0x105, 0x106, 0x107,	/* 200-207 */
	0x108, 0x109, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 208-20f */
	0x110, 0x111, 0x112, 0x113, 0x114, 0x115, 0x116, 0x117,	/* 210-217 */
	0x118, 0x119, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 218-21f */
	0x120, 0x121, 0x122, 0x123, 0x124, 0x125, 0x126, 0x127,	/* 220-227 */
	0x128, 0x129, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 228-22f */
	0x130, 0x131, 0x132, 0x133, 0x134, 0x135, 0x136, 0x137,	/* 230-237 */
	0x138, 0x139, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 238-23f */
	0x140, 0x141, 0x142, 0x143, 0x144, 0x145, 0x146, 0x147,	/* 240-247 */
	0x148, 0x149, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 248-24f */
	0x150, 0x151, 0x152, 0x153, 0x154, 0x155, 0x156, 0x157,	/* 250-257 */
	0x158, 0x159, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 258-25f */
	0x160, 0x161, 0x162, 0x163, 0x164, 0x165, 0x166, 0x167,	/* 260-267 */
	0x168, 0x169, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 268-26f */
	0x170, 0x171, 0x172, 0x173, 0x174, 0x175, 0x176, 0x177,	/* 270-277 */
	0x178, 0x179, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 278-27f */
	0x10a, 0x10b, 0x12a, 0x12b, 0x14a, 0x14b, 0x16a, 0x16b,	/* 280-287 */
	0x14e, 0x14f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 288-28f */
	0x11a, 0x11b, 0x13a, 0x13b, 0x15a, 0x15b, 0x17a, 0x17b,	/* 290-297 */
	0x15e, 0x15f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 298-29f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2a0-2a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2a8-2af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2b0-2b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2b8-2bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2c0-2c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2c8-2cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2d0-2d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2d8-2df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2e0-2e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2e8-2ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2f0-2f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 2f8-2ff */
	0x180, 0x181, 0x182, 0x183, 0x184, 0x185, 0x186, 0x187,	/* 300-307 */
	0x188, 0x189, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 308-30f */
	0x190, 0x191, 0x192, 0x193, 0x194, 0x195, 0x196, 0x197,	/* 310-317 */
	0x198, 0x199, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 318-31f */
	0x1a0, 0x1a1, 0x1a2, 0x1a3, 0x1a4, 0x1a5, 0x1a6, 0x1a7,	/* 320-327 */
	0x1a8, 0x1a9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 328-32f */
	0x1b0, 0x1b1, 0x1b2, 0x1b3, 0x1b4, 0x1b5, 0x1b6, 0x1b7,	/* 330-337 */
	0x1b8, 0x1b9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 338-33f */
	0x1c0, 0x1c1, 0x1c2, 0x1c3, 0x1c4, 0x1c5, 0x1c6, 0x1c7,	/* 340-347 */
	0x1c8, 0x1c9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 348-34f */
	0x1d0, 0x1d1, 0x1d2, 0x1d3, 0x1d4, 0x1d5, 0x1d6, 0x1d7,	/* 350-357 */
	0x1d8, 0x1d9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 358-35f */
	0x1e0, 0x1e1, 0x1e2, 0x1e3, 0x1e4, 0x1e5, 0x1e6, 0x1e7,	/* 360-367 */
	0x1e8, 0x1e9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 368-36f */
	0x1f0, 0x1f1, 0x1f2, 0x1f3, 0x1f4, 0x1f5, 0x1f6, 0x1f7,	/* 370-377 */
	0x1f8, 0x1f9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 378-37f */
	0x18a, 0x18b, 0x1aa, 0x1ab, 0x1ca, 0x1cb, 0x1ea, 0x1eb,	/* 380-387 */
	0x1ce, 0x1cf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 388-38f */
	0x19a, 0x19b, 0x1ba, 0x1bb, 0x1da, 0x1db, 0x1fa, 0x1fb,	/* 390-397 */
	0x1de, 0x1df, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 398-39f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3a0-3a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3a8-3af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3b0-3b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3b8-3bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3c0-3c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3c8-3cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3d0-3d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3d8-3df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3e0-3e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3e8-3ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3f0-3f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 3f8-3ff */
	0x200, 0x201, 0x202, 0x203, 0x204, 0x205, 0x206, 0x207,	/* 400-407 */
	0x208, 0x209, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 408-40f */
	0x210, 0x211, 0x212, 0x213, 0x214, 0x215, 0x216, 0x217,	/* 410-417 */
	0x218, 0x219, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 418-41f */
	0x220, 0x221, 0x222, 0x223, 0x224, 0x225, 0x226, 0x227,	/* 420-427 */
	0x228, 0x229, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 428-42f */
	0x230, 0x231, 0x232, 0x233, 0x234, 0x235, 0x236, 0x237,	/* 430-437 */
	0x238, 0x239, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 438-43f */
	0x240, 0x241, 0x242, 0x243, 0x244, 0x245, 0x246, 0x247,	/* 440-447 */
	0x248, 0x249, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 448-44f */
	0x250, 0x251, 0x252, 0x253, 0x254, 0x255, 0x256, 0x257,	/* 450-457 */
	0x258, 0x259, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 458-45f */
	0x260, 0x261, 0x262, 0x263, 0x264, 0x265, 0x266, 0x267,	/* 460-467 */
	0x268, 0x269, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 468-46f */
	0x270, 0x271, 0x272, 0x273, 0x274, 0x275, 0x276, 0x277,	/* 470-477 */
	0x278, 0x279, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 478-47f */
	0x20a, 0x20b, 0x22a, 0x22b, 0x24a, 0x24b, 0x26a, 0x26b,	/* 480-487 */
	0x24e, 0x24f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 488-48f */
	0x21a, 0x21b, 0x23a, 0x23b, 0x25a, 0x25b, 0x27a, 0x27b,	/* 490-497 */
	0x25e, 0x25f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 498-49f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4a0-4a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4a8-4af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4b0-4b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4b8-4bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4c0-4c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4c8-4cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4d0-4d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4d8-4df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4e0-4e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4e8-4ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4f0-4f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 4f8-4ff */
	0x280, 0x281, 0x282, 0x283, 0x284, 0x285, 0x286, 0x287,	/* 500-507 */
	0x288, 0x289, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 508-50f */
	0x290, 0x291, 0x292, 0x293, 0x294, 0x295, 0x296, 0x297,	/* 510-517 */
	0x298, 0x299, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 518-51f */
	0x2a0, 0x2a1, 0x2a2, 0x2a3, 0x2a4, 0x2a5, 0x2a6, 0x2a7,	/* 520-527 */
	0x2a8, 0x2a9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 528-52f */
	0x2b0, 0x2b1, 0x2b2, 0x2b3, 0x2b4, 0x2b5, 0x2b6, 0x2b7,	/* 530-537 */
	0x2b8, 0x2b9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 538-53f */
	0x2c0, 0x2c1, 0x2c2, 0x2c3, 0x2c4, 0x2c5, 0x2c6, 0x2c7,	/* 540-547 */
	0x2c8, 0x2c9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 548-54f */
	0x2d0, 0x2d1, 0x2d2, 0x2d3, 0x2d4, 0x2d5, 0x2d6, 0x2d7,	/* 550-557 */
	0x2d8, 0x2d9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 558-55f */
	0x2e0, 0x2e1, 0x2e2, 0x2e3, 0x2e4, 0x2e5, 0x2e6, 0x2e7,	/* 560-567 */
	0x2e8, 0x2e9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 568-56f */
	0x2f0, 0x2f1, 0x2f2, 0x2f3, 0x2f4, 0x2f5, 0x2f6, 0x2f7,	/* 570-577 */
	0x2f8, 0x2f9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 578-57f */
	0x28a, 0x28b, 0x2aa, 0x2ab, 0x2ca, 0x2cb, 0x2ea, 0x2eb,	/* 580-587 */
	0x2ce, 0x2cf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 588-58f */
	0x29a, 0x29b, 0x2ba, 0x2bb, 0x2da, 0x2db, 0x2fa, 0x2fb,	/* 590-597 */
	0x2de, 0x2df, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 598-59f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5a0-5a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5a8-5af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5b0-5b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5b8-5bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5c0-5c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5c8-5cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5d0-5d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5d8-5df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5e0-5e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5e8-5ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5f0-5f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 5f8-5ff */
	0x300, 0x301, 0x302, 0x303, 0x304, 0x305, 0x306, 0x307,	/* 600-607 */
	0x308, 0x309, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 608-60f */
	0x310, 0x311, 0x312, 0x313, 0x314, 0x315, 0x316, 0x317,	/* 610-617 */
	0x318, 0x319, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 618-61f */
	0x320, 0x321, 0x322, 0x323, 0x324, 0x325, 0x326, 0x327,	/* 620-627 */
	0x328, 0x329, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 628-62f */
	0x330, 0x331, 0x332, 0x333, 0x334, 0x335, 0x336, 0x337,	/* 630-637 */
	0x338, 0x339, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 638-63f */
	0x340, 0x341, 0x342, 0x343, 0x344, 0x345, 0x346, 0x347,	/* 640-647 */
	0x348, 0x349, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 648-64f */
	0x350, 0x351, 0x352, 0x353, 0x354, 0x355, 0x356, 0x357,	/* 650-657 */
	0x358, 0x359, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 658-65f */
	0x360, 0x361, 0x362, 0x363, 0x364, 0x365, 0x366, 0x367,	/* 660-667 */
	0x368, 0x369, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 668-66f */
	0x370, 0x371, 0x372, 0x373, 0x374, 0x375, 0x376, 0x377,	/* 670-677 */
	0x378, 0x379, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 678-67f */
	0x30a, 0x30b, 0x32a, 0x32b, 0x34a, 0x34b, 0x36a, 0x36b,	/* 680-687 */
	0x34e, 0x34f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 688-68f */
	0x31a, 0x31b, 0x33a, 0x33b, 0x35a, 0x35b, 0x37a, 0x37b,	/* 690-697 */
	0x35e, 0x35f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 698-69f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6a0-6a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6a8-6af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6b0-6b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6b8-6bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6c0-6c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6c8-6cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6d0-6d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6d8-6df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6e0-6e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6e8-6ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6f0-6f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 6f8-6ff */
	0x380, 0x381, 0x382, 0x383, 0x384, 0x385, 0x386, 0x387,	/* 700-707 */
	0x388, 0x389, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 708-70f */
	0x390, 0x391, 0x392, 0x393, 0x394, 0x395, 0x396, 0x397,	/* 710-717 */
	0x398, 0x399, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 718-71f */
	0x3a0, 0x3a1, 0x3a2, 0x3a3, 0x3a4, 0x3a5, 0x3a6, 0x3a7,	/* 720-727 */
	0x3a8, 0x3a9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 728-72f */
	0x3b0, 0x3b1, 0x3b2, 0x3b3, 0x3b4, 0x3b5, 0x3b6, 0x3b7,	/* 730-737 */
	0x3b8, 0x3b9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 738-73f */
	0x3c0, 0x3c1, 0x3c2, 0x3c3, 0x3c4, 0x3c5, 0x3c6, 0x3c7,	/* 740-747 */
	0x3c8, 0x3c9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 748-74f */
	0x3d0, 0x3d1, 0x3d2, 0x3d3, 0x3d4, 0x3d5, 0x3d6, 0x3d7,	/* 750-757 */
	0x3d8, 0x3d9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 758-75f */
	0x3e0, 0x3e1, 0x3e2, 0x3e3, 0x3e4, 0x3e5, 0x3e6, 0x3e7,	/* 760-767 */
	0x3e8, 0x3e9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 768-76f */
	0x3f0, 0x3f1, 0x3f2, 0x3f3, 0x000, 0x3f5, 0x3f6, 0x3f7,	/* 770-777 */
	0x3f8, 0x3f9, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 778-77f */
	0x38a, 0x38b, 0x3aa, 0x3ab, 0x3ca, 0x3cb, 0x3ea, 0x3eb,	/* 780-787 */
	0x3ce, 0x3cf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 788-78f */
	0x39a, 0x39b, 0x3ba, 0x3bb, 0x3da, 0x3db, 0x3fa, 0x3fb,	/* 790-797 */
	0x3de, 0x3df, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 798-79f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7a0-7a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7a8-7af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7b0-7b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7b8-7bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7c0-7c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7c8-7cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7d0-7d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7d8-7df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7e0-7e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7e8-7ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7f0-7f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 7f8-7ff */
	0x00c, 0x00d, 0x10c, 0x10d, 0x20c, 0x20d, 0x30c, 0x30d,	/* 800-807 */
	0x02e, 0x02f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 808-80f */
	0x01c, 0x01d, 0x11c, 0x11d, 0x21c, 0x21d, 0x31c, 0x31d,	/* 810-817 */
	0x03e, 0x03f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 818-81f */
	0x02c, 0x02d, 0x12c, 0x12d, 0x22c, 0x22d, 0x32c, 0x32d,	/* 820-827 */
	0x12e, 0x12f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 828-82f */
	0x03c, 0x03d, 0x13c, 0x13d, 0x23c, 0x23d, 0x33c, 0x33d,	/* 830-837 */
	0x13e, 0x13f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 838-83f */
	0x04c, 0x04d, 0x14c, 0x14d, 0x24c, 0x24d, 0x34c, 0x34d,	/* 840-847 */
	0x22e, 0x22f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 848-84f */
	0x05c, 0x05d, 0x15c, 0x15d, 0x25c, 0x25d, 0x35c, 0x35d,	/* 850-857 */
	0x23e, 0x23f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 858-85f */
	0x06c, 0x06d, 0x16c, 0x16d, 0x26c, 0x26d, 0x36c, 0x36d,	/* 860-867 */
	0x32e, 0x32f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 868-86f */
	0x07c, 0x07d, 0x17c, 0x17d, 0x27c, 0x27d, 0x37c, 0x37d,	/* 870-877 */
	0x33e, 0x33f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 878-87f */
	0x00e, 0x00f, 0x10e, 0x10f, 0x20e, 0x20f, 0x30e, 0x30f,	/* 880-887 */
	0x06e, 0x06f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 888-88f */
	0x01e, 0x01f, 0x11e, 0x11f, 0x21e, 0x21f, 0x31e, 0x31f,	/* 890-897 */
	0x07e, 0x07f, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 898-89f */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8a0-8a7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8a8-8af */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8b0-8b7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8b8-8bf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8c0-8c7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8c8-8cf */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8d0-8d7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8d8-8df */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8e0-8e7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8e8-8ef */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8f0-8f7 */
	0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 8f8-8ff */
	0x08c, 0x08d, 0x18c, 0x18d, 0x28c, 0x28d, 0x38c, 0x38d,	/* 900-907 */
	0x0ae, 0x0af, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 908-90f */
	0x09c, 0x09d, 0x19c, 0x19d, 0x29c, 0x29d, 0x39c, 0x39d,	/* 910-917 */
	0x0be, 0x0bf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 918-91f */
	0x0ac, 0x0ad, 0x1ac, 0x1ad, 0x2ac, 0x2ad, 0x3ac, 0x3ad,	/* 920-927 */
	0x1ae, 0x1af, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 928-92f */
	0x0bc, 0x0bd, 0x1bc, 0x1bd, 0x2bc, 0x2bd, 0x3bc, 0x3bd,	/* 930-937 */
	0x1be, 0x1bf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 938-93f */
	0x0cc, 0x0cd, 0x1cc, 0x1cd, 0x2cc, 0x2cd, 0x3cc, 0x3cd,	/* 940-947 */
	0x2ae, 0x2af, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 948-94f */
	0x0dc, 0x0dd, 0x1dc, 0x1dd, 0x2dc, 0x2dd, 0x3dc, 0x3dd,	/* 950-957 */
	0x2be, 0x2bf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 958-95f */
	0x0ec, 0x0ed, 0x1ec, 0x1ed, 0x2ec, 0x2ed, 0x3ec, 0x3ed,	/* 960-967 */
	0x3ae, 0x3af, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 968-96f */
	0x0fc, 0x0fd, 0x1fc, 0x1fd, 0x3f4, 0x2fd, 0x3fc, 0x3fd,	/* 970-977 */
	0x3be, 0x3bf, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 978-97f */
	0x08e, 0x08f, 0x18e, 0x18f, 0x28e, 0x28f, 0x38e, 0x38f,	/* 980-987 */
	0x0ee, 0x0ef, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 988-98f */
	0x09e, 0x09f, 0x19e, 0x19f, 0x29e, 0x29f, 0x39e, 0x39f,	/* 990-997 */
	0x0fe, 0x0ff, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,	/* 998-99f */

};

/* Convert a declet into a binary value. This is basically dpd_to_bcd sans
   0x.  */
const short int dpd_to_bin[1024] = {
	  0,   1,   2,   3,   4,   5,   6,   7,
	  8,   9,  80,  81, 800, 801, 880, 881,
	 10,  11,  12,  13,  14,  15,  16,  17,
	 18,  19,  90,  91, 810, 811, 890, 891,
	 20,  21,  22,  23,  24,  25,  26,  27,
	 28,  29,  82,  83, 820, 821, 808, 809,
	 30,  31,  32,  33,  34,  35,  36,  37,
	 38,  39,  92,  93, 830, 831, 818, 819,
	 40,  41,  42,  43,  44,  45,  46,  47,
	 48,  49,  84,  85, 840, 841,  88,  89,
	 50,  51,  52,  53,  54,  55,  56,  57,
	 58,  59,  94,  95, 850, 851,  98,  99,
	 60,  61,  62,  63,  64,  65,  66,  67,
	 68,  69,  86,  87, 860, 861, 888, 889,
	 70,  71,  72,  73,  74,  75,  76,  77,
	 78,  79,  96,  97, 870, 871, 898, 899,
	100, 101, 102, 103, 104, 105, 106, 107,
	108, 109, 180, 181, 900, 901, 980, 981,
	110, 111, 112, 113, 114, 115, 116, 117,
	118, 119, 190, 191, 910, 911, 990, 991,
	120, 121, 122, 123, 124, 125, 126, 127,
	128, 129, 182, 183, 920, 921, 908, 909,
	130, 131, 132, 133, 134, 135, 136, 137,
	138, 139, 192, 193, 930, 931, 918, 919,
	140, 141, 142, 143, 144, 145, 146, 147,
	148, 149, 184, 185, 940, 941, 188, 189,
	150, 151, 152, 153, 154, 155, 156, 157,
	158, 159, 194, 195, 950, 951, 198, 199,
	160, 161, 162, 163, 164, 165, 166, 167,
	168, 169, 186, 187, 960, 961, 988, 989,
	170, 171, 172, 173, 174, 175, 176, 177,
	178, 179, 196, 197, 970, 971, 998, 999,
	200, 201, 202, 203, 204, 205, 206, 207,
	208, 209, 280, 281, 802, 803, 882, 883,
	210, 211, 212, 213, 214, 215, 216, 217,
	218, 219, 290, 291, 812, 813, 892, 893,
	220, 221, 222, 223, 224, 225, 226, 227,
	228, 229, 282, 283, 822, 823, 828, 829,
	230, 231, 232, 233, 234, 235, 236, 237,
	238, 239, 292, 293, 832, 833, 838, 839,
	240, 241, 242, 243, 244, 245, 246, 247,
	248, 249, 284, 285, 842, 843, 288, 289,
	250, 251, 252, 253, 254, 255, 256, 257,
	258, 259, 294, 295, 852, 853, 298, 299,
	260, 261, 262, 263, 264, 265, 266, 267,
	268, 269, 286, 287, 862, 863, 888, 889,
	270, 271, 272, 273, 274, 275, 276, 277,
	278, 279, 296, 297, 872, 873, 898, 899,
	300, 301, 302, 303, 304, 305, 306, 307,
	308, 309, 380, 381, 902, 903, 982, 983,
	310, 311, 312, 313, 314, 315, 316, 317,
	318, 319, 390, 391, 912, 913, 992, 993,
	320, 321, 322, 323, 324, 325, 326, 327,
	328, 329, 382, 383, 922, 923, 928, 929,
	330, 331, 332, 333, 334, 335, 336, 337,
	338, 339, 392, 393, 932, 933, 938, 939,
	340, 341, 342, 343, 344, 345, 346, 347,
	348, 349, 384, 385, 942, 943, 388, 389,
	350, 351, 352, 353, 354, 355, 356, 357,
	358, 359, 394, 395, 952, 953, 398, 399,
	360, 361, 362, 363, 364, 365, 366, 367,
	368, 369, 386, 387, 962, 963, 988, 989,
	370, 371, 372, 373, 374, 375, 376, 377,
	378, 379, 396, 397, 972, 973, 998, 999,
	400, 401, 402, 403, 404, 405, 406, 407,
	408, 409, 480, 481, 804, 805, 884, 885,
	410, 411, 412, 413, 414, 415, 416, 417,
	418, 419, 490, 491, 814, 815, 894, 895,
	420, 421, 422, 423, 424, 425, 426, 427,
	428, 429, 482, 483, 824, 825, 848, 849,
	430, 431, 432, 433, 434, 435, 436, 437,
	438, 439, 492, 493, 834, 835, 858, 859,
	440, 441, 442, 443, 444, 445, 446, 447,
	448, 449, 484, 485, 844, 845, 488, 489,
	450, 451, 452, 453, 454, 455, 456, 457,
	458, 459, 494, 495, 854, 855, 498, 499,
	460, 461, 462, 463, 464, 465, 466, 467,
	468, 469, 486, 487, 864, 865, 888, 889,
	470, 471, 472, 473, 474, 475, 476, 477,
	478, 479, 496, 497, 874, 875, 898, 899,
	500, 501, 502, 503, 504, 505, 506, 507,
	508, 509, 580, 581, 904, 905, 984, 985,
	510, 511, 512, 513, 514, 515, 516, 517,
	518, 519, 590, 591, 914, 915, 994, 995,
	520, 521, 522, 523, 524, 525, 526, 527,
	528, 529, 582, 583, 924, 925, 948, 949,
	530, 531, 532, 533, 534, 535, 536, 537,
	538, 539, 592, 593, 934, 935, 958, 959,
	540, 541, 542, 543, 544, 545, 546, 547,
	548, 549, 584, 585, 944, 945, 588, 589,
	550, 551, 552, 553, 554, 555, 556, 557,
	558, 559, 594, 595, 954, 955, 598, 599,
	560, 561, 562, 563, 564, 565, 566, 567,
	568, 569, 586, 587, 964, 965, 988, 989,
	570, 571, 572, 573, 574, 575, 576, 577,
	578, 579, 596, 597, 974, 975, 998, 999,
	600, 601, 602, 603, 604, 605, 606, 607,
	608, 609, 680, 681, 806, 807, 886, 887,
	610, 611, 612, 613, 614, 615, 616, 617,
	618, 619, 690, 691, 816, 817, 896, 897,
	620, 621, 622, 623, 624, 625, 626, 627,
	628, 629, 682, 683, 826, 827, 868, 869,
	630, 631, 632, 633, 634, 635, 636, 637,
	638, 639, 692, 693, 836, 837, 878, 879,
	640, 641, 642, 643, 644, 645, 646, 647,
	648, 649, 684, 685, 846, 847, 688, 689,
	650, 651, 652, 653, 654, 655, 656, 657,
	658, 659, 694, 695, 856, 857, 698, 699,
	660, 661, 662, 663, 664, 665, 666, 667,
	668, 669, 686, 687, 866, 867, 888, 889,
	670, 671, 672, 673, 674, 675, 676, 677,
	678, 679, 696, 697, 876, 877, 898, 899,
	700, 701, 702, 703, 704, 705, 706, 707,
	708, 709, 780, 781, 906, 907, 986, 987,
	710, 711, 712, 713, 714, 715, 716, 717,
	718, 719, 790, 791, 916, 917, 996, 997,
	720, 721, 722, 723, 724, 725, 726, 727,
	728, 729, 782, 783, 926, 927, 968, 969,
	730, 731, 732, 733, 734, 735, 736, 737,
	738, 739, 792, 793, 936, 937, 978, 979,
	740, 741, 742, 743, 744, 745, 746, 747,
	748, 749, 784, 785, 946, 947, 788, 789,
	750, 751, 752, 753, 754, 755, 756, 757,
	758, 759, 794, 795, 956, 957, 798, 799,
	760, 761, 762, 763, 764, 765, 766, 767,
	768, 769, 786, 787, 966, 967, 988, 989,
	770, 771, 772, 773, 974, 775, 776, 777,
	778, 779, 796, 797, 976, 977, 998, 999
	};

extern void __get_dpd_digits (int bits, const void *const *args, char *str,
			      int *exp_p, int *sign_p, int *nan_p, int *inf_p);

void
__get_dpd_digits (int bits, const void *const *args, 
char *str, int *exp_p, int *sign_p, int *nan_p, int *inf_p)
{

  int exp = 0;
  struct ieee754r_c_field c_f;
if (bits == 32) 
{
  union ieee754r_Decimal32 d;
  d.sd = *(_Decimal32*)args[0];
  c_f =  c_decoder[d.ieee.c];
  exp = c_f.lm_exp << DECIMAL32_BEC_bits;
  exp += d.ieee.bec;
  exp -= DECIMAL32_Bias;

  __dpd_cpydeclet (str+1, dpd_to_char[d.ieee.cc0]);
  __dpd_cpydeclet (str+4, dpd_to_char[d.ieee.cc1]);
  str[7] = '\0';
  if (sign_p) *sign_p = d.ieee.negative;
}
else if (bits == 64) 
{
  union ieee754r_Decimal64 d;
  d.dd = *(_Decimal64*)args[0];
  c_f =  c_decoder[d.ieee.c];
  exp = c_f.lm_exp << DECIMAL64_BEC_bits;
  exp += d.ieee.bec;
  exp -= DECIMAL64_Bias;

  __dpd_cpydeclet (str+1, dpd_to_char[d.ieee.cc0]);
  /* Packed fields crossing a word boundary require special handling. */
  __dpd_cpydeclet (str+4, dpd_to_char[(d.ieee.cc1H8<<2) + d.ieee.cc1L2]);
  __dpd_cpydeclet (str+7, dpd_to_char[d.ieee.cc2]);
  __dpd_cpydeclet (str+10, dpd_to_char[d.ieee.cc3]);
  __dpd_cpydeclet (str+13, dpd_to_char[d.ieee.cc4]);
  str[16] = '\0';
  if (sign_p) *sign_p = d.ieee.negative;
}
//else if (bits == 128)
else
{
  union ieee754r_Decimal128 d;
  d.td = *(_Decimal128*)args[0];
  c_f =  c_decoder[d.ieee.c];
  exp = c_f.lm_exp << DECIMAL128_BEC_bits;
  exp += d.ieee.bec;
  exp -= DECIMAL128_Bias;

  __dpd_cpydeclet (str+1, dpd_to_char[d.ieee.cc0]);
  /* Packed fields crossing a word boundary require special handling. */
  __dpd_cpydeclet (str+4, dpd_to_char[(d.ieee.cc1H4<<6) + d.ieee.cc1L6]);
  __dpd_cpydeclet (str+7, dpd_to_char[d.ieee.cc2]);
  __dpd_cpydeclet (str+10, dpd_to_char[d.ieee.cc3]);
  __dpd_cpydeclet (str+13, dpd_to_char[(d.ieee.cc4H6<<4) + d.ieee.cc4L4]);
  __dpd_cpydeclet (str+16, dpd_to_char[d.ieee.cc5]);
  __dpd_cpydeclet (str+19, dpd_to_char[d.ieee.cc6]);
  __dpd_cpydeclet (str+22, dpd_to_char[(d.ieee.cc7H8<<2) + d.ieee.cc7L2]);
  __dpd_cpydeclet (str+25, dpd_to_char[d.ieee.cc8]);
  __dpd_cpydeclet (str+28, dpd_to_char[d.ieee.cc9]);
  __dpd_cpydeclet (str+31, dpd_to_char[d.ieee.cc10]);
  str[34] = '\0';
  if (sign_p) *sign_p = d.ieee.negative;
}

  str[0] = '0' + c_f.lmd;
  if (exp_p) *exp_p = exp;
  if (nan_p) *nan_p = c_f.is_nan;
  if (inf_p) *inf_p = c_f.is_inf;

}

void
__get_digits_d32 (_Decimal32 x, char *str, int *exp_p, int *sign_p, 
		  int *nan_p, int *inf_p)
{
  int exp = 0;
  struct ieee754r_c_field c_f;
  union ieee754r_Decimal32 d;
  d.sd = x;
  c_f =  c_decoder[d.ieee.c];
  /* Consider replacing with a call to getexp_d* for hardware support  */
  exp = c_f.lm_exp << DECIMAL32_BEC_bits;
  exp += d.ieee.bec;
  exp -= DECIMAL32_Bias;

  str[0] = '0' + c_f.lmd;
  __dpd_cpydeclet (str+1, dpd_to_char[d.ieee.cc0]);
  __dpd_cpydeclet (str+4, dpd_to_char[d.ieee.cc1]);
  str[7] = '\0';
  if (sign_p) *sign_p = d.ieee.negative;
  if (exp_p) *exp_p = exp;
  if (nan_p) *nan_p = c_f.is_nan;
  if (inf_p) *inf_p = c_f.is_inf;
}

void 
__get_digits_d64 (_Decimal64 x, char *str, int *exp_p, int *sign_p, 
		  int *nan_p, int *inf_p)
{
  int exp = 0;
  struct ieee754r_c_field c_f;
  union ieee754r_Decimal64 d;
  d.dd = x;
  c_f =  c_decoder[d.ieee.c];
  /* Consider replacing with a call to getexp_d* for hardware support  */
  exp = c_f.lm_exp << DECIMAL64_BEC_bits;
  exp += d.ieee.bec;
  exp -= DECIMAL64_Bias;

  str[0] = '0' + c_f.lmd;
  __dpd_cpydeclet (str+1, dpd_to_char[d.ieee.cc0]);
  /* Packed fields crossing a word boundary require special handling. */
  __dpd_cpydeclet (str+4, dpd_to_char[(d.ieee.cc1H8<<2) + d.ieee.cc1L2]);
  __dpd_cpydeclet (str+7, dpd_to_char[d.ieee.cc2]);
  __dpd_cpydeclet (str+10, dpd_to_char[d.ieee.cc3]);
  __dpd_cpydeclet (str+13, dpd_to_char[d.ieee.cc4]);
  str[16] = '\0';
  if (sign_p) *sign_p = d.ieee.negative;
  if (exp_p) *exp_p = exp;
  if (nan_p) *nan_p = c_f.is_nan;
  if (inf_p) *inf_p = c_f.is_inf;

}

void 
__get_digits_d128 (_Decimal128 x, char *str, int *exp_p, int *sign_p, 
		   int *nan_p, int *inf_p)
{
  int exp = 0;
  struct ieee754r_c_field c_f;
  union ieee754r_Decimal128 d;
  d.td = x;
  c_f =  c_decoder[d.ieee.c];
  /* Consider replacing with a call to getexp_d* for hardware support  */
  exp = c_f.lm_exp << DECIMAL128_BEC_bits;
  exp += d.ieee.bec;
  exp -= DECIMAL128_Bias;

  str[0] = '0' + c_f.lmd;
  __dpd_cpydeclet (str+1, dpd_to_char[d.ieee.cc0]);
  /* Packed fields crossing a word boundary require special handling. */
  __dpd_cpydeclet (str+4, dpd_to_char[(d.ieee.cc1H4<<6) + d.ieee.cc1L6]);
  __dpd_cpydeclet (str+7, dpd_to_char[d.ieee.cc2]);
  __dpd_cpydeclet (str+10, dpd_to_char[d.ieee.cc3]);
  __dpd_cpydeclet (str+13, dpd_to_char[(d.ieee.cc4H6<<4) + d.ieee.cc4L4]);
  __dpd_cpydeclet (str+16, dpd_to_char[d.ieee.cc5]);
  __dpd_cpydeclet (str+19, dpd_to_char[d.ieee.cc6]);
  __dpd_cpydeclet (str+22, dpd_to_char[(d.ieee.cc7H8<<2) + d.ieee.cc7L2]);
  __dpd_cpydeclet (str+25, dpd_to_char[d.ieee.cc8]);
  __dpd_cpydeclet (str+28, dpd_to_char[d.ieee.cc9]);
  __dpd_cpydeclet (str+31, dpd_to_char[d.ieee.cc10]);
  str[34] = '\0';
  if (sign_p) *sign_p = d.ieee.negative;
  if (exp_p) *exp_p = exp;
  if (nan_p) *nan_p = c_f.is_nan;
  if (inf_p) *inf_p = c_f.is_inf;
  
}

