#if !defined (__SCALARIDENTIFIERS_HPP)
#define __SCALARIDENTIFIERS_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if !defined IN_COMMON_HPP
   #error ScalarIdentifiers.hpp is included by Common.hpp only.
#endif
  
namespace corelinux
{
   /**
   ScalarIdentifier provides a templated interface
   for declaring CoreLinux Identifiers for simple scalar
   types
   */

   template< class ScalarType >
      class ScalarIdentifier : public Identifier
      {

      public:

         //
         // Constructors and destructor
         //

                        /// Default constructor

                        ScalarIdentifier( void )
                           :
                           Identifier(),
                           theScalar(0)
                        {
                           ;  // do nothing
                        }

                        /// Initializing constructor

                        ScalarIdentifier( ScalarType aType )
                           :
                           Identifier(),
                           theScalar(aType)
                        {
                           ;  // do nothing
                        }

                        /// Copy constructor

                        ScalarIdentifier( const ScalarIdentifier &aScalar )
                           :
                           Identifier(aScalar),
                           theScalar( aScalar.getScalar() )
                        {
                           ;  // do nothing
                        }

                        /// Virtual Destructor

         virtual        ~ScalarIdentifier( void )
                        {
                           ;  // do nothing
                        }

         //
         // Operator overloads
         //
                  /// Operator assignment for scalars

         inline   ScalarType & operator=( ScalarType aScalar )
                  {
                     theScalar = aScalar;
                     return ( *this );
                  }

                  /// Operator assignment of Identifiers

         inline   ScalarIdentifier & operator=( const ScalarIdentifier &aScalar )
                  {
                     theScalar = aScalar.getScalar();
                     return ( *this );
                  }

                  /// Reference cast operator

         inline   operator ScalarType &( void )
                  {
                     return theScalar;
                  }

                  /// Pointer cast operator

         inline   operator ScalarType *( void )
                  {
                     return &theScalar;         
                  }

         //
         // Accessors
         //
                  /// Returns a const reference

         inline   const ScalarType &  getScalar( void ) const
                  {
                     return theScalar;
                  }

      protected:

                  /**
                  Equality method
                  @param Identifier const reference
                  @return true if equal, false otherwise
                  */
            
         virtual  bool        isEqual( IdentifierCref aRef ) const
         {
            return ( theScalar == 
                     ((const ScalarIdentifier &)aRef).theScalar );
         }
            
                  /**
                  Less than method
                  @param Identifier const reference
                  @return true if less than, false otherwise
                  */
            
         virtual  bool        isLessThan( IdentifierCref aRef ) const 
         {
            return ( theScalar < 
                     ((const ScalarIdentifier &)aRef).theScalar );
         }
            
                  /**
                  Less than or equal method.
                  @param Identifier const reference
                  @return true if less than or equal, false otherwise
                  */
            
         virtual  bool        isLessThanOrEqual( IdentifierCref aRef ) const 
         {
            return ( theScalar <= 
                     ((const ScalarIdentifier &)aRef).theScalar );
         }
            
                  /**
                  Greater than method.
                  @param Identifier const reference
                  @return true if greater than, false otherwise
                  */
            
         virtual  bool        isGreaterThan( IdentifierCref aRef ) const 
         {
            return ( theScalar > 
                     ((const ScalarIdentifier &)aRef).theScalar );
         }
            
                  /**
                  Greater than or equal method.
                  @param Identifier const reference
                  @return true if greater than or equal, false otherwise
                  */
   
         virtual  bool        isGreaterThanOrEqual( IdentifierCref aRef ) const 
         {
            return ( theScalar >= 
                     ((const ScalarIdentifier &)aRef).theScalar );
         }

      private:

                  /// The scalar storage

                  ScalarType     theScalar;
      };

}

using corelinux::Dword;
using corelinux::Int;
using corelinux::Short;
using corelinux::Word;
using corelinux::UnsignedInt;

/// Integer Identifier (int)

DECLARE_TYPE( CORELINUX(ScalarIdentifier<Int>), IntIdentifier );

/// Unsigned integer identifier (UnsignedInt)

DECLARE_TYPE( CORELINUX(ScalarIdentifier<UnsignedInt>), UnsignedIdentifier );

/// Short Identifer (short int)

DECLARE_TYPE( CORELINUX(ScalarIdentifier<Short>), ShortIdentifier );

/// Dword Identifier (unsigned long)

DECLARE_TYPE( CORELINUX(ScalarIdentifier<Dword>), DwordIdentifier );

/// Word identifier (unsigned short)

DECLARE_TYPE( CORELINUX(ScalarIdentifier<Word>), WordIdentifier );

/// SemphoreIdentifier used by Semaphores and SemaphoreGroups

DECLARE_TYPE( ShortIdentifier, SemaphoreIdentifier );

/// SemphoreGroupIdentifier used by Semaphores and SemaphoreGroups

DECLARE_TYPE( IntIdentifier, SemaphoreGroupIdentifier );

/// ThreadIdentifier used by Threads

DECLARE_TYPE( IntIdentifier, ThreadIdentifier );

/// ProcessIdentifier used by ?

DECLARE_TYPE( IntIdentifier, ProcessIdentifier );

/// MemoryIdentifier used by Memory

DECLARE_TYPE( IntIdentifier, MemoryIdentifier );

/// User identifier

DECLARE_TYPE( UnsignedIdentifier, UserIdentifier );

/// Group identifier

DECLARE_TYPE( UnsignedIdentifier, GroupIdentifier );


#endif // if defined(__SCALARIDENTIFIERS_HPP)

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.2 $
   $Date: 2000/05/25 04:26:14 $
   $Locker:  $
*/


