/*
 Copyright 2006, CRIM, Stephane Bond

 This file is part of the MILLE-XTERM distribution.
 See the MILLE-XTERM (english) and/or the MILLE (french) project web site

 http://www.revolutionlinux.com/mille-xterm/
 http://www.mille.ca/

 The MILLE-XTERM framework is covered by the GNU General Public License. See
 the COPYING file in the top-level MILLE-XTERM directory. Software packages
 that are included in the MILLE-XTERM distribution have their own licenses.

 -------------------------------------------------------------------------
 
 This script creates function and triggers to manage the nodes nstree structure.

 Nstree are described in this article : 
 - http://www.sitepoint.com/article/hierarchical-data-database
 Trigger implementation example : 
 - http://openacs.org/forums/message-email?message_id=16799
 
 Revisions history:
	- 10 feb 2006: Creation
*/



/* The rebuild_tree(parent, left) function set leftval and rightval values of "parent"
   and his childs. The first left value ef the nested set are specified by the left
   parameter. 
   Tree is traversed recursively.
   Function returns the last right value of the set + 1.

*/
--
-- DROP FUNCTION rebuild_tree (INTEGER, INTEGER);
-- 

CREATE LANGUAGE plpgsql;
CREATE FUNCTION rebuild_tree (INTEGER, INTEGER) RETURNS INTEGER AS '
DECLARE v_parent ALIAS FOR $1;
        v_left ALIAS FOR $2;
        v_right INTEGER;
        child RECORD;
BEGIN
    -- The right value of this node is the left value + 1 
    v_right := v_left + 1;
    
    -- Get all children of this node ordered by id
    FOR child IN SELECT id FROM nodes WHERE id_parent = v_parent AND id != id_parent ORDER BY id LOOP
        -- Recursive execution of this function for each child of this node
        v_right := rebuild_tree(child.id, v_right);
    END LOOP;
    
    -- We`ve got the left value, and now that we`ve processed
    -- the children of this node we also know the right value 
    UPDATE nodes SET leftval = v_left, rightval = v_right WHERE id = v_parent;

    RETURN v_right + 1;
END;
'
LANGUAGE 'plpgsql';



/* The rebuild_tree() function execute rebuild_tree(parent, left) on all root nodes
*/
-- 
-- DROP FUNCTION rebuild_tree();
--
CREATE FUNCTION rebuild_tree () RETURNS BOOLEAN AS '
DECLARE v_next INTEGER;
        child RECORD;
BEGIN
    v_next := 1;
    -- Get all root nodes
    FOR child IN SELECT id FROM nodes WHERE id = id_parent LOOP
        v_next := rebuild_tree(child.id, v_next);
    END LOOP;
    RETURN true;
END;
'
LANGUAGE 'plpgsql';




/* The nodes_delete() trigger function update the nested tree for a leaf deletion.
   Deletion is canceled if the node contains a subtree
*/
--
-- DROP FUNCTION nodes_delete() CASCADE;
--
CREATE FUNCTION nodes_delete () RETURNS OPAQUE AS ' 
BEGIN 
    --  Prevent deletion of non empty nodes
    IF (OLD.rightval - OLD.leftval) != 1 THEN
        RAISE EXCEPTION ''Can not delete non-empty node : %'', OLD.id;
        RETURN NULL;
    ELSE
        UPDATE nodes SET leftval = leftval - 2 WHERE leftval > OLD.rightval;
        UPDATE nodes SET rightval = rightval - 2 WHERE rightval > OLD.rightval;
        RETURN OLD;
   END IF;
END;
'
LANGUAGE 'plpgsql';

CREATE TRIGGER nodes_delete_tr BEFORE DELETE ON nodes FOR EACH ROW EXECUTE PROCEDURE nodes_delete() ;



/* The nodes_insert() trigger function update the tree for the new node insertion
*/
--
-- DROP FUNCTION nodes_insert() CASCADE;
--
CREATE FUNCTION nodes_insert () RETURNS OPAQUE AS '
DECLARE v_max_rightval INTEGER;
        v_parent_rightval INTEGER;
BEGIN
    -- We differentiate two cases, depending on whether the newly
    -- inserted has parent or not.
    IF NEW.id = NEW.id_parent THEN
        SELECT coalesce(max(rightval),0) INTO v_max_rightval FROM nodes;
        NEW.leftval := v_max_rightval + 1;
        NEW.rightval := v_max_rightval + 2;
    ELSE
        --  New node is inserted at the right node of the parent,
        --  shifting everything two places to the right.
        SELECT rightval INTO v_parent_rightval FROM nodes WHERE id = NEW.id_parent;
        UPDATE nodes SET leftval = leftval + 2 WHERE leftval > v_parent_rightval;
        UPDATE nodes SET rightval = rightval + 2 WHERE rightval >= v_parent_rightval;
        NEW.leftval := v_parent_rightval;
        NEW.rightval := v_parent_rightval + 1;
    END IF;
    RETURN NEW;
END;
'
LANGUAGE 'plpgsql';

CREATE TRIGGER nodes_insert_tr BEFORE INSERT ON nodes FOR EACH ROW EXECUTE PROCEDURE nodes_insert() ;



/* The nodes_update_pre() function raise an exception if the update broke the tree integrity.
   A node can not be moved inside one of his child
*/
CREATE FUNCTION nodes_update_pre () RETURNS OPAQUE AS '
DECLARE v_result BOOLEAN;
BEGIN
    -- Exit if parent is not updated
    IF OLD.id_parent = NEW.id_parent THEN
        RETURN NEW;
    END IF;
    -- Can not move a node inside one of his child
    IF NEW.id != NEW.id_parent THEN
        SELECT true INTO v_result FROM nodes WHERE id = NEW.id_parent AND leftval BETWEEN OLD.leftval AND OLD.rightval;
        IF FOUND THEN
            RAISE EXCEPTION ''Can not move a node inside one of his child'';
            RETURN NULL;
        END IF;
    END IF;
    RETURN NEW;
END;
'
LANGUAGE 'plpgsql';

CREATE TRIGGER nodes_update_tr_pre BEFORE UPDATE ON nodes FOR EACH ROW EXECUTE PROCEDURE nodes_update_pre() ;



/* The nodes_update_post() function update the tree after the update operation.
   Operation must be done "after update" because the updated record will be re-updated 
   by the algorithm.

   This leads to 3 cases:
    1.  parent changes from leaf to root
    2.  parent changes from root to leaf
    3.  parent changes from leaf to leaf
*/
--
-- DROP FUNCTION nodes_update_post () CASCADE;
--
/*
-- contains bugs when moving child to the root level
CREATE FUNCTION nodes_update_post () RETURNS OPAQUE AS '
DECLARE v_max_rightval INTEGER;
        v_nested_set_width INTEGER;
        v_shift_offset INTEGER;
        v_parent_leftval INTEGER;
        v_parent_rightval INTEGER;
BEGIN
    -- Exit if parent is not updated
    IF OLD.id_parent = NEW.id_parent THEN
        RETURN NEW;
    END IF;

    -- Prepare move
    SELECT max(rightval) INTO v_max_rightval FROM nodes;
    v_nested_set_width := OLD.rightval - OLD.leftval + 1;
    v_shift_offset := v_max_rightval - OLD.leftval + 1;
    --  Shift nested subset out of its parent.
    UPDATE nodes SET leftval = leftval + v_shift_offset, rightval = rightval + v_shift_offset
     WHERE leftval >= OLD.leftval AND rightval <= OLD.rightval;

    -- Case 1 : change a leaf node to a root node
    IF OLD.id != OLD.id_parent AND NEW.id = NEW.id_parent THEN
        --  Since we have already lifted our subset out of its
        --  position, we will simply renumber the nodes to fill the gaps.
        UPDATE nodes SET leftval = leftval - v_nested_set_width WHERE leftval > OLD.rightval;
        UPDATE nodes SET rightval = rightval - v_nested_set_width WHERE rightval > OLD.rightval;
        RETURN NEW;
    END IF;
    -- Case 2 and 3 : move into a leaf
    IF NEW.id != NEW.id_parent THEN
        --  The tricky part here is that we must pay attention whether
        --  the gap is to the left or to the right of inserting point.
        SELECT leftval, rightval INTO v_parent_leftval, v_parent_rightval FROM nodes WHERE id = NEW.id_parent;
        --  Where will this subset be moved?
        IF v_parent_rightval < OLD.leftval THEN
            --  Gap is to the right of inserting point
            UPDATE nodes SET leftval = leftval + v_nested_set_width WHERE leftval > v_parent_rightval AND leftval < OLD.leftval;
            UPDATE nodes SET rightval = rightval + v_nested_set_width WHERE rightval >= v_parent_rightval AND rightval < OLD.leftval;
            --  Place the subset under its new parent
            v_shift_offset := v_max_rightval - v_parent_rightval + 1;
        ELSE
            --  The gap is to the LEFT of inserting point
            UPDATE nodes SET leftval = leftval - v_nested_set_width WHERE leftval <= v_parent_leftval AND leftval > OLD.rightval;
            UPDATE nodes SET rightval = rightval - v_nested_set_width WHERE rightval < v_parent_leftval AND rightval > OLD.rightval;
            v_shift_offset := v_max_rightval + v_nested_set_width - v_parent_leftval;
        END IF;
        UPDATE nodes SET rightval = rightval - v_shift_offset, leftval = leftval - v_shift_offset WHERE leftval > v_max_rightval;
        RETURN NEW;
    END IF;
END;
'
LANGUAGE 'plpgsql';
*/

CREATE FUNCTION nodes_update_post () RETURNS OPAQUE AS '
DECLARE v_result BOOLEAN;
BEGIN
    -- Exit if parent is not updated
    IF OLD.id_parent = NEW.id_parent THEN
        RETURN NEW;
    END IF;

    v_result := rebuild_tree();
    RETURN NEW;
END;
'
LANGUAGE 'plpgsql';

CREATE TRIGGER nodes_update_tr_post AFTER UPDATE ON nodes FOR EACH ROW EXECUTE PROCEDURE nodes_update_post() ;



/* 
  Rebuild the tree after triggers creation
*/
SELECT rebuild_tree();

