package tests::functional::LrLog2MailTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase tests::functional::TestSchemaFixture/;
use Lire::Config::Build qw/ac_info/;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/tempdir/;
use File::Basename qw/dirname/;
use Cwd 'realpath';
use Carp;
{
    use utf8;
    use MIME::Parser;
}

sub new {
    my $self = shift->SUPER::new(@_);

    $self->tests::functional::TestSchemaFixture::init();

    $self->{'email_to'} = "flacoste\@logreport.org wsl\@logreport.org";
    $self->{'testsdir'} = realpath(dirname( __FILE__ ) . "/..");
    $self->{'report_cfg'} = "$self->{'testsdir'}/data/test.cfg";
    $self->{'parser'} = new MIME::Parser;
    my $tmpdir = tempdir( "MIME::Parser_XXXXXX", 'CLEANUP' => 1 );
    $self->{'parser'}->output_under( $tmpdir );

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schemas();
    $self->set_up_test_specs();
    $self->set_up_TestDlfConverter();
    $self->set_up_test_analysers();

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();
    $self->tests::functional::TestSchemaFixture::tear_down();
}

sub test_version {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail -v" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/(ERROR|WARNING)/, $result->stderr(),
                                  "There were warnings or error messages." );
    my $VERSION = ac_info( "VERSION" );
    $self->assert_matches( qr/shipped with Lire version $VERSION/,
                           $result->stderr() );
}

sub test_ascii_report {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail -c $self->{'report_cfg'} -s 'Test Subject' -o txt test_newapi < $self->{'testsdir'}/data/test.dlf $self->{'email_to'}" );
    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/(ERROR|WARNING)/, $result->stderr(),
                                  "There were warnings or error messages." );

    my $msg = $result->sent_mail;
    $self->assert_equals( 1, scalar @$msg );
    $self->assert_deep_equals( [ split ' ', $self->{'email_to'} ],
                               $msg->[0]{'recipients'} );

    my $entity = $self->{'parser'}->parse_data( $msg->[0]{'message'} );
    my $head = $entity->head;
    $self->assert_equals( "Test Subject\n", $head->get( 'Subject' ) );
    $self->assert_equals( 'text/plain', $head->mime_type() );
}

sub test_bad_service {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail no_such_service $self->{'email_to'}" );
    $self->annotate( $result->stderr() );
    $self->assert_not_equals( 0, $result->status() );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/No such DLF converter/, $result->stderr() );
}

sub test_usage {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail test_newapi" );
    $self->annotate( $result->stderr() );
    $self->assert_not_equals( 0, $result->status() );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/Usage: lr_log2mail/, $result->stderr() );
}

1;
