package latexDraw.ui.components;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.Line2D;

import javax.swing.JComponent;

import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.DrawPanel;
import latexDraw.ui.dialog.ExceptionFrameDialog;


/** 
 * This class defines the Y-scale.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class YScale extends JComponent implements MouseListener
{
	private static final long serialVersionUID = 1L;

	/** The drawPanl which contains the pixels per centimetre */
	private DrawPanel drawPanel;
	
	/** The X-scale of the program */
	private XScale xScale;
	
	
	
	/**
	 * The constructor.
	 * @param drawP The drawPanel which contains the pixels per centimetre.
	 * @param xScale The XScale bounds to the YScale.
	 * @throws IllegalArgumentException If drawP is null.
	 */
	public YScale(DrawPanel drawP, XScale xScale)
	{
		super();
		
		if(drawP==null)
			throw new IllegalArgumentException();
		
		setPreferredSize(new Dimension(XScale.SIZE, 500));
		drawPanel = drawP;
		this.xScale = xScale;
		addMouseListener(this);
		setDoubleBuffered(true);
	}
	
	
	
	@Override
    public void paintComponent(Graphics g)
    {   	
		try
		{
	    	Dimension d = getSize();
	    	double zoom = drawPanel.getZoom();
	    	double height = d.getHeight()*(1/zoom), i, j, cpt;
	    	double pixPerCm = drawPanel.getPixelsPerCm();
	    	Graphics2D g2 = (Graphics2D)g;
	    	
	    	g2.scale(zoom, zoom);
	    	
	    	g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			g2.setStroke(new BasicStroke(0, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
			
			Color formerCol = g2.getColor();
	    	g2.setColor(Color.BLACK);
			
	    	
    		if(xScale.inchMenu.isSelected())
    			pixPerCm*=PSTricksConstants.INCH_VAL_CM;
	    	
    		if(pixPerCm/10.!=(int)(pixPerCm/10.))
			{
    			double tmp = pixPerCm/10.;
    			tmp = tmp-(int)tmp>0.5 ? ((int)tmp)+1 : (int)tmp;
    			pixPerCm = tmp*10;
			}
    		
	    	if(pixPerCm>20/zoom)
	    	{
	    		double pixPerCm10 = pixPerCm/10.;
	    		
	    		for(i=pixPerCm10; i<height; i+=pixPerCm)
	    			for(j=i, cpt=1; cpt<10; j+=pixPerCm10, cpt++)
	    				g2.draw(new Line2D.Double(XScale.SIZE/2./zoom,j,XScale.SIZE/zoom,j));
	    	}
	    	
	    	for(j=0; j<height;j+=pixPerCm)
	    		g2.draw(new Line2D.Double(0.,j,XScale.SIZE/zoom,j));
	    	
	    	g2.setColor(formerCol);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
    }


	
	/**
	 * sets the X-scale.
	 * @param x The new X-scale.
	 * @throws IllegalArgumentException If x is null.
	 */
	public void setXScale(XScale x)
	{
		if(x==null)
			throw new IllegalArgumentException();
		
		xScale = x;
	}
	
	
	
	
    /** 
     * Defines the actions that it must be realised by a right click of the mouse.
     * @param e Correspond to the click of the mouse.
     */
	public void mouseClickedRightButton(MouseEvent e)
    {
		if(xScale!=null && xScale.getPopupMenu()!=null)
			xScale.getPopupMenu().show(e.getComponent(), e.getX(), e.getY());
    }
	
	
	
	
	public void mouseClicked(MouseEvent e) 
	{
    	switch(e.getButton())
	    {
		    case MouseEvent.BUTTON1 :   	
		    			break;
			
		    case MouseEvent.BUTTON2 : 
		    			break;
	
		    case MouseEvent.BUTTON3 : 
						mouseClickedRightButton(e);    
						break;
	    }
	}



	public void mousePressed(MouseEvent e) 
	{
		/* No action to do. */
	}


	
	public void mouseReleased(MouseEvent e) 
	{
		/* No action to do. */
	}


	
	public void mouseEntered(MouseEvent e) 
	{
		/* No action to do. */
	}



	public void mouseExited(MouseEvent e) 
	{
		/* No action to do. */
	}
}
