package latexDraw.ui.components;

import java.awt.Image;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;

import latexDraw.figures.Draw;
import latexDraw.filters.LaTeXDrawTemplateFilter;
import latexDraw.filters.SVGFilter;
import latexDraw.generators.svg.SVGDocumentGenerator;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.parsers.svg.MalformedSVGDocument;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.dialog.ExceptionFrameDialog;
import latexDraw.util.LaTeXDrawException;
import latexDraw.util.LaTeXDrawPath;
import latexDraw.util.LaTeXDrawResources;


/** 
 * Defines the menu that contains the templates menu items.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 * <br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 * <br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 * <br>
 * 04/10/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class MenuTemplate extends JMenu implements MouseListener
{
	private static final long serialVersionUID = 1L;
	
	/** Define if the menu has been constructed. @since 1.9.2 */
	protected boolean constructed;
	
	/** The parent frame. @since 1.9.2 */
	protected LaTeXDrawFrame frame;
	
	
	/**
	 * The constructor.
	 * @param txt The text of the menu.
	 * @param f The parent frame
	 * @since 1.9.2
	 */
	public MenuTemplate(String txt, LaTeXDrawFrame f)
	{
		super(txt);
			
		frame = f;
		constructed = false;
		addMouseListener(this);
	}



	public void mouseClicked(MouseEvent e)
	{
		// nothing to do.
	}



	public void mouseEntered(MouseEvent e)
	{
		constructMenu();
	}

	
	

	/**
	 * Constructs the menu, only is constructed = false.
	 * @since 1.9.2
	 */
	public void constructMenu()
	{
		if(constructed)
			return ;
		
		Vector<JMenuItem> menus = createMenuItems(LaTeXDrawPath.PATH_TEMPLATES_DIR_USER, LaTeXDrawPath.PATH_CACHE_DIR, false);
		
		constructed = true;
		removeAll();
		
		for(JMenuItem menu : menus)
			add(menu);
	
		if(LaTeXDrawPath.PATH_TEMPLATES_SHARED!=null)
		{
			addSeparator();
			
			menus = createMenuItems(LaTeXDrawPath.PATH_TEMPLATES_SHARED, LaTeXDrawPath.PATH_CACHE_SHARE_DIR, true);
			
			for(JMenuItem menu : menus)
				add(menu);
		}
		
		addSeparator();
		JMenuItem menu = new JMenuItem(LaTeXDrawResources.LABEL_TEMPLATE_UPDATE, LaTeXDrawResources.reloadIcon);
		menu.addActionListener(frame);
		menu.setActionCommand(LaTeXDrawResources.LABEL_TEMPLATE_UPDATE);
		add(menu);
	}


	
	
	/**
	 * Creates a list of JMenuItem from the given directory of templates.
	 * @param pathTemplate The path of the templates.
	 * @param pathCache The path of the cache of the templates.
	 * @param sharedTemplates True if the templates are shared templates (in the shared directory).
	 * @return The created JMenuItem.
	 * @since 1.9.2
	 */
	private Vector<JMenuItem> createMenuItems(String pathTemplate, String pathCache, boolean sharedTemplates)
	{
		File templateDir = new File(pathTemplate);
		int i, id;
		SVGFilter filter = new SVGFilter();
		File[] files = templateDir.listFiles();
		char shared = sharedTemplates ? '1' : '0';
		Vector<JMenuItem> menuItems = new Vector<JMenuItem>();

		if(files!=null)
			for(i=0; i<files.length; i++)
				if(filter.accept(files[i]) && !files[i].isDirectory())
				{
					String name = files[i].getName()+LaTeXDrawResources.PNG_EXTENSION;
					JMenuItem menu = createTemplateMenuItem(name, pathCache);
					
					id = name.toLowerCase().indexOf(SVGFilter.SVG_EXTENSION.toLowerCase());
					
					if(menu!=null && id != -1)//remove the extension from the name		
					{
						name = LaTeXDrawResources.TEMPLATE_START_ACTION_CMD + shared + name.substring(0, id);
						menu.setActionCommand(name);
						menuItems.add(menu);
					}//if
				} //if
		
		return menuItems;
	}
	
	
	   

	/**
     * Allows to create a menu item from the name of the jpg file.<br>
     * @param nameThumb The name of the jpg file: file.jpg
     * @return The created JMenuItem. 
     */
    private JMenuItem createTemplateMenuItem(String nameThumb, String path)
    {
    	JMenuItem menu=null;
    	ImageIcon icon=null;
    	String name = nameThumb;
    	boolean updated = false;
    	boolean imageOk = false;
    	int id;
    	
    	while(!imageOk)
	    	try
	    	{
	    		Image image = LaTeXDrawFrame.convertFrame.getImage(new File(path+File.separator+name));
	    		icon = new ImageIcon(image);
	    		image.flush();
	    		imageOk = true;
	    	}
    		catch(Exception e)
	    	{
	    		if(updated)
	    			imageOk = true;
	    		else
	    		{
	    			updateTemplates();
	    			updated = true;
	    		}
	    	}
	    
		id = name.indexOf(LaTeXDrawResources.PNG_EXTENSION);
		
		if(id!=-1)
		{
		    name = name.substring(0, id);
		    id	 = name.indexOf(name.endsWith(SVGFilter.SVG_EXTENSION) ? SVGFilter.SVG_EXTENSION : LaTeXDrawResources.TEMPLATE_EXTENSION);
		    
		    if(id!=-1)
		    {
			    name = name.substring(0, id);
			    	
			    if(icon==null)
			    	icon = LaTeXDrawResources.emptyIcon;
			    
				menu = new JMenuItem(name,icon);
				menu.addActionListener(frame);
		    }
		}
		
    	return menu;
    }
  
    

	public void mouseExited(MouseEvent e)
	{
		// nothing to do.
	}



	public void mousePressed(MouseEvent e)
	{
		// nothing to do.
	}



	public void mouseReleased(MouseEvent e)
	{
		// nothing to do.
	}



	/**
	 * @return the constructed.
	 * @since 1.9.2
	 */
	public boolean isConstructed()
	{
		return constructed;
	}



	/**
	 * @param constructed the constructed to set.
	 * @since 1.9.2
	 */
	public void setConstructed(boolean constructed)
	{
		this.constructed = constructed;
	}




	
	/**
	 * Adds a menu item to the menu.
	 * @param templateName The name of the new template item.
	 * @since 1.9.2
	 */
	public void addItem(String templateName)
	{
		if(templateName==null)
			throw new IllegalArgumentException();
		
		JMenuItem menu = createTemplateMenuItem(templateName + SVGFilter.SVG_EXTENSION + LaTeXDrawResources.PNG_EXTENSION, 
												LaTeXDrawPath.PATH_CACHE_DIR);

		int i, size = getItemCount();
		boolean found = false;
		String ac = LaTeXDrawResources.TEMPLATE_START_ACTION_CMD+'0'+templateName, ac2;
	
		for(i=0; i<size && !found; i++)
		{
			if(getItem(i)==null)
				ac2=null;
			else
				ac2 = getItem(i).getActionCommand();
			
			if(ac2!=null && ac2.equals(ac))
				found = true;
		}
			
		if(found)
			updateTemplates();
		else
		{
			menu.setActionCommand(ac);			
			add(menu, 0);
		}
	}


	
	
	/**
	 * Allows to update the templates and their icon.<br>
	 */
	public void updateTemplates()
	{
		try
		{
			convertOldTemplates();
			updateTemplates(LaTeXDrawPath.PATH_TEMPLATES_DIR_USER, LaTeXDrawPath.PATH_CACHE_DIR);
			updateTemplates(LaTeXDrawPath.PATH_TEMPLATES_SHARED, LaTeXDrawPath.PATH_CACHE_SHARE_DIR);
		}
		catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}

	
	
	/**
	 * Allows to get the figures from a templates
	 * @param file The file containing the template
	 * @return The figures
	 */
	public static Draw readOldTemplate(File file)
	{
		try
		{
			Draw template;
			FileInputStream fis = new FileInputStream(file);
			ObjectInputStream in = new ObjectInputStream(fis);
			String nameApp = (String)in.readObject();
			
			if(!nameApp.equals(LaTeXDrawResources.TOKEN_TEMPLATE_FILE))
				throw new LaTeXDrawException(LaTeXDrawException.NOT_TEXDRAW_FILE);
			
			LaTeXDrawFrame.setVersionOfFile((String)in.readObject());
			
			template = (Draw)in.readObject();
			in.close();
			fis.close();
			
			return template;
			
		}catch(LaTeXDrawException e)
		{
			JOptionPane.showMessageDialog(null, 
		 		   LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.161")//$NON-NLS-1$
		 		   + file.getAbsolutePath()+", "+e.toString(), //$NON-NLS-1$
		 		   LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"),  //$NON-NLS-1$
		 		    JOptionPane.WARNING_MESSAGE); 
			
		}catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
		
		return null;
	}
	
	
	
	public void convertOldTemplates()
	{
		convertOldTemplates(LaTeXDrawPath.PATH_TEMPLATES_DIR_USER, LaTeXDrawPath.PATH_CACHE_DIR);
		convertOldTemplates(LaTeXDrawPath.PATH_TEMPLATES_SHARED, LaTeXDrawPath.PATH_CACHE_SHARE_DIR);
	}
	
	
	
	
	private void convertOldTemplates(String pathTemplates, String pathCache)
	{
		if(pathTemplates==null || pathCache==null)
			return ;
		
		LaTeXDrawTemplateFilter oldFilter = new LaTeXDrawTemplateFilter();
		File dir = new File(pathTemplates);
		
		if(dir.isDirectory() && dir.canRead() && dir.canWrite())
		{
			File[] files = new File(pathTemplates).listFiles();
			String path;
			int id;
			File testFile;
			Draw d;
			
			for(File f : files)
				if(oldFilter.accept(f))
				{
					path = f.getPath();
					id = path.indexOf(LaTeXDrawResources.TEMPLATE_EXTENSION);
					
					if(id!=-1)
					{
						testFile = new File(path.substring(0, id) + SVGFilter.SVG_EXTENSION);
						
						if(!testFile.exists())
						{
							d = readOldTemplate(f);
							
							try { SVGDocumentGenerator.exportAsSVG(testFile.getPath(), d); }
							catch(IOException e) { e.printStackTrace(); }
						}
					}
				}
		}
	}
	
	
	
	
	/**
	 * Updates the templates from the given path, in the given cache path.
	 * @param pathTemplate The path of the templates to update.
	 * @param pathCache The path where the cache of the thumbnails of the templates will be stored.
	 * @since 1.9.2
	 */
	private void updateTemplates(String pathTemplate, String pathCache)
	{
		File templateDir = new File(pathTemplate);
		File thumbnail;
		
		if(!templateDir.exists() || !templateDir.isDirectory())
			return ; // There is no template
		
		// We get the list of the templates
		SVGFilter filter = new SVGFilter();
		File[] files = templateDir.listFiles();
		int i;
		Draw template;
		
		// We export the updated template
		if(files!=null)
			for(i=0; i<files.length; i++)
				try
				{
					if(filter.accept(files[i]))
					{
						template = SVGDocumentGenerator.toLatexdraw(files[i], null, null);
							
						thumbnail = new File(pathCache+File.separator+files[i].getName()+LaTeXDrawResources.PNG_EXTENSION);
						frame.getDrawPanel().getDraw().createTemplateThumbnail(thumbnail, template);
					}
				}
				catch(MalformedSVGDocument e) { e.printStackTrace(); }

		constructed = false;
	}
}
