package latexDraw.parsers.svg;

import org.w3c.dom.*;

/**
 * Defines an SVG attribute.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/16/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class SVGAttr implements Attr, Cloneable
{
	/** The name of the attribute. @since 0.1 */
	protected String name;
	
	/** The owner of the attribute. @since 0.1 */
	protected Element owner;
	
	/** The value of the attribute. @since 0.1 */
	protected String value;
	
	
	
	public SVGAttr(String n, String v, Element parent)
	{
		if(n==null || parent==null || v==null)
			throw new IllegalArgumentException();
		
		name 	= n;
		owner 	= parent;
		value 	= v;
	}
	
	
	
	public String getName()
	{
		return name;
	}



	public Element getOwnerElement()
	{
		return owner;
	}



	public String getValue()
	{
		return value;
	}



	public boolean isId()
	{
		return name.equals(SVGAttributes.SVG_ID);
	}



	public void setValue(String value) throws DOMException
	{
		if(value==null)
			throw new DOMException(DOMException.SYNTAX_ERR, "A value cannot be null");//$NON-NLS-1$
		
		this.value = value;
	}


	
	/** No deep allow. */
	public Node cloneNode(boolean deep)
	{
		try { return (SVGAttr)super.clone(); }
		catch(CloneNotSupportedException e) { return null; }
	}



	public String getNodeName()
	{
		return name;
	}



	public short getNodeType()
	{
		return Node.ATTRIBUTE_NODE;
	}



	public String getNodeValue() throws DOMException
	{
		return value;
	}



	public Node getParentNode()
	{
		return owner;
	}


	public boolean hasAttributes()
	{
		return false;
	}


	public boolean hasChildNodes()
	{
		return false;
	}



	public boolean isEqualNode(Node arg)
	{
		return arg!=null && arg.getNodeName().equals(name) && arg.getNodeValue().equals(value) &&
			   arg.getNodeType()==getNodeType();
	}

	

	public boolean isSameNode(Node other)
	{
		return other!=null && other==this;
	}


	
	public void setNodeValue(String nodeValue) throws DOMException
	{
		setValue(nodeValue);
	}
	
	
	
	@Override
	public String toString()
	{
		StringBuffer stringBuffer = new StringBuffer();
		
		stringBuffer.append("["); //$NON-NLS-1$
		stringBuffer.append(name);
		stringBuffer.append(", "); //$NON-NLS-1$
		stringBuffer.append(value);
		stringBuffer.append("]"); //$NON-NLS-1$
		
		return stringBuffer.toString();
	}

	
	public String getPrefix()
	{
		if(getNodeName()==null)
			return null;
		
		int index = getName().indexOf(':');
		
		if(index!=-1)
			return getName().substring(0, index);
		
		return null;
	}

	
	public boolean getSpecified()
	{
		return true; 
	}
	
	
	public String getNamespaceURI()
	{ 
		return lookupNamespaceURI(getPrefix()); 
	}
	
	
	public String lookupNamespaceURI(String prefix)
	{ 
		return owner.lookupNamespaceURI(prefix);
	}
	
	
	public String getLocalName()
	{	
		return name;
	}
	
	
	public TypeInfo getSchemaTypeInfo()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public short compareDocumentPosition(Node other) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public String getBaseURI()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Object getFeature(String feature, String version)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Node getNextSibling()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Document getOwnerDocument()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Node getPreviousSibling()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public String getTextContent() throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Object getUserData(String key)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public boolean isDefaultNamespace(String namespaceURI)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public boolean isSupported(String feature, String version)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public String lookupPrefix(String namespaceURI)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void normalize()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }
	
	public Node replaceChild(Node newChild, Node oldChild) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setPrefix(String prefix) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public void setTextContent(String textContent) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Object setUserData(String key, Object data, UserDataHandler handler)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	
	public Node removeChild(Node oldChild) throws DOMException
	{ return null; }

	public Node insertBefore(Node newChild, Node refChild) throws DOMException
	{ return null; }
	
	public NamedNodeMap getAttributes()
	{ return null; }

	public NodeList getChildNodes()
	{ return null; }

	public Node getFirstChild()
	{ return null; }

	public Node getLastChild()
	{ return null; }

	public Node appendChild(Node newChild) throws DOMException
	{ return null; }
}
