/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggwidgetkeyhandler.h"


#include "iconsole.h"
#include "icontrolmodule.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iobjecthelp.h"
#include "ishell.h"

#include "iggmainwindow.h"

#include "ibgframesubject.h"


using namespace iParameter;


//
//  Template class
//
template<class T>
iggWidgetKeyHandler<T>::iggWidgetKeyHandler(int type, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyHandlerBase(type,key,rm,parent,index,indkey)
{
	this->mActionKey = false;
}


//
//  send the control information to the ControlModule
//
template<class T>
bool iggWidgetKeyHandler<T>::ExecuteControl(bool final)
{
	T val;
	this->QueryValue(val);
	return this->ExecuteControl(final,val);
}


template<class T>
bool iggWidgetKeyHandler<T>::ExecuteControl(bool final, T val)
{
	const iObjectKey &key = this->GetKey(); // cache the key

	this->mValue = val;

	this->GetMainWindow()->Block(true);

	iString ws;
	if(key.Dimension() == 1)
	{
		//
		//  Scalar key
		//
		this->GetShell()->GetControlModule()->PackCommand(ws,key,this->mValue);
	}
	else
	{
		//
		//  Vector key, use index (which must be >= 0 then)
		//
		if(this->mIndex>=0 && (key.Dimension()==0 || this->mIndex<key.Dimension()))
		{
			this->GetShell()->GetControlModule()->PackCommand(ws,key,mIndex,this->mValue);
		}
		else
		{
			//
			//  Full vector - all elements are the same
			//
			T *v = new T[key.Dimension()];
			for(int i=0; i<key.Dimension(); i++) v[i] = val;
			this->GetShell()->GetControlModule()->PackCommand(ws,key,v,key.Dimension());
			delete [] v;
		}
	}

	bool ren = (this->GetRenderMode()==RenderMode::Immediate || (final && this->GetRenderMode()!=RenderMode::NoRender));
	bool ret = this->GetShell()->GetControlModule()->Execute(ws,ren && this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());

	if(!ret && this->GetShell()->GetControlModule()->GetErrorStatus()->IsError())
	{
		iConsole::Display(iConsole::_HighError,this->GetShell()->GetControlModule()->GetErrorStatus()->Message());
	}

	if(this->mSelfChecking && (final || this->GetRenderMode()==RenderMode::Immediate))
	{
		this->UpdateWidget();
	}

	this->GetMainWindow()->Block(false);

	return ret;
}


//
//  query the control module about the proper state for this widget
//
template<class T>
void iggWidgetKeyHandler<T>::UpdateWidgetBody()
{
	this->UpdateHelp();

	const iObjectKey &key = this->GetKey(); // cache the key
	if(key.Dimension() == 1)
	{
		//
		//  Scalar key - index and index key are ignored
		//
	    if(this->GetShell()->GetControlModule()->QueryValue(key,this->mValue))
		{
			if(!this->mActionKey) this->Enable(true);
			this->UpdateValue(this->mValue);
		}
		else if(!this->mActionKey) this->Enable(false);
	}
	else
	{
		//
		//  Vector key, use index (which must be >= 0 then) and index key
		//
		if(this->mIndexKey != 0)
		{
			//
			//  Determine the index
			//
		    if(!this->GetShell()->GetControlModule()->QueryValue(*this->mIndexKey,this->mIndex))
			{
				this->UpdateFailed();
				if(!this->mActionKey) this->Enable(false);
				return;
			}
		}
		if(this->mIndex>=0 && (key.Dimension()==0 || this->mIndex<key.Dimension()))
		{
			if(this->GetShell()->GetControlModule()->QueryValue(key,this->mIndex,this->mValue,(key.Dimension()==0)?this->mIndex+1:key.Dimension(),-1,-1))
			{
				if(!this->mActionKey) this->Enable(true);
				this->UpdateValue(this->mValue);
			}
			else if(!this->mActionKey) this->Enable(false);
		}
		else
		{
			//
			//  Full vector - this is not implemented
			//
//			this->GetShell()->GetControlModule()->QueryValue(this->mKey,this->mValue,this->mKey.Dimension());
			IERROR_LOW("iggWidgetKeyHandler with the full vector argument is not implemented.");
			this->UpdateFailed();
			if(!this->mActionKey) this->Enable(false);
		}
	}
}

//
//  Dummy invisible widget with full KeyHandler functionality
//
template <class T>
iggWidgetKeyDummy<T>::iggWidgetKeyDummy(const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyHandler<T>(WidgetType::Other,key,RenderMode::Immediate,parent,index,indkey)
{
	this->mWasLaidOut = true; // a dummy does not need a layout
	this->mSubjectOwner = false; // a dummy does not own its subject
	this->mValue = T(0); //  initialize the value
	//
	//  Share the helper and the subject
	//
	this->AttachSubject(parent->GetSubject());
	this->mNeedsBaloonHelp = false;
}


template <class T>
void iggWidgetKeyDummy<T>::SetValue(T v)
{
	this->ExecuteControl(true,v);
}


template <class T>
void iggWidgetKeyDummy<T>::QueryValue(T &) const
{
	// nothing to do here
}

template <class T>
void iggWidgetKeyDummy<T>::UpdateValue(T)
{
	// nothing to do here
}

