// -*- C++ -*-
//
// TBDiagram.h is a part of Herwig++ - A multi-purpose Monte Carlo event generator
// Copyright (C) 2002-2011 The Herwig Collaboration
//
// Herwig++ is licenced under version 2 of the GPL, see COPYING for details.
// Please respect the MCnet academic guidelines, see GUIDELINES for details.
//
#ifndef HERWIG_TBDiagram_H
#define HERWIG_TBDiagram_H
//
// This is the declaration of the TBDiagram struct.
//

#include "ThePEG/Persistency/PersistentOStream.h"
#include "ThePEG/Persistency/PersistentIStream.h"
#include "ThePEG/Helicity/Vertex/VertexBase.h"

namespace Herwig {
using namespace ThePEG;
using Helicity::VertexBasePtr;

/** Pair of particle ids. */
typedef pair<long, long> IDPair;
  
/** Pair of bool's*/
typedef pair<bool, bool> BPair;

/** Convenient typedef of VertexBasePtr */
typedef VertexBasePtr VBPtr;

/** Pair of VertexBasePtrs */
typedef pair<VBPtr, VBPtr> VBPair;
  
/** Pair of int,double */
typedef pair<unsigned int, double> CFPair;
  
/**
 * The TBDiagram struct contains information about a \f$1\to3\f$ decay that 
 * has been automatically generated by ThreeBodyDecayConstructor.
 */  
struct TBDiagram {

  /** Enumeration for channel type */
  enum Channel {UNDEFINED = -1, channel23=0, channel13=1, channel12=2, fourPoint=3};

  /** Standard Constructor */
  TBDiagram()  
    : incoming(0), outgoing(0), outgoingPair(make_pair(0, 0)),
      channelType(UNDEFINED), colourFlow(0), ids(4, 0) 
  {}

  /** Constructor taking ids as arguments.*/
  TBDiagram(long a, long b, IDPair c)
    : incoming(a), outgoing(b), outgoingPair(c), 	
      channelType(UNDEFINED), colourFlow(0), ids(4, 0) {
    ids[0] = a;
    ids[1] = b;
    ids[2] = c.first;
    ids[3] = c.second;
  }
  
  /** Incoming particle id's */
  long incoming;
  
  /** Outgoing particle from first vertex */
  long outgoing;

  /** Outgoing particle id's fropm resonance*/
  IDPair outgoingPair;

  /** ParticleData pointer to intermediate, null for 4-point vertices */
  PDPtr intermediate;

  /** The two vertices for the diagram */
  VBPair vertices;
  
  /** Enum of channel type */
  Channel channelType;

  /** Store colour flow at \f$N_c=3\f$ information */
  mutable vector<CFPair> colourFlow;

  /** Store colour flow at \f$N_c=\infty\f$ information */
  mutable vector<CFPair> largeNcColourFlow;

  /** Store the ids in a vector for easy use of comparison operator. */
  vector<long> ids;

  /**
   * Test whether this and x are the same decay
   * @param x The other process to check
   */
  bool sameDecay(const TBDiagram & x) const {
    if(ids[0] != x.ids[0]) return false;
    bool used[4]={false,false,false,false};
    for(unsigned int ix=1;ix<4;++ix) {
      bool found=false;
      for(unsigned int iy=1;iy<4;++iy) {
	if(used[iy]) continue;
	if(ids[ix]==x.ids[iy]) {
	  used[iy]=true;
	  found=true;
	  break;
	}
      }
      if(!found) return false;
    }
    return true;
  }
};

/**
 * Test whether two diagrams are identical.
 */
inline bool operator==(const TBDiagram & x, const TBDiagram & y) {
  if( x.incoming     != y.incoming) return false;
  if( x.outgoing     != y.outgoing) return false;
  if( x.intermediate != y.intermediate) return false;
  if( x.vertices     != y.vertices    ) return false;
  if( (x.outgoingPair.first  == y.outgoingPair.first  &&
       x.outgoingPair.second == y.outgoingPair.second ) ||
      (x.outgoingPair.first  == y.outgoingPair.second &&
       x.outgoingPair.second == y.outgoingPair.first  ) ) return true;
  else return false;
}

/** 
 * Output operator to allow the structure to be persistently written
 * @param os The output stream
 * @param x The TBDiagram 
 */
inline PersistentOStream & operator<<(PersistentOStream & os, 
				      const TBDiagram  & x) {
  os << x.incoming << x.outgoing << x.outgoingPair << x.intermediate
     << x.vertices << x.channelType << x.colourFlow << x.largeNcColourFlow
     << x.ids;
  return os;
}
  
/** 
 * Input operator to allow persistently written data to be read in
 * @param is The input stream
 * @param x The TBDiagram 
 */
inline PersistentIStream & operator>>(PersistentIStream & is,
			       TBDiagram & x) {
  int chan;
  is >> x.incoming >> x.outgoing >> x.outgoingPair >> x.intermediate
     >> x.vertices >> chan >> x.colourFlow >> x.largeNcColourFlow >> x.ids;
  x.channelType = TBDiagram::Channel(chan);
  return is;
}

}

#endif /* HERWIG_TBDiagram_H */
