------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                       --
--  Copyright (C) 2006-2007 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: test_dbi.adb,v 1.11 2008/05/05 10:21:37 vagul Exp $

--  Example of using dynamically allocated bind variables.

with Ada.Calendar;
with Ada.Command_Line;
with Ada.Strings.Fixed;
with Ada.Text_IO;

with GNAT.MD5; -- easy could be removed for non GNAT Ada compiler.

with OCI.Thick.DB;
with OCI.Thick.Containers.Strings;
with OCI.Thick.Connections;
with OCI.Thick.Lobs;
with OCI.Thick.Number_Functions;

procedure Test_DBI is

   --  TODO: non-blocking support

   use Ada.Text_IO;
   use OCI.Thick;
   use OCI.Thick.Connections;
   use OCI.Thick.Containers;
   use OCI.Thick.DB;
   use ASCII;

   Stmt : Statement := Prepare ("declare"  & LF
     & "   Dfmt Varchar2 (24) := 'YYYY-MM-DD HH24:MI:SS';"  & LF
     & "   function Short_Image (Item Varchar2) return Varchar2 is"  & LF
     & "   begin"  & LF
     & "      if Length (Item) < 32 then"  & LF
     & "         return Item;"  & LF
     & "      else"  & LF
     & "         return Lower (DBMS_Obfuscation_Toolkit.MD5"  & LF
     & "                  (Input => UTL_RAW.Cast_To_RAW (Item)))"  & LF
     & "                  || ':' || Length (Item);"  & LF
     & "      end if;"  & LF
     & "   end Short_Image;"  & LF
     & "begin  " & LF
     & " DBMS_Output.Put ('#1 ' || NVL (To_Char (:Int1), 'nul'));"
     & " DBMS_Output.Put (' ' || NVL (To_Char (:Date1, Dfmt), 'nul'));" & LF
     & " DBMS_Output.Put (' ' || NVL (To_Char (:Float1), 'nul'));" & LF
     & " DBMS_Output.Put (' ' || NVL (To_Char (:Numb1), 'nul'));" & LF
     & " DBMS_Output.Put_Line (' ' || NVL (:Str1, 'nul'));" & LF
     & " DBMS_Output.Put ('#2 ' || NVL (To_Char (:Int2), 'nul'));"
     & " DBMS_Output.Put (' ' || NVL (To_Char (:Date2, Dfmt), 'nul'));" & LF
     & " DBMS_Output.Put (' ' || NVL (To_Char (:Float2), 'nul'));" & LF
     & " DBMS_Output.Put (' ' || NVL (To_Char (:Numb2), 'nul'));" & LF
     & " DBMS_Output.Put_Line (' ' || Short_Image (NVL (:Str2, 'nul')));" & LF
     & " case NVL (:Int1, :Int2)" & LF
     & " when 1 then :Int1    := 111;" & LF
     & " when 2 then :Str2 := 'STR-2.';" & LF
         & " :Date1 := :Date1 + 1;" & LF
     & " when 3 then :Float1  := 7000000;" & LF
       & " :Date1 := To_Date ('2006-10-18 13:21:24', Dfmt);" & LF
       & "for J in 0 .. 3199 loop" & LF
       & "  if J rem 80 = 0 then" & LF
       & "     :Str2 := :Str2 || Chr (10);" & LF
       & "  else" & LF
       & "     :Str2 := :Str2 || (J rem 10);" & LF
       & "  end if;" & LF
       & "end loop;" & LF
     & " end case;" & LF
     & " end;");

   Ind4 : constant String := "/-\|";

   function Connect_String return String is
   begin
      if Ada.Command_Line.Argument_Count >= 1 then
         return Ada.Command_Line.Argument (1);
      else
         return "scott/tiger";
      end if;
   end Connect_String;

   Stmt2 : Statement := Prepare ("select * from Emp order by EmpNo");

   Stmt3 : Statement
     := Prepare ("select RowNum, E1.EMPNO, E2.EMPNO, E3.EMPNO,"
                    & " LPad ('@', RowNum, '1234567890')"
                    & " from Emp E1, Emp E2, Emp E3 order by 1, 2, 3, 4");

   DE : constant array (Boolean) of Character
     := (False => ASCII.HT, True => ASCII.LF);

   Connect : Connection := Logon (Connect_String);

   Stmt_Output : Statement
     := Prepare (Connect,
                 "begin"
                 & " DBMS_Output.Get_Line "
                 & " (Line => :Output, Status => :Status);"
                 & " end;");

   Inc_Str : Statement
     := Prepare (Connect, "begin :Str := :Str || :Char; end;");

   Str : Data_Holder := To_Data ("start" & LF);
   Sample : constant String
     := "0123456789 #()[]{}<>?.,:;_+|\/-"
        & "abcdefghijklmnopqrstuvwxyz"
        & "ABCDEFGHIJKLMNOPQRSTUVWXYZ" & LF;
   S_Index : Natural := 1;

   procedure In_Str (Data : out Data_Holder; Position, Iteration : Positive) is
      Idx : Positive;
   begin
      case Position is
      when 1 => Data := Str;
      when 2 =>
         if S_Index rem 2 = 0 then
            Data := Null_Data;
         else
            Idx := S_Index / 2 rem Sample'Length + 1;
            Data := To_Data (Sample (Idx .. Idx));
         end if;

         S_Index := S_Index + 1;
      when others =>
         raise Program_Error with "Position" & Position'Img & " input absent.";
      end case;
   end In_Str;

   procedure Out_Str (Data : in Data_Holder; Position, Iteration : Positive) is
   begin
      if Position = 1 then
         Str := Data;
         Put (Ind4 (Iteration rem 4 + 1) & CR);
      else
         raise Program_Error with "Position" & Position'Img & " output error.";
      end if;
   end Out_Str;

   procedure Test_Out
     (Data : in Data_Holder; Position, Iteration : in Positive)
   is
      Cols : constant array (1 .. 5) of Ada.Text_IO.Count
        := (1, 5, 10, 15, 20);
      Max_Show : constant := 70;
   begin
      if Position = 1 and then Iteration /= Value (Data) then
         Put_Line ("error iteration" & Iteration'Img & " /="  &  Image (Data));
      end if;

      if Iteration <= Max_Show then
         Set_Col (Cols (Position));
         Put (Image (Data));

      elsif Position = 5 then
         declare
            Line : constant String := Value (Data);
         begin
            for J in Line'First .. Line'Last - 1 loop
               if Integer'Value ("" & Line (J)) /= J rem 10 then
                  Put_Line ("error " & Line);
               end if;
            end loop;

            if Line (Line'Last) /= '@' then
               Put_Line ("error " & Line);
            end if;

            if Iteration = Max_Show + 1 then
               New_Line;
            else
               Put (Ind4 (Iteration rem 4 + 1) & CR);
            end if;
         end;
      end if;
   end Test_Out;

   RSet, Data, Output : Strings.Container_Type;
   NC : Positive;

   BCols : constant array (1 .. 10) of Ada.Text_IO.Count
     := (1, 8, 28, 35, 42, 49, 56, 76, 83, 90);

   Cols : constant array (1 .. 8) of Ada.Text_IO.Count
     := (1, 7, 14, 24, 31, 51, 56, 65);

begin
   Execute (Connect, "begin DBMS_Output.Enable (Buffer_Size => 4096); end;");

   Bind (Stmt_Output, Type_String, "Output");
   Bind (Stmt_Output, Type_Integer, "Status");
   Output.Name_Bind_Positions (Stmt_Output);

   Bind (Stmt, Type_Integer, "Int1");
   Bind (Stmt, Type_Integer, "Int2");

   Bind (Stmt, Type_Long_Float, "Float1");
   Bind (Stmt, Type_Long_Float, "Float2");

   Bind (Stmt, Type_Number, "Numb1");
   Bind (Stmt, Type_Number, "Numb2");

   Bind (Stmt, Type_Date, "Date1");
   Bind (Stmt, Type_Date, "Date2");

   Bind (Stmt, Type_String, "Str1");
   Bind (Stmt, Type_String, "Str2");

   Data.Name_Bind_Positions (Stmt);

   Ada.Text_IO.Put_Line ("binded");

   for J in 1 .. 3 loop
      if J rem 2 = 0 then
         Data.Set (J, "Int1", Iteration => J);
         Data.Set (Long_Float (J), "Float1", Iteration => J);
         Data.Set (Number_Functions.To_Number (J), "Numb1", Iteration => J);
         Data.Set (Ada.Calendar.Time_Of (1998, 11, 22, 3660.0 + Duration (J)),
               "Date1", Iteration => J);
         Data.Set ("str" & Integer'Image (-J), "Str1", Iteration => J);
      else
         Data.Set (J, "Int2", Iteration => J);
         Data.Set (Long_Float (J), "Float2", Iteration => J);
         Data.Set (Number_Functions.To_Number (J), "Numb2", Iteration => J);
         Data.Set (Ada.Calendar.Time_Of (1998, 11, 22, 3660.0 + Duration (J)),
               "Date2", Iteration => J);
         Data.Set ("str" & Integer'Image (-J), "Str2", Iteration => J);
      end if;
   end loop;

   --  Read source code into Data.

   declare
      use Ada.Text_IO;
      use GNAT.MD5;
      File : File_Type;
      MD5  : Context;
      Length : Natural := 0;
   begin
      Open (File, In_File, "test_dbi.adb");

      Data.Set ("", 10, 2);

      while not End_Of_File (File) loop
         declare
            Line : constant String := Get_Line (File) & ASCII.LF;
         begin
            Update (MD5, Line);
            Data.Append (Line, 10, 2);
            Length := Length + Line'Length;
         end;
      end loop;

      Put_Line (Digest (MD5) & Integer'Image (Length));
   end;

   Execute (Connect, Stmt, Data);

   Output.Clear (1);

   loop
      Execute (Stmt_Output, Output);
      case Output.Get ("Status") is
      when 0 => Ada.Text_IO.Put_Line (Strings.Get (Output, "Output"));
      when 1 => exit;
      when others =>
         raise Constraint_Error with Integer'Image (Output.Get ("Status"));
      end case;
   end loop;

   for J in 1 .. Bind_Count (Stmt) loop
      Set_Col (BCols (J));
      Ada.Text_IO.Put (Bind_Name (Stmt, J));
   end loop;

   for J in 1 .. Data.Length loop
      for K in 1 .. Bind_Count (Stmt) loop
         Set_Col (BCols (K));
         Put (Image (Strings.Get (Data, K, J), "(null)"));
      end loop;
   end loop;

   Data.Clear;

   Execute (Connect, Stmt2);

   Define (Stmt2, Type_Integer, 1);
   Define (Stmt2, Type_String,  2);
   Define (Stmt2, Type_String,  3);
   Define (Stmt2, Type_Integer, 4);
   Define (Stmt2, Type_Date,    5);
   Define (Stmt2, Type_Number,  6);
   Define (Stmt2, Type_Long_Float, 7);
   Define (Stmt2, Type_String, 8);

   NC := Number_Of_Columns (Stmt2);

   for K in 1 .. NC loop
      Set_Col (Cols (K));
      Put (Column_Name (Stmt2, K));
   end loop;

   Fetch (Stmt2, RSet, 20);

   for J in 1 .. RSet.Length loop
      for K in 1 .. Number_Of_Columns (Stmt2) loop
         Set_Col (Cols (K));
         Put (Image (Strings.Get (RSet, K, J), "(null)"));
      end loop;
   end loop;

   New_Line;

   Bind (Inc_Str, Type_String, 1);
   Bind (Inc_Str, Type_String, 2);

   Execute (Inc_Str, In_Str'Access, Out_Str'Access, Count => 6400);

   Put_Line (Value (Str));

   Execute (Connect, Stmt3);

   Define (Stmt3, Type_Integer, 1);
   Define (Stmt3, Type_Number,  2);
   Define (Stmt3, Type_Number,  3);
   Define (Stmt3, Type_Number,  4);
   Define (Stmt3, Type_String,  5);

   if Fetch (Stmt3, Test_Out'Access, 3000) then
      raise Constraint_Error with "too many rows";
   end if;

   if Is_Objects_Support then
      Put_Line ("Objects supported environment.");
   end if;

   Put_Line ("Done.");
end Test_DBI;
