# Gmsh - Copyright (C) 1997-2021 C. Geuzaine, J.-F. Remacle
#
# See the LICENSE.txt file for license information. Please report all
# issues on https://gitlab.onelab.info/gmsh/gmsh/issues.

# This file defines the Gmsh Python API (v4.8.0).
#
# Do not edit it directly: it is automatically generated by `api/gen.py'.
#
# By design, the Gmsh Python API is purely functional, and only uses elementary
# Python types (as well as `numpy' arrays if `numpy' is available). See
# `tutorial/python' and `demos/api' for examples.

from ctypes import *
from ctypes.util import find_library
import signal
import os
import platform
from math import pi

GMSH_API_VERSION = "4.8.0"
GMSH_API_VERSION_MAJOR = 4
GMSH_API_VERSION_MINOR = 8
GMSH_API_VERSION_PATCH = 0

__version__ = GMSH_API_VERSION

oldsig = signal.signal(signal.SIGINT, signal.SIG_DFL)
libdir = os.path.dirname(os.path.realpath(__file__))
if platform.system() == "Windows":
    libpath = os.path.join(libdir, "gmsh-4.8.dll")
elif platform.system() == "Darwin":
    libpath = os.path.join(libdir, "libgmsh.dylib")
else:
    libpath = os.path.join(libdir, "libgmsh.so")

if not os.path.exists(libpath):
    libpath = find_library("gmsh")

lib = CDLL(libpath)

use_numpy = False
try:
    import numpy
    try:
        from weakref import finalize as weakreffinalize
    except:
        from backports.weakref import finalize as weakreffinalize
    use_numpy = True
except:
    pass

# Utility functions, not part of the Gmsh Python API

def _ostring(s):
    sp = s.value.decode("utf-8")
    lib.gmshFree(s)
    return sp

def _ovectorpair(ptr, size):
    v = list((ptr[i * 2], ptr[i * 2 + 1]) for i in range(size//2))
    lib.gmshFree(ptr)
    return v

def _ovectorint(ptr, size):
    if use_numpy:
        if size == 0 :
            lib.gmshFree(ptr)
            return numpy.ndarray((0,),numpy.int32)
        v = numpy.ctypeslib.as_array(ptr, (size, ))
        weakreffinalize(v, lib.gmshFree, ptr)
    else:
        v = list(ptr[i] for i in range(size))
        lib.gmshFree(ptr)
    return v

def _ovectorsize(ptr, size):
    if use_numpy:
        if size == 0 :
            lib.gmshFree(ptr)
            return numpy.ndarray((0,),numpy.uintp)
        v = numpy.ctypeslib.as_array(ptr, (size, ))
        weakreffinalize(v, lib.gmshFree, ptr)
    else:
        v = list(ptr[i] for i in range(size))
        lib.gmshFree(ptr)
    return v

def _ovectordouble(ptr, size):
    if use_numpy:
        if size == 0 :
            lib.gmshFree(ptr)
            return numpy.ndarray((0,),numpy.float64)
        v = numpy.ctypeslib.as_array(ptr, (size, ))
        weakreffinalize(v, lib.gmshFree, ptr)
    else:
        v = list(ptr[i] for i in range(size))
        lib.gmshFree(ptr)
    return v

def _ovectorstring(ptr, size):
    v = list(_ostring(cast(ptr[i], c_char_p)) for i in range(size))
    lib.gmshFree(ptr)
    return v

def _ovectorvectorint(ptr, size, n):
    v = [_ovectorint(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ovectorvectorsize(ptr, size, n):
    v = [_ovectorsize(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ovectorvectordouble(ptr, size, n):
    v = [_ovectordouble(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ovectorvectorpair(ptr, size, n):
    v = [_ovectorpair(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ivectorint(o):
    if use_numpy:
        array = numpy.ascontiguousarray(o, numpy.int32)
        if(len(o) and array.ndim != 1):
            raise Exception("Invalid data for input vector of integers")
        ct = array.ctypes
        ct.array = array
        return ct, c_size_t(len(o))
    else:
        return (c_int * len(o))(*o), c_size_t(len(o))

def _ivectorsize(o):
    if use_numpy:
        array = numpy.ascontiguousarray(o, numpy.uintp)
        if(len(o) and array.ndim != 1):
            raise Exception("Invalid data for input vector of sizes")
        ct = array.ctypes
        ct.array = array
        return ct, c_size_t(len(o))
    else:
        return (c_size_t * len(o))(*o), c_size_t(len(o))

def _ivectordouble(o):
    if use_numpy:
        array = numpy.ascontiguousarray(o, numpy.float64)
        if(len(o) and array.ndim != 1):
            raise Exception("Invalid data for input vector of doubles")
        ct = array.ctypes
        ct.array = array
        return  ct, c_size_t(len(o))
    else:
        return (c_double * len(o))(*o), c_size_t(len(o))

def _ivectorpair(o):
    if use_numpy:
        array = numpy.ascontiguousarray(o, numpy.int32)
        if(len(o) and (array.ndim != 2 or array.shape[1] != 2)):
            raise Exception("Invalid data for input vector of pairs")
        ct = array.ctypes
        ct.array = array
        return ct, c_size_t(len(o) * 2)
    else:
        if(len(o) and len(o[0]) != 2):
            raise Exception("Invalid data for input vector of pairs")
        return ((c_int * 2) * len(o))(*o), c_size_t(len(o) * 2)

def _ivectorstring(o):
    return (c_char_p * len(o))(*(s.encode() for s in o)), c_size_t(len(o))

def _ivectorvectorint(os):
    n = len(os)
    parrays = [_ivectorint(o) for o in os]
    sizes = (c_size_t * n)(*(a[1] for a in parrays))
    arrays = (POINTER(c_int) * n)(*(cast(a[0], POINTER(c_int)) for a in parrays))
    arrays.ref = [a[0] for a in parrays]
    size = c_size_t(n)
    return arrays, sizes, size

def _ivectorvectorsize(os):
    n = len(os)
    parrays = [_ivectorsize(o) for o in os]
    sizes = (c_size_t * n)(*(a[1] for a in parrays))
    arrays = (POINTER(c_size_t) * n)(*(cast(a[0], POINTER(c_size_t)) for a in parrays))
    arrays.ref = [a[0] for a in parrays]
    size = c_size_t(n)
    return arrays, sizes, size

def _ivectorvectordouble(os):
    n = len(os)
    parrays = [_ivectordouble(o) for o in os]
    sizes = (c_size_t * n)(*(a[1] for a in parrays))
    arrays = (POINTER(c_double) * n)(*(cast(a[0], POINTER(c_double)) for a in parrays))
    arrays.ref = [a[0] for a in parrays]
    size = c_size_t(n)
    return arrays, sizes, size

def _iargcargv(o):
    return c_int(len(o)), (c_char_p * len(o))(*(s.encode() for s in o))

# Gmsh Python API begins here

def initialize(argv=[], readConfigFiles=True):
    """
    gmsh.initialize(argv=[], readConfigFiles=True)

    Initialize Gmsh API. This must be called before any call to the other
    functions in the API. If `argc' and `argv' (or just `argv' in Python or
    Julia) are provided, they will be handled in the same way as the command
    line arguments in the Gmsh app. If `readConfigFiles' is set, read system
    Gmsh configuration files (gmshrc and gmsh-options). Initializing the API
    sets the options "General.Terminal" to 1 and "General.AbortOnError" to 2.
    """
    api_argc_, api_argv_ = _iargcargv(argv)
    ierr = c_int()
    lib.gmshInitialize(
        api_argc_, api_argv_,
        c_int(bool(readConfigFiles)),
        byref(ierr))
    if ierr.value != 0:
        raise Exception(logger.getLastError())

def finalize():
    """
    gmsh.finalize()

    Finalize the Gmsh API. This must be called when you are done using the Gmsh
    API.
    """
    ierr = c_int()
    lib.gmshFinalize(
        byref(ierr))
    if oldsig is not None:
        signal.signal(signal.SIGINT, oldsig)
    if ierr.value != 0:
        raise Exception(logger.getLastError())

def open(fileName):
    """
    gmsh.open(fileName)

    Open a file. Equivalent to the `File->Open' menu in the Gmsh app. Handling
    of the file depends on its extension and/or its contents: opening a file
    with model data will create a new model.
    """
    ierr = c_int()
    lib.gmshOpen(
        c_char_p(fileName.encode()),
        byref(ierr))
    if ierr.value != 0:
        raise Exception(logger.getLastError())

def merge(fileName):
    """
    gmsh.merge(fileName)

    Merge a file. Equivalent to the `File->Merge' menu in the Gmsh app.
    Handling of the file depends on its extension and/or its contents. Merging
    a file with model data will add the data to the current model.
    """
    ierr = c_int()
    lib.gmshMerge(
        c_char_p(fileName.encode()),
        byref(ierr))
    if ierr.value != 0:
        raise Exception(logger.getLastError())

def write(fileName):
    """
    gmsh.write(fileName)

    Write a file. The export format is determined by the file extension.
    """
    ierr = c_int()
    lib.gmshWrite(
        c_char_p(fileName.encode()),
        byref(ierr))
    if ierr.value != 0:
        raise Exception(logger.getLastError())

def clear():
    """
    gmsh.clear()

    Clear all loaded models and post-processing data, and add a new empty
    model.
    """
    ierr = c_int()
    lib.gmshClear(
        byref(ierr))
    if ierr.value != 0:
        raise Exception(logger.getLastError())


class option:
    """
    Option handling functions
    """

    @staticmethod
    def setNumber(name, value):
        """
        gmsh.option.setNumber(name, value)

        Set a numerical option to `value'. `name' is of the form "category.option"
        or "category[num].option". Available categories and options are listed in
        the Gmsh reference manual.
        """
        ierr = c_int()
        lib.gmshOptionSetNumber(
            c_char_p(name.encode()),
            c_double(value),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_number = setNumber

    @staticmethod
    def getNumber(name):
        """
        gmsh.option.getNumber(name)

        Get the `value' of a numerical option. `name' is of the form
        "category.option" or "category[num].option". Available categories and
        options are listed in the Gmsh reference manual.

        Return `value'.
        """
        api_value_ = c_double()
        ierr = c_int()
        lib.gmshOptionGetNumber(
            c_char_p(name.encode()),
            byref(api_value_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_value_.value
    get_number = getNumber

    @staticmethod
    def setString(name, value):
        """
        gmsh.option.setString(name, value)

        Set a string option to `value'. `name' is of the form "category.option" or
        "category[num].option". Available categories and options are listed in the
        Gmsh reference manual.
        """
        ierr = c_int()
        lib.gmshOptionSetString(
            c_char_p(name.encode()),
            c_char_p(value.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_string = setString

    @staticmethod
    def getString(name):
        """
        gmsh.option.getString(name)

        Get the `value' of a string option. `name' is of the form "category.option"
        or "category[num].option". Available categories and options are listed in
        the Gmsh reference manual.

        Return `value'.
        """
        api_value_ = c_char_p()
        ierr = c_int()
        lib.gmshOptionGetString(
            c_char_p(name.encode()),
            byref(api_value_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_value_)
    get_string = getString

    @staticmethod
    def setColor(name, r, g, b, a=255):
        """
        gmsh.option.setColor(name, r, g, b, a=255)

        Set a color option to the RGBA value (`r', `g', `b', `a'), where where `r',
        `g', `b' and `a' should be integers between 0 and 255. `name' is of the
        form "category.option" or "category[num].option". Available categories and
        options are listed in the Gmsh reference manual, with the "Color." middle
        string removed.
        """
        ierr = c_int()
        lib.gmshOptionSetColor(
            c_char_p(name.encode()),
            c_int(r),
            c_int(g),
            c_int(b),
            c_int(a),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_color = setColor

    @staticmethod
    def getColor(name):
        """
        gmsh.option.getColor(name)

        Get the `r', `g', `b', `a' value of a color option. `name' is of the form
        "category.option" or "category[num].option". Available categories and
        options are listed in the Gmsh reference manual, with the "Color." middle
        string removed.

        Return `r', `g', `b', `a'.
        """
        api_r_ = c_int()
        api_g_ = c_int()
        api_b_ = c_int()
        api_a_ = c_int()
        ierr = c_int()
        lib.gmshOptionGetColor(
            c_char_p(name.encode()),
            byref(api_r_),
            byref(api_g_),
            byref(api_b_),
            byref(api_a_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_r_.value,
            api_g_.value,
            api_b_.value,
            api_a_.value)
    get_color = getColor


class model:
    """
    Model functions
    """

    @staticmethod
    def add(name):
        """
        gmsh.model.add(name)

        Add a new model, with name `name', and set it as the current model.
        """
        ierr = c_int()
        lib.gmshModelAdd(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def remove():
        """
        gmsh.model.remove()

        Remove the current model.
        """
        ierr = c_int()
        lib.gmshModelRemove(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def list():
        """
        gmsh.model.list()

        List the names of all models.

        Return `names'.
        """
        api_names_, api_names_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshModelList(
            byref(api_names_), byref(api_names_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorstring(api_names_, api_names_n_.value)

    @staticmethod
    def getCurrent():
        """
        gmsh.model.getCurrent()

        Get the name of the current model.

        Return `name'.
        """
        api_name_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetCurrent(
            byref(api_name_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_name_)
    get_current = getCurrent

    @staticmethod
    def setCurrent(name):
        """
        gmsh.model.setCurrent(name)

        Set the current model to the model with name `name'. If several models have
        the same name, select the one that was added first.
        """
        ierr = c_int()
        lib.gmshModelSetCurrent(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_current = setCurrent

    @staticmethod
    def getFileName():
        """
        gmsh.model.getFileName()

        Get the file name (if any) associated with the current model. A file name
        is associated when a model is read from a file on disk.

        Return `fileName'.
        """
        api_fileName_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetFileName(
            byref(api_fileName_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_fileName_)
    get_file_name = getFileName

    @staticmethod
    def setFileName(fileName):
        """
        gmsh.model.setFileName(fileName)

        Set the file name associated with the current model.
        """
        ierr = c_int()
        lib.gmshModelSetFileName(
            c_char_p(fileName.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_file_name = setFileName

    @staticmethod
    def getEntities(dim=-1):
        """
        gmsh.model.getEntities(dim=-1)

        Get all the entities in the current model. If `dim' is >= 0, return only
        the entities of the specified dimension (e.g. points if `dim' == 0). The
        entities are returned as a vector of (dim, tag) integer pairs.

        Return `dimTags'.
        """
        api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetEntities(
            byref(api_dimTags_), byref(api_dimTags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorpair(api_dimTags_, api_dimTags_n_.value)
    get_entities = getEntities

    @staticmethod
    def setEntityName(dim, tag, name):
        """
        gmsh.model.setEntityName(dim, tag, name)

        Set the name of the entity of dimension `dim' and tag `tag'.
        """
        ierr = c_int()
        lib.gmshModelSetEntityName(
            c_int(dim),
            c_int(tag),
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_entity_name = setEntityName

    @staticmethod
    def getEntityName(dim, tag):
        """
        gmsh.model.getEntityName(dim, tag)

        Get the name of the entity of dimension `dim' and tag `tag'.

        Return `name'.
        """
        api_name_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetEntityName(
            c_int(dim),
            c_int(tag),
            byref(api_name_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_name_)
    get_entity_name = getEntityName

    @staticmethod
    def getPhysicalGroups(dim=-1):
        """
        gmsh.model.getPhysicalGroups(dim=-1)

        Get all the physical groups in the current model. If `dim' is >= 0, return
        only the entities of the specified dimension (e.g. physical points if `dim'
        == 0). The entities are returned as a vector of (dim, tag) integer pairs.

        Return `dimTags'.
        """
        api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPhysicalGroups(
            byref(api_dimTags_), byref(api_dimTags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorpair(api_dimTags_, api_dimTags_n_.value)
    get_physical_groups = getPhysicalGroups

    @staticmethod
    def getEntitiesForPhysicalGroup(dim, tag):
        """
        gmsh.model.getEntitiesForPhysicalGroup(dim, tag)

        Get the tags of the model entities making up the physical group of
        dimension `dim' and tag `tag'.

        Return `tags'.
        """
        api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetEntitiesForPhysicalGroup(
            c_int(dim),
            c_int(tag),
            byref(api_tags_), byref(api_tags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorint(api_tags_, api_tags_n_.value)
    get_entities_for_physical_group = getEntitiesForPhysicalGroup

    @staticmethod
    def getPhysicalGroupsForEntity(dim, tag):
        """
        gmsh.model.getPhysicalGroupsForEntity(dim, tag)

        Get the tags of the physical groups (if any) to which the model entity of
        dimension `dim' and tag `tag' belongs.

        Return `physicalTags'.
        """
        api_physicalTags_, api_physicalTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPhysicalGroupsForEntity(
            c_int(dim),
            c_int(tag),
            byref(api_physicalTags_), byref(api_physicalTags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorint(api_physicalTags_, api_physicalTags_n_.value)
    get_physical_groups_for_entity = getPhysicalGroupsForEntity

    @staticmethod
    def addPhysicalGroup(dim, tags, tag=-1):
        """
        gmsh.model.addPhysicalGroup(dim, tags, tag=-1)

        Add a physical group of dimension `dim', grouping the model entities with
        tags `tags'. Return the tag of the physical group, equal to `tag' if `tag'
        is positive, or a new tag if `tag' < 0.

        Return an integer value.
        """
        api_tags_, api_tags_n_ = _ivectorint(tags)
        ierr = c_int()
        api_result_ = lib.gmshModelAddPhysicalGroup(
            c_int(dim),
            api_tags_, api_tags_n_,
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    add_physical_group = addPhysicalGroup

    @staticmethod
    def removePhysicalGroups(dimTags=[]):
        """
        gmsh.model.removePhysicalGroups(dimTags=[])

        Remove the physical groups `dimTags' from the current model. If `dimTags'
        is empty, remove all groups.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelRemovePhysicalGroups(
            api_dimTags_, api_dimTags_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    remove_physical_groups = removePhysicalGroups

    @staticmethod
    def setPhysicalName(dim, tag, name):
        """
        gmsh.model.setPhysicalName(dim, tag, name)

        Set the name of the physical group of dimension `dim' and tag `tag'.
        """
        ierr = c_int()
        lib.gmshModelSetPhysicalName(
            c_int(dim),
            c_int(tag),
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_physical_name = setPhysicalName

    @staticmethod
    def removePhysicalName(name):
        """
        gmsh.model.removePhysicalName(name)

        Remove the physical name `name' from the current model.
        """
        ierr = c_int()
        lib.gmshModelRemovePhysicalName(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    remove_physical_name = removePhysicalName

    @staticmethod
    def getPhysicalName(dim, tag):
        """
        gmsh.model.getPhysicalName(dim, tag)

        Get the name of the physical group of dimension `dim' and tag `tag'.

        Return `name'.
        """
        api_name_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetPhysicalName(
            c_int(dim),
            c_int(tag),
            byref(api_name_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_name_)
    get_physical_name = getPhysicalName

    @staticmethod
    def getBoundary(dimTags, combined=True, oriented=True, recursive=False):
        """
        gmsh.model.getBoundary(dimTags, combined=True, oriented=True, recursive=False)

        Get the boundary of the model entities `dimTags'. Return in `outDimTags'
        the boundary of the individual entities (if `combined' is false) or the
        boundary of the combined geometrical shape formed by all input entities (if
        `combined' is true). Return tags multiplied by the sign of the boundary
        entity if `oriented' is true. Apply the boundary operator recursively down
        to dimension 0 (i.e. to points) if `recursive' is true.

        Return `outDimTags'.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetBoundary(
            api_dimTags_, api_dimTags_n_,
            byref(api_outDimTags_), byref(api_outDimTags_n_),
            c_int(bool(combined)),
            c_int(bool(oriented)),
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
    get_boundary = getBoundary

    @staticmethod
    def getAdjacencies(dim, tag):
        """
        gmsh.model.getAdjacencies(dim, tag)

        Get the upward and downward adjacencies of the model entity of dimension
        `dim' and tag `tag'. The `upward' vector returns the adjacent entities of
        dimension `dim' + 1; the `downward' vector returns the adjacent entities of
        dimension `dim' - 1.

        Return `upward', `downward'.
        """
        api_upward_, api_upward_n_ = POINTER(c_int)(), c_size_t()
        api_downward_, api_downward_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetAdjacencies(
            c_int(dim),
            c_int(tag),
            byref(api_upward_), byref(api_upward_n_),
            byref(api_downward_), byref(api_downward_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ovectorint(api_upward_, api_upward_n_.value),
            _ovectorint(api_downward_, api_downward_n_.value))
    get_adjacencies = getAdjacencies

    @staticmethod
    def getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim=-1):
        """
        gmsh.model.getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim=-1)

        Get the model entities in the bounding box defined by the two points
        (`xmin', `ymin', `zmin') and (`xmax', `ymax', `zmax'). If `dim' is >= 0,
        return only the entities of the specified dimension (e.g. points if `dim'
        == 0).

        Return `tags'.
        """
        api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetEntitiesInBoundingBox(
            c_double(xmin),
            c_double(ymin),
            c_double(zmin),
            c_double(xmax),
            c_double(ymax),
            c_double(zmax),
            byref(api_tags_), byref(api_tags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorpair(api_tags_, api_tags_n_.value)
    get_entities_in_bounding_box = getEntitiesInBoundingBox

    @staticmethod
    def getBoundingBox(dim, tag):
        """
        gmsh.model.getBoundingBox(dim, tag)

        Get the bounding box (`xmin', `ymin', `zmin'), (`xmax', `ymax', `zmax') of
        the model entity of dimension `dim' and tag `tag'. If `dim' and `tag' are
        negative, get the bounding box of the whole model.

        Return `xmin', `ymin', `zmin', `xmax', `ymax', `zmax'.
        """
        api_xmin_ = c_double()
        api_ymin_ = c_double()
        api_zmin_ = c_double()
        api_xmax_ = c_double()
        api_ymax_ = c_double()
        api_zmax_ = c_double()
        ierr = c_int()
        lib.gmshModelGetBoundingBox(
            c_int(dim),
            c_int(tag),
            byref(api_xmin_),
            byref(api_ymin_),
            byref(api_zmin_),
            byref(api_xmax_),
            byref(api_ymax_),
            byref(api_zmax_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_xmin_.value,
            api_ymin_.value,
            api_zmin_.value,
            api_xmax_.value,
            api_ymax_.value,
            api_zmax_.value)
    get_bounding_box = getBoundingBox

    @staticmethod
    def getDimension():
        """
        gmsh.model.getDimension()

        Get the geometrical dimension of the current model.

        Return an integer value.
        """
        ierr = c_int()
        api_result_ = lib.gmshModelGetDimension(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    get_dimension = getDimension

    @staticmethod
    def addDiscreteEntity(dim, tag=-1, boundary=[]):
        """
        gmsh.model.addDiscreteEntity(dim, tag=-1, boundary=[])

        Add a discrete model entity (defined by a mesh) of dimension `dim' in the
        current model. Return the tag of the new discrete entity, equal to `tag' if
        `tag' is positive, or a new tag if `tag' < 0. `boundary' specifies the tags
        of the entities on the boundary of the discrete entity, if any. Specifying
        `boundary' allows Gmsh to construct the topology of the overall model.

        Return an integer value.
        """
        api_boundary_, api_boundary_n_ = _ivectorint(boundary)
        ierr = c_int()
        api_result_ = lib.gmshModelAddDiscreteEntity(
            c_int(dim),
            c_int(tag),
            api_boundary_, api_boundary_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    add_discrete_entity = addDiscreteEntity

    @staticmethod
    def removeEntities(dimTags, recursive=False):
        """
        gmsh.model.removeEntities(dimTags, recursive=False)

        Remove the entities `dimTags' of the current model. If `recursive' is true,
        remove all the entities on their boundaries, down to dimension 0.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelRemoveEntities(
            api_dimTags_, api_dimTags_n_,
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    remove_entities = removeEntities

    @staticmethod
    def removeEntityName(name):
        """
        gmsh.model.removeEntityName(name)

        Remove the entity name `name' from the current model.
        """
        ierr = c_int()
        lib.gmshModelRemoveEntityName(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    remove_entity_name = removeEntityName

    @staticmethod
    def getType(dim, tag):
        """
        gmsh.model.getType(dim, tag)

        Get the type of the entity of dimension `dim' and tag `tag'.

        Return `entityType'.
        """
        api_entityType_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetType(
            c_int(dim),
            c_int(tag),
            byref(api_entityType_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_entityType_)
    get_type = getType

    @staticmethod
    def getParent(dim, tag):
        """
        gmsh.model.getParent(dim, tag)

        In a partitioned model, get the parent of the entity of dimension `dim' and
        tag `tag', i.e. from which the entity is a part of, if any. `parentDim' and
        `parentTag' are set to -1 if the entity has no parent.

        Return `parentDim', `parentTag'.
        """
        api_parentDim_ = c_int()
        api_parentTag_ = c_int()
        ierr = c_int()
        lib.gmshModelGetParent(
            c_int(dim),
            c_int(tag),
            byref(api_parentDim_),
            byref(api_parentTag_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_parentDim_.value,
            api_parentTag_.value)
    get_parent = getParent

    @staticmethod
    def getPartitions(dim, tag):
        """
        gmsh.model.getPartitions(dim, tag)

        In a partitioned model, return the tags of the partition(s) to which the
        entity belongs.

        Return `partitions'.
        """
        api_partitions_, api_partitions_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPartitions(
            c_int(dim),
            c_int(tag),
            byref(api_partitions_), byref(api_partitions_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorint(api_partitions_, api_partitions_n_.value)
    get_partitions = getPartitions

    @staticmethod
    def getValue(dim, tag, parametricCoord):
        """
        gmsh.model.getValue(dim, tag, parametricCoord)

        Evaluate the parametrization of the entity of dimension `dim' and tag `tag'
        at the parametric coordinates `parametricCoord'. Only valid for `dim' equal
        to 0 (with empty `parametricCoord'), 1 (with `parametricCoord' containing
        parametric coordinates on the curve) or 2 (with `parametricCoord'
        containing pairs of u, v parametric coordinates on the surface,
        concatenated: [p1u, p1v, p2u, ...]). Return triplets of x, y, z coordinates
        in `coord', concatenated: [p1x, p1y, p1z, p2x, ...].

        Return `coord'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetValue(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_coord_), byref(api_coord_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_coord_, api_coord_n_.value)
    get_value = getValue

    @staticmethod
    def getDerivative(dim, tag, parametricCoord):
        """
        gmsh.model.getDerivative(dim, tag, parametricCoord)

        Evaluate the derivative of the parametrization of the entity of dimension
        `dim' and tag `tag' at the parametric coordinates `parametricCoord'. Only
        valid for `dim' equal to 1 (with `parametricCoord' containing parametric
        coordinates on the curve) or 2 (with `parametricCoord' containing pairs of
        u, v parametric coordinates on the surface, concatenated: [p1u, p1v, p2u,
        ...]). For `dim' equal to 1 return the x, y, z components of the derivative
        with respect to u [d1ux, d1uy, d1uz, d2ux, ...]; for `dim' equal to 2
        return the x, y, z components of the derivative with respect to u and v:
        [d1ux, d1uy, d1uz, d1vx, d1vy, d1vz, d2ux, ...].

        Return `derivatives'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_derivatives_, api_derivatives_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetDerivative(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_derivatives_), byref(api_derivatives_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_derivatives_, api_derivatives_n_.value)
    get_derivative = getDerivative

    @staticmethod
    def getSecondDerivative(dim, tag, parametricCoord):
        """
        gmsh.model.getSecondDerivative(dim, tag, parametricCoord)

        Evaluate the second derivative of the parametrization of the entity of
        dimension `dim' and tag `tag' at the parametric coordinates
        `parametricCoord'. Only valid for `dim' equal to 1 (with `parametricCoord'
        containing parametric coordinates on the curve) or 2 (with
        `parametricCoord' containing pairs of u, v parametric coordinates on the
        surface, concatenated: [p1u, p1v, p2u, ...]). For `dim' equal to 1 return
        the x, y, z components of the second derivative with respect to u [d1uux,
        d1uuy, d1uuz, d2uux, ...]; for `dim' equal to 2 return the x, y, z
        components of the second derivative with respect to u and v, and the mixed
        derivative with respect to u and v: [d1uux, d1uuy, d1uuz, d1vvx, d1vvy,
        d1vvz, d1uvx, d1uvy, d1uvz, d2uux, ...].

        Return `derivatives'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_derivatives_, api_derivatives_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetSecondDerivative(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_derivatives_), byref(api_derivatives_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_derivatives_, api_derivatives_n_.value)
    get_second_derivative = getSecondDerivative

    @staticmethod
    def getCurvature(dim, tag, parametricCoord):
        """
        gmsh.model.getCurvature(dim, tag, parametricCoord)

        Evaluate the (maximum) curvature of the entity of dimension `dim' and tag
        `tag' at the parametric coordinates `parametricCoord'. Only valid for `dim'
        equal to 1 (with `parametricCoord' containing parametric coordinates on the
        curve) or 2 (with `parametricCoord' containing pairs of u, v parametric
        coordinates on the surface, concatenated: [p1u, p1v, p2u, ...]).

        Return `curvatures'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_curvatures_, api_curvatures_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetCurvature(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_curvatures_), byref(api_curvatures_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_curvatures_, api_curvatures_n_.value)
    get_curvature = getCurvature

    @staticmethod
    def getPrincipalCurvatures(tag, parametricCoord):
        """
        gmsh.model.getPrincipalCurvatures(tag, parametricCoord)

        Evaluate the principal curvatures of the surface with tag `tag' at the
        parametric coordinates `parametricCoord', as well as their respective
        directions. `parametricCoord' are given by pair of u and v coordinates,
        concatenated: [p1u, p1v, p2u, ...].

        Return `curvatureMax', `curvatureMin', `directionMax', `directionMin'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_curvatureMax_, api_curvatureMax_n_ = POINTER(c_double)(), c_size_t()
        api_curvatureMin_, api_curvatureMin_n_ = POINTER(c_double)(), c_size_t()
        api_directionMax_, api_directionMax_n_ = POINTER(c_double)(), c_size_t()
        api_directionMin_, api_directionMin_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPrincipalCurvatures(
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_curvatureMax_), byref(api_curvatureMax_n_),
            byref(api_curvatureMin_), byref(api_curvatureMin_n_),
            byref(api_directionMax_), byref(api_directionMax_n_),
            byref(api_directionMin_), byref(api_directionMin_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ovectordouble(api_curvatureMax_, api_curvatureMax_n_.value),
            _ovectordouble(api_curvatureMin_, api_curvatureMin_n_.value),
            _ovectordouble(api_directionMax_, api_directionMax_n_.value),
            _ovectordouble(api_directionMin_, api_directionMin_n_.value))
    get_principal_curvatures = getPrincipalCurvatures

    @staticmethod
    def getNormal(tag, parametricCoord):
        """
        gmsh.model.getNormal(tag, parametricCoord)

        Get the normal to the surface with tag `tag' at the parametric coordinates
        `parametricCoord'. `parametricCoord' are given by pairs of u and v
        coordinates, concatenated: [p1u, p1v, p2u, ...]. `normals' are returned as
        triplets of x, y, z components, concatenated: [n1x, n1y, n1z, n2x, ...].

        Return `normals'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_normals_, api_normals_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetNormal(
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_normals_), byref(api_normals_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_normals_, api_normals_n_.value)
    get_normal = getNormal

    @staticmethod
    def getParametrization(dim, tag, coord):
        """
        gmsh.model.getParametrization(dim, tag, coord)

        Get the parametric coordinates `parametricCoord' for the points `coord' on
        the entity of dimension `dim' and tag `tag'. `coord' are given as triplets
        of x, y, z coordinates, concatenated: [p1x, p1y, p1z, p2x, ...].
        `parametricCoord' returns the parametric coordinates t on the curve (if
        `dim' = 1) or pairs of u and v coordinates concatenated on the surface (if
        `dim' = 2), i.e. [p1t, p2t, ...] or [p1u, p1v, p2u, ...].

        Return `parametricCoord'.
        """
        api_coord_, api_coord_n_ = _ivectordouble(coord)
        api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetParametrization(
            c_int(dim),
            c_int(tag),
            api_coord_, api_coord_n_,
            byref(api_parametricCoord_), byref(api_parametricCoord_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value)
    get_parametrization = getParametrization

    @staticmethod
    def getParametrizationBounds(dim, tag):
        """
        gmsh.model.getParametrizationBounds(dim, tag)

        Get the `min' and `max' bounds of the parametric coordinates for the entity
        of dimension `dim' and tag `tag'.

        Return `min', `max'.
        """
        api_min_, api_min_n_ = POINTER(c_double)(), c_size_t()
        api_max_, api_max_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetParametrizationBounds(
            c_int(dim),
            c_int(tag),
            byref(api_min_), byref(api_min_n_),
            byref(api_max_), byref(api_max_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ovectordouble(api_min_, api_min_n_.value),
            _ovectordouble(api_max_, api_max_n_.value))
    get_parametrization_bounds = getParametrizationBounds

    @staticmethod
    def isInside(dim, tag, parametricCoord):
        """
        gmsh.model.isInside(dim, tag, parametricCoord)

        Check if the parametric coordinates provided in `parametricCoord'
        correspond to points inside the entitiy of dimension `dim' and tag `tag',
        and return the number of points inside. This feature is only available for
        a subset of curves and surfaces, depending on the underyling geometrical
        representation.

        Return an integer value.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        ierr = c_int()
        api_result_ = lib.gmshModelIsInside(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    is_inside = isInside

    @staticmethod
    def getClosestPoint(dim, tag, coord):
        """
        gmsh.model.getClosestPoint(dim, tag, coord)

        Get the points `closestCoord' on the entity of dimension `dim' and tag
        `tag' to the points `coord', by orthogonal projection. `coord' and
        `closestCoord' are given as triplets of x, y, z coordinates, concatenated:
        [p1x, p1y, p1z, p2x, ...]. `parametricCoord' returns the parametric
        coordinates t on the curve (if `dim' = 1) or pairs of u and v coordinates
        concatenated on the surface (if `dim' = 2), i.e. [p1t, p2t, ...] or [p1u,
        p1v, p2u, ...].

        Return `closestCoord', `parametricCoord'.
        """
        api_coord_, api_coord_n_ = _ivectordouble(coord)
        api_closestCoord_, api_closestCoord_n_ = POINTER(c_double)(), c_size_t()
        api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetClosestPoint(
            c_int(dim),
            c_int(tag),
            api_coord_, api_coord_n_,
            byref(api_closestCoord_), byref(api_closestCoord_n_),
            byref(api_parametricCoord_), byref(api_parametricCoord_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ovectordouble(api_closestCoord_, api_closestCoord_n_.value),
            _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))
    get_closest_point = getClosestPoint

    @staticmethod
    def reparametrizeOnSurface(dim, tag, parametricCoord, surfaceTag, which=0):
        """
        gmsh.model.reparametrizeOnSurface(dim, tag, parametricCoord, surfaceTag, which=0)

        Reparametrize the boundary entity (point or curve, i.e. with `dim' == 0 or
        `dim' == 1) of tag `tag' on the surface `surfaceTag'. If `dim' == 1,
        reparametrize all the points corresponding to the parametric coordinates
        `parametricCoord'. Multiple matches in case of periodic surfaces can be
        selected with `which'. This feature is only available for a subset of
        entities, depending on the underyling geometrical representation.

        Return `surfaceParametricCoord'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_surfaceParametricCoord_, api_surfaceParametricCoord_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelReparametrizeOnSurface(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            c_int(surfaceTag),
            byref(api_surfaceParametricCoord_), byref(api_surfaceParametricCoord_n_),
            c_int(which),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_surfaceParametricCoord_, api_surfaceParametricCoord_n_.value)
    reparametrize_on_surface = reparametrizeOnSurface

    @staticmethod
    def setVisibility(dimTags, value, recursive=False):
        """
        gmsh.model.setVisibility(dimTags, value, recursive=False)

        Set the visibility of the model entities `dimTags' to `value'. Apply the
        visibility setting recursively if `recursive' is true.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelSetVisibility(
            api_dimTags_, api_dimTags_n_,
            c_int(value),
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_visibility = setVisibility

    @staticmethod
    def getVisibility(dim, tag):
        """
        gmsh.model.getVisibility(dim, tag)

        Get the visibility of the model entity of dimension `dim' and tag `tag'.

        Return `value'.
        """
        api_value_ = c_int()
        ierr = c_int()
        lib.gmshModelGetVisibility(
            c_int(dim),
            c_int(tag),
            byref(api_value_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_value_.value
    get_visibility = getVisibility

    @staticmethod
    def setVisibilityPerWindow(value, windowIndex=0):
        """
        gmsh.model.setVisibilityPerWindow(value, windowIndex=0)

        Set the global visibility of the model per window to `value', where
        `windowIndex' identifies the window in the window list.
        """
        ierr = c_int()
        lib.gmshModelSetVisibilityPerWindow(
            c_int(value),
            c_int(windowIndex),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_visibility_per_window = setVisibilityPerWindow

    @staticmethod
    def setColor(dimTags, r, g, b, a=255, recursive=False):
        """
        gmsh.model.setColor(dimTags, r, g, b, a=255, recursive=False)

        Set the color of the model entities `dimTags' to the RGBA value (`r', `g',
        `b', `a'), where `r', `g', `b' and `a' should be integers between 0 and
        255. Apply the color setting recursively if `recursive' is true.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelSetColor(
            api_dimTags_, api_dimTags_n_,
            c_int(r),
            c_int(g),
            c_int(b),
            c_int(a),
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_color = setColor

    @staticmethod
    def getColor(dim, tag):
        """
        gmsh.model.getColor(dim, tag)

        Get the color of the model entity of dimension `dim' and tag `tag'.

        Return `r', `g', `b', `a'.
        """
        api_r_ = c_int()
        api_g_ = c_int()
        api_b_ = c_int()
        api_a_ = c_int()
        ierr = c_int()
        lib.gmshModelGetColor(
            c_int(dim),
            c_int(tag),
            byref(api_r_),
            byref(api_g_),
            byref(api_b_),
            byref(api_a_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_r_.value,
            api_g_.value,
            api_b_.value,
            api_a_.value)
    get_color = getColor

    @staticmethod
    def setCoordinates(tag, x, y, z):
        """
        gmsh.model.setCoordinates(tag, x, y, z)

        Set the `x', `y', `z' coordinates of a geometrical point.
        """
        ierr = c_int()
        lib.gmshModelSetCoordinates(
            c_int(tag),
            c_double(x),
            c_double(y),
            c_double(z),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_coordinates = setCoordinates


    class mesh:
        """
        Mesh functions
        """

        @staticmethod
        def generate(dim=3):
            """
            gmsh.model.mesh.generate(dim=3)

            Generate a mesh of the current model, up to dimension `dim' (0, 1, 2 or 3).
            """
            ierr = c_int()
            lib.gmshModelMeshGenerate(
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def partition(numPart):
            """
            gmsh.model.mesh.partition(numPart)

            Partition the mesh of the current model into `numPart' partitions.
            """
            ierr = c_int()
            lib.gmshModelMeshPartition(
                c_int(numPart),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def unpartition():
            """
            gmsh.model.mesh.unpartition()

            Unpartition the mesh of the current model.
            """
            ierr = c_int()
            lib.gmshModelMeshUnpartition(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def optimize(method, force=False, niter=1, dimTags=[]):
            """
            gmsh.model.mesh.optimize(method, force=False, niter=1, dimTags=[])

            Optimize the mesh of the current model using `method' (empty for default
            tetrahedral mesh optimizer, "Netgen" for Netgen optimizer, "HighOrder" for
            direct high-order mesh optimizer, "HighOrderElastic" for high-order elastic
            smoother, "HighOrderFastCurving" for fast curving algorithm, "Laplace2D"
            for Laplace smoothing, "Relocate2D" and "Relocate3D" for node relocation).
            If `force' is set apply the optimization also to discrete entities. If
            `dimTags' is given, only apply the optimizer to the given entities.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshOptimize(
                c_char_p(method.encode()),
                c_int(bool(force)),
                c_int(niter),
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def recombine():
            """
            gmsh.model.mesh.recombine()

            Recombine the mesh of the current model.
            """
            ierr = c_int()
            lib.gmshModelMeshRecombine(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def refine():
            """
            gmsh.model.mesh.refine()

            Refine the mesh of the current model by uniformly splitting the elements.
            """
            ierr = c_int()
            lib.gmshModelMeshRefine(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def setOrder(order):
            """
            gmsh.model.mesh.setOrder(order)

            Set the order of the elements in the mesh of the current model to `order'.
            """
            ierr = c_int()
            lib.gmshModelMeshSetOrder(
                c_int(order),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_order = setOrder

        @staticmethod
        def getLastEntityError():
            """
            gmsh.model.mesh.getLastEntityError()

            Get the last entities (if any) where a meshing error occurred. Currently
            only populated by the new 3D meshing algorithms.

            Return `dimTags'.
            """
            api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetLastEntityError(
                byref(api_dimTags_), byref(api_dimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_dimTags_, api_dimTags_n_.value)
        get_last_entity_error = getLastEntityError

        @staticmethod
        def getLastNodeError():
            """
            gmsh.model.mesh.getLastNodeError()

            Get the last nodes (if any) where a meshing error occurred. Currently only
            populated by the new 3D meshing algorithms.

            Return `nodeTags'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetLastNodeError(
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorsize(api_nodeTags_, api_nodeTags_n_.value)
        get_last_node_error = getLastNodeError

        @staticmethod
        def clear(dimTags=[]):
            """
            gmsh.model.mesh.clear(dimTags=[])

            Clear the mesh, i.e. delete all the nodes and elements, for the entities
            `dimTags'. if `dimTags' is empty, clear the whole mesh. Note that the mesh
            of an entity can only be cleared if this entity is not on the boundary of
            another entity with a non-empty mesh.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshClear(
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def getNodes(dim=-1, tag=-1, includeBoundary=False, returnParametricCoord=True):
            """
            gmsh.model.mesh.getNodes(dim=-1, tag=-1, includeBoundary=False, returnParametricCoord=True)

            Get the nodes classified on the entity of dimension `dim' and tag `tag'. If
            `tag' < 0, get the nodes for all entities of dimension `dim'. If `dim' and
            `tag' are negative, get all the nodes in the mesh. `nodeTags' contains the
            node tags (their unique, strictly positive identification numbers). `coord'
            is a vector of length 3 times the length of `nodeTags' that contains the x,
            y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. If
            `dim' >= 0 and `returnParamtricCoord' is set, `parametricCoord' contains
            the parametric coordinates ([u1, u2, ...] or [u1, v1, u2, ...]) of the
            nodes, if available. The length of `parametricCoord' can be 0 or `dim'
            times the length of `nodeTags'. If `includeBoundary' is set, also return
            the nodes classified on the boundary of the entity (which will be
            reparametrized on the entity if `dim' >= 0 in order to compute their
            parametric coordinates).

            Return `nodeTags', `coord', `parametricCoord'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNodes(
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(api_parametricCoord_), byref(api_parametricCoord_n_),
                c_int(dim),
                c_int(tag),
                c_int(bool(includeBoundary)),
                c_int(bool(returnParametricCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value),
                _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))
        get_nodes = getNodes

        @staticmethod
        def getNodesByElementType(elementType, tag=-1, returnParametricCoord=True):
            """
            gmsh.model.mesh.getNodesByElementType(elementType, tag=-1, returnParametricCoord=True)

            Get the nodes classified on the entity of tag `tag', for all the elements
            of type `elementType'. The other arguments are treated as in `getNodes'.

            Return `nodeTags', `coord', `parametricCoord'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNodesByElementType(
                c_int(elementType),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(api_parametricCoord_), byref(api_parametricCoord_n_),
                c_int(tag),
                c_int(bool(returnParametricCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value),
                _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))
        get_nodes_by_element_type = getNodesByElementType

        @staticmethod
        def getNode(nodeTag):
            """
            gmsh.model.mesh.getNode(nodeTag)

            Get the coordinates and the parametric coordinates (if any) of the node
            with tag `tag'. This function relies on an internal cache (a vector in case
            of dense node numbering, a map otherwise); for large meshes accessing nodes
            in bulk is often preferable.

            Return `coord', `parametricCoord'.
            """
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNode(
                c_size_t(nodeTag),
                byref(api_coord_), byref(api_coord_n_),
                byref(api_parametricCoord_), byref(api_parametricCoord_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectordouble(api_coord_, api_coord_n_.value),
                _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))
        get_node = getNode

        @staticmethod
        def setNode(nodeTag, coord, parametricCoord):
            """
            gmsh.model.mesh.setNode(nodeTag, coord, parametricCoord)

            Set the coordinates and the parametric coordinates (if any) of the node
            with tag `tag'. This function relies on an internal cache (a vector in case
            of dense node numbering, a map otherwise); for large meshes accessing nodes
            in bulk is often preferable.
            """
            api_coord_, api_coord_n_ = _ivectordouble(coord)
            api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
            ierr = c_int()
            lib.gmshModelMeshSetNode(
                c_size_t(nodeTag),
                api_coord_, api_coord_n_,
                api_parametricCoord_, api_parametricCoord_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_node = setNode

        @staticmethod
        def rebuildNodeCache(onlyIfNecessary=True):
            """
            gmsh.model.mesh.rebuildNodeCache(onlyIfNecessary=True)

            Rebuild the node cache.
            """
            ierr = c_int()
            lib.gmshModelMeshRebuildNodeCache(
                c_int(bool(onlyIfNecessary)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        rebuild_node_cache = rebuildNodeCache

        @staticmethod
        def rebuildElementCache(onlyIfNecessary=True):
            """
            gmsh.model.mesh.rebuildElementCache(onlyIfNecessary=True)

            Rebuild the element cache.
            """
            ierr = c_int()
            lib.gmshModelMeshRebuildElementCache(
                c_int(bool(onlyIfNecessary)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        rebuild_element_cache = rebuildElementCache

        @staticmethod
        def getNodesForPhysicalGroup(dim, tag):
            """
            gmsh.model.mesh.getNodesForPhysicalGroup(dim, tag)

            Get the nodes from all the elements belonging to the physical group of
            dimension `dim' and tag `tag'. `nodeTags' contains the node tags; `coord'
            is a vector of length 3 times the length of `nodeTags' that contains the x,
            y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...].

            Return `nodeTags', `coord'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNodesForPhysicalGroup(
                c_int(dim),
                c_int(tag),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))
        get_nodes_for_physical_group = getNodesForPhysicalGroup

        @staticmethod
        def addNodes(dim, tag, nodeTags, coord, parametricCoord=[]):
            """
            gmsh.model.mesh.addNodes(dim, tag, nodeTags, coord, parametricCoord=[])

            Add nodes classified on the model entity of dimension `dim' and tag `tag'.
            `nodeTags' contains the node tags (their unique, strictly positive
            identification numbers). `coord' is a vector of length 3 times the length
            of `nodeTags' that contains the x, y, z coordinates of the nodes,
            concatenated: [n1x, n1y, n1z, n2x, ...]. The optional `parametricCoord'
            vector contains the parametric coordinates of the nodes, if any. The length
            of `parametricCoord' can be 0 or `dim' times the length of `nodeTags'. If
            the `nodeTags' vector is empty, new tags are automatically assigned to the
            nodes.
            """
            api_nodeTags_, api_nodeTags_n_ = _ivectorsize(nodeTags)
            api_coord_, api_coord_n_ = _ivectordouble(coord)
            api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
            ierr = c_int()
            lib.gmshModelMeshAddNodes(
                c_int(dim),
                c_int(tag),
                api_nodeTags_, api_nodeTags_n_,
                api_coord_, api_coord_n_,
                api_parametricCoord_, api_parametricCoord_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        add_nodes = addNodes

        @staticmethod
        def reclassifyNodes():
            """
            gmsh.model.mesh.reclassifyNodes()

            Reclassify all nodes on their associated model entity, based on the
            elements. Can be used when importing nodes in bulk (e.g. by associating
            them all to a single volume), to reclassify them correctly on model
            surfaces, curves, etc. after the elements have been set.
            """
            ierr = c_int()
            lib.gmshModelMeshReclassifyNodes(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        reclassify_nodes = reclassifyNodes

        @staticmethod
        def relocateNodes(dim=-1, tag=-1):
            """
            gmsh.model.mesh.relocateNodes(dim=-1, tag=-1)

            Relocate the nodes classified on the entity of dimension `dim' and tag
            `tag' using their parametric coordinates. If `tag' < 0, relocate the nodes
            for all entities of dimension `dim'. If `dim' and `tag' are negative,
            relocate all the nodes in the mesh.
            """
            ierr = c_int()
            lib.gmshModelMeshRelocateNodes(
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        relocate_nodes = relocateNodes

        @staticmethod
        def getElements(dim=-1, tag=-1):
            """
            gmsh.model.mesh.getElements(dim=-1, tag=-1)

            Get the elements classified on the entity of dimension `dim' and tag `tag'.
            If `tag' < 0, get the elements for all entities of dimension `dim'. If
            `dim' and `tag' are negative, get all the elements in the mesh.
            `elementTypes' contains the MSH types of the elements (e.g. `2' for 3-node
            triangles: see `getElementProperties' to obtain the properties for a given
            element type). `elementTags' is a vector of the same length as
            `elementTypes'; each entry is a vector containing the tags (unique,
            strictly positive identifiers) of the elements of the corresponding type.
            `nodeTags' is also a vector of the same length as `elementTypes'; each
            entry is a vector of length equal to the number of elements of the given
            type times the number N of nodes for this type of element, that contains
            the node tags of all the elements of the given type, concatenated: [e1n1,
            e1n2, ..., e1nN, e2n1, ...].

            Return `elementTypes', `elementTags', `nodeTags'.
            """
            api_elementTypes_, api_elementTypes_n_ = POINTER(c_int)(), c_size_t()
            api_elementTags_, api_elementTags_n_, api_elementTags_nn_ = POINTER(POINTER(c_size_t))(), POINTER(c_size_t)(), c_size_t()
            api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_ = POINTER(POINTER(c_size_t))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElements(
                byref(api_elementTypes_), byref(api_elementTypes_n_),
                byref(api_elementTags_), byref(api_elementTags_n_), byref(api_elementTags_nn_),
                byref(api_nodeTags_), byref(api_nodeTags_n_), byref(api_nodeTags_nn_),
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorint(api_elementTypes_, api_elementTypes_n_.value),
                _ovectorvectorsize(api_elementTags_, api_elementTags_n_, api_elementTags_nn_),
                _ovectorvectorsize(api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_))
        get_elements = getElements

        @staticmethod
        def getElement(elementTag):
            """
            gmsh.model.mesh.getElement(elementTag)

            Get the type and node tags of the element with tag `tag'. This function
            relies on an internal cache (a vector in case of dense element numbering, a
            map otherwise); for large meshes accessing elements in bulk is often
            preferable.

            Return `elementType', `nodeTags'.
            """
            api_elementType_ = c_int()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElement(
                c_size_t(elementTag),
                byref(api_elementType_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_elementType_.value,
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value))
        get_element = getElement

        @staticmethod
        def getElementByCoordinates(x, y, z, dim=-1, strict=False):
            """
            gmsh.model.mesh.getElementByCoordinates(x, y, z, dim=-1, strict=False)

            Search the mesh for an element located at coordinates (`x', `y', `z'). This
            function performs a search in a spatial octree. If an element is found,
            return its tag, type and node tags, as well as the local coordinates (`u',
            `v', `w') within the reference element corresponding to search location. If
            `dim' is >= 0, only search for elements of the given dimension. If `strict'
            is not set, use a tolerance to find elements near the search location.

            Return `elementTag', `elementType', `nodeTags', `u', `v', `w'.
            """
            api_elementTag_ = c_size_t()
            api_elementType_ = c_int()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_u_ = c_double()
            api_v_ = c_double()
            api_w_ = c_double()
            ierr = c_int()
            lib.gmshModelMeshGetElementByCoordinates(
                c_double(x),
                c_double(y),
                c_double(z),
                byref(api_elementTag_),
                byref(api_elementType_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_u_),
                byref(api_v_),
                byref(api_w_),
                c_int(dim),
                c_int(bool(strict)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_elementTag_.value,
                api_elementType_.value,
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                api_u_.value,
                api_v_.value,
                api_w_.value)
        get_element_by_coordinates = getElementByCoordinates

        @staticmethod
        def getElementsByCoordinates(x, y, z, dim=-1, strict=False):
            """
            gmsh.model.mesh.getElementsByCoordinates(x, y, z, dim=-1, strict=False)

            Search the mesh for element(s) located at coordinates (`x', `y', `z'). This
            function performs a search in a spatial octree. Return the tags of all
            found elements in `elementTags'. Additional information about the elements
            can be accessed through `getElement' and `getLocalCoordinatesInElement'. If
            `dim' is >= 0, only search for elements of the given dimension. If `strict'
            is not set, use a tolerance to find elements near the search location.

            Return `elementTags'.
            """
            api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementsByCoordinates(
                c_double(x),
                c_double(y),
                c_double(z),
                byref(api_elementTags_), byref(api_elementTags_n_),
                c_int(dim),
                c_int(bool(strict)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorsize(api_elementTags_, api_elementTags_n_.value)
        get_elements_by_coordinates = getElementsByCoordinates

        @staticmethod
        def getLocalCoordinatesInElement(elementTag, x, y, z):
            """
            gmsh.model.mesh.getLocalCoordinatesInElement(elementTag, x, y, z)

            Return the local coordinates (`u', `v', `w') within the element
            `elementTag' corresponding to the model coordinates (`x', `y', `z'). This
            function relies on an internal cache (a vector in case of dense element
            numbering, a map otherwise); for large meshes accessing elements in bulk is
            often preferable.

            Return `u', `v', `w'.
            """
            api_u_ = c_double()
            api_v_ = c_double()
            api_w_ = c_double()
            ierr = c_int()
            lib.gmshModelMeshGetLocalCoordinatesInElement(
                c_size_t(elementTag),
                c_double(x),
                c_double(y),
                c_double(z),
                byref(api_u_),
                byref(api_v_),
                byref(api_w_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_u_.value,
                api_v_.value,
                api_w_.value)
        get_local_coordinates_in_element = getLocalCoordinatesInElement

        @staticmethod
        def getElementTypes(dim=-1, tag=-1):
            """
            gmsh.model.mesh.getElementTypes(dim=-1, tag=-1)

            Get the types of elements in the entity of dimension `dim' and tag `tag'.
            If `tag' < 0, get the types for all entities of dimension `dim'. If `dim'
            and `tag' are negative, get all the types in the mesh.

            Return `elementTypes'.
            """
            api_elementTypes_, api_elementTypes_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementTypes(
                byref(api_elementTypes_), byref(api_elementTypes_n_),
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorint(api_elementTypes_, api_elementTypes_n_.value)
        get_element_types = getElementTypes

        @staticmethod
        def getElementType(familyName, order, serendip=False):
            """
            gmsh.model.mesh.getElementType(familyName, order, serendip=False)

            Return an element type given its family name `familyName' ("Point", "Line",
            "Triangle", "Quadrangle", "Tetrahedron", "Pyramid", "Prism", "Hexahedron")
            and polynomial order `order'. If `serendip' is true, return the
            corresponding serendip element type (element without interior nodes).

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelMeshGetElementType(
                c_char_p(familyName.encode()),
                c_int(order),
                c_int(bool(serendip)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        get_element_type = getElementType

        @staticmethod
        def getElementProperties(elementType):
            """
            gmsh.model.mesh.getElementProperties(elementType)

            Get the properties of an element of type `elementType': its name
            (`elementName'), dimension (`dim'), order (`order'), number of nodes
            (`numNodes'), local coordinates of the nodes in the reference element
            (`localNodeCoord' vector, of length `dim' times `numNodes') and number of
            primary (first order) nodes (`numPrimaryNodes').

            Return `elementName', `dim', `order', `numNodes', `localNodeCoord', `numPrimaryNodes'.
            """
            api_elementName_ = c_char_p()
            api_dim_ = c_int()
            api_order_ = c_int()
            api_numNodes_ = c_int()
            api_localNodeCoord_, api_localNodeCoord_n_ = POINTER(c_double)(), c_size_t()
            api_numPrimaryNodes_ = c_int()
            ierr = c_int()
            lib.gmshModelMeshGetElementProperties(
                c_int(elementType),
                byref(api_elementName_),
                byref(api_dim_),
                byref(api_order_),
                byref(api_numNodes_),
                byref(api_localNodeCoord_), byref(api_localNodeCoord_n_),
                byref(api_numPrimaryNodes_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ostring(api_elementName_),
                api_dim_.value,
                api_order_.value,
                api_numNodes_.value,
                _ovectordouble(api_localNodeCoord_, api_localNodeCoord_n_.value),
                api_numPrimaryNodes_.value)
        get_element_properties = getElementProperties

        @staticmethod
        def getElementsByType(elementType, tag=-1, task=0, numTasks=1):
            """
            gmsh.model.mesh.getElementsByType(elementType, tag=-1, task=0, numTasks=1)

            Get the elements of type `elementType' classified on the entity of tag
            `tag'. If `tag' < 0, get the elements for all entities. `elementTags' is a
            vector containing the tags (unique, strictly positive identifiers) of the
            elements of the corresponding type. `nodeTags' is a vector of length equal
            to the number of elements of the given type times the number N of nodes for
            this type of element, that contains the node tags of all the elements of
            the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...]. If
            `numTasks' > 1, only compute and return the part of the data indexed by
            `task'.

            Return `elementTags', `nodeTags'.
            """
            api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementsByType(
                c_int(elementType),
                byref(api_elementTags_), byref(api_elementTags_n_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                c_int(tag),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_elementTags_, api_elementTags_n_.value),
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value))
        get_elements_by_type = getElementsByType

        @staticmethod
        def addElements(dim, tag, elementTypes, elementTags, nodeTags):
            """
            gmsh.model.mesh.addElements(dim, tag, elementTypes, elementTags, nodeTags)

            Add elements classified on the entity of dimension `dim' and tag `tag'.
            `types' contains the MSH types of the elements (e.g. `2' for 3-node
            triangles: see the Gmsh reference manual). `elementTags' is a vector of the
            same length as `types'; each entry is a vector containing the tags (unique,
            strictly positive identifiers) of the elements of the corresponding type.
            `nodeTags' is also a vector of the same length as `types'; each entry is a
            vector of length equal to the number of elements of the given type times
            the number N of nodes per element, that contains the node tags of all the
            elements of the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1,
            ...].
            """
            api_elementTypes_, api_elementTypes_n_ = _ivectorint(elementTypes)
            api_elementTags_, api_elementTags_n_, api_elementTags_nn_ = _ivectorvectorsize(elementTags)
            api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_ = _ivectorvectorsize(nodeTags)
            ierr = c_int()
            lib.gmshModelMeshAddElements(
                c_int(dim),
                c_int(tag),
                api_elementTypes_, api_elementTypes_n_,
                api_elementTags_, api_elementTags_n_, api_elementTags_nn_,
                api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        add_elements = addElements

        @staticmethod
        def addElementsByType(tag, elementType, elementTags, nodeTags):
            """
            gmsh.model.mesh.addElementsByType(tag, elementType, elementTags, nodeTags)

            Add elements of type `elementType' classified on the entity of tag `tag'.
            `elementTags' contains the tags (unique, strictly positive identifiers) of
            the elements of the corresponding type. `nodeTags' is a vector of length
            equal to the number of elements times the number N of nodes per element,
            that contains the node tags of all the elements, concatenated: [e1n1, e1n2,
            ..., e1nN, e2n1, ...]. If the `elementTag' vector is empty, new tags are
            automatically assigned to the elements.
            """
            api_elementTags_, api_elementTags_n_ = _ivectorsize(elementTags)
            api_nodeTags_, api_nodeTags_n_ = _ivectorsize(nodeTags)
            ierr = c_int()
            lib.gmshModelMeshAddElementsByType(
                c_int(tag),
                c_int(elementType),
                api_elementTags_, api_elementTags_n_,
                api_nodeTags_, api_nodeTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        add_elements_by_type = addElementsByType

        @staticmethod
        def getIntegrationPoints(elementType, integrationType):
            """
            gmsh.model.mesh.getIntegrationPoints(elementType, integrationType)

            Get the numerical quadrature information for the given element type
            `elementType' and integration rule `integrationType' (e.g. "Gauss4" for a
            Gauss quadrature suited for integrating 4th order polynomials).
            `localCoord' contains the u, v, w coordinates of the G integration points
            in the reference element: [g1u, g1v, g1w, ..., gGu, gGv, gGw]. `weights'
            contains the associated weights: [g1q, ..., gGq].

            Return `localCoord', `weights'.
            """
            api_localCoord_, api_localCoord_n_ = POINTER(c_double)(), c_size_t()
            api_weights_, api_weights_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetIntegrationPoints(
                c_int(elementType),
                c_char_p(integrationType.encode()),
                byref(api_localCoord_), byref(api_localCoord_n_),
                byref(api_weights_), byref(api_weights_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectordouble(api_localCoord_, api_localCoord_n_.value),
                _ovectordouble(api_weights_, api_weights_n_.value))
        get_integration_points = getIntegrationPoints

        @staticmethod
        def getJacobians(elementType, localCoord, tag=-1, task=0, numTasks=1):
            """
            gmsh.model.mesh.getJacobians(elementType, localCoord, tag=-1, task=0, numTasks=1)

            Get the Jacobians of all the elements of type `elementType' classified on
            the entity of tag `tag', at the G evaluation points `localCoord' given as
            concatenated triplets of coordinates in the reference element [g1u, g1v,
            g1w, ..., gGu, gGv, gGw]. Data is returned by element, with elements in the
            same order as in `getElements' and `getElementsByType'. `jacobians'
            contains for each element the 9 entries of the 3x3 Jacobian matrix at each
            evaluation point. The matrix is returned by column: [e1g1Jxu, e1g1Jyu,
            e1g1Jzu, e1g1Jxv, ..., e1g1Jzw, e1g2Jxu, ..., e1gGJzw, e2g1Jxu, ...], with
            Jxu=dx/du, Jyu=dy/du, etc. `determinants' contains for each element the
            determinant of the Jacobian matrix at each evaluation point: [e1g1, e1g2,
            ... e1gG, e2g1, ...]. `coord' contains for each element the x, y, z
            coordinates of the evaluation points. If `tag' < 0, get the Jacobian data
            for all entities. If `numTasks' > 1, only compute and return the part of
            the data indexed by `task'.

            Return `jacobians', `determinants', `coord'.
            """
            api_localCoord_, api_localCoord_n_ = _ivectordouble(localCoord)
            api_jacobians_, api_jacobians_n_ = POINTER(c_double)(), c_size_t()
            api_determinants_, api_determinants_n_ = POINTER(c_double)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetJacobians(
                c_int(elementType),
                api_localCoord_, api_localCoord_n_,
                byref(api_jacobians_), byref(api_jacobians_n_),
                byref(api_determinants_), byref(api_determinants_n_),
                byref(api_coord_), byref(api_coord_n_),
                c_int(tag),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectordouble(api_jacobians_, api_jacobians_n_.value),
                _ovectordouble(api_determinants_, api_determinants_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))
        get_jacobians = getJacobians

        @staticmethod
        def getJacobian(elementTag, localCoord):
            """
            gmsh.model.mesh.getJacobian(elementTag, localCoord)

            Get the Jacobian for a single element `elementTag', at the G evaluation
            points `localCoord' given as concatenated triplets of coordinates in the
            reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]. `jacobians' contains
            the 9 entries of the 3x3 Jacobian matrix at each evaluation point. The
            matrix is returned by column: [e1g1Jxu, e1g1Jyu, e1g1Jzu, e1g1Jxv, ...,
            e1g1Jzw, e1g2Jxu, ..., e1gGJzw, e2g1Jxu, ...], with Jxu=dx/du, Jyu=dy/du,
            etc. `determinants' contains the determinant of the Jacobian matrix at each
            evaluation point. `coord' contains the x, y, z coordinates of the
            evaluation points. This function relies on an internal cache (a vector in
            case of dense element numbering, a map otherwise); for large meshes
            accessing Jacobians in bulk is often preferable.

            Return `jacobians', `determinants', `coord'.
            """
            api_localCoord_, api_localCoord_n_ = _ivectordouble(localCoord)
            api_jacobians_, api_jacobians_n_ = POINTER(c_double)(), c_size_t()
            api_determinants_, api_determinants_n_ = POINTER(c_double)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetJacobian(
                c_size_t(elementTag),
                api_localCoord_, api_localCoord_n_,
                byref(api_jacobians_), byref(api_jacobians_n_),
                byref(api_determinants_), byref(api_determinants_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectordouble(api_jacobians_, api_jacobians_n_.value),
                _ovectordouble(api_determinants_, api_determinants_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))
        get_jacobian = getJacobian

        @staticmethod
        def getBasisFunctions(elementType, localCoord, functionSpaceType, wantedOrientations=[]):
            """
            gmsh.model.mesh.getBasisFunctions(elementType, localCoord, functionSpaceType, wantedOrientations=[])

            Get the basis functions of the element of type `elementType' at the
            evaluation points `localCoord' (given as concatenated triplets of
            coordinates in the reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]),
            for the function space `functionSpaceType' (e.g. "Lagrange" or
            "GradLagrange" for Lagrange basis functions or their gradient, in the u, v,
            w coordinates of the reference element; or "H1Legendre3" or
            "GradH1Legendre3" for 3rd order hierarchical H1 Legendre functions).
            `numComponents' returns the number C of components of a basis function.
            `basisFunctions' returns the value of the N basis functions at the
            evaluation points, i.e. [g1f1, g1f2, ..., g1fN, g2f1, ...] when C == 1 or
            [g1f1u, g1f1v, g1f1w, g1f2u, ..., g1fNw, g2f1u, ...] when C == 3. For basis
            functions that depend on the orientation of the elements, all values for
            the first orientation are returned first, followed by values for the
            second, etc. `numOrientations' returns the overall number of orientations.
            If `wantedOrientations' is not empty, only return the values for the
            desired orientation indices.

            Return `numComponents', `basisFunctions', `numOrientations'.
            """
            api_localCoord_, api_localCoord_n_ = _ivectordouble(localCoord)
            api_numComponents_ = c_int()
            api_basisFunctions_, api_basisFunctions_n_ = POINTER(c_double)(), c_size_t()
            api_numOrientations_ = c_int()
            api_wantedOrientations_, api_wantedOrientations_n_ = _ivectorint(wantedOrientations)
            ierr = c_int()
            lib.gmshModelMeshGetBasisFunctions(
                c_int(elementType),
                api_localCoord_, api_localCoord_n_,
                c_char_p(functionSpaceType.encode()),
                byref(api_numComponents_),
                byref(api_basisFunctions_), byref(api_basisFunctions_n_),
                byref(api_numOrientations_),
                api_wantedOrientations_, api_wantedOrientations_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_numComponents_.value,
                _ovectordouble(api_basisFunctions_, api_basisFunctions_n_.value),
                api_numOrientations_.value)
        get_basis_functions = getBasisFunctions

        @staticmethod
        def getBasisFunctionsOrientationForElements(elementType, functionSpaceType, tag=-1, task=0, numTasks=1):
            """
            gmsh.model.mesh.getBasisFunctionsOrientationForElements(elementType, functionSpaceType, tag=-1, task=0, numTasks=1)

            Get the orientation index of the elements of type `elementType' in the
            entity of tag `tag'. The arguments have the same meaning as in
            `getBasisFunctions'. `basisFunctionsOrientation' is a vector giving for
            each element the orientation index in the values returned by
            `getBasisFunctions'. For Lagrange basis functions the call is superfluous
            as it will return a vector of zeros.

            Return `basisFunctionsOrientation'.
            """
            api_basisFunctionsOrientation_, api_basisFunctionsOrientation_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetBasisFunctionsOrientationForElements(
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(api_basisFunctionsOrientation_), byref(api_basisFunctionsOrientation_n_),
                c_int(tag),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorint(api_basisFunctionsOrientation_, api_basisFunctionsOrientation_n_.value)
        get_basis_functions_orientation_for_elements = getBasisFunctionsOrientationForElements

        @staticmethod
        def getBasisFunctionsOrientationForElement(elementTag, functionSpaceType):
            """
            gmsh.model.mesh.getBasisFunctionsOrientationForElement(elementTag, functionSpaceType)

            Get the orientation of a single element `elementTag'.

            Return `basisFunctionsOrientation'.
            """
            api_basisFunctionsOrientation_ = c_int()
            ierr = c_int()
            lib.gmshModelMeshGetBasisFunctionsOrientationForElement(
                c_size_t(elementTag),
                c_char_p(functionSpaceType.encode()),
                byref(api_basisFunctionsOrientation_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_basisFunctionsOrientation_.value
        get_basis_functions_orientation_for_element = getBasisFunctionsOrientationForElement

        @staticmethod
        def getNumberOfOrientations(elementType, functionSpaceType):
            """
            gmsh.model.mesh.getNumberOfOrientations(elementType, functionSpaceType)

            Get the number of possible orientations for elements of type `elementType'
            and function space named `functionSpaceType'.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelMeshGetNumberOfOrientations(
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        get_number_of_orientations = getNumberOfOrientations

        @staticmethod
        def getEdges(nodeTags):
            """
            gmsh.model.mesh.getEdges(nodeTags)

            Get the global unique mesh edge identifiers `edgeTags' and orientations
            `edgeOrientation' for an input list of node tag pairs defining these edges,
            concatenated in the vector `nodeTags'.

            Return `edgeTags', `edgeOrientations'.
            """
            api_nodeTags_, api_nodeTags_n_ = _ivectorsize(nodeTags)
            api_edgeTags_, api_edgeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_edgeOrientations_, api_edgeOrientations_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetEdges(
                api_nodeTags_, api_nodeTags_n_,
                byref(api_edgeTags_), byref(api_edgeTags_n_),
                byref(api_edgeOrientations_), byref(api_edgeOrientations_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_edgeTags_, api_edgeTags_n_.value),
                _ovectorint(api_edgeOrientations_, api_edgeOrientations_n_.value))
        get_edges = getEdges

        @staticmethod
        def getFaces(faceType, nodeTags):
            """
            gmsh.model.mesh.getFaces(faceType, nodeTags)

            Get the global unique mesh face identifiers `faceTags' and orientations
            `faceOrientations' for an input list of node tag triplets (if `faceType' ==
            3) or quadruplets (if `faceType' == 4) defining these faces, concatenated
            in the vector `nodeTags'.

            Return `faceTags', `faceOrientations'.
            """
            api_nodeTags_, api_nodeTags_n_ = _ivectorsize(nodeTags)
            api_faceTags_, api_faceTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_faceOrientations_, api_faceOrientations_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetFaces(
                c_int(faceType),
                api_nodeTags_, api_nodeTags_n_,
                byref(api_faceTags_), byref(api_faceTags_n_),
                byref(api_faceOrientations_), byref(api_faceOrientations_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_faceTags_, api_faceTags_n_.value),
                _ovectorint(api_faceOrientations_, api_faceOrientations_n_.value))
        get_faces = getFaces

        @staticmethod
        def createEdges(dimTags=[]):
            """
            gmsh.model.mesh.createEdges(dimTags=[])

            Create unique mesh edges for the entities `dimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshCreateEdges(
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        create_edges = createEdges

        @staticmethod
        def createFaces(dimTags=[]):
            """
            gmsh.model.mesh.createFaces(dimTags=[])

            Create unique mesh faces for the entities `dimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshCreateFaces(
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        create_faces = createFaces

        @staticmethod
        def getLocalMultipliersForHcurl0(elementType, tag=-1):
            """
            gmsh.model.mesh.getLocalMultipliersForHcurl0(elementType, tag=-1)

            Get the local multipliers (to guarantee H(curl)-conformity) of the order 0
            H(curl) basis functions. Warning: this is an experimental feature and will
            probably change in a future release.

            Return `localMultipliers'.
            """
            api_localMultipliers_, api_localMultipliers_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetLocalMultipliersForHcurl0(
                c_int(elementType),
                byref(api_localMultipliers_), byref(api_localMultipliers_n_),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorint(api_localMultipliers_, api_localMultipliers_n_.value)
        get_local_multipliers_for_hcurl0 = getLocalMultipliersForHcurl0

        @staticmethod
        def getKeysForElements(elementType, functionSpaceType, tag=-1, returnCoord=True):
            """
            gmsh.model.mesh.getKeysForElements(elementType, functionSpaceType, tag=-1, returnCoord=True)

            Generate the `keys' for the elements of type `elementType' in the entity of
            tag `tag', for the `functionSpaceType' function space. Each key uniquely
            identifies a basis function in the function space. If `returnCoord' is set,
            the `coord' vector contains the x, y, z coordinates locating basis
            functions for sorting purposes. Warning: this is an experimental feature
            and will probably change in a future release.

            Return `keys', `coord'.
            """
            api_keys_, api_keys_n_ = POINTER(c_int)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetKeysForElements(
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(api_keys_), byref(api_keys_n_),
                byref(api_coord_), byref(api_coord_n_),
                c_int(tag),
                c_int(bool(returnCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorpair(api_keys_, api_keys_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))
        get_keys_for_elements = getKeysForElements

        @staticmethod
        def getKeysForElement(elementTag, functionSpaceType, returnCoord=True):
            """
            gmsh.model.mesh.getKeysForElement(elementTag, functionSpaceType, returnCoord=True)

            Get the keys for a single element `elementTag'.

            Return `keys', `coord'.
            """
            api_keys_, api_keys_n_ = POINTER(c_int)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetKeysForElement(
                c_size_t(elementTag),
                c_char_p(functionSpaceType.encode()),
                byref(api_keys_), byref(api_keys_n_),
                byref(api_coord_), byref(api_coord_n_),
                c_int(bool(returnCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorpair(api_keys_, api_keys_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))
        get_keys_for_element = getKeysForElement

        @staticmethod
        def getNumberOfKeysForElements(elementType, functionSpaceType):
            """
            gmsh.model.mesh.getNumberOfKeysForElements(elementType, functionSpaceType)

            Get the number of keys by elements of type `elementType' for function space
            named `functionSpaceType'.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelMeshGetNumberOfKeysForElements(
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        get_number_of_keys_for_elements = getNumberOfKeysForElements

        @staticmethod
        def getInformationForElements(keys, elementType, functionSpaceType):
            """
            gmsh.model.mesh.getInformationForElements(keys, elementType, functionSpaceType)

            Get information about the `keys'. `infoKeys' returns information about the
            functions associated with the `keys'. `infoKeys[0].first' describes the
            type of function (0 for  vertex function, 1 for edge function, 2 for face
            function and 3 for bubble function). `infoKeys[0].second' gives the order
            of the function associated with the key. Warning: this is an experimental
            feature and will probably change in a future release.

            Return `infoKeys'.
            """
            api_keys_, api_keys_n_ = _ivectorpair(keys)
            api_infoKeys_, api_infoKeys_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetInformationForElements(
                api_keys_, api_keys_n_,
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(api_infoKeys_), byref(api_infoKeys_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_infoKeys_, api_infoKeys_n_.value)
        get_information_for_elements = getInformationForElements

        @staticmethod
        def getBarycenters(elementType, tag, fast, primary, task=0, numTasks=1):
            """
            gmsh.model.mesh.getBarycenters(elementType, tag, fast, primary, task=0, numTasks=1)

            Get the barycenters of all elements of type `elementType' classified on the
            entity of tag `tag'. If `primary' is set, only the primary nodes of the
            elements are taken into account for the barycenter calculation. If `fast'
            is set, the function returns the sum of the primary node coordinates
            (without normalizing by the number of nodes). If `tag' < 0, get the
            barycenters for all entities. If `numTasks' > 1, only compute and return
            the part of the data indexed by `task'.

            Return `barycenters'.
            """
            api_barycenters_, api_barycenters_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetBarycenters(
                c_int(elementType),
                c_int(tag),
                c_int(bool(fast)),
                c_int(bool(primary)),
                byref(api_barycenters_), byref(api_barycenters_n_),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectordouble(api_barycenters_, api_barycenters_n_.value)
        get_barycenters = getBarycenters

        @staticmethod
        def getElementEdgeNodes(elementType, tag=-1, primary=False, task=0, numTasks=1):
            """
            gmsh.model.mesh.getElementEdgeNodes(elementType, tag=-1, primary=False, task=0, numTasks=1)

            Get the nodes on the edges of all elements of type `elementType' classified
            on the entity of tag `tag'. `nodeTags' contains the node tags of the edges
            for all the elements: [e1a1n1, e1a1n2, e1a2n1, ...]. Data is returned by
            element, with elements in the same order as in `getElements' and
            `getElementsByType'. If `primary' is set, only the primary (begin/end)
            nodes of the edges are returned. If `tag' < 0, get the edge nodes for all
            entities. If `numTasks' > 1, only compute and return the part of the data
            indexed by `task'.

            Return `nodeTags'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementEdgeNodes(
                c_int(elementType),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                c_int(tag),
                c_int(bool(primary)),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorsize(api_nodeTags_, api_nodeTags_n_.value)
        get_element_edge_nodes = getElementEdgeNodes

        @staticmethod
        def getElementFaceNodes(elementType, faceType, tag=-1, primary=False, task=0, numTasks=1):
            """
            gmsh.model.mesh.getElementFaceNodes(elementType, faceType, tag=-1, primary=False, task=0, numTasks=1)

            Get the nodes on the faces of type `faceType' (3 for triangular faces, 4
            for quadrangular faces) of all elements of type `elementType' classified on
            the entity of tag `tag'. `nodeTags' contains the node tags of the faces for
            all elements: [e1f1n1, ..., e1f1nFaceType, e1f2n1, ...]. Data is returned
            by element, with elements in the same order as in `getElements' and
            `getElementsByType'. If `primary' is set, only the primary (corner) nodes
            of the faces are returned. If `tag' < 0, get the face nodes for all
            entities. If `numTasks' > 1, only compute and return the part of the data
            indexed by `task'.

            Return `nodeTags'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementFaceNodes(
                c_int(elementType),
                c_int(faceType),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                c_int(tag),
                c_int(bool(primary)),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorsize(api_nodeTags_, api_nodeTags_n_.value)
        get_element_face_nodes = getElementFaceNodes

        @staticmethod
        def getGhostElements(dim, tag):
            """
            gmsh.model.mesh.getGhostElements(dim, tag)

            Get the ghost elements `elementTags' and their associated `partitions'
            stored in the ghost entity of dimension `dim' and tag `tag'.

            Return `elementTags', `partitions'.
            """
            api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_partitions_, api_partitions_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetGhostElements(
                c_int(dim),
                c_int(tag),
                byref(api_elementTags_), byref(api_elementTags_n_),
                byref(api_partitions_), byref(api_partitions_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorsize(api_elementTags_, api_elementTags_n_.value),
                _ovectorint(api_partitions_, api_partitions_n_.value))
        get_ghost_elements = getGhostElements

        @staticmethod
        def setSize(dimTags, size):
            """
            gmsh.model.mesh.setSize(dimTags, size)

            Set a mesh size constraint on the model entities `dimTags'. Currently only
            entities of dimension 0 (points) are handled.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshSetSize(
                api_dimTags_, api_dimTags_n_,
                c_double(size),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_size = setSize

        @staticmethod
        def setSizeAtParametricPoints(dim, tag, parametricCoord, sizes):
            """
            gmsh.model.mesh.setSizeAtParametricPoints(dim, tag, parametricCoord, sizes)

            Set mesh size constraints at the given parametric points `parametricCoord'
            on the model entity of dimension `dim' and tag `tag'. Currently only
            entities of dimension 1 (lines) are handled.
            """
            api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
            api_sizes_, api_sizes_n_ = _ivectordouble(sizes)
            ierr = c_int()
            lib.gmshModelMeshSetSizeAtParametricPoints(
                c_int(dim),
                c_int(tag),
                api_parametricCoord_, api_parametricCoord_n_,
                api_sizes_, api_sizes_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_size_at_parametric_points = setSizeAtParametricPoints

        @staticmethod
        def setSizeCallback(callback):
            """
            gmsh.model.mesh.setSizeCallback(callback)

            Set a global mesh size callback. The callback should take 5 arguments
            (`dim', `tag', `x', `y' and `z') and return the value of the mesh size at
            coordinates (`x', `y', `z').
            """
            global api_callback_type_
            api_callback_type_ = CFUNCTYPE(c_double, c_int, c_int, c_double, c_double, c_double, c_void_p)
            global api_callback_
            api_callback_ = api_callback_type_(lambda dim, tag, x, y, z, _ : callback(dim, tag, x, y, z))
            ierr = c_int()
            lib.gmshModelMeshSetSizeCallback(
                api_callback_, None,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_size_callback = setSizeCallback

        @staticmethod
        def removeSizeCallback():
            """
            gmsh.model.mesh.removeSizeCallback()

            Remove the global mesh size callback.
            """
            ierr = c_int()
            lib.gmshModelMeshRemoveSizeCallback(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_size_callback = removeSizeCallback

        @staticmethod
        def setTransfiniteCurve(tag, numNodes, meshType="Progression", coef=1.):
            """
            gmsh.model.mesh.setTransfiniteCurve(tag, numNodes, meshType="Progression", coef=1.)

            Set a transfinite meshing constraint on the curve `tag', with `numNodes'
            nodes distributed according to `meshType' and `coef'. Currently supported
            types are "Progression" (geometrical progression with power `coef'), "Bump"
            (refinement toward both extremities of the curve) and "Beta" (beta law).
            """
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteCurve(
                c_int(tag),
                c_int(numNodes),
                c_char_p(meshType.encode()),
                c_double(coef),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_transfinite_curve = setTransfiniteCurve

        @staticmethod
        def setTransfiniteSurface(tag, arrangement="Left", cornerTags=[]):
            """
            gmsh.model.mesh.setTransfiniteSurface(tag, arrangement="Left", cornerTags=[])

            Set a transfinite meshing constraint on the surface `tag'. `arrangement'
            describes the arrangement of the triangles when the surface is not flagged
            as recombined: currently supported values are "Left", "Right",
            "AlternateLeft" and "AlternateRight". `cornerTags' can be used to specify
            the (3 or 4) corners of the transfinite interpolation explicitly;
            specifying the corners explicitly is mandatory if the surface has more that
            3 or 4 points on its boundary.
            """
            api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteSurface(
                c_int(tag),
                c_char_p(arrangement.encode()),
                api_cornerTags_, api_cornerTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_transfinite_surface = setTransfiniteSurface

        @staticmethod
        def setTransfiniteVolume(tag, cornerTags=[]):
            """
            gmsh.model.mesh.setTransfiniteVolume(tag, cornerTags=[])

            Set a transfinite meshing constraint on the surface `tag'. `cornerTags' can
            be used to specify the (6 or 8) corners of the transfinite interpolation
            explicitly.
            """
            api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteVolume(
                c_int(tag),
                api_cornerTags_, api_cornerTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_transfinite_volume = setTransfiniteVolume

        @staticmethod
        def setTransfiniteAutomatic(dimTags=[], cornerAngle=2.35, recombine=True):
            """
            gmsh.model.mesh.setTransfiniteAutomatic(dimTags=[], cornerAngle=2.35, recombine=True)

            Set transfinite meshing constraints on the model entities in `dimTag'.
            Transfinite meshing constraints are added to the curves of the quadrangular
            surfaces and to the faces of 6-sided volumes. Quadragular faces with a
            corner angle superior to `cornerAngle' (in radians) are ignored. The number
            of points is automatically determined from the sizing constraints. If
            `dimTag' is empty, the constraints are applied to all entities in the
            model. If `recombine' is true, the recombine flag is automatically set on
            the transfinite surfaces.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteAutomatic(
                api_dimTags_, api_dimTags_n_,
                c_double(cornerAngle),
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_transfinite_automatic = setTransfiniteAutomatic

        @staticmethod
        def setRecombine(dim, tag):
            """
            gmsh.model.mesh.setRecombine(dim, tag)

            Set a recombination meshing constraint on the model entity of dimension
            `dim' and tag `tag'. Currently only entities of dimension 2 (to recombine
            triangles into quadrangles) are supported.
            """
            ierr = c_int()
            lib.gmshModelMeshSetRecombine(
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_recombine = setRecombine

        @staticmethod
        def setSmoothing(dim, tag, val):
            """
            gmsh.model.mesh.setSmoothing(dim, tag, val)

            Set a smoothing meshing constraint on the model entity of dimension `dim'
            and tag `tag'. `val' iterations of a Laplace smoother are applied.
            """
            ierr = c_int()
            lib.gmshModelMeshSetSmoothing(
                c_int(dim),
                c_int(tag),
                c_int(val),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_smoothing = setSmoothing

        @staticmethod
        def setReverse(dim, tag, val=True):
            """
            gmsh.model.mesh.setReverse(dim, tag, val=True)

            Set a reverse meshing constraint on the model entity of dimension `dim' and
            tag `tag'. If `val' is true, the mesh orientation will be reversed with
            respect to the natural mesh orientation (i.e. the orientation consistent
            with the orientation of the geometry). If `val' is false, the mesh is left
            as-is.
            """
            ierr = c_int()
            lib.gmshModelMeshSetReverse(
                c_int(dim),
                c_int(tag),
                c_int(bool(val)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_reverse = setReverse

        @staticmethod
        def setAlgorithm(dim, tag, val):
            """
            gmsh.model.mesh.setAlgorithm(dim, tag, val)

            Set the meshing algorithm on the model entity of dimension `dim' and tag
            `tag'. Currently only supported for `dim' == 2.
            """
            ierr = c_int()
            lib.gmshModelMeshSetAlgorithm(
                c_int(dim),
                c_int(tag),
                c_int(val),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_algorithm = setAlgorithm

        @staticmethod
        def setSizeFromBoundary(dim, tag, val):
            """
            gmsh.model.mesh.setSizeFromBoundary(dim, tag, val)

            Force the mesh size to be extended from the boundary, or not, for the model
            entity of dimension `dim' and tag `tag'. Currently only supported for `dim'
            == 2.
            """
            ierr = c_int()
            lib.gmshModelMeshSetSizeFromBoundary(
                c_int(dim),
                c_int(tag),
                c_int(val),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_size_from_boundary = setSizeFromBoundary

        @staticmethod
        def setCompound(dim, tags):
            """
            gmsh.model.mesh.setCompound(dim, tags)

            Set a compound meshing constraint on the model entities of dimension `dim'
            and tags `tags'. During meshing, compound entities are treated as a single
            discrete entity, which is automatically reparametrized.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            ierr = c_int()
            lib.gmshModelMeshSetCompound(
                c_int(dim),
                api_tags_, api_tags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_compound = setCompound

        @staticmethod
        def setOutwardOrientation(tag):
            """
            gmsh.model.mesh.setOutwardOrientation(tag)

            Set meshing constraints on the bounding surfaces of the volume of tag `tag'
            so that all surfaces are oriented with outward pointing normals. Currently
            only available with the OpenCASCADE kernel, as it relies on the STL
            triangulation.
            """
            ierr = c_int()
            lib.gmshModelMeshSetOutwardOrientation(
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_outward_orientation = setOutwardOrientation

        @staticmethod
        def removeConstraints(dimTags=[]):
            """
            gmsh.model.mesh.removeConstraints(dimTags=[])

            Remove all meshing constraints from the model entities `dimTags'. If
            `dimTags' is empty, remove all constraings.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshRemoveConstraints(
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_constraints = removeConstraints

        @staticmethod
        def embed(dim, tags, inDim, inTag):
            """
            gmsh.model.mesh.embed(dim, tags, inDim, inTag)

            Embed the model entities of dimension `dim' and tags `tags' in the
            (`inDim', `inTag') model entity. The dimension `dim' can 0, 1 or 2 and must
            be strictly smaller than `inDim', which must be either 2 or 3. The embedded
            entities should not intersect each other or be part of the boundary of the
            entity `inTag', whose mesh will conform to the mesh of the embedded
            entities. With the OpenCASCADE kernel, if the `fragment' operation is
            applied to entities of different dimensions, the lower dimensional entities
            will be automatically embedded in the higher dimensional entities if they
            are not on their boundary.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            ierr = c_int()
            lib.gmshModelMeshEmbed(
                c_int(dim),
                api_tags_, api_tags_n_,
                c_int(inDim),
                c_int(inTag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def removeEmbedded(dimTags, dim=-1):
            """
            gmsh.model.mesh.removeEmbedded(dimTags, dim=-1)

            Remove embedded entities from the model entities `dimTags'. if `dim' is >=
            0, only remove embedded entities of the given dimension (e.g. embedded
            points if `dim' == 0).
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshRemoveEmbedded(
                api_dimTags_, api_dimTags_n_,
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_embedded = removeEmbedded

        @staticmethod
        def getEmbedded(dim, tag):
            """
            gmsh.model.mesh.getEmbedded(dim, tag)

            Get the entities (if any) embedded in the model entity of dimension `dim'
            and tag `tag'.

            Return `dimTags'.
            """
            api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetEmbedded(
                c_int(dim),
                c_int(tag),
                byref(api_dimTags_), byref(api_dimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_dimTags_, api_dimTags_n_.value)
        get_embedded = getEmbedded

        @staticmethod
        def reorderElements(elementType, tag, ordering):
            """
            gmsh.model.mesh.reorderElements(elementType, tag, ordering)

            Reorder the elements of type `elementType' classified on the entity of tag
            `tag' according to `ordering'.
            """
            api_ordering_, api_ordering_n_ = _ivectorsize(ordering)
            ierr = c_int()
            lib.gmshModelMeshReorderElements(
                c_int(elementType),
                c_int(tag),
                api_ordering_, api_ordering_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        reorder_elements = reorderElements

        @staticmethod
        def renumberNodes():
            """
            gmsh.model.mesh.renumberNodes()

            Renumber the node tags in a continuous sequence.
            """
            ierr = c_int()
            lib.gmshModelMeshRenumberNodes(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        renumber_nodes = renumberNodes

        @staticmethod
        def renumberElements():
            """
            gmsh.model.mesh.renumberElements()

            Renumber the element tags in a continuous sequence.
            """
            ierr = c_int()
            lib.gmshModelMeshRenumberElements(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        renumber_elements = renumberElements

        @staticmethod
        def setPeriodic(dim, tags, tagsMaster, affineTransform):
            """
            gmsh.model.mesh.setPeriodic(dim, tags, tagsMaster, affineTransform)

            Set the meshes of the entities of dimension `dim' and tag `tags' as
            periodic copies of the meshes of entities `tagsMaster', using the affine
            transformation specified in `affineTransformation' (16 entries of a 4x4
            matrix, by row). If used after meshing, generate the periodic node
            correspondence information assuming the meshes of entities `tags'
            effectively match the meshes of entities `tagsMaster' (useful for
            structured and extruded meshes). Currently only available for @code{dim} ==
            1 and @code{dim} == 2.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            api_tagsMaster_, api_tagsMaster_n_ = _ivectorint(tagsMaster)
            api_affineTransform_, api_affineTransform_n_ = _ivectordouble(affineTransform)
            ierr = c_int()
            lib.gmshModelMeshSetPeriodic(
                c_int(dim),
                api_tags_, api_tags_n_,
                api_tagsMaster_, api_tagsMaster_n_,
                api_affineTransform_, api_affineTransform_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_periodic = setPeriodic

        @staticmethod
        def getPeriodicNodes(dim, tag, includeHighOrderNodes=False):
            """
            gmsh.model.mesh.getPeriodicNodes(dim, tag, includeHighOrderNodes=False)

            Get the master entity `tagMaster', the node tags `nodeTags' and their
            corresponding master node tags `nodeTagsMaster', and the affine transform
            `affineTransform' for the entity of dimension `dim' and tag `tag'. If
            `includeHighOrderNodes' is set, include high-order nodes in the returned
            data.

            Return `tagMaster', `nodeTags', `nodeTagsMaster', `affineTransform'.
            """
            api_tagMaster_ = c_int()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_nodeTagsMaster_, api_nodeTagsMaster_n_ = POINTER(c_size_t)(), c_size_t()
            api_affineTransform_, api_affineTransform_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetPeriodicNodes(
                c_int(dim),
                c_int(tag),
                byref(api_tagMaster_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_nodeTagsMaster_), byref(api_nodeTagsMaster_n_),
                byref(api_affineTransform_), byref(api_affineTransform_n_),
                c_int(bool(includeHighOrderNodes)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_tagMaster_.value,
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectorsize(api_nodeTagsMaster_, api_nodeTagsMaster_n_.value),
                _ovectordouble(api_affineTransform_, api_affineTransform_n_.value))
        get_periodic_nodes = getPeriodicNodes

        @staticmethod
        def removeDuplicateNodes():
            """
            gmsh.model.mesh.removeDuplicateNodes()

            Remove duplicate nodes in the mesh of the current model.
            """
            ierr = c_int()
            lib.gmshModelMeshRemoveDuplicateNodes(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_duplicate_nodes = removeDuplicateNodes

        @staticmethod
        def splitQuadrangles(quality=1., tag=-1):
            """
            gmsh.model.mesh.splitQuadrangles(quality=1., tag=-1)

            Split (into two triangles) all quadrangles in surface `tag' whose quality
            is lower than `quality'. If `tag' < 0, split quadrangles in all surfaces.
            """
            ierr = c_int()
            lib.gmshModelMeshSplitQuadrangles(
                c_double(quality),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        split_quadrangles = splitQuadrangles

        @staticmethod
        def classifySurfaces(angle, boundary=True, forReparametrization=False, curveAngle=pi, exportDiscrete=True):
            """
            gmsh.model.mesh.classifySurfaces(angle, boundary=True, forReparametrization=False, curveAngle=pi, exportDiscrete=True)

            Classify ("color") the surface mesh based on the angle threshold `angle'
            (in radians), and create new discrete surfaces, curves and points
            accordingly. If `boundary' is set, also create discrete curves on the
            boundary if the surface is open. If `forReparametrization' is set, create
            edges and surfaces that can be reparametrized using a single map. If
            `curveAngle' is less than Pi, also force curves to be split according to
            `curveAngle'. If `exportDiscrete' is set, clear any built-in CAD kernel
            entities and export the discrete entities in the built-in CAD kernel.
            """
            ierr = c_int()
            lib.gmshModelMeshClassifySurfaces(
                c_double(angle),
                c_int(bool(boundary)),
                c_int(bool(forReparametrization)),
                c_double(curveAngle),
                c_int(bool(exportDiscrete)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        classify_surfaces = classifySurfaces

        @staticmethod
        def createGeometry(dimTags=[]):
            """
            gmsh.model.mesh.createGeometry(dimTags=[])

            Create a geometry for the discrete entities `dimTags' (represented solely
            by a mesh, without an underlying CAD description), i.e. create a
            parametrization for discrete curves and surfaces, assuming that each can be
            parametrized with a single map. If `dimTags' is empty, create a geometry
            for all the discrete entities.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshCreateGeometry(
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        create_geometry = createGeometry

        @staticmethod
        def createTopology(makeSimplyConnected=True, exportDiscrete=True):
            """
            gmsh.model.mesh.createTopology(makeSimplyConnected=True, exportDiscrete=True)

            Create a boundary representation from the mesh if the model does not have
            one (e.g. when imported from mesh file formats with no BRep representation
            of the underlying model). If `makeSimplyConnected' is set, enforce simply
            connected discrete surfaces and volumes. If `exportDiscrete' is set, clear
            any built-in CAD kernel entities and export the discrete entities in the
            built-in CAD kernel.
            """
            ierr = c_int()
            lib.gmshModelMeshCreateTopology(
                c_int(bool(makeSimplyConnected)),
                c_int(bool(exportDiscrete)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        create_topology = createTopology

        @staticmethod
        def computeHomology(domainTags=[], subdomainTags=[], dims=[]):
            """
            gmsh.model.mesh.computeHomology(domainTags=[], subdomainTags=[], dims=[])

            Compute a basis representation for homology spaces after a mesh has been
            generated. The computation domain is given in a list of physical group tags
            `domainTags'; if empty, the whole mesh is the domain. The computation
            subdomain for relative homology computation is given in a list of physical
            group tags `subdomainTags'; if empty, absolute homology is computed. The
            dimensions homology bases to be computed are given in the list `dim'; if
            empty, all bases are computed. Resulting basis representation chains are
            stored as physical groups in the mesh.
            """
            api_domainTags_, api_domainTags_n_ = _ivectorint(domainTags)
            api_subdomainTags_, api_subdomainTags_n_ = _ivectorint(subdomainTags)
            api_dims_, api_dims_n_ = _ivectorint(dims)
            ierr = c_int()
            lib.gmshModelMeshComputeHomology(
                api_domainTags_, api_domainTags_n_,
                api_subdomainTags_, api_subdomainTags_n_,
                api_dims_, api_dims_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        compute_homology = computeHomology

        @staticmethod
        def computeCohomology(domainTags=[], subdomainTags=[], dims=[]):
            """
            gmsh.model.mesh.computeCohomology(domainTags=[], subdomainTags=[], dims=[])

            Compute a basis representation for cohomology spaces after a mesh has been
            generated. The computation domain is given in a list of physical group tags
            `domainTags'; if empty, the whole mesh is the domain. The computation
            subdomain for relative cohomology computation is given in a list of
            physical group tags `subdomainTags'; if empty, absolute cohomology is
            computed. The dimensions homology bases to be computed are given in the
            list `dim'; if empty, all bases are computed. Resulting basis
            representation cochains are stored as physical groups in the mesh.
            """
            api_domainTags_, api_domainTags_n_ = _ivectorint(domainTags)
            api_subdomainTags_, api_subdomainTags_n_ = _ivectorint(subdomainTags)
            api_dims_, api_dims_n_ = _ivectorint(dims)
            ierr = c_int()
            lib.gmshModelMeshComputeCohomology(
                api_domainTags_, api_domainTags_n_,
                api_subdomainTags_, api_subdomainTags_n_,
                api_dims_, api_dims_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        compute_cohomology = computeCohomology

        @staticmethod
        def computeCrossField():
            """
            gmsh.model.mesh.computeCrossField()

            Compute a cross field for the current mesh. The function creates 3 views:
            the H function, the Theta function and cross directions. Return the tags of
            the views.

            Return `viewTags'.
            """
            api_viewTags_, api_viewTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshComputeCrossField(
                byref(api_viewTags_), byref(api_viewTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorint(api_viewTags_, api_viewTags_n_.value)
        compute_cross_field = computeCrossField

        @staticmethod
        def triangulate(coord):
            """
            gmsh.model.mesh.triangulate(coord)

            Triangulate the points given in the `coord' vector as pairs of u, v
            coordinates, and return the node tags (with numbering starting at 1) of the
            resulting triangles in `tri'.

            Return `tri'.
            """
            api_coord_, api_coord_n_ = _ivectordouble(coord)
            api_tri_, api_tri_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshTriangulate(
                api_coord_, api_coord_n_,
                byref(api_tri_), byref(api_tri_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorsize(api_tri_, api_tri_n_.value)

        @staticmethod
        def tetrahedralize(coord):
            """
            gmsh.model.mesh.tetrahedralize(coord)

            Tetrahedralize the points given in the `coord' vector as triplets of x, y,
            z coordinates, and return the node tags (with numbering starting at 1) of
            the resulting tetrahedra in `tetra'.

            Return `tetra'.
            """
            api_coord_, api_coord_n_ = _ivectordouble(coord)
            api_tetra_, api_tetra_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshTetrahedralize(
                api_coord_, api_coord_n_,
                byref(api_tetra_), byref(api_tetra_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorsize(api_tetra_, api_tetra_n_.value)


        class field:
            """
            Mesh size field functions
            """

            @staticmethod
            def add(fieldType, tag=-1):
                """
                gmsh.model.mesh.field.add(fieldType, tag=-1)

                Add a new mesh size field of type `fieldType'. If `tag' is positive, assign
                the tag explicitly; otherwise a new tag is assigned automatically. Return
                the field tag.

                Return an integer value.
                """
                ierr = c_int()
                api_result_ = lib.gmshModelMeshFieldAdd(
                    c_char_p(fieldType.encode()),
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
                return api_result_

            @staticmethod
            def remove(tag):
                """
                gmsh.model.mesh.field.remove(tag)

                Remove the field with tag `tag'.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldRemove(
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())

            @staticmethod
            def setNumber(tag, option, value):
                """
                gmsh.model.mesh.field.setNumber(tag, option, value)

                Set the numerical option `option' to value `value' for field `tag'.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetNumber(
                    c_int(tag),
                    c_char_p(option.encode()),
                    c_double(value),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_number = setNumber

            @staticmethod
            def setString(tag, option, value):
                """
                gmsh.model.mesh.field.setString(tag, option, value)

                Set the string option `option' to value `value' for field `tag'.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetString(
                    c_int(tag),
                    c_char_p(option.encode()),
                    c_char_p(value.encode()),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_string = setString

            @staticmethod
            def setNumbers(tag, option, value):
                """
                gmsh.model.mesh.field.setNumbers(tag, option, value)

                Set the numerical list option `option' to value `value' for field `tag'.
                """
                api_value_, api_value_n_ = _ivectordouble(value)
                ierr = c_int()
                lib.gmshModelMeshFieldSetNumbers(
                    c_int(tag),
                    c_char_p(option.encode()),
                    api_value_, api_value_n_,
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_numbers = setNumbers

            @staticmethod
            def setAsBackgroundMesh(tag):
                """
                gmsh.model.mesh.field.setAsBackgroundMesh(tag)

                Set the field `tag' as the background mesh size field.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetAsBackgroundMesh(
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_as_background_mesh = setAsBackgroundMesh

            @staticmethod
            def setAsBoundaryLayer(tag):
                """
                gmsh.model.mesh.field.setAsBoundaryLayer(tag)

                Set the field `tag' as a boundary layer size field.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetAsBoundaryLayer(
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_as_boundary_layer = setAsBoundaryLayer


    class geo:
        """
        Built-in CAD kernel functions
        """

        @staticmethod
        def addPoint(x, y, z, meshSize=0., tag=-1):
            """
            gmsh.model.geo.addPoint(x, y, z, meshSize=0., tag=-1)

            Add a geometrical point in the built-in CAD representation, at coordinates
            (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint at that
            point. If `tag' is positive, set the tag explicitly; otherwise a new tag is
            selected automatically. Return the tag of the point. (Note that the point
            will be added in the current model only after `synchronize' is called. This
            behavior holds for all the entities added in the geo module.)

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddPoint(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(meshSize),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_point = addPoint

        @staticmethod
        def addLine(startTag, endTag, tag=-1):
            """
            gmsh.model.geo.addLine(startTag, endTag, tag=-1)

            Add a straight line segment in the built-in CAD representation, between the
            two points with tags `startTag' and `endTag'. If `tag' is positive, set the
            tag explicitly; otherwise a new tag is selected automatically. Return the
            tag of the line.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddLine(
                c_int(startTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_line = addLine

        @staticmethod
        def addCircleArc(startTag, centerTag, endTag, tag=-1, nx=0., ny=0., nz=0.):
            """
            gmsh.model.geo.addCircleArc(startTag, centerTag, endTag, tag=-1, nx=0., ny=0., nz=0.)

            Add a circle arc (strictly smaller than Pi) in the built-in CAD
            representation, between the two points with tags `startTag' and `endTag',
            and with center `centerTag'. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. If (`nx', `ny', `nz') != (0,
            0, 0), explicitly set the plane of the circle arc. Return the tag of the
            circle arc.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddCircleArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(endTag),
                c_int(tag),
                c_double(nx),
                c_double(ny),
                c_double(nz),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_circle_arc = addCircleArc

        @staticmethod
        def addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1, nx=0., ny=0., nz=0.):
            """
            gmsh.model.geo.addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1, nx=0., ny=0., nz=0.)

            Add an ellipse arc (strictly smaller than Pi) in the built-in CAD
            representation, between the two points `startTag' and `endTag', and with
            center `centerTag' and major axis point `majorTag'. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically. If
            (`nx', `ny', `nz') != (0, 0, 0), explicitly set the plane of the circle
            arc. Return the tag of the ellipse arc.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddEllipseArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(majorTag),
                c_int(endTag),
                c_int(tag),
                c_double(nx),
                c_double(ny),
                c_double(nz),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_ellipse_arc = addEllipseArc

        @staticmethod
        def addSpline(pointTags, tag=-1):
            """
            gmsh.model.geo.addSpline(pointTags, tag=-1)

            Add a spline (Catmull-Rom) curve in the built-in CAD representation, going
            through the points `pointTags'. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. Create a
            periodic curve if the first and last points are the same. Return the tag of
            the spline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_spline = addSpline

        @staticmethod
        def addBSpline(pointTags, tag=-1):
            """
            gmsh.model.geo.addBSpline(pointTags, tag=-1)

            Add a cubic b-spline curve in the built-in CAD representation, with
            `pointTags' control points. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Creates a periodic curve if
            the first and last points are the same. Return the tag of the b-spline
            curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddBSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bspline = addBSpline

        @staticmethod
        def addBezier(pointTags, tag=-1):
            """
            gmsh.model.geo.addBezier(pointTags, tag=-1)

            Add a Bezier curve in the built-in CAD representation, with `pointTags'
            control points. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically.  Return the tag of the Bezier curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddBezier(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bezier = addBezier

        @staticmethod
        def addPolyline(pointTags, tag=-1):
            """
            gmsh.model.geo.addPolyline(pointTags, tag=-1)

            Add a polyline curve in the built-in CAD representation, going through the
            points `pointTags'. If `tag' is positive, set the tag explicitly; otherwise
            a new tag is selected automatically. Create a periodic curve if the first
            and last points are the same. Return the tag of the polyline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddPolyline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_polyline = addPolyline

        @staticmethod
        def addCompoundSpline(curveTags, numIntervals=5, tag=-1):
            """
            gmsh.model.geo.addCompoundSpline(curveTags, numIntervals=5, tag=-1)

            Add a spline (Catmull-Rom) curve in the built-in CAD representation, going
            through points sampling the curves in `curveTags'. The density of sampling
            points on each curve is governed by `numIntervals'. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically.
            Return the tag of the spline.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddCompoundSpline(
                api_curveTags_, api_curveTags_n_,
                c_int(numIntervals),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_compound_spline = addCompoundSpline

        @staticmethod
        def addCompoundBSpline(curveTags, numIntervals=20, tag=-1):
            """
            gmsh.model.geo.addCompoundBSpline(curveTags, numIntervals=20, tag=-1)

            Add a b-spline curve in the built-in CAD representation, with control
            points sampling the curves in `curveTags'. The density of sampling points
            on each curve is governed by `numIntervals'. If `tag' is positive, set the
            tag explicitly; otherwise a new tag is selected automatically. Return the
            tag of the b-spline.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddCompoundBSpline(
                api_curveTags_, api_curveTags_n_,
                c_int(numIntervals),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_compound_bspline = addCompoundBSpline

        @staticmethod
        def addCurveLoop(curveTags, tag=-1, reorient=False):
            """
            gmsh.model.geo.addCurveLoop(curveTags, tag=-1, reorient=False)

            Add a curve loop (a closed wire) in the built-in CAD representation, formed
            by the curves `curveTags'. `curveTags' should contain (signed) tags of
            model entities of dimension 1 forming a closed loop: a negative tag
            signifies that the underlying curve is considered with reversed
            orientation. If `tag' is positive, set the tag explicitly; otherwise a new
            tag is selected automatically. If `reorient' is set, automatically reorient
            the curves if necessary. Return the tag of the curve loop.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddCurveLoop(
                api_curveTags_, api_curveTags_n_,
                c_int(tag),
                c_int(bool(reorient)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_curve_loop = addCurveLoop

        @staticmethod
        def addCurveLoops(curveTags):
            """
            gmsh.model.geo.addCurveLoops(curveTags)

            Add curve loops in the built-in CAD representation based on the curves
            `curveTags'. Return the `tags' of found curve loops, if any.

            Return `tags'.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelGeoAddCurveLoops(
                api_curveTags_, api_curveTags_n_,
                byref(api_tags_), byref(api_tags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorint(api_tags_, api_tags_n_.value)
        add_curve_loops = addCurveLoops

        @staticmethod
        def addPlaneSurface(wireTags, tag=-1):
            """
            gmsh.model.geo.addPlaneSurface(wireTags, tag=-1)

            Add a plane surface in the built-in CAD representation, defined by one or
            more curve loops `wireTags'. The first curve loop defines the exterior
            contour; additional curve loop define holes. If `tag' is positive, set the
            tag explicitly; otherwise a new tag is selected automatically. Return the
            tag of the surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddPlaneSurface(
                api_wireTags_, api_wireTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_plane_surface = addPlaneSurface

        @staticmethod
        def addSurfaceFilling(wireTags, tag=-1, sphereCenterTag=-1):
            """
            gmsh.model.geo.addSurfaceFilling(wireTags, tag=-1, sphereCenterTag=-1)

            Add a surface in the built-in CAD representation, filling the curve loops
            in `wireTags' using transfinite interpolation. Currently only a single
            curve loop is supported; this curve loop should be composed by 3 or 4
            curves only. If `tag' is positive, set the tag explicitly; otherwise a new
            tag is selected automatically. Return the tag of the surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddSurfaceFilling(
                api_wireTags_, api_wireTags_n_,
                c_int(tag),
                c_int(sphereCenterTag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_surface_filling = addSurfaceFilling

        @staticmethod
        def addSurfaceLoop(surfaceTags, tag=-1):
            """
            gmsh.model.geo.addSurfaceLoop(surfaceTags, tag=-1)

            Add a surface loop (a closed shell) formed by `surfaceTags' in the built-in
            CAD representation.  If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Return the tag of the shell.

            Return an integer value.
            """
            api_surfaceTags_, api_surfaceTags_n_ = _ivectorint(surfaceTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddSurfaceLoop(
                api_surfaceTags_, api_surfaceTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_surface_loop = addSurfaceLoop

        @staticmethod
        def addVolume(shellTags, tag=-1):
            """
            gmsh.model.geo.addVolume(shellTags, tag=-1)

            Add a volume (a region) in the built-in CAD representation, defined by one
            or more shells `shellTags'. The first surface loop defines the exterior
            boundary; additional surface loop define holes. If `tag' is positive, set
            the tag explicitly; otherwise a new tag is selected automatically. Return
            the tag of the volume.

            Return an integer value.
            """
            api_shellTags_, api_shellTags_n_ = _ivectorint(shellTags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddVolume(
                api_shellTags_, api_shellTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_volume = addVolume

        @staticmethod
        def extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.geo.extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False)

            Extrude the entities `dimTags' in the built-in CAD representation, using a
            translation along (`dx', `dy', `dz'). Return extruded entities in
            `outDimTags'. If `numElements' is not empty, also extrude the mesh: the
            entries in `numElements' give the number of elements in each layer. If
            `height' is not empty, it provides the (cumulative) height of the different
            layers, normalized to 1. If `recombine' is set, recombine the mesh in the
            layers.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoExtrude(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.geo.revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False)

            Extrude the entities `dimTags' in the built-in CAD representation, using a
            rotation of `angle' radians around the axis of revolution defined by the
            point (`x', `y', `z') and the direction (`ax', `ay', `az'). The angle
            should be strictly smaller than Pi. Return extruded entities in
            `outDimTags'. If `numElements' is not empty, also extrude the mesh: the
            entries in `numElements' give the number of elements in each layer. If
            `height' is not empty, it provides the (cumulative) height of the different
            layers, normalized to 1. If `recombine' is set, recombine the mesh in the
            layers.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoRevolve(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def twist(dimTags, x, y, z, dx, dy, dz, ax, ay, az, angle, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.geo.twist(dimTags, x, y, z, dx, dy, dz, ax, ay, az, angle, numElements=[], heights=[], recombine=False)

            Extrude the entities `dimTags' in the built-in CAD representation, using a
            combined translation and rotation of `angle' radians, along (`dx', `dy',
            `dz') and around the axis of revolution defined by the point (`x', `y',
            `z') and the direction (`ax', `ay', `az'). The angle should be strictly
            smaller than Pi. Return extruded entities in `outDimTags'. If `numElements'
            is not empty, also extrude the mesh: the entries in `numElements' give the
            number of elements in each layer. If `height' is not empty, it provides the
            (cumulative) height of the different layers, normalized to 1. If
            `recombine' is set, recombine the mesh in the layers.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoTwist(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def extrudeBoundaryLayer(dimTags, numElements=[1], heights=[], recombine=False, second=False, viewIndex=-1):
            """
            gmsh.model.geo.extrudeBoundaryLayer(dimTags, numElements=[1], heights=[], recombine=False, second=False, viewIndex=-1)

            Extrude the entities `dimTags' in the built-in CAD representation along the
            normals of the mesh, creating discrete boundary layer entities. Return
            extruded entities in `outDimTags'. The entries in `numElements' give the
            number of elements in each layer. If `height' is not empty, it provides the
            height of the different layers. If `recombine' is set, recombine the mesh
            in the layers. A second boundary layer can be created from the same
            entities if `second' is set. If `viewIndex' is >= 0, use the corresponding
            view to either specify the normals (if the view contains a vector field) or
            scale the normals (if the view is scalar).

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoExtrudeBoundaryLayer(
                api_dimTags_, api_dimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                c_int(bool(second)),
                c_int(viewIndex),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
        extrude_boundary_layer = extrudeBoundaryLayer

        @staticmethod
        def translate(dimTags, dx, dy, dz):
            """
            gmsh.model.geo.translate(dimTags, dx, dy, dz)

            Translate the entities `dimTags' in the built-in CAD representation along
            (`dx', `dy', `dz').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoTranslate(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def rotate(dimTags, x, y, z, ax, ay, az, angle):
            """
            gmsh.model.geo.rotate(dimTags, x, y, z, ax, ay, az, angle)

            Rotate the entities `dimTags' in the built-in CAD representation by `angle'
            radians around the axis of revolution defined by the point (`x', `y', `z')
            and the direction (`ax', `ay', `az').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoRotate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def dilate(dimTags, x, y, z, a, b, c):
            """
            gmsh.model.geo.dilate(dimTags, x, y, z, a, b, c)

            Scale the entities `dimTag' in the built-in CAD representation by factors
            `a', `b' and `c' along the three coordinate axes; use (`x', `y', `z') as
            the center of the homothetic transformation.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoDilate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(a),
                c_double(b),
                c_double(c),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def mirror(dimTags, a, b, c, d):
            """
            gmsh.model.geo.mirror(dimTags, a, b, c, d)

            Mirror the entities `dimTag' in the built-in CAD representation, with
            respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoMirror(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def symmetrize(dimTags, a, b, c, d):
            """
            gmsh.model.geo.symmetrize(dimTags, a, b, c, d)

            Mirror the entities `dimTag' in the built-in CAD representation, with
            respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0.
            (This is a synonym for `mirror', which will be deprecated in a future
            release.)
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoSymmetrize(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def copy(dimTags):
            """
            gmsh.model.geo.copy(dimTags)

            Copy the entities `dimTags' in the built-in CAD representation; the new
            entities are returned in `outDimTags'.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelGeoCopy(
                api_dimTags_, api_dimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def remove(dimTags, recursive=False):
            """
            gmsh.model.geo.remove(dimTags, recursive=False)

            Remove the entities `dimTags' in the built-in CAD representation. If
            `recursive' is true, remove all the entities on their boundaries, down to
            dimension 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoRemove(
                api_dimTags_, api_dimTags_n_,
                c_int(bool(recursive)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def removeAllDuplicates():
            """
            gmsh.model.geo.removeAllDuplicates()

            Remove all duplicate entities in the built-in CAD representation (different
            entities at the same geometrical location).
            """
            ierr = c_int()
            lib.gmshModelGeoRemoveAllDuplicates(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_all_duplicates = removeAllDuplicates

        @staticmethod
        def splitCurve(tag, pointTags):
            """
            gmsh.model.geo.splitCurve(tag, pointTags)

            Split the curve of tag `tag' in the built-in CAD representation, on the
            control points `pointTags'. Return the tags `curveTags' of the newly
            created curves.

            Return `curveTags'.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            api_curveTags_, api_curveTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelGeoSplitCurve(
                c_int(tag),
                api_pointTags_, api_pointTags_n_,
                byref(api_curveTags_), byref(api_curveTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorint(api_curveTags_, api_curveTags_n_.value)
        split_curve = splitCurve

        @staticmethod
        def getMaxTag(dim):
            """
            gmsh.model.geo.getMaxTag(dim)

            Get the maximum tag of entities of dimension `dim' in the built-in CAD
            representation.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelGeoGetMaxTag(
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        get_max_tag = getMaxTag

        @staticmethod
        def setMaxTag(dim, maxTag):
            """
            gmsh.model.geo.setMaxTag(dim, maxTag)

            Set the maximum tag `maxTag' for entities of dimension `dim' in the built-
            in CAD representation.
            """
            ierr = c_int()
            lib.gmshModelGeoSetMaxTag(
                c_int(dim),
                c_int(maxTag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_max_tag = setMaxTag

        @staticmethod
        def addPhysicalGroup(dim, tags, tag=-1):
            """
            gmsh.model.geo.addPhysicalGroup(dim, tags, tag=-1)

            Add a physical group of dimension `dim', grouping the entities with tags
            `tags' in the built-in CAD representation. Return the tag of the physical
            group, equal to `tag' if `tag' is positive, or a new tag if `tag' < 0.

            Return an integer value.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            ierr = c_int()
            api_result_ = lib.gmshModelGeoAddPhysicalGroup(
                c_int(dim),
                api_tags_, api_tags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_physical_group = addPhysicalGroup

        @staticmethod
        def removePhysicalGroups(dimTags=[]):
            """
            gmsh.model.geo.removePhysicalGroups(dimTags=[])

            Remove the physical groups `dimTags' from the built-in CAD representation.
            If `dimTags' is empty, remove all groups.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoRemovePhysicalGroups(
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_physical_groups = removePhysicalGroups

        @staticmethod
        def synchronize():
            """
            gmsh.model.geo.synchronize()

            Synchronize the built-in CAD representation with the current Gmsh model.
            This can be called at any time, but since it involves a non trivial amount
            of processing, the number of synchronization points should normally be
            minimized. Without synchronization the entities in the built-in CAD
            representation are not available to any function outside of the built-in
            CAD kernel functions.
            """
            ierr = c_int()
            lib.gmshModelGeoSynchronize(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())


        class mesh:
            """
            Built-in CAD kernel meshing constraints
            """

            @staticmethod
            def setSize(dimTags, size):
                """
                gmsh.model.geo.mesh.setSize(dimTags, size)

                Set a mesh size constraint on the entities `dimTags' in the built-in CAD
                kernel representation. Currently only entities of dimension 0 (points) are
                handled.
                """
                api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
                ierr = c_int()
                lib.gmshModelGeoMeshSetSize(
                    api_dimTags_, api_dimTags_n_,
                    c_double(size),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_size = setSize

            @staticmethod
            def setTransfiniteCurve(tag, nPoints, meshType="Progression", coef=1.):
                """
                gmsh.model.geo.mesh.setTransfiniteCurve(tag, nPoints, meshType="Progression", coef=1.)

                Set a transfinite meshing constraint on the curve `tag' in the built-in CAD
                kernel representation, with `numNodes' nodes distributed according to
                `meshType' and `coef'. Currently supported types are "Progression"
                (geometrical progression with power `coef') and "Bump" (refinement toward
                both extremities of the curve).
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetTransfiniteCurve(
                    c_int(tag),
                    c_int(nPoints),
                    c_char_p(meshType.encode()),
                    c_double(coef),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_transfinite_curve = setTransfiniteCurve

            @staticmethod
            def setTransfiniteSurface(tag, arrangement="Left", cornerTags=[]):
                """
                gmsh.model.geo.mesh.setTransfiniteSurface(tag, arrangement="Left", cornerTags=[])

                Set a transfinite meshing constraint on the surface `tag' in the built-in
                CAD kernel representation. `arrangement' describes the arrangement of the
                triangles when the surface is not flagged as recombined: currently
                supported values are "Left", "Right", "AlternateLeft" and "AlternateRight".
                `cornerTags' can be used to specify the (3 or 4) corners of the transfinite
                interpolation explicitly; specifying the corners explicitly is mandatory if
                the surface has more that 3 or 4 points on its boundary.
                """
                api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
                ierr = c_int()
                lib.gmshModelGeoMeshSetTransfiniteSurface(
                    c_int(tag),
                    c_char_p(arrangement.encode()),
                    api_cornerTags_, api_cornerTags_n_,
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_transfinite_surface = setTransfiniteSurface

            @staticmethod
            def setTransfiniteVolume(tag, cornerTags=[]):
                """
                gmsh.model.geo.mesh.setTransfiniteVolume(tag, cornerTags=[])

                Set a transfinite meshing constraint on the surface `tag' in the built-in
                CAD kernel representation. `cornerTags' can be used to specify the (6 or 8)
                corners of the transfinite interpolation explicitly.
                """
                api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
                ierr = c_int()
                lib.gmshModelGeoMeshSetTransfiniteVolume(
                    c_int(tag),
                    api_cornerTags_, api_cornerTags_n_,
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_transfinite_volume = setTransfiniteVolume

            @staticmethod
            def setRecombine(dim, tag, angle=45.):
                """
                gmsh.model.geo.mesh.setRecombine(dim, tag, angle=45.)

                Set a recombination meshing constraint on the entity of dimension `dim' and
                tag `tag' in the built-in CAD kernel representation. Currently only
                entities of dimension 2 (to recombine triangles into quadrangles) are
                supported.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetRecombine(
                    c_int(dim),
                    c_int(tag),
                    c_double(angle),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_recombine = setRecombine

            @staticmethod
            def setSmoothing(dim, tag, val):
                """
                gmsh.model.geo.mesh.setSmoothing(dim, tag, val)

                Set a smoothing meshing constraint on the entity of dimension `dim' and tag
                `tag' in the built-in CAD kernel representation. `val' iterations of a
                Laplace smoother are applied.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetSmoothing(
                    c_int(dim),
                    c_int(tag),
                    c_int(val),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_smoothing = setSmoothing

            @staticmethod
            def setReverse(dim, tag, val=True):
                """
                gmsh.model.geo.mesh.setReverse(dim, tag, val=True)

                Set a reverse meshing constraint on the entity of dimension `dim' and tag
                `tag' in the built-in CAD kernel representation. If `val' is true, the mesh
                orientation will be reversed with respect to the natural mesh orientation
                (i.e. the orientation consistent with the orientation of the geometry). If
                `val' is false, the mesh is left as-is.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetReverse(
                    c_int(dim),
                    c_int(tag),
                    c_int(bool(val)),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_reverse = setReverse

            @staticmethod
            def setAlgorithm(dim, tag, val):
                """
                gmsh.model.geo.mesh.setAlgorithm(dim, tag, val)

                Set the meshing algorithm on the entity of dimension `dim' and tag `tag' in
                the built-in CAD kernel representation. Currently only supported for `dim'
                == 2.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetAlgorithm(
                    c_int(dim),
                    c_int(tag),
                    c_int(val),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_algorithm = setAlgorithm

            @staticmethod
            def setSizeFromBoundary(dim, tag, val):
                """
                gmsh.model.geo.mesh.setSizeFromBoundary(dim, tag, val)

                Force the mesh size to be extended from the boundary, or not, for the
                entity of dimension `dim' and tag `tag' in the built-in CAD kernel
                representation. Currently only supported for `dim' == 2.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetSizeFromBoundary(
                    c_int(dim),
                    c_int(tag),
                    c_int(val),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_size_from_boundary = setSizeFromBoundary


    class occ:
        """
        OpenCASCADE CAD kernel functions
        """

        @staticmethod
        def addPoint(x, y, z, meshSize=0., tag=-1):
            """
            gmsh.model.occ.addPoint(x, y, z, meshSize=0., tag=-1)

            Add a geometrical point in the OpenCASCADE CAD representation, at
            coordinates (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint
            at that point. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the point. (Note that
            the point will be added in the current model only after `synchronize' is
            called. This behavior holds for all the entities added in the occ module.)

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddPoint(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(meshSize),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_point = addPoint

        @staticmethod
        def addLine(startTag, endTag, tag=-1):
            """
            gmsh.model.occ.addLine(startTag, endTag, tag=-1)

            Add a straight line segment in the OpenCASCADE CAD representation, between
            the two points with tags `startTag' and `endTag'. If `tag' is positive, set
            the tag explicitly; otherwise a new tag is selected automatically. Return
            the tag of the line.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddLine(
                c_int(startTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_line = addLine

        @staticmethod
        def addCircleArc(startTag, centerTag, endTag, tag=-1):
            """
            gmsh.model.occ.addCircleArc(startTag, centerTag, endTag, tag=-1)

            Add a circle arc in the OpenCASCADE CAD representation, between the two
            points with tags `startTag' and `endTag', with center `centerTag'. If `tag'
            is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the circle arc.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddCircleArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_circle_arc = addCircleArc

        @staticmethod
        def addCircle(x, y, z, r, tag=-1, angle1=0., angle2=2*pi):
            """
            gmsh.model.occ.addCircle(x, y, z, r, tag=-1, angle1=0., angle2=2*pi)

            Add a circle of center (`x', `y', `z') and radius `r' in the OpenCASCADE
            CAD representation. If `tag' is positive, set the tag explicitly; otherwise
            a new tag is selected automatically. If `angle1' and `angle2' are
            specified, create a circle arc between the two angles. Return the tag of
            the circle.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddCircle(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(r),
                c_int(tag),
                c_double(angle1),
                c_double(angle2),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_circle = addCircle

        @staticmethod
        def addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1):
            """
            gmsh.model.occ.addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1)

            Add an ellipse arc in the OpenCASCADE CAD representation, between the two
            points `startTag' and `endTag', and with center `centerTag' and major axis
            point `majorTag'. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the ellipse arc. Note
            that OpenCASCADE does not allow creating ellipse arcs with the major radius
            smaller than the minor radius.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddEllipseArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(majorTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_ellipse_arc = addEllipseArc

        @staticmethod
        def addEllipse(x, y, z, r1, r2, tag=-1, angle1=0., angle2=2*pi):
            """
            gmsh.model.occ.addEllipse(x, y, z, r1, r2, tag=-1, angle1=0., angle2=2*pi)

            Add an ellipse of center (`x', `y', `z') and radii `r1' and `r2' along the
            x- and y-axes, respectively, in the OpenCASCADE CAD representation. If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. If `angle1' and `angle2' are specified, create an ellipse
            arc between the two angles. Return the tag of the ellipse. Note that
            OpenCASCADE does not allow creating ellipses with the major radius (along
            the x-axis) smaller than or equal to the minor radius (along the y-axis):
            rotate the shape or use `addCircle' in such cases.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddEllipse(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(r1),
                c_double(r2),
                c_int(tag),
                c_double(angle1),
                c_double(angle2),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_ellipse = addEllipse

        @staticmethod
        def addSpline(pointTags, tag=-1):
            """
            gmsh.model.occ.addSpline(pointTags, tag=-1)

            Add a spline (C2 b-spline) curve in the OpenCASCADE CAD representation,
            going through the points `pointTags'. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. Create a
            periodic curve if the first and last points are the same. Return the tag of
            the spline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_spline = addSpline

        @staticmethod
        def addBSpline(pointTags, tag=-1, degree=3, weights=[], knots=[], multiplicities=[]):
            """
            gmsh.model.occ.addBSpline(pointTags, tag=-1, degree=3, weights=[], knots=[], multiplicities=[])

            Add a b-spline curve of degree `degree' in the OpenCASCADE CAD
            representation, with `pointTags' control points. If `weights', `knots' or
            `multiplicities' are not provided, default parameters are computed
            automatically. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Create a periodic curve if the first and
            last points are the same. Return the tag of the b-spline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            api_weights_, api_weights_n_ = _ivectordouble(weights)
            api_knots_, api_knots_n_ = _ivectordouble(knots)
            api_multiplicities_, api_multiplicities_n_ = _ivectorint(multiplicities)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                c_int(degree),
                api_weights_, api_weights_n_,
                api_knots_, api_knots_n_,
                api_multiplicities_, api_multiplicities_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bspline = addBSpline

        @staticmethod
        def addBezier(pointTags, tag=-1):
            """
            gmsh.model.occ.addBezier(pointTags, tag=-1)

            Add a Bezier curve in the OpenCASCADE CAD representation, with `pointTags'
            control points. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the Bezier curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBezier(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bezier = addBezier

        @staticmethod
        def addWire(curveTags, tag=-1, checkClosed=False):
            """
            gmsh.model.occ.addWire(curveTags, tag=-1, checkClosed=False)

            Add a wire (open or closed) in the OpenCASCADE CAD representation, formed
            by the curves `curveTags'. Note that an OpenCASCADE wire can be made of
            curves that share geometrically identical (but topologically different)
            points. If `tag' is positive, set the tag explicitly; otherwise a new tag
            is selected automatically. Return the tag of the wire.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddWire(
                api_curveTags_, api_curveTags_n_,
                c_int(tag),
                c_int(bool(checkClosed)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_wire = addWire

        @staticmethod
        def addCurveLoop(curveTags, tag=-1):
            """
            gmsh.model.occ.addCurveLoop(curveTags, tag=-1)

            Add a curve loop (a closed wire) in the OpenCASCADE CAD representation,
            formed by the curves `curveTags'. `curveTags' should contain tags of curves
            forming a closed loop. Note that an OpenCASCADE curve loop can be made of
            curves that share geometrically identical (but topologically different)
            points. If `tag' is positive, set the tag explicitly; otherwise a new tag
            is selected automatically. Return the tag of the curve loop.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddCurveLoop(
                api_curveTags_, api_curveTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_curve_loop = addCurveLoop

        @staticmethod
        def addRectangle(x, y, z, dx, dy, tag=-1, roundedRadius=0.):
            """
            gmsh.model.occ.addRectangle(x, y, z, dx, dy, tag=-1, roundedRadius=0.)

            Add a rectangle in the OpenCASCADE CAD representation, with lower left
            corner at (`x', `y', `z') and upper right corner at (`x' + `dx', `y' +
            `dy', `z'). If `tag' is positive, set the tag explicitly; otherwise a new
            tag is selected automatically. Round the corners if `roundedRadius' is
            nonzero. Return the tag of the rectangle.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddRectangle(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_int(tag),
                c_double(roundedRadius),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_rectangle = addRectangle

        @staticmethod
        def addDisk(xc, yc, zc, rx, ry, tag=-1):
            """
            gmsh.model.occ.addDisk(xc, yc, zc, rx, ry, tag=-1)

            Add a disk in the OpenCASCADE CAD representation, with center (`xc', `yc',
            `zc') and radius `rx' along the x-axis and `ry' along the y-axis. If `tag'
            is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the disk.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddDisk(
                c_double(xc),
                c_double(yc),
                c_double(zc),
                c_double(rx),
                c_double(ry),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_disk = addDisk

        @staticmethod
        def addPlaneSurface(wireTags, tag=-1):
            """
            gmsh.model.occ.addPlaneSurface(wireTags, tag=-1)

            Add a plane surface in the OpenCASCADE CAD representation, defined by one
            or more curve loops (or closed wires) `wireTags'. The first curve loop
            defines the exterior contour; additional curve loop define holes. If `tag'
            is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddPlaneSurface(
                api_wireTags_, api_wireTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_plane_surface = addPlaneSurface

        @staticmethod
        def addSurfaceFilling(wireTag, tag=-1, pointTags=[]):
            """
            gmsh.model.occ.addSurfaceFilling(wireTag, tag=-1, pointTags=[])

            Add a surface in the OpenCASCADE CAD representation, filling the curve loop
            `wireTag'. If `tag' is positive, set the tag explicitly; otherwise a new
            tag is selected automatically. Return the tag of the surface. If
            `pointTags' are provided, force the surface to pass through the given
            points.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddSurfaceFilling(
                c_int(wireTag),
                c_int(tag),
                api_pointTags_, api_pointTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_surface_filling = addSurfaceFilling

        @staticmethod
        def addBSplineFilling(wireTag, tag=-1, type=""):
            """
            gmsh.model.occ.addBSplineFilling(wireTag, tag=-1, type="")

            Add a BSpline surface in the OpenCASCADE CAD representation, filling the
            curve loop `wireTag'. The curve loop should be made of 2, 3 or 4 BSpline
            curves. The optional `type' argument specifies the type of filling:
            "Stretch" creates the flattest patch, "Curved" (the default) creates the
            most rounded patch, and "Coons" creates a rounded patch with less depth
            than "Curved". If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the surface.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBSplineFilling(
                c_int(wireTag),
                c_int(tag),
                c_char_p(type.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bspline_filling = addBSplineFilling

        @staticmethod
        def addBezierFilling(wireTag, tag=-1, type=""):
            """
            gmsh.model.occ.addBezierFilling(wireTag, tag=-1, type="")

            Add a Bezier surface in the OpenCASCADE CAD representation, filling the
            curve loop `wireTag'. The curve loop should be made of 2, 3 or 4 Bezier
            curves. The optional `type' argument specifies the type of filling:
            "Stretch" creates the flattest patch, "Curved" (the default) creates the
            most rounded patch, and "Coons" creates a rounded patch with less depth
            than "Curved". If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the surface.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBezierFilling(
                c_int(wireTag),
                c_int(tag),
                c_char_p(type.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bezier_filling = addBezierFilling

        @staticmethod
        def addBSplineSurface(pointTags, numPointsU, tag=-1, degreeU=3, degreeV=3, weights=[], knotsU=[], knotsV=[], multiplicitiesU=[], multiplicitiesV=[], wireTags=[], wire3D=False):
            """
            gmsh.model.occ.addBSplineSurface(pointTags, numPointsU, tag=-1, degreeU=3, degreeV=3, weights=[], knotsU=[], knotsV=[], multiplicitiesU=[], multiplicitiesV=[], wireTags=[], wire3D=False)

            Add a b-spline surface of degree `degreeU' x `degreeV' in the OpenCASCADE
            CAD representation, with `pointTags' control points given as a single
            vector [Pu1v1, ... Pu`numPointsU'v1, Pu1v2, ...]. If `weights', `knotsU',
            `knotsV', `multiplicitiesU' or `multiplicitiesV' are not provided, default
            parameters are computed automatically. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. If `wireTags' is
            provided, trim the b-spline patch using the provided wires: the first wire
            defines the external contour, the others define holes. If `wire3D' is set,
            consider wire curves as 3D curves and project them on the b-spline surface;
            otherwise consider the wire curves as defined in the parametric space of
            the surface. Return the tag of the b-spline surface.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            api_weights_, api_weights_n_ = _ivectordouble(weights)
            api_knotsU_, api_knotsU_n_ = _ivectordouble(knotsU)
            api_knotsV_, api_knotsV_n_ = _ivectordouble(knotsV)
            api_multiplicitiesU_, api_multiplicitiesU_n_ = _ivectorint(multiplicitiesU)
            api_multiplicitiesV_, api_multiplicitiesV_n_ = _ivectorint(multiplicitiesV)
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBSplineSurface(
                api_pointTags_, api_pointTags_n_,
                c_int(numPointsU),
                c_int(tag),
                c_int(degreeU),
                c_int(degreeV),
                api_weights_, api_weights_n_,
                api_knotsU_, api_knotsU_n_,
                api_knotsV_, api_knotsV_n_,
                api_multiplicitiesU_, api_multiplicitiesU_n_,
                api_multiplicitiesV_, api_multiplicitiesV_n_,
                api_wireTags_, api_wireTags_n_,
                c_int(bool(wire3D)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bspline_surface = addBSplineSurface

        @staticmethod
        def addBezierSurface(pointTags, numPointsU, tag=-1, wireTags=[], wire3D=False):
            """
            gmsh.model.occ.addBezierSurface(pointTags, numPointsU, tag=-1, wireTags=[], wire3D=False)

            Add a Bezier surface in the OpenCASCADE CAD representation, with
            `pointTags' control points given as a single vector [Pu1v1, ...
            Pu`numPointsU'v1, Pu1v2, ...]. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. If `wireTags' is
            provided, trim the Bezier patch using the provided wires: the first wire
            defines the external contour, the others define holes. If `wire3D' is set,
            consider wire curves as 3D curves and project them on the Bezier surface;
            otherwise consider the wire curves as defined in the parametric space of
            the surface. Return the tag of the Bezier surface.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBezierSurface(
                api_pointTags_, api_pointTags_n_,
                c_int(numPointsU),
                c_int(tag),
                api_wireTags_, api_wireTags_n_,
                c_int(bool(wire3D)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_bezier_surface = addBezierSurface

        @staticmethod
        def addTrimmedSurface(surfaceTag, wireTags=[], wire3D=False, tag=-1):
            """
            gmsh.model.occ.addTrimmedSurface(surfaceTag, wireTags=[], wire3D=False, tag=-1)

            Trim the surface `surfaceTag' with the wires `wireTags', replacing any
            existing trimming curves. The first wire defines the external contour, the
            others define holes. If `wire3D' is set, consider wire curves as 3D curves
            and project them on the surface; otherwise consider the wire curves as
            defined in the parametric space of the surface. If `tag' is positive, set
            the tag explicitly; otherwise a new tag is selected automatically. Return
            the tag of the trimmed surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddTrimmedSurface(
                c_int(surfaceTag),
                api_wireTags_, api_wireTags_n_,
                c_int(bool(wire3D)),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_trimmed_surface = addTrimmedSurface

        @staticmethod
        def addSurfaceLoop(surfaceTags, tag=-1, sewing=False):
            """
            gmsh.model.occ.addSurfaceLoop(surfaceTags, tag=-1, sewing=False)

            Add a surface loop (a closed shell) in the OpenCASCADE CAD representation,
            formed by `surfaceTags'.  If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Return the tag of the
            surface loop. Setting `sewing' allows to build a shell made of surfaces
            that share geometrically identical (but topologically different) curves.

            Return an integer value.
            """
            api_surfaceTags_, api_surfaceTags_n_ = _ivectorint(surfaceTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddSurfaceLoop(
                api_surfaceTags_, api_surfaceTags_n_,
                c_int(tag),
                c_int(bool(sewing)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_surface_loop = addSurfaceLoop

        @staticmethod
        def addVolume(shellTags, tag=-1):
            """
            gmsh.model.occ.addVolume(shellTags, tag=-1)

            Add a volume (a region) in the OpenCASCADE CAD representation, defined by
            one or more surface loops `shellTags'. The first surface loop defines the
            exterior boundary; additional surface loop define holes. If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the volume.

            Return an integer value.
            """
            api_shellTags_, api_shellTags_n_ = _ivectorint(shellTags)
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddVolume(
                api_shellTags_, api_shellTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_volume = addVolume

        @staticmethod
        def addSphere(xc, yc, zc, radius, tag=-1, angle1=-pi/2, angle2=pi/2, angle3=2*pi):
            """
            gmsh.model.occ.addSphere(xc, yc, zc, radius, tag=-1, angle1=-pi/2, angle2=pi/2, angle3=2*pi)

            Add a sphere of center (`xc', `yc', `zc') and radius `r' in the OpenCASCADE
            CAD representation. The optional `angle1' and `angle2' arguments define the
            polar angle opening (from -Pi/2 to Pi/2). The optional `angle3' argument
            defines the azimuthal opening (from 0 to 2*Pi). If `tag' is positive, set
            the tag explicitly; otherwise a new tag is selected automatically. Return
            the tag of the sphere.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddSphere(
                c_double(xc),
                c_double(yc),
                c_double(zc),
                c_double(radius),
                c_int(tag),
                c_double(angle1),
                c_double(angle2),
                c_double(angle3),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_sphere = addSphere

        @staticmethod
        def addBox(x, y, z, dx, dy, dz, tag=-1):
            """
            gmsh.model.occ.addBox(x, y, z, dx, dy, dz, tag=-1)

            Add a parallelepipedic box in the OpenCASCADE CAD representation, defined
            by a point (`x', `y', `z') and the extents along the x-, y- and z-axes. If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the box.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddBox(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_box = addBox

        @staticmethod
        def addCylinder(x, y, z, dx, dy, dz, r, tag=-1, angle=2*pi):
            """
            gmsh.model.occ.addCylinder(x, y, z, dx, dy, dz, r, tag=-1, angle=2*pi)

            Add a cylinder in the OpenCASCADE CAD representation, defined by the center
            (`x', `y', `z') of its first circular face, the 3 components (`dx', `dy',
            `dz') of the vector defining its axis and its radius `r'. The optional
            `angle' argument defines the angular opening (from 0 to 2*Pi). If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the cylinder.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddCylinder(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_double(r),
                c_int(tag),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_cylinder = addCylinder

        @staticmethod
        def addCone(x, y, z, dx, dy, dz, r1, r2, tag=-1, angle=2*pi):
            """
            gmsh.model.occ.addCone(x, y, z, dx, dy, dz, r1, r2, tag=-1, angle=2*pi)

            Add a cone in the OpenCASCADE CAD representation, defined by the center
            (`x', `y', `z') of its first circular face, the 3 components of the vector
            (`dx', `dy', `dz') defining its axis and the two radii `r1' and `r2' of the
            faces (these radii can be zero). If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. `angle' defines
            the optional angular opening (from 0 to 2*Pi). Return the tag of the cone.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddCone(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_double(r1),
                c_double(r2),
                c_int(tag),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_cone = addCone

        @staticmethod
        def addWedge(x, y, z, dx, dy, dz, tag=-1, ltx=0.):
            """
            gmsh.model.occ.addWedge(x, y, z, dx, dy, dz, tag=-1, ltx=0.)

            Add a right angular wedge in the OpenCASCADE CAD representation, defined by
            the right-angle point (`x', `y', `z') and the 3 extends along the x-, y-
            and z-axes (`dx', `dy', `dz'). If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. The optional
            argument `ltx' defines the top extent along the x-axis. Return the tag of
            the wedge.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddWedge(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_int(tag),
                c_double(ltx),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_wedge = addWedge

        @staticmethod
        def addTorus(x, y, z, r1, r2, tag=-1, angle=2*pi):
            """
            gmsh.model.occ.addTorus(x, y, z, r1, r2, tag=-1, angle=2*pi)

            Add a torus in the OpenCASCADE CAD representation, defined by its center
            (`x', `y', `z') and its 2 radii `r' and `r2'. If `tag' is positive, set the
            tag explicitly; otherwise a new tag is selected automatically. The optional
            argument `angle' defines the angular opening (from 0 to 2*Pi). Return the
            tag of the wedge.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccAddTorus(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(r1),
                c_double(r2),
                c_int(tag),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        add_torus = addTorus

        @staticmethod
        def addThruSections(wireTags, tag=-1, makeSolid=True, makeRuled=False, maxDegree=-1):
            """
            gmsh.model.occ.addThruSections(wireTags, tag=-1, makeSolid=True, makeRuled=False, maxDegree=-1)

            Add a volume (if the optional argument `makeSolid' is set) or surfaces in
            the OpenCASCADE CAD representation, defined through the open or closed
            wires `wireTags'. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. The new entities are returned in
            `outDimTags'. If the optional argument `makeRuled' is set, the surfaces
            created on the boundary are forced to be ruled surfaces. If `maxDegree' is
            positive, set the maximal degree of resulting surface.

            Return `outDimTags'.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccAddThruSections(
                api_wireTags_, api_wireTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(tag),
                c_int(bool(makeSolid)),
                c_int(bool(makeRuled)),
                c_int(maxDegree),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
        add_thru_sections = addThruSections

        @staticmethod
        def addThickSolid(volumeTag, excludeSurfaceTags, offset, tag=-1):
            """
            gmsh.model.occ.addThickSolid(volumeTag, excludeSurfaceTags, offset, tag=-1)

            Add a hollowed volume in the OpenCASCADE CAD representation, built from an
            initial volume `volumeTag' and a set of faces from this volume
            `excludeSurfaceTags', which are to be removed. The remaining faces of the
            volume become the walls of the hollowed solid, with thickness `offset'. If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically.

            Return `outDimTags'.
            """
            api_excludeSurfaceTags_, api_excludeSurfaceTags_n_ = _ivectorint(excludeSurfaceTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccAddThickSolid(
                c_int(volumeTag),
                api_excludeSurfaceTags_, api_excludeSurfaceTags_n_,
                c_double(offset),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
        add_thick_solid = addThickSolid

        @staticmethod
        def extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.occ.extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False)

            Extrude the entities `dimTags' in the OpenCASCADE CAD representation, using
            a translation along (`dx', `dy', `dz'). Return extruded entities in
            `outDimTags'. If `numElements' is not empty, also extrude the mesh: the
            entries in `numElements' give the number of elements in each layer. If
            `height' is not empty, it provides the (cumulative) height of the different
            layers, normalized to 1. If `recombine' is set, recombine the mesh in the
            layers.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelOccExtrude(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.occ.revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False)

            Extrude the entities `dimTags' in the OpenCASCADE CAD representation, using
            a rotation of `angle' radians around the axis of revolution defined by the
            point (`x', `y', `z') and the direction (`ax', `ay', `az'). Return extruded
            entities in `outDimTags'. If `numElements' is not empty, also extrude the
            mesh: the entries in `numElements' give the number of elements in each
            layer. If `height' is not empty, it provides the (cumulative) height of the
            different layers, normalized to 1. When the mesh is extruded the angle
            should be strictly smaller than 2*Pi. If `recombine' is set, recombine the
            mesh in the layers.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelOccRevolve(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def addPipe(dimTags, wireTag, trihedron=""):
            """
            gmsh.model.occ.addPipe(dimTags, wireTag, trihedron="")

            Add a pipe in the OpenCASCADE CAD representation, by extruding the entities
            `dimTags' along the wire `wireTag'. The type of sweep can be specified with
            `trihedron' (possible values: "DiscreteTrihedron", "CorrectedFrenet",
            "Fixed", "Frenet", "ConstantNormal", "Darboux", "GuideAC", "GuidePlan",
            "GuideACWithContact", "GuidePlanWithContact"). If `trihedron' is not
            provided, "DiscreteTrihedron" is assumed. Return the pipe in `outDimTags'.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccAddPipe(
                api_dimTags_, api_dimTags_n_,
                c_int(wireTag),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_char_p(trihedron.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
        add_pipe = addPipe

        @staticmethod
        def fillet(volumeTags, curveTags, radii, removeVolume=True):
            """
            gmsh.model.occ.fillet(volumeTags, curveTags, radii, removeVolume=True)

            Fillet the volumes `volumeTags' on the curves `curveTags' with radii
            `radii'. The `radii' vector can either contain a single radius, as many
            radii as `curveTags', or twice as many as `curveTags' (in which case
            different radii are provided for the begin and end points of the curves).
            Return the filleted entities in `outDimTags'. Remove the original volume if
            `removeVolume' is set.

            Return `outDimTags'.
            """
            api_volumeTags_, api_volumeTags_n_ = _ivectorint(volumeTags)
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            api_radii_, api_radii_n_ = _ivectordouble(radii)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccFillet(
                api_volumeTags_, api_volumeTags_n_,
                api_curveTags_, api_curveTags_n_,
                api_radii_, api_radii_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(bool(removeVolume)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def chamfer(volumeTags, curveTags, surfaceTags, distances, removeVolume=True):
            """
            gmsh.model.occ.chamfer(volumeTags, curveTags, surfaceTags, distances, removeVolume=True)

            Chamfer the volumes `volumeTags' on the curves `curveTags' with distances
            `distances' measured on surfaces `surfaceTags'. The `distances' vector can
            either contain a single distance, as many distances as `curveTags' and
            `surfaceTags', or twice as many as `curveTags' and `surfaceTags' (in which
            case the first in each pair is measured on the corresponding surface in
            `surfaceTags', the other on the other adjacent surface). Return the
            chamfered entities in `outDimTags'. Remove the original volume if
            `removeVolume' is set.

            Return `outDimTags'.
            """
            api_volumeTags_, api_volumeTags_n_ = _ivectorint(volumeTags)
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            api_surfaceTags_, api_surfaceTags_n_ = _ivectorint(surfaceTags)
            api_distances_, api_distances_n_ = _ivectordouble(distances)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccChamfer(
                api_volumeTags_, api_volumeTags_n_,
                api_curveTags_, api_curveTags_n_,
                api_surfaceTags_, api_surfaceTags_n_,
                api_distances_, api_distances_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(bool(removeVolume)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def fuse(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.fuse(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean union (the fusion) of the entities `objectDimTags' and
            `toolDimTags' in the OpenCASCADE CAD representation. Return the resulting
            entities in `outDimTags'. If `tag' is positive, try to set the tag
            explicitly (only valid if the boolean operation results in a single
            entity). Remove the object if `removeObject' is set. Remove the tool if
            `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccFuse(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def intersect(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.intersect(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean intersection (the common parts) of the entities
            `objectDimTags' and `toolDimTags' in the OpenCASCADE CAD representation.
            Return the resulting entities in `outDimTags'. If `tag' is positive, try to
            set the tag explicitly (only valid if the boolean operation results in a
            single entity). Remove the object if `removeObject' is set. Remove the tool
            if `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccIntersect(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def cut(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.cut(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean difference between the entities `objectDimTags' and
            `toolDimTags' in the OpenCASCADE CAD representation. Return the resulting
            entities in `outDimTags'. If `tag' is positive, try to set the tag
            explicitly (only valid if the boolean operation results in a single
            entity). Remove the object if `removeObject' is set. Remove the tool if
            `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccCut(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def fragment(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.fragment(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean fragments (general fuse) resulting from the
            intersection of the entities `objectDimTags' and `toolDimTags' in the
            OpenCASCADE CAD representation, making all iterfaces conformal. When
            applied to entities of different dimensions, the lower dimensional entities
            will be automatically embedded in the higher dimensional entities if they
            are not on their boundary. Return the resulting entities in `outDimTags'.
            If `tag' is positive, try to set the tag explicitly (only valid if the
            boolean operation results in a single entity). Remove the object if
            `removeObject' is set. Remove the tool if `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccFragment(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def translate(dimTags, dx, dy, dz):
            """
            gmsh.model.occ.translate(dimTags, dx, dy, dz)

            Translate the entities `dimTags' in the OpenCASCADE CAD representation
            along (`dx', `dy', `dz').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccTranslate(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def rotate(dimTags, x, y, z, ax, ay, az, angle):
            """
            gmsh.model.occ.rotate(dimTags, x, y, z, ax, ay, az, angle)

            Rotate the entities `dimTags' in the OpenCASCADE CAD representation by
            `angle' radians around the axis of revolution defined by the point (`x',
            `y', `z') and the direction (`ax', `ay', `az').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccRotate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def dilate(dimTags, x, y, z, a, b, c):
            """
            gmsh.model.occ.dilate(dimTags, x, y, z, a, b, c)

            Scale the entities `dimTags' in the OpenCASCADE CAD representation by
            factors `a', `b' and `c' along the three coordinate axes; use (`x', `y',
            `z') as the center of the homothetic transformation.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccDilate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(a),
                c_double(b),
                c_double(c),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def mirror(dimTags, a, b, c, d):
            """
            gmsh.model.occ.mirror(dimTags, a, b, c, d)

            Mirror the entities `dimTags' in the OpenCASCADE CAD representation, with
            respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccMirror(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def symmetrize(dimTags, a, b, c, d):
            """
            gmsh.model.occ.symmetrize(dimTags, a, b, c, d)

            Mirror the entities `dimTags' in the OpenCASCADE CAD representation, with
            respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0.
            (This is a synonym for `mirror', which will be deprecated in a future
            release.)
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccSymmetrize(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def affineTransform(dimTags, a):
            """
            gmsh.model.occ.affineTransform(dimTags, a)

            Apply a general affine transformation matrix `a' (16 entries of a 4x4
            matrix, by row; only the 12 first can be provided for convenience) to the
            entities `dimTags' in the OpenCASCADE CAD representation.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_a_, api_a_n_ = _ivectordouble(a)
            ierr = c_int()
            lib.gmshModelOccAffineTransform(
                api_dimTags_, api_dimTags_n_,
                api_a_, api_a_n_,
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        affine_transform = affineTransform

        @staticmethod
        def copy(dimTags):
            """
            gmsh.model.occ.copy(dimTags)

            Copy the entities `dimTags' in the OpenCASCADE CAD representation; the new
            entities are returned in `outDimTags'.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccCopy(
                api_dimTags_, api_dimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def remove(dimTags, recursive=False):
            """
            gmsh.model.occ.remove(dimTags, recursive=False)

            Remove the entities `dimTags' in the OpenCASCADE CAD representation. If
            `recursive' is true, remove all the entities on their boundaries, down to
            dimension 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccRemove(
                api_dimTags_, api_dimTags_n_,
                c_int(bool(recursive)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())

        @staticmethod
        def removeAllDuplicates():
            """
            gmsh.model.occ.removeAllDuplicates()

            Remove all duplicate entities in the OpenCASCADE CAD representation
            (different entities at the same geometrical location) after intersecting
            (using boolean fragments) all highest dimensional entities.
            """
            ierr = c_int()
            lib.gmshModelOccRemoveAllDuplicates(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        remove_all_duplicates = removeAllDuplicates

        @staticmethod
        def healShapes(dimTags=[], tolerance=1e-8, fixDegenerated=True, fixSmallEdges=True, fixSmallFaces=True, sewFaces=True, makeSolids=True):
            """
            gmsh.model.occ.healShapes(dimTags=[], tolerance=1e-8, fixDegenerated=True, fixSmallEdges=True, fixSmallFaces=True, sewFaces=True, makeSolids=True)

            Apply various healing procedures to the entities `dimTags' (or to all the
            entities in the model if `dimTags' is empty) in the OpenCASCADE CAD
            representation. Return the healed entities in `outDimTags'. Available
            healing options are listed in the Gmsh reference manual.

            Return `outDimTags'.
            """
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccHealShapes(
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_dimTags_, api_dimTags_n_,
                c_double(tolerance),
                c_int(bool(fixDegenerated)),
                c_int(bool(fixSmallEdges)),
                c_int(bool(fixSmallFaces)),
                c_int(bool(sewFaces)),
                c_int(bool(makeSolids)),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
        heal_shapes = healShapes

        @staticmethod
        def importShapes(fileName, highestDimOnly=True, format=""):
            """
            gmsh.model.occ.importShapes(fileName, highestDimOnly=True, format="")

            Import BREP, STEP or IGES shapes from the file `fileName' in the
            OpenCASCADE CAD representation. The imported entities are returned in
            `outDimTags'. If the optional argument `highestDimOnly' is set, only import
            the highest dimensional entities in the file. The optional argument
            `format' can be used to force the format of the file (currently "brep",
            "step" or "iges").

            Return `outDimTags'.
            """
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccImportShapes(
                c_char_p(fileName.encode()),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(bool(highestDimOnly)),
                c_char_p(format.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)
        import_shapes = importShapes

        @staticmethod
        def getEntities(dim=-1):
            """
            gmsh.model.occ.getEntities(dim=-1)

            Get all the OpenCASCADE entities. If `dim' is >= 0, return only the
            entities of the specified dimension (e.g. points if `dim' == 0). The
            entities are returned as a vector of (dim, tag) integer pairs.

            Return `dimTags'.
            """
            api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccGetEntities(
                byref(api_dimTags_), byref(api_dimTags_n_),
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_dimTags_, api_dimTags_n_.value)
        get_entities = getEntities

        @staticmethod
        def getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim=-1):
            """
            gmsh.model.occ.getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim=-1)

            Get the OpenCASCADE entities in the bounding box defined by the two points
            (`xmin', `ymin', `zmin') and (`xmax', `ymax', `zmax'). If `dim' is >= 0,
            return only the entities of the specified dimension (e.g. points if `dim'
            == 0).

            Return `tags'.
            """
            api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccGetEntitiesInBoundingBox(
                c_double(xmin),
                c_double(ymin),
                c_double(zmin),
                c_double(xmax),
                c_double(ymax),
                c_double(zmax),
                byref(api_tags_), byref(api_tags_n_),
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectorpair(api_tags_, api_tags_n_.value)
        get_entities_in_bounding_box = getEntitiesInBoundingBox

        @staticmethod
        def getBoundingBox(dim, tag):
            """
            gmsh.model.occ.getBoundingBox(dim, tag)

            Get the bounding box (`xmin', `ymin', `zmin'), (`xmax', `ymax', `zmax') of
            the OpenCASCADE entity of dimension `dim' and tag `tag'.

            Return `xmin', `ymin', `zmin', `xmax', `ymax', `zmax'.
            """
            api_xmin_ = c_double()
            api_ymin_ = c_double()
            api_zmin_ = c_double()
            api_xmax_ = c_double()
            api_ymax_ = c_double()
            api_zmax_ = c_double()
            ierr = c_int()
            lib.gmshModelOccGetBoundingBox(
                c_int(dim),
                c_int(tag),
                byref(api_xmin_),
                byref(api_ymin_),
                byref(api_zmin_),
                byref(api_xmax_),
                byref(api_ymax_),
                byref(api_zmax_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_xmin_.value,
                api_ymin_.value,
                api_zmin_.value,
                api_xmax_.value,
                api_ymax_.value,
                api_zmax_.value)
        get_bounding_box = getBoundingBox

        @staticmethod
        def getMass(dim, tag):
            """
            gmsh.model.occ.getMass(dim, tag)

            Get the mass of the OpenCASCADE entity of dimension `dim' and tag `tag'.

            Return `mass'.
            """
            api_mass_ = c_double()
            ierr = c_int()
            lib.gmshModelOccGetMass(
                c_int(dim),
                c_int(tag),
                byref(api_mass_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_mass_.value
        get_mass = getMass

        @staticmethod
        def getCenterOfMass(dim, tag):
            """
            gmsh.model.occ.getCenterOfMass(dim, tag)

            Get the center of mass of the OpenCASCADE entity of dimension `dim' and tag
            `tag'.

            Return `x', `y', `z'.
            """
            api_x_ = c_double()
            api_y_ = c_double()
            api_z_ = c_double()
            ierr = c_int()
            lib.gmshModelOccGetCenterOfMass(
                c_int(dim),
                c_int(tag),
                byref(api_x_),
                byref(api_y_),
                byref(api_z_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return (
                api_x_.value,
                api_y_.value,
                api_z_.value)
        get_center_of_mass = getCenterOfMass

        @staticmethod
        def getMatrixOfInertia(dim, tag):
            """
            gmsh.model.occ.getMatrixOfInertia(dim, tag)

            Get the matrix of inertia (by row) of the OpenCASCADE entity of dimension
            `dim' and tag `tag'.

            Return `mat'.
            """
            api_mat_, api_mat_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccGetMatrixOfInertia(
                c_int(dim),
                c_int(tag),
                byref(api_mat_), byref(api_mat_n_),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return _ovectordouble(api_mat_, api_mat_n_.value)
        get_matrix_of_inertia = getMatrixOfInertia

        @staticmethod
        def getMaxTag(dim):
            """
            gmsh.model.occ.getMaxTag(dim)

            Get the maximum tag of entities of dimension `dim' in the OpenCASCADE CAD
            representation.

            Return an integer value.
            """
            ierr = c_int()
            api_result_ = lib.gmshModelOccGetMaxTag(
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
            return api_result_
        get_max_tag = getMaxTag

        @staticmethod
        def setMaxTag(dim, maxTag):
            """
            gmsh.model.occ.setMaxTag(dim, maxTag)

            Set the maximum tag `maxTag' for entities of dimension `dim' in the
            OpenCASCADE CAD representation.
            """
            ierr = c_int()
            lib.gmshModelOccSetMaxTag(
                c_int(dim),
                c_int(maxTag),
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())
        set_max_tag = setMaxTag

        @staticmethod
        def synchronize():
            """
            gmsh.model.occ.synchronize()

            Synchronize the OpenCASCADE CAD representation with the current Gmsh model.
            This can be called at any time, but since it involves a non trivial amount
            of processing, the number of synchronization points should normally be
            minimized. Without synchronization the entities in the OpenCASCADE CAD
            representation are not available to any function outside of the OpenCASCADE
            CAD kernel functions.
            """
            ierr = c_int()
            lib.gmshModelOccSynchronize(
                byref(ierr))
            if ierr.value != 0:
                raise Exception(logger.getLastError())


        class mesh:
            """
            OpenCASCADE CAD kernel meshing constraints
            """

            @staticmethod
            def setSize(dimTags, size):
                """
                gmsh.model.occ.mesh.setSize(dimTags, size)

                Set a mesh size constraint on the entities `dimTags' in the OpenCASCADE CAD
                representation. Currently only entities of dimension 0 (points) are
                handled.
                """
                api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
                ierr = c_int()
                lib.gmshModelOccMeshSetSize(
                    api_dimTags_, api_dimTags_n_,
                    c_double(size),
                    byref(ierr))
                if ierr.value != 0:
                    raise Exception(logger.getLastError())
            set_size = setSize


class view:
    """
    Post-processing view functions
    """

    @staticmethod
    def add(name, tag=-1):
        """
        gmsh.view.add(name, tag=-1)

        Add a new post-processing view, with name `name'. If `tag' is positive use
        it (and remove the view with that tag if it already exists), otherwise
        associate a new tag. Return the view tag.

        Return an integer value.
        """
        ierr = c_int()
        api_result_ = lib.gmshViewAdd(
            c_char_p(name.encode()),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_

    @staticmethod
    def remove(tag):
        """
        gmsh.view.remove(tag)

        Remove the view with tag `tag'.
        """
        ierr = c_int()
        lib.gmshViewRemove(
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def getIndex(tag):
        """
        gmsh.view.getIndex(tag)

        Get the index of the view with tag `tag' in the list of currently loaded
        views. This dynamic index (it can change when views are removed) is used to
        access view options.

        Return an integer value.
        """
        ierr = c_int()
        api_result_ = lib.gmshViewGetIndex(
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    get_index = getIndex

    @staticmethod
    def getTags():
        """
        gmsh.view.getTags()

        Get the tags of all views.

        Return `tags'.
        """
        api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshViewGetTags(
            byref(api_tags_), byref(api_tags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorint(api_tags_, api_tags_n_.value)
    get_tags = getTags

    @staticmethod
    def addModelData(tag, step, modelName, dataType, tags, data, time=0., numComponents=-1, partition=0):
        """
        gmsh.view.addModelData(tag, step, modelName, dataType, tags, data, time=0., numComponents=-1, partition=0)

        Add model-based post-processing data to the view with tag `tag'.
        `modelName' identifies the model the data is attached to. `dataType'
        specifies the type of data, currently either "NodeData", "ElementData" or
        "ElementNodeData". `step' specifies the identifier (>= 0) of the data in a
        sequence. `tags' gives the tags of the nodes or elements in the mesh to
        which the data is associated. `data' is a vector of the same length as
        `tags': each entry is the vector of double precision numbers representing
        the data associated with the corresponding tag. The optional `time'
        argument associate a time value with the data. `numComponents' gives the
        number of data components (1 for scalar data, 3 for vector data, etc.) per
        entity; if negative, it is automatically inferred (when possible) from the
        input data. `partition' allows to specify data in several sub-sets.
        """
        api_tags_, api_tags_n_ = _ivectorsize(tags)
        api_data_, api_data_n_, api_data_nn_ = _ivectorvectordouble(data)
        ierr = c_int()
        lib.gmshViewAddModelData(
            c_int(tag),
            c_int(step),
            c_char_p(modelName.encode()),
            c_char_p(dataType.encode()),
            api_tags_, api_tags_n_,
            api_data_, api_data_n_, api_data_nn_,
            c_double(time),
            c_int(numComponents),
            c_int(partition),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    add_model_data = addModelData

    @staticmethod
    def addHomogeneousModelData(tag, step, modelName, dataType, tags, data, time=0., numComponents=-1, partition=0):
        """
        gmsh.view.addHomogeneousModelData(tag, step, modelName, dataType, tags, data, time=0., numComponents=-1, partition=0)

        Add homogeneous model-based post-processing data to the view with tag
        `tag'. The arguments have the same meaning as in `addModelData', except
        that `data' is supposed to be homogeneous and is thus flattened in a single
        vector. For data types that can lead to different data sizes per tag (like
        "ElementNodeData"), the data should be padded.
        """
        api_tags_, api_tags_n_ = _ivectorsize(tags)
        api_data_, api_data_n_ = _ivectordouble(data)
        ierr = c_int()
        lib.gmshViewAddHomogeneousModelData(
            c_int(tag),
            c_int(step),
            c_char_p(modelName.encode()),
            c_char_p(dataType.encode()),
            api_tags_, api_tags_n_,
            api_data_, api_data_n_,
            c_double(time),
            c_int(numComponents),
            c_int(partition),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    add_homogeneous_model_data = addHomogeneousModelData

    @staticmethod
    def getModelData(tag, step):
        """
        gmsh.view.getModelData(tag, step)

        Get model-based post-processing data from the view with tag `tag' at step
        `step'. Return the `data' associated to the nodes or the elements with tags
        `tags', as well as the `dataType' and the number of components
        `numComponents'.

        Return `dataType', `tags', `data', `time', `numComponents'.
        """
        api_dataType_ = c_char_p()
        api_tags_, api_tags_n_ = POINTER(c_size_t)(), c_size_t()
        api_data_, api_data_n_, api_data_nn_ = POINTER(POINTER(c_double))(), POINTER(c_size_t)(), c_size_t()
        api_time_ = c_double()
        api_numComponents_ = c_int()
        ierr = c_int()
        lib.gmshViewGetModelData(
            c_int(tag),
            c_int(step),
            byref(api_dataType_),
            byref(api_tags_), byref(api_tags_n_),
            byref(api_data_), byref(api_data_n_), byref(api_data_nn_),
            byref(api_time_),
            byref(api_numComponents_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ostring(api_dataType_),
            _ovectorsize(api_tags_, api_tags_n_.value),
            _ovectorvectordouble(api_data_, api_data_n_, api_data_nn_),
            api_time_.value,
            api_numComponents_.value)
    get_model_data = getModelData

    @staticmethod
    def getHomogeneousModelData(tag, step):
        """
        gmsh.view.getHomogeneousModelData(tag, step)

        Get homogeneous model-based post-processing data from the view with tag
        `tag' at step `step'. The arguments have the same meaning as in
        `getModelData', except that `data' is returned flattened in a single
        vector, with the appropriate padding if necessary.

        Return `dataType', `tags', `data', `time', `numComponents'.
        """
        api_dataType_ = c_char_p()
        api_tags_, api_tags_n_ = POINTER(c_size_t)(), c_size_t()
        api_data_, api_data_n_ = POINTER(c_double)(), c_size_t()
        api_time_ = c_double()
        api_numComponents_ = c_int()
        ierr = c_int()
        lib.gmshViewGetHomogeneousModelData(
            c_int(tag),
            c_int(step),
            byref(api_dataType_),
            byref(api_tags_), byref(api_tags_n_),
            byref(api_data_), byref(api_data_n_),
            byref(api_time_),
            byref(api_numComponents_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ostring(api_dataType_),
            _ovectorsize(api_tags_, api_tags_n_.value),
            _ovectordouble(api_data_, api_data_n_.value),
            api_time_.value,
            api_numComponents_.value)
    get_homogeneous_model_data = getHomogeneousModelData

    @staticmethod
    def addListData(tag, dataType, numEle, data):
        """
        gmsh.view.addListData(tag, dataType, numEle, data)

        Add list-based post-processing data to the view with tag `tag'. List-based
        datasets are independent from any model and any mesh. `dataType' identifies
        the data by concatenating the field type ("S" for scalar, "V" for vector,
        "T" for tensor) and the element type ("P" for point, "L" for line, "T" for
        triangle, "S" for tetrahedron, "I" for prism, "H" for hexaHedron, "Y" for
        pyramid). For example `dataType' should be "ST" for a scalar field on
        triangles. `numEle' gives the number of elements in the data. `data'
        contains the data for the `numEle' elements, concatenated, with node
        coordinates followed by values per node, repeated for each step: [e1x1,
        ..., e1xn, e1y1, ..., e1yn, e1z1, ..., e1zn, e1v1..., e1vN, e2x1, ...].
        """
        api_data_, api_data_n_ = _ivectordouble(data)
        ierr = c_int()
        lib.gmshViewAddListData(
            c_int(tag),
            c_char_p(dataType.encode()),
            c_int(numEle),
            api_data_, api_data_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    add_list_data = addListData

    @staticmethod
    def getListData(tag):
        """
        gmsh.view.getListData(tag)

        Get list-based post-processing data from the view with tag `tag'. Return
        the types `dataTypes', the number of elements `numElements' for each data
        type and the `data' for each data type.

        Return `dataType', `numElements', `data'.
        """
        api_dataType_, api_dataType_n_ = POINTER(POINTER(c_char))(), c_size_t()
        api_numElements_, api_numElements_n_ = POINTER(c_int)(), c_size_t()
        api_data_, api_data_n_, api_data_nn_ = POINTER(POINTER(c_double))(), POINTER(c_size_t)(), c_size_t()
        ierr = c_int()
        lib.gmshViewGetListData(
            c_int(tag),
            byref(api_dataType_), byref(api_dataType_n_),
            byref(api_numElements_), byref(api_numElements_n_),
            byref(api_data_), byref(api_data_n_), byref(api_data_nn_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ovectorstring(api_dataType_, api_dataType_n_.value),
            _ovectorint(api_numElements_, api_numElements_n_.value),
            _ovectorvectordouble(api_data_, api_data_n_, api_data_nn_))
    get_list_data = getListData

    @staticmethod
    def addListDataString(tag, coord, data, style=[]):
        """
        gmsh.view.addListDataString(tag, coord, data, style=[])

        Add a string to a list-based post-processing view with tag `tag'. If
        `coord' contains 3 coordinates the string is positioned in the 3D model
        space ("3D string"); if it contains 2 coordinates it is positioned in the
        2D graphics viewport ("2D string"). `data' contains one or more (for
        multistep views) strings. `style' contains key-value pairs of styling
        parameters, concatenated. Available keys are "Font" (possible values:
        "Times-Roman", "Times-Bold", "Times-Italic", "Times-BoldItalic",
        "Helvetica", "Helvetica-Bold", "Helvetica-Oblique", "Helvetica-
        BoldOblique", "Courier", "Courier-Bold", "Courier-Oblique", "Courier-
        BoldOblique", "Symbol", "ZapfDingbats", "Screen"), "FontSize" and "Align"
        (possible values: "Left" or "BottomLeft", "Center" or "BottomCenter",
        "Right" or "BottomRight", "TopLeft", "TopCenter", "TopRight", "CenterLeft",
        "CenterCenter", "CenterRight").
        """
        api_coord_, api_coord_n_ = _ivectordouble(coord)
        api_data_, api_data_n_ = _ivectorstring(data)
        api_style_, api_style_n_ = _ivectorstring(style)
        ierr = c_int()
        lib.gmshViewAddListDataString(
            c_int(tag),
            api_coord_, api_coord_n_,
            api_data_, api_data_n_,
            api_style_, api_style_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    add_list_data_string = addListDataString

    @staticmethod
    def getListDataStrings(tag, dim):
        """
        gmsh.view.getListDataStrings(tag, dim)

        Get list-based post-processing data strings (2D strings if `dim' = 2, 3D
        strings if `dim' = 3) from the view with tag `tag'. Return the coordinates
        in `coord', the strings in `data' and the styles in `style'.

        Return `coord', `data', `style'.
        """
        api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
        api_data_, api_data_n_ = POINTER(POINTER(c_char))(), c_size_t()
        api_style_, api_style_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshViewGetListDataStrings(
            c_int(tag),
            c_int(dim),
            byref(api_coord_), byref(api_coord_n_),
            byref(api_data_), byref(api_data_n_),
            byref(api_style_), byref(api_style_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            _ovectordouble(api_coord_, api_coord_n_.value),
            _ovectorstring(api_data_, api_data_n_.value),
            _ovectorstring(api_style_, api_style_n_.value))
    get_list_data_strings = getListDataStrings

    @staticmethod
    def setInterpolationMatrices(tag, type, d, coef, exp, dGeo=0, coefGeo=[], expGeo=[]):
        """
        gmsh.view.setInterpolationMatrices(tag, type, d, coef, exp, dGeo=0, coefGeo=[], expGeo=[])

        Set interpolation matrices for the element family `type' ("Line",
        "Triangle", "Quadrangle", "Tetrahedron", "Hexahedron", "Prism", "Pyramid")
        in the view `tag'. The approximation of the values over an element is
        written as a linear combination of `d' basis functions f_i(u, v, w) =
        sum_(j = 0, ..., `d' - 1) `coef'[i][j] u^`exp'[j][0] v^`exp'[j][1]
        w^`exp'[j][2], i = 0, ..., `d'-1, with u, v, w the coordinates in the
        reference element. The `coef' matrix (of size `d' x `d') and the `exp'
        matrix (of size `d' x 3) are stored as vectors, by row. If `dGeo' is
        positive, use `coefGeo' and `expGeo' to define the interpolation of the x,
        y, z coordinates of the element in terms of the u, v, w coordinates, in
        exactly the same way. If `d' < 0, remove the interpolation matrices.
        """
        api_coef_, api_coef_n_ = _ivectordouble(coef)
        api_exp_, api_exp_n_ = _ivectordouble(exp)
        api_coefGeo_, api_coefGeo_n_ = _ivectordouble(coefGeo)
        api_expGeo_, api_expGeo_n_ = _ivectordouble(expGeo)
        ierr = c_int()
        lib.gmshViewSetInterpolationMatrices(
            c_int(tag),
            c_char_p(type.encode()),
            c_int(d),
            api_coef_, api_coef_n_,
            api_exp_, api_exp_n_,
            c_int(dGeo),
            api_coefGeo_, api_coefGeo_n_,
            api_expGeo_, api_expGeo_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_interpolation_matrices = setInterpolationMatrices

    @staticmethod
    def addAlias(refTag, copyOptions=False, tag=-1):
        """
        gmsh.view.addAlias(refTag, copyOptions=False, tag=-1)

        Add a post-processing view as an `alias' of the reference view with tag
        `refTag'. If `copyOptions' is set, copy the options of the reference view.
        If `tag' is positive use it (and remove the view with that tag if it
        already exists), otherwise associate a new tag. Return the view tag.

        Return an integer value.
        """
        ierr = c_int()
        api_result_ = lib.gmshViewAddAlias(
            c_int(refTag),
            c_int(bool(copyOptions)),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    add_alias = addAlias

    @staticmethod
    def copyOptions(refTag, tag):
        """
        gmsh.view.copyOptions(refTag, tag)

        Copy the options from the view with tag `refTag' to the view with tag
        `tag'.
        """
        ierr = c_int()
        lib.gmshViewCopyOptions(
            c_int(refTag),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    copy_options = copyOptions

    @staticmethod
    def combine(what, how, remove=True, copyOptions=True):
        """
        gmsh.view.combine(what, how, remove=True, copyOptions=True)

        Combine elements (if `what' == "elements") or steps (if `what' == "steps")
        of all views (`how' == "all"), all visible views (`how' == "visible") or
        all views having the same name (`how' == "name"). Remove original views if
        `remove' is set.
        """
        ierr = c_int()
        lib.gmshViewCombine(
            c_char_p(what.encode()),
            c_char_p(how.encode()),
            c_int(bool(remove)),
            c_int(bool(copyOptions)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def probe(tag, x, y, z, step=-1, numComp=-1, gradient=False, tolerance=0., xElemCoord=[], yElemCoord=[], zElemCoord=[], dim=-1):
        """
        gmsh.view.probe(tag, x, y, z, step=-1, numComp=-1, gradient=False, tolerance=0., xElemCoord=[], yElemCoord=[], zElemCoord=[], dim=-1)

        Probe the view `tag' for its `value' at point (`x', `y', `z'). Return only
        the value at step `step' is `step' is positive. Return only values with
        `numComp' if `numComp' is positive. Return the gradient of the `value' if
        `gradient' is set. Probes with a geometrical tolerance (in the reference
        unit cube) of `tolerance' if `tolerance' is not zero. Return the result
        from the element described by its coordinates if `xElementCoord',
        `yElementCoord' and `zElementCoord' are provided. If `dim' is >= 0, return
        only elements of the specified dimension.

        Return `value'.
        """
        api_value_, api_value_n_ = POINTER(c_double)(), c_size_t()
        api_xElemCoord_, api_xElemCoord_n_ = _ivectordouble(xElemCoord)
        api_yElemCoord_, api_yElemCoord_n_ = _ivectordouble(yElemCoord)
        api_zElemCoord_, api_zElemCoord_n_ = _ivectordouble(zElemCoord)
        ierr = c_int()
        lib.gmshViewProbe(
            c_int(tag),
            c_double(x),
            c_double(y),
            c_double(z),
            byref(api_value_), byref(api_value_n_),
            c_int(step),
            c_int(numComp),
            c_int(bool(gradient)),
            c_double(tolerance),
            api_xElemCoord_, api_xElemCoord_n_,
            api_yElemCoord_, api_yElemCoord_n_,
            api_zElemCoord_, api_zElemCoord_n_,
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_value_, api_value_n_.value)

    @staticmethod
    def write(tag, fileName, append=False):
        """
        gmsh.view.write(tag, fileName, append=False)

        Write the view to a file `fileName'. The export format is determined by the
        file extension. Append to the file if `append' is set.
        """
        ierr = c_int()
        lib.gmshViewWrite(
            c_int(tag),
            c_char_p(fileName.encode()),
            c_int(bool(append)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def setVisibilityPerWindow(tag, value, windowIndex=0):
        """
        gmsh.view.setVisibilityPerWindow(tag, value, windowIndex=0)

        Set the global visibility of the view `tag' per window to `value', where
        `windowIndex' identifies the window in the window list.
        """
        ierr = c_int()
        lib.gmshViewSetVisibilityPerWindow(
            c_int(tag),
            c_int(value),
            c_int(windowIndex),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_visibility_per_window = setVisibilityPerWindow


class plugin:
    """
    Plugin functions
    """

    @staticmethod
    def setNumber(name, option, value):
        """
        gmsh.plugin.setNumber(name, option, value)

        Set the numerical option `option' to the value `value' for plugin `name'.
        """
        ierr = c_int()
        lib.gmshPluginSetNumber(
            c_char_p(name.encode()),
            c_char_p(option.encode()),
            c_double(value),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_number = setNumber

    @staticmethod
    def setString(name, option, value):
        """
        gmsh.plugin.setString(name, option, value)

        Set the string option `option' to the value `value' for plugin `name'.
        """
        ierr = c_int()
        lib.gmshPluginSetString(
            c_char_p(name.encode()),
            c_char_p(option.encode()),
            c_char_p(value.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_string = setString

    @staticmethod
    def run(name):
        """
        gmsh.plugin.run(name)

        Run the plugin `name'.
        """
        ierr = c_int()
        lib.gmshPluginRun(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())


class graphics:
    """
    Graphics functions
    """

    @staticmethod
    def draw():
        """
        gmsh.graphics.draw()

        Draw all the OpenGL scenes.
        """
        ierr = c_int()
        lib.gmshGraphicsDraw(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())


class fltk:
    """
    FLTK graphical user interface functions
    """

    @staticmethod
    def initialize():
        """
        gmsh.fltk.initialize()

        Create the FLTK graphical user interface. Can only be called in the main
        thread.
        """
        ierr = c_int()
        lib.gmshFltkInitialize(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def wait(time=-1.):
        """
        gmsh.fltk.wait(time=-1.)

        Wait at most `time' seconds for user interface events and return. If `time'
        < 0, wait indefinitely. First automatically create the user interface if it
        has not yet been initialized. Can only be called in the main thread.
        """
        ierr = c_int()
        lib.gmshFltkWait(
            c_double(time),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def update():
        """
        gmsh.fltk.update()

        Update the user interface (potentially creating new widgets and windows).
        First automatically create the user interface if it has not yet been
        initialized. Can only be called in the main thread: use `awake("update")'
        to trigger an update of the user interface from another thread.
        """
        ierr = c_int()
        lib.gmshFltkUpdate(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def awake(action=""):
        """
        gmsh.fltk.awake(action="")

        Awake the main user interface thread and process pending events, and
        optionally perform an action (currently the only `action' allowed is
        "update").
        """
        ierr = c_int()
        lib.gmshFltkAwake(
            c_char_p(action.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def lock():
        """
        gmsh.fltk.lock()

        Block the current thread until it can safely modify the user interface.
        """
        ierr = c_int()
        lib.gmshFltkLock(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def unlock():
        """
        gmsh.fltk.unlock()

        Release the lock that was set using lock.
        """
        ierr = c_int()
        lib.gmshFltkUnlock(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def run():
        """
        gmsh.fltk.run()

        Run the event loop of the graphical user interface, i.e. repeatedly call
        `wait()'. First automatically create the user interface if it has not yet
        been initialized. Can only be called in the main thread.
        """
        ierr = c_int()
        lib.gmshFltkRun(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def isAvailable():
        """
        gmsh.fltk.isAvailable()

        Check if the user interface is available (e.g. to detect if it has been
        closed).

        Return an integer value.
        """
        ierr = c_int()
        api_result_ = lib.gmshFltkIsAvailable(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    is_available = isAvailable

    @staticmethod
    def selectEntities(dim=-1):
        """
        gmsh.fltk.selectEntities(dim=-1)

        Select entities in the user interface. If `dim' is >= 0, return only the
        entities of the specified dimension (e.g. points if `dim' == 0).

        Return an integer value, `dimTags'.
        """
        api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        api_result_ = lib.gmshFltkSelectEntities(
            byref(api_dimTags_), byref(api_dimTags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_result_,
            _ovectorpair(api_dimTags_, api_dimTags_n_.value))
    select_entities = selectEntities

    @staticmethod
    def selectElements():
        """
        gmsh.fltk.selectElements()

        Select elements in the user interface.

        Return an integer value, `elementTags'.
        """
        api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
        ierr = c_int()
        api_result_ = lib.gmshFltkSelectElements(
            byref(api_elementTags_), byref(api_elementTags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_result_,
            _ovectorsize(api_elementTags_, api_elementTags_n_.value))
    select_elements = selectElements

    @staticmethod
    def selectViews():
        """
        gmsh.fltk.selectViews()

        Select views in the user interface.

        Return an integer value, `viewTags'.
        """
        api_viewTags_, api_viewTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        api_result_ = lib.gmshFltkSelectViews(
            byref(api_viewTags_), byref(api_viewTags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return (
            api_result_,
            _ovectorint(api_viewTags_, api_viewTags_n_.value))
    select_views = selectViews

    @staticmethod
    def splitCurrentWindow(how="v", ratio=0.5):
        """
        gmsh.fltk.splitCurrentWindow(how="v", ratio=0.5)

        Split the current window horizontally (if `how' = "h") or vertically (if
        `how' = "v"), using ratio `ratio'. If `how' = "u", restore a single window.
        """
        ierr = c_int()
        lib.gmshFltkSplitCurrentWindow(
            c_char_p(how.encode()),
            c_double(ratio),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    split_current_window = splitCurrentWindow

    @staticmethod
    def setCurrentWindow(windowIndex=0):
        """
        gmsh.fltk.setCurrentWindow(windowIndex=0)

        Set the current window by speficying its index (starting at 0) in the list
        of all windows. When new windows are created by splits, new windows are
        appended at the end of the list.
        """
        ierr = c_int()
        lib.gmshFltkSetCurrentWindow(
            c_int(windowIndex),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_current_window = setCurrentWindow

    @staticmethod
    def setStatusMessage(message, graphics=False):
        """
        gmsh.fltk.setStatusMessage(message, graphics=False)

        Set a status message in the current window. If `graphics' is set, display
        the message inside the graphic window instead of the status bar.
        """
        ierr = c_int()
        lib.gmshFltkSetStatusMessage(
            c_char_p(message.encode()),
            c_int(bool(graphics)),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_status_message = setStatusMessage

    @staticmethod
    def showContextWindow(dim, tag):
        """
        gmsh.fltk.showContextWindow(dim, tag)

        Show context window for the entity of dimension `dim' and tag `tag'.
        """
        ierr = c_int()
        lib.gmshFltkShowContextWindow(
            c_int(dim),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    show_context_window = showContextWindow

    @staticmethod
    def openTreeItem(name):
        """
        gmsh.fltk.openTreeItem(name)

        Open the `name' item in the menu tree.
        """
        ierr = c_int()
        lib.gmshFltkOpenTreeItem(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    open_tree_item = openTreeItem

    @staticmethod
    def closeTreeItem(name):
        """
        gmsh.fltk.closeTreeItem(name)

        Close the `name' item in the menu tree.
        """
        ierr = c_int()
        lib.gmshFltkCloseTreeItem(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    close_tree_item = closeTreeItem


class onelab:
    """
    ONELAB server functions
    """

    @staticmethod
    def set(data, format="json"):
        """
        gmsh.onelab.set(data, format="json")

        Set one or more parameters in the ONELAB database, encoded in `format'.
        """
        ierr = c_int()
        lib.gmshOnelabSet(
            c_char_p(data.encode()),
            c_char_p(format.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def get(name="", format="json"):
        """
        gmsh.onelab.get(name="", format="json")

        Get all the parameters (or a single one if `name' is specified) from the
        ONELAB database, encoded in `format'.

        Return `data'.
        """
        api_data_ = c_char_p()
        ierr = c_int()
        lib.gmshOnelabGet(
            byref(api_data_),
            c_char_p(name.encode()),
            c_char_p(format.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ostring(api_data_)

    @staticmethod
    def getNames(search=""):
        """
        gmsh.onelab.getNames(search="")

        Get the names of the parameters in the ONELAB database matching the
        `search' regular expression. If `search' is empty, return all the names.

        Return `names'.
        """
        api_names_, api_names_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshOnelabGetNames(
            byref(api_names_), byref(api_names_n_),
            c_char_p(search.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorstring(api_names_, api_names_n_.value)
    get_names = getNames

    @staticmethod
    def setNumber(name, value):
        """
        gmsh.onelab.setNumber(name, value)

        Set the value of the number parameter `name' in the ONELAB database. Create
        the parameter if it does not exist; update the value if the parameter
        exists.
        """
        api_value_, api_value_n_ = _ivectordouble(value)
        ierr = c_int()
        lib.gmshOnelabSetNumber(
            c_char_p(name.encode()),
            api_value_, api_value_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_number = setNumber

    @staticmethod
    def setString(name, value):
        """
        gmsh.onelab.setString(name, value)

        Set the value of the string parameter `name' in the ONELAB database. Create
        the parameter if it does not exist; update the value if the parameter
        exists.
        """
        api_value_, api_value_n_ = _ivectorstring(value)
        ierr = c_int()
        lib.gmshOnelabSetString(
            c_char_p(name.encode()),
            api_value_, api_value_n_,
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
    set_string = setString

    @staticmethod
    def getNumber(name):
        """
        gmsh.onelab.getNumber(name)

        Get the value of the number parameter `name' from the ONELAB database.
        Return an empty vector if the parameter does not exist.

        Return `value'.
        """
        api_value_, api_value_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshOnelabGetNumber(
            c_char_p(name.encode()),
            byref(api_value_), byref(api_value_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectordouble(api_value_, api_value_n_.value)
    get_number = getNumber

    @staticmethod
    def getString(name):
        """
        gmsh.onelab.getString(name)

        Get the value of the string parameter `name' from the ONELAB database.
        Return an empty vector if the parameter does not exist.

        Return `value'.
        """
        api_value_, api_value_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshOnelabGetString(
            c_char_p(name.encode()),
            byref(api_value_), byref(api_value_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorstring(api_value_, api_value_n_.value)
    get_string = getString

    @staticmethod
    def clear(name=""):
        """
        gmsh.onelab.clear(name="")

        Clear the ONELAB database, or remove a single parameter if `name' is given.
        """
        ierr = c_int()
        lib.gmshOnelabClear(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def run(name="", command=""):
        """
        gmsh.onelab.run(name="", command="")

        Run a ONELAB client. If `name' is provided, create a new ONELAB client with
        name `name' and executes `command'. If not, try to run a client that might
        be linked to the processed input files.
        """
        ierr = c_int()
        lib.gmshOnelabRun(
            c_char_p(name.encode()),
            c_char_p(command.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())


class logger:
    """
    Information logging functions
    """

    @staticmethod
    def write(message, level="info"):
        """
        gmsh.logger.write(message, level="info")

        Write a `message'. `level' can be "info", "warning" or "error".
        """
        ierr = c_int()
        lib.gmshLoggerWrite(
            c_char_p(message.encode()),
            c_char_p(level.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def start():
        """
        gmsh.logger.start()

        Start logging messages.
        """
        ierr = c_int()
        lib.gmshLoggerStart(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def get():
        """
        gmsh.logger.get()

        Get logged messages.

        Return `log'.
        """
        api_log_, api_log_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshLoggerGet(
            byref(api_log_), byref(api_log_n_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return _ovectorstring(api_log_, api_log_n_.value)

    @staticmethod
    def stop():
        """
        gmsh.logger.stop()

        Stop logging messages.
        """
        ierr = c_int()
        lib.gmshLoggerStop(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())

    @staticmethod
    def getWallTime():
        """
        gmsh.logger.getWallTime()

        Return wall clock time.

        Return a floating point value.
        """
        ierr = c_int()
        lib.gmshLoggerGetWallTime.restype = c_double
        api_result_ = lib.gmshLoggerGetWallTime(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    get_wall_time = getWallTime

    @staticmethod
    def getCpuTime():
        """
        gmsh.logger.getCpuTime()

        Return CPU time.

        Return a floating point value.
        """
        ierr = c_int()
        lib.gmshLoggerGetCpuTime.restype = c_double
        api_result_ = lib.gmshLoggerGetCpuTime(
            byref(ierr))
        if ierr.value != 0:
            raise Exception(logger.getLastError())
        return api_result_
    get_cpu_time = getCpuTime

    @staticmethod
    def getLastError():
        """
        gmsh.logger.getLastError()

        Return last error message, if any.

        Return `error'.
        """
        api_error_ = c_char_p()
        ierr = c_int()
        lib.gmshLoggerGetLastError(
            byref(api_error_),
            byref(ierr))
        if ierr.value != 0:
            raise Exception('Could not get last error')
        return _ostring(api_error_)
    get_last_error = getLastError
