/*
 * Gmsh - Copyright (C) 1997-2024 C. Geuzaine, J.-F. Remacle
 *
 * See the LICENSE.txt file in the Gmsh root directory for license information.
 * Please report all issues on https://gitlab.onelab.info/gmsh/gmsh/issues.
 */

#ifndef GMSHC_H
#define GMSHC_H

/*
 * Gmsh is an automatic three-dimensional finite element mesh generator with a
 * built-in CAD engine and post-processor. Its design goal is to provide a fast,
 * light and user-friendly meshing tool with parametric input and flexible
 * visualization capabilities. Gmsh is built around four modules (geometry, mesh,
 * solver and post-processing), which can be controlled with the graphical user
 * interface, from the command line, using text files written in Gmsh's own
 * scripting language (.geo files), or through the C++, C, Python, Julia and
 * Fortran application programming interface (API).
 *
 * This file defines the Gmsh C API (v4.12.1).
 *
 * Do not edit this file directly: it is automatically generated by `api/gen.py'.
 *
 * By design, the Gmsh C API is purely functional, and only uses elementary
 * C types. See `tutorials/c' and `examples/api' for tutorials and examples.
 */

#include <stddef.h>

#define GMSH_API_VERSION "4.12.1"
#define GMSH_API_VERSION_MAJOR 4
#define GMSH_API_VERSION_MINOR 12
#define GMSH_API_VERSION_PATCH 1

#if defined(GMSH_DLL)
#if defined(GMSH_DLL_EXPORT)
#define GMSH_API __declspec(dllexport)
#else
#define GMSH_API __declspec(dllimport)
#endif
#elif defined(__GNUC__)
#define GMSH_API __attribute__ ((visibility("default")))
#else
#define GMSH_API
#endif

/* All the functions in the Gmsh C API that return arrays allocate the
 * necessary memory with gmshMalloc(). These arrays should be deallocated
 * with gmshFree(). */
GMSH_API void gmshFree(void *p);
GMSH_API void *gmshMalloc(size_t n);

/* Initialize the Gmsh API. This must be called before any call to the other
 * functions in the API. If `argc' and `argv' (or just `argv' in Python or
 * Julia) are provided, they will be handled in the same way as the command
 * line arguments in the Gmsh app. If `readConfigFiles' is set, read system
 * Gmsh configuration files (gmshrc and gmsh-options). If `run' is set, run in
 * the same way as the Gmsh app, either interactively or in batch mode
 * depending on the command line arguments. If `run' is not set, initializing
 * the API sets the options "General.AbortOnError" to 2 and "General.Terminal"
 * to 1. */
GMSH_API void gmshInitialize(int argc, char ** argv,
                             const int readConfigFiles,
                             const int run,
                             int * ierr);

/* Return 1 if the Gmsh API is initialized, and 0 if not. */
GMSH_API int gmshIsInitialized(int * ierr);

/* Finalize the Gmsh API. This must be called when you are done using the Gmsh
 * API. */
GMSH_API void gmshFinalize(int * ierr);

/* Open a file. Equivalent to the `File->Open' menu in the Gmsh app. Handling
 * of the file depends on its extension and/or its contents: opening a file
 * with model data will create a new model. */
GMSH_API void gmshOpen(const char * fileName,
                       int * ierr);

/* Merge a file. Equivalent to the `File->Merge' menu in the Gmsh app.
 * Handling of the file depends on its extension and/or its contents. Merging
 * a file with model data will add the data to the current model. */
GMSH_API void gmshMerge(const char * fileName,
                        int * ierr);

/* Write a file. The export format is determined by the file extension. */
GMSH_API void gmshWrite(const char * fileName,
                        int * ierr);

/* Clear all loaded models and post-processing data, and add a new empty
 * model. */
GMSH_API void gmshClear(int * ierr);

/* Set a numerical option to `value'. `name' is of the form "Category.Option"
 * or "Category[num].Option". Available categories and options are listed in
 * the "Gmsh options" chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-options). */
GMSH_API void gmshOptionSetNumber(const char * name,
                                  const double value,
                                  int * ierr);

/* Get the `value' of a numerical option. `name' is of the form
 * "Category.Option" or "Category[num].Option". Available categories and
 * options are listed in the "Gmsh options" chapter of the Gmsh reference
 * manual (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-options). */
GMSH_API void gmshOptionGetNumber(const char * name,
                                  double * value,
                                  int * ierr);

/* Set a string option to `value'. `name' is of the form "Category.Option" or
 * "Category[num].Option". Available categories and options are listed in the
 * "Gmsh options" chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-options). */
GMSH_API void gmshOptionSetString(const char * name,
                                  const char * value,
                                  int * ierr);

/* Get the `value' of a string option. `name' is of the form "Category.Option"
 * or "Category[num].Option". Available categories and options are listed in
 * the "Gmsh options" chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-options). */
GMSH_API void gmshOptionGetString(const char * name,
                                  char ** value,
                                  int * ierr);

/* Set a color option to the RGBA value (`r', `g', `b', `a'), where where `r',
 * `g', `b' and `a' should be integers between 0 and 255. `name' is of the
 * form "Category.Color.Option" or "Category[num].Color.Option". Available
 * categories and options are listed in the "Gmsh options" chapter of the Gmsh
 * reference manual (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-options).
 * For conciseness "Color." can be ommitted in `name'. */
GMSH_API void gmshOptionSetColor(const char * name,
                                 const int r,
                                 const int g,
                                 const int b,
                                 const int a,
                                 int * ierr);

/* Get the `r', `g', `b', `a' value of a color option. `name' is of the form
 * "Category.Color.Option" or "Category[num].Color.Option". Available
 * categories and options are listed in the "Gmsh options" chapter of the Gmsh
 * reference manual (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-options).
 * For conciseness "Color." can be ommitted in `name'. */
GMSH_API void gmshOptionGetColor(const char * name,
                                 int * r,
                                 int * g,
                                 int * b,
                                 int * a,
                                 int * ierr);

/* Add a new model, with name `name', and set it as the current model. */
GMSH_API void gmshModelAdd(const char * name,
                           int * ierr);

/* Remove the current model. */
GMSH_API void gmshModelRemove(int * ierr);

/* List the names of all models. */
GMSH_API void gmshModelList(char *** names, size_t * names_n,
                            int * ierr);

/* Get the name of the current model. */
GMSH_API void gmshModelGetCurrent(char ** name,
                                  int * ierr);

/* Set the current model to the model with name `name'. If several models have
 * the same name, select the one that was added first. */
GMSH_API void gmshModelSetCurrent(const char * name,
                                  int * ierr);

/* Get the file name (if any) associated with the current model. A file name
 * is associated when a model is read from a file on disk. */
GMSH_API void gmshModelGetFileName(char ** fileName,
                                   int * ierr);

/* Set the file name associated with the current model. */
GMSH_API void gmshModelSetFileName(const char * fileName,
                                   int * ierr);

/* Get all the entities in the current model. A model entity is represented by
 * two integers: its dimension (dim == 0, 1, 2 or 3) and its tag (its unique,
 * strictly positive identifier). If `dim' is >= 0, return only the entities
 * of the specified dimension (e.g. points if `dim' == 0). The entities are
 * returned as a vector of (dim, tag) pairs. */
GMSH_API void gmshModelGetEntities(int ** dimTags, size_t * dimTags_n,
                                   const int dim,
                                   int * ierr);

/* Set the name of the entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelSetEntityName(const int dim,
                                     const int tag,
                                     const char * name,
                                     int * ierr);

/* Get the name of the entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetEntityName(const int dim,
                                     const int tag,
                                     char ** name,
                                     int * ierr);

/* Remove the entity name `name' from the current model. */
GMSH_API void gmshModelRemoveEntityName(const char * name,
                                        int * ierr);

/* Get all the physical groups in the current model. If `dim' is >= 0, return
 * only the entities of the specified dimension (e.g. physical points if `dim'
 * == 0). The entities are returned as a vector of (dim, tag) pairs. */
GMSH_API void gmshModelGetPhysicalGroups(int ** dimTags, size_t * dimTags_n,
                                         const int dim,
                                         int * ierr);

/* Get the tags of the model entities making up the physical group of
 * dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetEntitiesForPhysicalGroup(const int dim,
                                                   const int tag,
                                                   int ** tags, size_t * tags_n,
                                                   int * ierr);

/* Get the model entities (as a vector (dim, tag) pairs) making up the
 * physical group with name `name'. */
GMSH_API void gmshModelGetEntitiesForPhysicalName(const char * name,
                                                  int ** dimTags, size_t * dimTags_n,
                                                  int * ierr);

/* Get the tags of the physical groups (if any) to which the model entity of
 * dimension `dim' and tag `tag' belongs. */
GMSH_API void gmshModelGetPhysicalGroupsForEntity(const int dim,
                                                  const int tag,
                                                  int ** physicalTags, size_t * physicalTags_n,
                                                  int * ierr);

/* Add a physical group of dimension `dim', grouping the model entities with
 * tags `tags'. Return the tag of the physical group, equal to `tag' if `tag'
 * is positive, or a new tag if `tag' < 0. Set the name of the physical group
 * if `name' is not empty. */
GMSH_API int gmshModelAddPhysicalGroup(const int dim,
                                       const int * tags, const size_t tags_n,
                                       const int tag,
                                       const char * name,
                                       int * ierr);

/* Remove the physical groups `dimTags' (given as a vector of (dim, tag)
 * pairs) from the current model. If `dimTags' is empty, remove all groups. */
GMSH_API void gmshModelRemovePhysicalGroups(const int * dimTags, const size_t dimTags_n,
                                            int * ierr);

/* Set the name of the physical group of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelSetPhysicalName(const int dim,
                                       const int tag,
                                       const char * name,
                                       int * ierr);

/* Get the name of the physical group of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetPhysicalName(const int dim,
                                       const int tag,
                                       char ** name,
                                       int * ierr);

/* Remove the physical name `name' from the current model. */
GMSH_API void gmshModelRemovePhysicalName(const char * name,
                                          int * ierr);

/* Set the tag of the entity of dimension `dim' and tag `tag' to the new value
 * `newTag'. */
GMSH_API void gmshModelSetTag(const int dim,
                              const int tag,
                              const int newTag,
                              int * ierr);

/* Get the boundary of the model entities `dimTags', given as a vector of
 * (dim, tag) pairs. Return in `outDimTags' the boundary of the individual
 * entities (if `combined' is false) or the boundary of the combined
 * geometrical shape formed by all input entities (if `combined' is true).
 * Return tags multiplied by the sign of the boundary entity if `oriented' is
 * true. Apply the boundary operator recursively down to dimension 0 (i.e. to
 * points) if `recursive' is true. */
GMSH_API void gmshModelGetBoundary(const int * dimTags, const size_t dimTags_n,
                                   int ** outDimTags, size_t * outDimTags_n,
                                   const int combined,
                                   const int oriented,
                                   const int recursive,
                                   int * ierr);

/* Get the upward and downward adjacencies of the model entity of dimension
 * `dim' and tag `tag'. The `upward' vector returns the tags of adjacent
 * entities of dimension `dim' + 1; the `downward' vector returns the tags of
 * adjacent entities of dimension `dim' - 1. */
GMSH_API void gmshModelGetAdjacencies(const int dim,
                                      const int tag,
                                      int ** upward, size_t * upward_n,
                                      int ** downward, size_t * downward_n,
                                      int * ierr);

/* Get the model entities in the bounding box defined by the two points
 * (`xmin', `ymin', `zmin') and (`xmax', `ymax', `zmax'). If `dim' is >= 0,
 * return only the entities of the specified dimension (e.g. points if `dim'
 * == 0). */
GMSH_API void gmshModelGetEntitiesInBoundingBox(const double xmin,
                                                const double ymin,
                                                const double zmin,
                                                const double xmax,
                                                const double ymax,
                                                const double zmax,
                                                int ** dimTags, size_t * dimTags_n,
                                                const int dim,
                                                int * ierr);

/* Get the bounding box (`xmin', `ymin', `zmin'), (`xmax', `ymax', `zmax') of
 * the model entity of dimension `dim' and tag `tag'. If `dim' and `tag' are
 * negative, get the bounding box of the whole model. */
GMSH_API void gmshModelGetBoundingBox(const int dim,
                                      const int tag,
                                      double * xmin,
                                      double * ymin,
                                      double * zmin,
                                      double * xmax,
                                      double * ymax,
                                      double * zmax,
                                      int * ierr);

/* Return the geometrical dimension of the current model. */
GMSH_API int gmshModelGetDimension(int * ierr);

/* Add a discrete model entity (defined by a mesh) of dimension `dim' in the
 * current model. Return the tag of the new discrete entity, equal to `tag' if
 * `tag' is positive, or a new tag if `tag' < 0. `boundary' specifies the tags
 * of the entities on the boundary of the discrete entity, if any. Specifying
 * `boundary' allows Gmsh to construct the topology of the overall model. */
GMSH_API int gmshModelAddDiscreteEntity(const int dim,
                                        const int tag,
                                        const int * boundary, const size_t boundary_n,
                                        int * ierr);

/* Remove the entities `dimTags' (given as a vector of (dim, tag) pairs) of
 * the current model, provided that they are not on the boundary of (or
 * embedded in) higher-dimensional entities. If `recursive' is true, remove
 * all the entities on their boundaries, down to dimension 0. */
GMSH_API void gmshModelRemoveEntities(const int * dimTags, const size_t dimTags_n,
                                      const int recursive,
                                      int * ierr);

/* Get the type of the entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetType(const int dim,
                               const int tag,
                               char ** entityType,
                               int * ierr);

/* In a partitioned model, get the parent of the entity of dimension `dim' and
 * tag `tag', i.e. from which the entity is a part of, if any. `parentDim' and
 * `parentTag' are set to -1 if the entity has no parent. */
GMSH_API void gmshModelGetParent(const int dim,
                                 const int tag,
                                 int * parentDim,
                                 int * parentTag,
                                 int * ierr);

/* Return the number of partitions in the model. */
GMSH_API int gmshModelGetNumberOfPartitions(int * ierr);

/* In a partitioned model, return the tags of the partition(s) to which the
 * entity belongs. */
GMSH_API void gmshModelGetPartitions(const int dim,
                                     const int tag,
                                     int ** partitions, size_t * partitions_n,
                                     int * ierr);

/* Evaluate the parametrization of the entity of dimension `dim' and tag `tag'
 * at the parametric coordinates `parametricCoord'. Only valid for `dim' equal
 * to 0 (with empty `parametricCoord'), 1 (with `parametricCoord' containing
 * parametric coordinates on the curve) or 2 (with `parametricCoord'
 * containing u, v parametric coordinates on the surface, concatenated: [p1u,
 * p1v, p2u, ...]). Return x, y, z coordinates in `coord', concatenated: [p1x,
 * p1y, p1z, p2x, ...]. */
GMSH_API void gmshModelGetValue(const int dim,
                                const int tag,
                                const double * parametricCoord, const size_t parametricCoord_n,
                                double ** coord, size_t * coord_n,
                                int * ierr);

/* Evaluate the derivative of the parametrization of the entity of dimension
 * `dim' and tag `tag' at the parametric coordinates `parametricCoord'. Only
 * valid for `dim' equal to 1 (with `parametricCoord' containing parametric
 * coordinates on the curve) or 2 (with `parametricCoord' containing u, v
 * parametric coordinates on the surface, concatenated: [p1u, p1v, p2u, ...]).
 * For `dim' equal to 1 return the x, y, z components of the derivative with
 * respect to u [d1ux, d1uy, d1uz, d2ux, ...]; for `dim' equal to 2 return the
 * x, y, z components of the derivative with respect to u and v: [d1ux, d1uy,
 * d1uz, d1vx, d1vy, d1vz, d2ux, ...]. */
GMSH_API void gmshModelGetDerivative(const int dim,
                                     const int tag,
                                     const double * parametricCoord, const size_t parametricCoord_n,
                                     double ** derivatives, size_t * derivatives_n,
                                     int * ierr);

/* Evaluate the second derivative of the parametrization of the entity of
 * dimension `dim' and tag `tag' at the parametric coordinates
 * `parametricCoord'. Only valid for `dim' equal to 1 (with `parametricCoord'
 * containing parametric coordinates on the curve) or 2 (with
 * `parametricCoord' containing u, v parametric coordinates on the surface,
 * concatenated: [p1u, p1v, p2u, ...]). For `dim' equal to 1 return the x, y,
 * z components of the second derivative with respect to u [d1uux, d1uuy,
 * d1uuz, d2uux, ...]; for `dim' equal to 2 return the x, y, z components of
 * the second derivative with respect to u and v, and the mixed derivative
 * with respect to u and v: [d1uux, d1uuy, d1uuz, d1vvx, d1vvy, d1vvz, d1uvx,
 * d1uvy, d1uvz, d2uux, ...]. */
GMSH_API void gmshModelGetSecondDerivative(const int dim,
                                           const int tag,
                                           const double * parametricCoord, const size_t parametricCoord_n,
                                           double ** derivatives, size_t * derivatives_n,
                                           int * ierr);

/* Evaluate the (maximum) curvature of the entity of dimension `dim' and tag
 * `tag' at the parametric coordinates `parametricCoord'. Only valid for `dim'
 * equal to 1 (with `parametricCoord' containing parametric coordinates on the
 * curve) or 2 (with `parametricCoord' containing u, v parametric coordinates
 * on the surface, concatenated: [p1u, p1v, p2u, ...]). */
GMSH_API void gmshModelGetCurvature(const int dim,
                                    const int tag,
                                    const double * parametricCoord, const size_t parametricCoord_n,
                                    double ** curvatures, size_t * curvatures_n,
                                    int * ierr);

/* Evaluate the principal curvatures of the surface with tag `tag' at the
 * parametric coordinates `parametricCoord', as well as their respective
 * directions. `parametricCoord' are given by pair of u and v coordinates,
 * concatenated: [p1u, p1v, p2u, ...]. */
GMSH_API void gmshModelGetPrincipalCurvatures(const int tag,
                                              const double * parametricCoord, const size_t parametricCoord_n,
                                              double ** curvatureMax, size_t * curvatureMax_n,
                                              double ** curvatureMin, size_t * curvatureMin_n,
                                              double ** directionMax, size_t * directionMax_n,
                                              double ** directionMin, size_t * directionMin_n,
                                              int * ierr);

/* Get the normal to the surface with tag `tag' at the parametric coordinates
 * `parametricCoord'. The `parametricCoord' vector should contain u and v
 * coordinates, concatenated: [p1u, p1v, p2u, ...]. `normals' are returned as
 * a vector of x, y, z components, concatenated: [n1x, n1y, n1z, n2x, ...]. */
GMSH_API void gmshModelGetNormal(const int tag,
                                 const double * parametricCoord, const size_t parametricCoord_n,
                                 double ** normals, size_t * normals_n,
                                 int * ierr);

/* Get the parametric coordinates `parametricCoord' for the points `coord' on
 * the entity of dimension `dim' and tag `tag'. `coord' are given as x, y, z
 * coordinates, concatenated: [p1x, p1y, p1z, p2x, ...]. `parametricCoord'
 * returns the parametric coordinates t on the curve (if `dim' = 1) or u and v
 * coordinates concatenated on the surface (if `dim' == 2), i.e. [p1t, p2t,
 * ...] or [p1u, p1v, p2u, ...]. */
GMSH_API void gmshModelGetParametrization(const int dim,
                                          const int tag,
                                          const double * coord, const size_t coord_n,
                                          double ** parametricCoord, size_t * parametricCoord_n,
                                          int * ierr);

/* Get the `min' and `max' bounds of the parametric coordinates for the entity
 * of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetParametrizationBounds(const int dim,
                                                const int tag,
                                                double ** min, size_t * min_n,
                                                double ** max, size_t * max_n,
                                                int * ierr);

/* Check if the coordinates (or the parametric coordinates if `parametric' is
 * set) provided in `coord' correspond to points inside the entity of
 * dimension `dim' and tag `tag', and return the number of points inside. This
 * feature is only available for a subset of entities, depending on the
 * underlying geometrical representation. */
GMSH_API int gmshModelIsInside(const int dim,
                               const int tag,
                               const double * coord, const size_t coord_n,
                               const int parametric,
                               int * ierr);

/* Get the points `closestCoord' on the entity of dimension `dim' and tag
 * `tag' to the points `coord', by orthogonal projection. `coord' and
 * `closestCoord' are given as x, y, z coordinates, concatenated: [p1x, p1y,
 * p1z, p2x, ...]. `parametricCoord' returns the parametric coordinates t on
 * the curve (if `dim' == 1) or u and v coordinates concatenated on the
 * surface (if `dim' = 2), i.e. [p1t, p2t, ...] or [p1u, p1v, p2u, ...]. */
GMSH_API void gmshModelGetClosestPoint(const int dim,
                                       const int tag,
                                       const double * coord, const size_t coord_n,
                                       double ** closestCoord, size_t * closestCoord_n,
                                       double ** parametricCoord, size_t * parametricCoord_n,
                                       int * ierr);

/* Reparametrize the boundary entity (point or curve, i.e. with `dim' == 0 or
 * `dim' == 1) of tag `tag' on the surface `surfaceTag'. If `dim' == 1,
 * reparametrize all the points corresponding to the parametric coordinates
 * `parametricCoord'. Multiple matches in case of periodic surfaces can be
 * selected with `which'. This feature is only available for a subset of
 * entities, depending on the underlying geometrical representation. */
GMSH_API void gmshModelReparametrizeOnSurface(const int dim,
                                              const int tag,
                                              const double * parametricCoord, const size_t parametricCoord_n,
                                              const int surfaceTag,
                                              double ** surfaceParametricCoord, size_t * surfaceParametricCoord_n,
                                              const int which,
                                              int * ierr);

/* Set the visibility of the model entities `dimTags' (given as a vector of
 * (dim, tag) pairs) to `value'. Apply the visibility setting recursively if
 * `recursive' is true. */
GMSH_API void gmshModelSetVisibility(const int * dimTags, const size_t dimTags_n,
                                     const int value,
                                     const int recursive,
                                     int * ierr);

/* Get the visibility of the model entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelGetVisibility(const int dim,
                                     const int tag,
                                     int * value,
                                     int * ierr);

/* Set the global visibility of the model per window to `value', where
 * `windowIndex' identifies the window in the window list. */
GMSH_API void gmshModelSetVisibilityPerWindow(const int value,
                                              const int windowIndex,
                                              int * ierr);

/* Set the color of the model entities `dimTags' (given as a vector of (dim,
 * tag) pairs) to the RGBA value (`r', `g', `b', `a'), where `r', `g', `b' and
 * `a' should be integers between 0 and 255. Apply the color setting
 * recursively if `recursive' is true. */
GMSH_API void gmshModelSetColor(const int * dimTags, const size_t dimTags_n,
                                const int r,
                                const int g,
                                const int b,
                                const int a,
                                const int recursive,
                                int * ierr);

/* Get the color of the model entity of dimension `dim' and tag `tag'. If no
 * color is specified for the entity, return fully transparent blue, i.e. (0,
 * 0, 255, 0). */
GMSH_API void gmshModelGetColor(const int dim,
                                const int tag,
                                int * r,
                                int * g,
                                int * b,
                                int * a,
                                int * ierr);

/* Set the `x', `y', `z' coordinates of a geometrical point. */
GMSH_API void gmshModelSetCoordinates(const int tag,
                                      const double x,
                                      const double y,
                                      const double z,
                                      int * ierr);

/* Set the values of the attribute with name `name'. */
GMSH_API void gmshModelSetAttribute(const char * name,
                                    const char * const * values, const size_t values_n,
                                    int * ierr);

/* Get the values of the attribute with name `name'. */
GMSH_API void gmshModelGetAttribute(const char * name,
                                    char *** values, size_t * values_n,
                                    int * ierr);

/* Get the names of any optional attributes stored in the model. */
GMSH_API void gmshModelGetAttributeNames(char *** names, size_t * names_n,
                                         int * ierr);

/* Remove the attribute with name `name'. */
GMSH_API void gmshModelRemoveAttribute(const char * name,
                                       int * ierr);

/* Generate a mesh of the current model, up to dimension `dim' (0, 1, 2 or 3). */
GMSH_API void gmshModelMeshGenerate(const int dim,
                                    int * ierr);

/* Partition the mesh of the current model into `numPart' partitions.
 * Optionally, `elementTags' and `partitions' can be provided to specify the
 * partition of each element explicitly. */
GMSH_API void gmshModelMeshPartition(const int numPart,
                                     const size_t * elementTags, const size_t elementTags_n,
                                     const int * partitions, const size_t partitions_n,
                                     int * ierr);

/* Unpartition the mesh of the current model. */
GMSH_API void gmshModelMeshUnpartition(int * ierr);

/* Optimize the mesh of the current model using `method' (empty for default
 * tetrahedral mesh optimizer, "Netgen" for Netgen optimizer, "HighOrder" for
 * direct high-order mesh optimizer, "HighOrderElastic" for high-order elastic
 * smoother, "HighOrderFastCurving" for fast curving algorithm, "Laplace2D"
 * for Laplace smoothing, "Relocate2D" and "Relocate3D" for node relocation,
 * "QuadQuasiStructured" for quad mesh optimization, "UntangleMeshGeometry"
 * for untangling). If `force' is set apply the optimization also to discrete
 * entities. If `dimTags' (given as a vector of (dim, tag) pairs) is given,
 * only apply the optimizer to the given entities. */
GMSH_API void gmshModelMeshOptimize(const char * method,
                                    const int force,
                                    const int niter,
                                    const int * dimTags, const size_t dimTags_n,
                                    int * ierr);

/* Recombine the mesh of the current model. */
GMSH_API void gmshModelMeshRecombine(int * ierr);

/* Refine the mesh of the current model by uniformly splitting the elements. */
GMSH_API void gmshModelMeshRefine(int * ierr);

/* Set the order of the elements in the mesh of the current model to `order'. */
GMSH_API void gmshModelMeshSetOrder(const int order,
                                    int * ierr);

/* Get the last entities `dimTags' (as a vector of (dim, tag) pairs) where a
 * meshing error occurred. Currently only populated by the new 3D meshing
 * algorithms. */
GMSH_API void gmshModelMeshGetLastEntityError(int ** dimTags, size_t * dimTags_n,
                                              int * ierr);

/* Get the last node tags `nodeTags' where a meshing error occurred. Currently
 * only populated by the new 3D meshing algorithms. */
GMSH_API void gmshModelMeshGetLastNodeError(size_t ** nodeTags, size_t * nodeTags_n,
                                            int * ierr);

/* Clear the mesh, i.e. delete all the nodes and elements, for the entities
 * `dimTags', given as a vector of (dim, tag) pairs. If `dimTags' is empty,
 * clear the whole mesh. Note that the mesh of an entity can only be cleared
 * if this entity is not on the boundary of another entity with a non-empty
 * mesh. */
GMSH_API void gmshModelMeshClear(const int * dimTags, const size_t dimTags_n,
                                 int * ierr);

/* Reverse the orientation of the elements in the entities `dimTags', given as
 * a vector of (dim, tag) pairs. If `dimTags' is empty, reverse the
 * orientation of the elements in the whole mesh. */
GMSH_API void gmshModelMeshReverse(const int * dimTags, const size_t dimTags_n,
                                   int * ierr);

/* Reverse the orientation of the elements with tags `elementTags'. */
GMSH_API void gmshModelMeshReverseElements(const size_t * elementTags, const size_t elementTags_n,
                                           int * ierr);

/* Apply the affine transformation `affineTransform' (16 entries of a 4x4
 * matrix, by row; only the 12 first can be provided for convenience) to the
 * coordinates of the nodes classified on the entities `dimTags', given as a
 * vector of (dim, tag) pairs. If `dimTags' is empty, transform all the nodes
 * in the mesh. */
GMSH_API void gmshModelMeshAffineTransform(const double * affineTransform, const size_t affineTransform_n,
                                           const int * dimTags, const size_t dimTags_n,
                                           int * ierr);

/* Get the nodes classified on the entity of dimension `dim' and tag `tag'. If
 * `tag' < 0, get the nodes for all entities of dimension `dim'. If `dim' and
 * `tag' are negative, get all the nodes in the mesh. `nodeTags' contains the
 * node tags (their unique, strictly positive identification numbers). `coord'
 * is a vector of length 3 times the length of `nodeTags' that contains the x,
 * y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. If
 * `dim' >= 0 and `returnParamtricCoord' is set, `parametricCoord' contains
 * the parametric coordinates ([u1, u2, ...] or [u1, v1, u2, ...]) of the
 * nodes, if available. The length of `parametricCoord' can be 0 or `dim'
 * times the length of `nodeTags'. If `includeBoundary' is set, also return
 * the nodes classified on the boundary of the entity (which will be
 * reparametrized on the entity if `dim' >= 0 in order to compute their
 * parametric coordinates). */
GMSH_API void gmshModelMeshGetNodes(size_t ** nodeTags, size_t * nodeTags_n,
                                    double ** coord, size_t * coord_n,
                                    double ** parametricCoord, size_t * parametricCoord_n,
                                    const int dim,
                                    const int tag,
                                    const int includeBoundary,
                                    const int returnParametricCoord,
                                    int * ierr);

/* Get the nodes classified on the entity of tag `tag', for all the elements
 * of type `elementType'. The other arguments are treated as in `getNodes'. */
GMSH_API void gmshModelMeshGetNodesByElementType(const int elementType,
                                                 size_t ** nodeTags, size_t * nodeTags_n,
                                                 double ** coord, size_t * coord_n,
                                                 double ** parametricCoord, size_t * parametricCoord_n,
                                                 const int tag,
                                                 const int returnParametricCoord,
                                                 int * ierr);

/* Get the coordinates and the parametric coordinates (if any) of the node
 * with tag `tag', as well as the dimension `dim' and tag `tag' of the entity
 * on which the node is classified. This function relies on an internal cache
 * (a vector in case of dense node numbering, a map otherwise); for large
 * meshes accessing nodes in bulk is often preferable. */
GMSH_API void gmshModelMeshGetNode(const size_t nodeTag,
                                   double ** coord, size_t * coord_n,
                                   double ** parametricCoord, size_t * parametricCoord_n,
                                   int * dim,
                                   int * tag,
                                   int * ierr);

/* Set the coordinates and the parametric coordinates (if any) of the node
 * with tag `tag'. This function relies on an internal cache (a vector in case
 * of dense node numbering, a map otherwise); for large meshes accessing nodes
 * in bulk is often preferable. */
GMSH_API void gmshModelMeshSetNode(const size_t nodeTag,
                                   const double * coord, const size_t coord_n,
                                   const double * parametricCoord, const size_t parametricCoord_n,
                                   int * ierr);

/* Rebuild the node cache. */
GMSH_API void gmshModelMeshRebuildNodeCache(const int onlyIfNecessary,
                                            int * ierr);

/* Rebuild the element cache. */
GMSH_API void gmshModelMeshRebuildElementCache(const int onlyIfNecessary,
                                               int * ierr);

/* Get the nodes from all the elements belonging to the physical group of
 * dimension `dim' and tag `tag'. `nodeTags' contains the node tags; `coord'
 * is a vector of length 3 times the length of `nodeTags' that contains the x,
 * y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. */
GMSH_API void gmshModelMeshGetNodesForPhysicalGroup(const int dim,
                                                    const int tag,
                                                    size_t ** nodeTags, size_t * nodeTags_n,
                                                    double ** coord, size_t * coord_n,
                                                    int * ierr);

/* Get the maximum tag `maxTag' of a node in the mesh. */
GMSH_API void gmshModelMeshGetMaxNodeTag(size_t * maxTag,
                                         int * ierr);

/* Add nodes classified on the model entity of dimension `dim' and tag `tag'.
 * `nodeTags' contains the node tags (their unique, strictly positive
 * identification numbers). `coord' is a vector of length 3 times the length
 * of `nodeTags' that contains the x, y, z coordinates of the nodes,
 * concatenated: [n1x, n1y, n1z, n2x, ...]. The optional `parametricCoord'
 * vector contains the parametric coordinates of the nodes, if any. The length
 * of `parametricCoord' can be 0 or `dim' times the length of `nodeTags'. If
 * the `nodeTags' vector is empty, new tags are automatically assigned to the
 * nodes. */
GMSH_API void gmshModelMeshAddNodes(const int dim,
                                    const int tag,
                                    const size_t * nodeTags, const size_t nodeTags_n,
                                    const double * coord, const size_t coord_n,
                                    const double * parametricCoord, const size_t parametricCoord_n,
                                    int * ierr);

/* Reclassify all nodes on their associated model entity, based on the
 * elements. Can be used when importing nodes in bulk (e.g. by associating
 * them all to a single volume), to reclassify them correctly on model
 * surfaces, curves, etc. after the elements have been set. */
GMSH_API void gmshModelMeshReclassifyNodes(int * ierr);

/* Relocate the nodes classified on the entity of dimension `dim' and tag
 * `tag' using their parametric coordinates. If `tag' < 0, relocate the nodes
 * for all entities of dimension `dim'. If `dim' and `tag' are negative,
 * relocate all the nodes in the mesh. */
GMSH_API void gmshModelMeshRelocateNodes(const int dim,
                                         const int tag,
                                         int * ierr);

/* Get the elements classified on the entity of dimension `dim' and tag `tag'.
 * If `tag' < 0, get the elements for all entities of dimension `dim'. If
 * `dim' and `tag' are negative, get all the elements in the mesh.
 * `elementTypes' contains the MSH types of the elements (e.g. `2' for 3-node
 * triangles: see `getElementProperties' to obtain the properties for a given
 * element type). `elementTags' is a vector of the same length as
 * `elementTypes'; each entry is a vector containing the tags (unique,
 * strictly positive identifiers) of the elements of the corresponding type.
 * `nodeTags' is also a vector of the same length as `elementTypes'; each
 * entry is a vector of length equal to the number of elements of the given
 * type times the number N of nodes for this type of element, that contains
 * the node tags of all the elements of the given type, concatenated: [e1n1,
 * e1n2, ..., e1nN, e2n1, ...]. */
GMSH_API void gmshModelMeshGetElements(int ** elementTypes, size_t * elementTypes_n,
                                       size_t *** elementTags, size_t ** elementTags_n, size_t *elementTags_nn,
                                       size_t *** nodeTags, size_t ** nodeTags_n, size_t *nodeTags_nn,
                                       const int dim,
                                       const int tag,
                                       int * ierr);

/* Get the type and node tags of the element with tag `tag', as well as the
 * dimension `dim' and tag `tag' of the entity on which the element is
 * classified. This function relies on an internal cache (a vector in case of
 * dense element numbering, a map otherwise); for large meshes accessing
 * elements in bulk is often preferable. */
GMSH_API void gmshModelMeshGetElement(const size_t elementTag,
                                      int * elementType,
                                      size_t ** nodeTags, size_t * nodeTags_n,
                                      int * dim,
                                      int * tag,
                                      int * ierr);

/* Search the mesh for an element located at coordinates (`x', `y', `z'). This
 * function performs a search in a spatial octree. If an element is found,
 * return its tag, type and node tags, as well as the local coordinates (`u',
 * `v', `w') within the reference element corresponding to search location. If
 * `dim' is >= 0, only search for elements of the given dimension. If `strict'
 * is not set, use a tolerance to find elements near the search location. */
GMSH_API void gmshModelMeshGetElementByCoordinates(const double x,
                                                   const double y,
                                                   const double z,
                                                   size_t * elementTag,
                                                   int * elementType,
                                                   size_t ** nodeTags, size_t * nodeTags_n,
                                                   double * u,
                                                   double * v,
                                                   double * w,
                                                   const int dim,
                                                   const int strict,
                                                   int * ierr);

/* Search the mesh for element(s) located at coordinates (`x', `y', `z'). This
 * function performs a search in a spatial octree. Return the tags of all
 * found elements in `elementTags'. Additional information about the elements
 * can be accessed through `getElement' and `getLocalCoordinatesInElement'. If
 * `dim' is >= 0, only search for elements of the given dimension. If `strict'
 * is not set, use a tolerance to find elements near the search location. */
GMSH_API void gmshModelMeshGetElementsByCoordinates(const double x,
                                                    const double y,
                                                    const double z,
                                                    size_t ** elementTags, size_t * elementTags_n,
                                                    const int dim,
                                                    const int strict,
                                                    int * ierr);

/* Return the local coordinates (`u', `v', `w') within the element
 * `elementTag' corresponding to the model coordinates (`x', `y', `z'). This
 * function relies on an internal cache (a vector in case of dense element
 * numbering, a map otherwise); for large meshes accessing elements in bulk is
 * often preferable. */
GMSH_API void gmshModelMeshGetLocalCoordinatesInElement(const size_t elementTag,
                                                        const double x,
                                                        const double y,
                                                        const double z,
                                                        double * u,
                                                        double * v,
                                                        double * w,
                                                        int * ierr);

/* Get the types of elements in the entity of dimension `dim' and tag `tag'.
 * If `tag' < 0, get the types for all entities of dimension `dim'. If `dim'
 * and `tag' are negative, get all the types in the mesh. */
GMSH_API void gmshModelMeshGetElementTypes(int ** elementTypes, size_t * elementTypes_n,
                                           const int dim,
                                           const int tag,
                                           int * ierr);

/* Return an element type given its family name `familyName' ("Point", "Line",
 * "Triangle", "Quadrangle", "Tetrahedron", "Pyramid", "Prism", "Hexahedron")
 * and polynomial order `order'. If `serendip' is true, return the
 * corresponding serendip element type (element without interior nodes). */
GMSH_API int gmshModelMeshGetElementType(const char * familyName,
                                         const int order,
                                         const int serendip,
                                         int * ierr);

/* Get the properties of an element of type `elementType': its name
 * (`elementName'), dimension (`dim'), order (`order'), number of nodes
 * (`numNodes'), local coordinates of the nodes in the reference element
 * (`localNodeCoord' vector, of length `dim' times `numNodes') and number of
 * primary (first order) nodes (`numPrimaryNodes'). */
GMSH_API void gmshModelMeshGetElementProperties(const int elementType,
                                                char ** elementName,
                                                int * dim,
                                                int * order,
                                                int * numNodes,
                                                double ** localNodeCoord, size_t * localNodeCoord_n,
                                                int * numPrimaryNodes,
                                                int * ierr);

/* Get the elements of type `elementType' classified on the entity of tag
 * `tag'. If `tag' < 0, get the elements for all entities. `elementTags' is a
 * vector containing the tags (unique, strictly positive identifiers) of the
 * elements of the corresponding type. `nodeTags' is a vector of length equal
 * to the number of elements of the given type times the number N of nodes for
 * this type of element, that contains the node tags of all the elements of
 * the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...]. If
 * `numTasks' > 1, only compute and return the part of the data indexed by
 * `task' (for C++ only; output vectors must be preallocated). */
GMSH_API void gmshModelMeshGetElementsByType(const int elementType,
                                             size_t ** elementTags, size_t * elementTags_n,
                                             size_t ** nodeTags, size_t * nodeTags_n,
                                             const int tag,
                                             const size_t task,
                                             const size_t numTasks,
                                             int * ierr);

/* Get the maximum tag `maxTag' of an element in the mesh. */
GMSH_API void gmshModelMeshGetMaxElementTag(size_t * maxTag,
                                            int * ierr);

/* Preallocate data before calling `getElementsByType' with `numTasks' > 1.
 * For C++ only. */
GMSH_API void gmshModelMeshPreallocateElementsByType(const int elementType,
                                                     const int elementTag,
                                                     const int nodeTag,
                                                     size_t ** elementTags, size_t * elementTags_n,
                                                     size_t ** nodeTags, size_t * nodeTags_n,
                                                     const int tag,
                                                     int * ierr);

/* Get the quality `elementQualities' of the elements with tags `elementTags'.
 * `qualityType' is the requested quality measure: "minDetJac" and "maxDetJac"
 * for the adaptively computed minimal and maximal Jacobian determinant,
 * "minSJ" for the sampled minimal scaled jacobien, "minSICN" for the sampled
 * minimal signed inverted condition number, "minSIGE" for the sampled signed
 * inverted gradient error, "gamma" for the ratio of the inscribed to
 * circumcribed sphere radius, "innerRadius" for the inner radius,
 * "outerRadius" for the outerRadius, "minIsotropy" for the minimum isotropy
 * measure, "angleShape" for the angle shape measure, "minEdge" for the
 * minimum straight edge length, "maxEdge" for the maximum straight edge
 * length, "volume" for the volume. If `numTasks' > 1, only compute and return
 * the part of the data indexed by `task' (for C++ only; output vector must be
 * preallocated). */
GMSH_API void gmshModelMeshGetElementQualities(const size_t * elementTags, const size_t elementTags_n,
                                               double ** elementsQuality, size_t * elementsQuality_n,
                                               const char * qualityName,
                                               const size_t task,
                                               const size_t numTasks,
                                               int * ierr);

/* Add elements classified on the entity of dimension `dim' and tag `tag'.
 * `types' contains the MSH types of the elements (e.g. `2' for 3-node
 * triangles: see the Gmsh reference manual). `elementTags' is a vector of the
 * same length as `types'; each entry is a vector containing the tags (unique,
 * strictly positive identifiers) of the elements of the corresponding type.
 * `nodeTags' is also a vector of the same length as `types'; each entry is a
 * vector of length equal to the number of elements of the given type times
 * the number N of nodes per element, that contains the node tags of all the
 * elements of the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1,
 * ...]. */
GMSH_API void gmshModelMeshAddElements(const int dim,
                                       const int tag,
                                       const int * elementTypes, const size_t elementTypes_n,
                                       const size_t * const * elementTags, const size_t * elementTags_n, const size_t elementTags_nn,
                                       const size_t * const * nodeTags, const size_t * nodeTags_n, const size_t nodeTags_nn,
                                       int * ierr);

/* Add elements of type `elementType' classified on the entity of tag `tag'.
 * `elementTags' contains the tags (unique, strictly positive identifiers) of
 * the elements of the corresponding type. `nodeTags' is a vector of length
 * equal to the number of elements times the number N of nodes per element,
 * that contains the node tags of all the elements, concatenated: [e1n1, e1n2,
 * ..., e1nN, e2n1, ...]. If the `elementTag' vector is empty, new tags are
 * automatically assigned to the elements. */
GMSH_API void gmshModelMeshAddElementsByType(const int tag,
                                             const int elementType,
                                             const size_t * elementTags, const size_t elementTags_n,
                                             const size_t * nodeTags, const size_t nodeTags_n,
                                             int * ierr);

/* Get the numerical quadrature information for the given element type
 * `elementType' and integration rule `integrationType', where
 * `integrationType' concatenates the integration rule family name with the
 * desired order (e.g. "Gauss4" for a quadrature suited for integrating 4th
 * order polynomials). The "CompositeGauss" family uses tensor-product rules
 * based the 1D Gauss-Legendre rule; the "Gauss" family uses an economic
 * scheme when available (i.e. with a minimal number of points), and falls
 * back to "CompositeGauss" otherwise. Note that integration points for the
 * "Gauss" family can fall outside of the reference element for high-order
 * rules. `localCoord' contains the u, v, w coordinates of the G integration
 * points in the reference element: [g1u, g1v, g1w, ..., gGu, gGv, gGw].
 * `weights' contains the associated weights: [g1q, ..., gGq]. */
GMSH_API void gmshModelMeshGetIntegrationPoints(const int elementType,
                                                const char * integrationType,
                                                double ** localCoord, size_t * localCoord_n,
                                                double ** weights, size_t * weights_n,
                                                int * ierr);

/* Get the Jacobians of all the elements of type `elementType' classified on
 * the entity of tag `tag', at the G evaluation points `localCoord' given as
 * concatenated u, v, w coordinates in the reference element [g1u, g1v, g1w,
 * ..., gGu, gGv, gGw]. Data is returned by element, with elements in the same
 * order as in `getElements' and `getElementsByType'. `jacobians' contains for
 * each element the 9 entries of the 3x3 Jacobian matrix at each evaluation
 * point. The matrix is returned by column: [e1g1Jxu, e1g1Jyu, e1g1Jzu,
 * e1g1Jxv, ..., e1g1Jzw, e1g2Jxu, ..., e1gGJzw, e2g1Jxu, ...], with Jxu =
 * dx/du, Jyu = dy/du, etc. `determinants' contains for each element the
 * determinant of the Jacobian matrix at each evaluation point: [e1g1, e1g2,
 * ... e1gG, e2g1, ...]. `coord' contains for each element the x, y, z
 * coordinates of the evaluation points. If `tag' < 0, get the Jacobian data
 * for all entities. If `numTasks' > 1, only compute and return the part of
 * the data indexed by `task' (for C++ only; output vectors must be
 * preallocated). */
GMSH_API void gmshModelMeshGetJacobians(const int elementType,
                                        const double * localCoord, const size_t localCoord_n,
                                        double ** jacobians, size_t * jacobians_n,
                                        double ** determinants, size_t * determinants_n,
                                        double ** coord, size_t * coord_n,
                                        const int tag,
                                        const size_t task,
                                        const size_t numTasks,
                                        int * ierr);

/* Preallocate data before calling `getJacobians' with `numTasks' > 1. For C++
 * only. */
GMSH_API void gmshModelMeshPreallocateJacobians(const int elementType,
                                                const int numEvaluationPoints,
                                                const int allocateJacobians,
                                                const int allocateDeterminants,
                                                const int allocateCoord,
                                                double ** jacobians, size_t * jacobians_n,
                                                double ** determinants, size_t * determinants_n,
                                                double ** coord, size_t * coord_n,
                                                const int tag,
                                                int * ierr);

/* Get the Jacobian for a single element `elementTag', at the G evaluation
 * points `localCoord' given as concatenated u, v, w coordinates in the
 * reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]. `jacobians' contains
 * the 9 entries of the 3x3 Jacobian matrix at each evaluation point. The
 * matrix is returned by column: [e1g1Jxu, e1g1Jyu, e1g1Jzu, e1g1Jxv, ...,
 * e1g1Jzw, e1g2Jxu, ..., e1gGJzw, e2g1Jxu, ...], with Jxu = dx/du, Jyu =
 * dy/du, etc. `determinants' contains the determinant of the Jacobian matrix
 * at each evaluation point. `coord' contains the x, y, z coordinates of the
 * evaluation points. This function relies on an internal cache (a vector in
 * case of dense element numbering, a map otherwise); for large meshes
 * accessing Jacobians in bulk is often preferable. */
GMSH_API void gmshModelMeshGetJacobian(const size_t elementTag,
                                       const double * localCoord, const size_t localCoord_n,
                                       double ** jacobians, size_t * jacobians_n,
                                       double ** determinants, size_t * determinants_n,
                                       double ** coord, size_t * coord_n,
                                       int * ierr);

/* Get the basis functions of the element of type `elementType' at the
 * evaluation points `localCoord' (given as concatenated u, v, w coordinates
 * in the reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]), for the
 * function space `functionSpaceType'. Currently supported function spaces
 * include "Lagrange" and "GradLagrange" for isoparametric Lagrange basis
 * functions and their gradient in the u, v, w coordinates of the reference
 * element; "LagrangeN" and "GradLagrangeN", with N = 1, 2, ..., for N-th
 * order Lagrange basis functions; "H1LegendreN" and "GradH1LegendreN", with N
 * = 1, 2, ..., for N-th order hierarchical H1 Legendre functions;
 * "HcurlLegendreN" and "CurlHcurlLegendreN", with N = 1, 2, ..., for N-th
 * order curl-conforming basis functions. `numComponents' returns the number C
 * of components of a basis function (e.g. 1 for scalar functions and 3 for
 * vector functions). `basisFunctions' returns the value of the N basis
 * functions at the evaluation points, i.e. [g1f1, g1f2, ..., g1fN, g2f1, ...]
 * when C == 1 or [g1f1u, g1f1v, g1f1w, g1f2u, ..., g1fNw, g2f1u, ...] when C
 * == 3. For basis functions that depend on the orientation of the elements,
 * all values for the first orientation are returned first, followed by values
 * for the second, etc. `numOrientations' returns the overall number of
 * orientations. If the `wantedOrientations' vector is not empty, only return
 * the values for the desired orientation indices. */
GMSH_API void gmshModelMeshGetBasisFunctions(const int elementType,
                                             const double * localCoord, const size_t localCoord_n,
                                             const char * functionSpaceType,
                                             int * numComponents,
                                             double ** basisFunctions, size_t * basisFunctions_n,
                                             int * numOrientations,
                                             const int * wantedOrientations, const size_t wantedOrientations_n,
                                             int * ierr);

/* Get the orientation index of the elements of type `elementType' in the
 * entity of tag `tag'. The arguments have the same meaning as in
 * `getBasisFunctions'. `basisFunctionsOrientation' is a vector giving for
 * each element the orientation index in the values returned by
 * `getBasisFunctions'. For Lagrange basis functions the call is superfluous
 * as it will return a vector of zeros. If `numTasks' > 1, only compute and
 * return the part of the data indexed by `task' (for C++ only; output vector
 * must be preallocated). */
GMSH_API void gmshModelMeshGetBasisFunctionsOrientation(const int elementType,
                                                        const char * functionSpaceType,
                                                        int ** basisFunctionsOrientation, size_t * basisFunctionsOrientation_n,
                                                        const int tag,
                                                        const size_t task,
                                                        const size_t numTasks,
                                                        int * ierr);

/* Get the orientation of a single element `elementTag'. */
GMSH_API void gmshModelMeshGetBasisFunctionsOrientationForElement(const size_t elementTag,
                                                                  const char * functionSpaceType,
                                                                  int * basisFunctionsOrientation,
                                                                  int * ierr);

/* Get the number of possible orientations for elements of type `elementType'
 * and function space named `functionSpaceType'. */
GMSH_API int gmshModelMeshGetNumberOfOrientations(const int elementType,
                                                  const char * functionSpaceType,
                                                  int * ierr);

/* Preallocate data before calling `getBasisFunctionsOrientation' with
 * `numTasks' > 1. For C++ only. */
GMSH_API void gmshModelMeshPreallocateBasisFunctionsOrientation(const int elementType,
                                                                int ** basisFunctionsOrientation, size_t * basisFunctionsOrientation_n,
                                                                const int tag,
                                                                int * ierr);

/* Get the global unique mesh edge identifiers `edgeTags' and orientations
 * `edgeOrientation' for an input list of node tag pairs defining these edges,
 * concatenated in the vector `nodeTags'. Mesh edges are created e.g. by
 * `createEdges()', `getKeys()' or `addEdges()'. The reference positive
 * orientation is n1 < n2, where n1 and n2 are the tags of the two edge nodes,
 * which corresponds to the local orientation of edge-based basis functions as
 * well. */
GMSH_API void gmshModelMeshGetEdges(const size_t * nodeTags, const size_t nodeTags_n,
                                    size_t ** edgeTags, size_t * edgeTags_n,
                                    int ** edgeOrientations, size_t * edgeOrientations_n,
                                    int * ierr);

/* Get the global unique mesh face identifiers `faceTags' and orientations
 * `faceOrientations' for an input list of a multiple of three (if `faceType'
 * == 3) or four (if `faceType' == 4) node tags defining these faces,
 * concatenated in the vector `nodeTags'. Mesh faces are created e.g. by
 * `createFaces()', `getKeys()' or `addFaces()'. */
GMSH_API void gmshModelMeshGetFaces(const int faceType,
                                    const size_t * nodeTags, const size_t nodeTags_n,
                                    size_t ** faceTags, size_t * faceTags_n,
                                    int ** faceOrientations, size_t * faceOrientations_n,
                                    int * ierr);

/* Create unique mesh edges for the entities `dimTags', given as a vector of
 * (dim, tag) pairs. */
GMSH_API void gmshModelMeshCreateEdges(const int * dimTags, const size_t dimTags_n,
                                       int * ierr);

/* Create unique mesh faces for the entities `dimTags', given as a vector of
 * (dim, tag) pairs. */
GMSH_API void gmshModelMeshCreateFaces(const int * dimTags, const size_t dimTags_n,
                                       int * ierr);

/* Get the global unique identifiers `edgeTags' and the nodes `edgeNodes' of
 * the edges in the mesh. Mesh edges are created e.g. by `createEdges()',
 * `getKeys()' or addEdges(). */
GMSH_API void gmshModelMeshGetAllEdges(size_t ** edgeTags, size_t * edgeTags_n,
                                       size_t ** edgeNodes, size_t * edgeNodes_n,
                                       int * ierr);

/* Get the global unique identifiers `faceTags' and the nodes `faceNodes' of
 * the faces of type `faceType' in the mesh. Mesh faces are created e.g. by
 * `createFaces()', `getKeys()' or addFaces(). */
GMSH_API void gmshModelMeshGetAllFaces(const int faceType,
                                       size_t ** faceTags, size_t * faceTags_n,
                                       size_t ** faceNodes, size_t * faceNodes_n,
                                       int * ierr);

/* Add mesh edges defined by their global unique identifiers `edgeTags' and
 * their nodes `edgeNodes'. */
GMSH_API void gmshModelMeshAddEdges(const size_t * edgeTags, const size_t edgeTags_n,
                                    const size_t * edgeNodes, const size_t edgeNodes_n,
                                    int * ierr);

/* Add mesh faces of type `faceType' defined by their global unique
 * identifiers `faceTags' and their nodes `faceNodes'. */
GMSH_API void gmshModelMeshAddFaces(const int faceType,
                                    const size_t * faceTags, const size_t faceTags_n,
                                    const size_t * faceNodes, const size_t faceNodes_n,
                                    int * ierr);

/* Generate the pair of keys for the elements of type `elementType' in the
 * entity of tag `tag', for the `functionSpaceType' function space. Each pair
 * (`typeKey', `entityKey') uniquely identifies a basis function in the
 * function space. If `returnCoord' is set, the `coord' vector contains the x,
 * y, z coordinates locating basis functions for sorting purposes. Warning:
 * this is an experimental feature and will probably change in a future
 * release. */
GMSH_API void gmshModelMeshGetKeys(const int elementType,
                                   const char * functionSpaceType,
                                   int ** typeKeys, size_t * typeKeys_n,
                                   size_t ** entityKeys, size_t * entityKeys_n,
                                   double ** coord, size_t * coord_n,
                                   const int tag,
                                   const int returnCoord,
                                   int * ierr);

/* Get the pair of keys for a single element `elementTag'. */
GMSH_API void gmshModelMeshGetKeysForElement(const size_t elementTag,
                                             const char * functionSpaceType,
                                             int ** typeKeys, size_t * typeKeys_n,
                                             size_t ** entityKeys, size_t * entityKeys_n,
                                             double ** coord, size_t * coord_n,
                                             const int returnCoord,
                                             int * ierr);

/* Get the number of keys by elements of type `elementType' for function space
 * named `functionSpaceType'. */
GMSH_API int gmshModelMeshGetNumberOfKeys(const int elementType,
                                          const char * functionSpaceType,
                                          int * ierr);

/* Get information about the pair of `keys'. `infoKeys' returns information
 * about the functions associated with the pairs (`typeKeys', `entityKey').
 * `infoKeys[0].first' describes the type of function (0 for  vertex function,
 * 1 for edge function, 2 for face function and 3 for bubble function).
 * `infoKeys[0].second' gives the order of the function associated with the
 * key. Warning: this is an experimental feature and will probably change in a
 * future release. */
GMSH_API void gmshModelMeshGetKeysInformation(const int * typeKeys, const size_t typeKeys_n,
                                              const size_t * entityKeys, const size_t entityKeys_n,
                                              const int elementType,
                                              const char * functionSpaceType,
                                              int ** infoKeys, size_t * infoKeys_n,
                                              int * ierr);

/* Get the barycenters of all elements of type `elementType' classified on the
 * entity of tag `tag'. If `primary' is set, only the primary nodes of the
 * elements are taken into account for the barycenter calculation. If `fast'
 * is set, the function returns the sum of the primary node coordinates
 * (without normalizing by the number of nodes). If `tag' < 0, get the
 * barycenters for all entities. If `numTasks' > 1, only compute and return
 * the part of the data indexed by `task' (for C++ only; output vector must be
 * preallocated). */
GMSH_API void gmshModelMeshGetBarycenters(const int elementType,
                                          const int tag,
                                          const int fast,
                                          const int primary,
                                          double ** barycenters, size_t * barycenters_n,
                                          const size_t task,
                                          const size_t numTasks,
                                          int * ierr);

/* Preallocate data before calling `getBarycenters' with `numTasks' > 1. For
 * C++ only. */
GMSH_API void gmshModelMeshPreallocateBarycenters(const int elementType,
                                                  double ** barycenters, size_t * barycenters_n,
                                                  const int tag,
                                                  int * ierr);

/* Get the nodes on the edges of all elements of type `elementType' classified
 * on the entity of tag `tag'. `nodeTags' contains the node tags of the edges
 * for all the elements: [e1a1n1, e1a1n2, e1a2n1, ...]. Data is returned by
 * element, with elements in the same order as in `getElements' and
 * `getElementsByType'. If `primary' is set, only the primary (begin/end)
 * nodes of the edges are returned. If `tag' < 0, get the edge nodes for all
 * entities. If `numTasks' > 1, only compute and return the part of the data
 * indexed by `task' (for C++ only; output vector must be preallocated). */
GMSH_API void gmshModelMeshGetElementEdgeNodes(const int elementType,
                                               size_t ** nodeTags, size_t * nodeTags_n,
                                               const int tag,
                                               const int primary,
                                               const size_t task,
                                               const size_t numTasks,
                                               int * ierr);

/* Get the nodes on the faces of type `faceType' (3 for triangular faces, 4
 * for quadrangular faces) of all elements of type `elementType' classified on
 * the entity of tag `tag'. `nodeTags' contains the node tags of the faces for
 * all elements: [e1f1n1, ..., e1f1nFaceType, e1f2n1, ...]. Data is returned
 * by element, with elements in the same order as in `getElements' and
 * `getElementsByType'. If `primary' is set, only the primary (corner) nodes
 * of the faces are returned. If `tag' < 0, get the face nodes for all
 * entities. If `numTasks' > 1, only compute and return the part of the data
 * indexed by `task' (for C++ only; output vector must be preallocated). */
GMSH_API void gmshModelMeshGetElementFaceNodes(const int elementType,
                                               const int faceType,
                                               size_t ** nodeTags, size_t * nodeTags_n,
                                               const int tag,
                                               const int primary,
                                               const size_t task,
                                               const size_t numTasks,
                                               int * ierr);

/* Get the ghost elements `elementTags' and their associated `partitions'
 * stored in the ghost entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelMeshGetGhostElements(const int dim,
                                            const int tag,
                                            size_t ** elementTags, size_t * elementTags_n,
                                            int ** partitions, size_t * partitions_n,
                                            int * ierr);

/* Set a mesh size constraint on the model entities `dimTags', given as a
 * vector of (dim, tag) pairs. Currently only entities of dimension 0 (points)
 * are handled. */
GMSH_API void gmshModelMeshSetSize(const int * dimTags, const size_t dimTags_n,
                                   const double size,
                                   int * ierr);

/* Get the mesh size constraints (if any) associated with the model entities
 * `dimTags', given as a vector of (dim, tag) pairs. A zero entry in the
 * output `sizes' vector indicates that no size constraint is specified on the
 * corresponding entity. */
GMSH_API void gmshModelMeshGetSizes(const int * dimTags, const size_t dimTags_n,
                                    double ** sizes, size_t * sizes_n,
                                    int * ierr);

/* Set mesh size constraints at the given parametric points `parametricCoord'
 * on the model entity of dimension `dim' and tag `tag'. Currently only
 * entities of dimension 1 (lines) are handled. */
GMSH_API void gmshModelMeshSetSizeAtParametricPoints(const int dim,
                                                     const int tag,
                                                     const double * parametricCoord, const size_t parametricCoord_n,
                                                     const double * sizes, const size_t sizes_n,
                                                     int * ierr);

/* Set a mesh size callback for the current model. The callback function
 * should take six arguments as input (`dim', `tag', `x', `y', `z' and `lc').
 * The first two integer arguments correspond to the dimension `dim' and tag
 * `tag' of the entity being meshed. The next four double precision arguments
 * correspond to the coordinates `x', `y' and `z' around which to prescribe
 * the mesh size and to the mesh size `lc' that would be prescribed if the
 * callback had not been called. The callback function should return a double
 * precision number specifying the desired mesh size; returning `lc' is
 * equivalent to a no-op. */
GMSH_API void gmshModelMeshSetSizeCallback(double (*callback)(int dim, int tag, double x, double y, double z, double lc, void * data), void * callback_data,
                                           int * ierr);

/* Remove the mesh size callback from the current model. */
GMSH_API void gmshModelMeshRemoveSizeCallback(int * ierr);

/* Set a transfinite meshing constraint on the curve `tag', with `numNodes'
 * nodes distributed according to `meshType' and `coef'. Currently supported
 * types are "Progression" (geometrical progression with power `coef'), "Bump"
 * (refinement toward both extremities of the curve) and "Beta" (beta law). */
GMSH_API void gmshModelMeshSetTransfiniteCurve(const int tag,
                                               const int numNodes,
                                               const char * meshType,
                                               const double coef,
                                               int * ierr);

/* Set a transfinite meshing constraint on the surface `tag'. `arrangement'
 * describes the arrangement of the triangles when the surface is not flagged
 * as recombined: currently supported values are "Left", "Right",
 * "AlternateLeft" and "AlternateRight". `cornerTags' can be used to specify
 * the (3 or 4) corners of the transfinite interpolation explicitly;
 * specifying the corners explicitly is mandatory if the surface has more that
 * 3 or 4 points on its boundary. */
GMSH_API void gmshModelMeshSetTransfiniteSurface(const int tag,
                                                 const char * arrangement,
                                                 const int * cornerTags, const size_t cornerTags_n,
                                                 int * ierr);

/* Set a transfinite meshing constraint on the surface `tag'. `cornerTags' can
 * be used to specify the (6 or 8) corners of the transfinite interpolation
 * explicitly. */
GMSH_API void gmshModelMeshSetTransfiniteVolume(const int tag,
                                                const int * cornerTags, const size_t cornerTags_n,
                                                int * ierr);

/* Set transfinite meshing constraints on the model entities in `dimTags',
 * given as a vector of (dim, tag) pairs. Transfinite meshing constraints are
 * added to the curves of the quadrangular surfaces and to the faces of
 * 6-sided volumes. Quadragular faces with a corner angle superior to
 * `cornerAngle' (in radians) are ignored. The number of points is
 * automatically determined from the sizing constraints. If `dimTag' is empty,
 * the constraints are applied to all entities in the model. If `recombine' is
 * true, the recombine flag is automatically set on the transfinite surfaces. */
GMSH_API void gmshModelMeshSetTransfiniteAutomatic(const int * dimTags, const size_t dimTags_n,
                                                   const double cornerAngle,
                                                   const int recombine,
                                                   int * ierr);

/* Set a recombination meshing constraint on the model entity of dimension
 * `dim' and tag `tag'. Currently only entities of dimension 2 (to recombine
 * triangles into quadrangles) are supported; `angle' specifies the threshold
 * angle for the simple recombination algorithm.. */
GMSH_API void gmshModelMeshSetRecombine(const int dim,
                                        const int tag,
                                        const double angle,
                                        int * ierr);

/* Set a smoothing meshing constraint on the model entity of dimension `dim'
 * and tag `tag'. `val' iterations of a Laplace smoother are applied. */
GMSH_API void gmshModelMeshSetSmoothing(const int dim,
                                        const int tag,
                                        const int val,
                                        int * ierr);

/* Set a reverse meshing constraint on the model entity of dimension `dim' and
 * tag `tag'. If `val' is true, the mesh orientation will be reversed with
 * respect to the natural mesh orientation (i.e. the orientation consistent
 * with the orientation of the geometry). If `val' is false, the mesh is left
 * as-is. */
GMSH_API void gmshModelMeshSetReverse(const int dim,
                                      const int tag,
                                      const int val,
                                      int * ierr);

/* Set the meshing algorithm on the model entity of dimension `dim' and tag
 * `tag'. Supported values are those of the `Mesh.Algorithm' option, as listed
 * in the Gmsh reference manual. Currently only supported for `dim' == 2. */
GMSH_API void gmshModelMeshSetAlgorithm(const int dim,
                                        const int tag,
                                        const int val,
                                        int * ierr);

/* Force the mesh size to be extended from the boundary, or not, for the model
 * entity of dimension `dim' and tag `tag'. Currently only supported for `dim'
 * == 2. */
GMSH_API void gmshModelMeshSetSizeFromBoundary(const int dim,
                                               const int tag,
                                               const int val,
                                               int * ierr);

/* Set a compound meshing constraint on the model entities of dimension `dim'
 * and tags `tags'. During meshing, compound entities are treated as a single
 * discrete entity, which is automatically reparametrized. */
GMSH_API void gmshModelMeshSetCompound(const int dim,
                                       const int * tags, const size_t tags_n,
                                       int * ierr);

/* Set meshing constraints on the bounding surfaces of the volume of tag `tag'
 * so that all surfaces are oriented with outward pointing normals; and if a
 * mesh already exists, reorient it. Currently only available with the
 * OpenCASCADE kernel, as it relies on the STL triangulation. */
GMSH_API void gmshModelMeshSetOutwardOrientation(const int tag,
                                                 int * ierr);

/* Remove all meshing constraints from the model entities `dimTags', given as
 * a vector of (dim, tag) pairs. If `dimTags' is empty, remove all
 * constraings. */
GMSH_API void gmshModelMeshRemoveConstraints(const int * dimTags, const size_t dimTags_n,
                                             int * ierr);

/* Embed the model entities of dimension `dim' and tags `tags' in the
 * (`inDim', `inTag') model entity. The dimension `dim' can 0, 1 or 2 and must
 * be strictly smaller than `inDim', which must be either 2 or 3. The embedded
 * entities should not intersect each other or be part of the boundary of the
 * entity `inTag', whose mesh will conform to the mesh of the embedded
 * entities. With the OpenCASCADE kernel, if the `fragment' operation is
 * applied to entities of different dimensions, the lower dimensional entities
 * will be automatically embedded in the higher dimensional entities if they
 * are not on their boundary. */
GMSH_API void gmshModelMeshEmbed(const int dim,
                                 const int * tags, const size_t tags_n,
                                 const int inDim,
                                 const int inTag,
                                 int * ierr);

/* Remove embedded entities from the model entities `dimTags', given as a
 * vector of (dim, tag) pairs. if `dim' is >= 0, only remove embedded entities
 * of the given dimension (e.g. embedded points if `dim' == 0). */
GMSH_API void gmshModelMeshRemoveEmbedded(const int * dimTags, const size_t dimTags_n,
                                          const int dim,
                                          int * ierr);

/* Get the entities (if any) embedded in the model entity of dimension `dim'
 * and tag `tag'. */
GMSH_API void gmshModelMeshGetEmbedded(const int dim,
                                       const int tag,
                                       int ** dimTags, size_t * dimTags_n,
                                       int * ierr);

/* Reorder the elements of type `elementType' classified on the entity of tag
 * `tag' according to the `ordering' vector. */
GMSH_API void gmshModelMeshReorderElements(const int elementType,
                                           const int tag,
                                           const size_t * ordering, const size_t ordering_n,
                                           int * ierr);

/* Compute a renumbering vector `newTags' corresponding to the input tags
 * `oldTags' for a given list of element tags `elementTags'. If `elementTags'
 * is empty, compute the renumbering on the full mesh. If `method' is equal to
 * "RCMK", compute a node renumering with Reverse Cuthill McKee. If `method'
 * is equal to "Hilbert", compute a node renumering along a Hilbert curve. If
 * `method' is equal to "Metis", compute a node renumering using Metis.
 * Element renumbering is not available yet. */
GMSH_API void gmshModelMeshComputeRenumbering(size_t ** oldTags, size_t * oldTags_n,
                                              size_t ** newTags, size_t * newTags_n,
                                              const char * method,
                                              const size_t * elementTags, const size_t elementTags_n,
                                              int * ierr);

/* Renumber the node tags. If no explicit renumbering is provided through the
 * `oldTags' and `newTags' vectors, renumber the nodes in a continuous
 * sequence, taking into account the subset of elements to be saved later on
 * if the option "Mesh.SaveAll" is not set. */
GMSH_API void gmshModelMeshRenumberNodes(const size_t * oldTags, const size_t oldTags_n,
                                         const size_t * newTags, const size_t newTags_n,
                                         int * ierr);

/* Renumber the element tags in a continuous sequence. If no explicit
 * renumbering is provided through the `oldTags' and `newTags' vectors,
 * renumber the elements in a continuous sequence, taking into account the
 * subset of elements to be saved later on if the option "Mesh.SaveAll" is not
 * set. */
GMSH_API void gmshModelMeshRenumberElements(const size_t * oldTags, const size_t oldTags_n,
                                            const size_t * newTags, const size_t newTags_n,
                                            int * ierr);

/* Set the meshes of the entities of dimension `dim' and tag `tags' as
 * periodic copies of the meshes of entities `tagsMaster', using the affine
 * transformation specified in `affineTransformation' (16 entries of a 4x4
 * matrix, by row). If used after meshing, generate the periodic node
 * correspondence information assuming the meshes of entities `tags'
 * effectively match the meshes of entities `tagsMaster' (useful for
 * structured and extruded meshes). Currently only available for @code{dim} ==
 * 1 and @code{dim} == 2. */
GMSH_API void gmshModelMeshSetPeriodic(const int dim,
                                       const int * tags, const size_t tags_n,
                                       const int * tagsMaster, const size_t tagsMaster_n,
                                       const double * affineTransform, const size_t affineTransform_n,
                                       int * ierr);

/* Get master entities `tagsMaster' for the entities of dimension `dim' and
 * tags `tags'. */
GMSH_API void gmshModelMeshGetPeriodic(const int dim,
                                       const int * tags, const size_t tags_n,
                                       int ** tagMaster, size_t * tagMaster_n,
                                       int * ierr);

/* Get the master entity `tagMaster', the node tags `nodeTags' and their
 * corresponding master node tags `nodeTagsMaster', and the affine transform
 * `affineTransform' for the entity of dimension `dim' and tag `tag'. If
 * `includeHighOrderNodes' is set, include high-order nodes in the returned
 * data. */
GMSH_API void gmshModelMeshGetPeriodicNodes(const int dim,
                                            const int tag,
                                            int * tagMaster,
                                            size_t ** nodeTags, size_t * nodeTags_n,
                                            size_t ** nodeTagsMaster, size_t * nodeTagsMaster_n,
                                            double ** affineTransform, size_t * affineTransform_n,
                                            const int includeHighOrderNodes,
                                            int * ierr);

/* Get the master entity `tagMaster' and the key pairs (`typeKeyMaster',
 * `entityKeyMaster') corresponding to the entity `tag' and the key pairs
 * (`typeKey', `entityKey') for the elements of type `elementType' and
 * function space type `functionSpaceType'. If `returnCoord' is set, the
 * `coord' and `coordMaster' vectors contain the x, y, z coordinates locating
 * basis functions for sorting purposes. */
GMSH_API void gmshModelMeshGetPeriodicKeys(const int elementType,
                                           const char * functionSpaceType,
                                           const int tag,
                                           int * tagMaster,
                                           int ** typeKeys, size_t * typeKeys_n,
                                           int ** typeKeysMaster, size_t * typeKeysMaster_n,
                                           size_t ** entityKeys, size_t * entityKeys_n,
                                           size_t ** entityKeysMaster, size_t * entityKeysMaster_n,
                                           double ** coord, size_t * coord_n,
                                           double ** coordMaster, size_t * coordMaster_n,
                                           const int returnCoord,
                                           int * ierr);

/* Import the model STL representation (if available) as the current mesh. */
GMSH_API void gmshModelMeshImportStl(int * ierr);

/* Get the `tags' of any duplicate nodes in the mesh of the entities
 * `dimTags', given as a vector of (dim, tag) pairs. If `dimTags' is empty,
 * consider the whole mesh. */
GMSH_API void gmshModelMeshGetDuplicateNodes(size_t ** tags, size_t * tags_n,
                                             const int * dimTags, const size_t dimTags_n,
                                             int * ierr);

/* Remove duplicate nodes in the mesh of the entities `dimTags', given as a
 * vector of (dim, tag) pairs. If `dimTags' is empty, consider the whole mesh. */
GMSH_API void gmshModelMeshRemoveDuplicateNodes(const int * dimTags, const size_t dimTags_n,
                                                int * ierr);

/* Remove duplicate elements (defined by the same nodes, in the same entity)
 * in the mesh of the entities `dimTags', given as a vector of (dim, tag)
 * pairs. If `dimTags' is empty, consider the whole mesh. */
GMSH_API void gmshModelMeshRemoveDuplicateElements(const int * dimTags, const size_t dimTags_n,
                                                   int * ierr);

/* Split (into two triangles) all quadrangles in surface `tag' whose quality
 * is lower than `quality'. If `tag' < 0, split quadrangles in all surfaces. */
GMSH_API void gmshModelMeshSplitQuadrangles(const double quality,
                                            const int tag,
                                            int * ierr);

/* Set the visibility of the elements of tags `elementTags' to `value'. */
GMSH_API void gmshModelMeshSetVisibility(const size_t * elementTags, const size_t elementTags_n,
                                         const int value,
                                         int * ierr);

/* Get the visibility of the elements of tags `elementTags'. */
GMSH_API void gmshModelMeshGetVisibility(const size_t * elementTags, const size_t elementTags_n,
                                         int ** values, size_t * values_n,
                                         int * ierr);

/* Classify ("color") the surface mesh based on the angle threshold `angle'
 * (in radians), and create new discrete surfaces, curves and points
 * accordingly. If `boundary' is set, also create discrete curves on the
 * boundary if the surface is open. If `forReparametrization' is set, create
 * curves and surfaces that can be reparametrized using a single map. If
 * `curveAngle' is less than Pi, also force curves to be split according to
 * `curveAngle'. If `exportDiscrete' is set, clear any built-in CAD kernel
 * entities and export the discrete entities in the built-in CAD kernel. */
GMSH_API void gmshModelMeshClassifySurfaces(const double angle,
                                            const int boundary,
                                            const int forReparametrization,
                                            const double curveAngle,
                                            const int exportDiscrete,
                                            int * ierr);

/* Create a geometry for the discrete entities `dimTags' (given as a vector of
 * (dim, tag) pairs) represented solely by a mesh (without an underlying CAD
 * description), i.e. create a parametrization for discrete curves and
 * surfaces, assuming that each can be parametrized with a single map. If
 * `dimTags' is empty, create a geometry for all the discrete entities. */
GMSH_API void gmshModelMeshCreateGeometry(const int * dimTags, const size_t dimTags_n,
                                          int * ierr);

/* Create a boundary representation from the mesh if the model does not have
 * one (e.g. when imported from mesh file formats with no BRep representation
 * of the underlying model). If `makeSimplyConnected' is set, enforce simply
 * connected discrete surfaces and volumes. If `exportDiscrete' is set, clear
 * any built-in CAD kernel entities and export the discrete entities in the
 * built-in CAD kernel. */
GMSH_API void gmshModelMeshCreateTopology(const int makeSimplyConnected,
                                          const int exportDiscrete,
                                          int * ierr);

/* Add a request to compute a basis representation for homology spaces (if
 * `type' == "Homology") or cohomology spaces (if `type' == "Cohomology"). The
 * computation domain is given in a list of physical group tags `domainTags';
 * if empty, the whole mesh is the domain. The computation subdomain for
 * relative (co)homology computation is given in a list of physical group tags
 * `subdomainTags'; if empty, absolute (co)homology is computed. The
 * dimensions of the (co)homology bases to be computed are given in the list
 * `dim'; if empty, all bases are computed. Resulting basis representation
 * (co)chains are stored as physical groups in the mesh. If the request is
 * added before mesh generation, the computation will be performed at the end
 * of the meshing pipeline. */
GMSH_API void gmshModelMeshAddHomologyRequest(const char * type,
                                              const int * domainTags, const size_t domainTags_n,
                                              const int * subdomainTags, const size_t subdomainTags_n,
                                              const int * dims, const size_t dims_n,
                                              int * ierr);

/* Clear all (co)homology computation requests. */
GMSH_API void gmshModelMeshClearHomologyRequests(int * ierr);

/* Perform the (co)homology computations requested by addHomologyRequest().
 * The newly created physical groups are returned in `dimTags' as a vector of
 * (dim, tag) pairs. */
GMSH_API void gmshModelMeshComputeHomology(int ** dimTags, size_t * dimTags_n,
                                           int * ierr);

/* Compute a cross field for the current mesh. The function creates 3 views:
 * the H function, the Theta function and cross directions. Return the tags of
 * the views. */
GMSH_API void gmshModelMeshComputeCrossField(int ** viewTags, size_t * viewTags_n,
                                             int * ierr);

/* Triangulate the points given in the `coord' vector as pairs of u, v
 * coordinates, and return the node tags (with numbering starting at 1) of the
 * resulting triangles in `tri'. */
GMSH_API void gmshModelMeshTriangulate(const double * coord, const size_t coord_n,
                                       size_t ** tri, size_t * tri_n,
                                       int * ierr);

/* Tetrahedralize the points given in the `coord' vector as x, y, z
 * coordinates, concatenated, and return the node tags (with numbering
 * starting at 1) of the resulting tetrahedra in `tetra'. */
GMSH_API void gmshModelMeshTetrahedralize(const double * coord, const size_t coord_n,
                                          size_t ** tetra, size_t * tetra_n,
                                          int * ierr);

/* Add a new mesh size field of type `fieldType'. If `tag' is positive, assign
 * the tag explicitly; otherwise a new tag is assigned automatically. Return
 * the field tag. Available field types are listed in the "Gmsh mesh size
 * fields" chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-mesh-size-fields). */
GMSH_API int gmshModelMeshFieldAdd(const char * fieldType,
                                   const int tag,
                                   int * ierr);

/* Remove the field with tag `tag'. */
GMSH_API void gmshModelMeshFieldRemove(const int tag,
                                       int * ierr);

/* Get the list of all fields. */
GMSH_API void gmshModelMeshFieldList(int ** tags, size_t * tags_n,
                                     int * ierr);

/* Get the type `fieldType' of the field with tag `tag'. */
GMSH_API void gmshModelMeshFieldGetType(const int tag,
                                        char ** fileType,
                                        int * ierr);

/* Set the numerical option `option' to value `value' for field `tag'. */
GMSH_API void gmshModelMeshFieldSetNumber(const int tag,
                                          const char * option,
                                          const double value,
                                          int * ierr);

/* Get the value of the numerical option `option' for field `tag'. */
GMSH_API void gmshModelMeshFieldGetNumber(const int tag,
                                          const char * option,
                                          double * value,
                                          int * ierr);

/* Set the string option `option' to value `value' for field `tag'. */
GMSH_API void gmshModelMeshFieldSetString(const int tag,
                                          const char * option,
                                          const char * value,
                                          int * ierr);

/* Get the value of the string option `option' for field `tag'. */
GMSH_API void gmshModelMeshFieldGetString(const int tag,
                                          const char * option,
                                          char ** value,
                                          int * ierr);

/* Set the numerical list option `option' to value `values' for field `tag'. */
GMSH_API void gmshModelMeshFieldSetNumbers(const int tag,
                                           const char * option,
                                           const double * values, const size_t values_n,
                                           int * ierr);

/* Get the value of the numerical list option `option' for field `tag'. */
GMSH_API void gmshModelMeshFieldGetNumbers(const int tag,
                                           const char * option,
                                           double ** values, size_t * values_n,
                                           int * ierr);

/* Set the field `tag' as the background mesh size field. */
GMSH_API void gmshModelMeshFieldSetAsBackgroundMesh(const int tag,
                                                    int * ierr);

/* Set the field `tag' as a boundary layer size field. */
GMSH_API void gmshModelMeshFieldSetAsBoundaryLayer(const int tag,
                                                   int * ierr);

/* Add a geometrical point in the built-in CAD representation, at coordinates
 * (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint at that
 * point. If `tag' is positive, set the tag explicitly; otherwise a new tag is
 * selected automatically. Return the tag of the point. (Note that the point
 * will be added in the current model only after `synchronize' is called. This
 * behavior holds for all the entities added in the geo module.) */
GMSH_API int gmshModelGeoAddPoint(const double x,
                                  const double y,
                                  const double z,
                                  const double meshSize,
                                  const int tag,
                                  int * ierr);

/* Add a straight line segment in the built-in CAD representation, between the
 * two points with tags `startTag' and `endTag'. If `tag' is positive, set the
 * tag explicitly; otherwise a new tag is selected automatically. Return the
 * tag of the line. */
GMSH_API int gmshModelGeoAddLine(const int startTag,
                                 const int endTag,
                                 const int tag,
                                 int * ierr);

/* Add a circle arc (strictly smaller than Pi) in the built-in CAD
 * representation, between the two points with tags `startTag' and `endTag',
 * and with center `centerTag'. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. If (`nx', `ny', `nz') != (0,
 * 0, 0), explicitly set the plane of the circle arc. Return the tag of the
 * circle arc. */
GMSH_API int gmshModelGeoAddCircleArc(const int startTag,
                                      const int centerTag,
                                      const int endTag,
                                      const int tag,
                                      const double nx,
                                      const double ny,
                                      const double nz,
                                      int * ierr);

/* Add an ellipse arc (strictly smaller than Pi) in the built-in CAD
 * representation, between the two points `startTag' and `endTag', and with
 * center `centerTag' and major axis point `majorTag'. If `tag' is positive,
 * set the tag explicitly; otherwise a new tag is selected automatically. If
 * (`nx', `ny', `nz') != (0, 0, 0), explicitly set the plane of the circle
 * arc. Return the tag of the ellipse arc. */
GMSH_API int gmshModelGeoAddEllipseArc(const int startTag,
                                       const int centerTag,
                                       const int majorTag,
                                       const int endTag,
                                       const int tag,
                                       const double nx,
                                       const double ny,
                                       const double nz,
                                       int * ierr);

/* Add a spline (Catmull-Rom) curve in the built-in CAD representation, going
 * through the points `pointTags'. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Create a
 * periodic curve if the first and last points are the same. Return the tag of
 * the spline curve. */
GMSH_API int gmshModelGeoAddSpline(const int * pointTags, const size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a cubic b-spline curve in the built-in CAD representation, with
 * `pointTags' control points. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Creates a periodic curve if
 * the first and last points are the same. Return the tag of the b-spline
 * curve. */
GMSH_API int gmshModelGeoAddBSpline(const int * pointTags, const size_t pointTags_n,
                                    const int tag,
                                    int * ierr);

/* Add a Bezier curve in the built-in CAD representation, with `pointTags'
 * control points. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically.  Return the tag of the Bezier curve. */
GMSH_API int gmshModelGeoAddBezier(const int * pointTags, const size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a polyline curve in the built-in CAD representation, going through the
 * points `pointTags'. If `tag' is positive, set the tag explicitly; otherwise
 * a new tag is selected automatically. Create a periodic curve if the first
 * and last points are the same. Return the tag of the polyline curve. */
GMSH_API int gmshModelGeoAddPolyline(const int * pointTags, const size_t pointTags_n,
                                     const int tag,
                                     int * ierr);

/* Add a spline (Catmull-Rom) curve in the built-in CAD representation, going
 * through points sampling the curves in `curveTags'. The density of sampling
 * points on each curve is governed by `numIntervals'. If `tag' is positive,
 * set the tag explicitly; otherwise a new tag is selected automatically.
 * Return the tag of the spline. */
GMSH_API int gmshModelGeoAddCompoundSpline(const int * curveTags, const size_t curveTags_n,
                                           const int numIntervals,
                                           const int tag,
                                           int * ierr);

/* Add a b-spline curve in the built-in CAD representation, with control
 * points sampling the curves in `curveTags'. The density of sampling points
 * on each curve is governed by `numIntervals'. If `tag' is positive, set the
 * tag explicitly; otherwise a new tag is selected automatically. Return the
 * tag of the b-spline. */
GMSH_API int gmshModelGeoAddCompoundBSpline(const int * curveTags, const size_t curveTags_n,
                                            const int numIntervals,
                                            const int tag,
                                            int * ierr);

/* Add a curve loop (a closed wire) in the built-in CAD representation, formed
 * by the curves `curveTags'. `curveTags' should contain (signed) tags of
 * model entities of dimension 1 forming a closed loop: a negative tag
 * signifies that the underlying curve is considered with reversed
 * orientation. If `tag' is positive, set the tag explicitly; otherwise a new
 * tag is selected automatically. If `reorient' is set, automatically reorient
 * the curves if necessary. Return the tag of the curve loop. */
GMSH_API int gmshModelGeoAddCurveLoop(const int * curveTags, const size_t curveTags_n,
                                      const int tag,
                                      const int reorient,
                                      int * ierr);

/* Add curve loops in the built-in CAD representation based on the curves
 * `curveTags'. Return the `tags' of found curve loops, if any. */
GMSH_API void gmshModelGeoAddCurveLoops(const int * curveTags, const size_t curveTags_n,
                                        int ** tags, size_t * tags_n,
                                        int * ierr);

/* Add a plane surface in the built-in CAD representation, defined by one or
 * more curve loops `wireTags'. The first curve loop defines the exterior
 * contour; additional curve loop define holes. If `tag' is positive, set the
 * tag explicitly; otherwise a new tag is selected automatically. Return the
 * tag of the surface. */
GMSH_API int gmshModelGeoAddPlaneSurface(const int * wireTags, const size_t wireTags_n,
                                         const int tag,
                                         int * ierr);

/* Add a surface in the built-in CAD representation, filling the curve loops
 * in `wireTags' using transfinite interpolation. Currently only a single
 * curve loop is supported; this curve loop should be composed by 3 or 4
 * curves only. If `tag' is positive, set the tag explicitly; otherwise a new
 * tag is selected automatically. Return the tag of the surface. */
GMSH_API int gmshModelGeoAddSurfaceFilling(const int * wireTags, const size_t wireTags_n,
                                           const int tag,
                                           const int sphereCenterTag,
                                           int * ierr);

/* Add a surface loop (a closed shell) formed by `surfaceTags' in the built-in
 * CAD representation.  If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the shell. */
GMSH_API int gmshModelGeoAddSurfaceLoop(const int * surfaceTags, const size_t surfaceTags_n,
                                        const int tag,
                                        int * ierr);

/* Add a volume (a region) in the built-in CAD representation, defined by one
 * or more shells `shellTags'. The first surface loop defines the exterior
 * boundary; additional surface loop define holes. If `tag' is positive, set
 * the tag explicitly; otherwise a new tag is selected automatically. Return
 * the tag of the volume. */
GMSH_API int gmshModelGeoAddVolume(const int * shellTags, const size_t shellTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a `geometry' in the built-in CAD representation. `geometry' can
 * currently be one of "Sphere" or "PolarSphere" (where `numbers' should
 * contain the x, y, z coordinates of the center, followed by the radius), or
 * "Parametric" (where `strings' should contains three expression evaluating
 * to the x, y and z coordinates. If `tag' is positive, set the tag of the
 * geometry explicitly; otherwise a new tag is selected automatically. Return
 * the tag of the geometry. */
GMSH_API int gmshModelGeoAddGeometry(const char * geometry,
                                     const double * numbers, const size_t numbers_n,
                                     const char * const * strings, const size_t strings_n,
                                     const int tag,
                                     int * ierr);

/* Add a point in the built-in CAD representation, at coordinates (`x', `y',
 * `z') on the geometry `geometryTag'. If `meshSize' is > 0, add a meshing
 * constraint at that point. If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the point.
 * For surface geometries, only the `x' and `y' coordinates are used. */
GMSH_API int gmshModelGeoAddPointOnGeometry(const int geometryTag,
                                            const double x,
                                            const double y,
                                            const double z,
                                            const double meshSize,
                                            const int tag,
                                            int * ierr);

/* Extrude the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation, using a translation along (`dx', `dy',
 * `dz'). Return extruded entities in `outDimTags'. If the `numElements'
 * vector is not empty, also extrude the mesh: the entries in `numElements'
 * give the number of elements in each layer. If the `height' vector is not
 * empty, it provides the (cumulative) height of the different layers,
 * normalized to 1. If `recombine' is set, recombine the mesh in the layers. */
GMSH_API void gmshModelGeoExtrude(const int * dimTags, const size_t dimTags_n,
                                  const double dx,
                                  const double dy,
                                  const double dz,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const int * numElements, const size_t numElements_n,
                                  const double * heights, const size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Extrude the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation, using a rotation of `angle' radians around
 * the axis of revolution defined by the point (`x', `y', `z') and the
 * direction (`ax', `ay', `az'). The angle should be strictly smaller than Pi.
 * Return extruded entities in `outDimTags'. If the `numElements' vector is
 * not empty, also extrude the mesh: the entries in `numElements' give the
 * number of elements in each layer. If the `height' vector is not empty, it
 * provides the (cumulative) height of the different layers, normalized to 1.
 * If `recombine' is set, recombine the mesh in the layers. */
GMSH_API void gmshModelGeoRevolve(const int * dimTags, const size_t dimTags_n,
                                  const double x,
                                  const double y,
                                  const double z,
                                  const double ax,
                                  const double ay,
                                  const double az,
                                  const double angle,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const int * numElements, const size_t numElements_n,
                                  const double * heights, const size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Extrude the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation, using a combined translation and rotation
 * of `angle' radians, along (`dx', `dy', `dz') and around the axis of
 * revolution defined by the point (`x', `y', `z') and the direction (`ax',
 * `ay', `az'). The angle should be strictly smaller than Pi. Return extruded
 * entities in `outDimTags'. If the `numElements' vector is not empty, also
 * extrude the mesh: the entries in `numElements' give the number of elements
 * in each layer. If the `height' vector is not empty, it provides the
 * (cumulative) height of the different layers, normalized to 1. If
 * `recombine' is set, recombine the mesh in the layers. */
GMSH_API void gmshModelGeoTwist(const int * dimTags, const size_t dimTags_n,
                                const double x,
                                const double y,
                                const double z,
                                const double dx,
                                const double dy,
                                const double dz,
                                const double ax,
                                const double ay,
                                const double az,
                                const double angle,
                                int ** outDimTags, size_t * outDimTags_n,
                                const int * numElements, const size_t numElements_n,
                                const double * heights, const size_t heights_n,
                                const int recombine,
                                int * ierr);

/* Extrude the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation along the normals of the mesh, creating
 * discrete boundary layer entities. Return extruded entities in `outDimTags'.
 * The entries in `numElements' give the number of elements in each layer. If
 * the `height' vector is not empty, it provides the (cumulative) height of
 * the different layers. If `recombine' is set, recombine the mesh in the
 * layers. A second boundary layer can be created from the same entities if
 * `second' is set. If `viewIndex' is >= 0, use the corresponding view to
 * either specify the normals (if the view contains a vector field) or scale
 * the normals (if the view is scalar). */
GMSH_API void gmshModelGeoExtrudeBoundaryLayer(const int * dimTags, const size_t dimTags_n,
                                               int ** outDimTags, size_t * outDimTags_n,
                                               const int * numElements, const size_t numElements_n,
                                               const double * heights, const size_t heights_n,
                                               const int recombine,
                                               const int second,
                                               const int viewIndex,
                                               int * ierr);

/* Translate the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation along (`dx', `dy', `dz'). */
GMSH_API void gmshModelGeoTranslate(const int * dimTags, const size_t dimTags_n,
                                    const double dx,
                                    const double dy,
                                    const double dz,
                                    int * ierr);

/* Rotate the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation by `angle' radians around the axis of
 * revolution defined by the point (`x', `y', `z') and the direction (`ax',
 * `ay', `az'). */
GMSH_API void gmshModelGeoRotate(const int * dimTags, const size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double ax,
                                 const double ay,
                                 const double az,
                                 const double angle,
                                 int * ierr);

/* Scale the entities `dimTags' (given as a vector of (dim, tag) pairs) in the
 * built-in CAD representation by factors `a', `b' and `c' along the three
 * coordinate axes; use (`x', `y', `z') as the center of the homothetic
 * transformation. */
GMSH_API void gmshModelGeoDilate(const int * dimTags, const size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double a,
                                 const double b,
                                 const double c,
                                 int * ierr);

/* Mirror the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation, with respect to the plane of equation `a'
 * * x + `b' * y + `c' * z + `d' = 0. */
GMSH_API void gmshModelGeoMirror(const int * dimTags, const size_t dimTags_n,
                                 const double a,
                                 const double b,
                                 const double c,
                                 const double d,
                                 int * ierr);

/* Mirror the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation, with respect to the plane of equation `a'
 * * x + `b' * y + `c' * z + `d' = 0. (This is a synonym for `mirror', which
 * will be deprecated in a future release.) */
GMSH_API void gmshModelGeoSymmetrize(const int * dimTags, const size_t dimTags_n,
                                     const double a,
                                     const double b,
                                     const double c,
                                     const double d,
                                     int * ierr);

/* Copy the entities `dimTags' (given as a vector of (dim, tag) pairs) in the
 * built-in CAD representation; the new entities are returned in `outDimTags'. */
GMSH_API void gmshModelGeoCopy(const int * dimTags, const size_t dimTags_n,
                               int ** outDimTags, size_t * outDimTags_n,
                               int * ierr);

/* Remove the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the built-in CAD representation, provided that they are not on the boundary
 * of higher-dimensional entities. If `recursive' is true, remove all the
 * entities on their boundaries, down to dimension 0. */
GMSH_API void gmshModelGeoRemove(const int * dimTags, const size_t dimTags_n,
                                 const int recursive,
                                 int * ierr);

/* Remove all duplicate entities in the built-in CAD representation (different
 * entities at the same geometrical location). */
GMSH_API void gmshModelGeoRemoveAllDuplicates(int * ierr);

/* Split the curve of tag `tag' in the built-in CAD representation, on the
 * specified control points `pointTags'. This feature is only available for
 * lines, splines and b-splines. Return the tag(s) `curveTags' of the newly
 * created curve(s). */
GMSH_API void gmshModelGeoSplitCurve(const int tag,
                                     const int * pointTags, const size_t pointTags_n,
                                     int ** curveTags, size_t * curveTags_n,
                                     int * ierr);

/* Get the maximum tag of entities of dimension `dim' in the built-in CAD
 * representation. */
GMSH_API int gmshModelGeoGetMaxTag(const int dim,
                                   int * ierr);

/* Set the maximum tag `maxTag' for entities of dimension `dim' in the built-
 * in CAD representation. */
GMSH_API void gmshModelGeoSetMaxTag(const int dim,
                                    const int maxTag,
                                    int * ierr);

/* Add a physical group of dimension `dim', grouping the entities with tags
 * `tags' in the built-in CAD representation. Return the tag of the physical
 * group, equal to `tag' if `tag' is positive, or a new tag if `tag' < 0. Set
 * the name of the physical group if `name' is not empty. */
GMSH_API int gmshModelGeoAddPhysicalGroup(const int dim,
                                          const int * tags, const size_t tags_n,
                                          const int tag,
                                          const char * name,
                                          int * ierr);

/* Remove the physical groups `dimTags' (given as a vector of (dim, tag)
 * pairs) from the built-in CAD representation. If `dimTags' is empty, remove
 * all groups. */
GMSH_API void gmshModelGeoRemovePhysicalGroups(const int * dimTags, const size_t dimTags_n,
                                               int * ierr);

/* Synchronize the built-in CAD representation with the current Gmsh model.
 * This can be called at any time, but since it involves a non trivial amount
 * of processing, the number of synchronization points should normally be
 * minimized. Without synchronization the entities in the built-in CAD
 * representation are not available to any function outside of the built-in
 * CAD kernel functions. */
GMSH_API void gmshModelGeoSynchronize(int * ierr);

/* Set a mesh size constraint on the entities `dimTags' (given as a vector of
 * (dim, tag) pairs) in the built-in CAD kernel representation. Currently only
 * entities of dimension 0 (points) are handled. */
GMSH_API void gmshModelGeoMeshSetSize(const int * dimTags, const size_t dimTags_n,
                                      const double size,
                                      int * ierr);

/* Set a transfinite meshing constraint on the curve `tag' in the built-in CAD
 * kernel representation, with `numNodes' nodes distributed according to
 * `meshType' and `coef'. Currently supported types are "Progression"
 * (geometrical progression with power `coef') and "Bump" (refinement toward
 * both extremities of the curve). */
GMSH_API void gmshModelGeoMeshSetTransfiniteCurve(const int tag,
                                                  const int nPoints,
                                                  const char * meshType,
                                                  const double coef,
                                                  int * ierr);

/* Set a transfinite meshing constraint on the surface `tag' in the built-in
 * CAD kernel representation. `arrangement' describes the arrangement of the
 * triangles when the surface is not flagged as recombined: currently
 * supported values are "Left", "Right", "AlternateLeft" and "AlternateRight".
 * `cornerTags' can be used to specify the (3 or 4) corners of the transfinite
 * interpolation explicitly; specifying the corners explicitly is mandatory if
 * the surface has more that 3 or 4 points on its boundary. */
GMSH_API void gmshModelGeoMeshSetTransfiniteSurface(const int tag,
                                                    const char * arrangement,
                                                    const int * cornerTags, const size_t cornerTags_n,
                                                    int * ierr);

/* Set a transfinite meshing constraint on the surface `tag' in the built-in
 * CAD kernel representation. `cornerTags' can be used to specify the (6 or 8)
 * corners of the transfinite interpolation explicitly. */
GMSH_API void gmshModelGeoMeshSetTransfiniteVolume(const int tag,
                                                   const int * cornerTags, const size_t cornerTags_n,
                                                   int * ierr);

/* Set a recombination meshing constraint on the entity of dimension `dim' and
 * tag `tag' in the built-in CAD kernel representation. Currently only
 * entities of dimension 2 (to recombine triangles into quadrangles) are
 * supported; `angle' specifies the threshold angle for the simple
 * recombination algorithm. */
GMSH_API void gmshModelGeoMeshSetRecombine(const int dim,
                                           const int tag,
                                           const double angle,
                                           int * ierr);

/* Set a smoothing meshing constraint on the entity of dimension `dim' and tag
 * `tag' in the built-in CAD kernel representation. `val' iterations of a
 * Laplace smoother are applied. */
GMSH_API void gmshModelGeoMeshSetSmoothing(const int dim,
                                           const int tag,
                                           const int val,
                                           int * ierr);

/* Set a reverse meshing constraint on the entity of dimension `dim' and tag
 * `tag' in the built-in CAD kernel representation. If `val' is true, the mesh
 * orientation will be reversed with respect to the natural mesh orientation
 * (i.e. the orientation consistent with the orientation of the geometry). If
 * `val' is false, the mesh is left as-is. */
GMSH_API void gmshModelGeoMeshSetReverse(const int dim,
                                         const int tag,
                                         const int val,
                                         int * ierr);

/* Set the meshing algorithm on the entity of dimension `dim' and tag `tag' in
 * the built-in CAD kernel representation. Currently only supported for `dim'
 * == 2. */
GMSH_API void gmshModelGeoMeshSetAlgorithm(const int dim,
                                           const int tag,
                                           const int val,
                                           int * ierr);

/* Force the mesh size to be extended from the boundary, or not, for the
 * entity of dimension `dim' and tag `tag' in the built-in CAD kernel
 * representation. Currently only supported for `dim' == 2. */
GMSH_API void gmshModelGeoMeshSetSizeFromBoundary(const int dim,
                                                  const int tag,
                                                  const int val,
                                                  int * ierr);

/* Add a geometrical point in the OpenCASCADE CAD representation, at
 * coordinates (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint
 * at that point. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the point. (Note that
 * the point will be added in the current model only after `synchronize' is
 * called. This behavior holds for all the entities added in the occ module.) */
GMSH_API int gmshModelOccAddPoint(const double x,
                                  const double y,
                                  const double z,
                                  const double meshSize,
                                  const int tag,
                                  int * ierr);

/* Add a straight line segment in the OpenCASCADE CAD representation, between
 * the two points with tags `startTag' and `endTag'. If `tag' is positive, set
 * the tag explicitly; otherwise a new tag is selected automatically. Return
 * the tag of the line. */
GMSH_API int gmshModelOccAddLine(const int startTag,
                                 const int endTag,
                                 const int tag,
                                 int * ierr);

/* Add a circle arc in the OpenCASCADE CAD representation, between the two
 * points with tags `startTag' and `endTag', with middle point `middleTag'. If
 * `center' is true, the middle point is the center of the circle; otherwise
 * the circle goes through the middle point. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Return the tag
 * of the circle arc. */
GMSH_API int gmshModelOccAddCircleArc(const int startTag,
                                      const int middleTag,
                                      const int endTag,
                                      const int tag,
                                      const int center,
                                      int * ierr);

/* Add a circle of center (`x', `y', `z') and radius `r' in the OpenCASCADE
 * CAD representation. If `tag' is positive, set the tag explicitly; otherwise
 * a new tag is selected automatically. If `angle1' and `angle2' are
 * specified, create a circle arc between the two angles. If a vector `zAxis'
 * of size 3 is provided, use it as the normal to the circle plane (z-axis).
 * If a vector `xAxis' of size 3 is provided in addition to `zAxis', use it to
 * define the x-axis. Return the tag of the circle. */
GMSH_API int gmshModelOccAddCircle(const double x,
                                   const double y,
                                   const double z,
                                   const double r,
                                   const int tag,
                                   const double angle1,
                                   const double angle2,
                                   const double * zAxis, const size_t zAxis_n,
                                   const double * xAxis, const size_t xAxis_n,
                                   int * ierr);

/* Add an ellipse arc in the OpenCASCADE CAD representation, between the two
 * points `startTag' and `endTag', and with center `centerTag' and major axis
 * point `majorTag'. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the ellipse arc. Note
 * that OpenCASCADE does not allow creating ellipse arcs with the major radius
 * smaller than the minor radius. */
GMSH_API int gmshModelOccAddEllipseArc(const int startTag,
                                       const int centerTag,
                                       const int majorTag,
                                       const int endTag,
                                       const int tag,
                                       int * ierr);

/* Add an ellipse of center (`x', `y', `z') and radii `r1' and `r2' (with `r1'
 * >= `r2') along the x- and y-axes, respectively, in the OpenCASCADE CAD
 * representation. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. If `angle1' and `angle2' are specified,
 * create an ellipse arc between the two angles. If a vector `zAxis' of size 3
 * is provided, use it as the normal to the ellipse plane (z-axis). If a
 * vector `xAxis' of size 3 is provided in addition to `zAxis', use it to
 * define the x-axis. Return the tag of the ellipse. */
GMSH_API int gmshModelOccAddEllipse(const double x,
                                    const double y,
                                    const double z,
                                    const double r1,
                                    const double r2,
                                    const int tag,
                                    const double angle1,
                                    const double angle2,
                                    const double * zAxis, const size_t zAxis_n,
                                    const double * xAxis, const size_t xAxis_n,
                                    int * ierr);

/* Add a spline (C2 b-spline) curve in the OpenCASCADE CAD representation,
 * going through the points `pointTags'. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Create a
 * periodic curve if the first and last points are the same. Return the tag of
 * the spline curve. If the `tangents' vector contains 6 entries, use them as
 * concatenated x, y, z components of the initial and final tangents of the
 * b-spline; if it contains 3 times as many entries as the number of points,
 * use them as concatenated x, y, z components of the tangents at each point,
 * unless the norm of the tangent is zero. */
GMSH_API int gmshModelOccAddSpline(const int * pointTags, const size_t pointTags_n,
                                   const int tag,
                                   const double * tangents, const size_t tangents_n,
                                   int * ierr);

/* Add a b-spline curve of degree `degree' in the OpenCASCADE CAD
 * representation, with `pointTags' control points. If `weights', `knots' or
 * `multiplicities' are not provided, default parameters are computed
 * automatically. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Create a periodic curve if the first and
 * last points are the same. Return the tag of the b-spline curve. */
GMSH_API int gmshModelOccAddBSpline(const int * pointTags, const size_t pointTags_n,
                                    const int tag,
                                    const int degree,
                                    const double * weights, const size_t weights_n,
                                    const double * knots, const size_t knots_n,
                                    const int * multiplicities, const size_t multiplicities_n,
                                    int * ierr);

/* Add a Bezier curve in the OpenCASCADE CAD representation, with `pointTags'
 * control points. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the Bezier curve. */
GMSH_API int gmshModelOccAddBezier(const int * pointTags, const size_t pointTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a wire (open or closed) in the OpenCASCADE CAD representation, formed
 * by the curves `curveTags'. Note that an OpenCASCADE wire can be made of
 * curves that share geometrically identical (but topologically different)
 * points. If `tag' is positive, set the tag explicitly; otherwise a new tag
 * is selected automatically. Return the tag of the wire. */
GMSH_API int gmshModelOccAddWire(const int * curveTags, const size_t curveTags_n,
                                 const int tag,
                                 const int checkClosed,
                                 int * ierr);

/* Add a curve loop (a closed wire) in the OpenCASCADE CAD representation,
 * formed by the curves `curveTags'. `curveTags' should contain tags of curves
 * forming a closed loop. Negative tags can be specified for compatibility
 * with the built-in kernel, but are simply ignored: the wire is oriented
 * according to the orientation of its first curve. Note that an OpenCASCADE
 * curve loop can be made of curves that share geometrically identical (but
 * topologically different) points. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. Return the tag
 * of the curve loop. */
GMSH_API int gmshModelOccAddCurveLoop(const int * curveTags, const size_t curveTags_n,
                                      const int tag,
                                      int * ierr);

/* Add a rectangle in the OpenCASCADE CAD representation, with lower left
 * corner at (`x', `y', `z') and upper right corner at (`x' + `dx', `y' +
 * `dy', `z'). If `tag' is positive, set the tag explicitly; otherwise a new
 * tag is selected automatically. Round the corners if `roundedRadius' is
 * nonzero. Return the tag of the rectangle. */
GMSH_API int gmshModelOccAddRectangle(const double x,
                                      const double y,
                                      const double z,
                                      const double dx,
                                      const double dy,
                                      const int tag,
                                      const double roundedRadius,
                                      int * ierr);

/* Add a disk in the OpenCASCADE CAD representation, with center (`xc', `yc',
 * `zc') and radius `rx' along the x-axis and `ry' along the y-axis (`rx' >=
 * `ry'). If `tag' is positive, set the tag explicitly; otherwise a new tag is
 * selected automatically. If a vector `zAxis' of size 3 is provided, use it
 * as the normal to the disk (z-axis). If a vector `xAxis' of size 3 is
 * provided in addition to `zAxis', use it to define the x-axis. Return the
 * tag of the disk. */
GMSH_API int gmshModelOccAddDisk(const double xc,
                                 const double yc,
                                 const double zc,
                                 const double rx,
                                 const double ry,
                                 const int tag,
                                 const double * zAxis, const size_t zAxis_n,
                                 const double * xAxis, const size_t xAxis_n,
                                 int * ierr);

/* Add a plane surface in the OpenCASCADE CAD representation, defined by one
 * or more curve loops (or closed wires) `wireTags'. The first curve loop
 * defines the exterior contour; additional curve loop define holes. If `tag'
 * is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the surface. */
GMSH_API int gmshModelOccAddPlaneSurface(const int * wireTags, const size_t wireTags_n,
                                         const int tag,
                                         int * ierr);

/* Add a surface in the OpenCASCADE CAD representation, filling the curve loop
 * `wireTag'. If `tag' is positive, set the tag explicitly; otherwise a new
 * tag is selected automatically. Return the tag of the surface. If
 * `pointTags' are provided, force the surface to pass through the given
 * points. The other optional arguments are `degree' (the degree of the energy
 * criterion to minimize for computing the deformation of the surface),
 * `numPointsOnCurves' (the average number of points for discretisation of the
 * bounding curves), `numIter' (the maximum number of iterations of the
 * optimization process), `anisotropic' (improve performance when the ratio of
 * the length along the two parametric coordinates of the surface is high),
 * `tol2d' (tolerance to the constraints in the parametric plane of the
 * surface), `tol3d' (the maximum distance allowed between the support surface
 * and the constraints), `tolAng' (the maximum angle allowed between the
 * normal of the surface and the constraints), `tolCurv' (the maximum
 * difference of curvature allowed between the surface and the constraint),
 * `maxDegree' (the highest degree which the polynomial defining the filling
 * surface can have) and, `maxSegments' (the largest number of segments which
 * the filling surface can have). */
GMSH_API int gmshModelOccAddSurfaceFilling(const int wireTag,
                                           const int tag,
                                           const int * pointTags, const size_t pointTags_n,
                                           const int degree,
                                           const int numPointsOnCurves,
                                           const int numIter,
                                           const int anisotropic,
                                           const double tol2d,
                                           const double tol3d,
                                           const double tolAng,
                                           const double tolCurv,
                                           const int maxDegree,
                                           const int maxSegments,
                                           int * ierr);

/* Add a BSpline surface in the OpenCASCADE CAD representation, filling the
 * curve loop `wireTag'. The curve loop should be made of 2, 3 or 4 curves.
 * The optional `type' argument specifies the type of filling: "Stretch"
 * creates the flattest patch, "Curved" (the default) creates the most rounded
 * patch, and "Coons" creates a rounded patch with less depth than "Curved".
 * If `tag' is positive, set the tag explicitly; otherwise a new tag is
 * selected automatically. Return the tag of the surface. */
GMSH_API int gmshModelOccAddBSplineFilling(const int wireTag,
                                           const int tag,
                                           const char * type,
                                           int * ierr);

/* Add a Bezier surface in the OpenCASCADE CAD representation, filling the
 * curve loop `wireTag'. The curve loop should be made of 2, 3 or 4 Bezier
 * curves. The optional `type' argument specifies the type of filling:
 * "Stretch" creates the flattest patch, "Curved" (the default) creates the
 * most rounded patch, and "Coons" creates a rounded patch with less depth
 * than "Curved". If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. Return the tag of the surface. */
GMSH_API int gmshModelOccAddBezierFilling(const int wireTag,
                                          const int tag,
                                          const char * type,
                                          int * ierr);

/* Add a b-spline surface of degree `degreeU' x `degreeV' in the OpenCASCADE
 * CAD representation, with `pointTags' control points given as a single
 * vector [Pu1v1, ... Pu`numPointsU'v1, Pu1v2, ...]. If `weights', `knotsU',
 * `knotsV', `multiplicitiesU' or `multiplicitiesV' are not provided, default
 * parameters are computed automatically. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. If `wireTags' is
 * provided, trim the b-spline patch using the provided wires: the first wire
 * defines the external contour, the others define holes. If `wire3D' is set,
 * consider wire curves as 3D curves and project them on the b-spline surface;
 * otherwise consider the wire curves as defined in the parametric space of
 * the surface. Return the tag of the b-spline surface. */
GMSH_API int gmshModelOccAddBSplineSurface(const int * pointTags, const size_t pointTags_n,
                                           const int numPointsU,
                                           const int tag,
                                           const int degreeU,
                                           const int degreeV,
                                           const double * weights, const size_t weights_n,
                                           const double * knotsU, const size_t knotsU_n,
                                           const double * knotsV, const size_t knotsV_n,
                                           const int * multiplicitiesU, const size_t multiplicitiesU_n,
                                           const int * multiplicitiesV, const size_t multiplicitiesV_n,
                                           const int * wireTags, const size_t wireTags_n,
                                           const int wire3D,
                                           int * ierr);

/* Add a Bezier surface in the OpenCASCADE CAD representation, with
 * `pointTags' control points given as a single vector [Pu1v1, ...
 * Pu`numPointsU'v1, Pu1v2, ...]. If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. If `wireTags' is
 * provided, trim the Bezier patch using the provided wires: the first wire
 * defines the external contour, the others define holes. If `wire3D' is set,
 * consider wire curves as 3D curves and project them on the Bezier surface;
 * otherwise consider the wire curves as defined in the parametric space of
 * the surface. Return the tag of the Bezier surface. */
GMSH_API int gmshModelOccAddBezierSurface(const int * pointTags, const size_t pointTags_n,
                                          const int numPointsU,
                                          const int tag,
                                          const int * wireTags, const size_t wireTags_n,
                                          const int wire3D,
                                          int * ierr);

/* Trim the surface `surfaceTag' with the wires `wireTags', replacing any
 * existing trimming curves. The first wire defines the external contour, the
 * others define holes. If `wire3D' is set, consider wire curves as 3D curves
 * and project them on the surface; otherwise consider the wire curves as
 * defined in the parametric space of the surface. If `tag' is positive, set
 * the tag explicitly; otherwise a new tag is selected automatically. Return
 * the tag of the trimmed surface. */
GMSH_API int gmshModelOccAddTrimmedSurface(const int surfaceTag,
                                           const int * wireTags, const size_t wireTags_n,
                                           const int wire3D,
                                           const int tag,
                                           int * ierr);

/* Add a surface loop (a closed shell) in the OpenCASCADE CAD representation,
 * formed by `surfaceTags'.  If `tag' is positive, set the tag explicitly;
 * otherwise a new tag is selected automatically. Return the tag of the
 * surface loop. Setting `sewing' allows one to build a shell made of surfaces
 * that share geometrically identical (but topologically different) curves. */
GMSH_API int gmshModelOccAddSurfaceLoop(const int * surfaceTags, const size_t surfaceTags_n,
                                        const int tag,
                                        const int sewing,
                                        int * ierr);

/* Add a volume (a region) in the OpenCASCADE CAD representation, defined by
 * one or more surface loops `shellTags'. The first surface loop defines the
 * exterior boundary; additional surface loop define holes. If `tag' is
 * positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the volume. */
GMSH_API int gmshModelOccAddVolume(const int * shellTags, const size_t shellTags_n,
                                   const int tag,
                                   int * ierr);

/* Add a sphere of center (`xc', `yc', `zc') and radius `r' in the OpenCASCADE
 * CAD representation. The optional `angle1' and `angle2' arguments define the
 * polar angle opening (from -Pi/2 to Pi/2). The optional `angle3' argument
 * defines the azimuthal opening (from 0 to 2*Pi). If `tag' is positive, set
 * the tag explicitly; otherwise a new tag is selected automatically. Return
 * the tag of the sphere. */
GMSH_API int gmshModelOccAddSphere(const double xc,
                                   const double yc,
                                   const double zc,
                                   const double radius,
                                   const int tag,
                                   const double angle1,
                                   const double angle2,
                                   const double angle3,
                                   int * ierr);

/* Add a parallelepipedic box in the OpenCASCADE CAD representation, defined
 * by a point (`x', `y', `z') and the extents along the x-, y- and z-axes. If
 * `tag' is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the box. */
GMSH_API int gmshModelOccAddBox(const double x,
                                const double y,
                                const double z,
                                const double dx,
                                const double dy,
                                const double dz,
                                const int tag,
                                int * ierr);

/* Add a cylinder in the OpenCASCADE CAD representation, defined by the center
 * (`x', `y', `z') of its first circular face, the 3 components (`dx', `dy',
 * `dz') of the vector defining its axis and its radius `r'. The optional
 * `angle' argument defines the angular opening (from 0 to 2*Pi). If `tag' is
 * positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. Return the tag of the cylinder. */
GMSH_API int gmshModelOccAddCylinder(const double x,
                                     const double y,
                                     const double z,
                                     const double dx,
                                     const double dy,
                                     const double dz,
                                     const double r,
                                     const int tag,
                                     const double angle,
                                     int * ierr);

/* Add a cone in the OpenCASCADE CAD representation, defined by the center
 * (`x', `y', `z') of its first circular face, the 3 components of the vector
 * (`dx', `dy', `dz') defining its axis and the two radii `r1' and `r2' of the
 * faces (these radii can be zero). If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. `angle' defines
 * the optional angular opening (from 0 to 2*Pi). Return the tag of the cone. */
GMSH_API int gmshModelOccAddCone(const double x,
                                 const double y,
                                 const double z,
                                 const double dx,
                                 const double dy,
                                 const double dz,
                                 const double r1,
                                 const double r2,
                                 const int tag,
                                 const double angle,
                                 int * ierr);

/* Add a right angular wedge in the OpenCASCADE CAD representation, defined by
 * the right-angle point (`x', `y', `z') and the 3 extends along the x-, y-
 * and z-axes (`dx', `dy', `dz'). If `tag' is positive, set the tag
 * explicitly; otherwise a new tag is selected automatically. The optional
 * argument `ltx' defines the top extent along the x-axis. If a vector `zAxis'
 * of size 3 is provided, use it to define the z-axis. Return the tag of the
 * wedge. */
GMSH_API int gmshModelOccAddWedge(const double x,
                                  const double y,
                                  const double z,
                                  const double dx,
                                  const double dy,
                                  const double dz,
                                  const int tag,
                                  const double ltx,
                                  const double * zAxis, const size_t zAxis_n,
                                  int * ierr);

/* Add a torus in the OpenCASCADE CAD representation, defined by its center
 * (`x', `y', `z') and its 2 radii `r' and `r2'. If `tag' is positive, set the
 * tag explicitly; otherwise a new tag is selected automatically. The optional
 * argument `angle' defines the angular opening (from 0 to 2*Pi). If a vector
 * `zAxis' of size 3 is provided, use it to define the z-axis. Return the tag
 * of the torus. */
GMSH_API int gmshModelOccAddTorus(const double x,
                                  const double y,
                                  const double z,
                                  const double r1,
                                  const double r2,
                                  const int tag,
                                  const double angle,
                                  const double * zAxis, const size_t zAxis_n,
                                  int * ierr);

/* Add a volume (if the optional argument `makeSolid' is set) or surfaces in
 * the OpenCASCADE CAD representation, defined through the open or closed
 * wires `wireTags'. If `tag' is positive, set the tag explicitly; otherwise a
 * new tag is selected automatically. The new entities are returned in
 * `outDimTags' as a vector of (dim, tag) pairs. If the optional argument
 * `makeRuled' is set, the surfaces created on the boundary are forced to be
 * ruled surfaces. If `maxDegree' is positive, set the maximal degree of
 * resulting surface. The optional argument `continuity' allows to specify the
 * continuity of the resulting shape ("C0", "G1", "C1", "G2", "C2", "C3",
 * "CN"). The optional argument `parametrization' sets the parametrization
 * type ("ChordLength", "Centripetal", "IsoParametric"). The optional argument
 * `smoothing' determines if smoothing is applied. */
GMSH_API void gmshModelOccAddThruSections(const int * wireTags, const size_t wireTags_n,
                                          int ** outDimTags, size_t * outDimTags_n,
                                          const int tag,
                                          const int makeSolid,
                                          const int makeRuled,
                                          const int maxDegree,
                                          const char * continuity,
                                          const char * parametrization,
                                          const int smoothing,
                                          int * ierr);

/* Add a hollowed volume in the OpenCASCADE CAD representation, built from an
 * initial volume `volumeTag' and a set of faces from this volume
 * `excludeSurfaceTags', which are to be removed. The remaining faces of the
 * volume become the walls of the hollowed solid, with thickness `offset'. If
 * `tag' is positive, set the tag explicitly; otherwise a new tag is selected
 * automatically. */
GMSH_API void gmshModelOccAddThickSolid(const int volumeTag,
                                        const int * excludeSurfaceTags, const size_t excludeSurfaceTags_n,
                                        const double offset,
                                        int ** outDimTags, size_t * outDimTags_n,
                                        const int tag,
                                        int * ierr);

/* Extrude the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation, using a translation along (`dx', `dy',
 * `dz'). Return extruded entities in `outDimTags'. If the `numElements'
 * vector is not empty, also extrude the mesh: the entries in `numElements'
 * give the number of elements in each layer. If the `height' vector is not
 * empty, it provides the (cumulative) height of the different layers,
 * normalized to 1. If `recombine' is set, recombine the mesh in the layers. */
GMSH_API void gmshModelOccExtrude(const int * dimTags, const size_t dimTags_n,
                                  const double dx,
                                  const double dy,
                                  const double dz,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const int * numElements, const size_t numElements_n,
                                  const double * heights, const size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Extrude the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation, using a rotation of `angle' radians
 * around the axis of revolution defined by the point (`x', `y', `z') and the
 * direction (`ax', `ay', `az'). Return extruded entities in `outDimTags'. If
 * the `numElements' vector is not empty, also extrude the mesh: the entries
 * in `numElements' give the number of elements in each layer. If the `height'
 * vector is not empty, it provides the (cumulative) height of the different
 * layers, normalized to 1. When the mesh is extruded the angle should be
 * strictly smaller than 2*Pi. If `recombine' is set, recombine the mesh in
 * the layers. */
GMSH_API void gmshModelOccRevolve(const int * dimTags, const size_t dimTags_n,
                                  const double x,
                                  const double y,
                                  const double z,
                                  const double ax,
                                  const double ay,
                                  const double az,
                                  const double angle,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const int * numElements, const size_t numElements_n,
                                  const double * heights, const size_t heights_n,
                                  const int recombine,
                                  int * ierr);

/* Add a pipe in the OpenCASCADE CAD representation, by extruding the entities
 * `dimTags' (given as a vector of (dim, tag) pairs) along the wire `wireTag'.
 * The type of sweep can be specified with `trihedron' (possible values:
 * "DiscreteTrihedron", "CorrectedFrenet", "Fixed", "Frenet",
 * "ConstantNormal", "Darboux", "GuideAC", "GuidePlan", "GuideACWithContact",
 * "GuidePlanWithContact"). If `trihedron' is not provided,
 * "DiscreteTrihedron" is assumed. Return the pipe in `outDimTags'. */
GMSH_API void gmshModelOccAddPipe(const int * dimTags, const size_t dimTags_n,
                                  const int wireTag,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const char * trihedron,
                                  int * ierr);

/* Fillet the volumes `volumeTags' on the curves `curveTags' with radii
 * `radii'. The `radii' vector can either contain a single radius, as many
 * radii as `curveTags', or twice as many as `curveTags' (in which case
 * different radii are provided for the begin and end points of the curves).
 * Return the filleted entities in `outDimTags' as a vector of (dim, tag)
 * pairs. Remove the original volume if `removeVolume' is set. */
GMSH_API void gmshModelOccFillet(const int * volumeTags, const size_t volumeTags_n,
                                 const int * curveTags, const size_t curveTags_n,
                                 const double * radii, const size_t radii_n,
                                 int ** outDimTags, size_t * outDimTags_n,
                                 const int removeVolume,
                                 int * ierr);

/* Chamfer the volumes `volumeTags' on the curves `curveTags' with distances
 * `distances' measured on surfaces `surfaceTags'. The `distances' vector can
 * either contain a single distance, as many distances as `curveTags' and
 * `surfaceTags', or twice as many as `curveTags' and `surfaceTags' (in which
 * case the first in each pair is measured on the corresponding surface in
 * `surfaceTags', the other on the other adjacent surface). Return the
 * chamfered entities in `outDimTags'. Remove the original volume if
 * `removeVolume' is set. */
GMSH_API void gmshModelOccChamfer(const int * volumeTags, const size_t volumeTags_n,
                                  const int * curveTags, const size_t curveTags_n,
                                  const int * surfaceTags, const size_t surfaceTags_n,
                                  const double * distances, const size_t distances_n,
                                  int ** outDimTags, size_t * outDimTags_n,
                                  const int removeVolume,
                                  int * ierr);

/* Compute the boolean union (the fusion) of the entities `objectDimTags' and
 * `toolDimTags' (vectors of (dim, tag) pairs) in the OpenCASCADE CAD
 * representation. Return the resulting entities in `outDimTags'. If `tag' is
 * positive, try to set the tag explicitly (only valid if the boolean
 * operation results in a single entity). Remove the object if `removeObject'
 * is set. Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccFuse(const int * objectDimTags, const size_t objectDimTags_n,
                               const int * toolDimTags, const size_t toolDimTags_n,
                               int ** outDimTags, size_t * outDimTags_n,
                               int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                               const int tag,
                               const int removeObject,
                               const int removeTool,
                               int * ierr);

/* Compute the boolean intersection (the common parts) of the entities
 * `objectDimTags' and `toolDimTags' (vectors of (dim, tag) pairs) in the
 * OpenCASCADE CAD representation. Return the resulting entities in
 * `outDimTags'. If `tag' is positive, try to set the tag explicitly (only
 * valid if the boolean operation results in a single entity). Remove the
 * object if `removeObject' is set. Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccIntersect(const int * objectDimTags, const size_t objectDimTags_n,
                                    const int * toolDimTags, const size_t toolDimTags_n,
                                    int ** outDimTags, size_t * outDimTags_n,
                                    int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                                    const int tag,
                                    const int removeObject,
                                    const int removeTool,
                                    int * ierr);

/* Compute the boolean difference between the entities `objectDimTags' and
 * `toolDimTags' (given as vectors of (dim, tag) pairs) in the OpenCASCADE CAD
 * representation. Return the resulting entities in `outDimTags'. If `tag' is
 * positive, try to set the tag explicitly (only valid if the boolean
 * operation results in a single entity). Remove the object if `removeObject'
 * is set. Remove the tool if `removeTool' is set. */
GMSH_API void gmshModelOccCut(const int * objectDimTags, const size_t objectDimTags_n,
                              const int * toolDimTags, const size_t toolDimTags_n,
                              int ** outDimTags, size_t * outDimTags_n,
                              int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                              const int tag,
                              const int removeObject,
                              const int removeTool,
                              int * ierr);

/* Compute the boolean fragments (general fuse) resulting from the
 * intersection of the entities `objectDimTags' and `toolDimTags' (given as
 * vectors of (dim, tag) pairs) in the OpenCASCADE CAD representation, making
 * all interfaces conformal. When applied to entities of different dimensions,
 * the lower dimensional entities will be automatically embedded in the higher
 * dimensional entities if they are not on their boundary. Return the
 * resulting entities in `outDimTags'. If `tag' is positive, try to set the
 * tag explicitly (only valid if the boolean operation results in a single
 * entity). Remove the object if `removeObject' is set. Remove the tool if
 * `removeTool' is set. */
GMSH_API void gmshModelOccFragment(const int * objectDimTags, const size_t objectDimTags_n,
                                   const int * toolDimTags, const size_t toolDimTags_n,
                                   int ** outDimTags, size_t * outDimTags_n,
                                   int *** outDimTagsMap, size_t ** outDimTagsMap_n, size_t *outDimTagsMap_nn,
                                   const int tag,
                                   const int removeObject,
                                   const int removeTool,
                                   int * ierr);

/* Translate the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation along (`dx', `dy', `dz'). */
GMSH_API void gmshModelOccTranslate(const int * dimTags, const size_t dimTags_n,
                                    const double dx,
                                    const double dy,
                                    const double dz,
                                    int * ierr);

/* Rotate the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation by `angle' radians around the axis of
 * revolution defined by the point (`x', `y', `z') and the direction (`ax',
 * `ay', `az'). */
GMSH_API void gmshModelOccRotate(const int * dimTags, const size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double ax,
                                 const double ay,
                                 const double az,
                                 const double angle,
                                 int * ierr);

/* Scale the entities `dimTags' (given as a vector of (dim, tag) pairs) in the
 * OpenCASCADE CAD representation by factors `a', `b' and `c' along the three
 * coordinate axes; use (`x', `y', `z') as the center of the homothetic
 * transformation. */
GMSH_API void gmshModelOccDilate(const int * dimTags, const size_t dimTags_n,
                                 const double x,
                                 const double y,
                                 const double z,
                                 const double a,
                                 const double b,
                                 const double c,
                                 int * ierr);

/* Mirror the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation, with respect to the plane of equation
 * `a' * x + `b' * y + `c' * z + `d' = 0. */
GMSH_API void gmshModelOccMirror(const int * dimTags, const size_t dimTags_n,
                                 const double a,
                                 const double b,
                                 const double c,
                                 const double d,
                                 int * ierr);

/* Mirror the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation, with respect to the plane of equation
 * `a' * x + `b' * y + `c' * z + `d' = 0. (This is a deprecated synonym for
 * `mirror'.) */
GMSH_API void gmshModelOccSymmetrize(const int * dimTags, const size_t dimTags_n,
                                     const double a,
                                     const double b,
                                     const double c,
                                     const double d,
                                     int * ierr);

/* Apply a general affine transformation matrix `affineTransform' (16 entries
 * of a 4x4 matrix, by row; only the 12 first can be provided for convenience)
 * to the entities `dimTags' (given as a vector of (dim, tag) pairs) in the
 * OpenCASCADE CAD representation. */
GMSH_API void gmshModelOccAffineTransform(const int * dimTags, const size_t dimTags_n,
                                          const double * affineTransform, const size_t affineTransform_n,
                                          int * ierr);

/* Copy the entities `dimTags' in the OpenCASCADE CAD representation; the new
 * entities are returned in `outDimTags'. */
GMSH_API void gmshModelOccCopy(const int * dimTags, const size_t dimTags_n,
                               int ** outDimTags, size_t * outDimTags_n,
                               int * ierr);

/* Remove the entities `dimTags' (given as a vector of (dim, tag) pairs) in
 * the OpenCASCADE CAD representation, provided that they are not on the
 * boundary of higher-dimensional entities. If `recursive' is true, remove all
 * the entities on their boundaries, down to dimension 0. */
GMSH_API void gmshModelOccRemove(const int * dimTags, const size_t dimTags_n,
                                 const int recursive,
                                 int * ierr);

/* Remove all duplicate entities in the OpenCASCADE CAD representation
 * (different entities at the same geometrical location) after intersecting
 * (using boolean fragments) all highest dimensional entities. */
GMSH_API void gmshModelOccRemoveAllDuplicates(int * ierr);

/* Apply various healing procedures to the entities `dimTags' (given as a
 * vector of (dim, tag) pairs), or to all the entities in the model if
 * `dimTags' is empty, in the OpenCASCADE CAD representation. Return the
 * healed entities in `outDimTags'. */
GMSH_API void gmshModelOccHealShapes(int ** outDimTags, size_t * outDimTags_n,
                                     const int * dimTags, const size_t dimTags_n,
                                     const double tolerance,
                                     const int fixDegenerated,
                                     const int fixSmallEdges,
                                     const int fixSmallFaces,
                                     const int sewFaces,
                                     const int makeSolids,
                                     int * ierr);

/* Convert the entities `dimTags' to NURBS. */
GMSH_API void gmshModelOccConvertToNURBS(const int * dimTags, const size_t dimTags_n,
                                         int * ierr);

/* Import BREP, STEP or IGES shapes from the file `fileName' in the
 * OpenCASCADE CAD representation. The imported entities are returned in
 * `outDimTags', as a vector of (dim, tag) pairs. If the optional argument
 * `highestDimOnly' is set, only import the highest dimensional entities in
 * the file. The optional argument `format' can be used to force the format of
 * the file (currently "brep", "step" or "iges"). */
GMSH_API void gmshModelOccImportShapes(const char * fileName,
                                       int ** outDimTags, size_t * outDimTags_n,
                                       const int highestDimOnly,
                                       const char * format,
                                       int * ierr);

/* Imports an OpenCASCADE `shape' by providing a pointer to a native
 * OpenCASCADE `TopoDS_Shape' object (passed as a pointer to void). The
 * imported entities are returned in `outDimTags' as a vector of (dim, tag)
 * pairs. If the optional argument `highestDimOnly' is set, only import the
 * highest dimensional entities in `shape'. In Python, this function can be
 * used for integration with PythonOCC, in which the SwigPyObject pointer of
 * `TopoDS_Shape' must be passed as an int to `shape', i.e., `shape =
 * int(pythonocc_shape.this)'. Warning: this function is unsafe, as providing
 * an invalid pointer will lead to undefined behavior. */
GMSH_API void gmshModelOccImportShapesNativePointer(const void * shape,
                                                    int ** outDimTags, size_t * outDimTags_n,
                                                    const int highestDimOnly,
                                                    int * ierr);

/* Get all the OpenCASCADE entities. If `dim' is >= 0, return only the
 * entities of the specified dimension (e.g. points if `dim' == 0). The
 * entities are returned as a vector of (dim, tag) pairs. */
GMSH_API void gmshModelOccGetEntities(int ** dimTags, size_t * dimTags_n,
                                      const int dim,
                                      int * ierr);

/* Get the OpenCASCADE entities in the bounding box defined by the two points
 * (`xmin', `ymin', `zmin') and (`xmax', `ymax', `zmax'). If `dim' is >= 0,
 * return only the entities of the specified dimension (e.g. points if `dim'
 * == 0). */
GMSH_API void gmshModelOccGetEntitiesInBoundingBox(const double xmin,
                                                   const double ymin,
                                                   const double zmin,
                                                   const double xmax,
                                                   const double ymax,
                                                   const double zmax,
                                                   int ** dimTags, size_t * dimTags_n,
                                                   const int dim,
                                                   int * ierr);

/* Get the bounding box (`xmin', `ymin', `zmin'), (`xmax', `ymax', `zmax') of
 * the OpenCASCADE entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelOccGetBoundingBox(const int dim,
                                         const int tag,
                                         double * xmin,
                                         double * ymin,
                                         double * zmin,
                                         double * xmax,
                                         double * ymax,
                                         double * zmax,
                                         int * ierr);

/* Get the tags `curveLoopTags' of the curve loops making up the surface of
 * tag `surfaceTag', as well as the tags `curveTags' of the curves making up
 * each curve loop. */
GMSH_API void gmshModelOccGetCurveLoops(const int surfaceTag,
                                        int ** curveLoopTags, size_t * curveLoopTags_n,
                                        int *** curveTags, size_t ** curveTags_n, size_t *curveTags_nn,
                                        int * ierr);

/* Get the tags `surfaceLoopTags' of the surface loops making up the volume of
 * tag `volumeTag', as well as the tags `surfaceTags' of the surfaces making
 * up each surface loop. */
GMSH_API void gmshModelOccGetSurfaceLoops(const int volumeTag,
                                          int ** surfaceLoopTags, size_t * surfaceLoopTags_n,
                                          int *** surfaceTags, size_t ** surfaceTags_n, size_t *surfaceTags_nn,
                                          int * ierr);

/* Get the mass of the OpenCASCADE entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshModelOccGetMass(const int dim,
                                  const int tag,
                                  double * mass,
                                  int * ierr);

/* Get the center of mass of the OpenCASCADE entity of dimension `dim' and tag
 * `tag'. */
GMSH_API void gmshModelOccGetCenterOfMass(const int dim,
                                          const int tag,
                                          double * x,
                                          double * y,
                                          double * z,
                                          int * ierr);

/* Get the matrix of inertia (by row) of the OpenCASCADE entity of dimension
 * `dim' and tag `tag'. */
GMSH_API void gmshModelOccGetMatrixOfInertia(const int dim,
                                             const int tag,
                                             double ** mat, size_t * mat_n,
                                             int * ierr);

/* Get the maximum tag of entities of dimension `dim' in the OpenCASCADE CAD
 * representation. */
GMSH_API int gmshModelOccGetMaxTag(const int dim,
                                   int * ierr);

/* Set the maximum tag `maxTag' for entities of dimension `dim' in the
 * OpenCASCADE CAD representation. */
GMSH_API void gmshModelOccSetMaxTag(const int dim,
                                    const int maxTag,
                                    int * ierr);

/* Synchronize the OpenCASCADE CAD representation with the current Gmsh model.
 * This can be called at any time, but since it involves a non trivial amount
 * of processing, the number of synchronization points should normally be
 * minimized. Without synchronization the entities in the OpenCASCADE CAD
 * representation are not available to any function outside of the OpenCASCADE
 * CAD kernel functions. */
GMSH_API void gmshModelOccSynchronize(int * ierr);

/* Set a mesh size constraint on the entities `dimTags' (given as a vector of
 * (dim, tag) pairs) in the OpenCASCADE CAD representation. Currently only
 * entities of dimension 0 (points) are handled. */
GMSH_API void gmshModelOccMeshSetSize(const int * dimTags, const size_t dimTags_n,
                                      const double size,
                                      int * ierr);

/* Add a new post-processing view, with name `name'. If `tag' is positive use
 * it (and remove the view with that tag if it already exists), otherwise
 * associate a new tag. Return the view tag. */
GMSH_API int gmshViewAdd(const char * name,
                         const int tag,
                         int * ierr);

/* Remove the view with tag `tag'. */
GMSH_API void gmshViewRemove(const int tag,
                             int * ierr);

/* Get the index of the view with tag `tag' in the list of currently loaded
 * views. This dynamic index (it can change when views are removed) is used to
 * access view options. */
GMSH_API int gmshViewGetIndex(const int tag,
                              int * ierr);

/* Get the tags of all views. */
GMSH_API void gmshViewGetTags(int ** tags, size_t * tags_n,
                              int * ierr);

/* Add model-based post-processing data to the view with tag `tag'.
 * `modelName' identifies the model the data is attached to. `dataType'
 * specifies the type of data, currently either "NodeData", "ElementData" or
 * "ElementNodeData". `step' specifies the identifier (>= 0) of the data in a
 * sequence. `tags' gives the tags of the nodes or elements in the mesh to
 * which the data is associated. `data' is a vector of the same length as
 * `tags': each entry is the vector of double precision numbers representing
 * the data associated with the corresponding tag. The optional `time'
 * argument associate a time value with the data. `numComponents' gives the
 * number of data components (1 for scalar data, 3 for vector data, etc.) per
 * entity; if negative, it is automatically inferred (when possible) from the
 * input data. `partition' allows one to specify data in several sub-sets. */
GMSH_API void gmshViewAddModelData(const int tag,
                                   const int step,
                                   const char * modelName,
                                   const char * dataType,
                                   const size_t * tags, const size_t tags_n,
                                   const double * const * data, const size_t * data_n, const size_t data_nn,
                                   const double time,
                                   const int numComponents,
                                   const int partition,
                                   int * ierr);

/* Add homogeneous model-based post-processing data to the view with tag
 * `tag'. The arguments have the same meaning as in `addModelData', except
 * that `data' is supposed to be homogeneous and is thus flattened in a single
 * vector. For data types that can lead to different data sizes per tag (like
 * "ElementNodeData"), the data should be padded. */
GMSH_API void gmshViewAddHomogeneousModelData(const int tag,
                                              const int step,
                                              const char * modelName,
                                              const char * dataType,
                                              const size_t * tags, const size_t tags_n,
                                              const double * data, const size_t data_n,
                                              const double time,
                                              const int numComponents,
                                              const int partition,
                                              int * ierr);

/* Get model-based post-processing data from the view with tag `tag' at step
 * `step'. Return the `data' associated to the nodes or the elements with tags
 * `tags', as well as the `dataType' and the number of components
 * `numComponents'. */
GMSH_API void gmshViewGetModelData(const int tag,
                                   const int step,
                                   char ** dataType,
                                   size_t ** tags, size_t * tags_n,
                                   double *** data, size_t ** data_n, size_t *data_nn,
                                   double * time,
                                   int * numComponents,
                                   int * ierr);

/* Get homogeneous model-based post-processing data from the view with tag
 * `tag' at step `step'. The arguments have the same meaning as in
 * `getModelData', except that `data' is returned flattened in a single
 * vector, with the appropriate padding if necessary. */
GMSH_API void gmshViewGetHomogeneousModelData(const int tag,
                                              const int step,
                                              char ** dataType,
                                              size_t ** tags, size_t * tags_n,
                                              double ** data, size_t * data_n,
                                              double * time,
                                              int * numComponents,
                                              int * ierr);

/* Add list-based post-processing data to the view with tag `tag'. List-based
 * datasets are independent from any model and any mesh. `dataType' identifies
 * the data by concatenating the field type ("S" for scalar, "V" for vector,
 * "T" for tensor) and the element type ("P" for point, "L" for line, "T" for
 * triangle, "S" for tetrahedron, "I" for prism, "H" for hexaHedron, "Y" for
 * pyramid). For example `dataType' should be "ST" for a scalar field on
 * triangles. `numEle' gives the number of elements in the data. `data'
 * contains the data for the `numEle' elements, concatenated, with node
 * coordinates followed by values per node, repeated for each step: [e1x1,
 * ..., e1xn, e1y1, ..., e1yn, e1z1, ..., e1zn, e1v1..., e1vN, e2x1, ...]. */
GMSH_API void gmshViewAddListData(const int tag,
                                  const char * dataType,
                                  const int numEle,
                                  const double * data, const size_t data_n,
                                  int * ierr);

/* Get list-based post-processing data from the view with tag `tag'. Return
 * the types `dataTypes', the number of elements `numElements' for each data
 * type and the `data' for each data type. If `returnAdaptive' is set, return
 * the data obtained after adaptive refinement, if available. */
GMSH_API void gmshViewGetListData(const int tag,
                                  char *** dataType, size_t * dataType_n,
                                  int ** numElements, size_t * numElements_n,
                                  double *** data, size_t ** data_n, size_t *data_nn,
                                  const int returnAdaptive,
                                  int * ierr);

/* Add a string to a list-based post-processing view with tag `tag'. If
 * `coord' contains 3 coordinates the string is positioned in the 3D model
 * space ("3D string"); if it contains 2 coordinates it is positioned in the
 * 2D graphics viewport ("2D string"). `data' contains one or more (for
 * multistep views) strings. `style' contains key-value pairs of styling
 * parameters, concatenated. Available keys are "Font" (possible values:
 * "Times-Roman", "Times-Bold", "Times-Italic", "Times-BoldItalic",
 * "Helvetica", "Helvetica-Bold", "Helvetica-Oblique", "Helvetica-
 * BoldOblique", "Courier", "Courier-Bold", "Courier-Oblique", "Courier-
 * BoldOblique", "Symbol", "ZapfDingbats", "Screen"), "FontSize" and "Align"
 * (possible values: "Left" or "BottomLeft", "Center" or "BottomCenter",
 * "Right" or "BottomRight", "TopLeft", "TopCenter", "TopRight", "CenterLeft",
 * "CenterCenter", "CenterRight"). */
GMSH_API void gmshViewAddListDataString(const int tag,
                                        const double * coord, const size_t coord_n,
                                        const char * const * data, const size_t data_n,
                                        const char * const * style, const size_t style_n,
                                        int * ierr);

/* Get list-based post-processing data strings (2D strings if `dim' == 2, 3D
 * strings if `dim' = 3) from the view with tag `tag'. Return the coordinates
 * in `coord', the strings in `data' and the styles in `style'. */
GMSH_API void gmshViewGetListDataStrings(const int tag,
                                         const int dim,
                                         double ** coord, size_t * coord_n,
                                         char *** data, size_t * data_n,
                                         char *** style, size_t * style_n,
                                         int * ierr);

/* Set interpolation matrices for the element family `type' ("Line",
 * "Triangle", "Quadrangle", "Tetrahedron", "Hexahedron", "Prism", "Pyramid")
 * in the view `tag'. The approximation of the values over an element is
 * written as a linear combination of `d' basis functions f_i(u, v, w) =
 * sum_(j = 0, ..., `d' - 1) `coef'[i][j] u^`exp'[j][0] v^`exp'[j][1]
 * w^`exp'[j][2], i = 0, ..., `d'-1, with u, v, w the coordinates in the
 * reference element. The `coef' matrix (of size `d' x `d') and the `exp'
 * matrix (of size `d' x 3) are stored as vectors, by row. If `dGeo' is
 * positive, use `coefGeo' and `expGeo' to define the interpolation of the x,
 * y, z coordinates of the element in terms of the u, v, w coordinates, in
 * exactly the same way. If `d' < 0, remove the interpolation matrices. */
GMSH_API void gmshViewSetInterpolationMatrices(const int tag,
                                               const char * type,
                                               const int d,
                                               const double * coef, const size_t coef_n,
                                               const double * exp, const size_t exp_n,
                                               const int dGeo,
                                               const double * coefGeo, const size_t coefGeo_n,
                                               const double * expGeo, const size_t expGeo_n,
                                               int * ierr);

/* Add a post-processing view as an `alias' of the reference view with tag
 * `refTag'. If `copyOptions' is set, copy the options of the reference view.
 * If `tag' is positive use it (and remove the view with that tag if it
 * already exists), otherwise associate a new tag. Return the view tag. */
GMSH_API int gmshViewAddAlias(const int refTag,
                              const int copyOptions,
                              const int tag,
                              int * ierr);

/* Combine elements (if `what' == "elements") or steps (if `what' == "steps")
 * of all views (`how' == "all"), all visible views (`how' == "visible") or
 * all views having the same name (`how' == "name"). Remove original views if
 * `remove' is set. */
GMSH_API void gmshViewCombine(const char * what,
                              const char * how,
                              const int remove,
                              const int copyOptions,
                              int * ierr);

/* Probe the view `tag' for its `values' at point (`x', `y', `z'). If no match
 * is found, `value' is returned empty. Return only the value at step `step'
 * is `step' is positive. Return only values with `numComp' if `numComp' is
 * positive. Return the gradient of the `values' if `gradient' is set. If
 * `distanceMax' is zero, only return a result if an exact match inside an
 * element in the view is found; if `distanceMax' is positive and an exact
 * match is not found, return the value at the closest node if it is closer
 * than `distanceMax'; if `distanceMax' is negative and an exact match is not
 * found, always return the value at the closest node. The distance to the
 * match is returned in `distance'. Return the result from the element
 * described by its coordinates if `xElementCoord', `yElementCoord' and
 * `zElementCoord' are provided. If `dim' is >= 0, return only matches from
 * elements of the specified dimension. */
GMSH_API void gmshViewProbe(const int tag,
                            const double x,
                            const double y,
                            const double z,
                            double ** values, size_t * values_n,
                            double * distance,
                            const int step,
                            const int numComp,
                            const int gradient,
                            const double distanceMax,
                            const double * xElemCoord, const size_t xElemCoord_n,
                            const double * yElemCoord, const size_t yElemCoord_n,
                            const double * zElemCoord, const size_t zElemCoord_n,
                            const int dim,
                            int * ierr);

/* Write the view to a file `fileName'. The export format is determined by the
 * file extension. Append to the file if `append' is set. */
GMSH_API void gmshViewWrite(const int tag,
                            const char * fileName,
                            const int append,
                            int * ierr);

/* Set the global visibility of the view `tag' per window to `value', where
 * `windowIndex' identifies the window in the window list. */
GMSH_API void gmshViewSetVisibilityPerWindow(const int tag,
                                             const int value,
                                             const int windowIndex,
                                             int * ierr);

/* Set the numerical option `name' to value `value' for the view with tag
 * `tag'. */
GMSH_API void gmshViewOptionSetNumber(const int tag,
                                      const char * name,
                                      const double value,
                                      int * ierr);

/* Get the `value' of the numerical option `name' for the view with tag `tag'. */
GMSH_API void gmshViewOptionGetNumber(const int tag,
                                      const char * name,
                                      double * value,
                                      int * ierr);

/* Set the string option `name' to value `value' for the view with tag `tag'. */
GMSH_API void gmshViewOptionSetString(const int tag,
                                      const char * name,
                                      const char * value,
                                      int * ierr);

/* Get the `value' of the string option `name' for the view with tag `tag'. */
GMSH_API void gmshViewOptionGetString(const int tag,
                                      const char * name,
                                      char ** value,
                                      int * ierr);

/* Set the color option `name' to the RGBA value (`r', `g', `b', `a') for the
 * view with tag `tag', where where `r', `g', `b' and `a' should be integers
 * between 0 and 255. */
GMSH_API void gmshViewOptionSetColor(const int tag,
                                     const char * name,
                                     const int r,
                                     const int g,
                                     const int b,
                                     const int a,
                                     int * ierr);

/* Get the `r', `g', `b', `a' value of the color option `name' for the view
 * with tag `tag'. */
GMSH_API void gmshViewOptionGetColor(const int tag,
                                     const char * name,
                                     int * r,
                                     int * g,
                                     int * b,
                                     int * a,
                                     int * ierr);

/* Copy the options from the view with tag `refTag' to the view with tag
 * `tag'. */
GMSH_API void gmshViewOptionCopy(const int refTag,
                                 const int tag,
                                 int * ierr);

/* Set the numerical option `option' to the value `value' for plugin `name'.
 * Plugins available in the official Gmsh release are listed in the "Gmsh
 * plugins" chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-plugins). */
GMSH_API void gmshPluginSetNumber(const char * name,
                                  const char * option,
                                  const double value,
                                  int * ierr);

/* Set the string option `option' to the value `value' for plugin `name'.
 * Plugins available in the official Gmsh release are listed in the "Gmsh
 * plugins" chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-plugins). */
GMSH_API void gmshPluginSetString(const char * name,
                                  const char * option,
                                  const char * value,
                                  int * ierr);

/* Run the plugin `name'. Return the tag of the created view (if any). Plugins
 * available in the official Gmsh release are listed in the "Gmsh plugins"
 * chapter of the Gmsh reference manual
 * (https://gmsh.info/doc/texinfo/gmsh.html#Gmsh-plugins). */
GMSH_API int gmshPluginRun(const char * name,
                           int * ierr);

/* Draw all the OpenGL scenes. */
GMSH_API void gmshGraphicsDraw(int * ierr);

/* Create the FLTK graphical user interface. Can only be called in the main
 * thread. */
GMSH_API void gmshFltkInitialize(int * ierr);

/* Close the FLTK graphical user interface. Can only be called in the main
 * thread. */
GMSH_API void gmshFltkFinalize(int * ierr);

/* Wait at most `time' seconds for user interface events and return. If `time'
 * < 0, wait indefinitely. First automatically create the user interface if it
 * has not yet been initialized. Can only be called in the main thread. */
GMSH_API void gmshFltkWait(const double time,
                           int * ierr);

/* Update the user interface (potentially creating new widgets and windows).
 * First automatically create the user interface if it has not yet been
 * initialized. Can only be called in the main thread: use `awake("update")'
 * to trigger an update of the user interface from another thread. */
GMSH_API void gmshFltkUpdate(int * ierr);

/* Awake the main user interface thread and process pending events, and
 * optionally perform an action (currently the only `action' allowed is
 * "update"). */
GMSH_API void gmshFltkAwake(const char * action,
                            int * ierr);

/* Block the current thread until it can safely modify the user interface. */
GMSH_API void gmshFltkLock(int * ierr);

/* Release the lock that was set using lock. */
GMSH_API void gmshFltkUnlock(int * ierr);

/* Run the event loop of the graphical user interface, i.e. repeatedly call
 * `wait()'. First automatically create the user interface if it has not yet
 * been initialized. Can only be called in the main thread. */
GMSH_API void gmshFltkRun(int * ierr);

/* Check if the user interface is available (e.g. to detect if it has been
 * closed). */
GMSH_API int gmshFltkIsAvailable(int * ierr);

/* Select entities in the user interface. Return the selected entities as a
 * vector of (dim, tag) pairs. If `dim' is >= 0, return only the entities of
 * the specified dimension (e.g. points if `dim' == 0). */
GMSH_API int gmshFltkSelectEntities(int ** dimTags, size_t * dimTags_n,
                                    const int dim,
                                    int * ierr);

/* Select elements in the user interface. */
GMSH_API int gmshFltkSelectElements(size_t ** elementTags, size_t * elementTags_n,
                                    int * ierr);

/* Select views in the user interface. */
GMSH_API int gmshFltkSelectViews(int ** viewTags, size_t * viewTags_n,
                                 int * ierr);

/* Split the current window horizontally (if `how' == "h") or vertically (if
 * `how' == "v"), using ratio `ratio'. If `how' == "u", restore a single
 * window. */
GMSH_API void gmshFltkSplitCurrentWindow(const char * how,
                                         const double ratio,
                                         int * ierr);

/* Set the current window by speficying its index (starting at 0) in the list
 * of all windows. When new windows are created by splits, new windows are
 * appended at the end of the list. */
GMSH_API void gmshFltkSetCurrentWindow(const int windowIndex,
                                       int * ierr);

/* Set a status message in the current window. If `graphics' is set, display
 * the message inside the graphic window instead of the status bar. */
GMSH_API void gmshFltkSetStatusMessage(const char * message,
                                       const int graphics,
                                       int * ierr);

/* Show context window for the entity of dimension `dim' and tag `tag'. */
GMSH_API void gmshFltkShowContextWindow(const int dim,
                                        const int tag,
                                        int * ierr);

/* Open the `name' item in the menu tree. */
GMSH_API void gmshFltkOpenTreeItem(const char * name,
                                   int * ierr);

/* Close the `name' item in the menu tree. */
GMSH_API void gmshFltkCloseTreeItem(const char * name,
                                    int * ierr);

/* Get the names of the variables in the Gmsh parser matching the `search'
 * regular expression. If `search' is empty, return all the names. */
GMSH_API void gmshParserGetNames(char *** names, size_t * names_n,
                                 const char * search,
                                 int * ierr);

/* Set the value of the number variable `name' in the Gmsh parser. Create the
 * variable if it does not exist; update the value if the variable exists. */
GMSH_API void gmshParserSetNumber(const char * name,
                                  const double * value, const size_t value_n,
                                  int * ierr);

/* Set the value of the string variable `name' in the Gmsh parser. Create the
 * variable if it does not exist; update the value if the variable exists. */
GMSH_API void gmshParserSetString(const char * name,
                                  const char * const * value, const size_t value_n,
                                  int * ierr);

/* Get the value of the number variable `name' from the Gmsh parser. Return an
 * empty vector if the variable does not exist. */
GMSH_API void gmshParserGetNumber(const char * name,
                                  double ** value, size_t * value_n,
                                  int * ierr);

/* Get the value of the string variable `name' from the Gmsh parser. Return an
 * empty vector if the variable does not exist. */
GMSH_API void gmshParserGetString(const char * name,
                                  char *** value, size_t * value_n,
                                  int * ierr);

/* Clear all the Gmsh parser variables, or remove a single variable if `name'
 * is given. */
GMSH_API void gmshParserClear(const char * name,
                              int * ierr);

/* Parse the file `fileName' with the Gmsh parser. */
GMSH_API void gmshParserParse(const char * fileName,
                              int * ierr);

/* Set one or more parameters in the ONELAB database, encoded in `format'. */
GMSH_API void gmshOnelabSet(const char * data,
                            const char * format,
                            int * ierr);

/* Get all the parameters (or a single one if `name' is specified) from the
 * ONELAB database, encoded in `format'. */
GMSH_API void gmshOnelabGet(char ** data,
                            const char * name,
                            const char * format,
                            int * ierr);

/* Get the names of the parameters in the ONELAB database matching the
 * `search' regular expression. If `search' is empty, return all the names. */
GMSH_API void gmshOnelabGetNames(char *** names, size_t * names_n,
                                 const char * search,
                                 int * ierr);

/* Set the value of the number parameter `name' in the ONELAB database. Create
 * the parameter if it does not exist; update the value if the parameter
 * exists. */
GMSH_API void gmshOnelabSetNumber(const char * name,
                                  const double * value, const size_t value_n,
                                  int * ierr);

/* Set the value of the string parameter `name' in the ONELAB database. Create
 * the parameter if it does not exist; update the value if the parameter
 * exists. */
GMSH_API void gmshOnelabSetString(const char * name,
                                  const char * const * value, const size_t value_n,
                                  int * ierr);

/* Get the value of the number parameter `name' from the ONELAB database.
 * Return an empty vector if the parameter does not exist. */
GMSH_API void gmshOnelabGetNumber(const char * name,
                                  double ** value, size_t * value_n,
                                  int * ierr);

/* Get the value of the string parameter `name' from the ONELAB database.
 * Return an empty vector if the parameter does not exist. */
GMSH_API void gmshOnelabGetString(const char * name,
                                  char *** value, size_t * value_n,
                                  int * ierr);

/* Check if any parameters in the ONELAB database used by the client `name'
 * have been changed. */
GMSH_API int gmshOnelabGetChanged(const char * name,
                                  int * ierr);

/* Set the changed flag to value `value' for all the parameters in the ONELAB
 * database used by the client `name'. */
GMSH_API void gmshOnelabSetChanged(const char * name,
                                   const int value,
                                   int * ierr);

/* Clear the ONELAB database, or remove a single parameter if `name' is given. */
GMSH_API void gmshOnelabClear(const char * name,
                              int * ierr);

/* Run a ONELAB client. If `name' is provided, create a new ONELAB client with
 * name `name' and executes `command'. If not, try to run a client that might
 * be linked to the processed input files. */
GMSH_API void gmshOnelabRun(const char * name,
                            const char * command,
                            int * ierr);

/* Write a `message'. `level' can be "info", "warning" or "error". */
GMSH_API void gmshLoggerWrite(const char * message,
                              const char * level,
                              int * ierr);

/* Start logging messages. */
GMSH_API void gmshLoggerStart(int * ierr);

/* Get logged messages. */
GMSH_API void gmshLoggerGet(char *** log, size_t * log_n,
                            int * ierr);

/* Stop logging messages. */
GMSH_API void gmshLoggerStop(int * ierr);

/* Return wall clock time. */
GMSH_API double gmshLoggerGetWallTime(int * ierr);

/* Return CPU time. */
GMSH_API double gmshLoggerGetCpuTime(int * ierr);

/* Return last error message, if any. */
GMSH_API void gmshLoggerGetLastError(char ** error,
                                     int * ierr);

#endif
