{-# Language BangPatterns, QuasiQuotes, OverloadedStrings #-}
{-|
Module      : Digraphs
Description : Character mnemonics
Copyright   : (c) Eric Mertens, 2016
License     : ISC
Maintainer  : emertens@gmail.com

This module provides an implementation of /digraphs/ as implemented
in Vim and as specified in RFC 1345 (2-character only).

<https://tools.ietf.org/html/rfc1345>
<http://vimdoc.sourceforge.net/htmldoc/digraph.html>

-}
module Digraphs
  ( Digraph(..)
  , lookupDigraph
  , digraphs
  ) where

import           DigraphQuote
import           Data.Text (Text)
import qualified Data.Text as Text

-- | Two-character key for digraph lookup
data Digraph = Digraph !Char !Char
  deriving (Eq, Ord, Read, Show)

-- | States for the digraph lookup state machine
data St
  = Ready -- ^ ready to match two-character name
  | Match -- ^ first character matched, ready for second
  | Found -- ^ both characters matched, return the next
  | Skip2 -- ^ skip the next two characters
  | Skip1 -- ^ skip the next character

-- | Find the entry in the digraph table give a two-character
-- key and return the matched value.
lookupDigraph :: Digraph -> Maybe Char
lookupDigraph (Digraph x y) = Text.foldr step finish digraphs Ready
  where
    finish _st = Nothing
    step z k st =
      case st of
        Ready | x == z    -> k Match
              | otherwise -> k Skip2
        Match | y == z    -> k Found
              | otherwise -> k Skip1
        Skip1             -> k Ready
        Skip2             -> k Skip1
        Found             -> Just z

------------------------------------------------------------------------

-- These are explicitly prohibited by the IRC RFC
-- NU U+0000 -- NULL (NUL)
-- LF U+000a -- LINE FEED (LF)
-- CR U+000d -- CARRIAGE RETURN (CR)

digraphs :: Text
digraphs = [digraphTable|
  SH U+0001 -- START OF HEADING (SOH)
  SX U+0002 -- START OF TEXT (STX)
  EX U+0003 -- END OF TEXT (ETX)
  ET U+0004 -- END OF TRANSMISSION (EOT)
  EQ U+0005 -- ENQUIRY (ENQ)
  AK U+0006 -- ACKNOWLEDGE (ACK)
  BL U+0007 -- BELL (BEL)
  BS U+0008 -- BACKSPACE (BS)
  HT U+0009 -- CHARACTER TABULATION (HT)
  VT U+000B -- LINE TABULATION (VT)
  FF U+000C -- FORM FEED (FF)
  SO U+000E -- SHIFT OUT (SO)
  SI U+000F -- SHIFT IN (SI)
  DL U+0010 -- DATALINK ESCAPE (DLE)
  D1 U+0011 -- DEVICE CONTROL ONE (DC1)
  D2 U+0012 -- DEVICE CONTROL TWO (DC2)
  D3 U+0013 -- DEVICE CONTROL THREE (DC3)
  D4 U+0014 -- DEVICE CONTROL FOUR (DC4)
  NK U+0015 -- NEGATIVE ACKNOWLEDGE (NAK)
  SY U+0016 -- SYNCHRONOUS IDLE (SYN)
  EB U+0017 -- END OF TRANSMISSION BLOCK (ETB)
  CN U+0018 -- CANCEL (CAN)
  EM U+0019 -- END OF MEDIUM (EM)
  SB U+001A -- SUBSTITUTE (SUB)
  EC U+001B -- ESCAPE (ESC)
  FS U+001C -- FILE SEPARATOR (IS4)
  GS U+001D -- GROUP SEPARATOR (IS3)
  RS U+001E -- RECORD SEPARATOR (IS2)
  US U+001F -- UNIT SEPARATOR (IS1)
  SP U+0020 -- SPACE
  Nb U+0023 -- NUMBER SIGN
  DO U+0024 -- DOLLAR SIGN
  At U+0040 -- COMMERCIAL AT
  <( U+005B -- LEFT SQUARE BRACKET
  // U+005C -- REVERSE SOLIDUS
  )> U+005D -- RIGHT SQUARE BRACKET
  '> U+005E -- CIRCUMFLEX ACCENT
  '! U+0060 -- GRAVE ACCENT
  (! U+007B -- LEFT CURLY BRACKET
  !! U+007C -- VERTICAL LINE
  !) U+007D -- RIGHT CURLY BRACKET
  '? U+007E -- TILDE
  DT U+007F -- DELETE (DEL)
  PA U+0080 -- PADDING CHARACTER (PAD)
  HO U+0081 -- HIGH OCTET PRESET (HOP)
  BH U+0082 -- BREAK PERMITTED HERE (BPH)
  NH U+0083 -- NO BREAK HERE (NBH)
  IN U+0084 -- INDEX (IND)
  NL U+0085 -- NEXT LINE (NEL)
  SA U+0086 -- START OF SELECTED AREA (SSA)
  ES U+0087 -- END OF SELECTED AREA (ESA)
  HS U+0088 -- CHARACTER TABULATION SET (HTS)
  HJ U+0089 -- CHARACTER TABULATION WITH JUSTIFICATION (HTJ)
  VS U+008A -- LINE TABULATION SET (VTS)
  PD U+008B -- PARTIAL LINE FORWARD (PLD)
  PU U+008C -- PARTIAL LINE BACKWARD (PLU)
  RI U+008D -- REVERSE LINE FEED (RI)
  S2 U+008E -- SINGLE-SHIFT TWO (SS2)
  S3 U+008F -- SINGLE-SHIFT THREE (SS3)
  DC U+0090 -- DEVICE CONTROL STRING (DCS)
  P1 U+0091 -- PRIVATE USE ONE (PU1)
  P2 U+0092 -- PRIVATE USE TWO (PU2)
  TS U+0093 -- SET TRANSMIT STATE (STS)
  CC U+0094 -- CANCEL CHARACTER (CCH)
  MW U+0095 -- MESSAGE WAITING (MW)
  SG U+0096 -- START OF GUARDED AREA (SPA)
  EG U+0097 -- END OF GUARDED AREA (EPA)
  SS U+0098 -- START OF STRING (SOS)
  GC U+0099 -- SINGLE GRAPHIC CHARACTER INTRODUCER (SGCI)
  SC U+009A -- SINGLE CHARACTER INTRODUCER (SCI)
  CI U+009B -- CONTROL SEQUENCE INTRODUCER (CSI)
  ST U+009C -- STRING TERMINATOR (ST)
  OC U+009D -- OPERATING SYSTEM COMMAND (OSC)
  PM U+009E -- PRIVACY MESSAGE (PM)
  AC U+009F -- APPLICATION PROGRAM COMMAND (APC)
  NS U+00A0 -- NO-BREAK SPACE
  !I U+00A1 -- INVERTED EXCLAMATION MARK
  Ct U+00A2 -- CENT SIGN
  Pd U+00A3 -- POUND SIGN
  Cu U+00A4 -- CURRENCY SIGN
  Ye U+00A5 -- YEN SIGN
  BB U+00A6 -- BROKEN BAR
  SE U+00A7 -- SECTION SIGN
  ': U+00A8 -- DIAERESIS
  Co U+00A9 -- COPYRIGHT SIGN
  -a U+00AA -- FEMININE ORDINAL INDICATOR
  << U+00AB -- LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
  NO U+00AC -- NOT SIGN
  -- U+00AD -- SOFT HYPHEN
  Rg U+00AE -- REGISTERED SIGN
  'm U+00AF -- MACRON
  DG U+00B0 -- DEGREE SIGN
  +- U+00B1 -- PLUS-MINUS SIGN
  2S U+00B2 -- SUPERSCRIPT TWO
  3S U+00B3 -- SUPERSCRIPT THREE
  '' U+00B4 -- ACUTE ACCENT
  My U+00B5 -- MICRO SIGN
  PI U+00B6 -- PILCROW SIGN
  .M U+00B7 -- MIDDLE DOT
  ', U+00B8 -- CEDILLA
  1S U+00B9 -- SUPERSCRIPT ONE
  -o U+00BA -- MASCULINE ORDINAL INDICATOR
  >> U+00BB -- RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
  14 U+00BC -- VULGAR FRACTION ONE QUARTER
  12 U+00BD -- VULGAR FRACTION ONE HALF
  34 U+00BE -- VULGAR FRACTION THREE QUARTERS
  ?I U+00BF -- INVERTED QUESTION MARK
  A! U+00C0 -- LATIN CAPITAL LETTER A WITH GRAVE
  A' U+00C1 -- LATIN CAPITAL LETTER A WITH ACUTE
  A> U+00C2 -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX
  A? U+00C3 -- LATIN CAPITAL LETTER A WITH TILDE
  A: U+00C4 -- LATIN CAPITAL LETTER A WITH DIAERESIS
  AA U+00C5 -- LATIN CAPITAL LETTER A WITH RING ABOVE
  AE U+00C6 -- LATIN CAPITAL LETTER AE
  C, U+00C7 -- LATIN CAPITAL LETTER C WITH CEDILLA
  E! U+00C8 -- LATIN CAPITAL LETTER E WITH GRAVE
  E' U+00C9 -- LATIN CAPITAL LETTER E WITH ACUTE
  E> U+00CA -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX
  E: U+00CB -- LATIN CAPITAL LETTER E WITH DIAERESIS
  I! U+00CC -- LATIN CAPITAL LETTER I WITH GRAVE
  I' U+00CD -- LATIN CAPITAL LETTER I WITH ACUTE
  I> U+00CE -- LATIN CAPITAL LETTER I WITH CIRCUMFLEX
  I: U+00CF -- LATIN CAPITAL LETTER I WITH DIAERESIS
  D- U+00D0 -- LATIN CAPITAL LETTER ETH (Icelandic)
  N? U+00D1 -- LATIN CAPITAL LETTER N WITH TILDE
  O! U+00D2 -- LATIN CAPITAL LETTER O WITH GRAVE
  O' U+00D3 -- LATIN CAPITAL LETTER O WITH ACUTE
  O> U+00D4 -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX
  O? U+00D5 -- LATIN CAPITAL LETTER O WITH TILDE
  O: U+00D6 -- LATIN CAPITAL LETTER O WITH DIAERESIS
  *X U+00D7 -- MULTIPLICATION SIGN
  O/ U+00D8 -- LATIN CAPITAL LETTER O WITH STROKE
  U! U+00D9 -- LATIN CAPITAL LETTER U WITH GRAVE
  U' U+00DA -- LATIN CAPITAL LETTER U WITH ACUTE
  U> U+00DB -- LATIN CAPITAL LETTER U WITH CIRCUMFLEX
  U: U+00DC -- LATIN CAPITAL LETTER U WITH DIAERESIS
  Y' U+00DD -- LATIN CAPITAL LETTER Y WITH ACUTE
  TH U+00DE -- LATIN CAPITAL LETTER THORN (Icelandic)
  ss U+00DF -- LATIN SMALL LETTER SHARP S (German)
  a! U+00E0 -- LATIN SMALL LETTER A WITH GRAVE
  a' U+00E1 -- LATIN SMALL LETTER A WITH ACUTE
  a> U+00E2 -- LATIN SMALL LETTER A WITH CIRCUMFLEX
  a? U+00E3 -- LATIN SMALL LETTER A WITH TILDE
  a: U+00E4 -- LATIN SMALL LETTER A WITH DIAERESIS
  aa U+00E5 -- LATIN SMALL LETTER A WITH RING ABOVE
  ae U+00E6 -- LATIN SMALL LETTER AE
  c, U+00E7 -- LATIN SMALL LETTER C WITH CEDILLA
  e! U+00E8 -- LATIN SMALL LETTER E WITH GRAVE
  e' U+00E9 -- LATIN SMALL LETTER E WITH ACUTE
  e> U+00EA -- LATIN SMALL LETTER E WITH CIRCUMFLEX
  e: U+00EB -- LATIN SMALL LETTER E WITH DIAERESIS
  i! U+00EC -- LATIN SMALL LETTER I WITH GRAVE
  i' U+00ED -- LATIN SMALL LETTER I WITH ACUTE
  i> U+00EE -- LATIN SMALL LETTER I WITH CIRCUMFLEX
  i: U+00EF -- LATIN SMALL LETTER I WITH DIAERESIS
  d- U+00F0 -- LATIN SMALL LETTER ETH (Icelandic)
  n? U+00F1 -- LATIN SMALL LETTER N WITH TILDE
  o! U+00F2 -- LATIN SMALL LETTER O WITH GRAVE
  o' U+00F3 -- LATIN SMALL LETTER O WITH ACUTE
  o> U+00F4 -- LATIN SMALL LETTER O WITH CIRCUMFLEX
  o? U+00F5 -- LATIN SMALL LETTER O WITH TILDE
  o: U+00F6 -- LATIN SMALL LETTER O WITH DIAERESIS
  -: U+00F7 -- DIVISION SIGN
  o/ U+00F8 -- LATIN SMALL LETTER O WITH STROKE
  u! U+00F9 -- LATIN SMALL LETTER U WITH GRAVE
  u' U+00FA -- LATIN SMALL LETTER U WITH ACUTE
  u> U+00FB -- LATIN SMALL LETTER U WITH CIRCUMFLEX
  u: U+00FC -- LATIN SMALL LETTER U WITH DIAERESIS
  y' U+00FD -- LATIN SMALL LETTER Y WITH ACUTE
  th U+00FE -- LATIN SMALL LETTER THORN (Icelandic)
  y: U+00FF -- LATIN SMALL LETTER Y WITH DIAERESIS
  A- U+0100 -- LATIN CAPITAL LETTER A WITH MACRON
  a- U+0101 -- LATIN SMALL LETTER A WITH MACRON
  A( U+0102 -- LATIN CAPITAL LETTER A WITH BREVE
  a( U+0103 -- LATIN SMALL LETTER A WITH BREVE
  A; U+0104 -- LATIN CAPITAL LETTER A WITH OGONEK
  a; U+0105 -- LATIN SMALL LETTER A WITH OGONEK
  C' U+0106 -- LATIN CAPITAL LETTER C WITH ACUTE
  c' U+0107 -- LATIN SMALL LETTER C WITH ACUTE
  C> U+0108 -- LATIN CAPITAL LETTER C WITH CIRCUMFLEX
  c> U+0109 -- LATIN SMALL LETTER C WITH CIRCUMFLEX
  C. U+010A -- LATIN CAPITAL LETTER C WITH DOT ABOVE
  c. U+010B -- LATIN SMALL LETTER C WITH DOT ABOVE
  C< U+010C -- LATIN CAPITAL LETTER C WITH CARON
  c< U+010D -- LATIN SMALL LETTER C WITH CARON
  D< U+010E -- LATIN CAPITAL LETTER D WITH CARON
  d< U+010F -- LATIN SMALL LETTER D WITH CARON
  D/ U+0110 -- LATIN CAPITAL LETTER D WITH STROKE
  d/ U+0111 -- LATIN SMALL LETTER D WITH STROKE
  E- U+0112 -- LATIN CAPITAL LETTER E WITH MACRON
  e- U+0113 -- LATIN SMALL LETTER E WITH MACRON
  E( U+0114 -- LATIN CAPITAL LETTER E WITH BREVE
  e( U+0115 -- LATIN SMALL LETTER E WITH BREVE
  E. U+0116 -- LATIN CAPITAL LETTER E WITH DOT ABOVE
  e. U+0117 -- LATIN SMALL LETTER E WITH DOT ABOVE
  E; U+0118 -- LATIN CAPITAL LETTER E WITH OGONEK
  e; U+0119 -- LATIN SMALL LETTER E WITH OGONEK
  E< U+011A -- LATIN CAPITAL LETTER E WITH CARON
  e< U+011B -- LATIN SMALL LETTER E WITH CARON
  G> U+011C -- LATIN CAPITAL LETTER G WITH CIRCUMFLEX
  g> U+011D -- LATIN SMALL LETTER G WITH CIRCUMFLEX
  G( U+011E -- LATIN CAPITAL LETTER G WITH BREVE
  g( U+011F -- LATIN SMALL LETTER G WITH BREVE
  G. U+0120 -- LATIN CAPITAL LETTER G WITH DOT ABOVE
  g. U+0121 -- LATIN SMALL LETTER G WITH DOT ABOVE
  G, U+0122 -- LATIN CAPITAL LETTER G WITH CEDILLA
  g, U+0123 -- LATIN SMALL LETTER G WITH CEDILLA
  H> U+0124 -- LATIN CAPITAL LETTER H WITH CIRCUMFLEX
  h> U+0125 -- LATIN SMALL LETTER H WITH CIRCUMFLEX
  H/ U+0126 -- LATIN CAPITAL LETTER H WITH STROKE
  h/ U+0127 -- LATIN SMALL LETTER H WITH STROKE
  I? U+0128 -- LATIN CAPITAL LETTER I WITH TILDE
  i? U+0129 -- LATIN SMALL LETTER I WITH TILDE
  I- U+012A -- LATIN CAPITAL LETTER I WITH MACRON
  i- U+012B -- LATIN SMALL LETTER I WITH MACRON
  I( U+012C -- LATIN CAPITAL LETTER I WITH BREVE
  i( U+012D -- LATIN SMALL LETTER I WITH BREVE
  I; U+012E -- LATIN CAPITAL LETTER I WITH OGONEK
  i; U+012F -- LATIN SMALL LETTER I WITH OGONEK
  I. U+0130 -- LATIN CAPITAL LETTER I WITH DOT ABOVE
  i. U+0131 -- LATIN SMALL LETTER DOTLESS I
  IJ U+0132 -- LATIN CAPITAL LIGATURE IJ
  ij U+0133 -- LATIN SMALL LIGATURE IJ
  J> U+0134 -- LATIN CAPITAL LETTER J WITH CIRCUMFLEX
  j> U+0135 -- LATIN SMALL LETTER J WITH CIRCUMFLEX
  K, U+0136 -- LATIN CAPITAL LETTER K WITH CEDILLA
  k, U+0137 -- LATIN SMALL LETTER K WITH CEDILLA
  kk U+0138 -- LATIN SMALL LETTER KRA
  L' U+0139 -- LATIN CAPITAL LETTER L WITH ACUTE
  l' U+013A -- LATIN SMALL LETTER L WITH ACUTE
  L, U+013B -- LATIN CAPITAL LETTER L WITH CEDILLA
  l, U+013C -- LATIN SMALL LETTER L WITH CEDILLA
  L< U+013D -- LATIN CAPITAL LETTER L WITH CARON
  l< U+013E -- LATIN SMALL LETTER L WITH CARON
  L. U+013F -- LATIN CAPITAL LETTER L WITH MIDDLE DOT
  l. U+0140 -- LATIN SMALL LETTER L WITH MIDDLE DOT
  L/ U+0141 -- LATIN CAPITAL LETTER L WITH STROKE
  l/ U+0142 -- LATIN SMALL LETTER L WITH STROKE
  N' U+0143 -- LATIN CAPITAL LETTER N WITH ACUTE `
  n' U+0144 -- LATIN SMALL LETTER N WITH ACUTE `
  N, U+0145 -- LATIN CAPITAL LETTER N WITH CEDILLA `
  n, U+0146 -- LATIN SMALL LETTER N WITH CEDILLA `
  N< U+0147 -- LATIN CAPITAL LETTER N WITH CARON `
  n< U+0148 -- LATIN SMALL LETTER N WITH CARON `
  'n U+0149 -- LATIN SMALL LETTER N PRECEDED BY APOSTROPHE `
  NG U+014A -- LATIN CAPITAL LETTER ENG
  ng U+014B -- LATIN SMALL LETTER ENG
  O- U+014C -- LATIN CAPITAL LETTER O WITH MACRON
  o- U+014D -- LATIN SMALL LETTER O WITH MACRON
  O( U+014E -- LATIN CAPITAL LETTER O WITH BREVE
  o( U+014F -- LATIN SMALL LETTER O WITH BREVE
  O" U+0150 -- LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
  o" U+0151 -- LATIN SMALL LETTER O WITH DOUBLE ACUTE
  OE U+0152 -- LATIN CAPITAL LIGATURE OE
  oe U+0153 -- LATIN SMALL LIGATURE OE
  R' U+0154 -- LATIN CAPITAL LETTER R WITH ACUTE
  r' U+0155 -- LATIN SMALL LETTER R WITH ACUTE
  R, U+0156 -- LATIN CAPITAL LETTER R WITH CEDILLA
  r, U+0157 -- LATIN SMALL LETTER R WITH CEDILLA
  R< U+0158 -- LATIN CAPITAL LETTER R WITH CARON
  r< U+0159 -- LATIN SMALL LETTER R WITH CARON
  S' U+015A -- LATIN CAPITAL LETTER S WITH ACUTE
  s' U+015B -- LATIN SMALL LETTER S WITH ACUTE
  S> U+015C -- LATIN CAPITAL LETTER S WITH CIRCUMFLEX
  s> U+015D -- LATIN SMALL LETTER S WITH CIRCUMFLEX
  S, U+015E -- LATIN CAPITAL LETTER S WITH CEDILLA
  s, U+015F -- LATIN SMALL LETTER S WITH CEDILLA
  S< U+0160 -- LATIN CAPITAL LETTER S WITH CARON
  s< U+0161 -- LATIN SMALL LETTER S WITH CARON
  T, U+0162 -- LATIN CAPITAL LETTER T WITH CEDILLA
  t, U+0163 -- LATIN SMALL LETTER T WITH CEDILLA
  T< U+0164 -- LATIN CAPITAL LETTER T WITH CARON
  t< U+0165 -- LATIN SMALL LETTER T WITH CARON
  T/ U+0166 -- LATIN CAPITAL LETTER T WITH STROKE
  t/ U+0167 -- LATIN SMALL LETTER T WITH STROKE
  U? U+0168 -- LATIN CAPITAL LETTER U WITH TILDE
  u? U+0169 -- LATIN SMALL LETTER U WITH TILDE
  U- U+016A -- LATIN CAPITAL LETTER U WITH MACRON
  u- U+016B -- LATIN SMALL LETTER U WITH MACRON
  U( U+016C -- LATIN CAPITAL LETTER U WITH BREVE
  u( U+016D -- LATIN SMALL LETTER U WITH BREVE
  U0 U+016E -- LATIN CAPITAL LETTER U WITH RING ABOVE
  u0 U+016F -- LATIN SMALL LETTER U WITH RING ABOVE
  U" U+0170 -- LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
  u" U+0171 -- LATIN SMALL LETTER U WITH DOUBLE ACUTE
  U; U+0172 -- LATIN CAPITAL LETTER U WITH OGONEK
  u; U+0173 -- LATIN SMALL LETTER U WITH OGONEK
  W> U+0174 -- LATIN CAPITAL LETTER W WITH CIRCUMFLEX
  w> U+0175 -- LATIN SMALL LETTER W WITH CIRCUMFLEX
  Y> U+0176 -- LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
  y> U+0177 -- LATIN SMALL LETTER Y WITH CIRCUMFLEX
  Y: U+0178 -- LATIN CAPITAL LETTER Y WITH DIAERESIS
  Z' U+0179 -- LATIN CAPITAL LETTER Z WITH ACUTE
  z' U+017A -- LATIN SMALL LETTER Z WITH ACUTE
  Z. U+017B -- LATIN CAPITAL LETTER Z WITH DOT ABOVE
  z. U+017C -- LATIN SMALL LETTER Z WITH DOT ABOVE
  Z< U+017D -- LATIN CAPITAL LETTER Z WITH CARON
  z< U+017E -- LATIN SMALL LETTER Z WITH CARON
  O9 U+01A0 -- LATIN CAPITAL LETTER O WITH HORN
  o9 U+01A1 -- LATIN SMALL LETTER O WITH HORN
  OI U+01A2 -- LATIN CAPITAL LETTER OI
  oi U+01A3 -- LATIN SMALL LETTER OI
  yr U+01A6 -- LATIN LETTER YR
  U9 U+01AF -- LATIN CAPITAL LETTER U WITH HORN
  u9 U+01B0 -- LATIN SMALL LETTER U WITH HORN
  Z/ U+01B5 -- LATIN CAPITAL LETTER Z WITH STROKE
  z/ U+01B6 -- LATIN SMALL LETTER Z WITH STROKE
  ED U+01B7 -- LATIN CAPITAL LETTER EZH
  A< U+01CD -- LATIN CAPITAL LETTER A WITH CARON
  a< U+01CE -- LATIN SMALL LETTER A WITH CARON
  I< U+01CF -- LATIN CAPITAL LETTER I WITH CARON
  i< U+01D0 -- LATIN SMALL LETTER I WITH CARON
  O< U+01D1 -- LATIN CAPITAL LETTER O WITH CARON
  o< U+01D2 -- LATIN SMALL LETTER O WITH CARON
  U< U+01D3 -- LATIN CAPITAL LETTER U WITH CARON
  u< U+01D4 -- LATIN SMALL LETTER U WITH CARON
  A1 U+01DE -- LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
  a1 U+01DF -- LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
  A7 U+01E0 -- LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
  a7 U+01E1 -- LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
  A3 U+01E2 -- LATIN CAPITAL LETTER AE WITH MACRON
  a3 U+01E3 -- LATIN SMALL LETTER AE WITH MACRON
  G/ U+01E4 -- LATIN CAPITAL LETTER G WITH STROKE
  g/ U+01E5 -- LATIN SMALL LETTER G WITH STROKE
  G< U+01E6 -- LATIN CAPITAL LETTER G WITH CARON
  g< U+01E7 -- LATIN SMALL LETTER G WITH CARON
  K< U+01E8 -- LATIN CAPITAL LETTER K WITH CARON
  k< U+01E9 -- LATIN SMALL LETTER K WITH CARON
  O; U+01EA -- LATIN CAPITAL LETTER O WITH OGONEK
  o; U+01EB -- LATIN SMALL LETTER O WITH OGONEK
  O1 U+01EC -- LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
  o1 U+01ED -- LATIN SMALL LETTER O WITH OGONEK AND MACRON
  EZ U+01EE -- LATIN CAPITAL LETTER EZH WITH CARON
  ez U+01EF -- LATIN SMALL LETTER EZH WITH CARON
  j< U+01F0 -- LATIN SMALL LETTER J WITH CARON
  G' U+01F4 -- LATIN CAPITAL LETTER G WITH ACUTE
  g' U+01F5 -- LATIN SMALL LETTER G WITH ACUTE
  ;S U+02BF -- MODIFIER LETTER LEFT HALF RING
  '< U+02C7 -- CARON
  '( U+02D8 -- BREVE
  '. U+02D9 -- DOT ABOVE
  '0 U+02DA -- RING ABOVE
  '; U+02DB -- OGONEK
  '" U+02DD -- DOUBLE ACUTE ACCENT
  A% U+0386 -- GREEK CAPITAL LETTER ALPHA WITH TONOS
  E% U+0388 -- GREEK CAPITAL LETTER EPSILON WITH TONOS
  Y% U+0389 -- GREEK CAPITAL LETTER ETA WITH TONOS
  I% U+038A -- GREEK CAPITAL LETTER IOTA WITH TONOS
  O% U+038C -- GREEK CAPITAL LETTER OMICRON WITH TONOS
  U% U+038E -- GREEK CAPITAL LETTER UPSILON WITH TONOS
  W% U+038F -- GREEK CAPITAL LETTER OMEGA WITH TONOS
  i3 U+0390 -- GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
  A* U+0391 -- GREEK CAPITAL LETTER ALPHA
  B* U+0392 -- GREEK CAPITAL LETTER BETA
  G* U+0393 -- GREEK CAPITAL LETTER GAMMA
  D* U+0394 -- GREEK CAPITAL LETTER DELTA
  E* U+0395 -- GREEK CAPITAL LETTER EPSILON
  Z* U+0396 -- GREEK CAPITAL LETTER ZETA
  Y* U+0397 -- GREEK CAPITAL LETTER ETA
  H* U+0398 -- GREEK CAPITAL LETTER THETA
  I* U+0399 -- GREEK CAPITAL LETTER IOTA
  K* U+039A -- GREEK CAPITAL LETTER KAPPA
  L* U+039B -- GREEK CAPITAL LETTER LAMDA
  M* U+039C -- GREEK CAPITAL LETTER MU
  N* U+039D -- GREEK CAPITAL LETTER NU
  C* U+039E -- GREEK CAPITAL LETTER XI
  O* U+039F -- GREEK CAPITAL LETTER OMICRON
  P* U+03A0 -- GREEK CAPITAL LETTER PI
  R* U+03A1 -- GREEK CAPITAL LETTER RHO
  S* U+03A3 -- GREEK CAPITAL LETTER SIGMA
  T* U+03A4 -- GREEK CAPITAL LETTER TAU
  U* U+03A5 -- GREEK CAPITAL LETTER UPSILON
  F* U+03A6 -- GREEK CAPITAL LETTER PHI
  X* U+03A7 -- GREEK CAPITAL LETTER CHI
  Q* U+03A8 -- GREEK CAPITAL LETTER PSI
  W* U+03A9 -- GREEK CAPITAL LETTER OMEGA
  J* U+03AA -- GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
  V* U+03AB -- GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
  a% U+03AC -- GREEK SMALL LETTER ALPHA WITH TONOS
  e% U+03AD -- GREEK SMALL LETTER EPSILON WITH TONOS
  y% U+03AE -- GREEK SMALL LETTER ETA WITH TONOS
  i% U+03AF -- GREEK SMALL LETTER IOTA WITH TONOS
  u3 U+03B0 -- GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
  a* U+03B1 -- GREEK SMALL LETTER ALPHA
  b* U+03B2 -- GREEK SMALL LETTER BETA
  g* U+03B3 -- GREEK SMALL LETTER GAMMA
  d* U+03B4 -- GREEK SMALL LETTER DELTA
  e* U+03B5 -- GREEK SMALL LETTER EPSILON
  z* U+03B6 -- GREEK SMALL LETTER ZETA
  y* U+03B7 -- GREEK SMALL LETTER ETA
  h* U+03B8 -- GREEK SMALL LETTER THETA
  i* U+03B9 -- GREEK SMALL LETTER IOTA
  k* U+03BA -- GREEK SMALL LETTER KAPPA
  l* U+03BB -- GREEK SMALL LETTER LAMDA
  m* U+03BC -- GREEK SMALL LETTER MU
  n* U+03BD -- GREEK SMALL LETTER NU
  c* U+03BE -- GREEK SMALL LETTER XI
  o* U+03BF -- GREEK SMALL LETTER OMICRON
  p* U+03C0 -- GREEK SMALL LETTER PI
  r* U+03C1 -- GREEK SMALL LETTER RHO
  *s U+03C2 -- GREEK SMALL LETTER FINAL SIGMA
  s* U+03C3 -- GREEK SMALL LETTER SIGMA
  t* U+03C4 -- GREEK SMALL LETTER TAU
  u* U+03C5 -- GREEK SMALL LETTER UPSILON
  f* U+03C6 -- GREEK SMALL LETTER PHI
  x* U+03C7 -- GREEK SMALL LETTER CHI
  q* U+03C8 -- GREEK SMALL LETTER PSI
  w* U+03C9 -- GREEK SMALL LETTER OMEGA
  j* U+03CA -- GREEK SMALL LETTER IOTA WITH DIALYTIKA
  v* U+03CB -- GREEK SMALL LETTER UPSILON WITH DIALYTIKA
  o% U+03CC -- GREEK SMALL LETTER OMICRON WITH TONOS
  u% U+03CD -- GREEK SMALL LETTER UPSILON WITH TONOS
  w% U+03CE -- GREEK SMALL LETTER OMEGA WITH TONOS
  'G U+03D8 -- GREEK LETTER ARCHAIC KOPPA
  ,G U+03D9 -- GREEK SMALL LETTER ARCHAIC KOPPA
  T3 U+03DA -- GREEK LETTER STIGMA
  t3 U+03DB -- GREEK SMALL LETTER STIGMA
  M3 U+03DC -- GREEK LETTER DIGAMMA
  m3 U+03DD -- GREEK SMALL LETTER DIGAMMA
  K3 U+03DE -- GREEK LETTER KOPPA
  k3 U+03DF -- GREEK SMALL LETTER KOPPA
  P3 U+03E0 -- GREEK LETTER SAMPI
  p3 U+03E1 -- GREEK SMALL LETTER SAMPI
  '% U+03F4 -- GREEK CAPITAL THETA SYMBOL
  j3 U+03F5 -- GREEK LUNATE EPSILON SYMBOL
  IO U+0401 -- CYRILLIC CAPITAL LETTER IO
  D% U+0402 -- CYRILLIC CAPITAL LETTER DJE
  G% U+0403 -- CYRILLIC CAPITAL LETTER GJE
  IE U+0404 -- CYRILLIC CAPITAL LETTER UKRAINIAN IE
  DS U+0405 -- CYRILLIC CAPITAL LETTER DZE
  II U+0406 -- CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
  YI U+0407 -- CYRILLIC CAPITAL LETTER YI
  J% U+0408 -- CYRILLIC CAPITAL LETTER JE
  LJ U+0409 -- CYRILLIC CAPITAL LETTER LJE
  NJ U+040A -- CYRILLIC CAPITAL LETTER NJE
  Ts U+040B -- CYRILLIC CAPITAL LETTER TSHE
  KJ U+040C -- CYRILLIC CAPITAL LETTER KJE
  V% U+040E -- CYRILLIC CAPITAL LETTER SHORT U
  DZ U+040F -- CYRILLIC CAPITAL LETTER DZHE
  A= U+0410 -- CYRILLIC CAPITAL LETTER A
  B= U+0411 -- CYRILLIC CAPITAL LETTER BE
  V= U+0412 -- CYRILLIC CAPITAL LETTER VE
  G= U+0413 -- CYRILLIC CAPITAL LETTER GHE
  D= U+0414 -- CYRILLIC CAPITAL LETTER DE
  E= U+0415 -- CYRILLIC CAPITAL LETTER IE
  Z% U+0416 -- CYRILLIC CAPITAL LETTER ZHE
  Z= U+0417 -- CYRILLIC CAPITAL LETTER ZE
  I= U+0418 -- CYRILLIC CAPITAL LETTER I
  J= U+0419 -- CYRILLIC CAPITAL LETTER SHORT I
  K= U+041A -- CYRILLIC CAPITAL LETTER KA
  L= U+041B -- CYRILLIC CAPITAL LETTER EL
  M= U+041C -- CYRILLIC CAPITAL LETTER EM
  N= U+041D -- CYRILLIC CAPITAL LETTER EN
  O= U+041E -- CYRILLIC CAPITAL LETTER O
  P= U+041F -- CYRILLIC CAPITAL LETTER PE
  R= U+0420 -- CYRILLIC CAPITAL LETTER ER
  S= U+0421 -- CYRILLIC CAPITAL LETTER ES
  T= U+0422 -- CYRILLIC CAPITAL LETTER TE
  U= U+0423 -- CYRILLIC CAPITAL LETTER U
  F= U+0424 -- CYRILLIC CAPITAL LETTER EF
  H= U+0425 -- CYRILLIC CAPITAL LETTER HA
  C= U+0426 -- CYRILLIC CAPITAL LETTER TSE
  C% U+0427 -- CYRILLIC CAPITAL LETTER CHE
  S% U+0428 -- CYRILLIC CAPITAL LETTER SHA
  Sc U+0429 -- CYRILLIC CAPITAL LETTER SHCHA
  =' U+042A -- CYRILLIC CAPITAL LETTER HARD SIGN
  Y= U+042B -- CYRILLIC CAPITAL LETTER YERU
  %' U+042C -- CYRILLIC CAPITAL LETTER SOFT SIGN
  JE U+042D -- CYRILLIC CAPITAL LETTER E
  JU U+042E -- CYRILLIC CAPITAL LETTER YU
  JA U+042F -- CYRILLIC CAPITAL LETTER YA
  a= U+0430 -- CYRILLIC SMALL LETTER A
  b= U+0431 -- CYRILLIC SMALL LETTER BE
  v= U+0432 -- CYRILLIC SMALL LETTER VE
  g= U+0433 -- CYRILLIC SMALL LETTER GHE
  d= U+0434 -- CYRILLIC SMALL LETTER DE
  e= U+0435 -- CYRILLIC SMALL LETTER IE
  z% U+0436 -- CYRILLIC SMALL LETTER ZHE
  z= U+0437 -- CYRILLIC SMALL LETTER ZE
  i= U+0438 -- CYRILLIC SMALL LETTER I
  j= U+0439 -- CYRILLIC SMALL LETTER SHORT I
  k= U+043A -- CYRILLIC SMALL LETTER KA
  l= U+043B -- CYRILLIC SMALL LETTER EL
  m= U+043C -- CYRILLIC SMALL LETTER EM
  n= U+043D -- CYRILLIC SMALL LETTER EN
  o= U+043E -- CYRILLIC SMALL LETTER O
  p= U+043F -- CYRILLIC SMALL LETTER PE
  r= U+0440 -- CYRILLIC SMALL LETTER ER
  s= U+0441 -- CYRILLIC SMALL LETTER ES
  t= U+0442 -- CYRILLIC SMALL LETTER TE
  u= U+0443 -- CYRILLIC SMALL LETTER U
  f= U+0444 -- CYRILLIC SMALL LETTER EF
  h= U+0445 -- CYRILLIC SMALL LETTER HA
  c= U+0446 -- CYRILLIC SMALL LETTER TSE
  c% U+0447 -- CYRILLIC SMALL LETTER CHE
  s% U+0448 -- CYRILLIC SMALL LETTER SHA
  sc U+0449 -- CYRILLIC SMALL LETTER SHCHA
  =' U+044A -- CYRILLIC SMALL LETTER HARD SIGN
  y= U+044B -- CYRILLIC SMALL LETTER YERU
  %' U+044C -- CYRILLIC SMALL LETTER SOFT SIGN
  je U+044D -- CYRILLIC SMALL LETTER E
  ju U+044E -- CYRILLIC SMALL LETTER YU
  ja U+044F -- CYRILLIC SMALL LETTER YA
  io U+0451 -- CYRILLIC SMALL LETTER IO
  d% U+0452 -- CYRILLIC SMALL LETTER DJE
  g% U+0453 -- CYRILLIC SMALL LETTER GJE
  ie U+0454 -- CYRILLIC SMALL LETTER UKRAINIAN IE
  ds U+0455 -- CYRILLIC SMALL LETTER DZE
  ii U+0456 -- CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
  yi U+0457 -- CYRILLIC SMALL LETTER YI
  j% U+0458 -- CYRILLIC SMALL LETTER JE
  lj U+0459 -- CYRILLIC SMALL LETTER LJE
  nj U+045A -- CYRILLIC SMALL LETTER NJE
  ts U+045B -- CYRILLIC SMALL LETTER TSHE
  kj U+045C -- CYRILLIC SMALL LETTER KJE
  v% U+045E -- CYRILLIC SMALL LETTER SHORT U
  dz U+045F -- CYRILLIC SMALL LETTER DZHE
  Y3 U+0462 -- CYRILLIC CAPITAL LETTER YAT
  y3 U+0463 -- CYRILLIC SMALL LETTER YAT
  O3 U+046A -- CYRILLIC CAPITAL LETTER BIG YUS
  o3 U+046B -- CYRILLIC SMALL LETTER BIG YUS
  F3 U+0472 -- CYRILLIC CAPITAL LETTER FITA
  f3 U+0473 -- CYRILLIC SMALL LETTER FITA
  V3 U+0474 -- CYRILLIC CAPITAL LETTER IZHITSA
  v3 U+0475 -- CYRILLIC SMALL LETTER IZHITSA
  C3 U+0480 -- CYRILLIC CAPITAL LETTER KOPPA
  c3 U+0481 -- CYRILLIC SMALL LETTER KOPPA
  G3 U+0490 -- CYRILLIC CAPITAL LETTER GHE WITH UPTURN
  g3 U+0491 -- CYRILLIC SMALL LETTER GHE WITH UPTURN
  A+ U+05D0 -- HEBREW LETTER ALEF
  B+ U+05D1 -- HEBREW LETTER BET
  G+ U+05D2 -- HEBREW LETTER GIMEL
  D+ U+05D3 -- HEBREW LETTER DALET
  H+ U+05D4 -- HEBREW LETTER HE
  W+ U+05D5 -- HEBREW LETTER VAV
  Z+ U+05D6 -- HEBREW LETTER ZAYIN
  X+ U+05D7 -- HEBREW LETTER HET
  Tj U+05D8 -- HEBREW LETTER TET
  J+ U+05D9 -- HEBREW LETTER YOD
  K% U+05DA -- HEBREW LETTER FINAL KAF
  K+ U+05DB -- HEBREW LETTER KAF
  L+ U+05DC -- HEBREW LETTER LAMED
  M% U+05DD -- HEBREW LETTER FINAL MEM
  M+ U+05DE -- HEBREW LETTER MEM
  N% U+05DF -- HEBREW LETTER FINAL NUN `
  N+ U+05E0 -- HEBREW LETTER NUN `
  S+ U+05E1 -- HEBREW LETTER SAMEKH
  E+ U+05E2 -- HEBREW LETTER AYIN
  P% U+05E3 -- HEBREW LETTER FINAL PE
  P+ U+05E4 -- HEBREW LETTER PE
  Zj U+05E5 -- HEBREW LETTER FINAL TSADI
  ZJ U+05E6 -- HEBREW LETTER TSADI
  Q+ U+05E7 -- HEBREW LETTER QOF
  R+ U+05E8 -- HEBREW LETTER RESH
  Sh U+05E9 -- HEBREW LETTER SHIN
  T+ U+05EA -- HEBREW LETTER TAV
  ,+ U+060C -- ARABIC COMMA
  ;+ U+061B -- ARABIC SEMICOLON
  ?+ U+061F -- ARABIC QUESTION MARK
  H' U+0621 -- ARABIC LETTER HAMZA
  aM U+0622 -- ARABIC LETTER ALEF WITH MADDA ABOVE
  aH U+0623 -- ARABIC LETTER ALEF WITH HAMZA ABOVE
  wH U+0624 -- ARABIC LETTER WAW WITH HAMZA ABOVE
  ah U+0625 -- ARABIC LETTER ALEF WITH HAMZA BELOW
  yH U+0626 -- ARABIC LETTER YEH WITH HAMZA ABOVE
  a+ U+0627 -- ARABIC LETTER ALEF
  b+ U+0628 -- ARABIC LETTER BEH
  tm U+0629 -- ARABIC LETTER TEH MARBUTA
  t+ U+062A -- ARABIC LETTER TEH
  tk U+062B -- ARABIC LETTER THEH
  g+ U+062C -- ARABIC LETTER JEEM
  hk U+062D -- ARABIC LETTER HAH
  x+ U+062E -- ARABIC LETTER KHAH
  d+ U+062F -- ARABIC LETTER DAL
  dk U+0630 -- ARABIC LETTER THAL
  r+ U+0631 -- ARABIC LETTER REH
  z+ U+0632 -- ARABIC LETTER ZAIN
  s+ U+0633 -- ARABIC LETTER SEEN
  sn U+0634 -- ARABIC LETTER SHEEN
  c+ U+0635 -- ARABIC LETTER SAD
  dd U+0636 -- ARABIC LETTER DAD
  tj U+0637 -- ARABIC LETTER TAH
  zH U+0638 -- ARABIC LETTER ZAH
  e+ U+0639 -- ARABIC LETTER AIN
  i+ U+063A -- ARABIC LETTER GHAIN
  ++ U+0640 -- ARABIC TATWEEL
  f+ U+0641 -- ARABIC LETTER FEH
  q+ U+0642 -- ARABIC LETTER QAF
  k+ U+0643 -- ARABIC LETTER KAF
  l+ U+0644 -- ARABIC LETTER LAM
  m+ U+0645 -- ARABIC LETTER MEEM
  n+ U+0646 -- ARABIC LETTER NOON
  h+ U+0647 -- ARABIC LETTER HEH
  w+ U+0648 -- ARABIC LETTER WAW
  j+ U+0649 -- ARABIC LETTER ALEF MAKSURA
  y+ U+064A -- ARABIC LETTER YEH
  :+ U+064B -- ARABIC FATHATAN
  "+ U+064C -- ARABIC DAMMATAN
  =+ U+064D -- ARABIC KASRATAN
  /+ U+064E -- ARABIC FATHA
  '+ U+064F -- ARABIC DAMMA
  1+ U+0650 -- ARABIC KASRA
  3+ U+0651 -- ARABIC SHADDA
  0+ U+0652 -- ARABIC SUKUN
  aS U+0670 -- ARABIC LETTER SUPERSCRIPT ALEF
  p+ U+067E -- ARABIC LETTER PEH
  v+ U+06A4 -- ARABIC LETTER VEH
  gf U+06AF -- ARABIC LETTER GAF
  0a U+06F0 -- EXTENDED ARABIC-INDIC DIGIT ZERO
  1a U+06F1 -- EXTENDED ARABIC-INDIC DIGIT ONE
  2a U+06F2 -- EXTENDED ARABIC-INDIC DIGIT TWO
  3a U+06F3 -- EXTENDED ARABIC-INDIC DIGIT THREE
  4a U+06F4 -- EXTENDED ARABIC-INDIC DIGIT FOUR
  5a U+06F5 -- EXTENDED ARABIC-INDIC DIGIT FIVE
  6a U+06F6 -- EXTENDED ARABIC-INDIC DIGIT SIX
  7a U+06F7 -- EXTENDED ARABIC-INDIC DIGIT SEVEN
  8a U+06F8 -- EXTENDED ARABIC-INDIC DIGIT EIGHT
  9a U+06F9 -- EXTENDED ARABIC-INDIC DIGIT NINE
  B. U+1E02 -- LATIN CAPITAL LETTER B WITH DOT ABOVE
  b. U+1E03 -- LATIN SMALL LETTER B WITH DOT ABOVE
  B_ U+1E06 -- LATIN CAPITAL LETTER B WITH LINE BELOW
  b_ U+1E07 -- LATIN SMALL LETTER B WITH LINE BELOW
  D. U+1E0A -- LATIN CAPITAL LETTER D WITH DOT ABOVE
  d. U+1E0B -- LATIN SMALL LETTER D WITH DOT ABOVE
  D_ U+1E0E -- LATIN CAPITAL LETTER D WITH LINE BELOW
  d_ U+1E0F -- LATIN SMALL LETTER D WITH LINE BELOW
  D, U+1E10 -- LATIN CAPITAL LETTER D WITH CEDILLA
  d, U+1E11 -- LATIN SMALL LETTER D WITH CEDILLA
  F. U+1E1E -- LATIN CAPITAL LETTER F WITH DOT ABOVE
  f. U+1E1F -- LATIN SMALL LETTER F WITH DOT ABOVE
  G- U+1E20 -- LATIN CAPITAL LETTER G WITH MACRON
  g- U+1E21 -- LATIN SMALL LETTER G WITH MACRON
  H. U+1E22 -- LATIN CAPITAL LETTER H WITH DOT ABOVE
  h. U+1E23 -- LATIN SMALL LETTER H WITH DOT ABOVE
  H: U+1E26 -- LATIN CAPITAL LETTER H WITH DIAERESIS
  h: U+1E27 -- LATIN SMALL LETTER H WITH DIAERESIS
  H, U+1E28 -- LATIN CAPITAL LETTER H WITH CEDILLA
  h, U+1E29 -- LATIN SMALL LETTER H WITH CEDILLA
  K' U+1E30 -- LATIN CAPITAL LETTER K WITH ACUTE
  k' U+1E31 -- LATIN SMALL LETTER K WITH ACUTE
  K_ U+1E34 -- LATIN CAPITAL LETTER K WITH LINE BELOW
  k_ U+1E35 -- LATIN SMALL LETTER K WITH LINE BELOW
  L_ U+1E3A -- LATIN CAPITAL LETTER L WITH LINE BELOW
  l_ U+1E3B -- LATIN SMALL LETTER L WITH LINE BELOW
  M' U+1E3E -- LATIN CAPITAL LETTER M WITH ACUTE
  m' U+1E3F -- LATIN SMALL LETTER M WITH ACUTE
  M. U+1E40 -- LATIN CAPITAL LETTER M WITH DOT ABOVE
  m. U+1E41 -- LATIN SMALL LETTER M WITH DOT ABOVE
  N. U+1E44 -- LATIN CAPITAL LETTER N WITH DOT ABOVE `
  n. U+1E45 -- LATIN SMALL LETTER N WITH DOT ABOVE `
  N_ U+1E48 -- LATIN CAPITAL LETTER N WITH LINE BELOW `
  n_ U+1E49 -- LATIN SMALL LETTER N WITH LINE BELOW `
  P' U+1E54 -- LATIN CAPITAL LETTER P WITH ACUTE
  p' U+1E55 -- LATIN SMALL LETTER P WITH ACUTE
  P. U+1E56 -- LATIN CAPITAL LETTER P WITH DOT ABOVE
  p. U+1E57 -- LATIN SMALL LETTER P WITH DOT ABOVE
  R. U+1E58 -- LATIN CAPITAL LETTER R WITH DOT ABOVE
  r. U+1E59 -- LATIN SMALL LETTER R WITH DOT ABOVE
  R_ U+1E5E -- LATIN CAPITAL LETTER R WITH LINE BELOW
  r_ U+1E5F -- LATIN SMALL LETTER R WITH LINE BELOW
  S. U+1E60 -- LATIN CAPITAL LETTER S WITH DOT ABOVE
  s. U+1E61 -- LATIN SMALL LETTER S WITH DOT ABOVE
  T. U+1E6A -- LATIN CAPITAL LETTER T WITH DOT ABOVE
  t. U+1E6B -- LATIN SMALL LETTER T WITH DOT ABOVE
  T_ U+1E6E -- LATIN CAPITAL LETTER T WITH LINE BELOW
  t_ U+1E6F -- LATIN SMALL LETTER T WITH LINE BELOW
  V? U+1E7C -- LATIN CAPITAL LETTER V WITH TILDE
  v? U+1E7D -- LATIN SMALL LETTER V WITH TILDE
  W! U+1E80 -- LATIN CAPITAL LETTER W WITH GRAVE
  w! U+1E81 -- LATIN SMALL LETTER W WITH GRAVE
  W' U+1E82 -- LATIN CAPITAL LETTER W WITH ACUTE
  w' U+1E83 -- LATIN SMALL LETTER W WITH ACUTE
  W: U+1E84 -- LATIN CAPITAL LETTER W WITH DIAERESIS
  w: U+1E85 -- LATIN SMALL LETTER W WITH DIAERESIS
  W. U+1E86 -- LATIN CAPITAL LETTER W WITH DOT ABOVE
  w. U+1E87 -- LATIN SMALL LETTER W WITH DOT ABOVE
  X. U+1E8A -- LATIN CAPITAL LETTER X WITH DOT ABOVE
  x. U+1E8B -- LATIN SMALL LETTER X WITH DOT ABOVE
  X: U+1E8C -- LATIN CAPITAL LETTER X WITH DIAERESIS
  x: U+1E8D -- LATIN SMALL LETTER X WITH DIAERESIS
  Y. U+1E8E -- LATIN CAPITAL LETTER Y WITH DOT ABOVE
  y. U+1E8F -- LATIN SMALL LETTER Y WITH DOT ABOVE
  Z> U+1E90 -- LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
  z> U+1E91 -- LATIN SMALL LETTER Z WITH CIRCUMFLEX
  Z_ U+1E94 -- LATIN CAPITAL LETTER Z WITH LINE BELOW
  z_ U+1E95 -- LATIN SMALL LETTER Z WITH LINE BELOW
  h_ U+1E96 -- LATIN SMALL LETTER H WITH LINE BELOW
  t: U+1E97 -- LATIN SMALL LETTER T WITH DIAERESIS
  w0 U+1E98 -- LATIN SMALL LETTER W WITH RING ABOVE
  y0 U+1E99 -- LATIN SMALL LETTER Y WITH RING ABOVE
  A2 U+1EA2 -- LATIN CAPITAL LETTER A WITH HOOK ABOVE
  a2 U+1EA3 -- LATIN SMALL LETTER A WITH HOOK ABOVE
  E2 U+1EBA -- LATIN CAPITAL LETTER E WITH HOOK ABOVE
  e2 U+1EBB -- LATIN SMALL LETTER E WITH HOOK ABOVE
  E? U+1EBC -- LATIN CAPITAL LETTER E WITH TILDE
  e? U+1EBD -- LATIN SMALL LETTER E WITH TILDE
  I2 U+1EC8 -- LATIN CAPITAL LETTER I WITH HOOK ABOVE
  i2 U+1EC9 -- LATIN SMALL LETTER I WITH HOOK ABOVE
  O2 U+1ECE -- LATIN CAPITAL LETTER O WITH HOOK ABOVE
  o2 U+1ECF -- LATIN SMALL LETTER O WITH HOOK ABOVE
  U2 U+1EE6 -- LATIN CAPITAL LETTER U WITH HOOK ABOVE
  u2 U+1EE7 -- LATIN SMALL LETTER U WITH HOOK ABOVE
  Y! U+1EF2 -- LATIN CAPITAL LETTER Y WITH GRAVE
  y! U+1EF3 -- LATIN SMALL LETTER Y WITH GRAVE
  Y2 U+1EF6 -- LATIN CAPITAL LETTER Y WITH HOOK ABOVE
  y2 U+1EF7 -- LATIN SMALL LETTER Y WITH HOOK ABOVE
  Y? U+1EF8 -- LATIN CAPITAL LETTER Y WITH TILDE
  y? U+1EF9 -- LATIN SMALL LETTER Y WITH TILDE
  ;' U+1F00 -- GREEK SMALL LETTER ALPHA WITH PSILI
  ,' U+1F01 -- GREEK SMALL LETTER ALPHA WITH DASIA
  ;! U+1F02 -- GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA
  ,! U+1F03 -- GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA
  ?; U+1F04 -- GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA
  ?, U+1F05 -- GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA
  !: U+1F06 -- GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI
  ?: U+1F07 -- GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
  1N U+2002 -- EN SPACE
  1M U+2003 -- EM SPACE
  3M U+2004 -- THREE-PER-EM SPACE
  4M U+2005 -- FOUR-PER-EM SPACE
  6M U+2006 -- SIX-PER-EM SPACE
  1T U+2009 -- THIN SPACE
  1H U+200A -- HAIR SPACE
  -1 U+2010 -- HYPHEN
  -N U+2013 -- EN DASH `
  -M U+2014 -- EM DASH
  -3 U+2015 -- HORIZONTAL BAR
  !2 U+2016 -- DOUBLE VERTICAL LINE
  =2 U+2017 -- DOUBLE LOW LINE
  '6 U+2018 -- LEFT SINGLE QUOTATION MARK
  '9 U+2019 -- RIGHT SINGLE QUOTATION MARK
  .9 U+201A -- SINGLE LOW-9 QUOTATION MARK
  9' U+201B -- SINGLE HIGH-REVERSED-9 QUOTATION MARK
  "6 U+201C -- LEFT DOUBLE QUOTATION MARK
  "9 U+201D -- RIGHT DOUBLE QUOTATION MARK
  :9 U+201E -- DOUBLE LOW-9 QUOTATION MARK
  9" U+201F -- DOUBLE HIGH-REVERSED-9 QUOTATION MARK
  /- U+2020 -- DAGGER
  /= U+2021 -- DOUBLE DAGGER
  .. U+2025 -- TWO DOT LEADER
  %0 U+2030 -- PER MILLE SIGN
  1' U+2032 -- PRIME
  2' U+2033 -- DOUBLE PRIME
  3' U+2034 -- TRIPLE PRIME
  1" U+2035 -- REVERSED PRIME
  2" U+2036 -- REVERSED DOUBLE PRIME
  3" U+2037 -- REVERSED TRIPLE PRIME
  Ca U+2038 -- CARET
  <1 U+2039 -- SINGLE LEFT-POINTING ANGLE QUOTATION MARK
  >1 U+203A -- SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
  :X U+203B -- REFERENCE MARK
  '- U+203E -- OVERLINE
  /f U+2044 -- FRACTION SLASH
  0S U+2070 -- SUPERSCRIPT ZERO
  4S U+2074 -- SUPERSCRIPT FOUR
  5S U+2075 -- SUPERSCRIPT FIVE
  6S U+2076 -- SUPERSCRIPT SIX
  7S U+2077 -- SUPERSCRIPT SEVEN
  8S U+2078 -- SUPERSCRIPT EIGHT
  9S U+2079 -- SUPERSCRIPT NINE
  +S U+207A -- SUPERSCRIPT PLUS SIGN
  -S U+207B -- SUPERSCRIPT MINUS
  =S U+207C -- SUPERSCRIPT EQUALS SIGN
  (S U+207D -- SUPERSCRIPT LEFT PARENTHESIS
  )S U+207E -- SUPERSCRIPT RIGHT PARENTHESIS
  nS U+207F -- SUPERSCRIPT LATIN SMALL LETTER N `
  0s U+2080 -- SUBSCRIPT ZERO
  1s U+2081 -- SUBSCRIPT ONE
  2s U+2082 -- SUBSCRIPT TWO
  3s U+2083 -- SUBSCRIPT THREE
  4s U+2084 -- SUBSCRIPT FOUR
  5s U+2085 -- SUBSCRIPT FIVE
  6s U+2086 -- SUBSCRIPT SIX
  7s U+2087 -- SUBSCRIPT SEVEN
  8s U+2088 -- SUBSCRIPT EIGHT
  9s U+2089 -- SUBSCRIPT NINE
  +s U+208A -- SUBSCRIPT PLUS SIGN
  -s U+208B -- SUBSCRIPT MINUS
  =s U+208C -- SUBSCRIPT EQUALS SIGN
  (s U+208D -- SUBSCRIPT LEFT PARENTHESIS
  )s U+208E -- SUBSCRIPT RIGHT PARENTHESIS
  Li U+20A4 -- LIRA SIGN
  Pt U+20A7 -- PESETA SIGN
  W= U+20A9 -- WON SIGN
  Eu U+20AC -- EURO SIGN
  oC U+2103 -- DEGREE CELSIUS
  co U+2105 -- CARE OF
  oF U+2109 -- DEGREE FAHRENHEIT
  N0 U+2116 -- NUMERO SIGN
  PO U+2117 -- SOUND RECORDING COPYRIGHT
  Rx U+211E -- PRESCRIPTION TAKE
  SM U+2120 -- SERVICE MARK
  TM U+2122 -- TRADE MARK SIGN
  Om U+2126 -- OHM SIGN
  AO U+212B -- ANGSTROM SIGN
  13 U+2153 -- VULGAR FRACTION ONE THIRD
  23 U+2154 -- VULGAR FRACTION TWO THIRDS
  15 U+2155 -- VULGAR FRACTION ONE FIFTH
  25 U+2156 -- VULGAR FRACTION TWO FIFTHS
  35 U+2157 -- VULGAR FRACTION THREE FIFTHS
  45 U+2158 -- VULGAR FRACTION FOUR FIFTHS
  16 U+2159 -- VULGAR FRACTION ONE SIXTH
  56 U+215A -- VULGAR FRACTION FIVE SIXTHS
  18 U+215B -- VULGAR FRACTION ONE EIGHTH
  38 U+215C -- VULGAR FRACTION THREE EIGHTHS
  58 U+215D -- VULGAR FRACTION FIVE EIGHTHS
  78 U+215E -- VULGAR FRACTION SEVEN EIGHTHS
  1R U+2160 -- ROMAN NUMERAL ONE
  2R U+2161 -- ROMAN NUMERAL TWO
  3R U+2162 -- ROMAN NUMERAL THREE
  4R U+2163 -- ROMAN NUMERAL FOUR
  5R U+2164 -- ROMAN NUMERAL FIVE
  6R U+2165 -- ROMAN NUMERAL SIX
  7R U+2166 -- ROMAN NUMERAL SEVEN
  8R U+2167 -- ROMAN NUMERAL EIGHT
  9R U+2168 -- ROMAN NUMERAL NINE
  aR U+2169 -- ROMAN NUMERAL TEN
  bR U+216A -- ROMAN NUMERAL ELEVEN
  cR U+216B -- ROMAN NUMERAL TWELVE
  1r U+2170 -- SMALL ROMAN NUMERAL ONE
  2r U+2171 -- SMALL ROMAN NUMERAL TWO
  3r U+2172 -- SMALL ROMAN NUMERAL THREE
  4r U+2173 -- SMALL ROMAN NUMERAL FOUR
  5r U+2174 -- SMALL ROMAN NUMERAL FIVE
  6r U+2175 -- SMALL ROMAN NUMERAL SIX
  7r U+2176 -- SMALL ROMAN NUMERAL SEVEN
  8r U+2177 -- SMALL ROMAN NUMERAL EIGHT
  9r U+2178 -- SMALL ROMAN NUMERAL NINE
  ar U+2179 -- SMALL ROMAN NUMERAL TEN
  br U+217A -- SMALL ROMAN NUMERAL ELEVEN
  cr U+217B -- SMALL ROMAN NUMERAL TWELVE
  <- U+2190 -- LEFTWARDS ARROW
  -! U+2191 -- UPWARDS ARROW
  -> U+2192 -- RIGHTWARDS ARROW
  -v U+2193 -- DOWNWARDS ARROW
  <> U+2194 -- LEFT RIGHT ARROW
  UD U+2195 -- UP DOWN ARROW
  <= U+21D0 -- LEFTWARDS DOUBLE ARROW
  => U+21D2 -- RIGHTWARDS DOUBLE ARROW
  == U+21D4 -- LEFT RIGHT DOUBLE ARROW
  FA U+2200 -- FOR ALL
  dP U+2202 -- PARTIAL DIFFERENTIAL
  TE U+2203 -- THERE EXISTS
  /0 U+2205 -- EMPTY SET
  DE U+2206 -- INCREMENT
  NB U+2207 -- NABLA
  (- U+2208 -- ELEMENT OF
  -) U+220B -- CONTAINS AS MEMBER
  *P U+220F -- N-ARY PRODUCT `
  +Z U+2211 -- N-ARY SUMMATION `
  -2 U+2212 -- MINUS SIGN
  -+ U+2213 -- MINUS-OR-PLUS SIGN
  *- U+2217 -- ASTERISK OPERATOR
  Ob U+2218 -- RING OPERATOR
  Sb U+2219 -- BULLET OPERATOR
  RT U+221A -- SQUARE ROOT
  0( U+221D -- PROPORTIONAL TO
  00 U+221E -- INFINITY
  -L U+221F -- RIGHT ANGLE
  -V U+2220 -- ANGLE
  PP U+2225 -- PARALLEL TO
  AN U+2227 -- LOGICAL AND
  OR U+2228 -- LOGICAL OR
  (U U+2229 -- INTERSECTION
  )U U+222A -- UNION
  In U+222B -- INTEGRAL
  DI U+222C -- DOUBLE INTEGRAL
  Io U+222E -- CONTOUR INTEGRAL
  .: U+2234 -- THEREFORE
  :. U+2235 -- BECAUSE
  :R U+2236 -- RATIO
  :: U+2237 -- PROPORTION
  ?1 U+223C -- TILDE OPERATOR
  CG U+223E -- INVERTED LAZY S
  ?- U+2243 -- ASYMPTOTICALLY EQUAL TO
  ?= U+2245 -- APPROXIMATELY EQUAL TO
  ?2 U+2248 -- ALMOST EQUAL TO
  =? U+224C -- ALL EQUAL TO
  HI U+2253 -- IMAGE OF OR APPROXIMATELY EQUAL TO
  != U+2260 -- NOT EQUAL TO
  =3 U+2261 -- IDENTICAL TO
  =< U+2264 -- LESS-THAN OR EQUAL TO
  >= U+2265 -- GREATER-THAN OR EQUAL TO
  <* U+226A -- MUCH LESS-THAN
  *> U+226B -- MUCH GREATER-THAN
  !< U+226E -- NOT LESS-THAN
  !> U+226F -- NOT GREATER-THAN
  (C U+2282 -- SUBSET OF
  )C U+2283 -- SUPERSET OF
  (_ U+2286 -- SUBSET OF OR EQUAL TO
  )_ U+2287 -- SUPERSET OF OR EQUAL TO
  0. U+2299 -- CIRCLED DOT OPERATOR
  02 U+229A -- CIRCLED RING OPERATOR
  -T U+22A5 -- UP TACK
  .P U+22C5 -- DOT OPERATOR
  :3 U+22EE -- VERTICAL ELLIPSIS
  .3 U+22EF -- MIDLINE HORIZONTAL ELLIPSIS
  Eh U+2302 -- HOUSE
  <7 U+2308 -- LEFT CEILING
  >7 U+2309 -- RIGHT CEILING
  7< U+230A -- LEFT FLOOR
  7> U+230B -- RIGHT FLOOR
  NI U+2310 -- REVERSED NOT SIGN
  (A U+2312 -- ARC
  TR U+2315 -- TELEPHONE RECORDER
  Iu U+2320 -- TOP HALF INTEGRAL
  Il U+2321 -- BOTTOM HALF INTEGRAL
  </ U+2329 -- LEFT-POINTING ANGLE BRACKET
  /> U+232A -- RIGHT-POINTING ANGLE BRACKET
  Vs U+2423 -- OPEN BOX
  1h U+2440 -- OCR HOOK
  3h U+2441 -- OCR CHAIR
  2h U+2442 -- OCR FORK
  4h U+2443 -- OCR INVERTED FORK
  1j U+2446 -- OCR BRANCH BANK IDENTIFICATION
  2j U+2447 -- OCR AMOUNT OF CHECK
  3j U+2448 -- OCR DASH
  4j U+2449 -- OCR CUSTOMER ACCOUNT NUMBER
  1. U+2488 -- DIGIT ONE FULL STOP
  2. U+2489 -- DIGIT TWO FULL STOP
  3. U+248A -- DIGIT THREE FULL STOP
  4. U+248B -- DIGIT FOUR FULL STOP
  5. U+248C -- DIGIT FIVE FULL STOP
  6. U+248D -- DIGIT SIX FULL STOP
  7. U+248E -- DIGIT SEVEN FULL STOP
  8. U+248F -- DIGIT EIGHT FULL STOP
  9. U+2490 -- DIGIT NINE FULL STOP
  hh U+2500 -- BOX DRAWINGS LIGHT HORIZONTAL
  HH U+2501 -- BOX DRAWINGS HEAVY HORIZONTAL
  vv U+2502 -- BOX DRAWINGS LIGHT VERTICAL
  VV U+2503 -- BOX DRAWINGS HEAVY VERTICAL
  3- U+2504 -- BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL
  3_ U+2505 -- BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL
  3! U+2506 -- BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL
  3/ U+2507 -- BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL
  4- U+2508 -- BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL
  4_ U+2509 -- BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL
  4! U+250A -- BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL
  4/ U+250B -- BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL
  dr U+250C -- BOX DRAWINGS LIGHT DOWN AND RIGHT
  dR U+250D -- BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY
  Dr U+250E -- BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT
  DR U+250F -- BOX DRAWINGS HEAVY DOWN AND RIGHT
  dl U+2510 -- BOX DRAWINGS LIGHT DOWN AND LEFT
  dL U+2511 -- BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY
  Dl U+2512 -- BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT
  LD U+2513 -- BOX DRAWINGS HEAVY DOWN AND LEFT
  ur U+2514 -- BOX DRAWINGS LIGHT UP AND RIGHT
  uR U+2515 -- BOX DRAWINGS UP LIGHT AND RIGHT HEAVY
  Ur U+2516 -- BOX DRAWINGS UP HEAVY AND RIGHT LIGHT
  UR U+2517 -- BOX DRAWINGS HEAVY UP AND RIGHT
  ul U+2518 -- BOX VOICED SOUND MARKDRAWINGS LIGHT UP AND LEFT
  uL U+2519 -- BOX DRAWINGS UP LIGHT AND LEFT HEAVY
  Ul U+251A -- BOX DRAWINGS UP HEAVY AND LEFT LIGHT
  UL U+251B -- BOX DRAWINGS HEAVY UP AND LEFT
  vr U+251C -- BOX DRAWINGS LIGHT VERTICAL AND RIGHT
  vR U+251D -- BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
  Vr U+2520 -- BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
  VR U+2523 -- BOX DRAWINGS HEAVY VERTICAL AND RIGHT
  vl U+2524 -- BOX DRAWINGS LIGHT VERTICAL AND LEFT
  vL U+2525 -- BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
  Vl U+2528 -- BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
  VL U+252B -- BOX DRAWINGS HEAVY VERTICAL AND LEFT
  dh U+252C -- BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
  dH U+252F -- BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
  Dh U+2530 -- BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
  DH U+2533 -- BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
  uh U+2534 -- BOX DRAWINGS LIGHT UP AND HORIZONTAL
  uH U+2537 -- BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
  Uh U+2538 -- BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
  UH U+253B -- BOX DRAWINGS HEAVY UP AND HORIZONTAL
  vh U+253C -- BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
  vH U+253F -- BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
  Vh U+2542 -- BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
  VH U+254B -- BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
  FD U+2571 -- BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT
  BD U+2572 -- BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT
  TB U+2580 -- UPPER HALF BLOCK
  LB U+2584 -- LOWER HALF BLOCK
  FB U+2588 -- FULL BLOCK
  lB U+258C -- LEFT HALF BLOCK
  RB U+2590 -- RIGHT HALF BLOCK
  .S U+2591 -- LIGHT SHADE
  :S U+2592 -- MEDIUM SHADE
  ?S U+2593 -- DARK SHADE
  fS U+25A0 -- BLACK SQUARE
  OS U+25A1 -- WHITE SQUARE
  RO U+25A2 -- WHITE SQUARE WITH ROUNDED CORNERS
  Rr U+25A3 -- WHITE SQUARE CONTAINING BLACK SMALL SQUARE
  RF U+25A4 -- SQUARE WITH HORIZONTAL FILL
  RY U+25A5 -- SQUARE WITH VERTICAL FILL
  RH U+25A6 -- SQUARE WITH ORTHOGONAL CROSSHATCH FILL
  RZ U+25A7 -- SQUARE WITH UPPER LEFT TO LOWER RIGHT FILL
  RK U+25A8 -- SQUARE WITH UPPER RIGHT TO LOWER LEFT FILL
  RX U+25A9 -- SQUARE WITH DIAGONAL CROSSHATCH FILL
  sB U+25AA -- BLACK SMALL SQUARE
  SR U+25AC -- BLACK RECTANGLE
  Or U+25AD -- WHITE RECTANGLE
  UT U+25B2 -- BLACK UP-POINTING TRIANGLE
  uT U+25B3 -- WHITE UP-POINTING TRIANGLE
  PR U+25B6 -- BLACK RIGHT-POINTING TRIANGLE
  Tr U+25B7 -- WHITE RIGHT-POINTING TRIANGLE
  Dt U+25BC -- BLACK DOWN-POINTING TRIANGLE
  dT U+25BD -- WHITE DOWN-POINTING TRIANGLE
  PL U+25C0 -- BLACK LEFT-POINTING TRIANGLE
  Tl U+25C1 -- WHITE LEFT-POINTING TRIANGLE
  Db U+25C6 -- BLACK DIAMOND
  Dw U+25C7 -- WHITE DIAMOND
  LZ U+25CA -- LOZENGE
  0m U+25CB -- WHITE CIRCLE
  0o U+25CE -- BULLSEYE
  0M U+25CF -- BLACK CIRCLE
  0L U+25D0 -- CIRCLE WITH LEFT HALF BLACK
  0R U+25D1 -- CIRCLE WITH RIGHT HALF BLACK
  Sn U+25D8 -- INVERSE BULLET
  Ic U+25D9 -- INVERSE WHITE CIRCLE
  Fd U+25E2 -- BLACK LOWER RIGHT TRIANGLE
  Bd U+25E3 -- BLACK LOWER LEFT TRIANGLE
  *2 U+2605 -- BLACK STAR
  *1 U+2606 -- WHITE STAR
  <H U+261C -- WHITE LEFT POINTING INDEX
  >H U+261E -- WHITE RIGHT POINTING INDEX
  0u U+263A -- WHITE SMILING FACE
  0U U+263B -- BLACK SMILING FACE
  SU U+263C -- WHITE SUN WITH RAYS
  Fm U+2640 -- FEMALE SIGN
  Ml U+2642 -- MALE SIGN
  cS U+2660 -- BLACK SPADE SUIT
  cH U+2661 -- WHITE HEART SUIT
  cD U+2662 -- WHITE DIAMOND SUIT
  cC U+2663 -- BLACK CLUB SUIT
  Md U+2669 -- QUARTER NOTE `
  M8 U+266A -- EIGHTH NOTE `
  M2 U+266B -- BEAMED EIGHTH NOTES
  Mb U+266D -- MUSIC FLAT SIGN
  Mx U+266E -- MUSIC NATURAL SIGN
  MX U+266F -- MUSIC SHARP SIGN
  OK U+2713 -- CHECK MARK
  XX U+2717 -- BALLOT X
  -X U+2720 -- MALTESE CROSS
  IS U+3000 -- IDEOGRAPHIC SPACE
  ,_ U+3001 -- IDEOGRAPHIC COMMA
  ._ U+3002 -- IDEOGRAPHIC FULL STOP
  +" U+3003 -- DITTO MARK
  +_ U+3004 -- JAPANESE INDUSTRIAL STANDARD SYMBOL
  *_ U+3005 -- IDEOGRAPHIC ITERATION MARK
  ;_ U+3006 -- IDEOGRAPHIC CLOSING MARK
  0_ U+3007 -- IDEOGRAPHIC NUMBER ZERO
  <+ U+300A -- LEFT DOUBLE ANGLE BRACKET
  >+ U+300B -- RIGHT DOUBLE ANGLE BRACKET
  <' U+300C -- LEFT CORNER BRACKET
  >' U+300D -- RIGHT CORNER BRACKET
  <" U+300E -- LEFT WHITE CORNER BRACKET
  >" U+300F -- RIGHT WHITE CORNER BRACKET
  (" U+3010 -- LEFT BLACK LENTICULAR BRACKET
  )" U+3011 -- RIGHT BLACK LENTICULAR BRACKET
  =T U+3012 -- POSTAL MARK
  =_ U+3013 -- GETA MARK
  (' U+3014 -- LEFT TORTOISE SHELL BRACKET
  )' U+3015 -- RIGHT TORTOISE SHELL BRACKET
  (I U+3016 -- LEFT WHITE LENTICULAR BRACKET
  )I U+3017 -- RIGHT WHITE LENTICULAR BRACKET
  -? U+301C -- WAVE DASH
  A5 U+3041 -- HIRAGANA LETTER SMALL A
  a5 U+3042 -- HIRAGANA LETTER A
  I5 U+3043 -- HIRAGANA LETTER SMALL I
  i5 U+3044 -- HIRAGANA LETTER I
  U5 U+3045 -- HIRAGANA LETTER SMALL U
  u5 U+3046 -- HIRAGANA LETTER U
  E5 U+3047 -- HIRAGANA LETTER SMALL E
  e5 U+3048 -- HIRAGANA LETTER E
  O5 U+3049 -- HIRAGANA LETTER SMALL O
  o5 U+304A -- HIRAGANA LETTER O
  ka U+304B -- HIRAGANA LETTER KA
  ga U+304C -- HIRAGANA LETTER GA
  ki U+304D -- HIRAGANA LETTER KI
  gi U+304E -- HIRAGANA LETTER GI
  ku U+304F -- HIRAGANA LETTER KU
  gu U+3050 -- HIRAGANA LETTER GU
  ke U+3051 -- HIRAGANA LETTER KE
  ge U+3052 -- HIRAGANA LETTER GE
  ko U+3053 -- HIRAGANA LETTER KO
  go U+3054 -- HIRAGANA LETTER GO
  sa U+3055 -- HIRAGANA LETTER SA
  za U+3056 -- HIRAGANA LETTER ZA
  si U+3057 -- HIRAGANA LETTER SI
  zi U+3058 -- HIRAGANA LETTER ZI
  su U+3059 -- HIRAGANA LETTER SU
  zu U+305A -- HIRAGANA LETTER ZU
  se U+305B -- HIRAGANA LETTER SE
  ze U+305C -- HIRAGANA LETTER ZE
  so U+305D -- HIRAGANA LETTER SO
  zo U+305E -- HIRAGANA LETTER ZO
  ta U+305F -- HIRAGANA LETTER TA
  da U+3060 -- HIRAGANA LETTER DA
  ti U+3061 -- HIRAGANA LETTER TI
  di U+3062 -- HIRAGANA LETTER DI
  tU U+3063 -- HIRAGANA LETTER SMALL TU
  tu U+3064 -- HIRAGANA LETTER TU
  du U+3065 -- HIRAGANA LETTER DU
  te U+3066 -- HIRAGANA LETTER TE
  de U+3067 -- HIRAGANA LETTER DE
  to U+3068 -- HIRAGANA LETTER TO
  do U+3069 -- HIRAGANA LETTER DO
  na U+306A -- HIRAGANA LETTER NA
  ni U+306B -- HIRAGANA LETTER NI
  nu U+306C -- HIRAGANA LETTER NU
  ne U+306D -- HIRAGANA LETTER NE
  no U+306E -- HIRAGANA LETTER NO
  ha U+306F -- HIRAGANA LETTER HA
  ba U+3070 -- HIRAGANA LETTER BA
  pa U+3071 -- HIRAGANA LETTER PA
  hi U+3072 -- HIRAGANA LETTER HI
  bi U+3073 -- HIRAGANA LETTER BI
  pi U+3074 -- HIRAGANA LETTER PI
  hu U+3075 -- HIRAGANA LETTER HU
  bu U+3076 -- HIRAGANA LETTER BU
  pu U+3077 -- HIRAGANA LETTER PU
  he U+3078 -- HIRAGANA LETTER HE
  be U+3079 -- HIRAGANA LETTER BE
  pe U+307A -- HIRAGANA LETTER PE
  ho U+307B -- HIRAGANA LETTER HO
  bo U+307C -- HIRAGANA LETTER BO
  po U+307D -- HIRAGANA LETTER PO
  ma U+307E -- HIRAGANA LETTER MA
  mi U+307F -- HIRAGANA LETTER MI
  mu U+3080 -- HIRAGANA LETTER MU
  me U+3081 -- HIRAGANA LETTER ME
  mo U+3082 -- HIRAGANA LETTER MO
  yA U+3083 -- HIRAGANA LETTER SMALL YA
  ya U+3084 -- HIRAGANA LETTER YA
  yU U+3085 -- HIRAGANA LETTER SMALL YU
  yu U+3086 -- HIRAGANA LETTER YU
  yO U+3087 -- HIRAGANA LETTER SMALL YO
  yo U+3088 -- HIRAGANA LETTER YO
  ra U+3089 -- HIRAGANA LETTER RA
  ri U+308A -- HIRAGANA LETTER RI
  ru U+308B -- HIRAGANA LETTER RU
  re U+308C -- HIRAGANA LETTER RE
  ro U+308D -- HIRAGANA LETTER RO
  wA U+308E -- HIRAGANA LETTER SMALL WA
  wa U+308F -- HIRAGANA LETTER WA
  wi U+3090 -- HIRAGANA LETTER WI
  we U+3091 -- HIRAGANA LETTER WE
  wo U+3092 -- HIRAGANA LETTER WO
  n5 U+3093 -- HIRAGANA LETTER N `
  vu U+3094 -- HIRAGANA LETTER VU
  "5 U+309B -- KATAKANA-HIRAGANA VOICED SOUND MARK
  05 U+309C -- KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
  *5 U+309D -- HIRAGANA ITERATION MARK
  +5 U+309E -- HIRAGANA VOICED ITERATION MARK
  a6 U+30A1 -- KATAKANA LETTER SMALL A
  A6 U+30A2 -- KATAKANA LETTER A
  i6 U+30A3 -- KATAKANA LETTER SMALL I
  I6 U+30A4 -- KATAKANA LETTER I
  u6 U+30A5 -- KATAKANA LETTER SMALL U
  U6 U+30A6 -- KATAKANA LETTER U
  e6 U+30A7 -- KATAKANA LETTER SMALL E
  E6 U+30A8 -- KATAKANA LETTER E
  o6 U+30A9 -- KATAKANA LETTER SMALL O
  O6 U+30AA -- KATAKANA LETTER O
  Ka U+30AB -- KATAKANA LETTER KA
  Ga U+30AC -- KATAKANA LETTER GA
  Ki U+30AD -- KATAKANA LETTER KI
  Gi U+30AE -- KATAKANA LETTER GI
  Ku U+30AF -- KATAKANA LETTER KU
  Gu U+30B0 -- KATAKANA LETTER GU
  Ke U+30B1 -- KATAKANA LETTER KE
  Ge U+30B2 -- KATAKANA LETTER GE
  Ko U+30B3 -- KATAKANA LETTER KO
  Go U+30B4 -- KATAKANA LETTER GO
  Sa U+30B5 -- KATAKANA LETTER SA
  Za U+30B6 -- KATAKANA LETTER ZA
  Si U+30B7 -- KATAKANA LETTER SI
  Zi U+30B8 -- KATAKANA LETTER ZI
  Su U+30B9 -- KATAKANA LETTER SU
  Zu U+30BA -- KATAKANA LETTER ZU
  Se U+30BB -- KATAKANA LETTER SE
  Ze U+30BC -- KATAKANA LETTER ZE
  So U+30BD -- KATAKANA LETTER SO
  Zo U+30BE -- KATAKANA LETTER ZO
  Ta U+30BF -- KATAKANA LETTER TA
  Da U+30C0 -- KATAKANA LETTER DA
  Ti U+30C1 -- KATAKANA LETTER TI
  Di U+30C2 -- KATAKANA LETTER DI
  TU U+30C3 -- KATAKANA LETTER SMALL TU
  Tu U+30C4 -- KATAKANA LETTER TU
  Du U+30C5 -- KATAKANA LETTER DU
  Te U+30C6 -- KATAKANA LETTER TE
  De U+30C7 -- KATAKANA LETTER DE
  To U+30C8 -- KATAKANA LETTER TO
  Do U+30C9 -- KATAKANA LETTER DO
  Na U+30CA -- KATAKANA LETTER NA
  Ni U+30CB -- KATAKANA LETTER NI
  Nu U+30CC -- KATAKANA LETTER NU
  Ne U+30CD -- KATAKANA LETTER NE
  No U+30CE -- KATAKANA LETTER NO
  Ha U+30CF -- KATAKANA LETTER HA
  Ba U+30D0 -- KATAKANA LETTER BA
  Pa U+30D1 -- KATAKANA LETTER PA
  Hi U+30D2 -- KATAKANA LETTER HI
  Bi U+30D3 -- KATAKANA LETTER BI
  Pi U+30D4 -- KATAKANA LETTER PI
  Hu U+30D5 -- KATAKANA LETTER HU
  Bu U+30D6 -- KATAKANA LETTER BU
  Pu U+30D7 -- KATAKANA LETTER PU
  He U+30D8 -- KATAKANA LETTER HE
  Be U+30D9 -- KATAKANA LETTER BE
  Pe U+30DA -- KATAKANA LETTER PE
  Ho U+30DB -- KATAKANA LETTER HO
  Bo U+30DC -- KATAKANA LETTER BO
  Po U+30DD -- KATAKANA LETTER PO
  Ma U+30DE -- KATAKANA LETTER MA
  Mi U+30DF -- KATAKANA LETTER MI
  Mu U+30E0 -- KATAKANA LETTER MU
  Me U+30E1 -- KATAKANA LETTER ME
  Mo U+30E2 -- KATAKANA LETTER MO
  YA U+30E3 -- KATAKANA LETTER SMALL YA
  Ya U+30E4 -- KATAKANA LETTER YA
  YU U+30E5 -- KATAKANA LETTER SMALL YU
  Yu U+30E6 -- KATAKANA LETTER YU
  YO U+30E7 -- KATAKANA LETTER SMALL YO
  Yo U+30E8 -- KATAKANA LETTER YO
  Ra U+30E9 -- KATAKANA LETTER RA
  Ri U+30EA -- KATAKANA LETTER RI
  Ru U+30EB -- KATAKANA LETTER RU
  Re U+30EC -- KATAKANA LETTER RE
  Ro U+30ED -- KATAKANA LETTER RO
  WA U+30EE -- KATAKANA LETTER SMALL WA
  Wa U+30EF -- KATAKANA LETTER WA
  Wi U+30F0 -- KATAKANA LETTER WI
  We U+30F1 -- KATAKANA LETTER WE
  Wo U+30F2 -- KATAKANA LETTER WO
  N6 U+30F3 -- KATAKANA LETTER N `
  Vu U+30F4 -- KATAKANA LETTER VU
  KA U+30F5 -- KATAKANA LETTER SMALL KA
  KE U+30F6 -- KATAKANA LETTER SMALL KE
  Va U+30F7 -- KATAKANA LETTER VA
  Vi U+30F8 -- KATAKANA LETTER VI
  Ve U+30F9 -- KATAKANA LETTER VE
  Vo U+30FA -- KATAKANA LETTER VO
  .6 U+30FB -- KATAKANA MIDDLE DOT
  -6 U+30FC -- KATAKANA-HIRAGANA PROLONGED SOUND MARK
  *6 U+30FD -- KATAKANA ITERATION MARK
  +6 U+30FE -- KATAKANA VOICED ITERATION MARK
  b4 U+3105 -- BOPOMOFO LETTER B
  p4 U+3106 -- BOPOMOFO LETTER P
  m4 U+3107 -- BOPOMOFO LETTER M
  f4 U+3108 -- BOPOMOFO LETTER F
  d4 U+3109 -- BOPOMOFO LETTER D
  t4 U+310A -- BOPOMOFO LETTER T
  n4 U+310B -- BOPOMOFO LETTER N `
  l4 U+310C -- BOPOMOFO LETTER L
  g4 U+310D -- BOPOMOFO LETTER G
  k4 U+310E -- BOPOMOFO LETTER K
  h4 U+310F -- BOPOMOFO LETTER H
  j4 U+3110 -- BOPOMOFO LETTER J
  q4 U+3111 -- BOPOMOFO LETTER Q
  x4 U+3112 -- BOPOMOFO LETTER X
  zh U+3113 -- BOPOMOFO LETTER ZH
  ch U+3114 -- BOPOMOFO LETTER CH
  sh U+3115 -- BOPOMOFO LETTER SH
  r4 U+3116 -- BOPOMOFO LETTER R
  z4 U+3117 -- BOPOMOFO LETTER Z
  c4 U+3118 -- BOPOMOFO LETTER C
  s4 U+3119 -- BOPOMOFO LETTER S
  a4 U+311A -- BOPOMOFO LETTER A
  o4 U+311B -- BOPOMOFO LETTER O
  e4 U+311C -- BOPOMOFO LETTER E
  ai U+311E -- BOPOMOFO LETTER AI
  ei U+311F -- BOPOMOFO LETTER EI
  au U+3120 -- BOPOMOFO LETTER AU
  ou U+3121 -- BOPOMOFO LETTER OU
  an U+3122 -- BOPOMOFO LETTER AN
  en U+3123 -- BOPOMOFO LETTER EN
  aN U+3124 -- BOPOMOFO LETTER ANG
  eN U+3125 -- BOPOMOFO LETTER ENG
  er U+3126 -- BOPOMOFO LETTER ER
  i4 U+3127 -- BOPOMOFO LETTER I
  u4 U+3128 -- BOPOMOFO LETTER U
  iu U+3129 -- BOPOMOFO LETTER IU
  v4 U+312A -- BOPOMOFO LETTER V
  nG U+312B -- BOPOMOFO LETTER NG
  gn U+312C -- BOPOMOFO LETTER GN
  1c U+3220 -- PARENTHESIZED IDEOGRAPH ONE
  2c U+3221 -- PARENTHESIZED IDEOGRAPH TWO
  3c U+3222 -- PARENTHESIZED IDEOGRAPH THREE
  4c U+3223 -- PARENTHESIZED IDEOGRAPH FOUR
  5c U+3224 -- PARENTHESIZED IDEOGRAPH FIVE
  6c U+3225 -- PARENTHESIZED IDEOGRAPH SIX
  7c U+3226 -- PARENTHESIZED IDEOGRAPH SEVEN
  8c U+3227 -- PARENTHESIZED IDEOGRAPH EIGHT
  9c U+3228 -- PARENTHESIZED IDEOGRAPH NINE
  ff U+FB00 -- LATIN SMALL LIGATURE FF
  fi U+FB01 -- LATIN SMALL LIGATURE FI
  fl U+FB02 -- LATIN SMALL LIGATURE FL
  ft U+FB05 -- LATIN SMALL LIGATURE LONG S T
  st U+FB06 -- LATIN SMALL LIGATURE ST
  |]
