/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */

package org.apache.fop.render.intermediate.extensions;

import java.util.Collections;
import java.util.List;

/**
 * This class is a bookmark element for use in the intermediate format.
 */
public class Bookmark {

    private String title;
    private boolean show;
    private List childBookmarks;
    private AbstractAction action;

    /**
     * Creates a new bookmark.
     * @param title the bookmark's title
     * @param show true if the bookmark shall be shown, false for hidden
     * @param action the action performed when the bookmark is clicked
     */
    public Bookmark(String title, boolean show, AbstractAction action) {
        this.title = title;
        this.show = show;
        this.action = action;
    }

    /**
     * Returns the bookmark's title.
     * @return the title
     */
    public String getTitle() {
        return this.title;
    }

    /**
     * Indicates whether the bookmark shall be shown initially.
     * @return true if it shall be shown
     */
    public boolean isShown() {
        return this.show;
    }

    /**
     * Returns the action performed when the bookmark is clicked.
     * @return the action
     */
    public AbstractAction getAction() {
        return this.action;
    }

    /**
     * Sets the action performed when the bookmark is clicked.
     * @param action the action
     */
    public void setAction(AbstractAction action) {
        this.action = action;
    }

    /**
     * Adds a child bookmark.
     * @param bookmark the child bookmark
     */
    public void addChildBookmark(Bookmark bookmark) {
        if (this.childBookmarks == null) {
            this.childBookmarks = new java.util.ArrayList();
        }
        this.childBookmarks.add(bookmark);
    }

    /**
     * Returns a list of child bookmarks.
     * @return the child bookmarks
     */
    public List getChildBookmarks() {
        if (this.childBookmarks == null) {
            return Collections.EMPTY_LIST;
        } else {
            return Collections.unmodifiableList(this.childBookmarks);
        }
    }

}
