/* $Id: WakeAt.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * WakeAt: Tell the scheduler to wake the current process at a given 
 *         simulation time.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __WAKE_AT_HPP_INCLUDED
#define __WAKE_AT_HPP_INCLUDED

#include <cassert>

namespace intermediate {

//! register that the current process should be resumed at a given time.
/** This class will tell the scheduler, to resume the current process 
 *  at a given time. The scheduler may however also resume the current
 *  process earlier, in case a signal registered via WakeOn had an 
 *  event.
 *  In case the process resumes due to an event, the timeout is also
 *  cleared, and must eventually get reset.
 *  Issueing more than one WakeAt before a call to Suspend the current
 *  process leads to undefined behaviour.
 *
 *  read operands: wakeTime
 *  write operands: no explicit write operands
 *
 *  Operation: sched->setTimeOut(wakeTime);
 */
class WakeAt : public OpCode {
public:
	//! c'tor
	/** @param t simulation time value, at which the process should
	 *         resume.
	 */
	WakeAt(Operand *t) : wakeTime(t) {
		assert(t->type == OP_TYPE_INTEGER);
	}

	//! Accept a Visitor.
 	/** All intermediate code nodes need to implement this method.
         *
         *  @param v the Visitor that can visit this node.
         */
	virtual void accept(Visitor& v) {
		v.visit(*this);
	}

	//! scheduled resume time.
	Operand *wakeTime;

protected:
	virtual ~WakeAt() {
		util::MiscUtil::terminate(this->wakeTime);
	}
};

}; /* namespace intermediate */

#endif /* __WAKE_AT_HPP_INCLUDED */
