/* $Id: Subscript.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SUBSCRIPT_HPP_INCLUDED
#define __SUBSCRIPT_HPP_INCLUDED

#include <list>
#include "frontend/ast/Expression.hpp"

namespace ast {

//! subscript operation on an array.
/** Array subscription by an index, which returns the element.
 *  arraytype(index {, index} ) -> element of arraytype.
 */
class Subscript : public Expression {
public:
	//! c'tor
	/** @param sourceArrayType an Expression which returns an array.
	 *  @param idc list of indices.
	 *  @param loc Location of the Subscript operation.
	 */
	Subscript(
		Expression *sourceArrayType,
		std::list<Expression*> *idc,
		Location loc
		) : 	Expression(loc),
			source(sourceArrayType),
			indices(idc) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		stream << this->source << '(';
		util::MiscUtil::listPut(this->indices, stream, ", ");
		stream << ')';
	}


	//! source of the subscript.
	Expression *source;

	//! list of indices
	std::list<Expression*> *indices;
	
protected:
	/** Destructor */
	virtual ~Subscript() {
		util::MiscUtil::terminate(this->source);
		util::MiscUtil::lterminate(this->indices);
	}
};

}; /* namespace ast */

#endif /* __SUBSCRIPT_HPP_INCLUDED */
