C Copyright 1981-2012 ECMWF.
C
C This software is licensed under the terms of the Apache Licence 
C Version 2.0 which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
C
C In applying this licence, ECMWF does not waive the privileges and immunities 
C granted to it by virtue of its status as an intergovernmental organisation 
C nor does it submit to any jurisdiction.
C

      INTEGER FUNCTION WAVEXX1(NPARAM, ISEC2, ISEC4, NLATS, STEPNS,
     X                        STEPWE, OLDWAVE, NEWIDX, DISTNEW,
     X                         NEWWAVE, NORTH, WEST, PMISS)
C
C---->
C*****WAVEXX1*
C
C     PURPOSE
C     -------
C
C     Interpolates wave fields (except 2D spectra).
C
C
C     INTERFACE
C     ---------
C
C     IRET = WAVEXX1(ISEC2, ISEC4, NLATS, STEPNS, STEPWE,
C    X               OLDWAVE, NEWIDX, DISTNEW,
C    X               NEWWAVE, NORTH, WEST, PMISS)
C
C     Input arguments
C     ---------------
C
C     ISEC2   - Unpacked section 2 header from 2D wave spectra field
C     ISEC4   - Unpacked section 4 header from 2D wave spectra field
C     NLATS   - Number of points N-S in new grid
C     STEPNS  - Output grid north-south resolution (degrees)
C     STEPWE  - Output grid west-east resolution (degrees)
C     OLDWAVE - Original wave field
C     NORTH   - Output grid northernmost latitude (degrees)
C     WEST    - Output grid westernmost longitude (degrees)
C     PMISS   - Missing data value 
C
C     Output arguments
C     ----------------
C
C     NEWIDX  - Indices of four neighbouring old grid points used for
C               new grid points
C     DISTNEW - Distances of surrounding box edges from new grid points
C     NEWWAVE - New wave field
C
C     Function returns 0 if the interpolation was OK.
C
C
C     METHOD
C     ------
C
C     Builds the index of neighbouring points for the output grid.
C     Then works through the output grid points, checking for subarea
C     boundaries and looking up neighbouring point values and weights.
C     (Neighbours may have missing data).
C
C
C     EXTERNALS
C     ---------
C
C     WAVEIDX - determines which nearest-neighbour values to use for
C               interpolating to new output grid point
C     NUMPTWE - Calculates number of grid points between west/east
C               area boundaries
C     VALPINA - Looks up bitmap value
C     SOFFSET - Finds the section offsets in GRIB message
C     INTLOG  - Log error message
C
C
C     REFERENCE
C     ---------
C
C     None.
C
C
C     Author.
C     -------
C
C     J.D.Chambers      ECMWF    January 1998
C
C
C     Modifications
C     -------------
C     Handle reduced lat/long grid in 'pseudo-gaussian' layout
C     J.D.Chambers      ECMWF    November 2003
C
C
C----<
C
      IMPLICIT NONE
C
C     Parameters
C
      INTEGER JPROUTINE, JPMXLAT
      PARAMETER (JPROUTINE = 40100)
      PARAMETER (JPMXLAT = 1801)
      INTEGER JPNW, JPNE, JPSW, JPSE, JPN, JPS
      INTEGER JPDISNW, JPDISNE, JPDISSW, JPDISSE
      PARAMETER (JPNW = 1)
      PARAMETER (JPNE = 2)
      PARAMETER (JPSW = 3)
      PARAMETER (JPSE = 4)
      PARAMETER (JPN  = 5)
      PARAMETER (JPS  = 6)
      PARAMETER (JPDISNW = 7)
      PARAMETER (JPDISNE = 8)
      PARAMETER (JPDISSW = 9)
      PARAMETER (JPDISSE = 10)
C
C     Subroutine arguments
C
      INTEGER ISEC2, ISEC4, NLATS
      DIMENSION ISEC2(*), ISEC4(*)
      REAL STEPNS, STEPWE, OLDWAVE, NEWWAVE, NORTH, WEST, PMISS
      REAL ONORTH
      DIMENSION OLDWAVE(*), NEWWAVE(*)
      INTEGER NEWIDX
      DIMENSION NEWIDX(4,*)
      REAL*4 DISTNEW
      DIMENSION DISTNEW(10,*)
C
C     Local arguments
C
      INTEGER IEOFSET, IWOFSET, ISTART, IWEST
      REAL SOUTH, PTLAT, AWEST, EAST, PTLONG, OLDWEST, OLDEAST
      REAL REFVAL, SCALE
      INTEGER NSPT1, INSPT, NEXP, NMANT, NRETA, NRETB, NLENG, NBIT
      INTEGER NPARAM, ISCALE, NSCALE, IBITS, ITEMP
      INTEGER IRET, IS0, IS1, IS2, IS3, IS4, IEDITN
      INTEGER NEXT, NEXTWV, NROW, NCOL, INDEX, ILOCATE
      INTEGER MISSLAT, LOOP, NPTS, NUMLATS, NUMNEW, KOLDNUM, KNEWNUM
      DIMENSION NPTS(JPMXLAT), NUMNEW(JPMXLAT)
      REAL RLATINC, OLDLATS, NEWLATS, ROWINC
      DIMENSION OLDLATS(JPMXLAT), NEWLATS(JPMXLAT)
      LOGICAL LDIREC
      REAL*4 DISNW, DISNE, DISSW, DISSE
      REAL*4 NW_PT, NE_PT, SW_PT, SE_PT
      REAL*4 RAD
      DATA RAD/0.017453293/
      REAL*4 DI1N, DI1S, DI2N, DI2S, DK1, DK2
      REAL*4 CNW_PT, CNE_PT, CSW_PT, CSE_PT
      REAL*4 SNW_PT, SNE_PT, SSW_PT, SSE_PT
      REAL*4 U1, U2, C1, C2, S1, S2, CC, SS
      INTEGER INW, INE, ISW, ISE, JNW, JNE, JSW, JSE
C
C     Externals
C
      INTEGER WAVEIDX, SOFFSET, VALPINA, NUMPTWE
      EXTERNAL WAVEIDX, SOFFSET, VALPINA, NUMPTWE
C
#include "parim.h"
#include "nifld.common"
#include "nofld.common"
#include "grspace.h"
C
C ---------------------------------------------------------------------
C*    Section 1. Initalisation.
C ---------------------------------------------------------------------
C
  100 CONTINUE
C
      WAVEXX1 = 0

C
C     Initialise the bitmap value lookup function
C
      IRET = VALPINA(0,0,0)
C
      MISSLAT = 0
C
C     Calculate number of latitudes if grid had been full from
C     North pole to South pole
C
      NUMLATS = (180000/ISEC2(10)) + 1
      IF( NUMLATS.GT.JPMXLAT ) THEN
        CALL INTLOG(JP_ERROR,
     X    'WAVEXX1: Number of latitudes in input grid = ',NUMLATS)
        CALL INTLOG(JP_ERROR,
     X    'WAVEXX1: And is greater than allowed maximum = ',JPMXLAT)
        WAVEXX1 = JPROUTINE + 1
        GOTO 900
      ENDIF
C
cs      RLATINC = FLOAT(ISEC2(10))/1000
      RLATINC = FLOAT(ISEC2(10))
C
C     Fill an array with the number of points at each latitude for the
C     input field.
C
      IF( ISEC2(17).EQ.1 ) THEN
C
C       Input field is a reduced latitude/longitude grid
C
C       .. but it may be 'pseudo-gaussian' in layout
C       (ie global, symmetric about the equator but no latitude
C        at the equator)
C
        IF( (ISEC2(4).NE.90000).AND.(MOD(ISEC2(3),2).EQ.0) ) THEN
C
          NUMLATS = ISEC2(3)
cs          RLATINC = FLOAT(ISEC2(10))/1000
cs          ONORTH = FLOAT(ISEC2(4))/1000
          ONORTH = FLOAT(ISEC2(4))
C
          DO LOOP = 1, NUMLATS
cs            OLDLATS(LOOP) = ONORTH - (LOOP-1)*RLATINC
            OLDLATS(LOOP) = (ONORTH - (LOOP-1)*RLATINC)/1000
          ENDDO
C
          DO LOOP = 1, ISEC2(3)
            NPTS(LOOP+MISSLAT) = ISEC2(22+LOOP)
          ENDDO
C
        ELSE
C
          DO LOOP = 1, NUMLATS
            OLDLATS(LOOP) = (90000.0 - (LOOP-1)*RLATINC)/1000
cs            OLDLATS(LOOP) = 90.0 - (LOOP-1)*RLATINC
          ENDDO
C
          MISSLAT = (90000 - ISEC2(4))/ISEC2(10)
          DO LOOP = 1, MISSLAT
            NPTS(LOOP)    = 0
          ENDDO
          KOLDNUM = 1 + (90000 - ISEC2(7))/ISEC2(10)
          DO LOOP = 1, (KOLDNUM-MISSLAT)
            NPTS(LOOP+MISSLAT) = ISEC2(22+LOOP)
          ENDDO
          DO LOOP = (KOLDNUM+1), NUMLATS
            NPTS(LOOP)    = 0
          ENDDO
        ENDIF
C
      ELSE
C
C       Input field is a regular latitude/longitude grid
C
C
        DO LOOP = 1, NUMLATS
cs          OLDLATS(LOOP) = 90.0 - (LOOP-1)*RLATINC
            OLDLATS(LOOP) = (90000.0 - (LOOP-1)*RLATINC)/1000
        ENDDO
C
        MISSLAT = (90000 - ISEC2(4))/ISEC2(10)
        DO LOOP = 1, MISSLAT
          NPTS(LOOP)    = 0
        ENDDO
        KOLDNUM = 1 + (90000 - ISEC2(7))/ISEC2(10)
        DO LOOP = 1, (KOLDNUM-MISSLAT)
          NPTS(LOOP+MISSLAT) = ISEC2(2)
        ENDDO
        DO LOOP = (KOLDNUM+1), NUMLATS
          NPTS(LOOP)    = 0
        ENDDO
      ENDIF
C
C ---------------------------------------------------------------------
C*    Section 2. Setup number of points at each latitude for the
C                output latitude/longitude field.
C ---------------------------------------------------------------------
C
  200 CONTINUE
C
      IF( (NOREPR.EQ.JPQUASI) .OR. (NOREPR.EQ.JPGAUSSIAN) ) THEN
C
C       Reduced (quasi-regular) gaussian output
C
        KNEWNUM = NOGAUSS*2
        DO LOOP = 1, KNEWNUM
          NUMNEW(LOOP)  = NOLPTS(LOOP)
          NEWLATS(LOOP) = ROGAUSS(LOOP)
        ENDDO
C
      ELSE IF( NOREPR.EQ.JPREDLL ) THEN
C
C       Reduced (quasi-regular) lat/long output
C
        KNEWNUM = NOREDLL
        DO LOOP = 1, KNEWNUM
          NUMNEW(LOOP)  = NOLPTS(LOOP)
          NEWLATS(LOOP) = ROREDLL(LOOP)
        ENDDO
C
      ELSE
C
C       Regular output
C
        MISSLAT = NINT((90.0 - NORTH)/STEPNS)
        DO LOOP = 1, MISSLAT
          NUMNEW(LOOP)    = 0
        ENDDO
        DO LOOP = 1, NLATS
          NUMNEW(LOOP+MISSLAT) = NINT(360.0/STEPWE)
        ENDDO
C
        KNEWNUM = MISSLAT + NLATS
        DO LOOP = 1, KNEWNUM
          NEWLATS(LOOP) = 90.0 - (LOOP-1)*STEPNS
        ENDDO
C
      ENDIF
C
C ---------------------------------------------------------------------
C*    Section 3. Get the input GRIB field characteristics.
C ---------------------------------------------------------------------
C
  300 CONTINUE
C
C     Calculate the indices of the input grid points to be used for
C     the output points
C
      OLDWEST = FLOAT(NIAREA(2))/PPMULT
      OLDEAST = FLOAT(NIAREA(4))/PPMULT
cs      print*,'MARS NUMLATS ', NUMLATS,' RLATINC ',RLATINC
cs      print*,'MARS OLDEAST ', OLDEAST,' OLDWEST ',OLDWEST
cs      print*,'MARS KNEWNUM ', KNEWNUM,' STEPNS ',STEPNS

cs21    format(i6)
cs      OPEN (UNIT=12,FILE='./mars_offset.txt',STATUS='UNKNOWN',
cs     &    FORM='FORMATTED')
cs      WRITE (12,21)(NPTS(LOOP), LOOP=1,NUMLATS)

      WAVEXX1 = WAVEIDX(NUMLATS,NPTS,OLDLATS,OLDWEST,OLDEAST,
     X                  KNEWNUM, NUMNEW, NEWLATS,
     X                  NEWIDX, DISTNEW)
      IF( WAVEXX1.NE.0 ) THEN
        CALL INTLOG(JP_ERROR,
     X    'WAVEXX1: Unable to calculate output grid indices',JPQUIET)
        WAVEXX1 = JPROUTINE + 2
        GOTO 900
      ENDIF
C
C     Find bitmap (section 3) offset the GRIB product
C
cs      IRET = SOFFSET(OLDWAVE, IS0, IS1, IS2, IS3, IS4, IEDITN)
C
C     Get the overall length of OLDWAVE
C
cs#ifdef INTEGER_8
cs      NBIT = 64
cs#else
cs      NBIT = 32
cs#endif
cs      INSPT = (IS0 + 4) * 8
cs      CALL INXBIT(OLDWAVE,10,INSPT,NLENG,1,NBIT,24,'D',NRETA)
cs      IF( NRETA.NE.0 ) THEN
cs        CALL INTLOG(JP_ERROR,
cs     X    'WAVEXX1: Problem extracting overall length of GRIB',JPQUIET)
cs        WAVEXX1 = JPROUTINE + 3
cs        GOTO 900
cs      ENDIF
C
C     Get parameter identifier
C
cs      INSPT = (IS1 + 8) * 8
cs      CALL INXBIT(OLDWAVE,10,INSPT,NPARAM,1,NBIT,8,'D',NRETA)
cs      IF( NRETA.NE.0 ) THEN
cs        CALL INTLOG(JP_ERROR,
cs     X    'WAVEXX1: Problem getting parameter identifier',JPQUIET)
cs        WAVEXX1 = JPROUTINE + 4
cs        GOTO 900
cs      ENDIF
C
C     Wave direction parameters need special handling
C     (MWD, MDWW, MDPS, MDWI)
C
      LDIREC = ( (NPARAM.EQ.230) .OR.
     X           (NPARAM.EQ.235) .OR.
     X           (NPARAM.EQ.238) .OR.
     X           (NPARAM.EQ.242) )
C
C     Get the scale factor
C
cs      INSPT = (IS4 + 4) * 8
cs      CALL INXBIT(OLDWAVE,NLENG,INSPT,NSCALE,1,NBIT,16,'D',NRETA)
cs      IF( NRETA.NE.0 ) THEN
cs        CALL INTLOG(JP_ERROR,
cs     X    'WAVEXX1: Problem extracting scale factor in GRIB',JPQUIET)
cs        WAVEXX1 = JPROUTINE + 5
cs        GOTO 900
cs      ENDIF
cs      CALL DSGNBT(ISCALE,NSCALE,16,NRETA)
cs      IF( NRETA.NE.0 ) THEN
cs        CALL INTLOG(JP_ERROR,
cs     X    'WAVEXX1: Problem setting sign of scale factor',JPQUIET)
cs        WAVEXX1 = JPROUTINE + 6
cs        GOTO 900
cs      ENDIF
cs      SCALE = 2.0**(FLOAT(ISCALE))
C
C     Get the reference value
C
cs      CALL INXBIT(OLDWAVE,NLENG,INSPT,NEXP,1,NBIT,8,'D',NRETA)
cs      CALL INXBIT(OLDWAVE,NLENG,INSPT,NMANT,1,NBIT,24,'D',NRETB)
cs      IF( (NRETA.NE.0) .OR. (NRETB.NE.0) ) THEN
cs        CALL INTLOG(JP_ERROR,
cs     X    'WAVEXX1: Problem getting reference value from GRIB',JPQUIET)
cs        WAVEXX1 = JPROUTINE + 7
cs        GOTO 900
cs      ENDIF
cs      CALL DECFP2(REFVAL,NEXP,NMANT)
C
C     Get number of bits per packed value
C
cs      CALL INXBIT(OLDWAVE,NLENG,INSPT,IBITS,1,NBIT,8,'D',NRETA)
cs      IF( NRETA.NE.0 ) THEN
cs        CALL INTLOG(JP_ERROR,
cs     X    'WAVEXX1: Problem getting number of bits per word',JPQUIET)
cs        WAVEXX1 = JPROUTINE + 8
cs        GOTO 900
cs      ENDIF
cs      NSPT1 = INSPT
C
C ---------------------------------------------------------------------
C*    Section 4. Work through the output subarea.
C ---------------------------------------------------------------------
C
  400 CONTINUE
C
C     Fill in the wave spectra values
C
      NEXT = 0
      NEXTWV = 0
C
      SOUTH = NOAREA(3)/PPMULT
      EAST  = NOAREA(4)/PPMULT
      ISTART = 0
C
C     Work down through latitudes from north to south.
C

          OPEN (UNIT=16,FILE='./mars.txt',STATUS='UNKNOWN',
     &    FORM='FORMATTED')
c17            format(i4,1x,i4,4(1x,i6))
17            format(i4,1x,i4,4(1x,i6),2x,4(f9.2))
19            format(i4,1x,i4,4(1x,i6),2x,f9.2)

      DO NROW = 1, KNEWNUM
C
C       If inside north-south (subarea) boundaries ..
C
        IF( (NOREPR.EQ.JPGAUSSIAN).OR.(NOREPR.EQ.JPQUASI) ) THEN
          PTLAT = ROGAUSS(NROW)
        ELSE
          PTLAT = 90.0 - (NROW-1)*STEPNS
        ENDIF
        IF( (PTLAT.LE.NORTH).AND.(ABS(PTLAT-SOUTH).GT.-0.0005) ) THEN
C
C         Calculate number of points between west boundary of area and
C         Greenwich
C
          ROWINC = 360.0/NUMNEW(NROW)
C
          IWEST = INT(WEST/ROWINC)
          IF( (WEST.GT.0.0).AND.(WEST.NE.(IWEST*ROWINC)) )
     X      IWEST = IWEST + 1
          AWEST = IWEST * ROWINC
          IWOFSET = NUMPTWE(AWEST,0.0,ROWINC)
          IEOFSET = NUMPTWE(AWEST,EAST,ROWINC)
C
C         Work through subarea longitudes from west to east.
C
          DO NCOL = 1, NUMNEW(NROW)
            PTLONG = AWEST + (NCOL-1)*ROWINC
            NEXT = NUMPTWE(AWEST,PTLONG,ROWINC)
            IF( (NEXT.LE.IEOFSET).AND.(NEXT.GE.0) ) THEN
C
C             .. and inside west-east (subarea) boundaries
C
              NEXT = 1 + NEXT - IWOFSET
              IF( NEXT.LE.0) NEXT = NEXT + NUMNEW(NROW)
              NEXT = NEXT + ISTART
              NEXTWV = NEXTWV + 1
C
              INW = NEWIDX(JPNW,NEXT)
              INE = NEWIDX(JPNE,NEXT)
              ISW = NEWIDX(JPSW,NEXT)
              ISE = NEWIDX(JPSE,NEXT)
C
C             If there is a bitmap, ...
C
cs              IF( IS3.NE.0 ) THEN
cs                IF(INW.NE.0) JNW = VALPINA(OLDWAVE, IS3+6, INW)
cs                IF(INE.NE.0) JNE = VALPINA(OLDWAVE, IS3+6, INE)
cs                IF(ISW.NE.0) JSW = VALPINA(OLDWAVE, IS3+6, ISW)
cs                IF(ISE.NE.0) JSE = VALPINA(OLDWAVE, IS3+6, ISE)
cs              ELSE
cs                JNW = INW
cs                JNE = INE
cs                JSW = ISW
cs                JSE = ISE
cs              ENDIF
C
C             Test if any of the four neighbouring points is missing.
C
              IF( (INW.EQ.0) .OR. (OLDWAVE(INW).EQ.PMISS) .OR.
     X            (ISW.EQ.0) .OR. (OLDWAVE(ISW).EQ.PMISS) .OR.
     X            (INE.EQ.0) .OR. (OLDWAVE(INE).EQ.PMISS) .OR.
     X            (ISE.EQ.0) .OR. (OLDWAVE(ISE).EQ.PMISS) ) THEN
cs
cs              WRITE (16,17)NROW,NCOL,INW,INE,ISW,ISE
cs     X           ,OLDWAVE(INW),OLDWAVE(ISW)
cs     X           ,OLDWAVE(INE),OLDWAVE(ISE)
C
C               If so, take nearest grid point value.
C
                DISNW = DISTNEW(JPDISNW,NEXT)
                DISNE = DISTNEW(JPDISNE,NEXT)
                DISSW = DISTNEW(JPDISSW,NEXT)
                DISSE = DISTNEW(JPDISSE,NEXT)
C
                IF( (DISNW.LE.DISNE).AND.
     X              (DISNW.LE.DISSW).AND.
     X              (DISNW.LE.DISSE)) THEN
                  INDEX = INW
cs                  ILOCATE  = JNW
C
                ELSE IF( (DISNE.LE.DISSW).AND.
     X                   (DISNE.LE.DISSE) ) THEN
                  INDEX = INE
cs                  ILOCATE  = JNE
C
                ELSE IF( (DISSW.LE.DISSE) ) THEN
                  INDEX = ISW
cs                  ILOCATE  = JSW
C
                ELSE
                  INDEX = ISE
cs                  ILOCATE  = JSE
                ENDIF
C
                IF(INDEX.EQ.0.OR.(OLDWAVE(INDEX).EQ.PMISS)) THEN
C
C                 Nearest point is missing
C
                  NEWWAVE(NEXTWV) = PMISS
C
                ELSE
cs                  INSPT = NSPT1 + (ILOCATE - 1)*IBITS
cs                  CALL INXBIT(OLDWAVE,NLENG,INSPT,ITEMP,1,
cs     X                        NBIT,IBITS,'D',NRETA)
cs                  IF( NRETA.NE.0 ) THEN
ccss                    CALL INTLOG(JP_ERROR,
cs     X              'WAVEXX1: Problem getting nearest neighbour value',
cs     X              JPQUIET)
cs                    WAVEXX1 = JPROUTINE + 9
cs                  GOTO 900
cs                  ENDIF
cs                  NEWWAVE(NEXTWV) = REFVAL + FLOAT(ITEMP)*SCALE
                NEWWAVE(NEXTWV) =  OLDWAVE(INDEX)
              WRITE (16,19)NROW,NCOL,INW,INE,ISW,ISE,NEWWAVE(NEXTWV)
                ENDIF
C
              ELSE
C
C               Use bi-linear interpolation from four
C               neighbouring sea points.
C
cs                INSPT = NSPT1 + (JNW - 1)*IBITS
cs                CALL INXBIT(OLDWAVE,NLENG,INSPT,ITEMP,1,
cs     X                      NBIT,IBITS,'D',NRETA)
cs                IF( NRETA.NE.0 ) THEN
cs                  CALL INTLOG(JP_ERROR,
cs     X            'WAVEXX1: Problem getting NW neighbour value',
cs     X            JPQUIET)
cs                  WAVEXX1 = JPROUTINE + 10
cs                  GOTO 900
cs                ENDIF
cs                NW_PT = REFVAL + FLOAT(ITEMP)*SCALE
C
cs                INSPT = NSPT1 + (JNE - 1)*IBITS
cs                CALL INXBIT(OLDWAVE,NLENG,INSPT,ITEMP,1,
cs     X                      NBIT,IBITS,'D',NRETA)
cs                IF( NRETA.NE.0 ) THEN
cs                  CALL INTLOG(JP_ERROR,
cs     X            'WAVEXX1: Problem getting NE neighbour value',
cs     X            JPQUIET)
cs                  WAVEXX1 = JPROUTINE + 10
cs                  GOTO 900
cs                ENDIF
cs                NE_PT = REFVAL + FLOAT(ITEMP)*SCALE
C
cs                INSPT = NSPT1 + (JSW - 1)*IBITS
cs                CALL INXBIT(OLDWAVE,NLENG,INSPT,ITEMP,1,
cs     X                      NBIT,IBITS,'D',NRETA)
cs                IF( NRETA.NE.0 ) THEN
cs                  CALL INTLOG(JP_ERROR,
cs     X            'WAVEXX1: Problem getting SW neighbour value',
cs     X            JPQUIET)
cs                  WAVEXX1 = JPROUTINE + 10
cs                  GOTO 900
cs                ENDIF
cs                SW_PT = REFVAL + FLOAT(ITEMP)*SCALE
C
cs                INSPT = NSPT1 + (JSE - 1)*IBITS
cs                CALL INXBIT(OLDWAVE,NLENG,INSPT,ITEMP,1,
cs     X                      NBIT,IBITS,'D',NRETA)
cs                IF( NRETA.NE.0 ) THEN
cs                  CALL INTLOG(JP_ERROR,
cs     X            'WAVEXX1: Problem getting SE neighbour value',
cs     X            JPQUIET)
cs                  WAVEXX1 = JPROUTINE + 10
cs                  GOTO 900
cs                ENDIF
cs                SE_PT = REFVAL + FLOAT(ITEMP)*SCALE
C
                DI1N = DISTNEW(JPNW,NEXT)
                DI2N = DISTNEW(JPNE,NEXT)
                DI1S = DISTNEW(JPSW,NEXT)
                DI2S = DISTNEW(JPSE,NEXT)
                DK1  = DISTNEW(JPN,NEXT)
                DK2  = DISTNEW(JPS,NEXT)
C
                IF( .NOT. LDIREC ) THEN
                  U1 = NW_PT*DI2N + NE_PT*DI1N
                  U2 = SW_PT*DI2S + SE_PT*DI1S
                  NEWWAVE(NEXTWV) = U1*DK2 + U2*DK1
                ELSE
C
C                 Fields representing a 'direction': resolve into
C                 components and interpolate each separately.
C
                  CNW_PT = COS(NW_PT*RAD)
                  CNE_PT = COS(NE_PT*RAD)
                  CSW_PT = COS(SW_PT*RAD)
                  CSE_PT = COS(SE_PT*RAD)
                  SNW_PT = SIN(NW_PT*RAD)
                  SNE_PT = SIN(NE_PT*RAD)
                  SSW_PT = SIN(SW_PT*RAD)
                  SSE_PT = SIN(SE_PT*RAD)
                  C1 = CNW_PT*DI2N + CNE_PT*DI1N
                  C2 = CSW_PT*DI2S + CSE_PT*DI1S
                  CC = C1*DK2 + C2*DK1
                  S1 = SNW_PT*DI2N + SNE_PT*DI1N
                  S2 = SSW_PT*DI2S + SSE_PT*DI1S
                  SS = S1*DK2 + S2*DK1
                  IF( SS.LT.0.0 ) THEN
                    NEWWAVE(NEXTWV) = ATAN2(SS,CC)/RAD + 360.0
                  ELSE
                    NEWWAVE(NEXTWV) = ATAN2(SS,CC)/RAD
                  ENDIF
                ENDIF
              ENDIF
            ENDIF
          ENDDO
C
        ENDIF
        ISTART = ISTART + NUMNEW(NROW)
      ENDDO
C
C ---------------------------------------------------------------------
C*    Section 9. Closedown.
C ---------------------------------------------------------------------
C
  900 CONTINUE
      RETURN
      END
