/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *
 * License: GPL-2+
 *
 * Initial version: 2010-02-12
 *
 * CNModel runner (interpreter)
 */


#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <cassert>
#include <regex.h>
#include <list>
#include <initializer_list>

#include "config.h"

#if defined(HAVE_READLINE_READLINE_H)
#  include <readline/readline.h>
#elif defined(HAVE_READLINE_H)
#  include <readline.h>
#endif /* !defined(HAVE_READLINE_H) */

#if defined(HAVE_READLINE_HISTORY_H)
#  include <readline/history.h>
#elif defined(HAVE_HISTORY_H)
#  include <history.h>
#  endif

#include "runner.hh"
#include "libstilton/exprparser.hh"
#include "libcn/integrate-rk65.hh"
#include "libcn/base-unit.hh"

using namespace std;


const char* const cnrun_cmd[] = {
	"new_model",
	"load_nml",
	"merge_nml",
	"add_neuron",
	"add_synapse",
	"reset",
	"reset_revert_params",
	"reset_state_units",
	"advance_until",
	"advance",
	"putout",
	"decimate",
	"start_listen",
	"stop_listen",
	"listen_dt",
	"listen_mode",
	"integration_dt_min",
	"integration_dt_max",
	"integration_dt_cap",
	"start_log_spikes",
	"stop_log_spikes",
	"sxf_params",
	"new_source",
	"show_sources",
	"connect_source",
	"disconnect_source",
	"set_parm_neuron",
	"set_parm_synapse",
	"cull_deaf_synapses",
	"describe_model",
	"show_units",
	"exec",
	"verbosity",
	"show_vars",
	"clear_vars",
	"pause",
	"quit",
	nullptr
};



list<SVariable> *current_shell_variables;



namespace {

void
LOG( const char *fname, int lineno, int vrb, const char* fmt, ...)
{
	va_list ap;
	va_start (ap, fmt);

	char *buf1 = const_cast<char*>(""), *buf2;
	if ( lineno > 0 )
		assert (asprintf( &buf1, "%s:%d: ", fname, lineno) > 0);
	assert (vasprintf( &buf2, fmt, ap) > 0);
	va_end (ap);

	Log->msg( vrb, "CNrun", "%s%s", buf1, buf2);

	if ( lineno > 0 )
		free( buf1);
	free( buf2);
}

int do_single_cmd( const char*,
		   list<SVariable> &varlist,
		   int level = 0, const char *fname = "",
		   unsigned lineno = -1);


#define CN_INTERP_EXIT		 1
#define CN_INTERP_WARN		-1
#define CN_INTERP_PARSEERROR	-2
#define CN_INTERP_SYSFAIL	-3

#define CNRUN_HISTORY ".cnrun-history"




int
new_model( const char *model_name, const char *fname, unsigned lineno)
{
	if ( !(Model = new CModel( model_name,
				   new CIntegrateRK65( Options.integration_dt_min,
						       Options.integration_dt_max,
						       Options.integration_dt_max_cap),
				   0 |
				   (Options.log_dt ? CN_MDL_LOGDT : 0) |
				   (Options.log_spikers ? CN_MDL_LOGSPIKERS : 0) |
				   (Options.sort_units ? CN_MDL_SORTUNITS : 0) |
				   (Options.log_spikers_use_serial_id ? CN_MDL_LOGUSINGID : 0) |
				   (Options.display_progress_percent ? CN_MDL_DISPLAY_PROGRESS_PERCENT : 0) |
				   (Options.display_progress_time ? CN_MDL_DISPLAY_PROGRESS_TIME : 0) |
				   (Options.dont_coalesce ? CN_MDL_DONT_COALESCE : 0))) ) {
		LOG( fname, lineno, -1, "Failed to create a model");
		return CN_INTERP_SYSFAIL;
	}

	Model->verbosely = Options.verbosely;
	Model->listen_dt = Options.listen_dt;
	Model->spike_threshold = Options.spike_threshold /*,  Model->spike_lapse = Options.spike_lapse */;
	Log->msg( 3, nullptr,
		  "generator type: %s\n"
		  "         seed = %lu\n"
		  "  first value = %lu\n",
		  gsl_rng_name (Model->_rng), gsl_rng_default_seed, gsl_rng_get (Model->_rng));

	return 0;
}






int
do_single_cmd( const char* raw,
	       list<SVariable> &varlist,
	       int level, const char *fname, unsigned lineno)
{
	string	raw_s( raw);
	char	*cmd = strtok( &raw_s[0], " \t"),
		*operand = strtok( nullptr, "\n");

	CExpression expr;
	double result;

#define CHECK_MODEL \
	if ( !Model ) {							\
		LOG( fname, lineno, -1, "No model loaded");		\
		return CN_INTERP_WARN;					\
	}


	if ( strcmp( cmd, cnrun_cmd[CNCMD_new_model]) == 0 ) {
		if ( !operand ) {
			LOG( fname, lineno, -1, "Missing a name for the new model");
			return CN_INTERP_PARSEERROR;
		}
		delete Model;

		regenerate_unit_labels = true;
		return new_model( operand, fname, lineno);

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_load_nml]) == 0 ) {
		struct stat s;
		if ( stat( operand, &s) ) {
			LOG( fname, lineno, -1, "No such file: \"%s\"", operand);
			return CN_INTERP_SYSFAIL;
		}

		int retval = new_model( operand, fname, lineno);
		if ( retval )
			return retval;

		if ( Model->import_NetworkML( operand, false) < 0 ) {
			LOG( fname, lineno, -1, "Failed to create model topology from \"%s\"", operand);
			delete Model;
			Model = nullptr;
			return CN_INTERP_SYSFAIL;
		}

		Model -> cull_blind_synapses();
		regenerate_unit_labels = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_merge_nml]) == 0 ) {
		CHECK_MODEL;
		struct stat s;
		if ( stat( operand, &s) ) {
			LOG( fname, lineno, -1, "No such file: \"%s\"", operand);
			return CN_INTERP_SYSFAIL;
		}
		if ( Model->import_NetworkML( operand, true) < 0 ) {
			LOG( fname, lineno, -1, "Failed to import topology from \"%s\"", operand);
			return CN_INTERP_SYSFAIL;
		}

		regenerate_unit_labels = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_add_neuron]) == 0 ) {
		CHECK_MODEL;
		char *type_s, *label_s;
		if ( !operand ||
		     !(type_s = (strtok( operand, " \t"))) ||
		     !(label_s = strtok( nullptr, "\n")) ) {
			LOG( fname, lineno, -1, "Missing neuron type and/or label in `add_neuron'");
			return CN_INTERP_PARSEERROR;
		}
		if ( !Model->add_neuron_species( type_s, label_s, true) ) {
			LOG( fname, lineno, -1, "`add_neuron' failed");
			return CN_INTERP_PARSEERROR;
		}
		regenerate_unit_labels = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_add_synapse]) == 0 ) {
		CHECK_MODEL;
		char *type_s, *src_s, *tgt_s, *g_s;
		if ( !operand ||
		     !(type_s = (strtok( operand, " \t"))) ||
		     !(src_s = strtok( nullptr, " \t")) ||
		     !(tgt_s = strtok( nullptr, " \t")) ||
		     !(g_s = strtok( nullptr, "\n")) ) {
			LOG( fname, lineno, -1, "Missing synapse type, source or target label, and/or gsyn in `add_synapse'");
			return CN_INTERP_PARSEERROR;
		}
		double g;
		if ( expr( g_s, g, &varlist) ) {
			LOG( fname, lineno, -1, "Bad value for gsyn in `add_synapse'");
			return CN_INTERP_PARSEERROR;
		}

		if ( !Model->add_synapse_species( type_s, src_s, tgt_s, g, true, true) ) {
			LOG( fname, lineno, -1, "`add_synapse' failed (reason given above)", operand);
			return CN_INTERP_SYSFAIL;
		}
		regenerate_unit_labels = true;



	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_reset]) == 0 ) {
		CHECK_MODEL;
		Model->reset();
		Log->msg( 0, nullptr, "Reset model and state all units");

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_reset_revert_params]) == 0 ) {
		CHECK_MODEL;
		Model->reset( true);
		Log->msg( 0, nullptr, "Reset model and reverted all units' state and parameters");

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_reset_state_units]) == 0 ) {
		CHECK_MODEL;
		if ( !operand )
			operand = const_cast<char*>(".*");
		regex_t RE;
		if ( 0 != regcomp( &RE, operand, REG_EXTENDED | REG_NOSUB) ) {
			LOG( fname, lineno, -1, "Invalid regexp for `reset_state_units' arg");
			return CN_INTERP_PARSEERROR;
		}
		size_t cnt = 0;
		for_model_units (Model,U)
			if ( regexec( &RE, (*U)->label(), 0, 0, 0) == 0 ) {
				(*U) -> reset_state();
				++cnt;
			}
		if ( cnt )
			Log->msg( 0, nullptr, "Reset %zd unit(s)", cnt);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_advance_until]) == 0 ) {
		CHECK_MODEL;
		expr.silent = true;
		if ( !operand || expr( operand, result, &varlist) ) {
			LOG( fname, lineno, -1, "No or bad time value for `advance_until'");
			return CN_INTERP_PARSEERROR;
		}
		if ( Model->model_time() > result ) {
			LOG( fname, lineno, 0, "Cannot go back in time (now is %g)", Model->model_time());
			return CN_INTERP_WARN;
		}

		Model -> advance( result - Model->model_time());
		for_model_spikelogging_neurons (Model,N)
			(*N)->sync_spikelogging_history();


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_advance]) == 0 ) {
		CHECK_MODEL;
		if ( !operand || expr( operand, result, &varlist) ) {
			LOG( fname, lineno, -1, "No or bad time value for `advance'");
			return CN_INTERP_PARSEERROR;
		}

		Model -> advance( result);
		for_model_spikelogging_neurons (Model,N)
			(*N)->sync_spikelogging_history();


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_putout]) == 0 ) {
		CHECK_MODEL;
		char *label_s;
		if ( !operand ||
		     !(label_s = (strtok( operand, " \t"))) ) {
			LOG( fname, lineno, -1, "Missing label in `putout'");
			return CN_INTERP_PARSEERROR;
		}

		list<CModel::STagGroup> tags;
		tags.push_back( CModel::STagGroup (label_s));
		Model->process_putout_tags( tags);

		regenerate_unit_labels = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_decimate]) == 0 ) {
		CHECK_MODEL;
		char *label_s, *frac_s;
		if ( !operand ||
		     !(label_s = (strtok( operand, " \t"))) ||
		     !(frac_s = (strtok( nullptr, "\n"))) ) {
			LOG( fname, lineno, -1, "Missing fraction or label in `decimate'");
			return CN_INTERP_PARSEERROR;
		}
		if ( expr( frac_s, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable expression for decimation fraction: \"%s\"", operand);
			return CN_INTERP_PARSEERROR;
		}
		if ( result < 0. || result > 1. ) {
			LOG( fname, lineno, -1, "Decimation fraction outside [0..1]");
			return CN_INTERP_PARSEERROR;
		}

		list<CModel::STagGroupDecimate> tags;
		tags.push_back( CModel::STagGroupDecimate( label_s, result));
		Model -> process_decimate_tags( tags);

		regenerate_unit_labels = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_start_listen]) == 0 ) {
		CHECK_MODEL;
		if ( !operand ||
		     !(operand = (strtok( operand, " \t")) ) ) {
			LOG( fname, lineno, -1, "Missing label in `start_listen'");
			return CN_INTERP_PARSEERROR;
		}
		list<CModel::STagGroupListener> tags;
		tags.push_back( CModel::STagGroupListener (operand, true, 0
					      | (Options.listen_1varonly ? CN_ULISTENING_1VARONLY : 0)
					      | (Options.listen_deferwrite ? CN_ULISTENING_DEFERWRITE : 0)
					      | (Options.listen_binary ? CN_ULISTENING_BINARY : CN_ULISTENING_DISK)));
		Model->process_listener_tags( tags);

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_stop_listen]) == 0 ) {
		CHECK_MODEL;
		if ( !operand ||
		     !(operand = (strtok( operand, " \t"))) ) {
			LOG( fname, lineno, -1, "Missing label in `stop_listen'");
			return CN_INTERP_PARSEERROR;
		}
		list<CModel::STagGroupListener> tags;
		tags.push_back( CModel::STagGroupListener (operand, false));
		Model->process_listener_tags( tags);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_listen_dt]) == 0 ) {
		if ( !operand ) {
			Log->msg( 0, nullptr, "listen_dt is %g", Options.listen_dt);
			return 0;
		}
		if ( expr( operand, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable expression for value in `listen_dt'");
			return CN_INTERP_PARSEERROR;
		}
		if ( Model )
			Model->listen_dt = Options.listen_dt = result;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_listen_mode]) == 0 ) {
		if ( !operand )
			Log->msg( 0, nullptr, "listen mode is 1%sd%sb%s (%s%s%s)",
				  Options.listen_1varonly ? "+" : "",
				  Options.listen_deferwrite ? "+" : "",
				  Options.listen_binary ? "+" : "",
				  Options.listen_1varonly ? "one var-only, " : "all vars, ",
				  Options.listen_deferwrite ? "deferred write, " : "continuous write, ",
				  Options.listen_binary ? "binary" : "ascii");
		else {
			char *c;
			if ( (c = strchr( operand, '1')) ) Options.listen_1varonly   = (*(c+1) != '-');
			if ( (c = strchr( operand, 'd')) ) Options.listen_deferwrite = (*(c+1) != '-');
			if ( (c = strchr( operand, 'b')) ) Options.listen_binary     = (*(c+1) != '-');
		}

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_integration_dt_min]) == 0 ) {
		if ( !operand ) {
			Log->msg( 0, nullptr, "integration_dt_min is %g", Options.integration_dt_min);
			return 0;
		}
		if ( expr( operand, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable expression for value in `integration_dt_min'");
			return CN_INTERP_PARSEERROR;
		}
		Options.integration_dt_min = result;
		if ( Model )
			Model->dt_min() = result;

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_integration_dt_max]) == 0 ) {
		if ( !operand ) {
			Log->msg( 0, nullptr, "integration_dt_max is %g", Options.integration_dt_max);
			return 0;
		}
		if ( expr( operand, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable expression for value in `integration_dt_max'");
			return CN_INTERP_PARSEERROR;
		}
		Options.integration_dt_max = result;
		if ( Model )
			Model->dt_max() = result;

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_integration_dt_cap]) == 0 ) {
		if ( !operand ) {
			Log->msg( 0, nullptr, "integration_dt_cap is %g", Options.integration_dt_max_cap);
			return 0;
		}
		if ( expr( operand, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable expression for value in `integration_dt_cap'");
			return CN_INTERP_PARSEERROR;
		}
		Options.integration_dt_max_cap = result;
		if ( Model )
			(static_cast<CIntegrateRK65*>(Model->_integrator)) -> _dt_max_cap = result;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_set_sxf_params]) == 0 ) {
		if ( !operand ) {
			Log->msg( 0, nullptr, "sxf_start_delay:sxf_period:sdf_sigma is %g:%g:%g",
				  Options.sxf_start_delay, Options.sxf_sample, Options.sdf_sigma);
			return 0;
		}
		if ( sscanf( operand, "%g:%g:%g",
			     &Options.sxf_start_delay, &Options.sxf_sample, &Options.sdf_sigma) < 3 ) {
			LOG( fname, lineno, -1, "Expecting <double>:<double>:<double> with set_sxf_params");
			return CN_INTERP_PARSEERROR;
		}


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_start_log_spikes]) == 0 ) {
		CHECK_MODEL;
		char *label_s;
		if ( !operand ||
		     !(label_s = (strtok( operand, " \t"))) ) {
			LOG( fname, lineno, -1, "Missing label in `start_log_spikes'");
			return CN_INTERP_PARSEERROR;
		}
		if ( Options.sxf_sample <= 0. || Options.sdf_sigma <= 0. ) {
			LOG( fname, lineno, 1, "SDF parameters not set up, will only log spike times");
		}
		list<CModel::STagGroupSpikelogger> specs;
		specs.push_back( CModel::STagGroupSpikelogger (label_s, true,
							       Options.sxf_sample, Options.sdf_sigma, Options.sxf_start_delay));
		Model->process_spikelogger_tags( specs);

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_stop_log_spikes]) == 0 ) {
		CHECK_MODEL;
		char *label_s;
		if ( !operand ||
		     !(label_s = (strtok( operand, " \t"))) ) {
			LOG( fname, lineno, -1, "Missing label in `stop_log_spikes'");
			return CN_INTERP_PARSEERROR;
		}
		list<CModel::STagGroupSpikelogger> specs;
		specs.push_back( CModel::STagGroupSpikelogger (label_s, false));
		Model->process_spikelogger_tags( specs);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_set_parm_neuron]) == 0 ) {
		CHECK_MODEL;
		char *label_s, *parm_s, *value_s;
		if ( !operand ||
		     !(label_s = (strtok( operand, " \t"))) ||
		     !(parm_s = strtok( nullptr, " \t")) ||
		     !(value_s = strtok( nullptr, "\n")) ) {
			LOG( fname, lineno, -1, "Missing label, parameter or value in `set_parm_neuron'");
			return CN_INTERP_PARSEERROR;
		}
		if ( expr( value_s, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable expression for value in `set_parm_neuron'");
			return CN_INTERP_PARSEERROR;
		}
		list<CModel::STagGroupNeuronParmSet> specs = { CModel::STagGroupNeuronParmSet (label_s, true, parm_s, result) };
		Model->process_paramset_static_tags( specs);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_set_parm_synapse]) == 0 ) {
		CHECK_MODEL;
		char *src_s, *tgt_s, *parm_s, *value_s;
		if ( !operand ||
		     !(src_s = (strtok( operand, " \t"))) ||
		     !(tgt_s = (strtok( nullptr, " \t"))) ||
		     !(parm_s = strtok( nullptr, " \t")) ||
		     !(value_s = strtok( nullptr, "\n")) ) {
			LOG( fname, lineno, -1, "Missing source or target label, parameter and/or value in `set_parm_synapse'");
			return CN_INTERP_PARSEERROR;
		}
		if ( expr( value_s, result, &varlist) ) {
			LOG( fname, lineno, -1, "Unparsable value in `set_parm_synapse'");
			return CN_INTERP_PARSEERROR;
		}
		list<CModel::STagGroupSynapseParmSet> specs = { CModel::STagGroupSynapseParmSet (src_s, tgt_s, true, parm_s, result) };
		Model->process_paramset_static_tags( specs);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_new_source]) == 0 ) {
		CHECK_MODEL;
		char *type_s, *name_s;
		if ( !operand ||
		     !(type_s = (strtok( operand, " \t"))) ||
		     !(name_s = (strtok( nullptr, " \t"))) ) {
			LOG( fname, lineno, -1, "Missing source type or name in `new_source'");
			return CN_INTERP_PARSEERROR;
		}

		if ( Model->source_by_id( name_s) ) {
			LOG( fname, lineno, -1, "A source named \"%s\" already exists", name_s);
			return CN_INTERP_PARSEERROR;
		}

		char *arg1, *arg2;
		if ( strcmp( type_s, __SourceTypes[SRC_TAPE]) == 0 ) {
			if ( !(arg1 = strtok( nullptr, "\n")) ) {
				LOG( fname, lineno, -1, "Missing filename for a Tape source in `new_source'");
				return CN_INTERP_PARSEERROR;
			} else {
				CSourceTape *source = new CSourceTape( name_s, arg1);
				if ( source && source->name.size() )
					if ( count( Model->Sources.begin(), Model->Sources.end(), source) == 0 )
						Model->Sources.push_back( source);
					else {
						LOG( fname, lineno, -1, "Duplicate name (\"%s\") for a source", arg1);
						return CN_INTERP_SYSFAIL;
					}
				else {
					delete source;
					LOG( fname, lineno, -1, "Failed to set up a Tape source from \"%s\"", arg1);
					return CN_INTERP_SYSFAIL;
				}
			}
		} else if ( strcmp( type_s, __SourceTypes[SRC_PERIODIC]) == 0 ) {
			if ( !(arg1 = strtok( nullptr, "\n")) ) {
				LOG( fname, lineno, -1, "Missing filename for a Periodic source in `new_source'");
				return CN_INTERP_PARSEERROR;
			} else {
				CSourcePeriodic *source = new CSourcePeriodic( name_s, arg1);
				if ( source && source->name.size() )
					if ( count( Model->Sources.begin(), Model->Sources.end(), source) == 0 )
						Model->Sources.push_back( source);
					else {
						LOG( fname, lineno, -1, "Duplicate name (\"%s\") for a source", arg1);
						return CN_INTERP_SYSFAIL;
					}
				else {
					delete source;
					LOG( fname, lineno, -1, "Failed to set up a Periodic source from \"%s\"", arg1);
					return CN_INTERP_SYSFAIL;
				}
			}
		} else if ( strcmp( type_s, __SourceTypes[SRC_NOISE]) == 0 ) {
			if ( !(arg1 = strtok( nullptr, ":")) ||
			     !(arg2 = strtok( nullptr, "\n")) ) {
				LOG( fname, lineno, -1, "Incomplete min:max set for a Noise source in `new_source'");
				return CN_INTERP_PARSEERROR;
			} else {
				double _min, _max;
				if ( expr( arg1, _min, &varlist) ||
				     expr( arg2, _max, &varlist) ) {
					LOG( fname, lineno, -1, "Bad min:max values for a Noise source");
					return CN_INTERP_PARSEERROR;
				}
				CSourceNoise *source = new CSourceNoise( name_s, _min, _max);
				if ( source && source->name.size() ) {
					Model->Sources.push_back( source);
				} else {
					delete source;
					LOG( fname, lineno, -1, "Failed to set up a Noise source");
					return CN_INTERP_SYSFAIL;
				}
			}
		} else if ( strcmp( type_s, __SourceTypes[SRC_FUNCTION]) == 0 ) {
			LOG( fname, lineno, -1, "Go code, Chris!");
			return CN_INTERP_SYSFAIL;
		} else {
			LOG( fname, lineno, -1, "Unrecognised source type in `new_source'");
			return CN_INTERP_PARSEERROR;
		}

		regenerate_source_ids = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_show_sources]) == 0 ) {
		CHECK_MODEL;
		for ( list<C_BaseSource*>::iterator S = Model->Sources.begin(); S != Model->Sources.end(); S++ )
			(*S)->dump();


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_connect_source]) == 0 ) {
		CHECK_MODEL;
		char *label_s, *parm_s, *source_s;
		if ( !operand ||
		     !(source_s = strtok( operand, " \t")) ||
		     !(label_s = strtok( nullptr, " \t")) ||
		     !(parm_s = strtok( nullptr, "\n")) ) {
			LOG( fname, lineno, -1, "Missing source id, unit label and/or parameter in `connect_source'");
			return CN_INTERP_PARSEERROR;
		}
		C_BaseSource *source = Model->source_by_id( source_s);
		if ( !source ) {
			LOG( fname, lineno, -1, "Unknown source \"%s\"", source_s);
			return CN_INTERP_PARSEERROR;
		}

		list<CModel::STagGroupSource> tags;
		tags.push_back( CModel::STagGroupSource (label_s, true, parm_s, source));
		Model->process_paramset_source_tags( tags);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_disconnect_source]) == 0 ) {
		CHECK_MODEL;
		char *label_s, *parm_s, *source_s;
		if ( !operand ||
		     !(label_s = (strtok( operand, " \t"))) ||
		     !(parm_s = strtok( nullptr, " \t")) ||
		     !(source_s = strtok( nullptr, "\n")) ) {
			LOG( fname, lineno, -1, "Missing label, parameter or source in `disconnect_source'");
			return CN_INTERP_PARSEERROR;
		}
		C_BaseSource *source = Model->source_by_id( source_s);
		if ( !source ) {
			LOG( fname, lineno, -1, "Unknown source \"%s\"", source_s);
			return CN_INTERP_PARSEERROR;
		}

		list<CModel::STagGroupSource> specs;
		specs.push_back( CModel::STagGroupSource (label_s, false, parm_s, source));
		Model->process_paramset_source_tags( specs);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_cull_deaf_synapses]) == 0 ) {
	 	CHECK_MODEL;
	 	Model->cull_deaf_synapses();


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_describe_model]) == 0 ) {
		CHECK_MODEL;
		Model->dump_metrics();
		Model->dump_units();
		Model->dump_state();

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_show_units]) == 0 ) {
		CHECK_MODEL;
		if ( !operand )
			operand = const_cast<char*>(".*");

		regex_t RE;
		if ( 0 != regcomp( &RE, operand, REG_EXTENDED | REG_NOSUB) ) {
			LOG( fname, lineno, -1, "Invalid regexp for `show_units' arg");
			return CN_INTERP_PARSEERROR;
		}
		size_t cnt = 0;
		for_model_units (Model,U)
			if ( regexec( &RE, (*U)->label(), 0, 0, 0) == 0 ) {
				(*U) -> dump( true);
				cnt++;
			}
		if ( cnt )
			Log->msg_( 0, nullptr, "------------\n%zd units total\n", cnt);

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_exec]) == 0 ) {
		return interpreter_run( operand, level+1, Options.interp_howstrict,
					true, true, varlist);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_verbosity]) == 0 ) {
		if ( !operand )
			Log->msg( 0, nullptr, "verbosity level is %d", Options.verbosely);
		else if ( sscanf( operand, "%d", &Options.verbosely) < 1 ) {
			LOG( fname, lineno, -1, "Bad value for `verbosity'");
			return CN_INTERP_PARSEERROR;
		}
		if ( Model )
			Model->verbosely = Options.verbosely;
		__cn_verbosely = Options.verbosely;



	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_exit]) == 0 ) {
		delete Model;
		Model = nullptr;
		return CN_INTERP_EXIT;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_show_vars]) == 0 ) {
		if ( !operand )
			operand = const_cast<char*>(".*");
		regex_t RE;
		if ( 0 != regcomp( &RE, operand, REG_EXTENDED | REG_NOSUB) ) {
			LOG( fname, lineno, -1, "Invalid regexp for `show_vars' arg");
			return CN_INTERP_PARSEERROR;
		}
		size_t	cnt = 0;
		size_t	longest_id = 0;
		for ( auto& V : varlist )
			if ( regexec( &RE, V.name, 0, 0, 0) == 0 )
				if ( longest_id < strlen( V.name) )
					longest_id = strlen( V.name);
		for ( auto& V : varlist )
			if ( regexec( &RE, V.name, 0, 0, 0) == 0 ) {
				Log->msg( 0, nullptr, "  %*s = %g", longest_id, V.name, V.value);
				++cnt;
			}
		if ( cnt > 1 )
			Log->msg_( 0, nullptr, "---------- %u variables\n", cnt);


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_clear_vars]) == 0 ) {
		if ( !operand )
			varlist.clear();
		else {
			regex_t RE;
			if ( 0 != regcomp( &RE, operand, REG_EXTENDED | REG_NOSUB) ) {
				LOG( fname, lineno, -1, "Invalid regexp for `clear_vars' arg");
				return CN_INTERP_PARSEERROR;
			}
			for ( list<SVariable>::iterator V = varlist.begin(); V != varlist.end(); V++ )
				if ( regexec( &RE, V->name, 0, 0, 0) == 0 ) {
					varlist.erase( V);
					break;
				}
		}

		regenerate_var_names = true;


	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_pause]) == 0 ) {
		if ( operand ) {
			double s;
			if ( expr( operand, s, &varlist) )
				return CN_INTERP_PARSEERROR;
			if ( s < 0 ) {
				Log->msg( 0, nullptr, "Can't sleep backwards in time");
				return CN_INTERP_WARN;
			}
			printf( "(Paused for %u sec)", (unsigned int)s); fflush(stdin);
			sleep( rintf( s));
			printf( "\n");
		} else {
			printf( "Paused: press Enter ...");
			getchar();
		}

	} else {  // feed it to exprparser
		if ( expr( raw, result, &varlist) ) {
			LOG( fname, lineno, -1, "%s", expr.status_s());
			return CN_INTERP_PARSEERROR;
		}
		if ( expr.toplevel_op != '=' )
			Log->msg( 0, nullptr, "%g", result);

		regenerate_var_names = true;
	}

	return 0;
}


} // inline namespace




int
interpreter_run( const char *script_fname, int level, int howstrict,
		 bool env_import, bool env_export, list<SVariable> &varlist)
{
	int retval = 0;

	list<SVariable> our_variables;
	current_shell_variables = &our_variables;

	if ( env_import ) {
		our_variables.splice( our_variables.begin(), varlist);
//		printf( "imported %zu vars\n", our_variables.size());
	}

	if ( script_fname && strlen(script_fname) > 0 ) {
		ifstream script_stream( script_fname);
		if ( !script_stream.good() ) {
			Log->msg( -1, "CNrun", "Failed to open script file \"%s\"", script_fname);
			return -1;
		}
		Log->msg( 1, nullptr, "execing %s\n", script_fname);

		unsigned lineno = 0;
		string buf;
		while ( getline( script_stream, buf) || script_stream.gcount() ) {
			lineno++;
			if ( buf.size() ) {

				char	*pp = strchr( (char*)buf.c_str(), '#');
				if ( pp )
					buf.resize( pp - buf.c_str());
				size_t	buflen = buf.size();
				while ( strlen(buf.c_str()) && isspace(buf[buflen-1]) )
					buf[--buflen] = '\0';

				char	*_stmt = &buf[0],
					*stmt;
				while ( _stmt - &buf[0] < (int)buflen && (stmt = strtok( _stmt, ";\n")) ) {
					_stmt = _stmt + strlen(_stmt)+1;

					retval = do_single_cmd( stmt, our_variables, level, script_fname, lineno);
					if ( retval == CN_INTERP_EXIT ||
					     (retval < CN_INTERP_WARN && howstrict == CN_INTRP_STRICT) )
						break;
				}
			}
		}
	} else {
		if ( level == 0 ) {
			using_history();
			read_history( CNRUN_HISTORY);
			rl_attempted_completion_function = cnrun_completion;
			rl_bind_key( '\t', rl_complete);
		}
		HISTORY_STATE *the_history_state = history_get_history_state();
		if ( the_history_state && the_history_state->length == 0 )
			printf( "Hit TAB for context completions\n");

		char *buf, prompt[256];
		while ( snprintf( prompt, 255, "%s @%g%.*s ",
				  Model ? Model->name.c_str() : "-",
				  Model ? Model->model_time() : 0,
				  level+1, "]]]]]]]"),
			(buf = readline( prompt)) ) {

			the_history_state = history_get_history_state();
			if ( the_history_state &&
			     (the_history_state->length < 2 ||
			      (the_history_state->length > 1 &&
			       *buf &&
			       strcmp( history_get( the_history_state->length)->line, buf) != 0) ) ) {
				add_history( buf);
			}

			char	*pp = strchr( buf, '#');
			if ( pp )
				*pp = '\0';
			size_t	buflen = strlen( buf);
			while ( buflen && strchr( " \t", buf[buflen-1]) )
				buf[--buflen] = '\0';
			if ( !buflen )
				continue;

			char	*_stmt = buf,  // will hold the pointer to the next statement
				*stmt;
			while ( _stmt - buf < (int)buflen  &&  (stmt = strtok( _stmt, ";\n")) ) {
				_stmt += (strlen(_stmt) + 1);

				retval = do_single_cmd( stmt, our_variables, level, nullptr, -1);
				if ( retval == CN_INTERP_EXIT ) {
					free( buf);
					goto out;
				}
			}
			free( buf);
			if ( level == 0 ) {
//				rl_attempted_completion_function = cnrun_completion;
				rl_bind_key( '\t', rl_complete);
			}
		}
	out:
		if ( level == 0 )
			write_history( CNRUN_HISTORY);
		printf( "\n");
	}

	if ( env_export ) {
	      // update varlist
		varlist.merge( our_variables);
		varlist.sort();
		varlist.unique();
	}
	current_shell_variables = &varlist;

	return retval;
}

