/*
 * Copyright (C) 2014-2021 Brian L. Browning
 *
 * This file is part of Beagle
 *
 * Beagle is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beagle is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package vcf;

import blbutil.Const;
import java.util.stream.IntStream;

/**
 * <p>Class {@code BasicGT} represents genotypes for a list of markers and
 * samples.</p>
 *
 * <p>Instances of class {@code BasicGT} are immutable</p>
 *
 * @author Brian L. Browning {@code <browning@uw.edu>}
 */
public final class BasicGT implements GT {

    private final Samples samples;
    private final Markers markers;
    private final GTRec[] recs;
    private final boolean isRefData;

    /**
     * Returns the genotype index corresponding to the specified unordered
     * alleles.
     * @param a1 the first allele index of an unordered genotype
     * @param a2 the second allele index of an unordered genotype
     * @return the genotype index corresponding to the specified unordered
     * alleles
     * @throws IllegalArgumentException if {@code a1 < 0 || a2 < 0}
     */
    public static int genotype(int a1, int a2) {
        if (a1<=a2) {
            if (a1 < 0) {
                String s = "allele < 0: " + a1 + " " + a2;
                throw new IllegalArgumentException(s);
            }
            return (a2*(a2+1))/2 + a1;
        }
        else {
            if (a2<0) {
                String s = "allele < 0: " + a1 + " " + a2;
                throw new IllegalArgumentException(s);
            }
            return (a1*(a1+1))/2 + a2;
        }
    }

    /**
     * Constructs a {@code BasicGT} instance from the specified data
     * @param samples the list of samples with genotype data
     * @param recs genotype emission probabilities
     * @throws IllegalArgumentException
     * if elements of {@code recs} corresponding to the same chromosome
     * are not contiguous and sorted in chromosome position order
     * @throws IllegalArgumentException if any two elements of
     * {@code recs} correspond to the same genetic marker
     * @throws IllegalArgumentException if
     * {@code (recs[j].samples().equals(samples) == false)} for any {@code j}
     * satisfying {@code (0 <= j && j < recs.length)}
     * @throws NullPointerException if {@code samples == null || recs == null}
     * @throws NullPointerException if {@code (recs[j] == null)} any {@code j}
     * satisfying {@code (0 <= j && j < recs.length)}
     */
    public BasicGT(Samples samples, GTRec[] recs) {
        checkSamples(samples, recs);
        this.markers = markers(recs);
        this.samples = samples;
        this.recs = recs.clone();
        this.isRefData = isRefData(recs);
    }

    /**
     * Constructs a {@code BasicGT} instance from the specified data.
     * @param markers the list of markers with genotype data
     * @param samples the list of samples with genotype data
     * @param recs the genotype data for each marker
     * @throws IllegalArgumentException if
     * {@code (recs[j].marker().equals(markers.marker(j)) == false)} for any
     * {@code j} satisfying {@code (0 <= j && j < recs.length)}
     * @throws IllegalArgumentException if
     * {@code (recs[j].samples().equals(samples) == false)} for any {@code j}
     * satisfying {@code (0 <= j && j < recs.length)}
     * @throws NullPointerException if
     * {@code (markers == null || samples == null || recs == null)}
     * @throws NullPointerException if {@code (recs[j] == null)} any {@code j}
     * satisfying {@code (0 <= j && j < recs.length)}
     */
    public BasicGT(Markers markers, Samples samples, GTRec[] recs) {
        checkMarkersAndSamples(markers, samples, recs);
        this.markers = markers;
        this.samples = samples;
        this.recs = recs.clone();
        this.isRefData = isRefData(recs);
    }

    private static void checkSamples(Samples samples, GTRec[] recs) {
        for (int j=0; j<recs.length; ++j) {
            if (recs[j].samples().equals(samples)==false) {
                throw new IllegalArgumentException("inconsistent samples");
            }
        }
    }

    private static void checkMarkersAndSamples(Markers markers, Samples samples,
            GTRec[] recs) {
        for (int j=0; j<recs.length; ++j) {
            if (recs[j].marker().equals(markers.marker(j))==false) {
                throw new IllegalArgumentException("inconsistent markers");
            }
            if (recs[j].samples().equals(samples)==false) {
                throw new IllegalArgumentException("inconsistent samples");
            }
        }
    }

    private static Markers markers(GTRec[] recs) {
        Marker[] markers = new Marker[recs.length];
        for (int j=0; j<markers.length; ++j) {
            markers[j] = recs[j].marker();
        }
        return Markers.create(markers);
    }

    private static boolean isRefData(GTRec[] recs) {
        boolean isRefData = true;
        for (int j=0; j<recs.length && isRefData==true; ++j) {
            if (recs[j].isPhased()==false) {
                isRefData = false;
            }
        }
        return isRefData;
    }

    @Override
    public boolean isReversed() {
        return false;
    }

    @Override
    public int nMarkers() {
        return recs.length;
    }

    @Override
    public Marker marker(int markerIndex) {
        return markers.marker(markerIndex);
    }

    @Override
    public Markers markers() {
        return markers;
    }

    @Override
    public int nHaps() {
        return 2*samples.size();
    }

    @Override
    public int nSamples() {
        return samples.size();
    }

    @Override
    public Samples samples() {
        return samples;
    }

    @Override
    public boolean isPhased() {
        return isRefData;
    }

    @Override
    public int allele1(int marker, int sample) {
        return recs[marker].allele1(sample);
    }

    @Override
    public int allele2(int marker, int sample) {
        return recs[marker].allele2(sample);
    }

    @Override
    public int allele(int marker, int hap) {
        return recs[marker].get(hap);
    }

    @Override
    public GT restrict(Markers restrictedMarkers, int[] indices) {
        GTRec[] restrictedRecs = IntStream.range(0, indices.length)
                .mapToObj(j -> recs[indices[j]])
                .toArray(GTRec[]::new);
        return new BasicGT(restrictedMarkers, samples, restrictedRecs);
    }

    /**
     * Returns a string representation of {@code this}.  The exact details
     * of the representation are unspecified and subject to change.
     * @return a string representation of {@code this}
     */
    @Override
    public String toString() {
        StringBuilder sb  = new StringBuilder();
        sb.append("[BasicGT: nMarkers=");
        sb.append(nMarkers());
        sb.append(" nSamples=");
        sb.append(nSamples());
        for (GTRec rec : recs) {
            sb.append(Const.nl);
            sb.append(rec);
        }
        sb.append(']');
        return sb.toString();
    }
}
