#
# Copyright 2006 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#
# 
# page_enter, page_display, page_process, page_leave

import unittest
import albatross

from dummyrequest import DummyRequest

class Page:
    def __init__(self, name):
        self.name = name

    def page_enter(self, ctx):
        ctx.app.called.append('enter ' + self.name)

    def page_display(self, ctx):
        ctx.app.called.append('display ' + self.name)

    def page_process(self, ctx):
        ctx.app.called.append('process ' + self.name)
        if ctx.app.action:
            meth, args = ctx.app.action
            ctx.app.action = None
            getattr(ctx, meth)(*args)


    def page_leave(self, ctx):
        ctx.app.called.append('leave ' + self.name)

class TestApp(albatross.SimpleApp):
    def __init__(self):
        albatross.SimpleApp.__init__(self, 
                                     base_url='app.py',
                                     template_path='.',
                                     start_page='pageA',
                                     secret='-=-secret-=-')
        self.vars = {}
        self.ctx = None
        self.called = []
        self.action = None

    def run_with_action(self, req, meth, *args):
        self.action = meth, args
        self.run(req)

    def get_called(self):
        called, self.called = self.called, []
        return called

    def create_context(self):
        self.ctx = albatross.SimpleApp.create_context(self)
        return self.ctx

    def load_session(self, ctx):
        ctx.locals.__dict__.update(self.vars)

    def save_session(self, ctx):
        self.vars = ctx.locals.__dict__


class Case(unittest.TestCase):
    def assertActions(self, app, *actions):
        self.assertEqual(app.get_called(), list(actions))

    def pageclass(self):
        app = TestApp()
        app.register_page('pageA', Page('pageA'))
        app.register_page('pageB', Page('pageB'))
        app.register_page('pageC', Page('pageC'))
        req = DummyRequest()
        app.run(req)
        content = req.get_content()
        self.failIf(content, content)
        self.assertEqual(app.vars['__page__'], 'pageA')
        self.assertEqual(app.get_called(), 
                         ['enter pageA', 'process pageA', 'display pageA'])

        app.run_with_action(req, 'set_page', 'pageB')
        self.assertActions(app, 'process pageA', 'leave pageA', 'enter pageB', 
                                'display pageB')

        app.run_with_action(req, 'push_page', 'pageA')
        self.assertActions(app, 'process pageB', 'enter pageA', 'display pageA')

        app.run_with_action(req, 'pop_page')
        self.assertActions(app, 'process pageA', 'leave pageA', 'display pageB')

        app.run_with_action(req, 'push_page', 'pageA')
        app.run_with_action(req, 'push_page', 'pageC')
        self.assertActions(app, 'process pageB', 'enter pageA', 'display pageA',
                                'process pageA', 'enter pageC', 'display pageC')

        app.run_with_action(req, 'pop_page', 'pageB')
        self.assertActions(app, 'process pageC', 'leave pageC', 'leave pageA',
                                'display pageB')


class Suite(unittest.TestSuite):
    test_list = (
        'pageclass',
    )
    def __init__(self):
        unittest.TestSuite.__init__(self, map(Case, self.test_list))

def suite():
    return Suite()

if __name__ == '__main__':
    unittest.main(defaultTest='suite')
