; Standard Utilities Library
;
; Copyright (C) 2024 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (www.alessandrocoglio.info)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "STD")

(include-book "std/system/maybe-pseudo-event-formp" :dir :system)
(include-book "std/util/define" :dir :system)
(include-book "xdoc/defxdoc-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc defconstrained-recognizer

  :parents (std/util)

  :short "Introduce a constrained recognizer."

  :long

  (xdoc::topstring

   (xdoc::h3 "Introduction")

   (xdoc::p
    "ACL2 support the introduction of
     arbitrary consistency-preserving constrained functions
     via @(tsee encapsulate).
     The @('defconstrained-recognizer') macro abbreviates @(tsee encapsulate)s
     to introduce certain common kinds of constrained recognizers.")

   (xdoc::p
    "This macro currently provides just a few options.
     More will be added as the need for them arises.")

   (xdoc::h3 "General Form")

   (xdoc::codeblock
    "(defconstrained-recognizer name"
    "                           :nonempty ..."
    "  )")

   (xdoc::h3 "Inputs")

   (xdoc::desc
    "@('name')"
    (xdoc::p
     "The name of the recognizer."))

   (xdoc::desc
    "@(':nonempty')"
    (xdoc::p
     "Determines whether the recognizer is constrained be non-empty or not,
      and if so it provides the name of a witness function
      for the non-emptiness of the recognizer.")
    (xdoc::p
     "It must be one of the following:")
    (xdoc::ul
     (xdoc::li
      "A symbol that is not @('nil').
       In this case, the recognizer is constrained to be non-empty,
       by introducing, besides the recognizer,
       also a nullary function, whose name is this input,
       constrained to satisfy the recognizer.")
     (xdoc::li
      "@('nil') (the default).
       In this case, the recognizer is not constrained to be non-empty,
       i.e. it may be empty or non-empty.
       No constrained nullary function is introduced in this case.")))

   (xdoc::h3 "Generated Events")

   (xdoc::p
    "This macro generates an @(tsee encapsulate)
     that introduces the following functions
     with the following constraining theorems.")

   (xdoc::desc
    "@('name')"
    (xdoc::p
     "The recognizer, a unary function.")
    (xdoc::p
     "Its executable counterpart is disabled."))

   (xdoc::desc
    "@('booleanp-of-name')"
    (xdoc::p
     "A rewrite and type prescription rule
      saying that the recognizer returns a boolean:")
    (xdoc::codeblock
     "(defthm booleanp-of-name"
     "  (booleanp (name x))"
     "  :rule-classes (:rewrite :type-prescription))"))

   (xdoc::desc
    "@('nonempty')"
    (xdoc::p
     "A nullary function witnessing the non-emptiness of the recognizer.")
    (xdoc::p
     "This is generated exactly when @(':nonempty') is not @('nil')."))

   (xdoc::desc
    "@('name-of-nonempty')"
    (xdoc::codeblock
     "(defthm name-of-nonempty"
     "  (name (nonempty)))")
    (xdoc::p
     "This is generated exactly when @(':nonempty') is not @('nil')."))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(acl2::defxdoc+ defconstrained-recognizer-implementation
  :parents (deffixer)
  :short "Implementation of @(tsee defconstrained-recognizer)."
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define defconstrained-recognizer-fn (name nonempty)
  :returns (event acl2::maybe-pseudo-event-formp)
  :short "Event generated by @(tsee defconstrained-recognizer)."
  (b* (((unless (symbolp name))
        (raise "The NAME input must be a symbol, ~
                but it is ~x0 instead." name))
       ((unless (symbolp nonempty))
        (raise "The :NONEMPTY input must be a symbol, ~
                but it is ~x0 instead." nonempty))
       (pkg (symbol-package-name name))
       (pkg (if (equal pkg *main-lisp-package-name*) "ACL2" pkg))
       (pkg-witness (pkg-witness pkg))
       (x (intern-in-package-of-symbol "X" pkg-witness))
       (name-sig `((,name *) acl2::=> *))
       (nonempty-sig? (and nonempty (list `((,nonempty) acl2::=> *))))
       (name-def `(local (defun ,name (,x) (declare (ignore ,x)) t)))
       (nonempty-def? (and nonempty (list `(local (defun ,nonempty () nil)))))
       (booleanp-of-name
        `(defthm ,(acl2::packn-pos (list 'booleanp-of- name) pkg-witness)
           (booleanp (,name ,x))
           :rule-classes (:rewrite :type-prescription)))
       (name-of-nonempty?
        (and nonempty
             (list
              `(defthm ,(acl2::packn-pos (list name '-of- nonempty) pkg-witness)
                 (,name (,nonempty)))))))
    `(encapsulate
       ()
       (logic)
       (encapsulate
        (,name-sig
         ,@nonempty-sig?)
        ,name-def
        ,@nonempty-def?
        ,booleanp-of-name
        ,@name-of-nonempty?))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection defconstrained-recognizer-definition
  :short "Definition of the @(tsee defconstrained-recognizer) macro."
  :long (xdoc::topstring-@def "defconstrained-recognizer")
  (defmacro defconstrained-recognizer (name
                                       &key
                                       nonempty)
    `(make-event (defconstrained-recognizer-fn ',name ',nonempty))))
