; Copyright (C) 2022, ForrestHunt, Inc.
; Written by Matt Kaufmann
; License: A 3-clause BSD license.  See the LICENSE file distributed with ACL2.

; This is a trivial example of an abstract stobj with a child stobj fields that
; are hash-tables, one of which contains stobjs.  See absstobj-with-array.lisp
; for an analogous example using array fields instead.

; I say "trivial" above because the only difference between the abstract stobj
; and its foundational stobj is that while both have two fields, they form a
; two-element list in the foundational stobj but they are consed together in
; the abstract stobj.

(in-package "ACL2")

(defstobj st$c fld$c)

(defstobj top$c
  (ht1 :type (hash-table eq nil rational) :initially 0)
  (ht2 :type (hash-table equal nil st$c)))

(defun top$ap (x)
  (declare (xargs :guard t))
  (and (consp x)
       (let ((a (car x)))
         (and (symbol-alistp a)
              (rational-listp (strip-cdrs a))))
       (let ((a (cdr x)))
         (and (alistp a)
              (true-list-listp a)
              (alistp (strip-cdrs a))
              (all-nils (strip-cdrs (strip-cdrs a)))))))

(defun-nx top-corr (x y)
  (and (top$cp x)
       (top$ap y)
       (equal (car x) (car y))
       (equal (cadr x) (cdr y))))

(defun create-top$a ()
  (declare (xargs :guard t))
  (cons nil nil))

(defun lookup1$a (key s)
  (declare (xargs :guard (and (symbolp key)
                              (top$ap s))))
  (let ((pair (hons-assoc-equal key (car s))))
    (if pair (cdr pair) 0)))

(defun update1$a (key val s)
  (declare (xargs :guard (and (symbolp key)
                              (rationalp val)
                              (top$ap s))))
  (cons (acons key val (car s))
        (cdr s)))

(defun lookup2$a (key s)
  (declare (xargs :guard (top$ap s)))
  (let ((pair (hons-assoc-equal key (cdr s))))
    (if pair (cdr pair) (non-exec (create-st$c)))))

(defun update2$a (key st$c s)
  (declare (xargs :guard (top$ap s)
                  :stobjs st$c))
  (cons (car s)
        (acons key (non-exec st$c) (cdr s))))

; The capitalized events below have been generated by first running the
; defabsstobj form below.

(DEFTHM CREATE-TOP{CORRESPONDENCE}
        (TOP-CORR (CREATE-TOP$C) (CREATE-TOP$A))
        :RULE-CLASSES NIL)

(DEFTHM CREATE-TOP{PRESERVED}
        (TOP$AP (CREATE-TOP$A))
        :RULE-CLASSES NIL)

(DEFTHM LOOKUP1{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (SYMBOLP K)
                      (TOP$AP TOP))
                 (EQUAL (HT1-GET K TOP$C)
                        (LOOKUP1$A K TOP)))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE1{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (SYMBOLP K)
                      (RATIONALP V)
                      (TOP$AP TOP))
                 (TOP-CORR (HT1-PUT K V TOP$C)
                           (UPDATE1$A K V TOP)))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE1{PRESERVED}
        (IMPLIES (AND (SYMBOLP K)
                      (RATIONALP V)
                      (TOP$AP TOP))
                 (TOP$AP (UPDATE1$A K V TOP)))
        :RULE-CLASSES NIL)

(DEFTHM LOOKUP2{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP) (TOP$AP TOP))
                 (EQUAL (HT2-GET K TOP$C)
                        (LOOKUP2$A K TOP)))
        :RULE-CLASSES NIL)

(defthm equal-len-1
  (equal (equal (len x) 1)
         (and (consp x)
              (atom (cdr x)))))

(DEFTHM UPDATE2{CORRESPONDENCE}
        (IMPLIES (AND (TOP-CORR TOP$C TOP)
                      (ST$CP ST$C)
                      (TOP$AP TOP))
                 (TOP-CORR (HT2-PUT K ST$C TOP$C)
                           (UPDATE2$A K ST$C TOP)))
        :RULE-CLASSES NIL)

(DEFTHM UPDATE2{PRESERVED}
        (IMPLIES (AND (ST$CP ST$C) (TOP$AP TOP))
                 (TOP$AP (UPDATE2$A K ST$C TOP)))
        :RULE-CLASSES NIL)

(defabsstobj top
  :foundation top$c
  :corr-fn top-corr
  :recognizer (topp :logic top$ap :exec top$cp)
  :creator (create-top :logic create-top$a :exec create-top$c)
  :exports ((lookup1 :logic lookup1$a :exec ht1-get)
            (update1 :logic update1$a :exec ht1-put)
            (lookup2 :logic lookup2$a :exec ht2-get :updater update2)
            (update2 :logic update2$a :exec ht2-put)))
