/*
 * Copyright 2022-2024 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

#include "internal/quic_record_rx.h"
#include "internal/quic_rx_depack.h"
#include "internal/quic_record_tx.h"
#include "internal/quic_ackm.h"
#include "internal/quic_cc.h"
#include "internal/quic_ssl.h"
#include "testutil.h"
#include "quic_record_test_util.h"

static const QUIC_CONN_ID empty_conn_id = {0, {0}};

#define RX_TEST_OP_END                     0 /* end of script */
#define RX_TEST_OP_SET_SCID_LEN            1 /* change SCID length */
#define RX_TEST_OP_SET_INIT_LARGEST_PN     2 /* set initial largest PN */
#define RX_TEST_OP_SET_RX_DCID             3 /* register an RX DCID */
#define RX_TEST_OP_INJECT                  4 /* inject a datagram into demux */
#define RX_TEST_OP_PROVIDE_SECRET          5 /* provide RX secret */
#define RX_TEST_OP_PROVIDE_SECRET_INITIAL  6 /* provide RX secret for initial */
#define RX_TEST_OP_DISCARD_EL              7 /* discard an encryption level */
#define RX_TEST_OP_CHECK_PKT               8 /* read packet, compare to expected */
#define RX_TEST_OP_CHECK_NO_PKT            9 /* check no packet is available to read */
#define RX_TEST_OP_CHECK_KEY_EPOCH        10 /* check key epoch value matches */
#define RX_TEST_OP_KEY_UPDATE_TIMEOUT     11 /* complete key update process */
#define RX_TEST_OP_SET_INIT_KEY_PHASE     12 /* initial Key Phase bit value */
#define RX_TEST_OP_CHECK_PKT_EPOCH        13 /* check read key epoch matches */
#define RX_TEST_OP_ALLOW_1RTT             14 /* allow 1RTT packet processing */

struct rx_test_op {
    unsigned char op;
    unsigned char subop;
    const unsigned char *buf;
    size_t buf_len;
    const QUIC_PKT_HDR *hdr;
    uint32_t enc_level, suite_id;
    QUIC_PN largest_pn;
    const QUIC_CONN_ID *dcid;
    int (*new_qrx)(QUIC_DEMUX **demux, OSSL_QRX **qrx);

    /* For frame checking */
};

#define RX_OP_END \
    { RX_TEST_OP_END }
#define RX_OP_SET_SCID_LEN(scid_len) \
    { RX_TEST_OP_SET_SCID_LEN, 0, NULL, 0, NULL, (scid_len), 0, 0, NULL, NULL },
#define RX_OP_SET_INIT_LARGEST_PN(largest_pn) \
    { RX_TEST_OP_SET_INIT_LARGEST_PN, 0, NULL, 0, NULL, 0, 0, (largest_pn), NULL, NULL },
#define RX_OP_SET_RX_DCID(dcid) \
    { RX_TEST_OP_SET_RX_DCID, 0, NULL, 0, NULL, 0, 0, 0, &(dcid), NULL },
#define RX_OP_INJECT(dgram) \
    { RX_TEST_OP_INJECT, 0, (dgram), sizeof(dgram), NULL, 0, 0, 0, NULL },
#define RX_OP_PROVIDE_SECRET(el, suite, key)                           \
    {                                                               \
        RX_TEST_OP_PROVIDE_SECRET, 0, (key), sizeof(key),             \
        NULL, (el), (suite), 0, NULL, NULL                          \
    },
#define RX_OP_PROVIDE_SECRET_INITIAL(dcid) \
    { RX_TEST_OP_PROVIDE_SECRET_INITIAL, 0, NULL, 0, NULL, 0, 0, 0, &(dcid), NULL },
#define RX_OP_DISCARD_EL(el) \
    { RX_TEST_OP_DISCARD_EL, 0, NULL, 0, NULL, (el), 0, 0, NULL, NULL },
#define RX_OP_CHECK_PKT(expect_hdr, expect_body)                       \
    {                                                               \
        RX_TEST_OP_CHECK_PKT, 0, (expect_body), sizeof(expect_body),  \
        &(expect_hdr), 0, 0, 0, NULL, NULL                          \
    },
#define RX_OP_CHECK_NO_PKT() \
    { RX_TEST_OP_CHECK_NO_PKT, 0, NULL, 0, NULL, 0, 0, 0, NULL, NULL },
#define RX_OP_CHECK_KEY_EPOCH(expected) \
    { RX_TEST_OP_CHECK_KEY_EPOCH, 0, NULL, 0, NULL, 0, 0, (expected), NULL },
#define RX_OP_KEY_UPDATE_TIMEOUT(normal) \
    { RX_TEST_OP_KEY_UPDATE_TIMEOUT, 0, NULL, 0, NULL, (normal), 0, 0, NULL },
#define RX_OP_SET_INIT_KEY_PHASE(kp_bit) \
    { RX_TEST_OP_SET_INIT_KEY_PHASE, 0, NULL, 0, NULL, (kp_bit), 0, 0, NULL },
#define RX_OP_CHECK_PKT_EPOCH(expected) \
    { RX_TEST_OP_CHECK_PKT_EPOCH, 0, NULL, 0, NULL, 0, 0, (expected), NULL },
#define RX_OP_ALLOW_1RTT() \
    { RX_TEST_OP_ALLOW_1RTT, 0, NULL, 0, NULL, 0, 0, 0, NULL },

#define RX_OP_INJECT_N(n)                                          \
    RX_OP_INJECT(rx_script_##n##_in)
#define RX_OP_CHECK_PKT_N(n)                                       \
    RX_OP_CHECK_PKT(rx_script_##n##_expect_hdr, rx_script_##n##_body)

#define RX_OP_INJECT_CHECK(n)                                  \
    RX_OP_INJECT_N(n)                                              \
    RX_OP_CHECK_PKT_N(n)

/* 1. RFC 9001 - A.3 Server Initial */
static const unsigned char rx_script_1_in[] = {
    0xcf, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a,
    0x42, 0x62, 0xb5, 0x00, 0x40, 0x75, 0xc0, 0xd9, 0x5a, 0x48, 0x2c, 0xd0,
    0x99, 0x1c, 0xd2, 0x5b, 0x0a, 0xac, 0x40, 0x6a, 0x58, 0x16, 0xb6, 0x39,
    0x41, 0x00, 0xf3, 0x7a, 0x1c, 0x69, 0x79, 0x75, 0x54, 0x78, 0x0b, 0xb3,
    0x8c, 0xc5, 0xa9, 0x9f, 0x5e, 0xde, 0x4c, 0xf7, 0x3c, 0x3e, 0xc2, 0x49,
    0x3a, 0x18, 0x39, 0xb3, 0xdb, 0xcb, 0xa3, 0xf6, 0xea, 0x46, 0xc5, 0xb7,
    0x68, 0x4d, 0xf3, 0x54, 0x8e, 0x7d, 0xde, 0xb9, 0xc3, 0xbf, 0x9c, 0x73,
    0xcc, 0x3f, 0x3b, 0xde, 0xd7, 0x4b, 0x56, 0x2b, 0xfb, 0x19, 0xfb, 0x84,
    0x02, 0x2f, 0x8e, 0xf4, 0xcd, 0xd9, 0x37, 0x95, 0xd7, 0x7d, 0x06, 0xed,
    0xbb, 0x7a, 0xaf, 0x2f, 0x58, 0x89, 0x18, 0x50, 0xab, 0xbd, 0xca, 0x3d,
    0x20, 0x39, 0x8c, 0x27, 0x64, 0x56, 0xcb, 0xc4, 0x21, 0x58, 0x40, 0x7d,
    0xd0, 0x74, 0xee
};

static const unsigned char rx_script_1_body[] = {
    0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x40, 0x5a, 0x02, 0x00, 0x00,
    0x56, 0x03, 0x03, 0xee, 0xfc, 0xe7, 0xf7, 0xb3, 0x7b, 0xa1, 0xd1, 0x63,
    0x2e, 0x96, 0x67, 0x78, 0x25, 0xdd, 0xf7, 0x39, 0x88, 0xcf, 0xc7, 0x98,
    0x25, 0xdf, 0x56, 0x6d, 0xc5, 0x43, 0x0b, 0x9a, 0x04, 0x5a, 0x12, 0x00,
    0x13, 0x01, 0x00, 0x00, 0x2e, 0x00, 0x33, 0x00, 0x24, 0x00, 0x1d, 0x00,
    0x20, 0x9d, 0x3c, 0x94, 0x0d, 0x89, 0x69, 0x0b, 0x84, 0xd0, 0x8a, 0x60,
    0x99, 0x3c, 0x14, 0x4e, 0xca, 0x68, 0x4d, 0x10, 0x81, 0x28, 0x7c, 0x83,
    0x4d, 0x53, 0x11, 0xbc, 0xf3, 0x2b, 0xb9, 0xda, 0x1a, 0x00, 0x2b, 0x00,
    0x02, 0x03, 0x04
};

static const QUIC_CONN_ID rx_script_1_dcid = {
    8, { 0x83, 0x94, 0xc8, 0xf0, 0x3e, 0x51, 0x57, 0x08 }
};

static const QUIC_PKT_HDR rx_script_1_expect_hdr = {
    QUIC_PKT_TYPE_INITIAL,
    0, 0, 2, 0, 1, 0, 0, 1, { 0, {0} },
    { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } },
    { 0, 1, 0, 0 },
    NULL, 0,
    99, NULL
};

static const struct rx_test_op rx_script_1[] = {
    RX_OP_SET_SCID_LEN(2)
    RX_OP_SET_INIT_LARGEST_PN(0)
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_1_dcid)
    RX_OP_INJECT_CHECK(1)
    RX_OP_CHECK_NO_PKT()
    RX_OP_END
};

/* 2. RFC 9001 - A.5 ChaCha20-Poly1305 Short Header Packet */
#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
static const unsigned char rx_script_2_in[] = {
    0x4c, 0xfe, 0x41, 0x89, 0x65, 0x5e, 0x5c, 0xd5, 0x5c, 0x41, 0xf6, 0x90,
    0x80, 0x57, 0x5d, 0x79, 0x99, 0xc2, 0x5a, 0x5b, 0xfb
};

static const unsigned char rx_script_2_secret[] = {
    0x9a, 0xc3, 0x12, 0xa7, 0xf8, 0x77, 0x46, 0x8e, 0xbe, 0x69, 0x42, 0x27,
    0x48, 0xad, 0x00, 0xa1, 0x54, 0x43, 0xf1, 0x82, 0x03, 0xa0, 0x7d, 0x60,
    0x60, 0xf6, 0x88, 0xf3, 0x0f, 0x21, 0x63, 0x2b
};

static const unsigned char rx_script_2_body[] = {
    0x01
};

static const QUIC_PKT_HDR rx_script_2_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0, 0, 3, 0, 1, 0, 0, 0, {0, {0}}, {0, {0}},
    {0x00, 0xbf, 0xf4, 0x00},
    NULL, 0,
    1, NULL
};

static const struct rx_test_op rx_script_2[] = {
    RX_OP_ALLOW_1RTT()
    RX_OP_SET_INIT_LARGEST_PN(654360560)
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT, QRL_SUITE_CHACHA20POLY1305,
                         rx_script_2_secret)
    RX_OP_INJECT_CHECK(2)
    RX_OP_CHECK_NO_PKT()
    RX_OP_END
};
#endif /* !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305) */

/* 3. Real World - Version Negotiation Response */
static const unsigned char rx_script_3_in[] = {
    0xc7,                               /* Long; Random Bits */
    0x00, 0x00, 0x00, 0x00,             /* Version 0 (Version Negotiation) */
    0x00,                               /* DCID */
    0x0c, 0x35, 0x3c, 0x1b, 0x97, 0xca, /* SCID */
    0xf8, 0x99, 0x11, 0x39, 0xad, 0x79,
    0x1f,
    0x00, 0x00, 0x00, 0x01,             /* Supported Version: 1 */
    0xaa, 0x9a, 0x3a, 0x9a              /* Supported Version: Random (GREASE) */
};

static const QUIC_PKT_HDR rx_script_3_expect_hdr = {
    QUIC_PKT_TYPE_VERSION_NEG,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    0,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},                                   /* DCID */
    {12, {0x35, 0x3c, 0x1b, 0x97, 0xca, 0xf8,   /* SCID */
          0x99, 0x11, 0x39, 0xad, 0x79, 0x1f}},
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    8, NULL
};

static const unsigned char rx_script_3_body[] = {
    0x00, 0x00, 0x00, 0x01,
    0xaa, 0x9a, 0x3a, 0x9a
};

static const struct rx_test_op rx_script_3[] = {
    RX_OP_SET_RX_DCID(empty_conn_id)
    /*
     * This is a version negotiation packet, so doesn't have any frames.
     * However, the depacketizer still handles this sort of packet, so
     * we still pass the packet to it, to exercise what it does.
     */
    RX_OP_INJECT_CHECK(3)
    RX_OP_CHECK_NO_PKT()
    RX_OP_END
};

/* 4. Real World - Retry (S2C) */
static const unsigned char rx_script_4_in[] = {
    0xf0,                           /* Long; Retry */
    0x00, 0x00, 0x00, 0x01,         /* Version 1 */
    0x00,                           /* DCID */
    0x04, 0xad, 0x15, 0x3f, 0xae,   /* SCID */
    /* Retry Token, including 16-byte Retry Integrity Tag */
    0xf6, 0x8b, 0x6e, 0xa3, 0xdc, 0x40, 0x38, 0xc6, 0xa5, 0x99, 0x1c, 0xa9,
    0x77, 0xe6, 0x1d, 0x4f, 0x09, 0x36, 0x12, 0x26, 0x00, 0x56, 0x0b, 0x29,
    0x7d, 0x5e, 0xda, 0x39, 0xc6, 0x61, 0x57, 0x69, 0x15, 0xff, 0x93, 0x39,
    0x95, 0xf0, 0x57, 0xf1, 0xe5, 0x36, 0x08, 0xad, 0xd2, 0x75, 0xa9, 0x68,
    0x29, 0xed, 0xaa, 0x03, 0x0e, 0x5f, 0xac, 0xbd, 0x26, 0x07, 0x95, 0x4e,
    0x48, 0x61, 0x26, 0xc5, 0xe2, 0x6c, 0x60, 0xbf, 0xa8, 0x6f, 0x51, 0xbb,
    0x1d, 0xf7, 0x98, 0x95, 0x3b, 0x2c, 0x50, 0x79, 0xcc, 0xde, 0x27, 0x84,
    0x44, 0x9b, 0xb2, 0x4a, 0x94, 0x4d, 0x4d, 0x3d, 0xbc, 0x00, 0x9d, 0x69,
    0xad, 0x45, 0x89, 0x04, 0x48, 0xca, 0x04, 0xf6, 0x3a, 0x62, 0xc1, 0x38,
    0x9d, 0x82, 0xb3, 0x45, 0x62, 0x4c,
};

static const QUIC_PKT_HDR rx_script_4_expect_hdr = {
    QUIC_PKT_TYPE_RETRY,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    0,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0xad, 0x15, 0x3f, 0xae}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    114, NULL
};

static const unsigned char rx_script_4_body[] = {
    0xf6, 0x8b, 0x6e, 0xa3, 0xdc, 0x40, 0x38, 0xc6, 0xa5, 0x99, 0x1c, 0xa9,
    0x77, 0xe6, 0x1d, 0x4f, 0x09, 0x36, 0x12, 0x26, 0x00, 0x56, 0x0b, 0x29,
    0x7d, 0x5e, 0xda, 0x39, 0xc6, 0x61, 0x57, 0x69, 0x15, 0xff, 0x93, 0x39,
    0x95, 0xf0, 0x57, 0xf1, 0xe5, 0x36, 0x08, 0xad, 0xd2, 0x75, 0xa9, 0x68,
    0x29, 0xed, 0xaa, 0x03, 0x0e, 0x5f, 0xac, 0xbd, 0x26, 0x07, 0x95, 0x4e,
    0x48, 0x61, 0x26, 0xc5, 0xe2, 0x6c, 0x60, 0xbf, 0xa8, 0x6f, 0x51, 0xbb,
    0x1d, 0xf7, 0x98, 0x95, 0x3b, 0x2c, 0x50, 0x79, 0xcc, 0xde, 0x27, 0x84,
    0x44, 0x9b, 0xb2, 0x4a, 0x94, 0x4d, 0x4d, 0x3d, 0xbc, 0x00, 0x9d, 0x69,
    0xad, 0x45, 0x89, 0x04, 0x48, 0xca, 0x04, 0xf6, 0x3a, 0x62, 0xc1, 0x38,
    0x9d, 0x82, 0xb3, 0x45, 0x62, 0x4c
};

static const struct rx_test_op rx_script_4[] = {
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_INJECT_CHECK(4)
    RX_OP_CHECK_NO_PKT()
    RX_OP_END
};

/*
 * 5. Real World - S2C Multiple Packets
 *      - Initial, Handshake, 1-RTT (AES-128-GCM/SHA256)
 */
static const QUIC_CONN_ID rx_script_5_c2s_init_dcid = {
    4, {0xad, 0x15, 0x3f, 0xae}
};

static const unsigned char rx_script_5_handshake_secret[32] = {
    0x5e, 0xc6, 0x4a, 0x4d, 0x0d, 0x40, 0x43, 0x3b, 0xd5, 0xbd, 0xe0, 0x19,
    0x71, 0x47, 0x56, 0xf3, 0x59, 0x3a, 0xa6, 0xc9, 0x3e, 0xdc, 0x81, 0x1e,
    0xc7, 0x72, 0x9d, 0x83, 0xd8, 0x8f, 0x88, 0x77
};

static const unsigned char rx_script_5_1rtt_secret[32] = {
    0x53, 0xf2, 0x1b, 0x94, 0xa7, 0x65, 0xf7, 0x76, 0xfb, 0x06, 0x27, 0xaa,
    0xd2, 0x3f, 0xe0, 0x9a, 0xbb, 0xcf, 0x99, 0x6f, 0x13, 0x2c, 0x6a, 0x37,
    0x95, 0xf3, 0xda, 0x21, 0xcb, 0xcb, 0xa5, 0x26,
};

static const unsigned char rx_script_5_in[] = {
    /* First Packet: Initial */
    0xc4,                           /* Long, Initial, PN Length=2 bytes */
    0x00, 0x00, 0x00, 0x01,         /* Version */
    0x00,                           /* DCID */
    0x04, 0x83, 0xd0, 0x0a, 0x27,   /* SCID */
    0x00,                           /* Token Length */
    0x41, 0xd2,                     /* Length (466) */
    0xe3, 0xab,                     /* PN (0) */
    0x22, 0x35, 0x34, 0x12, 0xcf, 0x20, 0x2b, 0x16, 0xaf, 0x08, 0xd4, 0xe0,
    0x94, 0x8b, 0x1e, 0x62, 0xdf, 0x31, 0x61, 0xcc, 0xf9, 0xfa, 0x66, 0x4f,
    0x18, 0x61, 0x07, 0xcb, 0x13, 0xd3, 0xf9, 0xbf, 0xe2, 0x8e, 0x25, 0x8d,
    0xd1, 0xdf, 0x58, 0x9c, 0x05, 0x20, 0xf9, 0xf2, 0x01, 0x20, 0xe9, 0x39,
    0xc3, 0x80, 0x77, 0xec, 0xa4, 0x57, 0xcf, 0x57, 0x8c, 0xdd, 0x68, 0x82,
    0x91, 0xfe, 0x71, 0xa0, 0xfa, 0x56, 0x4c, 0xf2, 0xe7, 0x2b, 0xd0, 0xc0,
    0xda, 0x81, 0xe2, 0x39, 0xb5, 0xf0, 0x0f, 0xd9, 0x07, 0xd5, 0x67, 0x09,
    0x02, 0xf0, 0xff, 0x74, 0xb0, 0xa0, 0xd9, 0x3a, 0x7e, 0xb6, 0x57, 0x82,
    0x47, 0x18, 0x66, 0xed, 0xe2, 0x18, 0x4d, 0xc2, 0x5c, 0x9f, 0x05, 0x09,
    0x18, 0x24, 0x0e, 0x3f, 0x3d, 0xf9, 0x15, 0x8b, 0x08, 0xfd, 0x25, 0xe9,
    0xc9, 0xb7, 0x8c, 0x18, 0x7b, 0xf3, 0x37, 0x58, 0xf0, 0xf0, 0xac, 0x33,
    0x55, 0x3f, 0x39, 0xbc, 0x62, 0x03, 0x8a, 0xc0, 0xd6, 0xcc, 0x49, 0x47,
    0xeb, 0x85, 0xb6, 0x72, 0xd7, 0xf8, 0xdc, 0x01, 0x32, 0xec, 0x1b, 0x4e,
    0x38, 0x6e, 0x2c, 0xc5, 0x80, 0xf2, 0x43, 0x4a, 0xf5, 0xe5, 0xa2, 0xf8,
    0x76, 0xa7, 0xa8, 0x57, 0x32, 0x67, 0x72, 0xeb, 0x82, 0xac, 0x3e, 0xc0,
    0x15, 0x67, 0xac, 0x32, 0x19, 0x18, 0x0a, 0xef, 0x20, 0xa1, 0xe8, 0xaf,
    0xac, 0x33, 0x87, 0x4c, 0x55, 0x05, 0x9b, 0x78, 0xf0, 0x3a, 0xce, 0x02,
    0x28, 0x06, 0x84, 0x61, 0x97, 0xac, 0x87, 0x8f, 0x25, 0xe7, 0x1b, 0xa3,
    0x02, 0x08, 0x4c, 0x2e, 0xef, 0xbd, 0x4f, 0x82, 0xe7, 0x37, 0x6c, 0x27,
    0x6f, 0x85, 0xb4, 0xbc, 0x79, 0x38, 0x45, 0x80, 0x8a, 0xda, 0x2f, 0x11,
    0x11, 0xac, 0x9c, 0xf3, 0x93, 0xc1, 0x49, 0x1b, 0x94, 0x12, 0x77, 0x07,
    0xdc, 0xbf, 0xc2, 0xfd, 0x8b, 0xf6, 0xf1, 0x66, 0x1c, 0x7f, 0x07, 0xbf,
    0x1f, 0xae, 0x27, 0x6c, 0x66, 0xe9, 0xa3, 0x64, 0x7a, 0x96, 0x78, 0x45,
    0xfe, 0x4b, 0x8c, 0x6f, 0x7f, 0x03, 0x47, 0x3c, 0xd7, 0xf7, 0x63, 0x92,
    0x58, 0x5b, 0x63, 0x83, 0x03, 0x05, 0xc3, 0x5d, 0x36, 0x62, 0x63, 0x5e,
    0xcf, 0xfe, 0x0a, 0x29, 0xfa, 0xeb, 0xc8, 0xaf, 0xce, 0x31, 0x07, 0x6a,
    0x09, 0x41, 0xc0, 0x2d, 0x98, 0x70, 0x05, 0x3b, 0x41, 0xfc, 0x7d, 0x61,
    0xe0, 0x41, 0x7d, 0x13, 0x41, 0x51, 0x52, 0xb4, 0x78, 0xd5, 0x46, 0x51,
    0x3b, 0xf1, 0xcd, 0xcc, 0x2e, 0x49, 0x30, 0x8b, 0x2a, 0xd2, 0xe6, 0x69,
    0xb5, 0x6b, 0x7a, 0xf4, 0xbb, 0xd1, 0xf8, 0x4a, 0xe8, 0x53, 0x10, 0x46,
    0x85, 0x8d, 0x66, 0x8e, 0x2b, 0xe8, 0x5d, 0xab, 0x7e, 0xfe, 0x5a, 0x79,
    0xcf, 0xc5, 0x0c, 0x30, 0x9e, 0x98, 0x02, 0xb3, 0xa6, 0xd5, 0xfa, 0x25,
    0xa8, 0xc8, 0xc1, 0xd9, 0x51, 0x60, 0x57, 0x5d, 0xfe, 0x75, 0x97, 0x05,
    0xda, 0xbb, 0xc6, 0x6a, 0xbe, 0x5c, 0xa5, 0x65, 0x0a, 0x12, 0x33, 0x1c,
    0xdf, 0xee, 0x08, 0xa9, 0x13, 0x13, 0x28, 0xce, 0x61, 0x59, 0xd1, 0x4e,
    0xc7, 0x74, 0xfd, 0x64, 0xde, 0x08, 0xce, 0xda, 0x3f, 0xec, 0xad, 0xc9,
    0xe1, 0xf9, 0x1f, 0x74, 0xf6, 0x86, 0x37, 0x6a, 0xa0, 0xc8, 0x0b, 0x1b,
    0x94, 0x98, 0x86, 0x81, 0x3b, 0xfc, 0x47, 0x6c, 0xc9, 0x3e, 0x3c, 0x30,
    0xc5, 0x9e, 0xb2, 0x32, 0x47, 0xf5, 0x0c, 0x6f,

    /* Second Packet: Handshake */
    0xe6,                           /* Long, Handshake, PN Length=2 bytes */
    0x00, 0x00, 0x00, 0x01,         /* Version */
    0x00,                           /* DCID */
    0x04, 0x83, 0xd0, 0x0a, 0x27,   /* SCID */
    0x42, 0x9c,                     /* Length (668) */
    0x9c, 0x55,                     /* PN (0) */
    0x55, 0xd4, 0x50, 0x02, 0x1a, 0x57, 0x84, 0x22, 0xcd, 0x01, 0xe5, 0x42,
    0x1b, 0x1e, 0x06, 0xf1, 0x86, 0xe2, 0x90, 0xf8, 0x9c, 0x3d, 0xa2, 0x7c,
    0xde, 0x2b, 0xc9, 0x2e, 0xcd, 0xa8, 0x4f, 0x5a, 0x20, 0xca, 0x96, 0xb6,
    0x11, 0x4b, 0xc8, 0x71, 0x32, 0xb5, 0xc7, 0x1a, 0x69, 0x7f, 0x1e, 0x37,
    0x49, 0xfb, 0x08, 0xce, 0x83, 0x5f, 0x02, 0x6d, 0x8a, 0x8f, 0xe7, 0x5d,
    0xe1, 0x34, 0x31, 0x22, 0x53, 0x53, 0x32, 0xcb, 0x04, 0x21, 0xce, 0xbc,
    0xa5, 0x1b, 0xdd, 0x4d, 0xd5, 0x1c, 0xd6, 0x5d, 0x88, 0x29, 0x5a, 0x19,
    0x71, 0x6a, 0xc2, 0xfa, 0xb7, 0xb4, 0x7d, 0xd1, 0x72, 0x93, 0x8f, 0x7c,
    0xb5, 0x36, 0x1b, 0xea, 0xf3, 0xf1, 0xd7, 0x6e, 0xd3, 0x91, 0x96, 0x62,
    0x4d, 0xc6, 0xec, 0xb7, 0xb0, 0xb7, 0x9b, 0x95, 0x8b, 0x14, 0x8d, 0x1a,
    0x0d, 0xb6, 0x3e, 0xec, 0xfe, 0x3b, 0x51, 0xea, 0x1a, 0x05, 0x14, 0x12,
    0x93, 0x0e, 0x7e, 0xe6, 0xa2, 0xc5, 0x22, 0x87, 0x65, 0xf8, 0x5d, 0x3c,
    0x55, 0x18, 0xcb, 0xe9, 0xef, 0x23, 0x43, 0xfe, 0xe8, 0x0d, 0xb2, 0x0f,
    0xc5, 0xf4, 0xb3, 0xde, 0x0c, 0xea, 0xa4, 0x48, 0x8e, 0xbf, 0x1f, 0xc7,
    0x99, 0x53, 0x8c, 0xc1, 0x3d, 0xba, 0xf4, 0x8e, 0x8e, 0x02, 0x52, 0xf6,
    0x1f, 0xcf, 0x1d, 0xaa, 0xb3, 0xcb, 0x08, 0xc2, 0xe1, 0x70, 0x68, 0x74,
    0x78, 0xa9, 0x30, 0x67, 0xba, 0x2b, 0xea, 0x35, 0x63, 0x47, 0xff, 0x29,
    0x73, 0x29, 0xc6, 0xe8, 0x08, 0xa9, 0x1e, 0x8f, 0x28, 0x41, 0xa4, 0x24,
    0x54, 0x26, 0x5f, 0x42, 0x77, 0xb1, 0x2b, 0x3d, 0x65, 0x67, 0x60, 0xa7,
    0x23, 0x0d, 0xa7, 0xf4, 0xd6, 0xe9, 0x4e, 0x58, 0x43, 0x9f, 0x3c, 0x9e,
    0x77, 0x61, 0xe5, 0x04, 0x4f, 0x73, 0xc9, 0x10, 0x79, 0xd0, 0xda, 0x3b,
    0xc6, 0x19, 0x93, 0x9f, 0x48, 0x3b, 0x76, 0x38, 0xa1, 0x72, 0x49, 0x7d,
    0x86, 0x7f, 0xe8, 0x1b, 0xa9, 0x5b, 0xc0, 0x47, 0xa0, 0x9c, 0x3f, 0x65,
    0x60, 0x76, 0x59, 0xaf, 0x20, 0x2d, 0x40, 0xa6, 0x80, 0x49, 0x5a, 0x8f,
    0x09, 0xf8, 0xf6, 0x97, 0xc1, 0xbd, 0xe1, 0x9f, 0x9b, 0xa2, 0x4c, 0x7b,
    0x88, 0xac, 0xbe, 0x4b, 0x11, 0x28, 0xd7, 0x67, 0xe6, 0xad, 0xaf, 0xd0,
    0xad, 0x01, 0x29, 0xa4, 0x4a, 0xc4, 0xb8, 0x2e, 0x42, 0x79, 0x24, 0x9e,
    0xd5, 0x34, 0xae, 0x45, 0xf1, 0x0b, 0x38, 0x4a, 0x76, 0xfb, 0x50, 0xa2,
    0x99, 0xc9, 0x5b, 0x6d, 0xc0, 0xb7, 0x55, 0xd8, 0x8d, 0x49, 0xdd, 0x1b,
    0xb8, 0xec, 0x10, 0x57, 0x9e, 0x33, 0xb4, 0x10, 0x16, 0x19, 0xac, 0x69,
    0xa2, 0x19, 0x1b, 0xd0, 0x77, 0x45, 0xeb, 0x49, 0x5c, 0xc5, 0x7c, 0xbe,
    0x4b, 0x4a, 0x22, 0x5c, 0x3d, 0x0e, 0x6e, 0xe5, 0x4b, 0x36, 0x06, 0x63,
    0x03, 0x97, 0xab, 0xed, 0xdc, 0xea, 0x64, 0xc2, 0x70, 0xb6, 0x7e, 0x35,
    0xfb, 0x13, 0x66, 0x37, 0xa3, 0x3f, 0x28, 0x16, 0x6c, 0xe7, 0xd4, 0xe6,
    0xca, 0x26, 0x0f, 0x19, 0xdd, 0x02, 0xae, 0xc1, 0xcf, 0x18, 0x7d, 0x56,
    0xe6, 0x52, 0xf3, 0x37, 0xb5, 0x86, 0x9d, 0x1d, 0x55, 0xb3, 0x95, 0x19,
    0x19, 0xa5, 0x44, 0x95, 0x81, 0xed, 0x02, 0x18, 0xf1, 0x85, 0x57, 0x78,
    0x28, 0xc4, 0x9a, 0xba, 0xe8, 0x5e, 0x22, 0x8d, 0xc1, 0x7b, 0x2a, 0x8a,
    0xc8, 0xb9, 0xdd, 0x82, 0xb2, 0x7b, 0x9f, 0x3d, 0xf5, 0x27, 0x2a, 0x48,
    0x53, 0xc7, 0xa0, 0x70, 0x0e, 0x9d, 0x61, 0xaa, 0xe2, 0xad, 0x28, 0xf2,
    0xb4, 0xfc, 0x56, 0x6b, 0x89, 0xe7, 0xf9, 0x51, 0xc9, 0xe9, 0xd3, 0x8a,
    0x8c, 0x7e, 0x86, 0xdd, 0xba, 0x2f, 0x39, 0xbf, 0x26, 0x62, 0x23, 0xd6,
    0x98, 0x6d, 0x3e, 0x72, 0xd7, 0x1b, 0xe1, 0x62, 0x94, 0x35, 0xe2, 0x18,
    0x19, 0x46, 0xb8, 0x2c, 0xb5, 0x8f, 0x8f, 0xb0, 0x5b, 0x76, 0x7b, 0x7e,
    0xb8, 0xc6, 0xb7, 0xe9, 0x4e, 0x9d, 0x30, 0x68, 0x03, 0x1e, 0x19, 0x73,
    0xc5, 0x3e, 0x24, 0xe2, 0x95, 0x60, 0x1b, 0x27, 0x93, 0x7c, 0x17, 0xc2,
    0xc6, 0xa3, 0xbd, 0xbd, 0x70, 0xc6, 0x60, 0x59, 0xc8, 0x5c, 0xd7, 0x9a,
    0xc4, 0x29, 0xac, 0x0f, 0xaa, 0x0d, 0xa9, 0x92, 0xa3, 0x95, 0xd7, 0x0f,
    0x6f, 0x74, 0x99, 0x9b, 0xc1, 0xd3, 0x68, 0x6d, 0xac, 0x82, 0x2d, 0x32,
    0x41, 0x9e, 0x0c, 0xf7, 0x31, 0x59, 0x4c, 0x93, 0x1c, 0x3b, 0x71, 0x69,
    0xcf, 0xc5, 0xca, 0x2b, 0xdf, 0xe7, 0xaa, 0xfd, 0x1d, 0x71, 0x01, 0x7e,
    0x1c, 0x70, 0x62, 0x20, 0x61, 0xf8, 0x35, 0xc1, 0x71, 0xe7, 0x02, 0x0d,
    0x88, 0x44, 0xd9, 0x00, 0xc5, 0xcc, 0x63, 0xe4, 0xf0, 0x86, 0xa7, 0xd0,
    0xfe, 0xcc, 0xb7, 0x1d, 0xfc, 0x21, 0x61, 0x54, 0x15, 0xea, 0x81, 0x5e,
    0xc0, 0x31, 0xfa, 0xbf, 0x7d, 0xb9, 0x3b, 0xa2, 0x1e, 0x42, 0x73, 0x05,
    0x3c, 0xdb, 0x21, 0x59, 0x4f, 0x63,

    /* Third Packet: 1-RTT */
    0x5f,               /* Short, 1-RTT, Spin=0, KP=0, PN Length=2 bytes */
    0x68, 0x47,         /* PN (0) */
    0xa3, 0x3c, 0xa5, 0x27, 0x5e, 0xf9, 0x8d, 0xec, 0xea, 0x6c, 0x09, 0x18,
    0x40, 0x80, 0xee, 0x9f, 0x6f, 0x73, 0x5c, 0x49, 0xe3, 0xec, 0xb7, 0x58,
    0x05, 0x66, 0x8f, 0xa3, 0x52, 0x37, 0xa1, 0x22, 0x1f, 0xc6, 0x92, 0xd6,
    0x59, 0x04, 0x99, 0xcb, 0x44, 0xef, 0x66, 0x05, 0x2d, 0xd0, 0x85, 0x24,
    0xbb, 0xe3, 0xa1, 0xd1, 0xbe, 0xf7, 0x54, 0xad, 0x65, 0xf4, 0xd4, 0x59,
    0x54, 0x87, 0x4e, 0x22, 0x4f, 0x06, 0x07, 0xa7, 0x8a, 0x14, 0x89, 0xd1,
    0x3f, 0xd3, 0xe4, 0x6f, 0x71, 0x8f, 0x9a, 0xd2, 0x3b, 0x61, 0x0a, 0xba,
    0x9a, 0x31, 0x56, 0xc7,
};

static const QUIC_PKT_HDR rx_script_5a_expect_hdr = {
    QUIC_PKT_TYPE_INITIAL,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0x83, 0xd0, 0x0a, 0x27}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    448, NULL
};

static const unsigned char rx_script_5a_body[] = {
    0x02, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x40, 0x5a, 0x02, 0x00,
    0x00, 0x56, 0x03, 0x03, 0xe2, 0xd2, 0x0a, 0x3b, 0xa2, 0xc4, 0xd2, 0x29,
    0xc8, 0xe8, 0xba, 0x23, 0x31, 0x88, 0x2c, 0x71, 0xeb, 0xba, 0x42, 0x5f,
    0x94, 0xe9, 0x0a, 0x90, 0x35, 0x31, 0x1e, 0xca, 0xed, 0xf8, 0x8a, 0x8d,
    0x00, 0x13, 0x01, 0x00, 0x00, 0x2e, 0x00, 0x2b, 0x00, 0x02, 0x03, 0x04,
    0x00, 0x33, 0x00, 0x24, 0x00, 0x1d, 0x00, 0x20, 0x96, 0x0b, 0x4b, 0x30,
    0x66, 0x3a, 0x75, 0x01, 0x4a, 0xdc, 0x2a, 0x75, 0x1f, 0xce, 0x7a, 0x30,
    0x9d, 0x00, 0xca, 0x20, 0xb4, 0xe0, 0x6b, 0x81, 0x23, 0x18, 0x0b, 0x20,
    0x1f, 0x54, 0x86, 0x1d,
};

static const QUIC_PKT_HDR rx_script_5b_expect_hdr = {
    QUIC_PKT_TYPE_HANDSHAKE,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0x83, 0xd0, 0x0a, 0x27}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    650, NULL
};

static const unsigned char rx_script_5b_body[] = {
    0x06, 0x00, 0x42, 0x86, 0x08, 0x00, 0x00, 0x7d, 0x00, 0x7b, 0x00, 0x10,
    0x00, 0x08, 0x00, 0x06, 0x05, 0x64, 0x75, 0x6d, 0x6d, 0x79, 0x00, 0x39,
    0x00, 0x6b, 0x4b, 0x20, 0x0b, 0x1b, 0xe1, 0x1f, 0xd0, 0x78, 0xc0, 0x69,
    0x72, 0x9c, 0xe2, 0xf7, 0x05, 0x04, 0x80, 0x08, 0x00, 0x00, 0x06, 0x04,
    0x80, 0x08, 0x00, 0x00, 0x07, 0x04, 0x80, 0x08, 0x00, 0x00, 0x04, 0x04,
    0x80, 0x0c, 0x00, 0x00, 0x08, 0x02, 0x40, 0x64, 0x09, 0x02, 0x40, 0x64,
    0x01, 0x04, 0x80, 0x00, 0x75, 0x30, 0x03, 0x02, 0x45, 0xac, 0x0b, 0x01,
    0x1a, 0x0c, 0x00, 0x02, 0x10, 0x41, 0x94, 0x41, 0x8d, 0x0d, 0xfb, 0x60,
    0x7b, 0xdc, 0xcc, 0xa2, 0x9c, 0x3e, 0xa5, 0xdf, 0x8d, 0x00, 0x08, 0x2d,
    0x71, 0x8a, 0x38, 0xdf, 0xdd, 0xe0, 0x03, 0x0e, 0x01, 0x04, 0x0f, 0x04,
    0x83, 0xd0, 0x0a, 0x27, 0x10, 0x04, 0xad, 0x15, 0x3f, 0xae, 0x20, 0x01,
    0x00, 0x0b, 0x00, 0x01, 0x8f, 0x00, 0x00, 0x01, 0x8b, 0x00, 0x01, 0x86,
    0x30, 0x82, 0x01, 0x82, 0x30, 0x82, 0x01, 0x29, 0xa0, 0x03, 0x02, 0x01,
    0x02, 0x02, 0x14, 0x0a, 0x73, 0x0f, 0x86, 0x18, 0xf2, 0xc3, 0x30, 0x01,
    0xd2, 0xc0, 0xc1, 0x62, 0x52, 0x13, 0xf1, 0x9c, 0x13, 0x39, 0xb5, 0x30,
    0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30,
    0x17, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0c,
    0x6d, 0x61, 0x70, 0x61, 0x6b, 0x74, 0x2e, 0x6c, 0x6f, 0x63, 0x61, 0x6c,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x38, 0x30, 0x32, 0x31, 0x32,
    0x30, 0x30, 0x31, 0x38, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x39, 0x30,
    0x31, 0x31, 0x32, 0x30, 0x30, 0x31, 0x38, 0x5a, 0x30, 0x17, 0x31, 0x15,
    0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0c, 0x6d, 0x61, 0x70,
    0x61, 0x6b, 0x74, 0x2e, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04,
    0x67, 0xf4, 0xd3, 0x8f, 0x15, 0x6d, 0xee, 0x85, 0xcc, 0x2a, 0x77, 0xfc,
    0x0b, 0x8f, 0x9f, 0xcf, 0xa9, 0x95, 0x5d, 0x5b, 0xcd, 0xb7, 0x8b, 0xba,
    0x31, 0x0a, 0x73, 0x62, 0xc5, 0xd0, 0x0e, 0x07, 0x90, 0xae, 0x38, 0x43,
    0x79, 0xce, 0x5e, 0x33, 0xad, 0x31, 0xbf, 0x9f, 0x2a, 0x56, 0x83, 0xa5,
    0x24, 0x16, 0xab, 0x0c, 0xf1, 0x64, 0xbe, 0xe4, 0x93, 0xb5, 0x89, 0xd6,
    0x05, 0xe4, 0xf7, 0x7b, 0xa3, 0x53, 0x30, 0x51, 0x30, 0x1d, 0x06, 0x03,
    0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x02, 0x64, 0x0f, 0x55, 0x69,
    0x14, 0x91, 0x19, 0xed, 0xf9, 0x1a, 0xe9, 0x1d, 0xa5, 0x5a, 0xd0, 0x48,
    0x96, 0x9f, 0x60, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
    0x30, 0x16, 0x80, 0x14, 0x02, 0x64, 0x0f, 0x55, 0x69, 0x14, 0x91, 0x19,
    0xed, 0xf9, 0x1a, 0xe9, 0x1d, 0xa5, 0x5a, 0xd0, 0x48, 0x96, 0x9f, 0x60,
    0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05,
    0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48,
    0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20,
    0x0a, 0x82, 0x92, 0x6e, 0xd3, 0xc6, 0x66, 0xd9, 0xd3, 0x75, 0xff, 0x71,
    0x3b, 0x61, 0x46, 0x21, 0x00, 0xe6, 0x21, 0x5d, 0x9c, 0x86, 0xe9, 0x65,
    0x40, 0x4f, 0xeb, 0x70, 0x4f, 0x2c, 0xad, 0x00, 0x02, 0x20, 0x08, 0xc2,
    0x07, 0x5d, 0x16, 0xfc, 0x54, 0x34, 0x2b, 0xb4, 0x18, 0x67, 0x44, 0x81,
    0xc9, 0xa9, 0x67, 0x2e, 0xce, 0xa1, 0x02, 0x9f, 0x3b, 0xe5, 0x61, 0x16,
    0x0b, 0x50, 0xf6, 0xa1, 0x50, 0x94, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x4a,
    0x04, 0x03, 0x00, 0x46, 0x30, 0x44, 0x02, 0x20, 0x7d, 0x57, 0x17, 0x14,
    0x46, 0x09, 0x95, 0x70, 0x09, 0x45, 0xe8, 0x9e, 0x5c, 0x87, 0x55, 0xd9,
    0x08, 0xc6, 0x5e, 0x47, 0x73, 0x5e, 0xb1, 0xc9, 0xef, 0xcb, 0xe5, 0x7f,
    0xcc, 0xb0, 0x28, 0xbc, 0x02, 0x20, 0x5d, 0xe4, 0x2b, 0x83, 0xd9, 0x78,
    0x75, 0x45, 0xf3, 0x22, 0x2b, 0x38, 0xeb, 0x68, 0xe5, 0x71, 0x5d, 0xcb,
    0xc3, 0x68, 0xb3, 0x0e, 0x7d, 0x5e, 0x1d, 0xc2, 0x1b, 0x8a, 0x62, 0x80,
    0x48, 0x3e, 0x14, 0x00, 0x00, 0x20, 0x37, 0xcd, 0x55, 0xca, 0x3f, 0x4b,
    0xf0, 0x95, 0xf8, 0xe4, 0xfe, 0x59, 0xab, 0xbc, 0xc1, 0x8f, 0x0c, 0x3f,
    0x41, 0x59, 0xf6, 0x96, 0xdb, 0x75, 0xae, 0xe7, 0x86, 0x1a, 0x92, 0xa7,
    0x53, 0x0a,
};

static const QUIC_PKT_HDR rx_script_5c_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},                           /* DCID */
    {0, {0}},                           /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    72, NULL
};

static const unsigned char rx_script_5c_body[] = {
    0x18, 0x03, 0x00, 0x04, 0x92, 0xec, 0xaa, 0xd6, 0x47, 0xd8, 0x8b, 0x56,
    0x3b, 0x5f, 0x67, 0xe6, 0xb9, 0xb9, 0xca, 0x72, 0xca, 0xf2, 0x49, 0x7d,
    0x18, 0x02, 0x00, 0x04, 0xa9, 0x6e, 0x9b, 0x84, 0x26, 0x43, 0x00, 0xc7,
    0x55, 0x71, 0x67, 0x2e, 0x52, 0xdd, 0x47, 0xfd, 0x06, 0x51, 0x33, 0x08,
    0x18, 0x01, 0x00, 0x04, 0x36, 0xd5, 0x1f, 0x06, 0x4e, 0xbf, 0xb4, 0xc9,
    0xef, 0x97, 0x1e, 0x9a, 0x3c, 0xab, 0x1e, 0xfc, 0xb7, 0x90, 0xc3, 0x1a,
};

static const struct rx_test_op rx_script_5[] = {
    RX_OP_ALLOW_1RTT()
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_5_c2s_init_dcid)
    RX_OP_INJECT_N(5)
    RX_OP_CHECK_PKT_N(5a)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_AES128GCM, rx_script_5_handshake_secret)
    RX_OP_CHECK_PKT_N(5b)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_AES128GCM, rx_script_5_1rtt_secret)
    RX_OP_CHECK_PKT_N(5c)
    RX_OP_CHECK_NO_PKT()

    /* Discard Initial EL and try injecting the packet again */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_INITIAL)
    RX_OP_INJECT_N(5)
    /* Initial packet is not output because we have discarded Initial keys */
    RX_OP_CHECK_PKT_N(5b)
    RX_OP_CHECK_PKT_N(5c)
    RX_OP_CHECK_NO_PKT()
    /* Try again with discarded keys */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_HANDSHAKE)
    RX_OP_INJECT_N(5)
    RX_OP_CHECK_PKT_N(5c)
    RX_OP_CHECK_NO_PKT()
    /* Try again */
    RX_OP_INJECT_N(5)
    RX_OP_CHECK_PKT_N(5c)
    RX_OP_CHECK_NO_PKT()
    /* Try again with discarded 1-RTT keys */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_1RTT)
    RX_OP_INJECT_N(5)
    RX_OP_CHECK_NO_PKT()

    /* Recreate QRL, test reading packets received before key */
    RX_OP_SET_SCID_LEN(0)
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_INJECT_N(5)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_5_c2s_init_dcid)
    RX_OP_CHECK_PKT_N(5a)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_AES128GCM, rx_script_5_handshake_secret)
    RX_OP_CHECK_PKT_N(5b)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_AES128GCM, rx_script_5_1rtt_secret)
    RX_OP_CHECK_PKT_N(5c)
    RX_OP_CHECK_NO_PKT()

    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_INITIAL)
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_HANDSHAKE)
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_1RTT)
    RX_OP_INJECT_N(5)
    RX_OP_CHECK_NO_PKT()

    RX_OP_END
};

/*
 * 6. Real World - S2C Multiple Packets
 *      - Initial, Handshake, 1-RTT (AES-256-GCM/SHA384)
 */
static const QUIC_CONN_ID rx_script_6_c2s_init_dcid = {
    4, {0xac, 0x88, 0x95, 0xbd}
};

static const unsigned char rx_script_6_handshake_secret[48] = {
    0xd1, 0x41, 0xb0, 0xf6, 0x0d, 0x8b, 0xbd, 0xe8, 0x5b, 0xa8, 0xff, 0xd7,
    0x18, 0x9a, 0x23, 0x7b, 0x13, 0x5c, 0x1e, 0x90, 0x1d, 0x08, 0x95, 0xcc,
    0xc5, 0x8e, 0x73, 0x4e, 0x02, 0x6f, 0x3c, 0xb6, 0x26, 0x77, 0x8d, 0x53,
    0xc5, 0x62, 0x9f, 0xb5, 0xf0, 0x88, 0xfb, 0xe5, 0x14, 0x71, 0xab, 0xe6,
};

static const unsigned char rx_script_6_1rtt_secret[48] = {
    0x2d, 0x6b, 0x9d, 0xd4, 0x39, 0xa0, 0xe7, 0xff, 0x17, 0xe2, 0xcb, 0x5c,
    0x0d, 0x4a, 0xf6, 0x3f, 0xf4, 0xfe, 0xfc, 0xe5, 0x22, 0xfa, 0xf5, 0x5b,
    0xc0, 0xb2, 0x18, 0xbb, 0x92, 0x4d, 0x35, 0xea, 0x67, 0xa6, 0xe7, 0xc1,
    0x90, 0x10, 0xc9, 0x14, 0x46, 0xf5, 0x95, 0x57, 0x8b, 0x90, 0x88, 0x5d,
};

static const unsigned char rx_script_6_in[] = {
    /* First Packet: Initial */
    0xc5,                           /* Long, Initial, PN Length=2 bytes */
    0x00, 0x00, 0x00, 0x01,         /* Version */
    0x00,                           /* DCID */
    0x04, 0x36, 0xf4, 0x75, 0x2d,   /* SCID */
    0x00,                           /* Token Length */
    0x41, 0xbe,                     /* Length (446) */
    0xa9, 0xe2,                     /* PN (0) */
    0x83, 0x39, 0x95, 0x8f, 0x8f, 0x8c, 0xa9, 0xaf, 0x10, 0x29, 0x3d, 0xfc,
    0x56, 0x4a, 0x1c, 0x4b, 0xc9, 0x48, 0xb1, 0xaf, 0x36, 0xd5, 0xac, 0x95,
    0xbf, 0xfd, 0x2c, 0x4d, 0x70, 0x2e, 0x5b, 0x7c, 0x22, 0x5f, 0x5f, 0xee,
    0x10, 0x8f, 0xfb, 0x0b, 0x5f, 0x9d, 0x7e, 0x68, 0x2f, 0x94, 0x0b, 0xdb,
    0xed, 0xef, 0xfa, 0x4e, 0xc6, 0xd5, 0xe7, 0xef, 0xe0, 0x78, 0x3c, 0xdc,
    0xe9, 0xd8, 0xe8, 0x56, 0x71, 0xd7, 0xe7, 0x6c, 0x7f, 0x5d, 0xaa, 0x7a,
    0x52, 0x1d, 0x95, 0x7a, 0x80, 0x70, 0x38, 0xc0, 0x8b, 0xa1, 0x2f, 0x09,
    0x16, 0xd2, 0xec, 0xa3, 0x23, 0x72, 0x45, 0x3c, 0xbd, 0x8c, 0xda, 0xbb,
    0x37, 0x5a, 0x8d, 0xb2, 0x00, 0x7e, 0x67, 0x0c, 0xa0, 0x32, 0xdd, 0x80,
    0x07, 0x71, 0xb0, 0x95, 0x21, 0xbc, 0x1e, 0xbd, 0x63, 0x0a, 0x10, 0xe7,
    0x4b, 0x6e, 0x2e, 0x85, 0x3a, 0x65, 0xf7, 0x06, 0x6e, 0x7e, 0x8f, 0x65,
    0x8c, 0xb1, 0x93, 0xe9, 0x0d, 0xe8, 0x46, 0xe7, 0xcf, 0xa7, 0xd2, 0x8b,
    0x15, 0x23, 0xec, 0xc3, 0xec, 0x44, 0xda, 0x62, 0x15, 0x35, 0x34, 0x2f,
    0x62, 0x77, 0xc8, 0x1f, 0x83, 0x22, 0x00, 0xe5, 0xc0, 0x89, 0xb8, 0x97,
    0xd2, 0x37, 0x02, 0xea, 0xa2, 0x35, 0xbf, 0x19, 0xf0, 0xba, 0x1d, 0xb7,
    0xaa, 0x36, 0xbb, 0x11, 0x60, 0xc3, 0x45, 0x1f, 0xe5, 0x18, 0xde, 0x4c,
    0x01, 0x23, 0x2d, 0x17, 0x78, 0xdd, 0x4c, 0x8a, 0x1e, 0x1b, 0xd4, 0xda,
    0x56, 0x43, 0x13, 0xa4, 0x4f, 0xfd, 0xd5, 0x92, 0x6a, 0x05, 0x5f, 0x14,
    0x63, 0x85, 0x7d, 0xf1, 0x31, 0xb8, 0x27, 0x0b, 0xa6, 0xb5, 0x50, 0xca,
    0x8b, 0x0e, 0xa1, 0x0d, 0xf9, 0xc4, 0xea, 0x6a, 0x6e, 0x4b, 0x6d, 0xdf,
    0x49, 0xe8, 0x32, 0xf6, 0x85, 0xc4, 0x29, 0x26, 0x32, 0xfb, 0x5e, 0xa8,
    0x55, 0x6b, 0x67, 0xe9, 0xaa, 0x35, 0x33, 0x90, 0xd8, 0x2a, 0x71, 0x0b,
    0x6a, 0x48, 0xc4, 0xa3, 0x8b, 0xe0, 0xe7, 0x00, 0x3d, 0xee, 0x30, 0x70,
    0x84, 0xbd, 0xa3, 0x3c, 0x9e, 0xa3, 0x5c, 0x69, 0xab, 0x55, 0x7b, 0xe2,
    0xe5, 0x86, 0x13, 0xcb, 0x93, 0x3f, 0xcb, 0x3e, 0x6d, 0xc9, 0xc2, 0x10,
    0x2b, 0x00, 0x9b, 0x3f, 0x14, 0x4e, 0x04, 0x27, 0xc0, 0xae, 0x1d, 0x48,
    0x89, 0x3a, 0xf4, 0xac, 0xe0, 0x05, 0x07, 0xc9, 0x74, 0x6e, 0x21, 0x01,
    0xe9, 0x26, 0xfd, 0xb4, 0xb2, 0x2a, 0xda, 0x72, 0xda, 0xbf, 0x63, 0x9d,
    0x37, 0xaf, 0x90, 0x05, 0xd6, 0x89, 0xc7, 0xa6, 0x81, 0x4e, 0x2a, 0x30,
    0xe3, 0x05, 0x88, 0x9f, 0xd0, 0xba, 0x8d, 0xc4, 0x21, 0x52, 0x5a, 0x7a,
    0xe1, 0xad, 0xd3, 0x88, 0xc2, 0x18, 0xad, 0x4c, 0xb1, 0x66, 0x73, 0x1b,
    0xf2, 0xd1, 0xb9, 0x43, 0xaa, 0xc4, 0x66, 0xcd, 0x42, 0xfa, 0x80, 0xec,
    0xa1, 0x7c, 0x45, 0x02, 0x53, 0x45, 0xd5, 0x07, 0xd4, 0x70, 0x12, 0x1b,
    0x08, 0x05, 0x6e, 0x99, 0x0a, 0xd3, 0x5b, 0x99, 0x6b, 0x65, 0xc4, 0xc0,
    0x04, 0x1b, 0x75, 0xf2, 0x86, 0x99, 0x09, 0x4a, 0x50, 0x70, 0x00, 0x7a,
    0x93, 0xaa, 0xe6, 0xf4, 0x03, 0x29, 0x06, 0xa4, 0x30, 0x6d, 0x52, 0xbd,
    0x60, 0xd1, 0x7e, 0xd6, 0x07, 0xc0, 0x41, 0x01, 0x12, 0x3e, 0x16, 0x94,

    /* Second Packet: Handshake */
    0xea,                           /* Long, Handshake, PN Length=2 bytes */
    0x00, 0x00, 0x00, 0x01,         /* Version */
    0x00,                           /* DCID */
    0x04, 0x36, 0xf4, 0x75, 0x2d,   /* SCID */
    0x42, 0xb0,                     /* Length (688) */
    0x3a, 0xc5,                     /* PN (0) */
    0x3b, 0x8e, 0x4c, 0x01, 0x72, 0x6b, 0xfa, 0xbb, 0xad, 0xf9, 0x9e, 0x21,
    0xb1, 0xd0, 0x01, 0xf1, 0xd4, 0x67, 0x8d, 0x2c, 0xee, 0x04, 0x60, 0x4a,
    0xe2, 0xe4, 0xc6, 0x89, 0x01, 0xae, 0x3c, 0x1f, 0xf7, 0xe6, 0xf7, 0xac,
    0x26, 0xcf, 0x3c, 0x6d, 0x1d, 0xfd, 0x11, 0x02, 0x51, 0x73, 0xb5, 0xe1,
    0xb2, 0x44, 0x42, 0x32, 0x0f, 0xf5, 0x3d, 0x55, 0x2d, 0x1f, 0x02, 0x29,
    0x51, 0x35, 0xdb, 0xc7, 0x7a, 0x34, 0x4b, 0xec, 0x60, 0x49, 0xa2, 0x90,
    0x11, 0xef, 0x5a, 0xa9, 0x1c, 0xf7, 0xd9, 0x21, 0x68, 0x1c, 0x2b, 0xc6,
    0x57, 0xde, 0xb1, 0x0b, 0x31, 0xed, 0xef, 0x16, 0xba, 0x08, 0xb9, 0xe2,
    0xd9, 0xd0, 0xd8, 0x1f, 0xc4, 0x32, 0xe8, 0x45, 0x2a, 0x86, 0xe4, 0xd3,
    0xaf, 0x72, 0x4f, 0x30, 0x01, 0x71, 0x15, 0x9b, 0xa9, 0x55, 0x35, 0xf7,
    0x39, 0x7e, 0x6a, 0x59, 0x18, 0x4f, 0xe6, 0xdf, 0xb5, 0x0d, 0xc2, 0xe7,
    0xb2, 0xa1, 0xa6, 0xa3, 0x9c, 0xf0, 0x0d, 0x59, 0x05, 0x49, 0x95, 0xfa,
    0xcc, 0x72, 0xd7, 0xc0, 0x84, 0x2e, 0xc4, 0x1c, 0xd4, 0xa0, 0xe3, 0x6c,
    0x5a, 0x8c, 0x94, 0x4d, 0x37, 0x1a, 0x1c, 0x68, 0x93, 0x5f, 0xe5, 0x99,
    0x27, 0xc6, 0x06, 0xaa, 0x1f, 0x29, 0x17, 0xc5, 0x8c, 0x3d, 0x53, 0xa7,
    0x05, 0x3a, 0x44, 0x53, 0x86, 0xed, 0x56, 0x99, 0x4c, 0xe2, 0x7b, 0x3a,
    0x1e, 0x5d, 0x6d, 0xac, 0x78, 0x1e, 0xfa, 0x55, 0x58, 0x6e, 0x72, 0xee,
    0xf9, 0x33, 0x64, 0x7f, 0x93, 0x3c, 0xfe, 0x18, 0x97, 0x6b, 0x02, 0x74,
    0x90, 0x0d, 0xba, 0x89, 0xc0, 0x22, 0x0a, 0x0a, 0x37, 0x4c, 0x28, 0x74,
    0xa7, 0x3a, 0x44, 0x74, 0x42, 0xff, 0xf1, 0xd2, 0x8d, 0x0c, 0xc1, 0xed,
    0x98, 0x98, 0x8e, 0xa8, 0x6b, 0x95, 0x6a, 0x86, 0x0b, 0xb4, 0x95, 0x58,
    0x34, 0x12, 0xb0, 0xc0, 0xf8, 0x2d, 0x5b, 0x40, 0x51, 0x80, 0x07, 0x91,
    0x31, 0x77, 0xd3, 0x06, 0xa5, 0xe5, 0x1f, 0xe2, 0xf8, 0x92, 0xe4, 0x23,
    0x2b, 0xf0, 0x4c, 0xa9, 0xa5, 0x6c, 0x6f, 0xaf, 0xaf, 0xbf, 0x97, 0xcf,
    0x46, 0xf2, 0x8d, 0x61, 0x0e, 0x73, 0xcd, 0xc5, 0xde, 0xda, 0x50, 0x82,
    0x61, 0x6d, 0xb1, 0xa2, 0xbe, 0x6b, 0x99, 0xcd, 0x5b, 0x99, 0x8f, 0x66,
    0xab, 0x11, 0x78, 0xcc, 0xdb, 0x66, 0x98, 0xca, 0x19, 0x92, 0xf4, 0x05,
    0xae, 0xe6, 0xf3, 0xe7, 0xf0, 0x30, 0x28, 0x31, 0x74, 0xff, 0xe2, 0xb3,
    0x3a, 0x4f, 0x79, 0xe7, 0x2a, 0x9f, 0xe3, 0x41, 0xb2, 0x88, 0xc8, 0x8f,
    0x77, 0x57, 0x42, 0x65, 0xdb, 0x07, 0xf6, 0x5f, 0xb8, 0x34, 0x17, 0xe3,
    0x8d, 0x22, 0x5b, 0x88, 0x94, 0x60, 0x97, 0x32, 0x3d, 0x8a, 0x51, 0x9d,
    0xb5, 0xac, 0xd7, 0x99, 0x96, 0x23, 0x6d, 0xc9, 0xab, 0x61, 0x41, 0x8f,
    0x72, 0x1b, 0xf8, 0x84, 0xd9, 0x57, 0x88, 0x68, 0x3d, 0x73, 0x5f, 0xb1,
    0x18, 0x5c, 0x3a, 0x35, 0xd2, 0xc5, 0xb7, 0x29, 0xc7, 0x95, 0xdd, 0x21,
    0xc0, 0x78, 0x49, 0xf3, 0x24, 0xe0, 0x4c, 0x5c, 0x32, 0x08, 0xb7, 0x00,
    0x43, 0x70, 0x5a, 0x95, 0x23, 0x91, 0xf5, 0xb7, 0x61, 0x85, 0x6f, 0xb3,
    0xa4, 0x6b, 0x05, 0x9d, 0x39, 0xa3, 0xb1, 0x1c, 0x61, 0xc5, 0xa5, 0xe7,
    0x9a, 0xe9, 0x5d, 0xaa, 0xca, 0x11, 0xd8, 0x4b, 0xa4, 0x9c, 0x18, 0x4e,
    0x2b, 0x2d, 0x75, 0xc1, 0x12, 0x20, 0xe4, 0x66, 0xa5, 0x59, 0x67, 0x4b,
    0xcc, 0x52, 0x2d, 0xfa, 0xaa, 0xa4, 0xe9, 0xfc, 0x79, 0xd7, 0xff, 0x03,
    0x3e, 0xec, 0xba, 0x97, 0x37, 0x52, 0xc1, 0x57, 0x31, 0x8e, 0x57, 0x0c,
    0x54, 0x92, 0x9c, 0x25, 0x5c, 0xfa, 0x9f, 0xa5, 0x36, 0x18, 0xd0, 0xaa,
    0xf3, 0x3b, 0x5b, 0x59, 0xbd, 0x33, 0x5e, 0x7d, 0x74, 0x7c, 0xaf, 0xe9,
    0x54, 0x80, 0xc4, 0xb4, 0xa1, 0x24, 0x9e, 0x23, 0x0d, 0xbf, 0x4e, 0x0f,
    0xaf, 0xa5, 0x16, 0xcb, 0x3b, 0xfa, 0x33, 0xa5, 0x68, 0xa6, 0x64, 0x48,
    0x2f, 0x5e, 0xfa, 0x64, 0x4e, 0xe3, 0x27, 0x4f, 0x13, 0xe6, 0x37, 0xf6,
    0xb9, 0x63, 0x4b, 0xdc, 0x49, 0x3c, 0x5e, 0x9e, 0x06, 0xea, 0xac, 0xa3,
    0xdf, 0x6c, 0x49, 0xfb, 0xa1, 0x01, 0x4f, 0x6f, 0x74, 0x1f, 0xd3, 0x26,
    0xa1, 0x92, 0x3e, 0xe0, 0x73, 0xd6, 0x3b, 0x67, 0x13, 0x53, 0x2e, 0xcb,
    0xbc, 0x83, 0xd0, 0x6e, 0x28, 0xb1, 0xcb, 0xd9, 0x66, 0xe0, 0x33, 0x59,
    0x45, 0xd3, 0x13, 0xc2, 0x48, 0xd5, 0x9e, 0x88, 0xba, 0x75, 0x7b, 0xb1,
    0xfe, 0x6f, 0xec, 0xde, 0xff, 0x14, 0x59, 0x75, 0xbf, 0x1a, 0x74, 0x47,
    0xc5, 0xd8, 0xe8, 0x1b, 0x3c, 0x86, 0xd7, 0x1f, 0x99, 0x11, 0xd3, 0x29,
    0xfd, 0x5d, 0x22, 0x7e, 0x03, 0x78, 0xed, 0x62, 0x0e, 0xbe, 0x6d, 0x75,
    0xf4, 0xa8, 0x6e, 0xc7, 0x21, 0x76, 0xc5, 0xa0, 0x0c, 0xaa, 0x58, 0x78,
    0x7e, 0x6e, 0xfc, 0x1e, 0x2a, 0x1c, 0xdd, 0xe5, 0x78, 0x08, 0xbd, 0xdb,
    0xea, 0x8f, 0x8a, 0xa5, 0xbf, 0x93, 0xfe, 0x0f, 0x03, 0xa1, 0xc8, 0x64,
    0x9f, 0x4a,

    /* Third Packet: 1-RTT */
    0x48,               /* Short, 1-RTT, Spin=0, KP=0, PN Length=2 bytes */
    0x3e, 0x28,         /* PN (0) */
    0xb9, 0xdb, 0x61, 0xf8, 0x8b, 0x3a, 0xef, 0x26, 0x69, 0xf2, 0x57, 0xc6,
    0x84, 0x25, 0x6b, 0x77, 0xbe, 0x8c, 0x43, 0x32, 0xf3, 0x9a, 0xd1, 0x85,
    0x14, 0xbc, 0x89, 0x3b, 0x9c, 0xf3, 0xfc, 0x00, 0xa1, 0x3a, 0xc3, 0xc4,
    0x1e, 0xdf, 0xd0, 0x11, 0x70, 0xd9, 0x02, 0x7a, 0xd4, 0xef, 0x86, 0x67,
    0xb1, 0x1e, 0x5d, 0xe3, 0x7f, 0x82, 0x14, 0x52, 0xa5, 0x8a, 0x89, 0xa7,
    0x98, 0x75, 0x2f, 0x8a, 0x00, 0xf3, 0xbd, 0x49, 0x26, 0x4d, 0x0c, 0xc7,
    0x38, 0xe7, 0x91, 0x85, 0xc9, 0x21, 0x6a, 0x1c, 0xc4, 0xa3, 0x0e, 0xd8,
    0xfe, 0xb1, 0x25, 0x1a,
};

static const QUIC_PKT_HDR rx_script_6a_expect_hdr = {
    QUIC_PKT_TYPE_INITIAL,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0x36, 0xf4, 0x75, 0x2d}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    428, NULL
};

static const unsigned char rx_script_6a_body[] = {
    0x02, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00,
    0x40, 0x5a, 0x02, 0x00, 0x00, 0x56, 0x03, 0x03, 0xc3, 0x45, 0xe8, 0xb8,
    0xf9, 0x7c, 0x9f, 0x5d, 0xcf, 0x66, 0x25, 0xe4, 0x91, 0x0e, 0xb0, 0x5a,
    0x14, 0xce, 0xaf, 0xea, 0x83, 0x12, 0xde, 0x68, 0xd9, 0x31, 0xf2, 0x23,
    0x11, 0x3a, 0x15, 0xcb, 0x00, 0x13, 0x02, 0x00, 0x00, 0x2e, 0x00, 0x2b,
    0x00, 0x02, 0x03, 0x04, 0x00, 0x33, 0x00, 0x24, 0x00, 0x1d, 0x00, 0x20,
    0xab, 0xd3, 0xc6, 0x9f, 0x36, 0xd3, 0x52, 0x93, 0x87, 0xee, 0x92, 0x01,
    0xa2, 0xd6, 0x9a, 0x5e, 0x61, 0x43, 0xcc, 0x4a, 0xcc, 0x7a, 0xcd, 0x83,
    0xb2, 0xd9, 0xad, 0xd1, 0x14, 0xdc, 0x84, 0x61,
};

static const QUIC_PKT_HDR rx_script_6b_expect_hdr = {
    QUIC_PKT_TYPE_HANDSHAKE,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0x36, 0xf4, 0x75, 0x2d}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    670, NULL
};

static const unsigned char rx_script_6b_body[] = {
    0x06, 0x00, 0x42, 0x9a, 0x08, 0x00, 0x00, 0x80, 0x00, 0x7e, 0x00, 0x10,
    0x00, 0x08, 0x00, 0x06, 0x05, 0x64, 0x75, 0x6d, 0x6d, 0x79, 0x00, 0x39,
    0x00, 0x6e, 0x47, 0xfa, 0x05, 0x5a, 0xe0, 0xec, 0x4a, 0xf3, 0x05, 0x04,
    0x80, 0x08, 0x00, 0x00, 0x06, 0x04, 0x80, 0x08, 0x00, 0x00, 0x07, 0x04,
    0x80, 0x08, 0x00, 0x00, 0x04, 0x04, 0x80, 0x0c, 0x00, 0x00, 0x08, 0x02,
    0x40, 0x64, 0x09, 0x02, 0x40, 0x64, 0x01, 0x04, 0x80, 0x00, 0x75, 0x30,
    0x03, 0x02, 0x45, 0xac, 0x0b, 0x01, 0x1a, 0x0c, 0x00, 0x02, 0x10, 0x35,
    0xd7, 0x7d, 0x8b, 0xc5, 0xb1, 0x89, 0xb1, 0x5c, 0x23, 0x74, 0x50, 0xfd,
    0x47, 0xfe, 0xd2, 0x00, 0x11, 0x96, 0x38, 0x27, 0xde, 0x7d, 0xfb, 0x2b,
    0x38, 0x56, 0xe5, 0x2a, 0xb8, 0x6b, 0xfa, 0xaa, 0xde, 0x81, 0x0e, 0x01,
    0x04, 0x0f, 0x04, 0x36, 0xf4, 0x75, 0x2d, 0x10, 0x04, 0xac, 0x88, 0x95,
    0xbd, 0x20, 0x01, 0x00, 0x0b, 0x00, 0x01, 0x8f, 0x00, 0x00, 0x01, 0x8b,
    0x00, 0x01, 0x86, 0x30, 0x82, 0x01, 0x82, 0x30, 0x82, 0x01, 0x29, 0xa0,
    0x03, 0x02, 0x01, 0x02, 0x02, 0x14, 0x0a, 0x73, 0x0f, 0x86, 0x18, 0xf2,
    0xc3, 0x30, 0x01, 0xd2, 0xc0, 0xc1, 0x62, 0x52, 0x13, 0xf1, 0x9c, 0x13,
    0x39, 0xb5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04,
    0x03, 0x02, 0x30, 0x17, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0c, 0x0c, 0x6d, 0x61, 0x70, 0x61, 0x6b, 0x74, 0x2e, 0x6c, 0x6f,
    0x63, 0x61, 0x6c, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x38, 0x30,
    0x32, 0x31, 0x32, 0x30, 0x30, 0x31, 0x38, 0x5a, 0x17, 0x0d, 0x32, 0x32,
    0x30, 0x39, 0x30, 0x31, 0x31, 0x32, 0x30, 0x30, 0x31, 0x38, 0x5a, 0x30,
    0x17, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0c,
    0x6d, 0x61, 0x70, 0x61, 0x6b, 0x74, 0x2e, 0x6c, 0x6f, 0x63, 0x61, 0x6c,
    0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
    0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0x67, 0xf4, 0xd3, 0x8f, 0x15, 0x6d, 0xee, 0x85, 0xcc,
    0x2a, 0x77, 0xfc, 0x0b, 0x8f, 0x9f, 0xcf, 0xa9, 0x95, 0x5d, 0x5b, 0xcd,
    0xb7, 0x8b, 0xba, 0x31, 0x0a, 0x73, 0x62, 0xc5, 0xd0, 0x0e, 0x07, 0x90,
    0xae, 0x38, 0x43, 0x79, 0xce, 0x5e, 0x33, 0xad, 0x31, 0xbf, 0x9f, 0x2a,
    0x56, 0x83, 0xa5, 0x24, 0x16, 0xab, 0x0c, 0xf1, 0x64, 0xbe, 0xe4, 0x93,
    0xb5, 0x89, 0xd6, 0x05, 0xe4, 0xf7, 0x7b, 0xa3, 0x53, 0x30, 0x51, 0x30,
    0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x02, 0x64,
    0x0f, 0x55, 0x69, 0x14, 0x91, 0x19, 0xed, 0xf9, 0x1a, 0xe9, 0x1d, 0xa5,
    0x5a, 0xd0, 0x48, 0x96, 0x9f, 0x60, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x02, 0x64, 0x0f, 0x55, 0x69,
    0x14, 0x91, 0x19, 0xed, 0xf9, 0x1a, 0xe9, 0x1d, 0xa5, 0x5a, 0xd0, 0x48,
    0x96, 0x9f, 0x60, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01,
    0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0a, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30,
    0x44, 0x02, 0x20, 0x0a, 0x82, 0x92, 0x6e, 0xd3, 0xc6, 0x66, 0xd9, 0xd3,
    0x75, 0xff, 0x71, 0x3b, 0x61, 0x46, 0x21, 0x00, 0xe6, 0x21, 0x5d, 0x9c,
    0x86, 0xe9, 0x65, 0x40, 0x4f, 0xeb, 0x70, 0x4f, 0x2c, 0xad, 0x00, 0x02,
    0x20, 0x08, 0xc2, 0x07, 0x5d, 0x16, 0xfc, 0x54, 0x34, 0x2b, 0xb4, 0x18,
    0x67, 0x44, 0x81, 0xc9, 0xa9, 0x67, 0x2e, 0xce, 0xa1, 0x02, 0x9f, 0x3b,
    0xe5, 0x61, 0x16, 0x0b, 0x50, 0xf6, 0xa1, 0x50, 0x94, 0x00, 0x00, 0x0f,
    0x00, 0x00, 0x4b, 0x04, 0x03, 0x00, 0x47, 0x30, 0x45, 0x02, 0x20, 0x78,
    0x9e, 0xe0, 0x6a, 0x7a, 0xbd, 0xc3, 0x84, 0x3d, 0x25, 0x6a, 0x59, 0x23,
    0x97, 0x52, 0x64, 0x4e, 0xb6, 0x9f, 0xcc, 0xd3, 0xd7, 0xa9, 0x29, 0x44,
    0x75, 0x6d, 0x50, 0xfc, 0x22, 0xde, 0xd3, 0x02, 0x21, 0x00, 0xe5, 0x28,
    0xd6, 0x5a, 0xd1, 0xec, 0x4a, 0xcc, 0x20, 0xb4, 0xea, 0x15, 0xfb, 0x8e,
    0x73, 0xa8, 0x6b, 0xbb, 0x42, 0x70, 0x90, 0x08, 0x6e, 0x74, 0x6f, 0x5a,
    0x05, 0xb5, 0x39, 0xee, 0x01, 0x04, 0x14, 0x00, 0x00, 0x30, 0xff, 0x9f,
    0xb2, 0x1d, 0xcb, 0x4f, 0xfc, 0x7a, 0xac, 0xf4, 0x75, 0x24, 0x83, 0x5f,
    0x8d, 0xa3, 0x3e, 0x9d, 0xef, 0x43, 0x67, 0x89, 0x5d, 0x55, 0xc7, 0xce,
    0x80, 0xab, 0xc3, 0xc7, 0x74, 0xc7, 0xb2, 0x91, 0x27, 0xce, 0xd8, 0x5e,
    0xc4, 0x4e, 0x96, 0x19, 0x68, 0x2d, 0xbe, 0x6f, 0x49, 0xfa,
};

static const QUIC_PKT_HDR rx_script_6c_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},                           /* DCID */
    {0, {0}},                           /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    72, NULL
};

static const unsigned char rx_script_6c_body[] = {
    0x18, 0x03, 0x00, 0x04, 0xf2, 0x94, 0x49, 0xc3, 0x34, 0xa1, 0xf4, 0x0f,
    0xcb, 0xb8, 0x03, 0x04, 0x1f, 0xc8, 0x69, 0xb9, 0x3b, 0xd5, 0xc6, 0x93,
    0x18, 0x02, 0x00, 0x04, 0x9a, 0x4f, 0xec, 0x52, 0xde, 0xd2, 0xc8, 0xb7,
    0x1c, 0x0c, 0xf3, 0x4e, 0x46, 0xf0, 0x6c, 0x54, 0x34, 0x1b, 0x0d, 0x98,
    0x18, 0x01, 0x00, 0x04, 0xe3, 0x33, 0x9e, 0x59, 0x00, 0x69, 0xc3, 0xac,
    0xfc, 0x58, 0x0e, 0xa4, 0xf4, 0xf3, 0x23, 0x1b, 0xd6, 0x8e, 0x5b, 0x08,
};

static const struct rx_test_op rx_script_6[] = {
    RX_OP_ALLOW_1RTT()
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_6_c2s_init_dcid)
    RX_OP_INJECT_N(6)
    RX_OP_CHECK_PKT_N(6a)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_AES256GCM, rx_script_6_handshake_secret)
    RX_OP_CHECK_PKT_N(6b)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_AES256GCM, rx_script_6_1rtt_secret)
    RX_OP_CHECK_PKT_N(6c)
    RX_OP_CHECK_NO_PKT()

    /* Discard Initial EL and try injecting the packet again */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_INITIAL)
    RX_OP_INJECT_N(6)
    /* Initial packet is not output because we have discarded Initial keys */
    RX_OP_CHECK_PKT_N(6b)
    RX_OP_CHECK_PKT_N(6c)
    RX_OP_CHECK_NO_PKT()
    /* Try again with discarded keys */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_HANDSHAKE)
    RX_OP_INJECT_N(6)
    RX_OP_CHECK_PKT_N(6c)
    RX_OP_CHECK_NO_PKT()
    /* Try again */
    RX_OP_INJECT_N(6)
    RX_OP_CHECK_PKT_N(6c)
    RX_OP_CHECK_NO_PKT()
    /* Try again with discarded 1-RTT keys */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_1RTT)
    RX_OP_INJECT_N(6)
    RX_OP_CHECK_NO_PKT()

    /* Recreate QRL, test reading packets received before key */
    RX_OP_SET_SCID_LEN(0)
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_INJECT_N(6)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_6_c2s_init_dcid)
    RX_OP_CHECK_PKT_N(6a)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_AES256GCM, rx_script_6_handshake_secret)
    RX_OP_CHECK_PKT_N(6b)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_AES256GCM, rx_script_6_1rtt_secret)
    RX_OP_CHECK_PKT_N(6c)
    RX_OP_CHECK_NO_PKT()

    RX_OP_END
};

/*
 * 7. Real World - S2C Multiple Packets
 *      - Initial, Handshake, 1-RTT (ChaCha20-Poly1305)
 */
#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
static const QUIC_CONN_ID rx_script_7_c2s_init_dcid = {
    4, {0xfa, 0x5d, 0xd6, 0x80}
};

static const unsigned char rx_script_7_handshake_secret[32] = {
    0x85, 0x44, 0xa4, 0x02, 0x46, 0x5b, 0x2a, 0x92, 0x80, 0x71, 0xfd, 0x11,
    0x89, 0x73, 0x84, 0xeb, 0x3e, 0x0d, 0x89, 0x4f, 0x71, 0xdc, 0x9c, 0xdd,
    0x55, 0x77, 0x9e, 0x79, 0x7b, 0xeb, 0xfa, 0x86,
};

static const unsigned char rx_script_7_1rtt_secret[32] = {
    0x4a, 0x77, 0xb6, 0x0e, 0xfd, 0x90, 0xca, 0xbf, 0xc0, 0x1a, 0x64, 0x9f,
    0xc0, 0x03, 0xd3, 0x8d, 0xc5, 0x41, 0x04, 0x50, 0xb1, 0x5b, 0x74, 0xe7,
    0xe3, 0x99, 0x0c, 0xdf, 0x74, 0x61, 0x35, 0xe6,
};

static const unsigned char rx_script_7_in[] = {
    /* First Packet: Initial */
    0xc2,                           /* Long, Initial, PN Length=2 bytes */
    0x00, 0x00, 0x00, 0x01,         /* Version */
    0x00,                           /* DCID */
    0x04, 0x03, 0x45, 0x0c, 0x7a,   /* SCID */
    0x00,                           /* Token Length */
    0x41, 0xcb,                     /* Length (459) */
    0x3c, 0xe0,                     /* PN (0) */
    0x85, 0x05, 0xc2, 0x4d, 0x0f, 0xf3, 0x62, 0x51, 0x04, 0x33, 0xfa, 0xb5,
    0xa3, 0x02, 0xbd, 0x5c, 0x22, 0x0c, 0x1d, 0xda, 0x06, 0xf1, 0xd7, 0xe0,
    0xc8, 0x56, 0xb0, 0x3d, 0xc1, 0x49, 0x8c, 0xc2, 0x88, 0x5a, 0x0e, 0xd5,
    0x67, 0x72, 0xec, 0xcc, 0x7a, 0x2b, 0x46, 0x17, 0x49, 0x4b, 0x28, 0x6a,
    0x89, 0x71, 0xfd, 0x31, 0x9a, 0xa1, 0x97, 0x64, 0xe2, 0xbf, 0xa0, 0x6d,
    0xf6, 0x76, 0x83, 0x28, 0xc4, 0xd5, 0x39, 0x87, 0x22, 0x7c, 0x11, 0x9a,
    0x53, 0x66, 0xb4, 0x27, 0xf1, 0xab, 0x6f, 0x49, 0x43, 0x3f, 0x9a, 0x23,
    0xd3, 0x53, 0x06, 0xe8, 0x14, 0xfd, 0xc0, 0x67, 0x1f, 0x88, 0x2a, 0xa8,
    0xae, 0x5f, 0x05, 0x0a, 0xeb, 0x66, 0x72, 0x8c, 0x46, 0xcc, 0x54, 0x21,
    0x5e, 0x14, 0xfe, 0x68, 0xc7, 0xf7, 0x60, 0x67, 0xb5, 0xa7, 0x0d, 0xf4,
    0xe1, 0xff, 0x60, 0xe3, 0x11, 0x38, 0x92, 0x90, 0xc2, 0x48, 0x28, 0xbf,
    0xf3, 0x85, 0x27, 0xfe, 0xbf, 0x42, 0x26, 0x1a, 0x4e, 0x78, 0xf1, 0xf0,
    0x88, 0x16, 0x1b, 0x64, 0x5f, 0x66, 0x02, 0x0b, 0x45, 0x3d, 0x38, 0xd9,
    0x09, 0xd5, 0xff, 0xc2, 0x68, 0x02, 0x2c, 0xc4, 0x3f, 0x60, 0x6e, 0x2f,
    0x7f, 0x43, 0xf7, 0x1a, 0x37, 0xcc, 0xe0, 0xe0, 0x4b, 0x96, 0xc1, 0xb1,
    0x8b, 0x1c, 0x7c, 0x6e, 0x80, 0xe3, 0x92, 0x9b, 0x86, 0x87, 0x1f, 0x9a,
    0x6a, 0x62, 0x18, 0xf4, 0x86, 0xc2, 0x3e, 0x33, 0xa3, 0xbf, 0x43, 0x96,
    0x6e, 0xff, 0x94, 0xaf, 0x6d, 0x23, 0x5c, 0x42, 0xed, 0xe7, 0xb9, 0x2c,
    0x33, 0xb0, 0xc6, 0x3d, 0x44, 0x00, 0x0b, 0xa3, 0x39, 0xa8, 0xeb, 0x8c,
    0x81, 0x1a, 0x99, 0x20, 0xbd, 0xfa, 0xf3, 0xf4, 0xf0, 0x11, 0xd8, 0x41,
    0x31, 0x8d, 0xdc, 0x0d, 0x00, 0xa6, 0x31, 0x40, 0xc6, 0xc6, 0xad, 0x74,
    0x93, 0x62, 0x1c, 0x55, 0xce, 0x5f, 0x8c, 0x5b, 0x3c, 0xcb, 0x25, 0x5e,
    0xbf, 0xed, 0xbb, 0x3c, 0x97, 0x4b, 0x62, 0xe0, 0xba, 0xf1, 0xb0, 0x30,
    0xbf, 0x35, 0x89, 0x7e, 0x25, 0x61, 0x54, 0x86, 0x52, 0x11, 0x86, 0x90,
    0xc3, 0xf5, 0xad, 0xa0, 0x96, 0x30, 0xb2, 0xf0, 0xa6, 0x79, 0x39, 0x1c,
    0x51, 0x42, 0xa1, 0x00, 0x6f, 0x55, 0x7d, 0xdc, 0xd0, 0x7c, 0xcf, 0x01,
    0x88, 0x03, 0xd7, 0x2d, 0x65, 0x2b, 0x40, 0xee, 0xba, 0x10, 0xd8, 0x0c,
    0x85, 0x14, 0xb7, 0x4d, 0x9e, 0x7d, 0x7c, 0xde, 0x7f, 0x0d, 0x0e, 0x3b,
    0x3d, 0xe3, 0xd3, 0x63, 0xc2, 0xed, 0xc7, 0x41, 0xaf, 0x05, 0x85, 0x87,
    0x46, 0x55, 0x7e, 0xbe, 0x14, 0x5b, 0x98, 0xae, 0x6e, 0x67, 0x1a, 0x65,
    0xc6, 0xcf, 0xe1, 0x28, 0x50, 0x6b, 0xb4, 0xf6, 0xba, 0x63, 0xbc, 0xf1,
    0xd7, 0xa4, 0x97, 0x2d, 0x4d, 0x04, 0x26, 0x96, 0xec, 0x0c, 0xd4, 0xae,
    0x6a, 0xca, 0x7e, 0x65, 0xc5, 0x43, 0x7e, 0xf8, 0x77, 0x61, 0xd0, 0x2c,
    0xe5, 0x37, 0x0a, 0xb3, 0x7a, 0x8c, 0x2a, 0xa1, 0xdc, 0x29, 0xdb, 0xec,
    0xca, 0xdc, 0xfe, 0xdd, 0x38, 0xd2, 0x13, 0x9f, 0x94, 0x6d, 0x5b, 0x87,
    0xf3, 0x15, 0xa8, 0xe5, 0xe9, 0x65, 0x1d, 0x4f, 0x92, 0x1b, 0xf4, 0xa6,
    0xa4, 0xd6, 0x22, 0xfc, 0x26, 0x1b, 0x35, 0xa4, 0x1c, 0x88, 0x9f, 0x7d,
    0xe0, 0x9a, 0x89, 0x0f, 0x6c, 0xc1, 0xda, 0x6e, 0x45, 0xce, 0x74, 0xb1,
    0xff,

    /* Second Packet: Handshake */
    0xeb,                           /* Long, Handshake, PN Length=2 bytes */
    0x00, 0x00, 0x00, 0x01,         /* Version */
    0x00,                           /* DCID */
    0x04, 0x03, 0x45, 0x0c, 0x7a,   /* SCID */
    0x42, 0xa3,                     /* Length (675) */
    0x43, 0x29,                     /* PN (0) */
    0xff, 0xdb, 0xcf, 0x3c, 0x17, 0xcf, 0xdc, 0x42, 0x3a, 0x59, 0x88, 0xdb,
    0x13, 0xef, 0x09, 0x3d, 0xf2, 0x24, 0xf3, 0xeb, 0xca, 0xb0, 0xe1, 0xa4,
    0x67, 0x64, 0x65, 0x80, 0x5f, 0x73, 0x29, 0x69, 0x29, 0xba, 0x03, 0x77,
    0x22, 0xc8, 0xa8, 0xd5, 0x21, 0xf2, 0xa2, 0x30, 0x7f, 0x86, 0x3a, 0x8a,
    0xdd, 0x92, 0x33, 0xa6, 0x57, 0x21, 0x39, 0xdd, 0x34, 0xb4, 0x39, 0xa7,
    0x6f, 0x0a, 0x14, 0xba, 0x9e, 0x3b, 0x3a, 0x6a, 0x4b, 0xc5, 0xda, 0x44,
    0x82, 0xca, 0x52, 0x86, 0x68, 0x8a, 0x0c, 0x5e, 0xeb, 0x1e, 0x81, 0x43,
    0x3a, 0x59, 0x2c, 0x26, 0x63, 0xa3, 0x89, 0x92, 0x80, 0xe9, 0x75, 0xc2,
    0xdb, 0xb9, 0x58, 0x6d, 0xab, 0xfd, 0x21, 0xe0, 0x35, 0x79, 0x2e, 0x56,
    0x7b, 0xfb, 0xb3, 0x7a, 0x05, 0x33, 0x0f, 0x13, 0xe5, 0xef, 0x04, 0x41,
    0x69, 0x85, 0x91, 0x24, 0xce, 0xb5, 0x21, 0x8d, 0x0a, 0x13, 0xda, 0xae,
    0x86, 0x2f, 0x25, 0x1f, 0x9c, 0x70, 0x8a, 0xaa, 0x05, 0xeb, 0x30, 0x93,
    0x50, 0xc1, 0x39, 0xab, 0x99, 0x8a, 0x31, 0xc1, 0xc1, 0x5e, 0x39, 0xcf,
    0x64, 0x3f, 0x9f, 0x5c, 0xa5, 0xa1, 0x88, 0xb2, 0x5f, 0x23, 0xcb, 0x76,
    0xe5, 0xf3, 0x2d, 0xa0, 0xed, 0xad, 0xcf, 0x30, 0x05, 0x44, 0xdc, 0xa5,
    0x81, 0xb1, 0x7f, 0x78, 0x0d, 0x4d, 0x96, 0xa3, 0xcb, 0xcb, 0x45, 0xcf,
    0x5f, 0x22, 0xb8, 0x93, 0x2b, 0x16, 0xe0, 0x1c, 0x53, 0x34, 0x76, 0x3b,
    0x7b, 0x78, 0xa1, 0x46, 0x40, 0x43, 0x4b, 0x0e, 0x1c, 0xfd, 0xcf, 0x01,
    0xf1, 0x2c, 0xee, 0xd0, 0xbd, 0x9f, 0x44, 0xd2, 0xd7, 0x13, 0xf9, 0x65,
    0x82, 0xf5, 0x42, 0xec, 0x9f, 0x5d, 0x51, 0x5a, 0x7b, 0xf2, 0x39, 0xbb,
    0xa6, 0x19, 0x5c, 0x73, 0x95, 0x65, 0x5b, 0x64, 0x2f, 0xda, 0x50, 0xd0,
    0x02, 0x34, 0x3f, 0x35, 0xc1, 0xd6, 0x31, 0x3b, 0xcf, 0x3f, 0x81, 0x8d,
    0xe0, 0x40, 0xfd, 0x6d, 0x32, 0x68, 0xa4, 0xf2, 0x4e, 0x3a, 0x4a, 0x42,
    0x2c, 0x07, 0x2d, 0x27, 0xa3, 0x34, 0xe7, 0x27, 0x87, 0x80, 0x76, 0xc0,
    0xa0, 0x72, 0x05, 0xf2, 0x88, 0x81, 0xe3, 0x32, 0x00, 0x76, 0x8d, 0x24,
    0x5c, 0x97, 0x2d, 0xd6, 0xb8, 0x34, 0xf8, 0x1c, 0x1a, 0x6d, 0xc7, 0x3f,
    0xcf, 0x56, 0xae, 0xec, 0x26, 0x74, 0x53, 0x69, 0xcd, 0x7a, 0x97, 0x29,
    0xab, 0x12, 0x7d, 0x75, 0xf8, 0x8d, 0x5b, 0xc0, 0x77, 0x20, 0xb6, 0x6a,
    0x0b, 0xce, 0x98, 0x50, 0xca, 0x47, 0x42, 0x1e, 0x5d, 0xc3, 0x24, 0x5a,
    0x47, 0x48, 0x3b, 0xa0, 0x9e, 0x43, 0xe9, 0x8d, 0x18, 0x23, 0xda, 0x6f,
    0x8c, 0xda, 0xd0, 0x3e, 0xdb, 0x37, 0xff, 0xfc, 0x7e, 0x17, 0xbe, 0x42,
    0xfd, 0xdb, 0x51, 0xb1, 0xa4, 0xfd, 0x9a, 0x20, 0x27, 0x24, 0x17, 0x04,
    0x70, 0xb6, 0x21, 0x87, 0x88, 0xe9, 0xda, 0x63, 0xcb, 0xcb, 0x1d, 0xaf,
    0x4a, 0x46, 0x76, 0x88, 0xa1, 0xf8, 0x48, 0x6c, 0x06, 0xb4, 0x62, 0x1a,
    0x67, 0x18, 0xb0, 0x1d, 0x58, 0x6a, 0xfe, 0x1f, 0xf1, 0x48, 0xff, 0xcb,
    0xa4, 0xd1, 0xa8, 0x12, 0x1f, 0x45, 0x94, 0x2f, 0x55, 0x80, 0x6a, 0x06,
    0xcc, 0x7b, 0xb0, 0xcc, 0xb8, 0x06, 0x52, 0x16, 0xe3, 0x6e, 0x7e, 0xb0,
    0x42, 0xfd, 0x3b, 0x7e, 0x0a, 0x42, 0x7b, 0x73, 0xaf, 0x2c, 0xf3, 0xbd,
    0xe5, 0x72, 0x8c, 0x16, 0xb2, 0xd7, 0x7a, 0x11, 0xb6, 0x9f, 0xd1, 0x69,
    0xc1, 0x1a, 0xe0, 0x26, 0x26, 0x13, 0xe2, 0x75, 0xf5, 0x74, 0xae, 0x3f,
    0xee, 0x1e, 0x09, 0x63, 0x5a, 0x30, 0x19, 0xa5, 0x59, 0x48, 0x90, 0x9b,
    0x46, 0x56, 0xd8, 0x6f, 0x6b, 0x76, 0x82, 0x32, 0xc7, 0x29, 0x76, 0x2e,
    0x32, 0xb6, 0x23, 0x99, 0xeb, 0x92, 0x5d, 0xc4, 0x4c, 0xa1, 0xe9, 0x26,
    0x37, 0x9a, 0x7d, 0x4c, 0x16, 0x9c, 0x18, 0xe9, 0xc0, 0xff, 0x48, 0x79,
    0xb1, 0x7b, 0x0b, 0x1e, 0x6f, 0xb1, 0x77, 0xa5, 0xd2, 0xc6, 0x9a, 0xa9,
    0xfc, 0xd1, 0x0f, 0x69, 0xf3, 0xe0, 0x49, 0x70, 0x57, 0x80, 0x86, 0xa7,
    0x3f, 0x54, 0xa8, 0x60, 0xfb, 0xe4, 0x06, 0xa3, 0x13, 0xb9, 0x2f, 0xa7,
    0x37, 0x80, 0x0c, 0x43, 0xac, 0x2f, 0xae, 0x6e, 0x62, 0x2b, 0x53, 0xe4,
    0xfe, 0x58, 0xd7, 0x8b, 0x96, 0xdc, 0xe6, 0xd3, 0x86, 0xb8, 0xd6, 0x42,
    0x5b, 0x68, 0x03, 0x48, 0x3f, 0xcd, 0xee, 0x39, 0x8b, 0xc4, 0x53, 0x30,
    0x87, 0x48, 0x2a, 0x01, 0x9d, 0x6f, 0x8e, 0x36, 0x75, 0x73, 0xef, 0x77,
    0x3a, 0x82, 0xd8, 0x4c, 0x0e, 0x7f, 0xb3, 0x8f, 0x16, 0xd1, 0x10, 0xcf,
    0x2f, 0xa3, 0xdf, 0x65, 0xba, 0x91, 0x79, 0xf6, 0x93, 0x60, 0x08, 0xe5,
    0xdb, 0x73, 0x02, 0x7a, 0x0b, 0x0e, 0xcc, 0x3b, 0x1f, 0x08, 0x2d, 0x51,
    0x3e, 0x87, 0x48, 0xd3, 0xd3, 0x75, 0xc2, 0x28, 0xa3, 0xf3, 0x02, 0xde,
    0x8f, 0xa6, 0xbd, 0xb3, 0x19, 0xa0, 0xdb, 0x48, 0x51, 0x03, 0x5f, 0x98,
    0xbe,

    /* Third Packet: 1-RTT */
    0x5c,               /* Short, 1-RTT, Spin=0, KP=0, PN Length=2 bytes */
    0x4f, 0x33,         /* PN (0) */
    0x16, 0x75, 0x98, 0x67, 0x04, 0x16, 0x61, 0xe3, 0x00, 0xb7, 0x9d, 0x5c,
    0x53, 0x4c, 0x26, 0x90, 0x92, 0x8e, 0x0e, 0xc0, 0x9c, 0x6d, 0x8b, 0xac,
    0x15, 0x6d, 0x89, 0x74, 0x2f, 0xe7, 0x84, 0xe3, 0x46, 0x46, 0x8c, 0xc1,
    0x21, 0x7c, 0x44, 0xa5, 0x00, 0x29, 0xca, 0xf2, 0x11, 0x18, 0xe0, 0x04,
    0x40, 0x55, 0xd2, 0xa7, 0xe5, 0x9d, 0x22, 0xa2, 0x2a, 0x6c, 0x03, 0x87,
    0xa3, 0xa3, 0xfa, 0xf5, 0x6c, 0xd7, 0x7d, 0xae, 0x3f, 0x28, 0x01, 0xae,
    0x06, 0x11, 0x69, 0x67, 0x90, 0x57, 0x5a, 0xd0, 0xeb, 0xdd, 0xac, 0xbd,
    0x7f, 0x33, 0x86, 0xbb,
};

static const QUIC_PKT_HDR rx_script_7a_expect_hdr = {
    QUIC_PKT_TYPE_INITIAL,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0x03, 0x45, 0x0c, 0x7a}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    441, NULL
};

static const unsigned char rx_script_7a_body[] = {
    0x02, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06,
    0x00, 0x40, 0x5a, 0x02, 0x00, 0x00, 0x56, 0x03, 0x03, 0xd5, 0xfb, 0x6a,
    0x81, 0x1c, 0xdb, 0xa2, 0x5c, 0x11, 0x31, 0xda, 0x15, 0x28, 0x97, 0x94,
    0x83, 0xfd, 0x9d, 0x91, 0x0e, 0x87, 0x71, 0x46, 0x64, 0xb4, 0xd9, 0x9e,
    0xbd, 0xa8, 0x48, 0x32, 0xbf, 0x00, 0x13, 0x03, 0x00, 0x00, 0x2e, 0x00,
    0x2b, 0x00, 0x02, 0x03, 0x04, 0x00, 0x33, 0x00, 0x24, 0x00, 0x1d, 0x00,
    0x20, 0xef, 0xbb, 0x46, 0xe9, 0xb4, 0xf6, 0x54, 0xc4, 0x07, 0x71, 0xdc,
    0x50, 0xd5, 0x69, 0x40, 0xbc, 0x85, 0x7f, 0xf9, 0x48, 0x14, 0xe3, 0xd6,
    0x08, 0xa9, 0x0b, 0xfd, 0xbe, 0xf1, 0x57, 0x21, 0x34,
};

static const QUIC_PKT_HDR rx_script_7b_expect_hdr = {
    QUIC_PKT_TYPE_HANDSHAKE,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    1,          /* Version */
    {0, {0}},                           /* DCID */
    {4, {0x03, 0x45, 0x0c, 0x7a}},      /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    657, NULL
};

static const unsigned char rx_script_7b_body[] = {
    0x06, 0x00, 0x42, 0x8d, 0x08, 0x00, 0x00, 0x82, 0x00, 0x80, 0x00, 0x10,
    0x00, 0x08, 0x00, 0x06, 0x05, 0x64, 0x75, 0x6d, 0x6d, 0x79, 0x00, 0x39,
    0x00, 0x70, 0x46, 0x0a, 0x0d, 0xdc, 0x59, 0xf0, 0x4e, 0xb2, 0x2c, 0xac,
    0x69, 0x6a, 0xc9, 0x77, 0xa9, 0x99, 0x05, 0x04, 0x80, 0x08, 0x00, 0x00,
    0x06, 0x04, 0x80, 0x08, 0x00, 0x00, 0x07, 0x04, 0x80, 0x08, 0x00, 0x00,
    0x04, 0x04, 0x80, 0x0c, 0x00, 0x00, 0x08, 0x02, 0x40, 0x64, 0x09, 0x02,
    0x40, 0x64, 0x01, 0x04, 0x80, 0x00, 0x75, 0x30, 0x03, 0x02, 0x45, 0xac,
    0x0b, 0x01, 0x1a, 0x0c, 0x00, 0x02, 0x10, 0x42, 0xf0, 0xed, 0x09, 0x07,
    0x5b, 0xd9, 0x5a, 0xb2, 0x39, 0x5d, 0x73, 0x2c, 0x57, 0x1f, 0x50, 0x00,
    0x0b, 0xe0, 0x3e, 0xf3, 0xd6, 0x91, 0x6f, 0x9c, 0xcc, 0x31, 0xf7, 0xa5,
    0x0e, 0x01, 0x04, 0x0f, 0x04, 0x03, 0x45, 0x0c, 0x7a, 0x10, 0x04, 0xfa,
    0x5d, 0xd6, 0x80, 0x20, 0x01, 0x00, 0x0b, 0x00, 0x01, 0x8f, 0x00, 0x00,
    0x01, 0x8b, 0x00, 0x01, 0x86, 0x30, 0x82, 0x01, 0x82, 0x30, 0x82, 0x01,
    0x29, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x14, 0x0a, 0x73, 0x0f, 0x86,
    0x18, 0xf2, 0xc3, 0x30, 0x01, 0xd2, 0xc0, 0xc1, 0x62, 0x52, 0x13, 0xf1,
    0x9c, 0x13, 0x39, 0xb5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x04, 0x03, 0x02, 0x30, 0x17, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x0c, 0x6d, 0x61, 0x70, 0x61, 0x6b, 0x74, 0x2e,
    0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x32, 0x30,
    0x38, 0x30, 0x32, 0x31, 0x32, 0x30, 0x30, 0x31, 0x38, 0x5a, 0x17, 0x0d,
    0x32, 0x32, 0x30, 0x39, 0x30, 0x31, 0x31, 0x32, 0x30, 0x30, 0x31, 0x38,
    0x5a, 0x30, 0x17, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03,
    0x0c, 0x0c, 0x6d, 0x61, 0x70, 0x61, 0x6b, 0x74, 0x2e, 0x6c, 0x6f, 0x63,
    0x61, 0x6c, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x67, 0xf4, 0xd3, 0x8f, 0x15, 0x6d, 0xee,
    0x85, 0xcc, 0x2a, 0x77, 0xfc, 0x0b, 0x8f, 0x9f, 0xcf, 0xa9, 0x95, 0x5d,
    0x5b, 0xcd, 0xb7, 0x8b, 0xba, 0x31, 0x0a, 0x73, 0x62, 0xc5, 0xd0, 0x0e,
    0x07, 0x90, 0xae, 0x38, 0x43, 0x79, 0xce, 0x5e, 0x33, 0xad, 0x31, 0xbf,
    0x9f, 0x2a, 0x56, 0x83, 0xa5, 0x24, 0x16, 0xab, 0x0c, 0xf1, 0x64, 0xbe,
    0xe4, 0x93, 0xb5, 0x89, 0xd6, 0x05, 0xe4, 0xf7, 0x7b, 0xa3, 0x53, 0x30,
    0x51, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14,
    0x02, 0x64, 0x0f, 0x55, 0x69, 0x14, 0x91, 0x19, 0xed, 0xf9, 0x1a, 0xe9,
    0x1d, 0xa5, 0x5a, 0xd0, 0x48, 0x96, 0x9f, 0x60, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x02, 0x64, 0x0f,
    0x55, 0x69, 0x14, 0x91, 0x19, 0xed, 0xf9, 0x1a, 0xe9, 0x1d, 0xa5, 0x5a,
    0xd0, 0x48, 0x96, 0x9f, 0x60, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13,
    0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0a,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47,
    0x00, 0x30, 0x44, 0x02, 0x20, 0x0a, 0x82, 0x92, 0x6e, 0xd3, 0xc6, 0x66,
    0xd9, 0xd3, 0x75, 0xff, 0x71, 0x3b, 0x61, 0x46, 0x21, 0x00, 0xe6, 0x21,
    0x5d, 0x9c, 0x86, 0xe9, 0x65, 0x40, 0x4f, 0xeb, 0x70, 0x4f, 0x2c, 0xad,
    0x00, 0x02, 0x20, 0x08, 0xc2, 0x07, 0x5d, 0x16, 0xfc, 0x54, 0x34, 0x2b,
    0xb4, 0x18, 0x67, 0x44, 0x81, 0xc9, 0xa9, 0x67, 0x2e, 0xce, 0xa1, 0x02,
    0x9f, 0x3b, 0xe5, 0x61, 0x16, 0x0b, 0x50, 0xf6, 0xa1, 0x50, 0x94, 0x00,
    0x00, 0x0f, 0x00, 0x00, 0x4c, 0x04, 0x03, 0x00, 0x48, 0x30, 0x46, 0x02,
    0x21, 0x00, 0xaa, 0x18, 0x61, 0x93, 0xdf, 0xbb, 0x79, 0xe7, 0x34, 0x7e,
    0x2e, 0x61, 0x13, 0x8c, 0xa0, 0x33, 0xfb, 0x33, 0xca, 0xfc, 0xd2, 0x45,
    0xb0, 0xc7, 0x89, 0x3d, 0xf1, 0xd6, 0x54, 0x94, 0x05, 0xb6, 0x02, 0x21,
    0x00, 0xef, 0x6c, 0xb6, 0xf2, 0x00, 0xb2, 0x32, 0xb1, 0xf3, 0x3f, 0x59,
    0xf5, 0xc8, 0x18, 0xbe, 0x39, 0xbb, 0x27, 0xf8, 0x67, 0xac, 0xcb, 0x63,
    0xa4, 0x29, 0xfb, 0x8e, 0x88, 0x0f, 0xe5, 0xe9, 0x7e, 0x14, 0x00, 0x00,
    0x20, 0xfc, 0x2c, 0x4c, 0xa7, 0x77, 0x24, 0x79, 0x29, 0xa8, 0x82, 0x1a,
    0x4d, 0x58, 0x9d, 0x82, 0xe2, 0x09, 0x36, 0x63, 0x0e, 0x0b, 0x55, 0x51,
    0x80, 0x93, 0x40, 0xda, 0x41, 0x33, 0x08, 0x10, 0x2c,
};

static const QUIC_PKT_HDR rx_script_7c_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},                           /* DCID */
    {0, {0}},                           /* SCID */
    {0},        /* PN */
    NULL, 0,    /* Token/Token Len */
    72, NULL
};

static const unsigned char rx_script_7c_body[] = {
    0x18, 0x03, 0x00, 0x04, 0xf7, 0x75, 0x72, 0xa2, 0xfd, 0x17, 0xd4, 0x82,
    0x8e, 0xe9, 0x5b, 0xce, 0xed, 0xec, 0x88, 0xb9, 0x73, 0xbf, 0x36, 0x9f,
    0x18, 0x02, 0x00, 0x04, 0x5f, 0x43, 0x96, 0xe4, 0x15, 0xdc, 0x56, 0x6b,
    0x67, 0x4c, 0x36, 0xb2, 0xe2, 0x77, 0xdc, 0x6e, 0xb9, 0x2c, 0x0d, 0x79,
    0x18, 0x01, 0x00, 0x04, 0xcb, 0x83, 0x4a, 0xf4, 0x8d, 0x7b, 0x69, 0x90,
    0xaf, 0x0d, 0xd2, 0x38, 0xa4, 0xf1, 0x94, 0xff, 0x63, 0x24, 0xd3, 0x7a,
};

static const struct rx_test_op rx_script_7[] = {
    RX_OP_ALLOW_1RTT()
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_7_c2s_init_dcid)
    RX_OP_INJECT_N(7)
    RX_OP_CHECK_PKT_N(7a)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_CHACHA20POLY1305, rx_script_7_handshake_secret)
    RX_OP_CHECK_PKT_N(7b)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_CHACHA20POLY1305, rx_script_7_1rtt_secret)
    RX_OP_CHECK_PKT_N(7c)
    RX_OP_CHECK_NO_PKT()

    /* Discard Initial EL and try injecting the packet again */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_INITIAL)
    RX_OP_INJECT_N(7)
    /* Initial packet is not output because we have discarded Initial keys */
    RX_OP_CHECK_PKT_N(7b)
    RX_OP_CHECK_PKT_N(7c)
    RX_OP_CHECK_NO_PKT()
    /* Try again with discarded keys */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_HANDSHAKE)
    RX_OP_INJECT_N(7)
    RX_OP_CHECK_PKT_N(7c)
    RX_OP_CHECK_NO_PKT()
    /* Try again */
    RX_OP_INJECT_N(7)
    RX_OP_CHECK_PKT_N(7c)
    RX_OP_CHECK_NO_PKT()
    /* Try again with discarded 1-RTT keys */
    RX_OP_DISCARD_EL(QUIC_ENC_LEVEL_1RTT)
    RX_OP_INJECT_N(7)
    RX_OP_CHECK_NO_PKT()

    /* Recreate QRL, test reading packets received before key */
    RX_OP_SET_SCID_LEN(0)
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_INJECT_N(7)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_7_c2s_init_dcid)
    RX_OP_CHECK_PKT_N(7a)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_CHACHA20POLY1305, rx_script_7_handshake_secret)
    RX_OP_CHECK_PKT_N(7b)
    RX_OP_CHECK_NO_PKT()
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_CHACHA20POLY1305, rx_script_7_1rtt_secret)
    RX_OP_CHECK_PKT_N(7c)
    RX_OP_CHECK_NO_PKT()

    RX_OP_END
};
#endif /* !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305) */

/*
 * 8. Real World - S2C Multiple Packets with Peer Initiated Key Phase Update
 */
static const unsigned char rx_script_8_1rtt_secret[32] = {
    0x5f, 0x1f, 0x47, 0xea, 0xc3, 0xb2, 0xce, 0x73, 0xfb, 0xa2, 0x9f, 0xac,
    0xc3, 0xa0, 0xfe, 0x9b, 0xf3, 0xc0, 0xde, 0x5d, 0x33, 0x11, 0x1c, 0x70,
    0xdd, 0xb4, 0x06, 0xcc, 0xdf, 0x7d, 0xe9, 0x9a
};

static const unsigned char rx_script_8a_in[] = {
    0x51,                           /* Short, 1-RTT, PN Length=2 bytes, KP=0 */
    0xcb, 0xf4,                     /* PN (4) */
    0x3f, 0x68, 0x7b, 0xa8, 0x2b, 0xb9, 0xfa, 0x7d, 0xe4, 0x6b, 0x20, 0x48,
    0xd1, 0x3c, 0xcb, 0x4b, 0xef, 0xb1, 0xfd, 0x5e, 0x1b, 0x19, 0x83, 0xa9,
    0x47, 0x62, 0xc1, 0x6e, 0xef, 0x27, 0xc3, 0x9b, 0x8f, 0x3f, 0xce, 0x11,
    0x68, 0xf5, 0x73, 0x0d, 0xf2, 0xdc, 0xe0, 0x28, 0x28, 0x79, 0xa6, 0x39,
    0xc3, 0xb9, 0xd3,
};

static const QUIC_PKT_HDR rx_script_8a_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},   /* DCID */
    {0, {0}},   /* SCID */
    {0, 4},     /* PN */
    NULL, 0,    /* Token/Token Len */
    35, NULL
};

static const unsigned char rx_script_8a_body[] = {
    0x02, 0x03, 0x06, 0x00, 0x03, 0x0c, 0x00, 0x1b, 0x49, 0x27, 0x6d, 0x20,
    0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f, 0x6e,
    0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65
};

static const unsigned char rx_script_8b_in[] = {
    0x52,                           /* Short, 1-RTT, PN Length=2 bytes, KP=1 */
    0x21, 0x8e,                     /* PN (5) */
    0xa2, 0x6a, 0x9c, 0x83, 0x24, 0x48, 0xae, 0x60, 0x1e, 0xc2, 0xa5, 0x91,
    0xfa, 0xe5, 0xf2, 0x05, 0x14, 0x37, 0x04, 0x6a, 0xa8, 0xae, 0x06, 0x58,
    0xd7, 0x85, 0x48, 0xd7, 0x3b, 0x85, 0x9e, 0x5a, 0xb3, 0x46, 0x89, 0x1b,
    0x4b, 0x6e, 0x1d, 0xd1, 0xfc, 0xb7, 0x47, 0xda, 0x6a, 0x64, 0x4b, 0x8e,
    0xf2, 0x69, 0x16,
};

static const QUIC_PKT_HDR rx_script_8b_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    1,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},   /* DCID */
    {0, {0}},   /* SCID */
    {0, 5},     /* PN */
    NULL, 0,    /* Token/Token Len */
    35, NULL
};

static const unsigned char rx_script_8b_body[] = {
    0x02, 0x04, 0x03, 0x00, 0x00, 0x0c, 0x00, 0x36, 0x49, 0x27, 0x6d, 0x20,
    0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f, 0x6e,
    0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char rx_script_8c_in[] = {
    0x5b,                           /* Short, 1-RTT, PN Length=2 bytes, KP=0 */
    0x98, 0xd6,                     /* PN (3) */
    0x3c, 0x6f, 0x94, 0x20, 0x5e, 0xfc, 0x5b, 0x3a, 0x4a, 0x65, 0x1a, 0x9a,
    0x6c, 0x00, 0x52, 0xb6, 0x0c, 0x9b, 0x07, 0xf9, 0x6f, 0xbc, 0x3d, 0xb4,
    0x57, 0xe0, 0x15, 0x74, 0xfe, 0x76, 0xea, 0x1f, 0x23, 0xae, 0x22, 0x62,
    0xb7, 0x90, 0x94, 0x89, 0x38, 0x9b, 0x5b, 0x47, 0xed,
};

static const QUIC_PKT_HDR rx_script_8c_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},   /* DCID */
    {0, {0}},   /* SCID */
    {0, 3},     /* PN */
    NULL, 0,    /* Token/Token Len */
    29, NULL
};

static const unsigned char rx_script_8c_body[] = {
    0x08, 0x00, 0x49, 0x27, 0x6d, 0x20, 0x68, 0x61, 0x76, 0x69, 0x6e, 0x67,
    0x20, 0x61, 0x20, 0x77, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x66, 0x75, 0x6c,
    0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char rx_script_8d_in[] = {
    0x55,                           /* Short, 1-RTT, PN Length=2 bytes, KP=1 */
    0x98, 0x20,                     /* PN (6) */
    0x45, 0x53, 0x05, 0x29, 0x30, 0x42, 0x29, 0x02, 0xf2, 0xa7, 0x27, 0xd6,
    0xb0, 0xb7, 0x30, 0xad, 0x45, 0xd8, 0x73, 0xd7, 0xe3, 0x65, 0xee, 0xd9,
    0x35, 0x33, 0x03, 0x3a, 0x35, 0x0b, 0x59, 0xa7, 0xbc, 0x23, 0x37, 0xc2,
    0x5e, 0x13, 0x88, 0x18, 0x79, 0x94, 0x6c, 0x15, 0xe3, 0x1f, 0x0d, 0xd1,
    0xc3, 0xfa, 0x40, 0xff,
};

static const QUIC_PKT_HDR rx_script_8d_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    1,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},   /* DCID */
    {0, {0}},   /* SCID */
    {0, 6},     /* PN */
    NULL, 0,    /* Token/Token Len */
    36, NULL
};

static const unsigned char rx_script_8d_body[] = {
    0x02, 0x05, 0x03, 0x00, 0x00, 0x0c, 0x00, 0x40, 0x51, 0x49, 0x27, 0x6d,
    0x20, 0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f,
    0x6e, 0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char rx_script_8e_in[] = {
    0x55,                           /* Short, 1-RTTT, PN Length=2 bytes, KP=0 */
    0x76, 0x25,                     /* PN (10) */
    0x1c, 0x0d, 0x70, 0x4c, 0x2b, 0xc5, 0x7d, 0x7b, 0x77, 0x64, 0x03, 0x27,
    0xb3, 0x5d, 0x83, 0x9e, 0x35, 0x05, 0x10, 0xd2, 0xa4, 0x5c, 0x83, 0xd6,
    0x94, 0x12, 0x18, 0xc5, 0xb3, 0x0f, 0x0a, 0xb1, 0x8a, 0x82, 0x9f, 0xd6,
    0xa9, 0xab, 0x40, 0xc1, 0x05, 0xe8, 0x1b, 0x74, 0xaa, 0x8e, 0xd6, 0x8b,
    0xa5, 0xa3, 0x77, 0x79,
};

static const QUIC_PKT_HDR rx_script_8e_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    0,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},   /* DCID */
    {0, {0}},   /* SCID */
    {0, 10},    /* PN */
    NULL, 0,    /* Token/Token Len */
    36, NULL
};

static const unsigned char rx_script_8e_body[] = {
    0x02, 0x09, 0x04, 0x00, 0x00, 0x0c, 0x00, 0x40, 0xbd, 0x49, 0x27, 0x6d,
    0x20, 0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f,
    0x6e, 0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char rx_script_8f_in[] = {
    0x48,                           /* Short, 1-RTT, PN Length=2 Bytes, KP=1 */
    0x4d, 0xf6,                     /* PN (15) */
    0x42, 0x86, 0xa1, 0xfa, 0x69, 0x6b, 0x1a, 0x45, 0xf2, 0xcd, 0xf6, 0x92,
    0xe1, 0xe6, 0x1a, 0x49, 0x37, 0xd7, 0x10, 0xae, 0x09, 0xbd
};

static const QUIC_PKT_HDR rx_script_8f_expect_hdr = {
    QUIC_PKT_TYPE_1RTT,
    0,          /* Spin Bit */
    1,          /* Key Phase */
    2,          /* PN Length */
    0,          /* Partial */
    1,          /* Fixed */
    0,          /* Unused */
    0,          /* Reserved */
    0,          /* Version */
    {0, {0}},   /* DCID */
    {0, {0}},   /* SCID */
    {0, 15},    /* PN */
    NULL, 0,    /* Token/Token Len */
    6, NULL
};

static const unsigned char rx_script_8f_body[] = {
    0x02, 0x0e, 0x4c, 0x54, 0x00, 0x02
};

static const struct rx_test_op rx_script_8[] = {
    RX_OP_ALLOW_1RTT()
    RX_OP_SET_RX_DCID(empty_conn_id)
    /* Inject before we get the keys */
    RX_OP_INJECT_N(8a)
    /* Nothing yet */
    RX_OP_CHECK_NO_PKT()
    /* Provide keys */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                         QRL_SUITE_AES128GCM, rx_script_8_1rtt_secret)
    /* Now the injected packet is successfully returned */
    RX_OP_CHECK_PKT_N(8a)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(0)
    RX_OP_CHECK_PKT_EPOCH(0)

    /* Packet with new key phase */
    RX_OP_INJECT_N(8b)
    /* Packet is successfully decrypted and returned */
    RX_OP_CHECK_PKT_N(8b)
    RX_OP_CHECK_NO_PKT()
    /* Key epoch has increased */
    RX_OP_CHECK_KEY_EPOCH(1)
    RX_OP_CHECK_PKT_EPOCH(1)

    /*
     * Now inject an old packet with the old keys (perhaps reordered in
     * network).
     */
    RX_OP_INJECT_N(8c)
    /* Should still be decrypted OK */
    RX_OP_CHECK_PKT_N(8c)
    RX_OP_CHECK_NO_PKT()
    /* Epoch has not changed */
    RX_OP_CHECK_KEY_EPOCH(1)
    RX_OP_CHECK_PKT_EPOCH(0)

    /* Another packet with the new keys. */
    RX_OP_INJECT_N(8d)
    RX_OP_CHECK_PKT_N(8d)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(1)
    RX_OP_CHECK_PKT_EPOCH(1)

    /* We can inject the old packet multiple times and it still works */
    RX_OP_INJECT_N(8c)
    RX_OP_CHECK_PKT_N(8c)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(1)
    RX_OP_CHECK_PKT_EPOCH(0)

    /* Until we move from UPDATING to COOLDOWN */
    RX_OP_KEY_UPDATE_TIMEOUT(0)
    RX_OP_INJECT_N(8c)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(1)

    /*
     * Injecting a packet from the next epoch (epoch 2) while in COOLDOWN
     * doesn't work
     */
    RX_OP_INJECT_N(8e)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(1)

    /* Move from COOLDOWN to NORMAL and try again */
    RX_OP_KEY_UPDATE_TIMEOUT(1)
    RX_OP_INJECT_N(8e)
    RX_OP_CHECK_PKT_N(8e)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(2)
    RX_OP_CHECK_PKT_EPOCH(2)

    /* Can still receive old packet */
    RX_OP_INJECT_N(8d)
    RX_OP_CHECK_PKT_N(8d)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(2)
    RX_OP_CHECK_PKT_EPOCH(1)

    /* Move straight from UPDATING to NORMAL */
    RX_OP_KEY_UPDATE_TIMEOUT(1)

    /* Try a packet from epoch 3 */
    RX_OP_INJECT_N(8f)
    RX_OP_CHECK_PKT_N(8f)
    RX_OP_CHECK_NO_PKT()
    RX_OP_CHECK_KEY_EPOCH(3)
    RX_OP_CHECK_PKT_EPOCH(3)

    RX_OP_END
};

/* 9. 1-RTT Deferral Test */
static const struct rx_test_op rx_script_9[] = {
    RX_OP_SET_RX_DCID(empty_conn_id)
    RX_OP_PROVIDE_SECRET_INITIAL(rx_script_5_c2s_init_dcid)
    RX_OP_INJECT_N(5)

    RX_OP_CHECK_PKT_N(5a)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_HANDSHAKE,
                      QRL_SUITE_AES128GCM, rx_script_5_handshake_secret)
    RX_OP_CHECK_PKT_N(5b)
    RX_OP_CHECK_NO_PKT() /* not got secret for next packet yet */
    RX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT,
                      QRL_SUITE_AES128GCM, rx_script_5_1rtt_secret)
    RX_OP_CHECK_NO_PKT() /* still nothing - 1-RTT not enabled */
    RX_OP_ALLOW_1RTT()
    RX_OP_CHECK_PKT_N(5c) /* now we get the 1-RTT packet */
    RX_OP_CHECK_NO_PKT()

    RX_OP_END
};

static const struct rx_test_op *rx_scripts[] = {
    rx_script_1,
#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
    rx_script_2,
#endif
    rx_script_3,
    rx_script_4,
    rx_script_5,
    rx_script_6,
#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
    rx_script_7,
#endif
    rx_script_8,
    rx_script_9
};

struct rx_state {
    QUIC_DEMUX         *demux;

    /* OSSL_QRX with necessary data */
    OSSL_QRX           *qrx;
    OSSL_QRX_ARGS       args;

    /* Used for the RX depacketizer */
    SSL_CTX            *quic_ssl_ctx;
    QUIC_CONNECTION    *quic_conn;

    QUIC_CONN_ID        rx_dcid;

    int                 allow_1rtt;
};

static void rx_state_teardown(struct rx_state *s)
{
    if (s->quic_conn != NULL) {
        SSL_free((SSL *)s->quic_conn);
        s->quic_conn = NULL;
    }
    if (s->quic_ssl_ctx != NULL) {
        SSL_CTX_free(s->quic_ssl_ctx);
        s->quic_ssl_ctx = NULL;
    }

    if (s->qrx != NULL) {
        ossl_qrx_free(s->qrx);
        s->qrx = NULL;
    }

    if (s->demux != NULL) {
        ossl_quic_demux_free(s->demux);
        s->demux = NULL;
    }
}

static uint64_t time_counter = 0;

static OSSL_TIME expected_time(uint64_t counter)
{
    return ossl_time_multiply(ossl_ticks2time(OSSL_TIME_MS), counter);
}

static OSSL_TIME fake_time(void *arg)
{
    return expected_time(++time_counter);
}

static void demux_default_handler(QUIC_URXE *e, void *arg,
                                  const QUIC_CONN_ID *dcid)
{
    struct rx_state *s = arg;

    if (dcid == NULL || !ossl_quic_conn_id_eq(dcid, &s->rx_dcid))
        return;

    ossl_qrx_inject_urxe(s->qrx, e);
}

static int rx_state_ensure(struct rx_state *s)
{
    if (s->demux == NULL
        && !TEST_ptr(s->demux = ossl_quic_demux_new(NULL,
                                                    s->args.short_conn_id_len,
                                                    fake_time,
                                                    NULL)))
        return 0;

    s->args.demux           = s->demux;
    s->args.max_deferred    = 32;

    /* Initialise OSSL_QRX */
    if (s->qrx == NULL
        && !TEST_ptr(s->qrx = ossl_qrx_new(&s->args)))
        return 0;

    ossl_quic_demux_set_default_handler(s->demux, demux_default_handler, s);

    if (s->allow_1rtt)
        ossl_qrx_allow_1rtt_processing(s->qrx);

    return 1;
}

static int rx_run_script(const struct rx_test_op *script)
{
    int testresult = 0;
    struct rx_state s = {0};
    size_t i;
    OSSL_QRX_PKT *pkt = NULL;
    const struct rx_test_op *op = script;
    uint64_t last_key_epoch = UINT64_MAX;

    for (; op->op != RX_TEST_OP_END; ++op)
        switch (op->op) {
            case RX_TEST_OP_SET_SCID_LEN:
                rx_state_teardown(&s);
                s.args.short_conn_id_len = op->enc_level;
                break;
            case RX_TEST_OP_SET_INIT_LARGEST_PN:
                rx_state_teardown(&s);
                for (i = 0; i < QUIC_PN_SPACE_NUM; ++i)
                    s.args.init_largest_pn[i] = op->largest_pn;
                break;
            case RX_TEST_OP_SET_RX_DCID:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;
                s.rx_dcid = *op->dcid;
                break;
            case RX_TEST_OP_PROVIDE_SECRET:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;
                if (!TEST_true(ossl_qrx_provide_secret(s.qrx, op->enc_level,
                                                       op->suite_id, NULL,
                                                       op->buf,
                                                       op->buf_len)))
                    goto err;
                break;
            case RX_TEST_OP_PROVIDE_SECRET_INITIAL:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;
                if (!TEST_true(ossl_quic_provide_initial_secret(NULL, NULL,
                                                                op->dcid, 0,
                                                                s.qrx, NULL)))
                    goto err;
                break;
            case RX_TEST_OP_DISCARD_EL:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;
                if (!TEST_true(ossl_qrx_discard_enc_level(s.qrx, op->enc_level)))
                    goto err;
                break;
            case RX_TEST_OP_INJECT:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;
                if (!TEST_true(ossl_quic_demux_inject(s.demux,
                                                      op->buf, op->buf_len,
                                                      NULL, NULL)))
                    goto err;
                break;
            case RX_TEST_OP_CHECK_PKT:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;

                if (!TEST_true(ossl_qrx_read_pkt(s.qrx, &pkt)))
                    goto err;

                if (!TEST_ptr(pkt) || !TEST_ptr(pkt->hdr))
                    goto err;

                if (!TEST_mem_eq(pkt->hdr->data, pkt->hdr->len,
                                 op->buf, op->buf_len))
                    goto err;

                if (!TEST_true(cmp_pkt_hdr(pkt->hdr, op->hdr,
                                           op->buf, op->buf_len, 1)))
                    goto err;

                last_key_epoch = pkt->key_epoch;

                ossl_qrx_pkt_release(pkt);
                pkt = NULL;
                break;
            case RX_TEST_OP_CHECK_NO_PKT:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;

                if (!TEST_false(ossl_qrx_read_pkt(s.qrx, &pkt)))
                    goto err;

                break;
            case RX_TEST_OP_CHECK_KEY_EPOCH:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;

                if (!TEST_uint64_t_eq(ossl_qrx_get_key_epoch(s.qrx),
                                      op->largest_pn))
                    goto err;

                break;
            case RX_TEST_OP_CHECK_PKT_EPOCH:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;

                if (!TEST_uint64_t_eq(last_key_epoch, op->largest_pn))
                    goto err;

                break;
            case RX_TEST_OP_KEY_UPDATE_TIMEOUT:
                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;

                if (!TEST_true(ossl_qrx_key_update_timeout(s.qrx,
                                                           op->enc_level)))
                    goto err;

                break;
            case RX_TEST_OP_SET_INIT_KEY_PHASE:
                rx_state_teardown(&s);
                s.args.init_key_phase_bit = (unsigned char)op->enc_level;
                break;
            case RX_TEST_OP_ALLOW_1RTT:
                s.allow_1rtt = 1;

                if (!TEST_true(rx_state_ensure(&s)))
                    goto err;

                break;
            default:
                OPENSSL_assert(0);
                goto err;
        }

    testresult = 1;
err:
    ossl_qrx_pkt_release(pkt);
    rx_state_teardown(&s);
    return testresult;
}

static int test_rx_script(int idx)
{
    return rx_run_script(rx_scripts[idx]);
}

/* Packet Header Tests */
struct pkt_hdr_test {
    QUIC_PKT_HDR hdr;
    const unsigned char *expected;
    size_t expected_len;
    const unsigned char *payload;
    size_t payload_len;
    size_t short_conn_id_len;
    /*
     * Minimum number of bytes which should be required for a successful decode.
     * SIZE_MAX if should never decode successfully.
     */
    size_t min_success_len;
    size_t pn_offset, sample_offset;
};

/* Packet Header Test 1: INITIAL With SCID */
static const unsigned char pkt_hdr_test_1_expected[] = {
    0xc1,                     /* Long|Fixed, Type=Initial, PN Len=2 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x00,                     /* DCID Length */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x00,                     /* Token Length */
    0x15,                     /* Length=21 */
    0x33, 0x44,               /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_1_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_1 = {
    {
        QUIC_PKT_TYPE_INITIAL,  /* type */
        0,                      /* spin bit */
        0,                      /* key phase */
        2,                      /* PN length */
        0,                      /* partial */
        1,                      /* fixed */
        0,                      /* unused */
        0,                      /* reserved */
        1,                      /* version */
        { 0, {0} },             /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33, 0x44 },         /* PN */
        NULL, 0,                /* Token/Token Len */
        19, NULL                /* Len/Data */
    },
    pkt_hdr_test_1_expected, OSSL_NELEM(pkt_hdr_test_1_expected),
    pkt_hdr_test_1_payload,  OSSL_NELEM(pkt_hdr_test_1_payload),
    0, sizeof(pkt_hdr_test_1_expected),
    17, 21
};

/* Packet Header Test 2: INITIAL With SCID and Token */
static const unsigned char pkt_hdr_test_2_expected[] = {
    0xc1,                     /* Long|Fixed, Type=Initial, PN Len=2 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x00,                     /* DCID Length */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x07,                     /* Token Length */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
    0x15,                     /* Length=21 */
    0x33, 0x44,               /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_2_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_2_token[] = {
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96
};

static const struct pkt_hdr_test pkt_hdr_test_2 = {
    {
        QUIC_PKT_TYPE_INITIAL,  /* type */
        0,                      /* spin bit */
        0,                      /* key phase */
        2,                      /* PN length */
        0,                      /* partial */
        1,                      /* fixed */
        0,                      /* unused */
        0,                      /* reserved */
        1,                      /* version */
        { 0, {0} },             /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33, 0x44 },         /* PN */
        pkt_hdr_test_2_token, sizeof(pkt_hdr_test_2_token), /* Token */
        19, NULL                /* Len/Data */
    },
    pkt_hdr_test_2_expected, OSSL_NELEM(pkt_hdr_test_2_expected),
    pkt_hdr_test_2_payload,  OSSL_NELEM(pkt_hdr_test_2_payload),
    0, sizeof(pkt_hdr_test_2_expected),
    24, 28
};

/* Packet Header Test 3: INITIAL With DCID and SCID and Token */
static const unsigned char pkt_hdr_test_3_expected[] = {
    0xc1,                     /* Long|Fixed, Type=Initial, PN Len=2 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x06,                     /* Token Length */
    0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
    0x15,                     /* Length=21 */
    0x33, 0x44,               /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_3_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_3_token[] = {
    0x91, 0x92, 0x93, 0x94, 0x95, 0x96
};

static const struct pkt_hdr_test pkt_hdr_test_3 = {
    {
        QUIC_PKT_TYPE_INITIAL,  /* type */
        0,                      /* spin bit */
        0,                      /* key phase */
        2,                      /* PN length */
        0,                      /* partial */
        1,                      /* fixed */
        0,                      /* unused */
        0,                      /* reserved */
        1,                      /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33, 0x44 },         /* PN */
        pkt_hdr_test_3_token, sizeof(pkt_hdr_test_3_token), /* Token */
        19, NULL                /* Len/Data */
    },
    pkt_hdr_test_3_expected, OSSL_NELEM(pkt_hdr_test_3_expected),
    pkt_hdr_test_3_payload,  OSSL_NELEM(pkt_hdr_test_3_payload),
    0, sizeof(pkt_hdr_test_3_expected),
    26, 30
};

/* Packet Header Test 4: 0-RTT */
static const unsigned char pkt_hdr_test_4_expected[] = {
    0xd0,                     /* Long|Fixed, Type=0-RTT, PN Len=1 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x14,                     /* Length=20 */
    0x33,                     /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_4_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_4 = {
    {
        QUIC_PKT_TYPE_0RTT,     /* type */
        0,                      /* spin bit */
        0,                      /* key phase */
        1,                      /* PN length */
        0,                      /* partial */
        1,                      /* fixed */
        0,                      /* unused */
        0,                      /* reserved */
        1,                      /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33 },               /* PN */
        NULL, 0,                /* Token */
        19, NULL                /* Len/Data */
    },
    pkt_hdr_test_4_expected, OSSL_NELEM(pkt_hdr_test_4_expected),
    pkt_hdr_test_4_payload,  OSSL_NELEM(pkt_hdr_test_4_payload),
    0, sizeof(pkt_hdr_test_4_expected),
    19, 23
};

/* Packet Header Test 5: Handshake */
static const unsigned char pkt_hdr_test_5_expected[] = {
    0xe0,                     /* Long|Fixed, Type=Handshake, PN Len=1 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x14,                     /* Length=20 */
    0x33,                     /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_5_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_5 = {
    {
        QUIC_PKT_TYPE_HANDSHAKE,    /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        1,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        1,                          /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33 },                   /* PN */
        NULL, 0,                    /* Token */
        19, NULL                    /* Len/Data */
    },
    pkt_hdr_test_5_expected, OSSL_NELEM(pkt_hdr_test_5_expected),
    pkt_hdr_test_5_payload,  OSSL_NELEM(pkt_hdr_test_5_payload),
    0, sizeof(pkt_hdr_test_5_expected),
    19, 23
};

/* Packet Header Test 6: Retry */
static const unsigned char pkt_hdr_test_6_expected[] = {
    0xf0,                     /* Long|Fixed, Type=Retry */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,       /* Retry Token */
    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
    0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f              /* Retry Integrity Tag */
};

static const unsigned char pkt_hdr_test_6_payload[] = {
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,       /* Retry Token */
    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
    0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f              /* Retry Integrity Tag */
};

static const struct pkt_hdr_test pkt_hdr_test_6 = {
    {
        QUIC_PKT_TYPE_RETRY,        /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        0,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        1,                          /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0 },                      /* PN */
        NULL, 0,                    /* Token */
        24, NULL                    /* Len/Data */
    },
    pkt_hdr_test_6_expected, OSSL_NELEM(pkt_hdr_test_6_expected),
    pkt_hdr_test_6_payload,  OSSL_NELEM(pkt_hdr_test_6_payload),
    0, 21,
    SIZE_MAX, SIZE_MAX
};

/* Packet Header Test 7: 1-RTT */
static const unsigned char pkt_hdr_test_7_expected[] = {
    0x42,                     /* Short|Fixed, Type=1-RTT, PN Len=3 */
    0x70, 0x71, 0x72,         /* DCID */
    0x50, 0x51, 0x52,         /* PN */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const unsigned char pkt_hdr_test_7_payload[] = {
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const struct pkt_hdr_test pkt_hdr_test_7 = {
    {
        QUIC_PKT_TYPE_1RTT,         /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        3,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 0, {0} },                 /* SCID */
        { 0x50, 0x51, 0x52 },       /* PN */
        NULL, 0,                    /* Token */
        18, NULL                    /* Len/Data */
    },
    pkt_hdr_test_7_expected, OSSL_NELEM(pkt_hdr_test_7_expected),
    pkt_hdr_test_7_payload,  OSSL_NELEM(pkt_hdr_test_7_payload),
    3, 21,
    4, 8
};

/* Packet Header Test 8: 1-RTT with Spin Bit */
static const unsigned char pkt_hdr_test_8_expected[] = {
    0x62,                     /* Short|Fixed, Type=1-RTT, PN Len=3, Spin=1 */
    0x70, 0x71, 0x72,         /* DCID */
    0x50, 0x51, 0x52,         /* PN */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const unsigned char pkt_hdr_test_8_payload[] = {
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const struct pkt_hdr_test pkt_hdr_test_8 = {
    {
        QUIC_PKT_TYPE_1RTT,         /* type */
        1,                          /* spin bit */
        0,                          /* key phase */
        3,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 0, {0} },                 /* SCID */
        { 0x50, 0x51, 0x52 },       /* PN */
        NULL, 0,                    /* Token */
        18, NULL                    /* Len/Data */
    },
    pkt_hdr_test_8_expected, OSSL_NELEM(pkt_hdr_test_8_expected),
    pkt_hdr_test_8_payload,  OSSL_NELEM(pkt_hdr_test_8_payload),
    3, 21,
    4, 8
};

/* Packet Header Test 9: 1-RTT with Key Phase Bit */
static const unsigned char pkt_hdr_test_9_expected[] = {
    0x46,                     /* Short|Fixed, Type=1-RTT, PN Len=3, Key Phase=1 */
    0x70, 0x71, 0x72,         /* DCID */
    0x50, 0x51, 0x52,         /* PN */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const unsigned char pkt_hdr_test_9_payload[] = {
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const struct pkt_hdr_test pkt_hdr_test_9 = {
    {
        QUIC_PKT_TYPE_1RTT,         /* type */
        0,                          /* spin bit */
        1,                          /* key phase */
        3,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 0, {0} },                 /* SCID */
        { 0x50, 0x51, 0x52 },       /* PN */
        NULL, 0,                    /* Token */
        18, NULL                    /* Len/Data */
    },
    pkt_hdr_test_9_expected, OSSL_NELEM(pkt_hdr_test_9_expected),
    pkt_hdr_test_9_payload,  OSSL_NELEM(pkt_hdr_test_9_payload),
    3, 21,
    4, 8
};

/* Packet Header Test 10: Handshake with 4-Byte PN */
static const unsigned char pkt_hdr_test_10_expected[] = {
    0xe3,                     /* Long|Fixed, Type=Handshake, PN Len=4 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x17,                     /* Length=20 */
    0x33, 0x44, 0x55, 0x66,   /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_10_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_10 = {
    {
        QUIC_PKT_TYPE_HANDSHAKE,    /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        4,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        1,                          /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33, 0x44, 0x55, 0x66 }, /* PN */
        NULL, 0,                    /* Token */
        19, NULL                    /* Len/Data */
    },
    pkt_hdr_test_10_expected, OSSL_NELEM(pkt_hdr_test_10_expected),
    pkt_hdr_test_10_payload,  OSSL_NELEM(pkt_hdr_test_10_payload),
    0, sizeof(pkt_hdr_test_10_expected),
    19, 23
};

/* Packet Header Test 11: 1-RTT with 4-Byte PN */
static const unsigned char pkt_hdr_test_11_expected[] = {
    0x43,                     /* Short|Fixed, Type=1-RTT, PN Len=4 */
    0x70, 0x71, 0x72,         /* DCID */
    0x50, 0x51, 0x52, 0x53,   /* PN */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const unsigned char pkt_hdr_test_11_payload[] = {
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const struct pkt_hdr_test pkt_hdr_test_11 = {
    {
        QUIC_PKT_TYPE_1RTT,         /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        4,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 0, {0} },                 /* SCID */
        { 0x50, 0x51, 0x52, 0x53 }, /* PN */
        NULL, 0,                    /* Token */
        18, NULL                    /* Len/Data */
    },
    pkt_hdr_test_11_expected, OSSL_NELEM(pkt_hdr_test_11_expected),
    pkt_hdr_test_11_payload,  OSSL_NELEM(pkt_hdr_test_11_payload),
    3, 21,
    4, 8
};

/* Packet Header Test 12: Version Negotiation */
static const unsigned char pkt_hdr_test_12_expected[] = {
    0xc0,                       /* Long|Fixed, Type=Version Neg */
    0x00, 0x00, 0x00, 0x00,     /* Version (0) */
    0x03, 0x70, 0x71, 0x72,     /* DCID */
    0x02, 0x81, 0x82,           /* SCID */
    0x11, 0x22, 0x33, 0x44      /* One Version */
};

static const unsigned char pkt_hdr_test_12_payload[] = {
    0x11, 0x22, 0x33, 0x44
};

static const struct pkt_hdr_test pkt_hdr_test_12 = {
    {
        QUIC_PKT_TYPE_VERSION_NEG,  /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        0,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 2, {0x81, 0x82} },        /* SCID */
        { 0 },                      /* PN */
        NULL, 0,                    /* Token */
        4, NULL                     /* Len/Data */
    },
    pkt_hdr_test_12_expected, OSSL_NELEM(pkt_hdr_test_12_expected),
    pkt_hdr_test_12_payload,  OSSL_NELEM(pkt_hdr_test_12_payload),
    0, 12,
    SIZE_MAX, SIZE_MAX
};

/* Packet Header Test 13: Version Negotiation without Fixed Bit */
static const unsigned char pkt_hdr_test_13_expected[] = {
    0x80,                       /* Long|Fixed, Type=Version Neg */
    0x00, 0x00, 0x00, 0x00,     /* Version (0) */
    0x03, 0x70, 0x71, 0x72,     /* DCID */
    0x02, 0x81, 0x82,           /* SCID */
    0x11, 0x22, 0x33, 0x44      /* One Version */
};

static const unsigned char pkt_hdr_test_13_payload[] = {
    0x11, 0x22, 0x33, 0x44
};

static const struct pkt_hdr_test pkt_hdr_test_13 = {
    {
        QUIC_PKT_TYPE_VERSION_NEG,  /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        0,                          /* PN length */
        0,                          /* partial */
        0,                          /* fixed */
        0,                          /* unused */
        0,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 2, {0x81, 0x82} },        /* SCID */
        { 0 },                      /* PN */
        NULL, 0,                    /* Token */
        4, NULL                     /* Len/Data */
    },
    pkt_hdr_test_13_expected, OSSL_NELEM(pkt_hdr_test_13_expected),
    pkt_hdr_test_13_payload,  OSSL_NELEM(pkt_hdr_test_13_payload),
    0, 12,
    SIZE_MAX, SIZE_MAX
};

/* Packet Header Test 14: 1-RTT - Malformed - No Fixed Bit */
static const unsigned char pkt_hdr_test_14_expected[] = {
    0x02,                     /* Fixed, Type=1-RTT, PN Len=3 */
    0x70, 0x71, 0x72,         /* DCID */
    0x50, 0x51, 0x52,         /* PN */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const struct pkt_hdr_test pkt_hdr_test_14 = {
    { 0 },
    pkt_hdr_test_14_expected, OSSL_NELEM(pkt_hdr_test_14_expected),
    NULL, 0,
    3, SIZE_MAX,
    4, 8
};

/* Packet Header Test 15: Handshake - Malformed - No Fixed Bit */
static const unsigned char pkt_hdr_test_15_expected[] = {
    0xa0,                     /* Long, Type=Handshake, PN Len=1 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x14,                     /* Length=20 */
    0x33,                     /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_15 = {
    { 0 },
    pkt_hdr_test_15_expected, OSSL_NELEM(pkt_hdr_test_15_expected),
    NULL, 0,
    0, SIZE_MAX,
    19, 23
};

/* Packet Header Test 16: Handshake - Malformed - Wrong Version */
static const unsigned char pkt_hdr_test_16_expected[] = {
    0xe0,                     /* Long|Fixed, Type=Handshake, PN Len=1 */
    0x00, 0x00, 0x00, 0x02,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x14,                     /* Length=20 */
    0x33,                     /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_16 = {
    { 0 },
    pkt_hdr_test_16_expected, OSSL_NELEM(pkt_hdr_test_16_expected),
    NULL, 0,
    0, SIZE_MAX,
    19, 23
};

/* Packet Header Test 17: Initial - Non-Zero Reserved Bits */
static const unsigned char pkt_hdr_test_17_expected[] = {
    0xcd,                     /* Long|Fixed, Type=Initial, PN Len=2 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x00,                     /* DCID Length */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x00,                     /* Token Length */
    0x15,                     /* Length=21 */
    0x33, 0x44,               /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_17_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_17 = {
    {
        QUIC_PKT_TYPE_INITIAL,  /* type */
        0,                      /* spin bit */
        0,                      /* key phase */
        2,                      /* PN length */
        0,                      /* partial */
        1,                      /* fixed */
        0,                      /* unused */
        3,                      /* reserved */
        1,                      /* version */
        { 0, {0} },             /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33, 0x44 },         /* PN */
        NULL, 0,                /* Token/Token Len */
        19, NULL                /* Len/Data */
    },
    pkt_hdr_test_17_expected, OSSL_NELEM(pkt_hdr_test_17_expected),
    pkt_hdr_test_17_payload,  OSSL_NELEM(pkt_hdr_test_17_payload),
    0, sizeof(pkt_hdr_test_17_expected),
    17, 21
};

/* Packet Header Test 18: 0-RTT - Non-Zero Reserved Bits */
static const unsigned char pkt_hdr_test_18_expected[] = {
    0xd8,                     /* Long|Fixed, Type=0-RTT, PN Len=1 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x14,                     /* Length=20 */
    0x33,                     /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_18_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_18 = {
    {
        QUIC_PKT_TYPE_0RTT,     /* type */
        0,                      /* spin bit */
        0,                      /* key phase */
        1,                      /* PN length */
        0,                      /* partial */
        1,                      /* fixed */
        0,                      /* unused */
        2,                      /* reserved */
        1,                      /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33 },               /* PN */
        NULL, 0,                /* Token */
        19, NULL                /* Len/Data */
    },
    pkt_hdr_test_18_expected, OSSL_NELEM(pkt_hdr_test_18_expected),
    pkt_hdr_test_18_payload,  OSSL_NELEM(pkt_hdr_test_18_payload),
    0, sizeof(pkt_hdr_test_18_expected),
    19, 23
};

/* Packet Header Test 19: Handshake - Non-Zero Reserved Bits */
static const unsigned char pkt_hdr_test_19_expected[] = {
    0xe4,                     /* Long|Fixed, Type=Handshake, PN Len=1 */
    0x00, 0x00, 0x00, 0x01,   /* Version */
    0x03,                     /* DCID Length */
    0x70, 0x71, 0x72,         /* DCID */
    0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5, /* SCID Length, SCID */
    0x14,                     /* Length=20 */
    0x33,                     /* Encoded PN */
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, /* Payload */
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const unsigned char pkt_hdr_test_19_payload[] = {
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22
};

static const struct pkt_hdr_test pkt_hdr_test_19 = {
    {
        QUIC_PKT_TYPE_HANDSHAKE,    /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        1,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        1,                          /* reserved */
        1,                          /* version */
        { 3, {0x70, 0x71, 0x72} },                                /* DCID */
        { 8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5 } }, /* SCID */
        { 0x33 },                   /* PN */
        NULL, 0,                    /* Token */
        19, NULL                    /* Len/Data */
    },
    pkt_hdr_test_19_expected, OSSL_NELEM(pkt_hdr_test_19_expected),
    pkt_hdr_test_19_payload,  OSSL_NELEM(pkt_hdr_test_19_payload),
    0, sizeof(pkt_hdr_test_19_expected),
    19, 23
};

/* Packet Header Test 20: 1-RTT with Non-Zero Reserved Bits */
static const unsigned char pkt_hdr_test_20_expected[] = {
    0x5a,                     /* Short|Fixed, Type=1-RTT, PN Len=3 */
    0x70, 0x71, 0x72,         /* DCID */
    0x50, 0x51, 0x52,         /* PN */
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const unsigned char pkt_hdr_test_20_payload[] = {
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1
};

static const struct pkt_hdr_test pkt_hdr_test_20 = {
    {
        QUIC_PKT_TYPE_1RTT,         /* type */
        0,                          /* spin bit */
        0,                          /* key phase */
        3,                          /* PN length */
        0,                          /* partial */
        1,                          /* fixed */
        0,                          /* unused */
        3,                          /* reserved */
        0,                          /* version */
        { 3, {0x70, 0x71, 0x72} },  /* DCID */
        { 0, {0} },                 /* SCID */
        { 0x50, 0x51, 0x52 },       /* PN */
        NULL, 0,                    /* Token */
        18, NULL                    /* Len/Data */
    },
    pkt_hdr_test_20_expected, OSSL_NELEM(pkt_hdr_test_20_expected),
    pkt_hdr_test_20_payload,  OSSL_NELEM(pkt_hdr_test_20_payload),
    3, 21,
    4, 8
};

static const struct pkt_hdr_test *const pkt_hdr_tests[] = {
    &pkt_hdr_test_1,
    &pkt_hdr_test_2,
    &pkt_hdr_test_3,
    &pkt_hdr_test_4,
    &pkt_hdr_test_5,
    &pkt_hdr_test_6,
    &pkt_hdr_test_7,
    &pkt_hdr_test_8,
    &pkt_hdr_test_9,
    &pkt_hdr_test_10,
    &pkt_hdr_test_11,
    &pkt_hdr_test_12,
    &pkt_hdr_test_13,
    &pkt_hdr_test_14,
    &pkt_hdr_test_15,
    &pkt_hdr_test_16,
    &pkt_hdr_test_17,
    &pkt_hdr_test_18,
    &pkt_hdr_test_19,
    &pkt_hdr_test_20
};

#define HPR_REPEAT_COUNT 4
#define HPR_CIPHER_COUNT 3

/*
 * Count of number of times we observed an unchanged (u) or changed (c) bit in
 * each header-protectable bit over all test suites.
 */
static unsigned int counts_u[HPR_CIPHER_COUNT][37] = {0};
static unsigned int counts_c[HPR_CIPHER_COUNT][37] = {0};

#define TEST_PKT_BUF_LEN 20000

static int test_wire_pkt_hdr_actual(int tidx, int repeat, int cipher,
                                    size_t trunc_len)
{
    int testresult = 0;
    const struct pkt_hdr_test *t = pkt_hdr_tests[tidx];
    QUIC_PKT_HDR hdr = {0};
    QUIC_PKT_HDR_PTRS ptrs = {0}, wptrs = {0};
    PACKET pkt = {0};
    WPACKET wpkt = {0};
    unsigned char *buf = NULL;
    size_t l = 0, i, j;
    QUIC_HDR_PROTECTOR hpr = {0};
    unsigned char hpr_key[32] = {0,1,2,3,4,5,6,7};
    int have_hpr = 0, hpr_cipher_id, hpr_key_len;
    unsigned char *hbuf = NULL;
    int is_trunc = trunc_len < t->expected_len;
    int expect_fail = trunc_len < t->min_success_len;
    hpr_key[8] = (unsigned char)tidx;
    hpr_key[9] = (unsigned char)repeat;

    if (is_trunc && trunc_len > t->min_success_len
        && t->hdr.type == QUIC_PKT_TYPE_VERSION_NEG
        && ((trunc_len - t->min_success_len) % 4) != 0)
        expect_fail = 1;

    switch (cipher) {
        case 0:
            hpr_cipher_id = QUIC_HDR_PROT_CIPHER_AES_128;
            hpr_key_len   = 16;
            break;
        case 1:
            hpr_cipher_id = QUIC_HDR_PROT_CIPHER_AES_256;
            hpr_key_len   = 32;
            break;
        case 2:
            /*
             * In a build without CHACHA, we rerun the AES 256 tests.
             * Removing all dependence on CHACHA is more difficult and these
             * tests are fast enough.
             */
#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
            hpr_cipher_id = QUIC_HDR_PROT_CIPHER_CHACHA;
#else
            hpr_cipher_id = QUIC_HDR_PROT_CIPHER_AES_256;
#endif
            hpr_key_len   = 32;
            break;
        default:
            goto err;
    }

    if (!TEST_ptr(buf = OPENSSL_malloc(TEST_PKT_BUF_LEN)))
        goto err;

    if (!TEST_true(WPACKET_init_static_len(&wpkt, buf, TEST_PKT_BUF_LEN, 0)))
        goto err;

    if (!TEST_true(PACKET_buf_init(&pkt, t->expected, trunc_len)))
        goto err;

    if (!TEST_int_eq(ossl_quic_wire_decode_pkt_hdr(&pkt, t->short_conn_id_len,
                                                   0, 0, &hdr, &ptrs),
                     !expect_fail))
        goto err;

    if (!expect_fail && !is_trunc) {
        if (!TEST_true(cmp_pkt_hdr(&hdr, &t->hdr, t->payload, t->payload_len, 1)))
            goto err;

        if (!TEST_ptr_eq(ptrs.raw_start, t->expected))
            goto err;

        if (t->pn_offset == SIZE_MAX) {
            if (!TEST_ptr_null(ptrs.raw_pn))
                goto err;
        } else {
            if (!TEST_ptr_eq(ptrs.raw_pn, t->expected + t->pn_offset))
                goto err;
        }

        if (t->sample_offset != SIZE_MAX) {
            if (!TEST_ptr_eq(ptrs.raw_sample, t->expected + t->sample_offset))
                goto err;
            if (!TEST_size_t_eq(ptrs.raw_sample_len,
                                t->expected_len - t->sample_offset))
                goto err;
        }

        if (!TEST_true(ossl_quic_wire_encode_pkt_hdr(&wpkt, t->short_conn_id_len, &hdr, &wptrs)))
            goto err;

        if (!TEST_true(WPACKET_memcpy(&wpkt, t->payload, t->payload_len)))
            goto err;

        if (!TEST_true(WPACKET_get_total_written(&wpkt, &l)))
            goto err;

        if (!TEST_mem_eq(buf, l, t->expected, t->expected_len))
            goto err;

        /* Test header protection. */
        if (t->sample_offset != SIZE_MAX) { /* if packet type has protection */
            if (!TEST_true(ossl_quic_hdr_protector_init(&hpr, NULL, NULL,
                                                        hpr_cipher_id,
                                                        hpr_key,
                                                        hpr_key_len)))
                goto err;

            have_hpr = 1;

            /*
             * Copy into a duplicate buffer to test header protection by
             * comparing it against the original.
             */
            hbuf = OPENSSL_malloc(t->expected_len);
            if (!TEST_ptr(hbuf))
                goto err;

            memcpy(hbuf, t->expected, t->expected_len);

            /* Fixup pointers to new buffer and encrypt. */
            ptrs.raw_pn         = hbuf + (ptrs.raw_pn     - ptrs.raw_start);
            ptrs.raw_sample     = hbuf + (ptrs.raw_sample - ptrs.raw_start);
            ptrs.raw_start      = hbuf;
            if (!TEST_true(ossl_quic_hdr_protector_encrypt(&hpr, &ptrs)))
                goto err;

            /* Ensure that bytes which should not have changed did not change */
            for (i = 0; i < t->expected_len; ++i) {
                unsigned char d = t->expected[i] ^ hbuf[i], rej_mask = 0xff;
                size_t jrel = 0;
                if (i == 0) {
                    /* Bits in first byte which must not change */
                    rej_mask = (t->hdr.type == QUIC_PKT_TYPE_1RTT) ? ~0x1f : ~0xf;
                } else if (i >= t->pn_offset && i < t->pn_offset + t->hdr.pn_len) {
                    /* PN bytes change */
                    rej_mask = 0;
                    jrel = 5 + (i - t->pn_offset) * 8;
                }

                if (rej_mask != 0xff)
                    for (j = 0; j < 8; ++j) {
                        if (((1U << j) & rej_mask) != 0)
                            /*
                             * Bit unrelated to header protection, do not record
                             * stats about it.
                             */
                            continue;

                        OPENSSL_assert(jrel + j < OSSL_NELEM(counts_u[cipher]));
                        if ((d & (1U << j)) != 0)
                            ++counts_c[cipher][jrel + j]; /* bit did change */
                        else
                            ++counts_u[cipher][jrel + j]; /* bit did not change */
                    }

                /* Bits in rej_mask must not change */
                if (!TEST_int_eq(d & rej_mask, 0))
                    goto err;
            }

            /* Decrypt and check matches original. */
            if (!TEST_true(ossl_quic_hdr_protector_decrypt(&hpr, &ptrs)))
                goto err;

            if (!TEST_mem_eq(hbuf, t->expected_len, t->expected, t->expected_len))
                goto err;
        }
    }

    testresult = 1;
err:
    if (have_hpr)
        ossl_quic_hdr_protector_cleanup(&hpr);
    WPACKET_finish(&wpkt);
    OPENSSL_free(buf);
    OPENSSL_free(hbuf);
    return testresult;
}

static int test_wire_pkt_hdr_inner(int tidx, int repeat, int cipher)
{
    int testresult = 0;
    const struct pkt_hdr_test *t = pkt_hdr_tests[tidx];
    size_t i;

    /* Test with entire packet */
    if (!TEST_true(test_wire_pkt_hdr_actual(tidx, repeat, cipher,
                                            t->expected_len)))
        goto err;

    /* Now repeat for every possible truncation of the packet */
    for (i = 0; i < t->expected_len; ++i)
        if (!TEST_true(test_wire_pkt_hdr_actual(tidx, repeat, cipher, i)))
            goto err;

    testresult = 1;
err:
    return testresult;
}

static int test_hdr_prot_stats(void)
{
    int testresult = 0;
    size_t i, cipher;

    /*
     * Test that, across all previously executed tests for each header
     * protection cipher, every bit which can have header protection applied a)
     * was changed in at least one test of applying header protection, and b)
     * was unchanged in at least one test of applying header protection.
     */
    for (cipher = 0; cipher < HPR_CIPHER_COUNT; ++cipher)
        for (i = 0; i < OSSL_NELEM(counts_u[0]); ++i) {
            if (!TEST_uint_gt(counts_u[cipher][i], 0))
                goto err;
            if (!TEST_uint_gt(counts_c[cipher][i], 0))
                goto err;
        }

    testresult = 1;
err:
    return testresult;
}

#define NUM_WIRE_PKT_HDR_TESTS \
    (OSSL_NELEM(pkt_hdr_tests) * HPR_REPEAT_COUNT * HPR_CIPHER_COUNT)

static int test_wire_pkt_hdr(int idx)
{
    int tidx, repeat, cipher;

    if (idx == NUM_WIRE_PKT_HDR_TESTS)
        return test_hdr_prot_stats();

    cipher = idx % HPR_CIPHER_COUNT;
    idx /= HPR_CIPHER_COUNT;

    repeat = idx % HPR_REPEAT_COUNT;
    idx /= HPR_REPEAT_COUNT;

    tidx = idx;

    return test_wire_pkt_hdr_inner(tidx, repeat, cipher);
}

/* TX Tests */
#define TX_TEST_OP_END                     0 /* end of script */
#define TX_TEST_OP_WRITE                   1 /* write packet */
#define TX_TEST_OP_PROVIDE_SECRET          2 /* provide TX secret */
#define TX_TEST_OP_PROVIDE_SECRET_INITIAL  3 /* provide TX secret for initial */
#define TX_TEST_OP_DISCARD_EL              4 /* discard an encryption level */
#define TX_TEST_OP_CHECK_DGRAM             5 /* read datagram, compare to expected */
#define TX_TEST_OP_CHECK_NO_DGRAM          6 /* check no datagram is in queue */
#define TX_TEST_OP_KEY_UPDATE              7 /* perform key update for 1-RTT */

struct tx_test_op {
    unsigned char op;
    const unsigned char *buf;
    size_t buf_len;
    const OSSL_QTX_PKT *pkt;
    uint32_t enc_level, suite_id;
    const QUIC_CONN_ID *dcid;
};

#define TX_OP_END                                                       \
    { TX_TEST_OP_END }
#define TX_OP_WRITE(pkt)                                                \
    { TX_TEST_OP_WRITE, NULL, 0, &(pkt), 0, 0, NULL },
#define TX_OP_PROVIDE_SECRET(el, suite, key)                            \
    {                                                                   \
        TX_TEST_OP_PROVIDE_SECRET, (key), sizeof(key),                  \
        NULL, (el), (suite), NULL                                       \
    },
#define TX_OP_PROVIDE_SECRET_INITIAL(dcid, is_server)                   \
    { TX_TEST_OP_PROVIDE_SECRET_INITIAL,                                \
      NULL, 0, NULL, 0, (is_server), &(dcid) },
#define TX_OP_DISCARD_EL(el)                                            \
    { TX_TEST_OP_DISCARD_EL, NULL, 0, NULL, (el), 0, NULL },
#define TX_OP_CHECK_DGRAM(expect_dgram)                                 \
    {                                                                   \
        TX_TEST_OP_CHECK_DGRAM, (expect_dgram), sizeof(expect_dgram),   \
        NULL, 0, 0, NULL                                                \
    },
#define TX_OP_CHECK_NO_DGRAM() \
    { TX_TEST_OP_CHECK_NO_PKT, NULL, 0, NULL, 0, 0, NULL },

#define TX_OP_WRITE_N(n)                                                \
    TX_OP_WRITE(tx_script_##n##_pkt)
#define TX_OP_CHECK_DGRAM_N(n)                                          \
    TX_OP_CHECK_DGRAM(tx_script_##n##_dgram)

#define TX_OP_WRITE_CHECK(n)                                            \
    TX_OP_WRITE_N(n)                                                    \
    TX_OP_CHECK_DGRAM_N(n)

#define TX_OP_KEY_UPDATE()                                              \
    { TX_TEST_OP_KEY_UPDATE, NULL, 0, NULL, 0, 0, NULL },

/* 1. RFC 9001 - A.2 Client Initial */
static const unsigned char tx_script_1_body[1162] = {
    0x06, 0x00, 0x40, 0xf1, 0x01, 0x00, 0x00, 0xed, 0x03, 0x03, 0xeb, 0xf8,
    0xfa, 0x56, 0xf1, 0x29, 0x39, 0xb9, 0x58, 0x4a, 0x38, 0x96, 0x47, 0x2e,
    0xc4, 0x0b, 0xb8, 0x63, 0xcf, 0xd3, 0xe8, 0x68, 0x04, 0xfe, 0x3a, 0x47,
    0xf0, 0x6a, 0x2b, 0x69, 0x48, 0x4c, 0x00, 0x00, 0x04, 0x13, 0x01, 0x13,
    0x02, 0x01, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x10, 0x00, 0x0e, 0x00,
    0x00, 0x0b, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f,
    0x6d, 0xff, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06,
    0x00, 0x1d, 0x00, 0x17, 0x00, 0x18, 0x00, 0x10, 0x00, 0x07, 0x00, 0x05,
    0x04, 0x61, 0x6c, 0x70, 0x6e, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x33, 0x00, 0x26, 0x00, 0x24, 0x00, 0x1d, 0x00, 0x20,
    0x93, 0x70, 0xb2, 0xc9, 0xca, 0xa4, 0x7f, 0xba, 0xba, 0xf4, 0x55, 0x9f,
    0xed, 0xba, 0x75, 0x3d, 0xe1, 0x71, 0xfa, 0x71, 0xf5, 0x0f, 0x1c, 0xe1,
    0x5d, 0x43, 0xe9, 0x94, 0xec, 0x74, 0xd7, 0x48, 0x00, 0x2b, 0x00, 0x03,
    0x02, 0x03, 0x04, 0x00, 0x0d, 0x00, 0x10, 0x00, 0x0e, 0x04, 0x03, 0x05,
    0x03, 0x06, 0x03, 0x02, 0x03, 0x08, 0x04, 0x08, 0x05, 0x08, 0x06, 0x00,
    0x2d, 0x00, 0x02, 0x01, 0x01, 0x00, 0x1c, 0x00, 0x02, 0x40, 0x01, 0x00,
    0x39, 0x00, 0x32, 0x04, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x05, 0x04, 0x80, 0x00, 0xff, 0xff, 0x07, 0x04, 0x80, 0x00, 0xff,
    0xff, 0x08, 0x01, 0x10, 0x01, 0x04, 0x80, 0x00, 0x75, 0x30, 0x09, 0x01,
    0x10, 0x0f, 0x08, 0x83, 0x94, 0xc8, 0xf0, 0x3e, 0x51, 0x57, 0x08, 0x06,
    0x04, 0x80, 0x00, 0xff, 0xff /* followed by zero padding */
};

static const unsigned char tx_script_1_dgram[] = {
    0xc0, 0x00, 0x00, 0x00, 0x01, 0x08, 0x83, 0x94, 0xc8, 0xf0, 0x3e, 0x51,
    0x57, 0x08, 0x00, 0x00, 0x44, 0x9e, 0x7b, 0x9a, 0xec, 0x34, 0xd1, 0xb1,
    0xc9, 0x8d, 0xd7, 0x68, 0x9f, 0xb8, 0xec, 0x11, 0xd2, 0x42, 0xb1, 0x23,
    0xdc, 0x9b, 0xd8, 0xba, 0xb9, 0x36, 0xb4, 0x7d, 0x92, 0xec, 0x35, 0x6c,
    0x0b, 0xab, 0x7d, 0xf5, 0x97, 0x6d, 0x27, 0xcd, 0x44, 0x9f, 0x63, 0x30,
    0x00, 0x99, 0xf3, 0x99, 0x1c, 0x26, 0x0e, 0xc4, 0xc6, 0x0d, 0x17, 0xb3,
    0x1f, 0x84, 0x29, 0x15, 0x7b, 0xb3, 0x5a, 0x12, 0x82, 0xa6, 0x43, 0xa8,
    0xd2, 0x26, 0x2c, 0xad, 0x67, 0x50, 0x0c, 0xad, 0xb8, 0xe7, 0x37, 0x8c,
    0x8e, 0xb7, 0x53, 0x9e, 0xc4, 0xd4, 0x90, 0x5f, 0xed, 0x1b, 0xee, 0x1f,
    0xc8, 0xaa, 0xfb, 0xa1, 0x7c, 0x75, 0x0e, 0x2c, 0x7a, 0xce, 0x01, 0xe6,
    0x00, 0x5f, 0x80, 0xfc, 0xb7, 0xdf, 0x62, 0x12, 0x30, 0xc8, 0x37, 0x11,
    0xb3, 0x93, 0x43, 0xfa, 0x02, 0x8c, 0xea, 0x7f, 0x7f, 0xb5, 0xff, 0x89,
    0xea, 0xc2, 0x30, 0x82, 0x49, 0xa0, 0x22, 0x52, 0x15, 0x5e, 0x23, 0x47,
    0xb6, 0x3d, 0x58, 0xc5, 0x45, 0x7a, 0xfd, 0x84, 0xd0, 0x5d, 0xff, 0xfd,
    0xb2, 0x03, 0x92, 0x84, 0x4a, 0xe8, 0x12, 0x15, 0x46, 0x82, 0xe9, 0xcf,
    0x01, 0x2f, 0x90, 0x21, 0xa6, 0xf0, 0xbe, 0x17, 0xdd, 0xd0, 0xc2, 0x08,
    0x4d, 0xce, 0x25, 0xff, 0x9b, 0x06, 0xcd, 0xe5, 0x35, 0xd0, 0xf9, 0x20,
    0xa2, 0xdb, 0x1b, 0xf3, 0x62, 0xc2, 0x3e, 0x59, 0x6d, 0x11, 0xa4, 0xf5,
    0xa6, 0xcf, 0x39, 0x48, 0x83, 0x8a, 0x3a, 0xec, 0x4e, 0x15, 0xda, 0xf8,
    0x50, 0x0a, 0x6e, 0xf6, 0x9e, 0xc4, 0xe3, 0xfe, 0xb6, 0xb1, 0xd9, 0x8e,
    0x61, 0x0a, 0xc8, 0xb7, 0xec, 0x3f, 0xaf, 0x6a, 0xd7, 0x60, 0xb7, 0xba,
    0xd1, 0xdb, 0x4b, 0xa3, 0x48, 0x5e, 0x8a, 0x94, 0xdc, 0x25, 0x0a, 0xe3,
    0xfd, 0xb4, 0x1e, 0xd1, 0x5f, 0xb6, 0xa8, 0xe5, 0xeb, 0xa0, 0xfc, 0x3d,
    0xd6, 0x0b, 0xc8, 0xe3, 0x0c, 0x5c, 0x42, 0x87, 0xe5, 0x38, 0x05, 0xdb,
    0x05, 0x9a, 0xe0, 0x64, 0x8d, 0xb2, 0xf6, 0x42, 0x64, 0xed, 0x5e, 0x39,
    0xbe, 0x2e, 0x20, 0xd8, 0x2d, 0xf5, 0x66, 0xda, 0x8d, 0xd5, 0x99, 0x8c,
    0xca, 0xbd, 0xae, 0x05, 0x30, 0x60, 0xae, 0x6c, 0x7b, 0x43, 0x78, 0xe8,
    0x46, 0xd2, 0x9f, 0x37, 0xed, 0x7b, 0x4e, 0xa9, 0xec, 0x5d, 0x82, 0xe7,
    0x96, 0x1b, 0x7f, 0x25, 0xa9, 0x32, 0x38, 0x51, 0xf6, 0x81, 0xd5, 0x82,
    0x36, 0x3a, 0xa5, 0xf8, 0x99, 0x37, 0xf5, 0xa6, 0x72, 0x58, 0xbf, 0x63,
    0xad, 0x6f, 0x1a, 0x0b, 0x1d, 0x96, 0xdb, 0xd4, 0xfa, 0xdd, 0xfc, 0xef,
    0xc5, 0x26, 0x6b, 0xa6, 0x61, 0x17, 0x22, 0x39, 0x5c, 0x90, 0x65, 0x56,
    0xbe, 0x52, 0xaf, 0xe3, 0xf5, 0x65, 0x63, 0x6a, 0xd1, 0xb1, 0x7d, 0x50,
    0x8b, 0x73, 0xd8, 0x74, 0x3e, 0xeb, 0x52, 0x4b, 0xe2, 0x2b, 0x3d, 0xcb,
    0xc2, 0xc7, 0x46, 0x8d, 0x54, 0x11, 0x9c, 0x74, 0x68, 0x44, 0x9a, 0x13,
    0xd8, 0xe3, 0xb9, 0x58, 0x11, 0xa1, 0x98, 0xf3, 0x49, 0x1d, 0xe3, 0xe7,
    0xfe, 0x94, 0x2b, 0x33, 0x04, 0x07, 0xab, 0xf8, 0x2a, 0x4e, 0xd7, 0xc1,
    0xb3, 0x11, 0x66, 0x3a, 0xc6, 0x98, 0x90, 0xf4, 0x15, 0x70, 0x15, 0x85,
    0x3d, 0x91, 0xe9, 0x23, 0x03, 0x7c, 0x22, 0x7a, 0x33, 0xcd, 0xd5, 0xec,
    0x28, 0x1c, 0xa3, 0xf7, 0x9c, 0x44, 0x54, 0x6b, 0x9d, 0x90, 0xca, 0x00,
    0xf0, 0x64, 0xc9, 0x9e, 0x3d, 0xd9, 0x79, 0x11, 0xd3, 0x9f, 0xe9, 0xc5,
    0xd0, 0xb2, 0x3a, 0x22, 0x9a, 0x23, 0x4c, 0xb3, 0x61, 0x86, 0xc4, 0x81,
    0x9e, 0x8b, 0x9c, 0x59, 0x27, 0x72, 0x66, 0x32, 0x29, 0x1d, 0x6a, 0x41,
    0x82, 0x11, 0xcc, 0x29, 0x62, 0xe2, 0x0f, 0xe4, 0x7f, 0xeb, 0x3e, 0xdf,
    0x33, 0x0f, 0x2c, 0x60, 0x3a, 0x9d, 0x48, 0xc0, 0xfc, 0xb5, 0x69, 0x9d,
    0xbf, 0xe5, 0x89, 0x64, 0x25, 0xc5, 0xba, 0xc4, 0xae, 0xe8, 0x2e, 0x57,
    0xa8, 0x5a, 0xaf, 0x4e, 0x25, 0x13, 0xe4, 0xf0, 0x57, 0x96, 0xb0, 0x7b,
    0xa2, 0xee, 0x47, 0xd8, 0x05, 0x06, 0xf8, 0xd2, 0xc2, 0x5e, 0x50, 0xfd,
    0x14, 0xde, 0x71, 0xe6, 0xc4, 0x18, 0x55, 0x93, 0x02, 0xf9, 0x39, 0xb0,
    0xe1, 0xab, 0xd5, 0x76, 0xf2, 0x79, 0xc4, 0xb2, 0xe0, 0xfe, 0xb8, 0x5c,
    0x1f, 0x28, 0xff, 0x18, 0xf5, 0x88, 0x91, 0xff, 0xef, 0x13, 0x2e, 0xef,
    0x2f, 0xa0, 0x93, 0x46, 0xae, 0xe3, 0x3c, 0x28, 0xeb, 0x13, 0x0f, 0xf2,
    0x8f, 0x5b, 0x76, 0x69, 0x53, 0x33, 0x41, 0x13, 0x21, 0x19, 0x96, 0xd2,
    0x00, 0x11, 0xa1, 0x98, 0xe3, 0xfc, 0x43, 0x3f, 0x9f, 0x25, 0x41, 0x01,
    0x0a, 0xe1, 0x7c, 0x1b, 0xf2, 0x02, 0x58, 0x0f, 0x60, 0x47, 0x47, 0x2f,
    0xb3, 0x68, 0x57, 0xfe, 0x84, 0x3b, 0x19, 0xf5, 0x98, 0x40, 0x09, 0xdd,
    0xc3, 0x24, 0x04, 0x4e, 0x84, 0x7a, 0x4f, 0x4a, 0x0a, 0xb3, 0x4f, 0x71,
    0x95, 0x95, 0xde, 0x37, 0x25, 0x2d, 0x62, 0x35, 0x36, 0x5e, 0x9b, 0x84,
    0x39, 0x2b, 0x06, 0x10, 0x85, 0x34, 0x9d, 0x73, 0x20, 0x3a, 0x4a, 0x13,
    0xe9, 0x6f, 0x54, 0x32, 0xec, 0x0f, 0xd4, 0xa1, 0xee, 0x65, 0xac, 0xcd,
    0xd5, 0xe3, 0x90, 0x4d, 0xf5, 0x4c, 0x1d, 0xa5, 0x10, 0xb0, 0xff, 0x20,
    0xdc, 0xc0, 0xc7, 0x7f, 0xcb, 0x2c, 0x0e, 0x0e, 0xb6, 0x05, 0xcb, 0x05,
    0x04, 0xdb, 0x87, 0x63, 0x2c, 0xf3, 0xd8, 0xb4, 0xda, 0xe6, 0xe7, 0x05,
    0x76, 0x9d, 0x1d, 0xe3, 0x54, 0x27, 0x01, 0x23, 0xcb, 0x11, 0x45, 0x0e,
    0xfc, 0x60, 0xac, 0x47, 0x68, 0x3d, 0x7b, 0x8d, 0x0f, 0x81, 0x13, 0x65,
    0x56, 0x5f, 0xd9, 0x8c, 0x4c, 0x8e, 0xb9, 0x36, 0xbc, 0xab, 0x8d, 0x06,
    0x9f, 0xc3, 0x3b, 0xd8, 0x01, 0xb0, 0x3a, 0xde, 0xa2, 0xe1, 0xfb, 0xc5,
    0xaa, 0x46, 0x3d, 0x08, 0xca, 0x19, 0x89, 0x6d, 0x2b, 0xf5, 0x9a, 0x07,
    0x1b, 0x85, 0x1e, 0x6c, 0x23, 0x90, 0x52, 0x17, 0x2f, 0x29, 0x6b, 0xfb,
    0x5e, 0x72, 0x40, 0x47, 0x90, 0xa2, 0x18, 0x10, 0x14, 0xf3, 0xb9, 0x4a,
    0x4e, 0x97, 0xd1, 0x17, 0xb4, 0x38, 0x13, 0x03, 0x68, 0xcc, 0x39, 0xdb,
    0xb2, 0xd1, 0x98, 0x06, 0x5a, 0xe3, 0x98, 0x65, 0x47, 0x92, 0x6c, 0xd2,
    0x16, 0x2f, 0x40, 0xa2, 0x9f, 0x0c, 0x3c, 0x87, 0x45, 0xc0, 0xf5, 0x0f,
    0xba, 0x38, 0x52, 0xe5, 0x66, 0xd4, 0x45, 0x75, 0xc2, 0x9d, 0x39, 0xa0,
    0x3f, 0x0c, 0xda, 0x72, 0x19, 0x84, 0xb6, 0xf4, 0x40, 0x59, 0x1f, 0x35,
    0x5e, 0x12, 0xd4, 0x39, 0xff, 0x15, 0x0a, 0xab, 0x76, 0x13, 0x49, 0x9d,
    0xbd, 0x49, 0xad, 0xab, 0xc8, 0x67, 0x6e, 0xef, 0x02, 0x3b, 0x15, 0xb6,
    0x5b, 0xfc, 0x5c, 0xa0, 0x69, 0x48, 0x10, 0x9f, 0x23, 0xf3, 0x50, 0xdb,
    0x82, 0x12, 0x35, 0x35, 0xeb, 0x8a, 0x74, 0x33, 0xbd, 0xab, 0xcb, 0x90,
    0x92, 0x71, 0xa6, 0xec, 0xbc, 0xb5, 0x8b, 0x93, 0x6a, 0x88, 0xcd, 0x4e,
    0x8f, 0x2e, 0x6f, 0xf5, 0x80, 0x01, 0x75, 0xf1, 0x13, 0x25, 0x3d, 0x8f,
    0xa9, 0xca, 0x88, 0x85, 0xc2, 0xf5, 0x52, 0xe6, 0x57, 0xdc, 0x60, 0x3f,
    0x25, 0x2e, 0x1a, 0x8e, 0x30, 0x8f, 0x76, 0xf0, 0xbe, 0x79, 0xe2, 0xfb,
    0x8f, 0x5d, 0x5f, 0xbb, 0xe2, 0xe3, 0x0e, 0xca, 0xdd, 0x22, 0x07, 0x23,
    0xc8, 0xc0, 0xae, 0xa8, 0x07, 0x8c, 0xdf, 0xcb, 0x38, 0x68, 0x26, 0x3f,
    0xf8, 0xf0, 0x94, 0x00, 0x54, 0xda, 0x48, 0x78, 0x18, 0x93, 0xa7, 0xe4,
    0x9a, 0xd5, 0xaf, 0xf4, 0xaf, 0x30, 0x0c, 0xd8, 0x04, 0xa6, 0xb6, 0x27,
    0x9a, 0xb3, 0xff, 0x3a, 0xfb, 0x64, 0x49, 0x1c, 0x85, 0x19, 0x4a, 0xab,
    0x76, 0x0d, 0x58, 0xa6, 0x06, 0x65, 0x4f, 0x9f, 0x44, 0x00, 0xe8, 0xb3,
    0x85, 0x91, 0x35, 0x6f, 0xbf, 0x64, 0x25, 0xac, 0xa2, 0x6d, 0xc8, 0x52,
    0x44, 0x25, 0x9f, 0xf2, 0xb1, 0x9c, 0x41, 0xb9, 0xf9, 0x6f, 0x3c, 0xa9,
    0xec, 0x1d, 0xde, 0x43, 0x4d, 0xa7, 0xd2, 0xd3, 0x92, 0xb9, 0x05, 0xdd,
    0xf3, 0xd1, 0xf9, 0xaf, 0x93, 0xd1, 0xaf, 0x59, 0x50, 0xbd, 0x49, 0x3f,
    0x5a, 0xa7, 0x31, 0xb4, 0x05, 0x6d, 0xf3, 0x1b, 0xd2, 0x67, 0xb6, 0xb9,
    0x0a, 0x07, 0x98, 0x31, 0xaa, 0xf5, 0x79, 0xbe, 0x0a, 0x39, 0x01, 0x31,
    0x37, 0xaa, 0xc6, 0xd4, 0x04, 0xf5, 0x18, 0xcf, 0xd4, 0x68, 0x40, 0x64,
    0x7e, 0x78, 0xbf, 0xe7, 0x06, 0xca, 0x4c, 0xf5, 0xe9, 0xc5, 0x45, 0x3e,
    0x9f, 0x7c, 0xfd, 0x2b, 0x8b, 0x4c, 0x8d, 0x16, 0x9a, 0x44, 0xe5, 0x5c,
    0x88, 0xd4, 0xa9, 0xa7, 0xf9, 0x47, 0x42, 0x41, 0xe2, 0x21, 0xaf, 0x44,
    0x86, 0x00, 0x18, 0xab, 0x08, 0x56, 0x97, 0x2e, 0x19, 0x4c, 0xd9, 0x34
};

static QUIC_PKT_HDR tx_script_1_hdr = {
    QUIC_PKT_TYPE_INITIAL,      /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    4,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    1,                          /* version */
    {8, {0x83, 0x94, 0xc8, 0xf0, 0x3e, 0x51, 0x57, 0x08}}, /* DCID */
    { 0, {0} },                 /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_1_iovec[] = {
    { tx_script_1_body, sizeof(tx_script_1_body) }
};

static const OSSL_QTX_PKT tx_script_1_pkt = {
    &tx_script_1_hdr,
    tx_script_1_iovec,
    OSSL_NELEM(tx_script_1_iovec),
    NULL, NULL,
    2,
    0
};

static const struct tx_test_op tx_script_1[] = {
    TX_OP_PROVIDE_SECRET_INITIAL(tx_script_1_hdr.dst_conn_id, 0)
    TX_OP_WRITE_CHECK(1)
    TX_OP_END
};

/* 2. RFC 9001 - A.3 Server Initial */
static const unsigned char tx_script_2_body[] = {
    0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x40, 0x5a, 0x02, 0x00, 0x00,
    0x56, 0x03, 0x03, 0xee, 0xfc, 0xe7, 0xf7, 0xb3, 0x7b, 0xa1, 0xd1, 0x63,
    0x2e, 0x96, 0x67, 0x78, 0x25, 0xdd, 0xf7, 0x39, 0x88, 0xcf, 0xc7, 0x98,
    0x25, 0xdf, 0x56, 0x6d, 0xc5, 0x43, 0x0b, 0x9a, 0x04, 0x5a, 0x12, 0x00,
    0x13, 0x01, 0x00, 0x00, 0x2e, 0x00, 0x33, 0x00, 0x24, 0x00, 0x1d, 0x00,
    0x20, 0x9d, 0x3c, 0x94, 0x0d, 0x89, 0x69, 0x0b, 0x84, 0xd0, 0x8a, 0x60,
    0x99, 0x3c, 0x14, 0x4e, 0xca, 0x68, 0x4d, 0x10, 0x81, 0x28, 0x7c, 0x83,
    0x4d, 0x53, 0x11, 0xbc, 0xf3, 0x2b, 0xb9, 0xda, 0x1a, 0x00, 0x2b, 0x00,
    0x02, 0x03, 0x04
};

static const unsigned char tx_script_2_dgram[] = {

    0xcf, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0xf0, 0x67, 0xa5, 0x50, 0x2a,
    0x42, 0x62, 0xb5, 0x00, 0x40, 0x75, 0xc0, 0xd9, 0x5a, 0x48, 0x2c, 0xd0,
    0x99, 0x1c, 0xd2, 0x5b, 0x0a, 0xac, 0x40, 0x6a, 0x58, 0x16, 0xb6, 0x39,
    0x41, 0x00, 0xf3, 0x7a, 0x1c, 0x69, 0x79, 0x75, 0x54, 0x78, 0x0b, 0xb3,
    0x8c, 0xc5, 0xa9, 0x9f, 0x5e, 0xde, 0x4c, 0xf7, 0x3c, 0x3e, 0xc2, 0x49,
    0x3a, 0x18, 0x39, 0xb3, 0xdb, 0xcb, 0xa3, 0xf6, 0xea, 0x46, 0xc5, 0xb7,
    0x68, 0x4d, 0xf3, 0x54, 0x8e, 0x7d, 0xde, 0xb9, 0xc3, 0xbf, 0x9c, 0x73,
    0xcc, 0x3f, 0x3b, 0xde, 0xd7, 0x4b, 0x56, 0x2b, 0xfb, 0x19, 0xfb, 0x84,
    0x02, 0x2f, 0x8e, 0xf4, 0xcd, 0xd9, 0x37, 0x95, 0xd7, 0x7d, 0x06, 0xed,
    0xbb, 0x7a, 0xaf, 0x2f, 0x58, 0x89, 0x18, 0x50, 0xab, 0xbd, 0xca, 0x3d,
    0x20, 0x39, 0x8c, 0x27, 0x64, 0x56, 0xcb, 0xc4, 0x21, 0x58, 0x40, 0x7d,
    0xd0, 0x74, 0xee
};

static QUIC_PKT_HDR tx_script_2_hdr = {
    QUIC_PKT_TYPE_INITIAL,      /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    2,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    1,                          /* version */
    { 0, {0} },                 /* DCID */
    {8, {0xf0, 0x67, 0xa5, 0x50, 0x2a, 0x42, 0x62, 0xb5}}, /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_2_iovec[] = {
    { tx_script_2_body, sizeof(tx_script_2_body) }
};

static const OSSL_QTX_PKT tx_script_2_pkt = {
    &tx_script_2_hdr,
    tx_script_2_iovec,
    OSSL_NELEM(tx_script_2_iovec),
    NULL, NULL,
    1,
    0
};

static const struct tx_test_op tx_script_2[] = {
    TX_OP_PROVIDE_SECRET_INITIAL(tx_script_1_hdr.dst_conn_id, 1)
    TX_OP_WRITE_CHECK(2)
    TX_OP_END
};

#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
/* 3. RFC 9001 - A.5 ChaCha20-Poly1305 Short Header Packet */
static const unsigned char tx_script_3_body[] = {
    0x01
};

static const unsigned char tx_script_3_dgram[] = {
    0x4c, 0xfe, 0x41, 0x89, 0x65, 0x5e, 0x5c, 0xd5, 0x5c, 0x41, 0xf6, 0x90,
    0x80, 0x57, 0x5d, 0x79, 0x99, 0xc2, 0x5a, 0x5b, 0xfb
};
static const unsigned char tx_script_3_secret[] = {
    0x9a, 0xc3, 0x12, 0xa7, 0xf8, 0x77, 0x46, 0x8e, 0xbe, 0x69, 0x42, 0x27,
    0x48, 0xad, 0x00, 0xa1, 0x54, 0x43, 0xf1, 0x82, 0x03, 0xa0, 0x7d, 0x60,
    0x60, 0xf6, 0x88, 0xf3, 0x0f, 0x21, 0x63, 0x2b
};

static QUIC_PKT_HDR tx_script_3_hdr = {
    QUIC_PKT_TYPE_1RTT,         /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    3,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    0,                          /* version */
    { 0, {0} },                 /* DCID */
    { 0, {0} },                 /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_3_iovec[] = {
    { tx_script_3_body, sizeof(tx_script_3_body) }
};

static const OSSL_QTX_PKT tx_script_3_pkt = {
    &tx_script_3_hdr,
    tx_script_3_iovec,
    OSSL_NELEM(tx_script_3_iovec),
    NULL, NULL,
    654360564,
    0
};

static const struct tx_test_op tx_script_3[] = {
    TX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT, QRL_SUITE_CHACHA20POLY1305, tx_script_3_secret)
    TX_OP_WRITE_CHECK(3)
    TX_OP_END
};
#endif /* !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305) */

/* 4. Real World - AES-128-GCM Key Update */
static const unsigned char tx_script_4_secret[] = {
    0x70, 0x82, 0xc0, 0x45, 0x61, 0x4d, 0xfe, 0x04, 0x76, 0xa6, 0x4e, 0xf0,
    0x38, 0xe6, 0x63, 0xd9, 0xdd, 0x4a, 0x75, 0x16, 0xa8, 0xa0, 0x06, 0x5a,
    0xf2, 0x56, 0xfd, 0x84, 0x78, 0xfd, 0xf6, 0x5e
};

static const unsigned char tx_script_4a_body[] = {
    0x02, 0x03, 0x09, 0x00, 0x03, 0x0c, 0x00, 0x36, 0x49, 0x27, 0x6d, 0x20,
    0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f, 0x6e,
    0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char tx_script_4a_dgram[] = {
    0x47, 0x6e, 0x4e, 0xbd, 0x49, 0x7e, 0xbd, 0x15, 0x1c, 0xd1, 0x3e, 0xc8,
    0xcd, 0x43, 0x87, 0x6b, 0x84, 0xdb, 0xeb, 0x06, 0x8b, 0x8a, 0xae, 0x37,
    0xed, 0x9c, 0xeb, 0xbc, 0xcf, 0x0d, 0x3c, 0xf0, 0xa1, 0x6f, 0xee, 0xd2,
    0x7c, 0x07, 0x6e, 0xd1, 0xbe, 0x40, 0x6a, 0xd4, 0x53, 0x38, 0x9e, 0x63,
    0xb5, 0xde, 0x35, 0x09, 0xb2, 0x78, 0x94, 0xe4, 0x2b, 0x37
};

static QUIC_PKT_HDR tx_script_4a_hdr = {
    QUIC_PKT_TYPE_1RTT,         /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    2,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    0,                          /* version */
    { 4, {0x6e, 0x4e, 0xbd, 0x49} }, /* DCID */
    { 0, {0} },                 /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_4a_iovec[] = {
    { tx_script_4a_body, sizeof(tx_script_4a_body) }
};

static const OSSL_QTX_PKT tx_script_4a_pkt = {
    &tx_script_4a_hdr,
    tx_script_4a_iovec,
    OSSL_NELEM(tx_script_4a_iovec),
    NULL, NULL,
    4,
    0
};

static const unsigned char tx_script_4b_body[] = {
    0x02, 0x04, 0x07, 0x00, 0x00, 0x0c, 0x00, 0x40, 0x51, 0x49, 0x27, 0x6d,
    0x20, 0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f,
    0x6e, 0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char tx_script_4b_dgram[] = {
    0x58, 0x6e, 0x4e, 0xbd, 0x49, 0xa4, 0x43, 0x33, 0xea, 0x11, 0x3a, 0x6c,
    0xf5, 0x20, 0xef, 0x55, 0x8d, 0x25, 0xe2, 0x3b, 0x0e, 0x8c, 0xea, 0x17,
    0xfc, 0x2b, 0x7a, 0xab, 0xfa, 0x3d, 0x07, 0xda, 0xa7, 0x7c, 0xc7, 0x47,
    0x82, 0x02, 0x46, 0x40, 0x4f, 0x01, 0xad, 0xb2, 0x9d, 0x97, 0xdb, 0xfc,
    0x9c, 0x4b, 0x46, 0xb1, 0x5a, 0x7f, 0x0b, 0x12, 0xaf, 0x49, 0xdf,
};

static QUIC_PKT_HDR tx_script_4b_hdr = {
    QUIC_PKT_TYPE_1RTT,         /* type */
    0,                          /* spin bit */
    1,                          /* key phase */
    2,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    0,                          /* version */
    { 4, {0x6e, 0x4e, 0xbd, 0x49} }, /* DCID */
    { 0, {0} },                 /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_4b_iovec[] = {
    { tx_script_4b_body, sizeof(tx_script_4b_body) }
};

static const OSSL_QTX_PKT tx_script_4b_pkt = {
    &tx_script_4b_hdr,
    tx_script_4b_iovec,
    OSSL_NELEM(tx_script_4b_iovec),
    NULL, NULL,
    5,
    0
};

static const unsigned char tx_script_4c_body[] = {
    0x02, 0x09, 0x0e, 0x00, 0x00, 0x0c, 0x00, 0x40, 0xd8, 0x49, 0x27, 0x6d,
    0x20, 0x68, 0x61, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x77, 0x6f,
    0x6e, 0x64, 0x65, 0x72, 0x66, 0x75, 0x6c, 0x20, 0x74, 0x69, 0x6d, 0x65,
};

static const unsigned char tx_script_4c_dgram[] = {
    0x49, 0x6e, 0x4e, 0xbd, 0x49, 0x4d, 0xd9, 0x85, 0xba, 0x26, 0xfb, 0x68,
    0x83, 0x9b, 0x94, 0x34, 0x7d, 0xc1, 0x7a, 0x05, 0xb7, 0x38, 0x43, 0x21,
    0xe2, 0xec, 0x2b, 0xc1, 0x81, 0x74, 0x2d, 0xda, 0x24, 0xba, 0xbd, 0x99,
    0x69, 0xd2, 0x56, 0xfa, 0xae, 0x29, 0x24, 0xb2, 0xaa, 0xda, 0xbd, 0x82,
    0x80, 0xf1, 0xbb, 0x6a, 0xfd, 0xae, 0xda, 0x0e, 0x09, 0xcf, 0x09,
};

static QUIC_PKT_HDR tx_script_4c_hdr = {
    QUIC_PKT_TYPE_1RTT,         /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    2,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    0,                          /* version */
    { 4, {0x6e, 0x4e, 0xbd, 0x49} }, /* DCID */
    { 0, {0} },                 /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_4c_iovec[] = {
    { tx_script_4c_body, sizeof(tx_script_4c_body) }
};

static const OSSL_QTX_PKT tx_script_4c_pkt = {
    &tx_script_4c_hdr,
    tx_script_4c_iovec,
    OSSL_NELEM(tx_script_4c_iovec),
    NULL, NULL,
    10,
    0
};

static const struct tx_test_op tx_script_4[] = {
    TX_OP_PROVIDE_SECRET(QUIC_ENC_LEVEL_1RTT, QRL_SUITE_AES128GCM, tx_script_4_secret)
    TX_OP_WRITE_CHECK(4a)
    TX_OP_KEY_UPDATE()
    TX_OP_WRITE_CHECK(4b)
    TX_OP_KEY_UPDATE()
    TX_OP_WRITE_CHECK(4c)
    TX_OP_END
};

/* 5. Real World - Retry Packet */
static const unsigned char tx_script_5_body[] = {
    /* Retry Token */
    0x92, 0xe7, 0xc6, 0xd8, 0x09, 0x65, 0x72, 0x55, 0xe5, 0xe2, 0x73, 0x04,
    0xf3, 0x07, 0x5b, 0x21, 0x9f, 0x50, 0xcb, 0xbc, 0x79, 0xc5, 0x77, 0x5a,
    0x29, 0x43, 0x65, 0x49, 0xf0, 0x6e, 0xc1, 0xc0, 0x3a, 0xe8, 0xca, 0xd2,
    0x44, 0x69, 0xdd, 0x23, 0x31, 0x93, 0x52, 0x02, 0xf7, 0x42, 0x07, 0x78,
    0xa1, 0x81, 0x61, 0x9c, 0x39, 0x07, 0x18, 0x69, 0x6e, 0x4f, 0xdc, 0xa0,
    0xbe, 0x4b, 0xe5, 0xf2, 0xe9, 0xd2, 0xa4, 0xa7, 0x34, 0x55, 0x5e, 0xf3,
    0xf8, 0x9c, 0x49, 0x8f, 0x0c, 0xc8, 0xb2, 0x75, 0x4b, 0x4d, 0x2f, 0xfe,
    0x05, 0x5a, 0xdd, 0x4b, 0xe6, 0x14, 0xb4, 0xd2, 0xc0, 0x93, 0x6e, 0x0e,
    0x84, 0x41, 0x4d, 0x31,
    /* Retry Integrity Tag */
    0x43, 0x8e, 0xab, 0xcd, 0xce, 0x24, 0x44, 0xc2, 0x20, 0xe1, 0xe2, 0xc8,
    0xae, 0xa3, 0x8d, 0x4e,
};

static const unsigned char tx_script_5_dgram[] = {
    0xf0, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0xa9, 0x20, 0xcc, 0xc2, 0x92,
    0xe7, 0xc6, 0xd8, 0x09, 0x65, 0x72, 0x55, 0xe5, 0xe2, 0x73, 0x04, 0xf3,
    0x07, 0x5b, 0x21, 0x9f, 0x50, 0xcb, 0xbc, 0x79, 0xc5, 0x77, 0x5a, 0x29,
    0x43, 0x65, 0x49, 0xf0, 0x6e, 0xc1, 0xc0, 0x3a, 0xe8, 0xca, 0xd2, 0x44,
    0x69, 0xdd, 0x23, 0x31, 0x93, 0x52, 0x02, 0xf7, 0x42, 0x07, 0x78, 0xa1,
    0x81, 0x61, 0x9c, 0x39, 0x07, 0x18, 0x69, 0x6e, 0x4f, 0xdc, 0xa0, 0xbe,
    0x4b, 0xe5, 0xf2, 0xe9, 0xd2, 0xa4, 0xa7, 0x34, 0x55, 0x5e, 0xf3, 0xf8,
    0x9c, 0x49, 0x8f, 0x0c, 0xc8, 0xb2, 0x75, 0x4b, 0x4d, 0x2f, 0xfe, 0x05,
    0x5a, 0xdd, 0x4b, 0xe6, 0x14, 0xb4, 0xd2, 0xc0, 0x93, 0x6e, 0x0e, 0x84,
    0x41, 0x4d, 0x31, 0x43, 0x8e, 0xab, 0xcd, 0xce, 0x24, 0x44, 0xc2, 0x20,
    0xe1, 0xe2, 0xc8, 0xae, 0xa3, 0x8d, 0x4e,
};

static QUIC_PKT_HDR tx_script_5_hdr = {
    QUIC_PKT_TYPE_RETRY,        /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    0,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    1,                          /* version */
    { 0, {0} },                 /* DCID */
    { 4, {0xa9, 0x20, 0xcc, 0xc2} },   /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_5_iovec[] = {
    { tx_script_5_body, sizeof(tx_script_5_body) }
};

static const OSSL_QTX_PKT tx_script_5_pkt = {
    &tx_script_5_hdr,
    tx_script_5_iovec,
    OSSL_NELEM(tx_script_5_iovec),
    NULL, NULL,
    0,
    0
};

static const struct tx_test_op tx_script_5[] = {
    TX_OP_WRITE_CHECK(5)
    TX_OP_END
};

/* 6. Real World - Version Negotiation Packet */
static const unsigned char tx_script_6_body[] = {
    0x00, 0x00, 0x00, 0x01,             /* Supported Version: 1 */
    0xaa, 0x9a, 0x3a, 0x9a              /* Supported Version: Random (GREASE) */
};

static const unsigned char tx_script_6_dgram[] = {
    0x80,                               /* Long */
    0x00, 0x00, 0x00, 0x00,             /* Version 0 (Version Negotiation) */
    0x00,                               /* DCID */
    0x0c, 0x35, 0x3c, 0x1b, 0x97, 0xca, /* SCID */
    0xf8, 0x99, 0x11, 0x39, 0xad, 0x79,
    0x1f,
    0x00, 0x00, 0x00, 0x01,             /* Supported Version: 1 */
    0xaa, 0x9a, 0x3a, 0x9a              /* Supported Version: Random (GREASE) */
};

static QUIC_PKT_HDR tx_script_6_hdr = {
    QUIC_PKT_TYPE_VERSION_NEG,  /* type */
    0,                          /* spin bit */
    0,                          /* key phase */
    0,                          /* PN length */
    0,                          /* partial */
    0,                          /* fixed */
    0,                          /* unused */
    0,                          /* reserved */
    0,                          /* version */
    { 0, {0} },                 /* DCID */
    { 12, {0x35, 0x3c, 0x1b, 0x97, 0xca, 0xf8, 0x99,
           0x11, 0x39, 0xad, 0x79, 0x1f} }, /* SCID */
    { 0 },                      /* PN */
    NULL, 0,                    /* Token */
    5555, NULL                  /* Len/Data */
};

static const OSSL_QTX_IOVEC tx_script_6_iovec[] = {
    { tx_script_6_body, sizeof(tx_script_6_body) }
};

static const OSSL_QTX_PKT tx_script_6_pkt = {
    &tx_script_6_hdr,
    tx_script_6_iovec,
    OSSL_NELEM(tx_script_6_iovec),
    NULL, NULL,
    0,
    0
};

static const struct tx_test_op tx_script_6[] = {
    TX_OP_WRITE_CHECK(6)
    TX_OP_END
};

static const struct tx_test_op *const tx_scripts[] = {
    tx_script_1,
    tx_script_2,
#if !defined(OPENSSL_NO_CHACHA) && !defined(OPENSSL_NO_POLY1305)
    tx_script_3,
#endif
    tx_script_4,
    tx_script_5,
    tx_script_6
};

static int tx_run_script(const struct tx_test_op *script)
{
    int testresult = 0;
    const struct tx_test_op *op = script;
    OSSL_QTX *qtx = NULL;
    BIO_MSG msg = {0};
    OSSL_QTX_ARGS args = {0};

    args.mdpl = 1472;

    if (!TEST_ptr(qtx = ossl_qtx_new(&args)))
        goto err;

    for (; op->op != TX_TEST_OP_END; ++op)
        switch (op->op) {
            case TX_TEST_OP_PROVIDE_SECRET:
                if (!TEST_true(ossl_qtx_provide_secret(qtx, op->enc_level,
                                                       op->suite_id, NULL,
                                                       op->buf, op->buf_len)))
                    goto err;
                break;
            case TX_TEST_OP_PROVIDE_SECRET_INITIAL:
                if (!TEST_true(ossl_quic_provide_initial_secret(NULL, NULL,
                                                                op->dcid,
                                                                (int)op->suite_id,
                                                                NULL, qtx)))
                    goto err;
                break;
            case TX_TEST_OP_DISCARD_EL:
                if (!TEST_true(ossl_qtx_discard_enc_level(qtx, op->enc_level)))
                    goto err;
                break;
            case TX_TEST_OP_WRITE:
                {
                    uint32_t enc_level
                        = ossl_quic_pkt_type_to_enc_level(op->pkt->hdr->type);
                    uint64_t old_value = 0, new_value, max_value;

                    if (enc_level < QUIC_ENC_LEVEL_NUM) { /* encrypted packet */
                        max_value = ossl_qtx_get_max_epoch_pkt_count(qtx, enc_level);

                        if (!TEST_uint64_t_lt(max_value, UINT64_MAX))
                            goto err;

                        old_value = ossl_qtx_get_cur_epoch_pkt_count(qtx, enc_level);
                        if (!TEST_uint64_t_lt(old_value, UINT64_MAX))
                            goto err;
                    }

                    if (!TEST_true(ossl_qtx_write_pkt(qtx, op->pkt)))
                        goto err;

                    if (enc_level < QUIC_ENC_LEVEL_NUM) {
                        new_value = ossl_qtx_get_cur_epoch_pkt_count(qtx, enc_level);
                        if (!TEST_uint64_t_eq(old_value + 1, new_value))
                            goto err;
                    }
                }
                break;
            case TX_TEST_OP_CHECK_DGRAM:
                if (!TEST_true(ossl_qtx_pop_net(qtx, &msg)))
                    goto err;

                if (!TEST_mem_eq(msg.data, msg.data_len, op->buf, op->buf_len))
                    goto err;

                break;
            case TX_TEST_OP_CHECK_NO_DGRAM:
                if (!TEST_false(ossl_qtx_pop_net(qtx, &msg)))
                    goto err;
                break;
            case TX_TEST_OP_KEY_UPDATE:
                if (!TEST_true(ossl_qtx_trigger_key_update(qtx)))
                    goto err;
                break;
            default:
                OPENSSL_assert(0);
                goto err;
        }

    testresult = 1;
err:
    if (qtx != NULL)
        ossl_qtx_free(qtx);

    return testresult;
}

static int test_tx_script(int idx)
{
    return tx_run_script(tx_scripts[idx]);
}

int setup_tests(void)
{
    ADD_ALL_TESTS(test_rx_script, OSSL_NELEM(rx_scripts));
    /*
     * Each instance of this test is executed multiple times to get enough
     * statistical coverage for our statistical test, as well as for each
     * supported key type.
     *
     * We call the statistical test as the last index in the wire_pkt_hdr
     * test rather than as a separate case, as it needs to execute last
     * and otherwise random test ordering will cause itt to randomly fail.
     */
    ADD_ALL_TESTS(test_wire_pkt_hdr, NUM_WIRE_PKT_HDR_TESTS + 1);
    ADD_ALL_TESTS(test_tx_script, OSSL_NELEM(tx_scripts));
    return 1;
}
