/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * Authors:
 *  Olivier Tilloy <olivier.tilloy@canonical.com>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; version 3.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "bamf-view.h"
#include "bamf-view-proxy.h"
#include "bamf-factory.h"

BamfView::BamfView(QString path) :
    m_proxy(NULL)
{
    m_proxy = new OrgAyatanaBamfViewInterface("org.ayatana.bamf", path,
                                              QDBusConnection::sessionBus(),
                                              static_cast<QObject*>(this));
    // Forward signals from the proxy
    connect(m_proxy, SIGNAL(ActiveChanged(bool)),
            this, SIGNAL(ActiveChanged(bool)));
    connect(m_proxy, SIGNAL(Closed()),
            this, SLOT(OnViewClosed()));
    connect(m_proxy, SIGNAL(ChildAdded(const QString &)),
            this, SLOT(OnChildAdded(const QString &)));
    connect(m_proxy, SIGNAL(ChildRemoved(const QString &)),
            this, SLOT(OnChildRemoved(const QString &)));
    connect(m_proxy, SIGNAL(RunningChanged(bool)),
            this, SIGNAL(RunningChanged(bool)));
    connect(m_proxy, SIGNAL(UrgentChanged(bool)),
            this, SIGNAL(UrgentChanged(bool)));
    connect(m_proxy, SIGNAL(UserVisibleChanged(bool)),
            this, SIGNAL(UserVisibleChanged(bool)));
}

BamfView::~BamfView()
{
    delete m_proxy;
}

bool
BamfView::active() const
{
    QDBusPendingReply<bool> reply = m_proxy->IsActive();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return false;
    } else {
        return reply.value();
    }
}

const QString
BamfView::path() const
{
    return m_proxy->path();
}

bool
BamfView::running() const
{
    QDBusPendingReply<bool> reply = m_proxy->IsRunning();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return false;
    } else {
        return reply.value();
    }
}

bool
BamfView::urgent() const
{
    QDBusPendingReply<bool> reply = m_proxy->IsUrgent();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return false;
    } else {
        return reply.value();
    }
}

bool
BamfView::user_visible() const
{
    QDBusPendingReply<bool> reply = m_proxy->UserVisible();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return false;
    } else {
        return reply.value();
    }
}

const QString
BamfView::name() const
{
    QDBusPendingReply<QString> reply = m_proxy->Name();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return QString();
    } else {
        return reply.value();
    }
}

const QString
BamfView::icon() const
{
    QDBusPendingReply<QString> reply = m_proxy->Icon();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return QString();
    } else {
        return reply.value();
    }
}

const QString
BamfView::view_type() const
{
    QDBusPendingReply<QString> reply = m_proxy->ViewType();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return QString();
    } else {
        return reply.value();
    }
}

BamfViewList *
BamfView::children() const
{
    QList<BamfView *> result;
    QDBusPendingReply<QStringList> reply = m_proxy->Children();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
    } else {
        QStringList list = reply.value();
        for (int i = 0; i < list.size(); ++i) {
            BamfView *view = BamfFactory::get_default().view_for_path(list.at(i));
            result.prepend(view);
        }
    }
    return new BamfViewList(result);
}

void
BamfView::OnViewClosed()
{
    Closed();
}

void
BamfView::OnChildAdded(const QString &path)
{
    BamfView *view = BamfFactory::get_default().view_for_path(path);
    ChildAdded(view);
}

void
BamfView::OnChildRemoved(const QString &path)
{
    BamfView *view = BamfFactory::get_default().view_for_path(path);
    ChildRemoved(view);
}

