#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# generate-plugin.py - xed plugin skeletton generator
# This file is part of xed
#
# Copyright (C) 2006 - Steve Frécinaux
#
# xed is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# xed is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with xed; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor,
# Boston, MA  02110-1301  USA

import re
import os
import argparse
from datetime import date
import preprocessor

TEMPLATE_DIR = os.path.join(os.path.dirname(__file__), 'plugin_template')
PLUGIN_DIR = os.path.normpath(os.path.join(os.path.dirname(__file__), '../plugins'))

# Parsing command line options
parser = argparse.ArgumentParser()
parser.add_argument('-S', '--standalone', dest='standalone', action='store_true',
                    help='indicates that this plugin is intended to be distributed as a standalone package')
parser.add_argument('-s', '--with-side-pane', dest='side_pane', action='store_true',
                    help='Indicates that this plugin will use a side pane')
parser.add_argument('-b', '--with-bottom-pane', dest='bottom_pane', action='store_true',
                    help='Indicates that this plugin will use a bottom pane')
parser.add_argument('-m', '--with-menu', dest='menu', action='store_true',
                    help='Indicates that this plugin will use menu entries')
parser.add_argument('-c', '--with-config', dest='config', action='store_true',
                    help='Indicates that this plugin will use a configuration dialog')
parser.add_argument('-d', '--description', dest='description', default='Type here a short description of your plugin', metavar='DESC',
                    help='Description of the plugin')
parser.add_argument('-a', '--author', dest='author', default=os.getenv('USERNAME'), metavar='AUTH',
                    help='Author of the plugin')
parser.add_argument('-e', '--email', dest='email', default=os.getenv('LOGNAME') + '@email.com', metavar='EMAIL',
                    help='Email address of the author')
parser.add_argument('-l', '--language', dest='language', default='c', metavar='LANG',
                    help='Language of the plugin')
parser.add_argument('-o', '--output-directory', dest='directory', default=None, metavar='LANG',
                    help='Language of the plugin')
parser.add_argument('name', metavar='PLUGIN_NAME',
                    help='The name of the plugin')

args = parser.parse_args()

plugin_name = args.name
plugin_id = re.sub('[^a-z0-9_]', '', plugin_name.lower().replace(' ', '_'))
plugin_module = plugin_id.replace('_', '-')

directives = {
    'PLUGIN_NAME'       : plugin_name,
    'PLUGIN_MODULE'     : plugin_module,
    'PLUGIN_ID'         : plugin_id,
    'AUTHOR_FULLNAME'   : args.author,
    'AUTHOR_EMAIL'      : args.email,
    'DATE_YEAR'         : date.today().year,
    'DESCRIPTION'       : args.description,
}

# Files to be generated by the preprocessor, in the form "template : outfile"
output_files = {
    'meson.build': 'meson.build',
    'xed-plugin.desktop.in': '%s.plugin.desktop.in' % plugin_module
}

if args.language == 'c':
    directives['HAS_C_FILES'] = True
    output_files['xed-plugin.c'] = 'xed-%s-plugin.c' % plugin_module
    output_files['xed-plugin.h'] = 'xed-%s-plugin.h' % plugin_module

if args.side_pane:
    directives['WITH_SIDE_PANE'] = True

if args.bottom_pane:
    directives['WITH_BOTTOM_PANE'] = True

if args.menu:
    directives['WITH_MENU'] = True

if args.config:
    directives['WITH_CONFIGURE_DIALOG'] = True

if args.directory is None:
    directory = os.getcwd() if args.standalone else PLUGIN_DIR
elif os.path.isdir(args.directory):
    directory = args.directory
else:
    print('Unable to create plugin: %s does not exist or is not a directory' % args.directory)
    quit(1)

directory = os.path.join(directory, plugin_module)
if os.path.exists(directory):
    print('Unable to create plugin: directory %s already exists' % directory)
    quit(1)
else:
    os.makedirs(directory)

if not args.standalone:
    with open(os.path.join(PLUGIN_DIR, 'meson.build'), 'r') as f:
        contents = f.read()
        lines = contents.split('\n')

        start = False
        for i in range(len(lines)):
            line = lines[i].rstrip()
            print(line)
            if line.startswith('subdir('):
                start = True
            elif start:
                break
            else:
                continue

            if line[8:-2] > plugin_module:
                break

        lines.insert(i, 'subdir(\'%s\')' % plugin_module)

    with open(os.path.join(PLUGIN_DIR, 'meson.build'), 'w') as f:
        f.write('\n'.join(lines))

# Generate the plugin base
for infile, outfile in output_files.iteritems():
    print('Generating file %s from template %s...' % (outfile, infile))

    file_directives = directives.copy()

    infile = os.path.join(TEMPLATE_DIR, infile)
    outfile = os.path.join(directory, outfile)

    if not os.path.isfile(infile):
        print('Input file %s does not exist: skipping' % os.path.basename(infile))
        continue

    # Make sure the destination directory exists
    if not os.path.isdir(os.path.split(outfile)[0]):
        os.makedirs(os.path.split(outfile)[0])

    # Variables relative to the generated file
    file_directives['DIRNAME'], file_directives['FILENAME'] = os.path.split(outfile)

    # Generate the file
    preprocessor.process(infile, outfile, file_directives)

print('Done')

# ex:ts=4:et:
